-- SKTools Totem Tracking
-- All totem tracking logic + Shaman settings tab (both main & combat)

local _, ns = ...

-----------------------------
-- Action Bar Prefixes (used by totem button detection)
-----------------------------
local ACTION_BAR_PREFIX = {
    "ActionButton",                -- Bar 1 (Main)
    "MultiBarBottomLeftButton",    -- Bar 2
    "MultiBarBottomRightButton",   -- Bar 3
    "MultiBarRightButton",         -- Bar 4
    "MultiBarLeftButton",          -- Bar 5
    "MultiBar5Button",             -- Bar 6
    "MultiBar6Button",             -- Bar 7
    "MultiBar7Button",             -- Bar 8
}

-----------------------------
-- Totem Duration Tracking
-----------------------------
-- Short-duration totems (defensive/utility)
local TOTEM_COUNTDOWN_GROUPS = {
    { key = "groundingTotem",      label = "Grounding Totem",      id = 204336 },
    { key = "tremorTotem",         label = "Tremor Totem",         id = 8143 },
    { key = "capacitorTotem",      label = "Capacitor Totem",      id = 192058 },
    { key = "counterstrikeTotem",  label = "Counterstrike Totem",  id = 204331 },
    { key = "staticFieldTotem",    label = "Static Field Totem",   id = 355580 },
    { key = "healingTideTotem",    label = "Healing Tide Totem",   id = 108280 },
    { key = "spiritLinkTotem",     label = "Spirit Link Totem",    id = 98008 },
    { key = "windrushTotem",       label = "Windrush Totem",       id = 192077 },
    { key = "earthgrabTotem",      label = "Earthgrab Totem",      id = 51485 },
    { key = "poisonCleansingTotem", label = "Poison Cleansing Totem", id = 383013 },
}
ns.TOTEM_COUNTDOWN_GROUPS = TOTEM_COUNTDOWN_GROUPS

-- Long-duration totems (healing/support)
local TOTEM_BAR_GROUPS = {
    { key = "healingStreamTotem", label = "Healing Stream Totem", id = 5394 },
    { key = "stormstreamTotem",   label = "Stormstream Totem",    id = 1267068 },
    { key = "surgingTotem",       label = "Surging Totem",        id = 444995 },
    { key = "totemOfWrath",       label = "Totem of Wrath",       id = 204330 },
}
ns.TOTEM_BAR_GROUPS = TOTEM_BAR_GROUPS

-- Display duration overrides (show this instead of GetTotemInfo duration)
local TOTEM_DURATION_OVERRIDE = {
    [192058] = 2,  -- Capacitor Totem: stun triggers at 2s, totem lives 3s
}

-- Fallback durations for instanced PvP where GetTotemInfo is blocked (secret values)
local TOTEM_KNOWN_DURATION = {
    [204336] = 3,    -- Grounding Totem
    [8143]   = 10,   -- Tremor Totem
    [192058] = 3,    -- Capacitor Totem
    [204331] = 15,   -- Counterstrike Totem
    [355580] = 6,    -- Static Field Totem
    [108280] = 10,   -- Healing Tide Totem
    [98008]  = 6,    -- Spirit Link Totem
    [192077] = 15,   -- Windrush Totem
    [51485]  = 20,   -- Earthgrab Totem
    [383013] = 6,    -- Poison Cleansing Totem
    [5394]   = 15,   -- Healing Stream Totem
    [1267068]= 15,   -- Stormstream Totem
    [444995] = 24,   -- Surging Totem
    [204330] = 15,   -- Totem of Wrath
}

-- All spell IDs that need button detection
local totemAllSpellIDs = {}
for _, g in ipairs(TOTEM_COUNTDOWN_GROUPS) do totemAllSpellIDs[g.id] = true end
for _, g in ipairs(TOTEM_BAR_GROUPS) do totemAllSpellIDs[g.id] = true end

-- Built at runtime from enabled settings
local TOTEM_ENABLED = {}       -- [spellID] = true

local function Totem_RebuildTracked()
    wipe(TOTEM_ENABLED)
    local spells = SKToolsDB and SKToolsDB.totemTrackSpells or {}
    for _, group in ipairs(TOTEM_COUNTDOWN_GROUPS) do
        if spells[group.key] then TOTEM_ENABLED[group.id] = true end
    end
    for _, group in ipairs(TOTEM_BAR_GROUPS) do
        if spells[group.key] then TOTEM_ENABLED[group.id] = true end
    end
end
ns.Totem_RebuildTracked = Totem_RebuildTracked

local totemNames = {}       -- [spellID] = localized name
local totemIcons = {}       -- [spellID] = icon texture ID
local cachedTotemDuration = {}  -- [spellID] = real duration from GetTotemInfo (talent-adjusted)
local totemFrame = CreateFrame("Frame")

-- Only called outside PvP where GetTotemInfo returns clean values
local function SafeGetTotemInfo(slot)
    local ok, active, name, startTime, duration = pcall(GetTotemInfo, slot)
    if not ok then return false, nil, 0, 0 end
    -- GetTotemInfo returns secret/tainted booleans in instanced PvP (Midnight)
    -- Convert to a clean boolean by testing inside pcall
    local boolOk, boolVal = pcall(function() if active then return true else return false end end)
    if not boolOk then return false, nil, 0, 0 end
    return boolVal, name, startTime or 0, duration or 0
end


local cachedButtons = {}      -- [spellID] = { button frames }
local buttonCacheDirty = true -- invalidated by Totem_DetectAllButtons
local EMPTY_TABLE = {}        -- reusable empty table (never modified)

local function Totem_FindAllButtons(spellID)
    if not buttonCacheDirty and cachedButtons[spellID] ~= nil then
        return cachedButtons[spellID]
    end
    local saved = SKToolsDB and SKToolsDB.totemButtonSlots and SKToolsDB.totemButtonSlots[spellID]
    if not saved then
        cachedButtons[spellID] = EMPTY_TABLE
        return EMPTY_TABLE
    end
    local buttons = {}
    for _, entry in ipairs(saved) do
        local prefix = ACTION_BAR_PREFIX[entry[1]]
        if prefix then
            local btn = _G[prefix .. entry[2]]
            if btn then buttons[#buttons + 1] = btn end
        end
    end
    cachedButtons[spellID] = buttons
    return buttons
end

-- Scan all action bars and save button assignments for tracked spells
local function Totem_DetectAllButtons()
    if not SKToolsDB then return 0 end
    if InCombatLockdown() then return -1 end

    local slots = {}
    for barIdx, prefix in ipairs(ACTION_BAR_PREFIX) do
        for btnIdx = 1, 12 do
            local btn = _G[prefix .. btnIdx]
            if btn and btn.action then
                local hasOk, hasAct = pcall(HasAction, btn.action)
                if hasOk and hasAct then
                    local ok, aType, id = pcall(GetActionInfo, btn.action)
                    if ok then
                        local matchID
                        if aType == "spell" and totemAllSpellIDs[id] then
                            matchID = id
                        elseif aType == "macro" then
                            local ok2, macroSpell = pcall(GetMacroSpell, id)
                            if ok2 and macroSpell then
                                for spellID in pairs(totemAllSpellIDs) do
                                    if totemNames[spellID] == macroSpell then
                                        matchID = spellID
                                        break
                                    end
                                end
                            end
                            if not matchID then
                                local ok3, actionIcon = pcall(GetActionTexture, btn.action)
                                if ok3 and actionIcon then
                                    for spellID in pairs(totemAllSpellIDs) do
                                        if totemIcons[spellID] == actionIcon then
                                            matchID = spellID
                                            break
                                        end
                                    end
                                end
                            end
                        end
                        if matchID then
                            if not slots[matchID] then slots[matchID] = {} end
                            table.insert(slots[matchID], {barIdx, btnIdx})
                        end
                    end
                end
            end
        end
    end
    -- Merge into saved data: update spells that were found, keep old entries for spells
    -- not detected this scan (e.g. PvP talents not on bar outside PvP instances)
    if not SKToolsDB.totemButtonSlots then SKToolsDB.totemButtonSlots = {} end
    for spellID, entries in pairs(slots) do
        SKToolsDB.totemButtonSlots[spellID] = entries
    end
    -- Invalidate button cache so next layout picks up new assignments
    wipe(cachedButtons)
    buttonCacheDirty = false
    local count = 0
    for _ in pairs(SKToolsDB.totemButtonSlots) do count = count + 1 end
    return count
end

-- Cache talent-adjusted totem durations from GetTotemInfo (only works outside PvP)
local function Totem_CacheDurations()
    for spellID in pairs(TOTEM_ENABLED) do
        local name = totemNames[spellID]
        if name then
            for slot = 1, (MAX_TOTEMS or 5) do
                local haveTotem, totemName, startTime, duration = SafeGetTotemInfo(slot)
                if haveTotem and totemName and totemName == name then
                    if duration and duration > 0 then
                        cachedTotemDuration[spellID] = duration
                        if SKToolsDB then
                            if not SKToolsDB.cachedTotemDurations then SKToolsDB.cachedTotemDurations = {} end
                            SKToolsDB.cachedTotemDurations[spellID] = duration
                        end
                    end
                    break
                end
            end
        end
    end
end


-----------------------------
-- Standalone Totem Tracker (circular badges on action bar buttons)
-----------------------------
local standaloneSlots = {}   -- array of slot data tables
local standaloneNextID = 1
local layoutBySpell = {}     -- reusable table for Standalone_Layout grouping
local STANDALONE_BADGE_SIZE = 16
local STANDALONE_BADGE_BORDER = 2
local STANDALONE_BADGE_GAP = 2
local STANDALONE_BORDER_COLOR = {0.75, 0.6, 0.1, 0.9}  -- gold, like Blizzard totem frames

-- Create a single circular badge frame (used for primary and clones)
local function Standalone_CreateBadge(spellID)
    local SIZE = STANDALONE_BADGE_SIZE
    local BORDER_WIDTH = 2

    local slot = CreateFrame("Frame", nil, UIParent)
    slot:SetFrameStrata("MEDIUM")
    slot:SetFrameLevel(100)
    slot:SetSize(SIZE, SIZE)

    -- Circular mask for the icon area
    local mask = slot:CreateMaskTexture()
    mask:SetAllPoints()
    mask:SetTexture("Interface\\CHARACTERFRAME\\TempPortraitAlphaMask", "CLAMPTOBLACKADDITIVE", "CLAMPTOBLACKADDITIVE")

    -- Circular border ring (gold, slightly larger, behind everything)
    local borderSize = SIZE + BORDER_WIDTH * 2
    local border = slot:CreateTexture(nil, "BACKGROUND", nil, -1)
    border:SetSize(borderSize, borderSize)
    border:SetPoint("CENTER")
    border:SetColorTexture(STANDALONE_BORDER_COLOR[1], STANDALONE_BORDER_COLOR[2], STANDALONE_BORDER_COLOR[3], STANDALONE_BORDER_COLOR[4])
    local borderMask = slot:CreateMaskTexture()
    borderMask:SetSize(borderSize, borderSize)
    borderMask:SetPoint("CENTER")
    borderMask:SetTexture("Interface\\CHARACTERFRAME\\TempPortraitAlphaMask", "CLAMPTOBLACKADDITIVE", "CLAMPTOBLACKADDITIVE")
    border:AddMaskTexture(borderMask)

    -- Dark circular background
    local bg = slot:CreateTexture(nil, "BACKGROUND")
    bg:SetAllPoints()
    bg:SetColorTexture(0, 0, 0, 0.85)
    bg:AddMaskTexture(mask)

    -- Icon texture (circular, fills frame)
    local icon = slot:CreateTexture(nil, "ARTWORK")
    icon:SetAllPoints()
    icon:SetTexCoord(0.08, 0.92, 0.08, 0.92)
    local tex = totemIcons[spellID]
    if tex then icon:SetTexture(tex) end
    icon:AddMaskTexture(mask)

    -- Cooldown sweep inside the badge (circular, like Blizzard totem frames)
    local cd = CreateFrame("Cooldown", nil, slot, "CooldownFrameTemplate")
    cd:SetAllPoints()
    cd:SetDrawSwipe(true)
    cd:SetSwipeColor(0, 0, 0, 0.6)
    cd:SetDrawEdge(true)
    cd:SetHideCountdownNumbers(true)
    cd:SetReverse(true)
    cd:SetUseCircularEdge(true)

    -- Countdown text below the icon (like Blizzard totem frames)
    local text = slot:CreateFontString(nil, "OVERLAY")
    text:SetFont(STANDARD_TEXT_FONT, 9, "OUTLINE")
    text:SetPoint("TOP", slot, "BOTTOM", 0, -1)
    text:SetTextColor(1, 1, 1, 1)
    text:SetText("")

    slot:Hide()
    return { frame = slot, icon = icon, text = text, cooldown = cd }
end

local function Standalone_CreateSlotFrame(spellID)
    local badge = Standalone_CreateBadge(spellID)
    return {
        frame = badge.frame, icon = badge.icon, text = badge.text,
        cooldown = badge.cooldown,
        timer = nil, active = false, spellID = spellID,
        expiry = 0, id = 0, startTime = 0, duration = 0, clones = {}
    }
end

-- Get or create a clone badge for showing on additional buttons
local function Standalone_GetClone(data, index)
    if not data.clones[index] then
        local badge = Standalone_CreateBadge(data.spellID)
        -- Copy the icon texture from the primary
        local tex = data.icon:GetTexture()
        if tex then badge.icon:SetTexture(tex) end
        data.clones[index] = badge
    end
    return data.clones[index]
end

-- Find an inactive slot for reuse, or create a new one
local function Standalone_AcquireSlot(spellID)
    for _, data in ipairs(standaloneSlots) do
        if not data.active and data.spellID == spellID then
            data.id = standaloneNextID
            standaloneNextID = standaloneNextID + 1
            return data
        end
    end
    local data = Standalone_CreateSlotFrame(spellID)
    data.id = standaloneNextID
    standaloneNextID = standaloneNextID + 1
    table.insert(standaloneSlots, data)
    return data
end

local function Standalone_Layout()
    -- Hide all clones first (prevents orphans when buttons disappear)
    for _, data in ipairs(standaloneSlots) do
        if data.active then
            for _, clone in pairs(data.clones) do clone.frame:Hide() end
        end
    end

    -- Group active slots by spellID (reuse module-level tables)
    for _, instances in pairs(layoutBySpell) do wipe(instances) end
    for _, data in ipairs(standaloneSlots) do
        if data.active then
            if not layoutBySpell[data.spellID] then layoutBySpell[data.spellID] = {} end
            layoutBySpell[data.spellID][#layoutBySpell[data.spellID] + 1] = data
        end
    end

    for spellID, instances in pairs(layoutBySpell) do
        if #instances == 0 then -- skip empty reused tables
        elseif #instances > 1 then
            table.sort(instances, function(a, b) return a.expiry < b.expiry end)
        end
        local buttons = Totem_FindAllButtons(spellID)
        if #instances > 0 and buttons and #buttons > 0 then
            local count = #instances
            local step = STANDALONE_BADGE_SIZE + STANDALONE_BADGE_BORDER * 2 + STANDALONE_BADGE_GAP
            local totalW = count * (STANDALONE_BADGE_SIZE + STANDALONE_BADGE_BORDER * 2) + (count - 1) * STANDALONE_BADGE_GAP
            local startX = -totalW / 2

            for btnIdx, btn in ipairs(buttons) do
                for i, data in ipairs(instances) do
                    local xOffset = startX + (i - 1) * step
                    if btnIdx == 1 then
                        data.frame:ClearAllPoints()
                        data.frame:SetPoint("BOTTOM", btn, "BOTTOM", xOffset + (STANDALONE_BADGE_SIZE / 2 + STANDALONE_BADGE_BORDER), -2)
                        if data.cooldown and data.startTime and data.duration > 0 then
                            data.cooldown:SetCooldown(data.startTime, data.duration)
                        end
                        data.frame:Show()
                    else
                        local cloneIdx = (btnIdx - 2) * count + i
                        local clone = Standalone_GetClone(data, cloneIdx)
                        clone.frame:ClearAllPoints()
                        clone.frame:SetPoint("BOTTOM", btn, "BOTTOM", xOffset + (STANDALONE_BADGE_SIZE / 2 + STANDALONE_BADGE_BORDER), -2)
                        if clone.cooldown and data.startTime and data.duration > 0 then
                            clone.cooldown:SetCooldown(data.startTime, data.duration)
                        end
                        clone.frame:Show()
                    end
                end
            end
        else
            for _, data in ipairs(instances) do
                data.frame:Hide()
            end
        end
    end
end

-- Hide all clone frames for a slot
local function Standalone_HideClones(data)
    for _, clone in pairs(data.clones) do
        clone.frame:Hide()
        clone.text:SetText("")
        if clone.cooldown then clone.cooldown:Clear() end
    end
end

local function Standalone_HideSlot(data)
    if data.timer then data.timer:Cancel(); data.timer = nil end
    data.active = false
    data.frame:Hide()
    if data.text then data.text:SetText("") end
    if data.cooldown then data.cooldown:Clear() end
    Standalone_HideClones(data)
end

local function Standalone_Show(spellID, startTime, duration)
    local data = Standalone_AcquireSlot(spellID)

    -- Update icon if missing
    if not data.icon:GetTexture() then
        local tex = totemIcons[spellID]
        if tex then data.icon:SetTexture(tex) end
    end

    data.active = true
    data.expiry = startTime + duration
    data.startTime = startTime
    data.duration = duration
    -- SetCooldown is handled by Standalone_Layout() below, no need to call twice

    if data.timer then data.timer:Cancel(); data.timer = nil end

    local remaining = data.expiry - GetTime()
    if remaining > 0 then
        data.timer = C_Timer.NewTimer(remaining, function()
            data.timer = nil
            Standalone_HideSlot(data)
            Standalone_Layout()
        end)
    end

    Standalone_Layout()
    return data
end

local function Standalone_HideAll()
    for _, data in ipairs(standaloneSlots) do
        Standalone_HideSlot(data)
    end
    Standalone_Layout()
end

-- Countdown text ticker (started/stopped by SetTotemTracking)
local standaloneTextTicker = nil

local function Standalone_StartTicker()
    if standaloneTextTicker then return end
    standaloneTextTicker = C_Timer.NewTicker(0.2, function()
        local ok, err = pcall(function()
            local now = GetTime()
            for _, data in ipairs(standaloneSlots) do
                if data.active and data.expiry then
                    local remain = data.expiry - now
                    local txt = remain > 0 and (math.ceil(remain) .. "s") or ""
                    if data.text then data.text:SetText(txt) end
                    for _, clone in pairs(data.clones) do
                        if clone.frame and clone.frame:IsShown() and clone.text then
                            clone.text:SetText(txt)
                        end
                    end
                end
            end
        end)
        if not ok then ns.SK_ReportError("Totem:Ticker", err) end
    end)
end

local function Standalone_StopTicker()
    if standaloneTextTicker then
        standaloneTextTicker:Cancel()
        standaloneTextTicker = nil
    end
end

-- Slot-based totem tracking: maps WoW totem slot → standaloneSlot data
local totemSlotMap = {}       -- [wowSlot] = standaloneSlotData
local pendingCastData = nil   -- set by UNIT_SPELLCAST_SUCCEEDED, consumed by next PLAYER_TOTEM_UPDATE

local function Totem_CacheNames()
    local allGroups = {}
    for _, g in ipairs(TOTEM_COUNTDOWN_GROUPS) do table.insert(allGroups, g) end
    for _, g in ipairs(TOTEM_BAR_GROUPS) do table.insert(allGroups, g) end
    for _, group in ipairs(allGroups) do
        local name = C_Spell.GetSpellName(group.id)
        if name then totemNames[group.id] = name end
        local icon = C_Spell.GetSpellTexture(group.id)
        if icon then totemIcons[group.id] = icon end
    end
end

local function SetTotemTracking(enabled)
    -- Totem tracking is Shaman-only; skip entirely for other classes
    if select(2, UnitClass("player")) ~= "SHAMAN" then return end
    if enabled then
        Totem_RebuildTracked()
        Totem_CacheNames()
        Standalone_StartTicker()

        totemFrame:RegisterEvent("PLAYER_TOTEM_UPDATE")
        totemFrame:RegisterEvent("PLAYER_ENTERING_WORLD")
        totemFrame:RegisterEvent("SPELLS_CHANGED")
        totemFrame:RegisterEvent("TRAIT_CONFIG_UPDATED")
        totemFrame:RegisterEvent("UNIT_SPELLCAST_SUCCEEDED")
        local function TotemEventHandler(event, ...)
            if not SKToolsDB or not SKToolsDB.totemTracking then return end

            if event == "TRAIT_CONFIG_UPDATED" then
                wipe(cachedTotemDuration)
                if SKToolsDB then SKToolsDB.cachedTotemDurations = nil end

            elseif event == "PLAYER_ENTERING_WORLD" or event == "SPELLS_CHANGED" then
                Totem_CacheNames()
                if event == "PLAYER_ENTERING_WORLD" then
                    Standalone_HideAll()
                    wipe(totemSlotMap)
                    pendingCastData = nil
                    local _, instanceType = IsInInstance()
                    local inPvPInstance = (instanceType == "arena" or instanceType == "pvp")
                    if not inPvPInstance then
                        Totem_DetectAllButtons()
                    end
                end

            elseif event == "UNIT_SPELLCAST_SUCCEEDED" then
                local unit, _, spellID = ...
                if unit ~= "player" then return end
                local knownDur = TOTEM_KNOWN_DURATION[spellID]
                if not knownDur then return end
                if not TOTEM_ENABLED[spellID] then return end
                local now = GetTime()
                local displayDur = TOTEM_DURATION_OVERRIDE[spellID] or cachedTotemDuration[spellID] or knownDur
                local data = Standalone_Show(spellID, now, displayDur)
                pendingCastData = data

            elseif event == "PLAYER_TOTEM_UPDATE" then
                local slot = ...

                -- Slot tracking: map cast to slot, or detect death
                local existing = totemSlotMap[slot]
                if pendingCastData then
                    if existing and existing.active and existing ~= pendingCastData then
                        Standalone_HideSlot(existing)
                    end
                    totemSlotMap[slot] = pendingCastData
                    pendingCastData = nil
                    Standalone_Layout()
                elseif existing then
                    if existing.active then
                        Standalone_HideSlot(existing)
                        Standalone_Layout()
                    end
                    totemSlotMap[slot] = nil
                end

                -- Cache durations outside PvP (GetTotemInfo returns clean values there)
                local _, instanceType = IsInInstance()
                if instanceType ~= "arena" and instanceType ~= "pvp" then
                    Totem_CacheDurations()
                end
            end
        end
        totemFrame:SetScript("OnEvent", function(self, event, ...)
            local ok, err = pcall(TotemEventHandler, event, ...)
            if not ok then ns.SK_ReportError("Totem:" .. event, err) end
        end)
    else
        totemFrame:UnregisterAllEvents()
        totemFrame:SetScript("OnEvent", nil)
        Standalone_StopTicker()
        Standalone_HideAll()
        wipe(totemSlotMap)
        pendingCastData = nil
    end
end
ns.SetTotemTracking = SetTotemTracking

-- Restore cached durations from SavedVariables (called during init)
function ns.Totem_RestoreCache()
    if SKToolsDB and SKToolsDB.cachedTotemDurations then
        for k, v in pairs(SKToolsDB.cachedTotemDurations) do
            cachedTotemDuration[k] = v
        end
    end
end


-----------------------------
-- Shaman Settings Builder (main panel)
-----------------------------
function ns.BuildShamanSettings(content, anchor)
    local totemCard = ns.CreateSectionCard(content)
    totemCard:SetPoint("TOPLEFT", anchor, "BOTTOMLEFT", -4, -12)
    local AddCB, GetLast, SetLast = ns.MakeCheckboxFactory(content, anchor)

    AddCB("TotemTracking", "totemTracking", "Totem Duration Tracking",
        "Shows circular badge icons with countdown timers on tracked totem action bar buttons.",
        SetTotemTracking)

    -- Totem spell toggles (two columns with icons)
    do
        local lastEl = GetLast()
        local totemLabel = content:CreateFontString(nil, "ARTWORK", "GameFontNormalSmall")
        totemLabel:SetPoint("TOPLEFT", lastEl, "BOTTOMLEFT", 0, -8)
        totemLabel:SetText("Tracked Totems")
        totemLabel:SetTextColor(0.95, 0.95, 0.95)

        local allTotemGroups = {}
        for _, g in ipairs(TOTEM_COUNTDOWN_GROUPS) do table.insert(allTotemGroups, g) end
        for _, g in ipairs(TOTEM_BAR_GROUPS) do table.insert(allTotemGroups, g) end

        local COL_WIDTH = 220
        local ROW_HEIGHT = 28
        local ICON_SIZE = 16
        local perCol = math.ceil(#allTotemGroups / 2)
        local totemChildFrames = {}

        for idx, group in ipairs(allTotemGroups) do
            local col = (idx <= perCol) and 0 or 1
            local rowInCol = (idx <= perCol) and (idx - 1) or (idx - perCol - 1)
            local yOff = -(rowInCol * ROW_HEIGHT) - 4

            local cb = ns.CreateToggleSwitch(content, "SKToolsTotem_" .. group.key)
            cb:SetPoint("TOPLEFT", totemLabel, "BOTTOMLEFT", col * COL_WIDTH, yOff)
            cb:SetChecked(SKToolsDB.totemTrackSpells[group.key])

            local icon = content:CreateTexture(nil, "ARTWORK")
            icon:SetSize(ICON_SIZE, ICON_SIZE)
            icon:SetPoint("LEFT", cb, "RIGHT", 4, 0)
            icon:SetTexCoord(0.08, 0.92, 0.08, 0.92)
            local spellIcon = C_Spell.GetSpellTexture(group.id)
            if spellIcon then icon:SetTexture(spellIcon) end

            cb.Text:ClearAllPoints()
            cb.Text:SetPoint("LEFT", icon, "RIGHT", 4, 0)
            cb.Text:SetText(group.label)

            cb:SetScript("OnClick", function(self)
                SKToolsDB.totemTrackSpells[group.key] = self:GetChecked()
                Totem_RebuildTracked()
            end)

            table.insert(totemChildFrames, { cb = cb, icon = icon })
        end

        -- Grey out children when parent toggle is disabled
        local function UpdateTotemChildrenState()
            local enabled = SKToolsDB.totemTracking
            local alpha = enabled and 1.0 or 0.4
            totemLabel:SetAlpha(alpha)
            for _, child in ipairs(totemChildFrames) do
                child.cb:SetEnabled(enabled)
                child.icon:SetAlpha(alpha)
            end
        end

        -- Hook into parent checkbox
        local parentCB = SKToolsTotemTrackingCB
        local origOnClick = parentCB:GetScript("OnClick")
        parentCB:SetScript("OnClick", function(self)
            origOnClick(self)
            UpdateTotemChildrenState()
        end)
        UpdateTotemChildrenState()

        local spacer = content:CreateFontString(nil, "ARTWORK")
        spacer:SetPoint("TOPLEFT", totemLabel, "BOTTOMLEFT", 0, -(perCol * ROW_HEIGHT) - 4)
        spacer:SetHeight(1)
        totemCard:SetPoint("BOTTOM", spacer, "BOTTOM", 0, -8)
        totemCard:SetPoint("RIGHT", content, "RIGHT", -8, 0)
        SetLast(spacer)
    end
end


-----------------------------
-- Shaman Combat Settings Builder
-----------------------------
function ns.BuildShamanCombatSettings(content, csSyncControls)
    local shamAnchor = ns.CreateTabTitle(content, "Totem Tracking", "")
    local totemCard = ns.CreateSectionCard(content)
    totemCard:SetPoint("TOPLEFT", shamAnchor, "BOTTOMLEFT", -4, -12)

    local AddCB, GetLast, SetLast = ns.MakeCombatCBFactory(content, shamAnchor, csSyncControls)
    local totemParentCB = AddCB("TotemTracking", "totemTracking", "Totem Duration Tracking",
        "Shows circular badge icons with countdown timers on tracked totem action bar buttons.", SetTotemTracking)

    do
        local lastEl = GetLast()
        local totemLabel = content:CreateFontString(nil, "ARTWORK", "GameFontNormalSmall")
        totemLabel:SetPoint("TOPLEFT", lastEl, "BOTTOMLEFT", 0, -8)
        totemLabel:SetText("Tracked Totems")
        totemLabel:SetTextColor(0.95, 0.95, 0.95)
        local allTotemGroups = {}
        for _, g in ipairs(TOTEM_COUNTDOWN_GROUPS) do table.insert(allTotemGroups, g) end
        for _, g in ipairs(TOTEM_BAR_GROUPS) do table.insert(allTotemGroups, g) end
        local COL_WIDTH, ROW_HT, ICON_SIZE = 220, 28, 16
        local perCol = math.ceil(#allTotemGroups / 2)
        local totemChildFrames = {}
        for idx, group in ipairs(allTotemGroups) do
            local col = (idx <= perCol) and 0 or 1
            local rowInCol = (idx <= perCol) and (idx - 1) or (idx - perCol - 1)
            local yOff = -(rowInCol * ROW_HT) - 4
            local cb = ns.CreateToggleSwitch(content, "SKToolsCS_Totem_" .. group.key)
            cb:SetPoint("TOPLEFT", totemLabel, "BOTTOMLEFT", col * COL_WIDTH, yOff)
            cb:SetChecked(SKToolsDB.totemTrackSpells[group.key])
            local icon = content:CreateTexture(nil, "ARTWORK")
            icon:SetSize(ICON_SIZE, ICON_SIZE)
            icon:SetPoint("LEFT", cb, "RIGHT", 4, 0)
            icon:SetTexCoord(0.08, 0.92, 0.08, 0.92)
            local spellIcon = C_Spell.GetSpellTexture(group.id)
            if spellIcon then icon:SetTexture(spellIcon) end
            cb.Text:ClearAllPoints()
            cb.Text:SetPoint("LEFT", icon, "RIGHT", 4, 0)
            cb.Text:SetText(group.label)
            cb:SetScript("OnClick", function(self)
                SKToolsDB.totemTrackSpells[group.key] = self:GetChecked()
                Totem_RebuildTracked()
            end)
            table.insert(totemChildFrames, { cb = cb, icon = icon, key = group.key })
        end
        local function UpdateCSTotemChildrenState()
            local enabled = SKToolsDB.totemTracking
            local alpha = enabled and 1.0 or 0.4
            totemLabel:SetAlpha(alpha)
            for _, child in ipairs(totemChildFrames) do
                child.cb:SetEnabled(enabled)
                child.icon:SetAlpha(alpha)
            end
        end
        totemParentCB:HookScript("OnClick", UpdateCSTotemChildrenState)
        UpdateCSTotemChildrenState()
        table.insert(csSyncControls, { type = "custom", refresh = function()
            for _, child in ipairs(totemChildFrames) do
                child.cb:SetChecked(SKToolsDB.totemTrackSpells[child.key])
            end
            UpdateCSTotemChildrenState()
        end })

        local spacer = content:CreateFontString(nil, "ARTWORK")
        spacer:SetPoint("TOPLEFT", totemLabel, "BOTTOMLEFT", 0, -(perCol * ROW_HT) - 4)
        spacer:SetHeight(1)
        totemCard:SetPoint("BOTTOM", spacer, "BOTTOM", 0, -8)
        totemCard:SetPoint("RIGHT", content, "RIGHT", -8, 0)
    end
end
