-- SKTools Stat Tracker
-- Secondary stat diminishing returns tracking + tooltips
-- Data sourced from SimC: https://github.com/simulationcraft/simc/blob/midnight/engine/dbc/generated/sc_scale_data.inc

local _, ns = ...

-----------------------------
-- Conversion Tables (SimC)
-----------------------------
local hst_cnv = {
    2.923638162, 2.923638162, 2.923638162, 2.923638162, 2.923638162, -- 5
    2.923638162, 2.923638162, 2.923638162, 2.923638162, 2.923638162, -- 10
    2.923638162, 3.06982007,  3.216001978, 3.362183886, 3.508365794, -- 15
    3.654547702, 3.80072961,  3.946911518, 4.093093426, 4.239275334, -- 20
    4.283081179, 4.328767379, 4.37638382,  4.425982836, 4.477619303, -- 25
    4.531350734, 4.587237394, 4.6453424,   4.705731852, 4.768474943, -- 30
    4.833644101, 4.901315118, 4.971567301, 5.044483622, 5.120150876, -- 35
    5.198659856, 5.280105527, 5.364587216, 5.452208807, 5.543078954, -- 40
    5.637311296, 5.735024692, 5.836343461, 5.941397644, 6.050323267, -- 45
    6.163262635, 6.280364625, 6.401785008, 6.527686779, 6.658240515, -- 50
    6.793624739, 6.934026317, 7.079640869, 7.230673208, 7.387337794, -- 55
    7.549859225, 7.718472748, 7.893424797, 8.074973567, 8.263389617, -- 60
    8.458956505, 8.661971461, 8.8727461,   9.09160717,  9.31889735,  -- 65
    9.554976083, 9.800220469, 10.0550262,  10.31980856, 10.59500345, -- 70
    10.69418334, 10.79429165, 10.89533707, 10.99732838, 11.10027443, -- 75
    11.20418416, 11.30906659, 11.41493083, 11.52178606, 11.62964156, -- 80
    12.99342144, 14.51712849, 16.21951696, 18.12154041, 20.24660954, -- 85
    22.62088037, 25.27357619, 28.23734722, 31.5486725,  44,          -- 90
}

local crt_cnv = {
    3.056530805, 3.056530805, 3.056530805, 3.056530805, 3.056530805, -- 5
    3.056530805, 3.056530805, 3.056530805, 3.056530805, 3.056530805, -- 10
    3.056530805, 3.209357346, 3.362183886, 3.515010426, 3.667836966, -- 15
    3.820663507, 3.973490047, 4.126316587, 4.279143127, 4.431969668, -- 20
    4.477766688, 4.525529532, 4.575310357, 4.627163874, 4.681147453, -- 25
    4.737321222, 4.795748184, 4.856494328, 4.919628754, 4.985223804, -- 30
    5.053355196, 5.124102169, 5.197547633, 5.273778332, 5.352885007, -- 35
    5.434962577, 5.520110324, 5.608432089, 5.70003648,  5.795037088, -- 40
    5.893552718, 5.995707632, 6.1016318,   6.211461173, 6.325337961, -- 45
    6.443410936, 6.565835744, 6.692775235, 6.824399815, 6.960887811, -- 50
    7.102425863, 7.249209331, 7.401442727, 7.559340172, 7.723125876, -- 55
    7.893034645, 8.069312419, 8.252216833, 8.442017821, 8.638998236, -- 60
    8.843454528, 9.055697437, 9.276052741, 9.504862042, 9.742483593, -- 65
    9.989293177, 10.24568504, 10.51207285, 10.78889076, 11.07659452, -- 70
    11.18028258, 11.28494127, 11.39057967, 11.49720694, 11.60483236, -- 75
    11.71346526, 11.82311507, 11.93379132, 12.04550361, 12.15826163, -- 80
    13.58403151, 15.17699796, 16.95676773, 18.94524679, 21.16690997, -- 85
    23.6491022,  26.42237511, 29.520863,   32.98270306, 46,          -- 90
}

local mst_cnv = {
    3.056530805, 3.056530805, 3.056530805, 3.056530805, 3.056530805, -- 5
    3.056530805, 3.056530805, 3.056530805, 3.056530805, 3.056530805, -- 10
    3.056530805, 3.209357346, 3.362183886, 3.515010426, 3.667836966, -- 15
    3.820663507, 3.973490047, 4.126316587, 4.279143127, 4.431969668, -- 20
    4.477766688, 4.525529532, 4.575310357, 4.627163874, 4.681147453, -- 25
    4.737321222, 4.795748184, 4.856494328, 4.919628754, 4.985223804, -- 30
    5.053355196, 5.124102169, 5.197547633, 5.273778332, 5.352885007, -- 35
    5.434962577, 5.520110324, 5.608432089, 5.70003648,  5.795037088, -- 40
    5.893552718, 5.995707632, 6.1016318,   6.211461173, 6.325337961, -- 45
    6.443410936, 6.565835744, 6.692775235, 6.824399815, 6.960887811, -- 50
    7.102425863, 7.249209331, 7.401442727, 7.559340172, 7.723125876, -- 55
    7.893034645, 8.069312419, 8.252216833, 8.442017821, 8.638998236, -- 60
    8.843454528, 9.055697437, 9.276052741, 9.504862042, 9.742483593, -- 65
    9.989293177, 10.24568504, 10.51207285, 10.78889076, 11.07659452, -- 70
    11.18028258, 11.28494127, 11.39057967, 11.49720694, 11.60483236, -- 75
    11.71346526, 11.82311507, 11.93379132, 12.04550361, 12.15826163, -- 80
    13.58403151, 15.17699796, 16.95676773, 18.94524679, 21.16690997, -- 85
    23.6491022,  26.42237511, 29.520863,   32.98270306, 46,          -- 90
}

local vrs_cnv = {
    3.58810138,  3.58810138,  3.58810138,  3.58810138,  3.58810138,  -- 5
    3.58810138,  3.58810138,  3.58810138,  3.58810138,  3.58810138,  -- 10
    3.58810138,  3.767506449, 3.946911518, 4.126316587, 4.305721656, -- 15
    4.485126725, 4.664531794, 4.843936863, 5.023341932, 5.202747001, -- 20
    5.25650872,  5.312578146, 5.371016506, 5.431888026, 5.495260053, -- 25
    5.561203174, 5.629791347, 5.701102037, 5.775216363, 5.852219248, -- 30
    5.932199578, 6.015250372, 6.101468961, 6.190957172, 6.28382153,  -- 35
    6.38017346,  6.480129511, 6.583811583, 6.691347172, 6.802869625, -- 40
    6.918518409, 7.038439394, 7.162785157, 7.29171529,  7.425396737, -- 45
    7.564004143, 7.707720221, 7.856736146, 8.011251956, 8.171476996, -- 50
    8.337630361, 8.509941389, 8.688650158, 8.874008028, 9.066278202, -- 55
    9.265736322, 9.4726711,   9.687384978, 9.910194833, 10.14143271, -- 60
    10.38144662, 10.63060134, 10.8892793,  11.15788153, 11.43682857, -- 65
    11.72656156, 12.0275433,  12.34025943, 12.66521959, 13.00295878, -- 70
    13.12467955, 13.24753975, 13.37155004, 13.4967212,  13.62306408, -- 75
    13.75058965, 13.879309,   14.00923329, 14.1403738,  14.27274192, -- 80
    15.94647177, 17.81647587, 19.90577082, 22.24007232, 24.8481117,  -- 85
    27.76198954, 31.01757078, 34.65492613, 38.71882534, 54,          -- 90
}

local lee_cnv = {
    4.584861654, 4.584861654, 4.584861654, 4.584861654, 4.584861654, -- 5
    4.584861654, 4.584861654, 4.584861654, 4.584861654, 4.584861654, -- 10
    4.584861654, 4.814104737, 5.04334782,  5.272590902, 5.501833985, -- 15
    5.731077068, 5.96032015,  6.189563233, 6.418806316, 6.648049399, -- 20
    6.716745909, 6.788391199, 6.863063502, 6.940844888, 7.021821412, -- 25
    7.106083269, 7.193724963, 7.284845479, 7.37954847,  7.47794245,  -- 30
    7.580140996, 7.68626297,  7.79643274,  7.91078042,  8.029442126, -- 35
    8.152560239, 8.280283682, 8.412768221, 8.550176769, 8.692679715, -- 40
    8.84045527,  8.993689828, 9.152578349, 9.317324759, 9.488142379, -- 45
    9.665254371, 9.848894204, 10.03930616, 10.23674585, 10.44148076, -- 50
    10.65379087, 10.87396922, 11.10232257, 11.33917212, 11.58485418, -- 55
    11.83972097, 12.10414141, 12.37850195, 12.66320749, 12.95868233, -- 60
    13.26537115, 13.58374006, 13.91427773, 14.25749658, 14.613934,   -- 65
    14.98415366, 15.36874693, 15.76833435, 16.18356716, 16.61512895, -- 70
    16.77066326, 16.92765353, 17.08611339, 17.24605659, 17.40749702, -- 75
    17.5704487,  17.73492577, 17.90094251, 18.06851333, 18.23765278, -- 80
    20.37633812, 22.76582191, 25.43551468, 28.41827584, 31.75081818, -- 85
    35.47415968, 39.63412842, 44.2819266,  49.47476082, 69.00098495, -- 90
}

local avd_cnv = {
    2.445259549, 2.445259549, 2.445259549, 2.445259549, 2.445259549, -- 5
    2.445259549, 2.445259549, 2.445259549, 2.445259549, 2.445259549, -- 10
    2.445259549, 2.567522526, 2.689785504, 2.812048481, 2.934311459, -- 15
    3.056574436, 3.178837414, 3.301100391, 3.423363368, 3.545626346, -- 20
    3.582264485, 3.620475306, 3.660300534, 3.70178394,  3.74497142,  -- 25
    3.789911077, 3.836653313, 3.885250922, 3.935759184, 3.988235973, -- 30
    4.042741865, 4.099340251, 4.158097461, 4.219082891, 4.282369134, -- 35
    4.348032127, 4.416151297, 4.486809718, 4.560094277, 4.636095848, -- 40
    4.714909477, 4.796634575, 4.881375119, 4.969239871, 5.060342602, -- 45
    5.154802331, 5.252743575, 5.354296618, 5.459597785, 5.56878974,  -- 50
    5.682021798, 5.799450249, 5.921238704, 6.047558463, 6.178588896, -- 55
    6.314517852, 6.455542084, 6.601867705, 6.753710662, 6.911297244, -- 60
    7.074864612, 7.244661363, 7.420948123, 7.603998176, 7.794098131, -- 65
    7.991548617, 8.196665031, 8.409778322, 8.631235818, 8.861402106, -- 70
    8.94435374,  9.028081885, 9.112593809, 9.19789685,  9.283998413, -- 75
    9.370905974, 9.458627076, 9.547169336, 9.636540441, 9.726748149, -- 80
    10.86738033, 12.14177169, 13.56560783, 15.15641378, 16.9337697,  -- 85
    18.91955183, 21.13820182, 23.61702752, 26.3865391,  36.80052531, -- 90
}

local spd_cnv = {
    0.764143609, 0.764143609, 0.764143609, 0.764143609, 0.764143609, -- 5
    0.764143609, 0.764143609, 0.764143609, 0.764143609, 0.764143609, -- 10
    0.764143609, 0.802350789, 0.84055797,  0.87876515,  0.916972331, -- 15
    0.955179511, 0.993386692, 1.031593872, 1.069801053, 1.108008233, -- 20
    1.119457652, 1.131398533, 1.143843917, 1.156807481, 1.170303569, -- 25
    1.184347211, 1.19895416,  1.214140913, 1.229924745, 1.246323742, -- 30
    1.263356833, 1.281043828, 1.299405457, 1.318463403, 1.338240354, -- 35
    1.35876004,  1.38004728,  1.402128037, 1.425029461, 1.448779953, -- 40
    1.473409212, 1.498948305, 1.525429725, 1.55288746,  1.581357063, -- 45
    1.610875728, 1.641482367, 1.673217693, 1.706124308, 1.740246794, -- 50
    1.775631812, 1.812328203, 1.850387095, 1.88986202,  1.93080903,  -- 55
    1.973286829, 2.017356901, 2.063083658, 2.110534582, 2.159780389, -- 60
    2.210895191, 2.263956676, 2.319046288, 2.37624943,  2.435655666, -- 65
    2.497358943, 2.561457822, 2.628055726, 2.697261193, 2.769188158, -- 70
    2.795110544, 2.821275589, 2.847685565, 2.874342766, 2.901249504, -- 75
    2.928408117, 2.955820961, 2.983490418, 3.011418888, 3.039608797, -- 80
    3.396056354, 3.794303652, 4.239252446, 4.736379307, 5.29180303,  -- 85
    5.912359947, 6.605688069, 7.3803211,   8.24579347,  11.50016416, -- 90
}

-----------------------------
-- DR Bracket Tables
-----------------------------
local STAT_BRACKETS = {
    { size = 30,     penalty = 0 },
    { size = 10,     penalty = 0.1 },
    { size = 10,     penalty = 0.2 },
    { size = 10,     penalty = 0.3 },
    { size = 20,     penalty = 0.4 },
    { size = 120,    penalty = 0.5 },
    { size = 100000, penalty = 1.0 },
}

local TERT_BRACKETS = {
    { size = 10,     penalty = 0 },
    { size = 5,      penalty = 0.2 },
    { size = 5,      penalty = 0.4 },
    { size = 80,     penalty = 0.6 },
    { size = 100000, penalty = 1.0 },
}

-----------------------------
-- Stat ID Mapping
-----------------------------
local STAT_MAP = {
    [CR_CRIT_SPELL]              = { conv = crt_cnv, brackets = STAT_BRACKETS, label = STAT_CRITICAL_STRIKE },
    [CR_HASTE_SPELL]             = { conv = hst_cnv, brackets = STAT_BRACKETS, label = STAT_HASTE },
    [CR_MASTERY]                 = { conv = mst_cnv, brackets = STAT_BRACKETS, label = STAT_MASTERY },
    [CR_VERSATILITY_DAMAGE_DONE] = { conv = vrs_cnv, brackets = STAT_BRACKETS, label = STAT_VERSATILITY },
    [CR_LIFESTEAL]               = { conv = lee_cnv, brackets = TERT_BRACKETS, label = STAT_LIFESTEAL },
    [CR_AVOIDANCE]               = { conv = avd_cnv, brackets = TERT_BRACKETS, label = STAT_AVOIDANCE },
    [CR_SPEED]                   = { conv = spd_cnv, brackets = TERT_BRACKETS, label = STAT_SPEED },
}

-- Main 4 stats for the dashboard
local MAIN_STATS = {
    { id = CR_CRIT_SPELL,              label = "Critical Strike", color = {1.0, 0.82, 0.20} },
    { id = CR_HASTE_SPELL,             label = "Haste",           color = {0.40, 0.85, 0.40} },
    { id = CR_MASTERY,                 label = "Mastery",         color = {0.85, 0.55, 0.20} },
    { id = CR_VERSATILITY_DAMAGE_DONE, label = "Versatility",     color = {0.50, 0.70, 1.00} },
}

-----------------------------
-- State
-----------------------------
local convFactors = {}   -- [statId] = conversion factor for current level
local baseRatings = {}   -- [statId] = raw combat rating
local statInfo = {}      -- [statId] = { effectivePct, rawPct, bracketPenalty, nextBracketPenalty, ratingUntilNext, bracketRating, bracketMaxRating, baseRating, convFactor }
local tooltipHooked = false
local statFrame = nil

-----------------------------
-- Core Math
-----------------------------
local function SetupConversionFactors()
    local level = math.max(UnitLevel("player"), 1)
    for statId, data in pairs(STAT_MAP) do
        convFactors[statId] = data.conv[level] or data.conv[90]
    end
end

local function GetStatDiminishBracket(statId, addedRating)
    local conv = convFactors[statId]
    if not conv or conv == 0 then return 0, 0, 0, 0, 0, 0 end

    local rating = (baseRatings[statId] or 0) + (addedRating or 0)
    local percent = rating / conv
    local brackets = STAT_MAP[statId].brackets
    local trueRating = 0
    local bracketPenalty = 0
    local bracketRating = 0
    local bracketMaxRating = 0
    local nextBracketPenalty = 0
    local ratingUntilNext = 0

    for i, bracket in ipairs(brackets) do
        if percent < bracket.size then
            bracketRating = math.floor(0.5 + percent * conv)
            bracketMaxRating = math.floor(0.5 + bracket.size * conv)
            bracketPenalty = bracket.penalty
            ratingUntilNext = bracketMaxRating - bracketRating
            local nextBracket = brackets[i + 1]
            nextBracketPenalty = nextBracket and nextBracket.penalty or bracket.penalty
            trueRating = trueRating + (percent * conv * (1.0 - bracket.penalty))
            break
        else
            trueRating = trueRating + (bracket.size * conv * (1.0 - bracket.penalty))
        end
        percent = percent - bracket.size
    end

    trueRating = math.floor(0.005 + 100 * trueRating) / 100
    return trueRating, bracketPenalty, bracketRating, bracketMaxRating, nextBracketPenalty, ratingUntilNext
end

local function RecalculateStats()
    baseRatings[CR_CRIT_SPELL]              = GetCombatRating(CR_CRIT_SPELL)
    baseRatings[CR_HASTE_SPELL]             = GetCombatRating(CR_HASTE_SPELL)
    baseRatings[CR_MASTERY]                 = GetCombatRating(CR_MASTERY)
    baseRatings[CR_VERSATILITY_DAMAGE_DONE] = GetCombatRating(CR_VERSATILITY_DAMAGE_DONE)
    baseRatings[CR_LIFESTEAL]               = GetCombatRating(CR_LIFESTEAL)
    baseRatings[CR_AVOIDANCE]               = GetCombatRating(CR_AVOIDANCE)
    baseRatings[CR_SPEED]                   = GetCombatRating(CR_SPEED)

    -- Get mastery spec coefficient (second return of GetMasteryEffect)
    local _, masteryCoeff = GetMasteryEffect()
    masteryCoeff = masteryCoeff or 1

    for statId in pairs(STAT_MAP) do
        local _, bracketPenalty, bracketRating, bracketMaxRating, nextBracketPenalty, ratingUntilNext = GetStatDiminishBracket(statId)
        -- Use game API for authoritative after-DR percentage
        local effectivePct = GetCombatRatingBonus(statId)
        local conv = convFactors[statId] or 1
        local rawPct = (baseRatings[statId] or 0) / conv

        -- Rating per 1% (base and at current DR bracket)
        local drMult = 1 - bracketPenalty
        local basePerOnePct, ratingPerOnePct
        if statId == CR_MASTERY then
            basePerOnePct = conv / masteryCoeff
            ratingPerOnePct = (drMult > 0) and (basePerOnePct / drMult) or 0
        else
            basePerOnePct = conv
            ratingPerOnePct = (drMult > 0) and (conv / drMult) or 0
        end

        statInfo[statId] = {
            effectivePct = effectivePct,
            rawPct = rawPct,
            bracketPenalty = bracketPenalty,
            nextBracketPenalty = nextBracketPenalty,
            ratingUntilNext = ratingUntilNext,
            bracketRating = bracketRating,
            bracketMaxRating = bracketMaxRating,
            baseRating = baseRatings[statId] or 0,
            convFactor = conv,
            basePerOnePct = basePerOnePct,
            ratingPerOnePct = ratingPerOnePct,
        }
    end
end

local function GetTrueStatAdded(statId, amount)
    -- Use bracket math for both sides so the diff is consistent (no API mismatch)
    local currentEffective = GetStatDiminishBracket(statId, 0)
    local addedEffective = GetStatDiminishBracket(statId, amount)
    return math.floor(addedEffective - currentEffective + 0.5)
end

-----------------------------
-- Tooltip: Character Panel Stat Tooltips
-----------------------------
local CYAN = ns.CYAN

-- Per-stat colors matching the dashboard
local STAT_COLORS = {
    [CR_CRIT_SPELL]              = { 1.0, 0.82, 0.20 },
    [CR_HASTE_SPELL]             = { 0.40, 0.85, 0.40 },
    [CR_MASTERY]                 = { 0.85, 0.55, 0.20 },
    [CR_VERSATILITY_DAMAGE_DONE] = { 0.50, 0.70, 1.00 },
    [CR_LIFESTEAL]               = { 0.70, 0.90, 0.70 },
    [CR_AVOIDANCE]               = { 0.60, 0.60, 0.80 },
    [CR_SPEED]                   = { 0.80, 0.80, 0.50 },
}

-- Map stat tooltip header text to stat IDs
local statLabelMap = {}
local masteryName = nil

local function BuildStatLabelMap()
    statLabelMap = {}
    -- Standard stats use PAPERDOLLFRAME_TOOLTIP_FORMAT
    statLabelMap[CR_CRIT_SPELL] = format(PAPERDOLLFRAME_TOOLTIP_FORMAT, STAT_CRITICAL_STRIKE)
    statLabelMap[CR_HASTE_SPELL] = format(PAPERDOLLFRAME_TOOLTIP_FORMAT, STAT_HASTE)
    statLabelMap[CR_VERSATILITY_DAMAGE_DONE] = format(PAPERDOLLFRAME_TOOLTIP_FORMAT, STAT_VERSATILITY)
    statLabelMap[CR_LIFESTEAL] = format(PAPERDOLLFRAME_TOOLTIP_FORMAT, STAT_LIFESTEAL)
    statLabelMap[CR_AVOIDANCE] = format(PAPERDOLLFRAME_TOOLTIP_FORMAT, STAT_AVOIDANCE)
    statLabelMap[CR_SPEED] = format(PAPERDOLLFRAME_TOOLTIP_FORMAT, STAT_SPEED)
    -- Mastery uses spell name
    local spec = GetSpecialization()
    if spec then
        local masterySpell = GetSpecializationMasterySpells(spec)
        if masterySpell then
            masteryName = C_Spell.GetSpellName(masterySpell)
        end
    end
end

local function AddStatTooltipInfo(tooltip, statId)
    local info = statInfo[statId]
    if not info then return end

    local sc = STAT_COLORS[statId] or { CYAN.r, CYAN.g, CYAN.b }
    local r, g, b = sc[1], sc[2], sc[3]
    local hexColor = string.format("%02x%02x%02x", r * 255, g * 255, b * 255)

    -- Effective rating derived from game API (GetCombatRatingBonus * convFactor)
    local effectiveRating = math.floor(info.effectivePct * info.convFactor)
    local lostRating = info.baseRating - effectiveRating
    local lostPct = info.rawPct - info.effectivePct

    tooltip:AddLine(" ")

    -- Effective Rating
    tooltip:AddDoubleLine(
        "|cff" .. hexColor .. "Effective Rating|r",
        string.format("|cff%s%s|r", hexColor, BreakUpLargeNumbers(effectiveRating)),
        nil, nil, nil,
        nil, nil, nil)

    -- Diminishing Returns (always shown)
    if info.bracketPenalty > 0 then
        tooltip:AddDoubleLine(
            "|cffffffffDiminishing Returns|r",
            string.format("|cffff6666-%d%%|r", info.bracketPenalty * 100),
            nil, nil, nil,
            nil, nil, nil)
    else
        tooltip:AddDoubleLine(
            "|cffffffffDiminishing Returns|r",
            "|cffffffffNone|r",
            nil, nil, nil,
            nil, nil, nil)
    end

    -- Lost to DR (rating + percentage)
    if lostRating > 0 then
        tooltip:AddDoubleLine(
            "|cffff6666Lost to DR|r",
            string.format("|cffff6666%s rating  (%.2f%%)|r", BreakUpLargeNumbers(lostRating), lostPct),
            nil, nil, nil,
            nil, nil, nil)
    end

    -- How far until DR gets worse
    if info.nextBracketPenalty > info.bracketPenalty and info.nextBracketPenalty < 1.0 then
        tooltip:AddDoubleLine(
            "|cffaaaaaa" .. BreakUpLargeNumbers(info.ratingUntilNext) .. " rating until|r",
            string.format("|cffaaaaaa-%d%% DR|r", info.nextBracketPenalty * 100),
            nil, nil, nil,
            nil, nil, nil)
    end

    -- Rating per 1%
    if info.ratingPerOnePct and info.ratingPerOnePct > 0 then
        local perPctStr
        if info.bracketPenalty > 0 then
            perPctStr = string.format("|cffaaaaaa%.1f  (%.1f base)|r", info.ratingPerOnePct, info.basePerOnePct)
        else
            perPctStr = string.format("|cffaaaaaa%.1f|r", info.ratingPerOnePct)
        end
        tooltip:AddDoubleLine(
            "|cffaaaaaaRating per 1%|r",
            perPctStr,
            nil, nil, nil,
            nil, nil, nil)
    end

    -- Progress bar showing bracket position (X / Y)
    local barLabel = string.format("%s / %s",
        BreakUpLargeNumbers(info.bracketRating),
        BreakUpLargeNumbers(info.bracketMaxRating))
    GameTooltip_ShowProgressBar(tooltip, 0, info.bracketMaxRating, info.bracketRating, barLabel)
    local frames = tooltip.insertedFrames
    if frames then
        local insertedFrame = frames[#frames]
        if insertedFrame and insertedFrame.Bar then
            insertedFrame.Bar:SetStatusBarColor(r, g, b, 1)
        end
    end

    tooltip:Show()
end

local blackList = {
    ["Leeching Poison"] = true,
    ["Thief's Versatility"] = true,
    ["Voodoo Mastery"] = true,
}

local function OnGameTooltipShow(tooltip)
    if not SKToolsDB or not SKToolsDB.statTooltips then return end

    local tt1 = GameTooltipTextLeft1
    if not tt1 then return end
    local text = tt1:GetText()
    if not text or issecretvalue(text) or blackList[text] then return end

    -- Skip mastery on OnShow — it rebuilds as spell tooltip causing a flash.
    -- OnTooltipSetSpell handles mastery instead.
    local masteryLabel = format(PAPERDOLLFRAME_TOOLTIP_FORMAT, STAT_MASTERY)
    if masteryLabel then
        local escaped = masteryLabel:gsub("%-", "%%-")
        if text:find(escaped) then return end
    end
    if masteryName and text:find(masteryName) then return end

    -- Match other stats
    for statId, label in pairs(statLabelMap) do
        if label then
            local escaped = label:gsub("%-", "%%-")
            local s = text:find(escaped)
            if s and s <= 11 then
                AddStatTooltipInfo(tooltip, statId)
                return
            end
        end
    end
end

local function OnTooltipSetSpell(tooltip)
    if tooltip ~= GameTooltip then return end
    if not SKToolsDB or not SKToolsDB.statTooltips then return end
    if not masteryName then return end

    local tt1 = GameTooltipTextLeft1
    if not tt1 then return end
    local text = tt1:GetText()
    if not text or issecretvalue(text) then return end

    if text:find(masteryName) then
        AddStatTooltipInfo(tooltip, CR_MASTERY)
    end
end

-----------------------------
-- Tooltip: Item Tooltips
-----------------------------
local itemPatterns = {}

local deltaPatterns = {}

local function BuildItemPatterns()
    itemPatterns = {
        [CR_CRIT_SPELL]              = "%+([,0-9]+) " .. STAT_CRITICAL_STRIKE,
        [CR_HASTE_SPELL]             = "%+([,0-9]+) " .. STAT_HASTE,
        [CR_MASTERY]                 = "%+([,0-9]+) " .. STAT_MASTERY,
        [CR_VERSATILITY_DAMAGE_DONE] = "%+([,0-9]+) " .. STAT_VERSATILITY,
        [CR_LIFESTEAL]               = "%+([,0-9]+) " .. STAT_LIFESTEAL,
        [CR_AVOIDANCE]               = "%+([,0-9]+) " .. STAT_AVOIDANCE,
        [CR_SPEED]                   = "%+([,0-9]+) " .. STAT_SPEED,
    }
    -- Comparison delta patterns: handles color codes like "|cff00ff00+20 Critical Strike|r"
    deltaPatterns = {
        [CR_CRIT_SPELL]              = "([+-]%d+) " .. STAT_CRITICAL_STRIKE,
        [CR_HASTE_SPELL]             = "([+-]%d+) " .. STAT_HASTE,
        [CR_MASTERY]                 = "([+-]%d+) " .. STAT_MASTERY,
        [CR_VERSATILITY_DAMAGE_DONE] = "([+-]%d+) " .. STAT_VERSATILITY,
        [CR_LIFESTEAL]               = "([+-]%d+) " .. STAT_LIFESTEAL,
        [CR_AVOIDANCE]               = "([+-]%d+) " .. STAT_AVOIDANCE,
        [CR_SPEED]                   = "([+-]%d+) " .. STAT_SPEED,
    }
end

local function GetDeltaPercent(statId, deltaRating)
    local conv = convFactors[statId]
    if not conv or conv == 0 then return 0 end

    -- For mastery, apply the spec coefficient
    local coeff = 1
    if statId == CR_MASTERY then
        local _, masteryCoeff = GetMasteryEffect()
        coeff = masteryCoeff or 1
    end

    if deltaRating > 0 then
        local trueAdded = GetTrueStatAdded(statId, deltaRating)
        return (trueAdded / conv) * coeff
    else
        -- Losing stats: calculate effective loss
        local trueLost = GetTrueStatAdded(statId, deltaRating)
        return (trueLost / conv) * coeff
    end
end

local function IsCharacterSlot(tooltip)
    local owner = tooltip:GetOwner()
    if not owner then return false end
    local ownerName = owner.GetName and owner:GetName() or ""
    return ownerName:match("^Character") and ownerName:match("Slot") and true or false
end

local function StripColorCodes(text)
    return text:gsub("|c%x%x%x%x%x%x%x%x", ""):gsub("|r", "")
end

-----------------------------
-- Companion Frame for Comparison Deltas
-----------------------------
local deltaFrame = nil
local MAX_DELTA_ROWS = 7

local function GetDeltaFrame()
    if deltaFrame then return deltaFrame end

    local CC = ns.COLORS
    local f = CreateFrame("Frame", "SKToolsStatDeltaFrame", UIParent, "BackdropTemplate")
    f:SetFrameStrata("TOOLTIP")
    f:SetBackdrop(ns.BACKDROP_PANEL)
    f:SetBackdropColor(CC.bgFrame[1], CC.bgFrame[2], CC.bgFrame[3], 0.95)
    f:SetBackdropBorderColor(CC.border[1], CC.border[2], CC.border[3], 0.80)
    f:Hide()

    local title = f:CreateFontString(nil, "ARTWORK", "GameFontNormalSmall")
    title:SetPoint("TOPLEFT", 8, -6)
    title:SetText("|cff00E5EEStat Changes|r")
    f.title = title

    f.rows = {}
    for i = 1, MAX_DELTA_ROWS do
        local label = f:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        local value = f:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        value:SetPoint("RIGHT", f, "RIGHT", -8, 0)
        label:Hide()
        value:Hide()
        f.rows[i] = { label = label, value = value }
    end

    deltaFrame = f
    return f
end

local function ScanTooltipForDeltas(prefix)
    local deltas = {}
    local inComparison = false

    for i = 1, 40 do
        local textLeft = _G[prefix .. "TextLeft" .. tostring(i)]
        if not textLeft or not textLeft.GetText then break end
        local text = textLeft:GetText()
        if not text or text == "" or issecretvalue(text) then
            -- skip
        else
            local clean = StripColorCodes(text)
            if clean:find("the following stat changes") then
                inComparison = true
            elseif inComparison then
                for statId, pattern in pairs(deltaPatterns) do
                    local delta = string.match(clean, pattern)
                    if delta then
                        local deltaNum = tonumber(delta)
                        if deltaNum and deltaNum ~= 0 then
                            local pct = GetDeltaPercent(statId, deltaNum)
                            local sc = STAT_COLORS[statId] or { CYAN.r, CYAN.g, CYAN.b }
                            local statLabel = STAT_MAP[statId] and STAT_MAP[statId].label or "?"
                            table.insert(deltas, {
                                label = statLabel,
                                delta = deltaNum,
                                pct = pct,
                                color = sc,
                            })
                        end
                        break
                    end
                end
            end
        end
    end

    return deltas
end

local function ShowDeltaFrame()
    local f = GetDeltaFrame()

    -- Scan all tooltip frames for comparison section
    local anchorTooltip = GameTooltip
    local deltas = ScanTooltipForDeltas("GameTooltip")
    if #deltas == 0 and ShoppingTooltip1 and ShoppingTooltip1:IsShown() then
        deltas = ScanTooltipForDeltas("ShoppingTooltip1")
        if #deltas > 0 then anchorTooltip = ShoppingTooltip1 end
    end
    if #deltas == 0 and ShoppingTooltip2 and ShoppingTooltip2:IsShown() then
        deltas = ScanTooltipForDeltas("ShoppingTooltip2")
        if #deltas > 0 then anchorTooltip = ShoppingTooltip2 end
    end

    if #deltas == 0 then
        f:Hide()
        return
    end

    -- Position rows
    local anchor = f.title
    local rowCount = math.min(#deltas, MAX_DELTA_ROWS)
    for i = 1, MAX_DELTA_ROWS do
        local row = f.rows[i]
        if i <= rowCount then
            local d = deltas[i]
            local hexStr = string.format("%02x%02x%02x", d.color[1] * 255, d.color[2] * 255, d.color[3] * 255)
            local sign = d.delta >= 0 and "+" or ""
            row.label:ClearAllPoints()
            row.label:SetPoint("TOPLEFT", anchor, i == 1 and "BOTTOMLEFT" or "BOTTOMLEFT", 0, i == 1 and -6 or -2)
            row.label:SetText(string.format("|cff%s%s%d %s|r", hexStr, sign, d.delta, d.label))
            row.label:Show()

            local pctSign = d.pct >= 0 and "+" or ""
            row.value:ClearAllPoints()
            row.value:SetPoint("RIGHT", f, "RIGHT", -8, 0)
            row.value:SetPoint("TOP", row.label, "TOP")
            row.value:SetText(string.format("|cff%s%s%.2f%%|r", hexStr, pctSign, d.pct))
            row.value:Show()

            anchor = row.label
        else
            row.label:Hide()
            row.value:Hide()
        end
    end

    -- Size frame to fit
    local h = 6 + 12 + 6 + (rowCount * 14) + 6
    f:SetSize(220, h)

    -- Anchor to bottom of GameTooltip
    f:ClearAllPoints()
    f:SetPoint("TOPLEFT", anchorTooltip, "BOTTOMLEFT", 0, -2)
    f:Show()
end

local function OnTooltipSetItem(tooltip)
    if tooltip ~= GameTooltip then return end
    if not SKToolsDB or not SKToolsDB.statItemTooltips then return end

    local name, itemLink = tooltip:GetItem()
    if not name or not itemLink then return end

    -- Skip items shown from character panel equipment slots
    if IsCharacterSlot(tooltip) then return end

    -- Annotate item stat lines with DR info
    for i = 1, 40 do
        local textLeft = _G["GameTooltipTextLeft" .. tostring(i)]
        if textLeft and textLeft.GetText then
            local text = textLeft:GetText()
            if text and type(text) == "string" and not issecretvalue(text) and text ~= "" then
                for statId, pattern in pairs(itemPatterns) do
                    local amount = string.match(text, pattern)
                    if amount then
                        local _, e = string.find(text, pattern)
                        local amountNum = tonumber((amount:gsub(",", "")))
                        if amountNum and amountNum > 0 then
                            local trueAmount = GetTrueStatAdded(statId, amountNum)
                            if trueAmount < amountNum then
                                local sc = STAT_COLORS[statId] or { CYAN.r, CYAN.g, CYAN.b }
                                local hexStr = string.format("%02x%02x%02x", sc[1] * 255, sc[2] * 255, sc[3] * 255)
                                local trueText = text:sub(1, e)
                                    .. " |cff" .. hexStr .. "(" .. trueAmount .. ")|r"
                                    .. text:sub(e + 1)
                                textLeft:SetText(trueText)
                            end
                        end
                        break
                    end
                end
            end
        end
    end

    -- Show companion frame with comparison delta percentages (delayed — comparison section
    -- is added by Blizzard's TooltipComparisonManager after our handler fires)
    if SKToolsDB.statCompareTooltips then
        C_Timer.After(0, ShowDeltaFrame)
    end
end

-----------------------------
-- Tooltip Hook Setup
-----------------------------
local function HookTooltips()
    if tooltipHooked then return end
    tooltipHooked = true

    C_Timer.After(0.2, function()
        GameTooltip:HookScript("OnShow", OnGameTooltipShow)
        GameTooltip:HookScript("OnHide", function()
            if deltaFrame then deltaFrame:Hide() end
        end)
    end)
    TooltipDataProcessor.AddTooltipPostCall(Enum.TooltipDataType.Item, OnTooltipSetItem)
    TooltipDataProcessor.AddTooltipPostCall(Enum.TooltipDataType.Spell, OnTooltipSetSpell)
end

-----------------------------
-- Event Frame
-----------------------------
local function OnEvent(self, event)
    if event == "PLAYER_ENTERING_WORLD" then
        SetupConversionFactors()
        RecalculateStats()
        BuildStatLabelMap()
        BuildItemPatterns()
    elseif event == "COMBAT_RATING_UPDATE" then
        RecalculateStats()
    elseif event == "PLAYER_LEVEL_UP" then
        SetupConversionFactors()
        RecalculateStats()
    elseif event == "PLAYER_SPECIALIZATION_CHANGED" then
        BuildStatLabelMap()
        RecalculateStats()
    end
end

function ns.SetStatTracking(enabled)
    if enabled then
        if not statFrame then
            statFrame = CreateFrame("Frame")
            statFrame:SetScript("OnEvent", OnEvent)
        end
        statFrame:RegisterEvent("PLAYER_ENTERING_WORLD")
        statFrame:RegisterEvent("COMBAT_RATING_UPDATE")
        statFrame:RegisterEvent("PLAYER_LEVEL_UP")
        statFrame:RegisterEvent("PLAYER_SPECIALIZATION_CHANGED")
        HookTooltips()
    else
        if statFrame then
            statFrame:UnregisterAllEvents()
        end
    end
end

-----------------------------
-- Stat Dashboard (for /sk panel)
-----------------------------
local C = ns.COLORS

local function CreateStatRow(parent, label, color)
    local hexColor = string.format("%02x%02x%02x", color[1] * 255, color[2] * 255, color[3] * 255)

    local row = CreateFrame("Frame", nil, parent)
    row:SetHeight(90)

    -- Line 1: Stat name + base rating
    local nameText = row:CreateFontString(nil, "ARTWORK", "GameFontNormal")
    nameText:SetPoint("TOPLEFT", 4, -2)
    nameText:SetText(label)
    nameText:SetTextColor(color[1], color[2], color[3])

    local ratingText = row:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
    ratingText:SetPoint("LEFT", nameText, "RIGHT", 8, 0)
    ratingText:SetTextColor(0.70, 0.70, 0.73)

    -- Line 2: Effective Rating (left) + value (right)
    local effLabel = row:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
    effLabel:SetPoint("TOPLEFT", nameText, "BOTTOMLEFT", 0, -4)
    effLabel:SetTextColor(color[1], color[2], color[3])

    local effValue = row:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
    effValue:SetPoint("RIGHT", row, "RIGHT", -4, 0)
    effValue:SetPoint("TOP", effLabel, "TOP")

    -- Line 3: Diminishing Returns (left) + value (right)
    local drLabel = row:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
    drLabel:SetPoint("TOPLEFT", effLabel, "BOTTOMLEFT", 0, -2)
    drLabel:SetText("Diminishing Returns")
    drLabel:SetTextColor(1, 1, 1)

    local drValue = row:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
    drValue:SetPoint("RIGHT", row, "RIGHT", -4, 0)
    drValue:SetPoint("TOP", drLabel, "TOP")

    -- Line 4: Lost to DR (conditional)
    local lostLabel = row:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
    lostLabel:SetPoint("TOPLEFT", drLabel, "BOTTOMLEFT", 0, -2)
    lostLabel:SetTextColor(1, 0.4, 0.4)

    local lostValue = row:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
    lostValue:SetPoint("RIGHT", row, "RIGHT", -4, 0)
    lostValue:SetPoint("TOP", lostLabel, "TOP")
    lostValue:SetTextColor(1, 0.4, 0.4)

    -- Line 5: Next bracket (conditional)
    local nextLabel = row:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
    nextLabel:SetTextColor(0.67, 0.67, 0.67)

    local nextValue = row:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
    nextValue:SetPoint("RIGHT", row, "RIGHT", -4, 0)
    nextValue:SetTextColor(0.67, 0.67, 0.67)

    -- Line 6: Rating per 1% (always shown)
    local perPctLabel = row:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
    perPctLabel:SetTextColor(0.67, 0.67, 0.67)

    local perPctValue = row:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
    perPctValue:SetPoint("RIGHT", row, "RIGHT", -4, 0)
    perPctValue:SetTextColor(0.67, 0.67, 0.67)

    -- Progress bar
    local barBg = CreateFrame("Frame", nil, row, "BackdropTemplate")
    barBg:SetHeight(14)
    barBg:SetPoint("RIGHT", row, "RIGHT", -4, 0)
    barBg:SetBackdrop(ns.BACKDROP_CONTROL)
    barBg:SetBackdropColor(C.bgControl[1], C.bgControl[2], C.bgControl[3], C.bgControl[4])
    barBg:SetBackdropBorderColor(C.border[1], C.border[2], C.border[3], C.border[4])

    local fill = barBg:CreateTexture(nil, "ARTWORK")
    fill:SetPoint("TOPLEFT", 1, -1)
    fill:SetPoint("BOTTOMLEFT", 1, 1)
    fill:SetColorTexture(color[1], color[2], color[3], 0.6)

    local barText = barBg:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
    barText:SetPoint("CENTER")
    barText:SetTextColor(0.9, 0.9, 0.9)

    function row:Update(statId)
        local info = statInfo[statId]
        if not info then
            ratingText:SetText("")
            effLabel:SetText(""); effValue:SetText("")
            drValue:SetText("")
            lostLabel:SetText(""); lostLabel:Hide(); lostValue:SetText(""); lostValue:Hide()
            nextLabel:SetText(""); nextLabel:Hide(); nextValue:SetText(""); nextValue:Hide()
            perPctLabel:SetText(""); perPctValue:SetText("")
            barText:SetText("")
            fill:SetWidth(0.001)
            return
        end

        local effectiveRating = math.floor(info.effectivePct * info.convFactor)
        local lostRating = info.baseRating - effectiveRating
        local lostPct = info.rawPct - info.effectivePct
        local hasDR = lostPct > 0.005

        -- Line 1: base rating + raw %
        ratingText:SetText(string.format("%s rating  ·  %.2f%%",
            BreakUpLargeNumbers(info.baseRating), info.rawPct))

        -- Line 2: Effective Rating
        effLabel:SetText("Effective Rating")
        effLabel:SetTextColor(color[1], color[2], color[3])
        effValue:SetText(string.format("|cff%s%s|r", hexColor, BreakUpLargeNumbers(effectiveRating)))

        -- Line 3: Diminishing Returns
        if info.bracketPenalty > 0 then
            drValue:SetText(string.format("|cffff6666-%d%%|r", info.bracketPenalty * 100))
        else
            drValue:SetText("|cffffffffNone|r")
        end

        -- Line 4: Lost to DR
        local barAnchor
        if hasDR then
            lostLabel:SetText("Lost to DR")
            lostLabel:Show()
            lostValue:SetText(string.format("%s rating  (%.2f%%)",
                BreakUpLargeNumbers(lostRating), lostPct))
            lostValue:Show()
            barAnchor = lostLabel
        else
            lostLabel:Hide()
            lostValue:Hide()
            barAnchor = drLabel
        end

        -- Line 5: Next bracket
        if info.nextBracketPenalty > info.bracketPenalty and info.nextBracketPenalty < 1.0 then
            nextLabel:ClearAllPoints()
            nextLabel:SetPoint("TOPLEFT", barAnchor, "BOTTOMLEFT", 0, -2)
            nextLabel:SetText(string.format("%s rating until",
                BreakUpLargeNumbers(info.ratingUntilNext)))
            nextLabel:Show()
            nextValue:ClearAllPoints()
            nextValue:SetPoint("RIGHT", row, "RIGHT", -4, 0)
            nextValue:SetPoint("TOP", nextLabel, "TOP")
            nextValue:SetText(string.format("-%d%% DR", info.nextBracketPenalty * 100))
            nextValue:Show()
            barAnchor = nextLabel
        else
            nextLabel:Hide()
            nextValue:Hide()
        end

        -- Rating per 1%
        perPctLabel:ClearAllPoints()
        perPctLabel:SetPoint("TOPLEFT", barAnchor, "BOTTOMLEFT", 0, -2)
        perPctLabel:SetText("Rating per 1%")
        perPctValue:ClearAllPoints()
        perPctValue:SetPoint("RIGHT", row, "RIGHT", -4, 0)
        perPctValue:SetPoint("TOP", perPctLabel, "TOP")
        if info.ratingPerOnePct and info.ratingPerOnePct > 0 then
            if info.bracketPenalty > 0 then
                perPctValue:SetText(string.format("%.1f  |cffaaaaaa(%.1f base)|r", info.ratingPerOnePct, info.basePerOnePct))
            else
                perPctValue:SetText(string.format("%.1f", info.ratingPerOnePct))
            end
        else
            perPctValue:SetText("—")
        end
        barAnchor = perPctLabel

        -- Progress bar with X / Y
        barBg:ClearAllPoints()
        barBg:SetPoint("TOPLEFT", barAnchor, "BOTTOMLEFT", 0, -4)
        barBg:SetPoint("RIGHT", row, "RIGHT", -4, 0)

        local bracketMax = info.bracketMaxRating
        local bracketCur = info.bracketRating
        barText:SetText(string.format("%s / %s",
            BreakUpLargeNumbers(bracketCur),
            BreakUpLargeNumbers(bracketMax)))

        if bracketMax > 0 then
            local pct = math.min(bracketCur / bracketMax, 1)
            local barWidth = barBg:GetWidth() - 2
            if barWidth > 0 then
                fill:SetWidth(math.max(pct * barWidth, 0.001))
            end
        else
            fill:SetWidth(0.001)
        end

        -- Adjust row height based on visible lines
        local h = 2 + 14 + 4 + 12 + 2 + 12 -- name + eff + dr
        if hasDR then h = h + 2 + 12 end -- lost
        if info.nextBracketPenalty > info.bracketPenalty and info.nextBracketPenalty < 1.0 then
            h = h + 2 + 12
        end
        h = h + 2 + 12 -- rating per 1%
        h = h + 4 + 14 + 4 -- bar + bottom padding
        row:SetHeight(h)
    end

    return row
end

-----------------------------
-- Settings Panel Builder (Main)
-----------------------------
function ns.BuildStatSettings(content, anchor)
    local AddCB, GetLast, SetLast = ns.MakeCheckboxFactory(content, anchor)

    AddCB("StatTooltips", "statTooltips", "Stat Tooltips",
        "Show diminishing return info on character panel stat tooltips (effective rating, lost rating, bracket bar).",
        function(enabled)
            ns.SetStatTracking(SKToolsDB.statTooltips or SKToolsDB.statItemTooltips or SKToolsDB.statCompareTooltips)
        end)

    AddCB("StatItemTooltips", "statItemTooltips", "Item Tooltips",
        "Show effective stat values on item tooltips after diminishing returns.",
        function(enabled)
            ns.SetStatTracking(SKToolsDB.statTooltips or SKToolsDB.statItemTooltips or SKToolsDB.statCompareTooltips)
        end)

    AddCB("StatCompareTooltips", "statCompareTooltips", "Comparison Percentages",
        "Show effective percentage changes on item comparison stat deltas.",
        function(enabled)
            ns.SetStatTracking(SKToolsDB.statTooltips or SKToolsDB.statItemTooltips or SKToolsDB.statCompareTooltips)
        end)

    -- Stats Overview dashboard
    local header = ns.AddSectionHeader(content, GetLast(), "Stats Overview", false)
    local card = ns.CreateSectionCard(content)
    card:SetPoint("TOPLEFT", header, "TOPLEFT", -8, 8)

    local rows = {}
    local lastRow = header
    for i, def in ipairs(MAIN_STATS) do
        local row = CreateStatRow(content, def.label, def.color)
        row:SetPoint("TOPLEFT", lastRow, "BOTTOMLEFT", i == 1 and 2 or 0, i == 1 and -8 or -6)
        row:SetPoint("RIGHT", content, "RIGHT", -20, 0)
        rows[i] = { frame = row, statId = def.id }
        lastRow = row
    end

    card:SetPoint("BOTTOM", lastRow, "BOTTOM", 0, -8)
    card:SetPoint("RIGHT", content, "RIGHT", -8, 0)

    -- Refresh on panel show (works even if tooltips are disabled)
    local function Refresh()
        SetupConversionFactors()
        RecalculateStats()
        for _, r in ipairs(rows) do
            r.frame:Update(r.statId)
        end
    end

    local scrollParent = content:GetParent()
    if scrollParent then
        local oldOnShow = scrollParent:GetScript("OnShow")
        scrollParent:SetScript("OnShow", function(self, ...)
            if oldOnShow then oldOnShow(self, ...) end
            Refresh()
        end)
    end
    -- Also refresh on first build
    C_Timer.After(0.1, Refresh)

    return GetLast()
end

-----------------------------
-- Settings Panel Builder (Combat)
-----------------------------
function ns.BuildStatCombatSettings(content, anchor, csSyncControls)
    local AddCB, GetLast, SetLast = ns.MakeCombatCBFactory(content, anchor, csSyncControls)

    AddCB("StatTooltips", "statTooltips", "Stat Tooltips",
        "Show diminishing return info on character panel stat tooltips (effective rating, lost rating, bracket bar).",
        function(enabled)
            ns.SetStatTracking(SKToolsDB.statTooltips or SKToolsDB.statItemTooltips or SKToolsDB.statCompareTooltips)
        end)

    AddCB("StatItemTooltips", "statItemTooltips", "Item Tooltips",
        "Show effective stat values on item tooltips after diminishing returns.",
        function(enabled)
            ns.SetStatTracking(SKToolsDB.statTooltips or SKToolsDB.statItemTooltips or SKToolsDB.statCompareTooltips)
        end)

    AddCB("StatCompareTooltips", "statCompareTooltips", "Comparison Percentages",
        "Show effective percentage changes on item comparison stat deltas.",
        function(enabled)
            ns.SetStatTracking(SKToolsDB.statTooltips or SKToolsDB.statItemTooltips or SKToolsDB.statCompareTooltips)
        end)

    -- Stats Overview dashboard
    local header = ns.AddSectionHeader(content, GetLast(), "Stats Overview", false)
    local card = ns.CreateSectionCard(content)
    card:SetPoint("TOPLEFT", header, "TOPLEFT", -8, 8)

    local rows = {}
    local lastRow = header
    for i, def in ipairs(MAIN_STATS) do
        local row = CreateStatRow(content, def.label, def.color)
        row:SetPoint("TOPLEFT", lastRow, "BOTTOMLEFT", i == 1 and 2 or 0, i == 1 and -8 or -6)
        row:SetPoint("RIGHT", content, "RIGHT", -20, 0)
        rows[i] = { frame = row, statId = def.id }
        lastRow = row
    end

    card:SetPoint("BOTTOM", lastRow, "BOTTOM", 0, -8)
    card:SetPoint("RIGHT", content, "RIGHT", -8, 0)

    -- Refresh on combat panel show (works even if tooltips are disabled)
    table.insert(csSyncControls, { type = "custom", refresh = function()
        SetupConversionFactors()
        RecalculateStats()
        for _, r in ipairs(rows) do
            r.frame:Update(r.statId)
        end
    end })

    return GetLast()
end

-----------------------------
-- Standalone Stats Overview Window
-----------------------------
local statsFrame = nil

local function CreateStatsFrame()
    if statsFrame then return end

    local CC = ns.COLORS

    -- Outer frame — same style as /sk combat settings frame
    local f = CreateFrame("Frame", "SKToolsStatsOverviewFrame", UIParent, "BackdropTemplate")
    f:SetSize(680, 200) -- width matches /sk content area; height auto-sized
    f:SetPoint("CENTER")
    f:SetMovable(true)
    f:EnableMouse(true)
    f:SetClampedToScreen(true)
    f:SetFrameStrata("DIALOG")
    f:SetBackdrop(ns.BACKDROP_PANEL)
    f:SetBackdropColor(CC.bgFrame[1], CC.bgFrame[2], CC.bgFrame[3], CC.bgFrame[4])
    f:SetBackdropBorderColor(CC.border[1], CC.border[2], CC.border[3], 0.80)
    f:Hide()
    tinsert(UISpecialFrames, "SKToolsStatsOverviewFrame")

    ns.CreateDropShadow(f, 8)
    ns.CreateCyanAccentLine(f)

    -- ESC to close
    f:SetScript("OnKeyDown", function(self, key)
        if key == "ESCAPE" then
            pcall(self.SetPropagateKeyboardInput, self, false)
            self:Hide()
        else
            pcall(self.SetPropagateKeyboardInput, self, true)
        end
    end)
    pcall(f.EnableKeyboard, f, true)

    -- Draggable
    f:RegisterForDrag("LeftButton")
    f:SetScript("OnDragStart", f.StartMoving)
    f:SetScript("OnDragStop", f.StopMovingOrSizing)

    -- Close button — same VS Code style as /sk panel
    local closeBtn = CreateFrame("Button", nil, f, "BackdropTemplate")
    closeBtn:SetSize(24, 24)
    closeBtn:SetPoint("TOPRIGHT", -6, -6)
    closeBtn:SetBackdrop(ns.BACKDROP_CONTROL)
    closeBtn:SetBackdropColor(0, 0, 0, 0)
    closeBtn:SetBackdropBorderColor(0, 0, 0, 0)
    local closeLbl = closeBtn:CreateFontString(nil, "OVERLAY", "GameFontNormal")
    closeLbl:SetPoint("CENTER", 0, 0)
    closeLbl:SetText("x")
    closeLbl:SetTextColor(CC.textMuted[1], CC.textMuted[2], CC.textMuted[3])
    closeBtn:SetScript("OnClick", function() f:Hide() end)
    closeBtn:SetScript("OnEnter", function(self)
        self:SetBackdropColor(0.60, 0.15, 0.15, 0.6)
        self:SetBackdropBorderColor(0.60, 0.20, 0.20, 0.5)
        closeLbl:SetTextColor(1, 1, 1)
    end)
    closeBtn:SetScript("OnLeave", function(self)
        self:SetBackdropColor(0, 0, 0, 0)
        self:SetBackdropBorderColor(0, 0, 0, 0)
        closeLbl:SetTextColor(CC.textMuted[1], CC.textMuted[2], CC.textMuted[3])
    end)

    -- Content area — same bgContent background as /sk content container
    local contentArea = CreateFrame("Frame", nil, f)
    contentArea:SetPoint("TOPLEFT", 4, -4)
    contentArea:SetPoint("BOTTOMRIGHT", -4, 4)
    local contentBg = contentArea:CreateTexture(nil, "BACKGROUND")
    contentBg:SetAllPoints()
    contentBg:SetColorTexture(CC.bgContent[1], CC.bgContent[2], CC.bgContent[3], CC.bgContent[4])

    -- Content scroll child — same as MakeTabContent produces
    local content = CreateFrame("Frame", nil, contentArea)
    content:SetPoint("TOPLEFT", 0, 0)
    content:SetPoint("RIGHT", contentArea, "RIGHT", -22, 0)

    -- Tab title — same as ns.CreateTabTitle used in /sk stats tab
    local tabAnchor = ns.CreateTabTitle(content, "Stat Tracker", "Secondary stat diminishing returns and information.")

    -- Section header — identical to settings panel (AddSectionHeader with isFirst=false)
    local header = ns.AddSectionHeader(content, tabAnchor, "Stats Overview", false)

    -- Section card — identical to settings panel
    local card = ns.CreateSectionCard(content)
    card:SetPoint("TOPLEFT", header, "TOPLEFT", -8, 8)

    -- Stat rows — identical layout to settings panel
    local rows = {}
    local lastRow = header
    for i, def in ipairs(MAIN_STATS) do
        local row = CreateStatRow(content, def.label, def.color)
        row:SetPoint("TOPLEFT", lastRow, "BOTTOMLEFT", i == 1 and 2 or 0, i == 1 and -8 or -6)
        row:SetPoint("RIGHT", content, "RIGHT", -20, 0)
        rows[i] = { frame = row, statId = def.id }
        lastRow = row
    end

    card:SetPoint("BOTTOM", lastRow, "BOTTOM", 0, -8)
    card:SetPoint("RIGHT", content, "RIGHT", -8, 0)

    -- Refresh + auto-size outer frame to fit content
    local function Refresh()
        SetupConversionFactors()
        RecalculateStats()
        for _, r in ipairs(rows) do
            r.frame:Update(r.statId)
        end
        C_Timer.After(0, function()
            -- title(16) + subtitle(8+12) + gap(40) + header(14)
            local totalH = 16 + 14 + 8 + 12 + 40 + 14
            for i, r in ipairs(rows) do
                totalH = totalH + (i == 1 and 8 or 6) + r.frame:GetHeight()
            end
            -- 4+4 frame insets + 8 card bottom + 16 bottom padding
            f:SetHeight(totalH + 4 + 4 + 8 + 24)
        end)
    end

    f:SetToplevel(true)
    f:SetScript("OnShow", function(self) self:Raise(); Refresh() end)

    statsFrame = f
end

function ns.StatsOverview_Toggle()
    CreateStatsFrame()
    if statsFrame:IsShown() then
        statsFrame:Hide()
    else
        statsFrame:Show()
        ns.FadeIn(statsFrame, 0.15)
    end
end
