-- SKTools Smart Macros
-- Auto-replace role placeholders (@healer, @tank, @dps1-3) with party member names

local _, ns = ...

-----------------------------
-- Constants
-----------------------------
local SMARTMACRO_PLACEHOLDERS = { "@healer", "@tank", "@dps1", "@dps2", "@dps3" }
ns.SMARTMACRO_PLACEHOLDERS = SMARTMACRO_PLACEHOLDERS

-----------------------------
-- Namespace character macros (121+) by name-realm
-----------------------------
local function SmartMacro_CharPrefix()
    return UnitName("player") .. "-" .. (GetNormalizedRealmName() or "") .. "-"
end

local function SmartMacro_Key(idx)
    if idx <= 120 then return idx end
    return SmartMacro_CharPrefix() .. idx
end

local function SmartMacro_ParseKey(key)
    if type(key) == "number" then return key end
    return tonumber(key:match("-(%d+)$"))
end

local function SmartMacro_IsMyKey(key)
    if type(key) == "number" then return true end
    return key:find(SmartMacro_CharPrefix(), 1, true) == 1
end

ns.SmartMacro_Key = SmartMacro_Key
ns.SmartMacro_ParseKey = SmartMacro_ParseKey
ns.SmartMacro_IsMyKey = SmartMacro_IsMyKey

-----------------------------
-- Core Logic
-----------------------------
local function SmartMacro_GetPartyRoles()
    local healer, tank, dps = nil, nil, {}
    for i = 1, GetNumGroupMembers() - 1 do
        local unit = "party" .. i
        if UnitExists(unit) then
            local name = UnitName(unit)
            if name and name ~= "Unknown" then
                local role = UnitGroupRolesAssigned(unit)
                if role == "HEALER" then
                    healer = healer or name
                elseif role == "TANK" then
                    tank = tank or name
                else
                    dps[#dps + 1] = name
                end
            end
        end
    end
    return healer, tank, dps
end

-- Case-insensitive gsub helper: builds a pattern like "[hH][eE][aA]..."
local function ciPattern(str)
    return str:gsub("%a", function(c)
        return "[" .. c:lower() .. c:upper() .. "]"
    end)
end

local function SmartMacro_Resolve(template, healer, tank, dps)
    local body = template
    if healer then body = body:gsub(ciPattern("@healer"), "@" .. healer) end
    if tank then body = body:gsub(ciPattern("@tank"), "@" .. tank) end
    if dps[1] then body = body:gsub(ciPattern("@dps1"), "@" .. dps[1]) end
    if dps[2] then body = body:gsub(ciPattern("@dps2"), "@" .. dps[2]) end
    if dps[3] then body = body:gsub(ciPattern("@dps3"), "@" .. dps[3]) end
    return body
end

-- Reverse-substitute resolved names back to placeholders using last known mapping
-- Sorted by longest name first to prevent substring collisions
local function SmartMacro_ReverseResolve(body, mapping)
    if not mapping then return body end
    local subs = {}
    if mapping.healer then subs[#subs + 1] = { mapping.healer, "@healer" } end
    if mapping.tank then subs[#subs + 1] = { mapping.tank, "@tank" } end
    if mapping.dps then
        for i, n in ipairs(mapping.dps) do subs[#subs + 1] = { n, "@dps" .. i } end
    end
    table.sort(subs, function(a, b) return #a[1] > #b[1] end)
    for _, s in ipairs(subs) do
        body = body:gsub("@" .. s[1], s[2])
    end
    return body
end

local function SmartMacro_Update()
    if not SKToolsDB or not SKToolsDB.smartMacros then return end
    if GetNumGroupMembers() <= 1 then return end
    if InCombatLockdown() then return false end
    if MacroFrame and MacroFrame:IsShown() then return false end

    local healer, tank, dps = SmartMacro_GetPartyRoles()
    local templates = SKToolsDB.smartMacroTemplates
    local macroNames = SKToolsDB.smartMacroNames
    local lastResolved = SKToolsDB.smartMacroLastResolved
    local charPrefix = SmartMacro_CharPrefix()
    local lastMapping = SKToolsDB.smartMacroLastMapping[charPrefix]
    local numAccount, numChar = GetNumMacros()

    -- Phase 0: Validate tracked macros — detect index shifts from deleted macros
    for key, template in pairs(templates) do
        if SmartMacro_IsMyKey(key) then
            local idx = SmartMacro_ParseKey(key)
            local name = GetMacroInfo(idx)
            if not name then
                templates[key] = nil
                lastResolved[key] = nil
                macroNames[key] = nil
            elseif macroNames[key] and name ~= macroNames[key] then
                templates[key] = nil
                lastResolved[key] = nil
                macroNames[key] = nil
            end
        end
    end

    -- Phase 1: Scan all macros for NEW placeholders (freshly written by user)
    for idx = 1, numAccount do
        local name, _, body = GetMacroInfo(idx)
        if body then
            for _, ph in ipairs(SMARTMACRO_PLACEHOLDERS) do
                if body:lower():find(ph, 1, true) then
                    local key = SmartMacro_Key(idx)
                    templates[key] = body
                    macroNames[key] = name
                    lastResolved[key] = nil
                    break
                end
            end
        end
    end
    for idx = 121, 120 + numChar do
        local name, _, body = GetMacroInfo(idx)
        if body then
            for _, ph in ipairs(SMARTMACRO_PLACEHOLDERS) do
                if body:lower():find(ph, 1, true) then
                    local key = SmartMacro_Key(idx)
                    templates[key] = body
                    macroNames[key] = name
                    lastResolved[key] = nil
                    break
                end
            end
        end
    end

    -- Phase 1.5: Detect user edits on tracked macros
    for key, template in pairs(templates) do
        if SmartMacro_IsMyKey(key) then
            local idx = SmartMacro_ParseKey(key)
            local currentBody = GetMacroBody(idx)
            if currentBody and lastResolved[key] and currentBody ~= lastResolved[key] then
                local updatedTemplate = SmartMacro_ReverseResolve(currentBody, lastMapping)
                templates[key] = updatedTemplate
                lastResolved[key] = nil
            end
        end
    end

    -- Phase 2: Resolve all stored templates (only this character's keys)
    local resolved = 0
    for key, template in pairs(templates) do
        if SmartMacro_IsMyKey(key) then
            local idx = SmartMacro_ParseKey(key)
            local name = GetMacroInfo(idx)
            if not name then
                templates[key] = nil
                lastResolved[key] = nil
                macroNames[key] = nil
            else
                local newBody = SmartMacro_Resolve(template, healer, tank, dps)
                if newBody ~= GetMacroBody(idx) then
                    EditMacro(idx, nil, nil, newBody)
                    lastResolved[key] = newBody
                    resolved = resolved + 1
                else
                    lastResolved[key] = newBody
                end
            end
        end
    end

    -- Store the mapping per character for reverse-substitution
    local mapping = { healer = healer, tank = tank, dps = {} }
    for i, name in ipairs(dps) do mapping.dps[i] = name end
    SKToolsDB.smartMacroLastMapping[charPrefix] = mapping

    if resolved > 0 then
        local parts = {}
        if healer then parts[#parts + 1] = "Healer: |cff00ff00" .. healer .. "|r" end
        if tank then parts[#parts + 1] = "Tank: |cff00ff00" .. tank .. "|r" end
        for i, name in ipairs(dps) do
            parts[#parts + 1] = "DPS" .. i .. ": |cff00ff00" .. name .. "|r"
        end
        print("|cff00E5EESKTools|r Smart Macros updated (" .. resolved .. "): " .. table.concat(parts, ", "))
    end
    return true
end

-----------------------------
-- Event Handling
-----------------------------
local smartMacroFrame = CreateFrame("Frame")
local smartMacroPending = false
local smartMacroSafetyTimer = nil

local function SmartMacro_AllNamesReady()
    for i = 1, GetNumGroupMembers() - 1 do
        local unit = "party" .. i
        if UnitExists(unit) then
            local name = UnitName(unit)
            if not name or name == "Unknown" then return false end
        end
    end
    return true
end

local function SmartMacro_StopListening()
    smartMacroFrame:UnregisterEvent("UNIT_NAME_UPDATE")
    if smartMacroSafetyTimer then smartMacroSafetyTimer:Cancel(); smartMacroSafetyTimer = nil end
end

local function SmartMacro_TryUpdate()
    if InCombatLockdown() then
        smartMacroPending = true
        smartMacroFrame:RegisterEvent("PLAYER_REGEN_ENABLED")
        SmartMacro_StopListening()
        return
    end
    local result = SmartMacro_Update()
    if result == false then return end
    if SmartMacro_AllNamesReady() then
        SmartMacro_StopListening()
    end
end

local function SmartMacro_StartListening()
    SmartMacro_TryUpdate()
    if not InCombatLockdown() and not SmartMacro_AllNamesReady() then
        smartMacroFrame:RegisterEvent("UNIT_NAME_UPDATE")
        if smartMacroSafetyTimer then smartMacroSafetyTimer:Cancel() end
        smartMacroSafetyTimer = C_Timer.NewTimer(5, function()
            smartMacroSafetyTimer = nil
            smartMacroFrame:UnregisterEvent("UNIT_NAME_UPDATE")
            SmartMacro_TryUpdate()
        end)
    end
end

local function SmartMacro_OnEvent(self, event)
    if not SKToolsDB or not SKToolsDB.smartMacros then return end
    if event == "GROUP_ROSTER_UPDATE" then
        local n = GetNumGroupMembers()
        if n <= 1 or n > 5 then return end  -- skip solo and large groups (BGs/raids)
        local ok, err = pcall(SmartMacro_StartListening)
        if not ok then ns.SK_ReportError("SmartMacro:GROUP_ROSTER", err) end
    elseif event == "UNIT_NAME_UPDATE" then
        if GetNumGroupMembers() <= 1 then return end
        local ok, err = pcall(SmartMacro_TryUpdate)
        if not ok then ns.SK_ReportError("SmartMacro:UNIT_NAME", err) end
    elseif event == "PLAYER_REGEN_ENABLED" then
        smartMacroFrame:UnregisterEvent("PLAYER_REGEN_ENABLED")
        if smartMacroPending then
            smartMacroPending = false
            local ok, err = pcall(SmartMacro_StartListening)
            if not ok then ns.SK_ReportError("SmartMacro:REGEN", err) end
        end
    end
end

local function SetSmartMacros(enabled)
    if enabled then
        smartMacroFrame:RegisterEvent("GROUP_ROSTER_UPDATE")
        smartMacroFrame:SetScript("OnEvent", SmartMacro_OnEvent)
    else
        smartMacroFrame:UnregisterAllEvents()
        SmartMacro_StopListening()
    end
end

ns.SetSmartMacros = SetSmartMacros

-----------------------------
-- Tracked Macro List Builder (shared by both panels)
-----------------------------
local function BuildTrackedMacroList(parentPanel, anchor)
    local AddSectionHeader = ns.AddSectionHeader
    local trackedHeader = AddSectionHeader(parentPanel, anchor, "Tracked Macros", false)
    local trackedContainer = CreateFrame("Frame", nil, parentPanel)
    trackedContainer:SetPoint("TOPLEFT", trackedHeader, "BOTTOMLEFT", 0, -8)
    trackedContainer:SetPoint("RIGHT", parentPanel, "RIGHT", -8, 0)
    trackedContainer:SetHeight(300)

    local emptyText = trackedContainer:CreateFontString(nil, "ARTWORK", "GameFontDisableSmall")
    emptyText:SetPoint("TOPLEFT", 0, 0)
    emptyText:SetText("No tracked macros yet. Create a macro with any of the placeholders above and join a group.")

    local macroRows = {}

    local function RefreshMacroList()
        for _, row in ipairs(macroRows) do row:Hide() end
        wipe(macroRows)

        local templates = SKToolsDB.smartMacroTemplates or {}
        local sNames = SKToolsDB.smartMacroNames or {}
        local numAccount, numChar = GetNumMacros()
        for idx = 1, numAccount do
            local name, _, body = GetMacroInfo(idx)
            if body then
                for _, ph in ipairs(SMARTMACRO_PLACEHOLDERS) do
                    if body:lower():find(ph, 1, true) then
                        local key = SmartMacro_Key(idx)
                        templates[key] = body
                        sNames[key] = name
                        break
                    end
                end
            end
        end
        for idx = 121, 120 + numChar do
            local name, _, body = GetMacroInfo(idx)
            if body then
                for _, ph in ipairs(SMARTMACRO_PLACEHOLDERS) do
                    if body:lower():find(ph, 1, true) then
                        local key = SmartMacro_Key(idx)
                        templates[key] = body
                        sNames[key] = name
                        break
                    end
                end
            end
        end

        local count = 0
        local yOff = 0

        for key, template in pairs(templates) do
            if SmartMacro_IsMyKey(key) then
                local idx = SmartMacro_ParseKey(key)
                local name, iconID = GetMacroInfo(idx)
                if not name or (sNames[key] and name ~= sNames[key]) then
                    templates[key] = nil
                    SKToolsDB.smartMacroLastResolved[key] = nil
                    sNames[key] = nil
                else
                    count = count + 1
                    local row = CreateFrame("Frame", nil, trackedContainer)
                    row:SetHeight(26)
                    row:SetPoint("TOPLEFT", trackedContainer, "TOPLEFT", 0, yOff)
                    row:SetPoint("RIGHT", trackedContainer, "RIGHT", 0, 0)

                    local icon = row:CreateTexture(nil, "ARTWORK")
                    icon:SetSize(20, 20)
                    icon:SetPoint("LEFT", 0, 0)
                    icon:SetTexCoord(0.08, 0.92, 0.08, 0.92)
                    if iconID then icon:SetTexture(iconID) end

                    local nameStr = row:CreateFontString(nil, "ARTWORK", "GameFontNormal")
                    nameStr:SetPoint("LEFT", icon, "RIGHT", 6, 0)
                    nameStr:SetText(name)

                    local preview = row:CreateFontString(nil, "ARTWORK", "GameFontDisableSmall")
                    preview:SetPoint("LEFT", nameStr, "RIGHT", 8, 0)
                    preview:SetPoint("RIGHT", row, "RIGHT", -70, 0)
                    preview:SetJustifyH("LEFT")
                    preview:SetWordWrap(false)
                    local currentBody = GetMacroBody(idx) or template
                    local flat = currentBody:gsub("\n", " "); preview:SetText(#flat > 50 and flat:sub(1, 50) .. "..." or flat)

                    local removeBtn = ns.CreateThemedButton(row, "Remove", 60, 22, "danger")
                    removeBtn:SetPoint("RIGHT", row, "RIGHT", 0, 0)
                    removeBtn:SetScript("OnClick", function()
                        SKToolsDB.smartMacroTemplates[key] = nil
                        SKToolsDB.smartMacroLastResolved[key] = nil
                        SKToolsDB.smartMacroNames[key] = nil
                        RefreshMacroList()
                    end)

                    macroRows[#macroRows + 1] = row
                    yOff = yOff - 28
                end
            end
        end

        if count > 0 then
            emptyText:Hide()
            trackedContainer:SetHeight(math.max(30, count * 28))
        else
            emptyText:Show()
            trackedContainer:SetHeight(30)
        end
    end

    return trackedContainer, RefreshMacroList
end

-----------------------------
-- Main Settings Builder
-----------------------------
function ns.BuildMacroSettings(content, anchor)
    local MakeCheckboxFactory = ns.MakeCheckboxFactory
    local AddSectionHeader = ns.AddSectionHeader

    local macroCard = ns.CreateSectionCard(content)
    macroCard:SetPoint("TOPLEFT", anchor, "BOTTOMLEFT", -4, -12)
    local AddCB, GetLast, SetLast = MakeCheckboxFactory(content, anchor)

    AddCB("SmartMacros", "smartMacros", "Enable Smart Macros",
        "Automatically scans macros for @healer, @tank, @dps1, @dps2, @dps3 and replaces with party member names.",
        SetSmartMacros)

    macroCard:SetPoint("BOTTOM", GetLast(), "BOTTOM", 0, -8)
    macroCard:SetPoint("RIGHT", content, "RIGHT", -8, 0)

    -- Setup instructions
    do
        local howHeader = AddSectionHeader(content, GetLast(), "How to Use", false)
        local howCard = ns.CreateSectionCard(content)
        howCard:SetPoint("TOPLEFT", howHeader, "TOPLEFT", -8, 8)

        local howText = content:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        howText:SetPoint("TOPLEFT", howHeader, "BOTTOMLEFT", 0, -8)
        howText:SetPoint("RIGHT", content, "RIGHT", -8, 0)
        howText:SetJustifyH("LEFT")
        howText:SetSpacing(2)
        howText:SetText(
            "1. Open the WoW macro editor (|cff00E5EE/macro|r) and create or edit a macro.\n" ..
            "2. Use these placeholders anywhere you'd normally put a player name:\n\n" ..
            "    |cff00E5EE@healer|r  — replaced with the party member who has the Healer role\n" ..
            "    |cff00E5EE@tank|r  — replaced with the party member who has the Tank role\n" ..
            "    |cff00E5EE@dps1|r  — replaced with the 1st DPS or unassigned party member\n" ..
            "    |cff00E5EE@dps2|r  — replaced with the 2nd DPS party member\n" ..
            "    |cff00E5EE@dps3|r  — replaced with the 3rd DPS party member\n\n" ..
            "3. That's it! When you join a group, the placeholders are auto-replaced with real names.\n" ..
            "    The macro updates again whenever your group roster changes.\n\n" ..
            "|cff888888Example:|r\n" ..
            "    /cast [@healer] Blessing of Protection\n" ..
            "    /cast [@dps1] Innervate")
        howCard:SetPoint("BOTTOM", howText, "BOTTOM", 0, -8)
        howCard:SetPoint("RIGHT", content, "RIGHT", -8, 0)
        SetLast(howText)
    end

    -- Tracked macros list
    do
        local trackedContainer, RefreshMacroList = BuildTrackedMacroList(content, GetLast())

        -- Refresh on every show of the outer panel AND its parent settings frame
        local outerPanel = content:GetParent():GetParent() -- content → scrollchild → scrollframe → outer
        if outerPanel then
            outerPanel:HookScript("OnShow", RefreshMacroList)
        end
        C_Timer.After(0, function()
            if SettingsPanel then
                SettingsPanel:HookScript("OnShow", function()
                    C_Timer.After(0.1, function()
                        if outerPanel and outerPanel:IsVisible() then RefreshMacroList() end
                    end)
                end)
            end
        end)

        SetLast(trackedContainer)
    end

    return GetLast()
end

-----------------------------
-- Combat Settings Builder
-----------------------------
function ns.BuildMacroCombatSettings(content, csSyncControls)
    local MakeCombatCBFactory = ns.MakeCombatCBFactory
    local AddSectionHeader = ns.AddSectionHeader

    local macAnchor = ns.CreateTabTitle(content, "Smart Macros", "Auto-replace @healer, @tank, @dps1-3 in macros with party member names.")

    local macroCard = ns.CreateSectionCard(content)
    macroCard:SetPoint("TOPLEFT", macAnchor, "BOTTOMLEFT", -4, -12)
    local AddCB, GetLast, SetLast = MakeCombatCBFactory(content, macAnchor, csSyncControls)
    AddCB("SmartMacros", "smartMacros", "Enable Smart Macros",
        "Automatically scans macros for @healer, @tank, @dps1, @dps2, @dps3 and replaces with party member names.",
        SetSmartMacros)
    macroCard:SetPoint("BOTTOM", GetLast(), "BOTTOM", 0, -8)
    macroCard:SetPoint("RIGHT", content, "RIGHT", -8, 0)

    -- How to Use
    local howHeader = AddSectionHeader(content, GetLast(), "How to Use", false)
    local howCard = ns.CreateSectionCard(content)
    howCard:SetPoint("TOPLEFT", howHeader, "TOPLEFT", -8, 8)
    local howText = content:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
    howText:SetPoint("TOPLEFT", howHeader, "BOTTOMLEFT", 0, -8)
    howText:SetPoint("RIGHT", content, "RIGHT", -8, 0)
    howText:SetJustifyH("LEFT")
    howText:SetSpacing(2)
    howText:SetText(
        "1. Open the WoW macro editor (|cff00E5EE/macro|r) and create or edit a macro.\n" ..
        "2. Use these placeholders anywhere you'd normally put a player name:\n\n" ..
        "    |cff00E5EE@healer|r  — replaced with the party member who has the Healer role\n" ..
        "    |cff00E5EE@tank|r  — replaced with the party member who has the Tank role\n" ..
        "    |cff00E5EE@dps1|r  — replaced with the 1st DPS or unassigned party member\n" ..
        "    |cff00E5EE@dps2|r  — replaced with the 2nd DPS party member\n" ..
        "    |cff00E5EE@dps3|r  — replaced with the 3rd DPS party member\n\n" ..
        "3. That's it! When you join a group, the placeholders are auto-replaced with real names.\n" ..
        "    The macro updates again whenever your group roster changes.\n\n" ..
        "|cff888888Example:|r\n" ..
        "    /cast [@healer] Blessing of Protection\n" ..
        "    /cast [@dps1] Innervate")
    howCard:SetPoint("BOTTOM", howText, "BOTTOM", 0, -8)
    howCard:SetPoint("RIGHT", content, "RIGHT", -8, 0)
    SetLast(howText)

    -- Tracked macros list
    local trackedContainer, RefreshCSMacroList = BuildTrackedMacroList(content, GetLast())
    table.insert(csSyncControls, { type = "custom", refresh = RefreshCSMacroList })
end

