-- SKTools PvP Titles
-- Inspect player achievements to show Gladiator/R1/Duelist/Rival badges on nameplates
-- Ctrl+mouseover to inspect, cached results shown automatically

local _, ns = ...

-----------------------------
-- Achievement & Statistic IDs
-----------------------------

-- 3v3 Arena Rank 1 achievements (named gladiator titles = top 0.1%)
local RANK1_ACHIEVEMENTS = {
    -- TBC
    418, 419, 420,
    -- WotLK
    3336, 3436, 3758, 4599,
    -- Cata
    6002, 6124, 6938,
    -- MoP
    8214, 8791, 8643, 8666,
    -- WoD
    9232, 10096, 10097,
    -- Legion
    11012, 11014, 11037, 11062, 12010, 12134, 12185,
    -- BfA
    12945, 13200, 13630, 13957,
    -- Shadowlands
    14690, 14973, 15353, 15606,
    -- Dragonflight
    15951, 17764, 19132, 19454,
    -- The War Within
    40380, 41354, 42036,
    -- Midnight
    61180,  -- Galactic Gladiator: Midnight S1
}

-- Gladiator achievements (40 total: generic S1 + named titles S2-S13 + "Gladiator: Season X" S14+)
local GLAD_ACHIEVEMENTS = {
    -- TBC
    2091,           -- Gladiator (generic — S1, no named title)
    418, 419, 420,
    -- WotLK
    3336, 3436, 3758, 4599,
    -- Cata
    6002, 6124, 6938,
    -- MoP S12-S13 (named titles)
    8214, 8791,
    -- MoP S14-S15 (separate Glad tier achievements start here)
    8644, 8667,
    -- WoD
    9239, 10098, 10110,
    -- Legion
    11011, 11013, 11038, 11061, 12045, 12167, 12168,
    -- BfA
    12961, 13212, 13647, 13967,
    -- Shadowlands
    14689, 14972, 15352, 15605,
    -- Dragonflight
    15957, 17740, 19091, 19490,
    -- The War Within
    40393, 41032, 41049,
    -- Midnight
    61188,  -- Gladiator: Midnight S1
}

-- Solo Shuffle R1 achievements (top 0.1% SS ladder — Named Legend titles, DF+)
local SOLOSHUFFLE_R1_ACHIEVEMENTS = {
    -- Dragonflight (Named Legend = SS R1)
    16734,  -- Crimson Legend: DF S1
    17767,  -- Obsidian Legend: DF S2
    19131,  -- Verdant Legend: DF S3
    19453,  -- Draconic Legend: DF S4
    -- The War Within (Named Legend = SS R1)
    40381,  -- Forged Legend: TWW S1
    41355,  -- Prized Legend: TWW S2
    42033,  -- Astral Legend: TWW S3
    -- Midnight
    61179,  -- Galactic Legend: Midnight S1
}

-- BG Blitz R1 (Marshal/Warlord) achievements (top 0.1% Blitz rating)
-- Both Horde (Warlord) and Alliance (Marshal) variants
local BLITZ_R1_ACHIEVEMENTS = {
    -- The War Within
    40234, 40235,  -- Forged Warlord/Marshal: TWW S1
    41356, 41357,  -- Prized Warlord/Marshal: TWW S2
    42034, 42035,  -- Astral Warlord/Marshal: TWW S3
    -- Midnight
    61177, 61178,  -- Galactic Marshal/Warlord: Midnight S1
}

-- Export achievement arrays to ns for reuse (e.g. CharacterProfile self-scan)
ns.R1_ACHIEVEMENTS = RANK1_ACHIEVEMENTS
ns.GLAD_ACHIEVEMENTS = GLAD_ACHIEVEMENTS
ns.SOLOSHUFFLE_R1_ACHIEVEMENTS = SOLOSHUFFLE_R1_ACHIEVEMENTS
ns.BLITZ_R1_ACHIEVEMENTS = BLITZ_R1_ACHIEVEMENTS

-- General PvP achievements
local GLADIATOR_ACHIEVEMENT = 2091
local DUELIST_ACHIEVEMENT = 2092
local RIVAL_ACHIEVEMENT = 2093
local CHALLENGER_ACHIEVEMENT = 2090
local HERO_HORDE_ACHIEVEMENT = 6941
local HERO_ALLIANCE_ACHIEVEMENT = 6942

-- Rating statistics
local STAT_HIGHEST_2V2 = 370
local STAT_HIGHEST_3V3 = 595

-- Season progress achievements (for self-tracking: Gladiator/Legend/Strategist 50-win requirements)
local SEASON_GLAD = {       -- Gladiator: 50 wins at 2400+ in 3v3
    [30]=14689, [31]=14972, [32]=15352, [33]=15605,  -- SL
    [34]=15957, [35]=17740, [36]=19091, [37]=19490,  -- DF
    [38]=40393, [39]=41032, [40]=41049, [41]=61188,  -- TWW+MN
}
local SEASON_LEGEND = {     -- Legend: 50 wins at 2400+ in Solo Shuffle
    [36]=19304, [37]=19500,                          -- DF
    [38]=40395, [39]=41358, [40]=42023, [41]=61190,  -- TWW+MN
}
local SEASON_STRATEGIST = { -- Strategist: 50 wins at 2400+ in BG Blitz
    [38]=40233, [39]=41363, [40]=42024, [41]=61194,  -- TWW+MN
}

-----------------------------
-- Display Config
-----------------------------
local R1_ICON = "Interface\\Icons\\Achievement_Featsofstrength_Gladiator_08"  -- red helmet (same as SKTools button)
local GLAD_ICON = 236539    -- Duelist (2092) blue helmet
local SSR1_ICON = 236540    -- Gladiator (2091) green helmet
local THREEK_ICON = 236537  -- Challenger (2090) gold helmet
local BLITZ_R1_ICON = 236542    -- achievement_featsofstrength_gladiator_09 (Elite helmet)
local MURLOC_ICON = "Interface\\Icons\\INV_Misc_Head_Murloc_01"  -- murloc for no titles

-- Fallback icons for lower ranks (crossed swords)
local FALLBACK_ICONS = {
    hero = "Interface\\Icons\\Achievement_PVP_P_14",  -- Grand Marshal shield
    duelist = 132147,    -- ability_dualwield (crossed swords)
    rival = 132147,      -- ability_dualwield (crossed swords)
    challenger = 132147, -- ability_dualwield (crossed swords)
}
local CIRCLE_MASK = "Interface\\CHARACTERFRAME\\TempPortraitAlphaMask"
local BADGE_SIZE = 26
local BADGE_BORDER = 2
local BADGE_INNER = BADGE_SIZE - BADGE_BORDER * 2
local BADGE_ICON_SIZE = BADGE_INNER - 2
local BADGE_SPACING = 5

local RANK_FALLBACK = {
    hero       = { icon = FALLBACK_ICONS.hero,        label = "Hero",       color = {0.6, 0.6, 0.6} },
    duelist    = { icon = FALLBACK_ICONS.duelist,      label = "Duelist",    color = {0.2, 0.7, 0.9} },
    rival      = { icon = FALLBACK_ICONS.rival,        label = "Rival",      color = {0, 0.8, 0.4} },
    challenger = { icon = FALLBACK_ICONS.challenger,   label = "Challenger", color = {0.6, 0.6, 0.6} },
}

-----------------------------
-- Badge Type Definitions (exported for reuse)
-----------------------------
ns.PVP_BADGE_TYPES = {
    r1      = { icon = R1_ICON,        ring = {0.8, 0.15, 0.15}, countColor = {1, 0.3, 0.3} },
    ssR1    = { icon = SSR1_ICON,      ring = {0.15, 0.6, 0.15}, countColor = {0.3, 1, 0.3} },
    glad    = { icon = GLAD_ICON,      ring = {0.15, 0.3, 0.8},  countColor = {0.4, 0.6, 1} },
    blitzR1 = { icon = BLITZ_R1_ICON,  ring = {0.6, 0.2, 0.8},  countColor = {0.8, 0.5, 1} },
    threek  = { icon = THREEK_ICON,    ring = {0.8, 0.65, 0.1},  countColor = {1, 0.84, 0} },
    hero    = { icon = FALLBACK_ICONS.hero,    ring = {0.6, 0.6, 0.6}, countColor = {1, 0.84, 0} },
    duelist = { icon = FALLBACK_ICONS.duelist,  ring = {0.2, 0.7, 0.9}, countColor = {0.2, 0.7, 0.9} },
    rival   = { icon = FALLBACK_ICONS.rival,    ring = {0, 0.8, 0.4},   countColor = {0, 0.8, 0.4} },
    challenger = { icon = FALLBACK_ICONS.challenger, ring = {0.6, 0.6, 0.6}, countColor = {0.6, 0.6, 0.6} },
}

ns.PVP_BADGE_GENERAL = {
    gladiator  = GLADIATOR_ACHIEVEMENT,
    duelist    = DUELIST_ACHIEVEMENT,
    rival      = RIVAL_ACHIEVEMENT,
    challenger = CHALLENGER_ACHIEVEMENT,
    heroHorde  = HERO_HORDE_ACHIEVEMENT,
    heroAlliance = HERO_ALLIANCE_ACHIEVEMENT,
}

-----------------------------
-- Circular Badge Helper
-----------------------------
local function CreateCircularBadge(parent, iconTexture, ringR, ringG, ringB, optSize)
    local sz = optSize or BADGE_SIZE
    local border = math.max(2, math.floor(sz * BADGE_BORDER / BADGE_SIZE))
    local inner = sz - border * 2
    local iconSz = inner - 2

    local f = CreateFrame("Frame", nil, parent)
    f:SetSize(sz, sz)

    -- Circular mask
    local mask = f:CreateMaskTexture()
    mask:SetAllPoints()
    mask:SetTexture(CIRCLE_MASK, "CLAMPTOBLACKADDITIVE", "CLAMPTOBLACKADDITIVE")

    local innerMask = f:CreateMaskTexture()
    innerMask:SetSize(inner, inner)
    innerMask:SetPoint("CENTER")
    innerMask:SetTexture(CIRCLE_MASK, "CLAMPTOBLACKADDITIVE", "CLAMPTOBLACKADDITIVE")

    -- Border ring (colored)
    local ring = f:CreateTexture(nil, "BACKGROUND")
    ring:SetAllPoints()
    ring:SetColorTexture(ringR, ringG, ringB, 0.9)
    ring:AddMaskTexture(mask)
    f.ring = ring

    -- Dark fill
    local fill = f:CreateTexture(nil, "BORDER")
    fill:SetSize(inner, inner)
    fill:SetPoint("CENTER")
    fill:SetColorTexture(0.05, 0.05, 0.05, 0.9)
    fill:AddMaskTexture(innerMask)

    -- Icon (circular masked, NO vertex color tint)
    local icon = f:CreateTexture(nil, "ARTWORK")
    icon:SetSize(iconSz, iconSz)
    icon:SetPoint("CENTER")
    icon:SetTexture(iconTexture)
    icon:SetTexCoord(0.08, 0.92, 0.08, 0.92)
    icon:AddMaskTexture(innerMask)
    f.icon = icon

    -- Count text (bottom-right corner)
    local count = f:CreateFontString(nil, "OVERLAY")
    count:SetFont(STANDARD_TEXT_FONT, 10, "OUTLINE")
    count:SetPoint("BOTTOMRIGHT", f, "BOTTOMRIGHT", 3, -3)
    count:Hide()
    f.count = count

    -- Label text (below badge, for fallback ranks)
    local label = f:CreateFontString(nil, "OVERLAY")
    label:SetFont(STANDARD_TEXT_FONT, 8, "OUTLINE")
    label:SetPoint("TOP", f, "BOTTOM", 0, -1)
    label:Hide()
    f.label = label

    f:Hide()
    return f
end

ns.CreateCircularBadge = CreateCircularBadge

-----------------------------
-- Scan Animation (scope lock-on effect)
-----------------------------
local scanAnimFrame = nil

local function KillScanAnim()
    if scanAnimFrame then
        scanAnimFrame:SetScript("OnUpdate", nil)
        scanAnimFrame:Hide()
    end
end

local function PlayScanAnim(plate)
    KillScanAnim()
    if not plate or not plate:IsVisible() then return end

    -- Create frame once, reuse
    if not scanAnimFrame then
        local f = CreateFrame("Frame", nil, UIParent)
        f:SetFrameStrata("HIGH")

        -- 4 corner brackets (L-shapes)
        f.corners = {}
        for i = 1, 4 do
            local c = {}
            c.h = f:CreateTexture(nil, "OVERLAY")
            c.v = f:CreateTexture(nil, "OVERLAY")
            c.h:SetColorTexture(0, 0.9, 0.9, 0.9)
            c.v:SetColorTexture(0, 0.9, 0.9, 0.9)
            f.corners[i] = c
        end

        -- Crosshair lines (horizontal + vertical through center)
        local hLine = f:CreateTexture(nil, "ARTWORK")
        hLine:SetColorTexture(0, 0.9, 0.9, 0.5)
        hLine:SetSize(1, 1)  -- sized dynamically
        f.hLine = hLine

        local vLine = f:CreateTexture(nil, "ARTWORK")
        vLine:SetColorTexture(0, 0.9, 0.9, 0.5)
        vLine:SetSize(1, 1)
        f.vLine = vLine

        scanAnimFrame = f
    end

    local f = scanAnimFrame
    f:SetParent(plate)
    local okLvl, lvl = pcall(plate.GetFrameLevel, plate)
    f:SetFrameLevel((okLvl and lvl or 0) + 20)
    f:SetAllPoints(plate)
    f:SetAlpha(1)
    f:Show()

    local pw, ph = plate:GetSize()
    local armLen = 8
    local armThick = 2
    local dur = 0.6  -- animation duration
    local startScale = 3.0  -- how far out things start
    local elapsed = 0

    -- Size the corner bracket arms
    for _, c in ipairs(f.corners) do
        c.h:SetSize(armLen, armThick)
        c.v:SetSize(armThick, armLen)
        c.h:Show()
        c.v:Show()
    end
    f.hLine:Show()
    f.vLine:Show()

    f:SetScript("OnUpdate", function(self, dt)
        local ok, err = pcall(function()
            if not plate or not plate:IsVisible() then
                KillScanAnim()
                return
            end

            elapsed = elapsed + dt
            local t = elapsed / dur
            if t >= 1 then
                KillScanAnim()
                return
            end

            -- Ease out (fast start, slow end)
            local ease = 1 - (1 - t) * (1 - t)
            local scale = startScale + (1.0 - startScale) * ease
            local alpha = 0.9 * (1 - t * 0.5)  -- fade slightly

            self:SetAlpha(alpha)

            -- Corner offsets: converge from outside to plate edges
            local ox = (pw * 0.5) * scale
            local oy = (ph * 0.5) * scale
            local corners = f.corners

            -- Top-left
            corners[1].h:ClearAllPoints()
            corners[1].h:SetPoint("TOPRIGHT", f, "CENTER", -ox + armLen, oy)
            corners[1].v:ClearAllPoints()
            corners[1].v:SetPoint("BOTTOMLEFT", f, "CENTER", -ox, oy - armLen)

            -- Top-right
            corners[2].h:ClearAllPoints()
            corners[2].h:SetPoint("TOPLEFT", f, "CENTER", ox - armLen, oy)
            corners[2].v:ClearAllPoints()
            corners[2].v:SetPoint("BOTTOMRIGHT", f, "CENTER", ox, oy - armLen)

            -- Bottom-left
            corners[3].h:ClearAllPoints()
            corners[3].h:SetPoint("BOTTOMRIGHT", f, "CENTER", -ox + armLen, -oy)
            corners[3].v:ClearAllPoints()
            corners[3].v:SetPoint("TOPLEFT", f, "CENTER", -ox, -oy + armLen)

            -- Bottom-right
            corners[4].h:ClearAllPoints()
            corners[4].h:SetPoint("BOTTOMLEFT", f, "CENTER", ox - armLen, -oy)
            corners[4].v:ClearAllPoints()
            corners[4].v:SetPoint("TOPRIGHT", f, "CENTER", ox, -oy + armLen)

            -- Crosshair lines: shrink inward with the brackets
            local lineThick = 1
            local hWidth = pw * scale * 1.2
            local vHeight = ph * scale * 1.2
            f.hLine:SetSize(hWidth, lineThick)
            f.hLine:ClearAllPoints()
            f.hLine:SetPoint("CENTER", f, "CENTER")
            f.vLine:SetSize(lineThick, vHeight)
            f.vLine:ClearAllPoints()
            f.vLine:SetPoint("CENTER", f, "CENTER")
        end)
        if not ok then KillScanAnim() end
    end)
end

-----------------------------
-- Inspect State Machine
-----------------------------
local INSPECT_TIMEOUT = 6.0
local INSPECT_COOLDOWN = 0.5

local inspectState = "IDLE"
local inspectUnit = nil
local inspectGUID = nil
local inspectPlate = nil
local inspectTimer = nil
local inspectPlayerKey = nil  -- name-realm stored at scan start (stable even if token goes stale)
local inspectColorName = nil  -- class-colored first name for chat messages
local pendingGUID = nil  -- single pending scan (replaces queue)
local lastInspectTime = 0
local inspectRetries = 0
local MAX_RETRIES = 2

-- Runtime GUID → Name-Realm lookup (populated from SavedVar + inspects)
local guidToKey = {}

-- Overlay frames keyed by nameplate
local pvpTitleOverlays = {}

-- One-time hint flag (per session)
local nameplateHintShown = false

-- Track failed scan GUIDs with timestamps to prevent re-scan spam
local failedGUIDs = {}
local FAILED_COOLDOWN = 1

-- Track which GUIDs have already had their summary printed to chat (per session)
local printedGUIDs = {}

-----------------------------
-- Cache Helpers
-----------------------------
local function GetClassColorHex(unit)
    local ok, _, classFile = pcall(UnitClass, unit)
    if not ok or not classFile then return "FFFFFF" end
    if issecretvalue and issecretvalue(classFile) then return "FFFFFF" end
    local color = RAID_CLASS_COLORS[classFile]
    if not color then return "FFFFFF" end
    return string.format("%02x%02x%02x", color.r * 255, color.g * 255, color.b * 255)
end

local function ClassColorName(unit)
    local name = UnitName(unit)
    if not name then return "Unknown" end
    local ok = pcall(function() local t = {}; t[name] = true end)
    if not ok then return "Unknown" end
    local hex = GetClassColorHex(unit)
    return "|cff" .. hex .. name .. "|r"
end

local function GetPlayerKey(unit)
    local name, realm = UnitName(unit)
    if not name then return nil end
    -- Detect secret values (instanced PvP taint) — secrets can't be used as table keys
    local ok = pcall(function() local t = {}; t[name] = true end)
    if not ok then return nil end
    realm = (realm and realm ~= "") and realm or GetNormalizedRealmName()
    if not realm then return name end
    return name .. "-" .. realm
end

local function GetCachedData(guid)
    if not guid or not SKToolsPvPTitlesDB then return nil end
    local key = guidToKey[guid]
    if not key then return nil end
    return SKToolsPvPTitlesDB.players[key]
end

local CACHE_VERSION = 6  -- bump: added Midnight S1 achievement IDs

function ns.PvPTitles_RestoreCache()
    local db = SKToolsPvPTitlesDB
    if not db or not db.players then return end

    -- Wipe cache if version is outdated
    if (db.version or 1) < CACHE_VERSION then
        wipe(db.players)
        wipe(db.guidToKey)
        db.version = CACHE_VERSION
        print("|cff00E5EESKTools:|r PvP title cache cleared (updated achievement data).")
        return
    end

    for key, entry in pairs(db.players) do
        if entry.guid then
            guidToKey[entry.guid] = key
        end
    end
end

-----------------------------
-- Inspect Queue
-----------------------------

-- Find a unit token by GUID — prioritize stable tokens (target/focus) over transient ones
local function FindUnitByGUID(guid)
    -- Check stable tokens first (don't change when you move your mouse)
    for _, token in ipairs({"target", "focus", "mouseover"}) do
        if UnitExists(token) then
            local ok, unitGUID = pcall(UnitGUID, token)
            if ok and unitGUID == guid then
                return token, C_NamePlate.GetNamePlateForUnit(token)
            end
        end
    end
    -- Fall back to nameplate tokens
    for _, plate in ipairs(C_NamePlate.GetNamePlates()) do
        local unit = plate.namePlateUnitToken
        if unit then
            local ok, unitGUID = pcall(UnitGUID, unit)
            if ok and unitGUID == guid then
                return unit, plate
            end
        end
    end
    return nil, nil
end

local function ResetInspect()
    inspectState = "IDLE"
    inspectUnit = nil
    inspectGUID = nil
    inspectPlate = nil
    inspectPlayerKey = nil
    inspectColorName = nil
    if inspectTimer then
        inspectTimer:Cancel()
        inspectTimer = nil
    end
    pcall(ClearAchievementComparisonUnit)
    pcall(ClearInspectPlayer)
end

local function ProcessPending()  -- forward declared, defined below
end

-- Try to silently retry a failed scan. Returns true if retry was queued.
local function TryRetry(guid, reason)
    if inspectRetries < MAX_RETRIES then
        local unit = FindUnitByGUID(guid)
        if unit then
            inspectRetries = inspectRetries + 1
            local savedKey, savedColor = inspectPlayerKey, inspectColorName
            ResetInspect()
            inspectPlayerKey = savedKey
            inspectColorName = savedColor
            pendingGUID = guid
            ProcessPending()
            return true
        end
    end
    -- Final failure
    print("|cff00E5EESKTools:|r |cffFF5555Scan failed:|r " .. reason .. ".")
    failedGUIDs[guid] = GetTime()
    inspectRetries = 0
    ResetInspect()
    ProcessPending()
    return false
end

local pendingTimer = nil  -- guard against overlapping cooldown timers

local function StartInspect(unit, guid)
    -- Verify unit still matches GUID before starting (pcall: may be tainted in PvP)
    local ok2, currentGUID = pcall(UnitGUID, unit)
    if not ok2 or currentGUID ~= guid then
        ResetInspect()
        return
    end

    -- Verify unit is inspectable
    local ok3, canInspect = pcall(CanInspect, unit)
    if not ok3 or not canInspect then
        ResetInspect()
        return
    end

    -- Range check — fail fast instead of waiting for timeout
    local okVis, isVisible = pcall(UnitIsVisible, unit)
    if not okVis or not isVisible then
        local colorName = ClassColorName(unit)
        print("|cff00E5EESKTools:|r |cffFF5555Scan failed:|r " .. colorName .. " is out of range.")
        failedGUIDs[guid] = GetTime()
        ResetInspect()
        return
    end

    inspectState = "INSPECTING"
    inspectUnit = unit
    inspectGUID = guid
    inspectPlate = nil  -- don't store stale plate ref, look it up fresh when needed
    lastInspectTime = GetTime()

    -- Store playerKey and class-colored name early while unit token is still valid
    inspectPlayerKey = GetPlayerKey(unit) or "Unknown"
    inspectColorName = ClassColorName(unit)
    if inspectRetries == 0 then
        print("|cff00E5EESKTools:|r Scanning " .. inspectColorName .. "...")
        -- Play scope lock-on animation on the nameplate
        if not SKToolsDB or SKToolsDB.pvpTitlesScanAnim ~= false then
            local plate = C_NamePlate.GetNamePlateForUnit(unit)
            if plate then pcall(PlayScanAnim, plate) end
        end
    end

    local ok, err = pcall(NotifyInspect, unit)
    if not ok then
        ns.SK_ReportError("PvPTitles:NotifyInspect", err)
        TryRetry(guid, "could not inspect " .. (inspectColorName or inspectPlayerKey))
        return
    end

    inspectTimer = C_Timer.After(INSPECT_TIMEOUT, function()
        if inspectState ~= "IDLE" then
            local g = inspectGUID
            TryRetry(g, "scan timed out")
        end
    end)
end

ProcessPending = function()
    if inspectState ~= "IDLE" then return end
    if not pendingGUID then return end

    local now = GetTime()
    if now - lastInspectTime < INSPECT_COOLDOWN then
        -- Only schedule one cooldown timer at a time
        if not pendingTimer then
            pendingTimer = C_Timer.After(INSPECT_COOLDOWN - (now - lastInspectTime), function()
                pendingTimer = nil
                ProcessPending()
            end)
        end
        return
    end

    local guid = pendingGUID
    pendingGUID = nil

    -- Skip if it got cached while waiting
    if GetCachedData(guid) then return end

    -- Look up a fresh unit token for this GUID
    local unit = FindUnitByGUID(guid)
    if not unit then return end

    local ok, canInspect = pcall(CanInspect, unit)
    if not ok or not canInspect then return end

    StartInspect(unit, guid)
end

local function QueueInspect(guid)
    -- Skip if already cached
    if GetCachedData(guid) then return end
    -- Skip if recently failed (prevent spam)
    local failedAt = failedGUIDs[guid]
    if failedAt and (GetTime() - failedAt) < FAILED_COOLDOWN then return end
    -- Skip if currently inspecting this GUID
    if inspectGUID == guid then return end
    -- Replace any pending scan with this one (single slot)
    pendingGUID = guid
    -- Cancel stale cooldown timer so ProcessPending picks up the new GUID
    if pendingTimer then
        pendingTimer:Cancel()
        pendingTimer = nil
    end
    ProcessPending()
end

-----------------------------
-- Achievement Reading
-----------------------------
local function ReadAchievements(unit, guid)
    -- Verify the unit still points to the expected player (pcall: may be tainted in PvP)
    local ok2, currentGUID = pcall(UnitGUID, unit)
    if not ok2 or currentGUID ~= guid then return nil, "player moved or left range" end
    -- Reject if unit is the local player (comparison API would read own data)
    if UnitIsUnit(unit, "player") then return nil, "cannot scan yourself" end
    local playerKey = GetPlayerKey(unit)
    if not playerKey then return nil, "could not identify player" end

    -- Count Rank 1 (3v3) achievements
    local r1Count = 0
    for _, achieveID in ipairs(RANK1_ACHIEVEMENTS) do
        local ok, completed = pcall(GetAchievementComparisonInfo, achieveID)
        if ok and completed then r1Count = r1Count + 1 end
    end

    -- Count Solo Shuffle R1 achievements
    local ssR1Count = 0
    for _, achieveID in ipairs(SOLOSHUFFLE_R1_ACHIEVEMENTS) do
        local ok, completed = pcall(GetAchievementComparisonInfo, achieveID)
        if ok and completed then ssR1Count = ssR1Count + 1 end
    end

    -- Count BG Blitz R1 (Marshal) achievements
    local blitzR1Count = 0
    for _, achieveID in ipairs(BLITZ_R1_ACHIEVEMENTS) do
        local ok, completed = pcall(GetAchievementComparisonInfo, achieveID)
        if ok and completed then blitzR1Count = blitzR1Count + 1 end
    end

    -- Count Gladiator achievements
    local gladCount = 0
    for _, achieveID in ipairs(GLAD_ACHIEVEMENTS) do
        local ok, completed = pcall(GetAchievementComparisonInfo, achieveID)
        if ok and completed then
            gladCount = gladCount + 1
        end
    end

    -- Determine highest rank
    local highestRank = "none"
    if r1Count > 0 then
        highestRank = "r1"
    elseif gladCount > 0 then
        highestRank = "gladiator"
    else
        local okH, heroH = pcall(GetAchievementComparisonInfo, HERO_HORDE_ACHIEVEMENT)
        local okA, heroA = pcall(GetAchievementComparisonInfo, HERO_ALLIANCE_ACHIEVEMENT)
        if (okH and heroH) or (okA and heroA) then
            highestRank = "hero"
        else
            local okD, duel = pcall(GetAchievementComparisonInfo, DUELIST_ACHIEVEMENT)
            if okD and duel then
                highestRank = "duelist"
            else
                local okR, rival = pcall(GetAchievementComparisonInfo, RIVAL_ACHIEVEMENT)
                if okR and rival then
                    highestRank = "rival"
                else
                    local okC, chal = pcall(GetAchievementComparisonInfo, CHALLENGER_ACHIEVEMENT)
                    if okC and chal then
                        highestRank = "challenger"
                    end
                end
            end
        end
    end

    -- Read rating statistics
    local highest2v2, highest3v3 = 0, 0
    local ok6, val2v2 = pcall(GetComparisonStatistic, STAT_HIGHEST_2V2)
    if ok6 and val2v2 then highest2v2 = tonumber(val2v2) or 0 end
    local ok7, val3v3 = pcall(GetComparisonStatistic, STAT_HIGHEST_3V3)
    if ok7 and val3v3 then highest3v3 = tonumber(val3v3) or 0 end

    local entry = {
        r1Count = r1Count,
        ssR1Count = ssR1Count,
        blitzR1Count = blitzR1Count,
        gladCount = gladCount,
        highestRank = highestRank,
        highest2v2 = highest2v2,
        highest3v3 = highest3v3,
        classColorHex = GetClassColorHex(unit),
        timestamp = time(),
        guid = guid,
    }

    SKToolsPvPTitlesDB.players[playerKey] = entry
    SKToolsPvPTitlesDB.guidToKey[guid] = playerKey
    guidToKey[guid] = playerKey

    return entry
end

-----------------------------
-- Nameplate Overlay Display
-----------------------------
local function GetOrCreateOverlay(plate)
    if pvpTitleOverlays[plate] then return pvpTitleOverlays[plate] end

    local parent = plate
    local uf = plate.UnitFrame
    if uf then
        local hb = uf.healthBar or uf.HealthBar
        if hb then parent = hb end
    end

    local overlay = CreateFrame("Frame", nil, parent)
    overlay:SetSize(120, BADGE_SIZE + 4)
    overlay:SetPoint("BOTTOM", plate, "TOP", 0, 2)
    overlay:SetFrameLevel(parent:GetFrameLevel() + 15)

    -- R1 badge (red helmet, red border)
    overlay.r1Badge = CreateCircularBadge(overlay, R1_ICON, 0.8, 0.15, 0.15)
    overlay.r1Badge.count:SetTextColor(1, 0.3, 0.3)

    -- Solo Shuffle R1 badge (green gladiator helmet, green border)
    overlay.ssR1Badge = CreateCircularBadge(overlay, SSR1_ICON, 0.15, 0.6, 0.15)
    overlay.ssR1Badge.count:SetTextColor(0.3, 1, 0.3)

    -- Glad badge (duelist helmet, blue border)
    overlay.gladBadge = CreateCircularBadge(overlay, GLAD_ICON, 0.15, 0.3, 0.8)
    overlay.gladBadge.count:SetTextColor(0.4, 0.6, 1)

    -- BG Blitz R1 badge (rival helmet, purple border)
    overlay.blitzR1Badge = CreateCircularBadge(overlay, BLITZ_R1_ICON, 0.6, 0.2, 0.8)
    overlay.blitzR1Badge.count:SetTextColor(0.8, 0.5, 1)

    -- 3k+ badge (challenger helmet, gold border)
    overlay.threekBadge = CreateCircularBadge(overlay, THREEK_ICON, 0.8, 0.65, 0.1)

    -- Murloc badge (no titles, white border)
    overlay.murlocBadge = CreateCircularBadge(overlay, MURLOC_ICON, 0.7, 0.7, 0.7)

    -- Fallback badge (for duelist/rival/etc — icon set dynamically)
    overlay.fallbackBadge = CreateCircularBadge(overlay, GLAD_ICON, 0.3, 0.3, 0.3)

    -- Tooltip
    overlay:EnableMouse(true)
    overlay:SetScript("OnEnter", function(self)
        if not self._data then return end
        local data = self._data
        GameTooltip:SetOwner(self, "ANCHOR_RIGHT")
        GameTooltip:AddLine("PvP Scanner", ns.CYAN.r, ns.CYAN.g, ns.CYAN.b)
        if data.r1Count > 0 then
            GameTooltip:AddLine("Rank 1 (3v3): " .. data.r1Count .. " season(s)", 1, 0.2, 0.2)
        end
        if (data.ssR1Count or 0) > 0 then
            GameTooltip:AddLine("Rank 1 (Solo Shuffle): " .. data.ssR1Count .. " season(s)", 0.3, 1, 0.3)
        end
        if (data.blitzR1Count or 0) > 0 then
            GameTooltip:AddLine("Rank 1 (BG Blitz): " .. data.blitzR1Count .. " season(s)", 0.8, 0.5, 1)
        end
        if data.gladCount > 0 then
            GameTooltip:AddLine("Gladiator: " .. data.gladCount .. " season(s)", 0.4, 0.6, 1)
        end
        if data.highestRank == "hero" then
            GameTooltip:AddLine("Hero of the Horde/Alliance", 1, 0.84, 0)
        elseif data.highestRank == "duelist" then
            GameTooltip:AddLine("Duelist", 0.2, 0.7, 0.9)
        elseif data.highestRank == "rival" then
            GameTooltip:AddLine("Rival", 0, 0.8, 0.4)
        elseif data.highestRank == "challenger" then
            GameTooltip:AddLine("Challenger", 0.6, 0.6, 0.6)
        end
        if (data.highest3v3 or 0) > 0 then
            GameTooltip:AddDoubleLine("Peak 3v3", tostring(data.highest3v3), 0.7, 0.7, 0.7, 1, 1, 1)
        end
        if (data.highest2v2 or 0) > 0 then
            GameTooltip:AddDoubleLine("Peak 2v2", tostring(data.highest2v2), 0.7, 0.7, 0.7, 1, 1, 1)
        end
        local peak = math.max(data.highest2v2 or 0, data.highest3v3 or 0)
        if peak >= 3000 then
            GameTooltip:AddLine("3000+ Peak Rating", 1, 0.84, 0)
        end
        if data.highestRank == "none" and data.r1Count == 0 and (data.ssR1Count or 0) == 0 and (data.blitzR1Count or 0) == 0 and data.gladCount == 0 and peak < 3000 then
            GameTooltip:AddLine("No notable PvP titles found", 0.5, 0.5, 0.5)
        end
        GameTooltip:Show()
    end)
    overlay:SetScript("OnLeave", function() GameTooltip:Hide() end)

    overlay:Hide()
    pvpTitleOverlays[plate] = overlay
    return overlay
end

local function SetBadgeCount(badge, count)
    if count > 1 then
        badge.count:SetText("x" .. count)
        badge.count:Show()
    else
        badge.count:Hide()
    end
end

local function UpdateDisplay(plate, guid)
    if not plate or not guid then return end

    local data = GetCachedData(guid)
    if not data then return end

    local peakRating = math.max(data.highest2v2 or 0, data.highest3v3 or 0)
    local hasR1 = data.r1Count > 0
    local hasSSR1 = (data.ssR1Count or 0) > 0
    local hasBlitzR1 = (data.blitzR1Count or 0) > 0
    local hasGlad = data.gladCount > 0
    local has3k = peakRating >= 3000
    local hasBadge = hasR1 or hasSSR1 or hasBlitzR1 or hasGlad or has3k

    local overlay = GetOrCreateOverlay(plate)
    overlay._data = data

    -- Hide all badges first
    overlay.r1Badge:Hide()
    overlay.ssR1Badge:Hide()
    overlay.blitzR1Badge:Hide()
    overlay.gladBadge:Hide()
    overlay.threekBadge:Hide()
    overlay.murlocBadge:Hide()
    overlay.fallbackBadge:Hide()
    overlay.fallbackBadge.label:Hide()

    if hasBadge then
        -- Collect active badges in display order: R1 > SS R1 > Blitz R1 > Glad > 3k+
        local badges = {}

        if hasR1 then
            SetBadgeCount(overlay.r1Badge, data.r1Count)
            badges[#badges + 1] = overlay.r1Badge
        end

        if hasSSR1 then
            SetBadgeCount(overlay.ssR1Badge, data.ssR1Count or 0)
            badges[#badges + 1] = overlay.ssR1Badge
        end

        if hasBlitzR1 then
            SetBadgeCount(overlay.blitzR1Badge, data.blitzR1Count or 0)
            badges[#badges + 1] = overlay.blitzR1Badge
        end

        if hasGlad then
            SetBadgeCount(overlay.gladBadge, data.gladCount)
            badges[#badges + 1] = overlay.gladBadge
        end

        if has3k then
            overlay.threekBadge.count:SetText("3k")
            overlay.threekBadge.count:SetTextColor(1, 0.84, 0)
            overlay.threekBadge.count:Show()
            badges[#badges + 1] = overlay.threekBadge
        end

        -- Center badges horizontally
        local totalWidth = #badges * BADGE_SIZE + (#badges - 1) * BADGE_SPACING
        local startX = -totalWidth / 2
        for i, badge in ipairs(badges) do
            badge:ClearAllPoints()
            badge:SetPoint("LEFT", overlay, "CENTER", startX + (i - 1) * (BADGE_SIZE + BADGE_SPACING), 0)
            badge:Show()
        end
    elseif data.highestRank ~= "none" then
        -- Fallback: single circular icon with label below
        local info = RANK_FALLBACK[data.highestRank]
        if info then
            local b = overlay.fallbackBadge
            b.icon:SetTexture(info.icon)
            b.ring:SetColorTexture(info.color[1] * 0.5, info.color[2] * 0.5, info.color[3] * 0.5, 0.9)
            b.label:SetText(info.label)
            b.label:SetTextColor(unpack(info.color))
            b.label:Show()
            b:ClearAllPoints()
            b:SetPoint("CENTER", overlay, "CENTER", 0, 4)
            b:Show()
        end
    else
        -- No titles at all — show murloc with white border
        local b = overlay.murlocBadge
        b:ClearAllPoints()
        b:SetPoint("CENTER", overlay, "CENTER", 0, 0)
        b:Show()
    end

    -- Respect Always Hide Badges and Hide in Combat settings
    if SKToolsDB and SKToolsDB.pvpTitlesHideBadges then return end
    if SKToolsDB and SKToolsDB.pvpTitlesHideInCombat and InCombatLockdown() then return end

    overlay:Show()
end

-----------------------------
-- PvP Instance Checks
-----------------------------
local function IsInPvPInstance()
    local _, instanceType = IsInInstance()
    return instanceType == "arena" or instanceType == "pvp"
end

local function IsInRatedPvP()
    if not IsInPvPInstance() then return false end
    return C_PvP and C_PvP.IsRatedMap and C_PvP.IsRatedMap()
end

-----------------------------
-- Chat Fallback Helpers
-----------------------------
local function PrintTitleSummary(playerKey, data)
    local parts = {}
    if data.r1Count > 0 then
        parts[#parts + 1] = "|cffFF3333R1 x" .. data.r1Count .. "|r"
    end
    if (data.ssR1Count or 0) > 0 then
        parts[#parts + 1] = "|cff33FF33SSR1 x" .. data.ssR1Count .. "|r"
    end
    if (data.blitzR1Count or 0) > 0 then
        parts[#parts + 1] = "|cffCC77FFBlitzR1 x" .. data.blitzR1Count .. "|r"
    end
    if data.gladCount > 0 then
        parts[#parts + 1] = "|cff5599FFGlad x" .. data.gladCount .. "|r"
    end
    -- Fallback rank with badge colors (only when no major badges)
    if #parts == 0 then
        if data.highestRank == "hero" then
            parts[#parts + 1] = "|cffFFD600Hero|r"
        elseif data.highestRank == "duelist" then
            parts[#parts + 1] = "|cff33B3E6Duelist|r"
        elseif data.highestRank == "rival" then
            parts[#parts + 1] = "|cff00CC66Rival|r"
        elseif data.highestRank == "challenger" then
            parts[#parts + 1] = "|cff999999Challenger|r"
        else
            parts[#parts + 1] = "|cff808080No PvP titles|r"
        end
    end
    -- Peak 3v3 rating
    local peak3v3 = data.highest3v3 or 0
    if peak3v3 > 0 then
        parts[#parts + 1] = "Peak 3v3: |cffFFFFFF" .. peak3v3 .. "|r"
    end
    local firstName = playerKey:match("^([^%-]+)") or playerKey
    local hex = (data.classColorHex and data.classColorHex ~= "") and data.classColorHex or "FFFFFF"
    print("|cff00E5EESKTools:|r |cff" .. hex .. firstName .. "|r — " .. table.concat(parts, ", "))
end

local function ShowNameplateHint(unit)
    if nameplateHintShown then return end
    nameplateHintShown = true
    if unit and UnitIsFriend(unit, "player") then
        print("|cff00E5EESKTools:|r |cffFFD633Tip:|r Enable friendly nameplates (default: |cffFFFFFFShift+V|r) to see PvP title badges above players.")
    else
        print("|cff00E5EESKTools:|r |cffFFD633Tip:|r Enable enemy nameplates (default: |cffFFFFFFV|r) to see PvP title badges above players.")
    end
end

-----------------------------
-- Event Handler
-----------------------------
local pvpTitleEventFrame = CreateFrame("Frame")

-- Shared scan trigger — called from both mouseover and target-change events
local function TriggerScan(unitToken, forceRefresh, aggressive)
    if not UnitIsPlayer(unitToken) then return end
    if UnitIsUnit(unitToken, "player") then return end

    -- pcall: UnitGUID may return secret/tainted values in instanced PvP
    local ok, guid = pcall(UnitGUID, unitToken)
    if not ok or not guid then return end

    -- Verify guid is usable as a table key (not tainted/secret)
    local testOk = pcall(function() local _ = printedGUIDs[guid] end)
    if not testOk then return end

    -- Aggressive mode (Ctrl+Click): clear cooldowns, cancel stale scans
    if aggressive then
        failedGUIDs[guid] = nil
        if inspectState ~= "IDLE" and inspectGUID ~= guid then
            inspectRetries = 0
            ResetInspect()
        end
    end

    if forceRefresh then
        -- Cancel any in-progress scan and retries
        inspectRetries = 0
        ResetInspect()
        -- Remove from cache and failure cooldown so QueueInspect doesn't skip it
        local playerKey = guidToKey[guid]
        if playerKey and SKToolsPvPTitlesDB then
            SKToolsPvPTitlesDB.players[playerKey] = nil
        end
        failedGUIDs[guid] = nil
    end

    -- Show cached data immediately if available (and not forcing)
    local cached = GetCachedData(guid)
    if cached and not forceRefresh then
        local plate = C_NamePlate.GetNamePlateForUnit(unitToken)
        if plate then
            UpdateDisplay(plate, guid)
        else
            -- No nameplate — print results to chat (once per player per session)
            if not printedGUIDs[guid] then
                local playerKey = guidToKey[guid] or GetPlayerKey(unitToken) or "Unknown"
                PrintTitleSummary(playerKey, cached)
                printedGUIDs[guid] = true
                ShowNameplateHint(unitToken)
            end
        end
        return
    end

    -- Queue inspect by GUID only — ProcessPending looks up fresh unit token
    QueueInspect(guid)
end

local function OnEvent(event, ...)
    if event == "UPDATE_MOUSEOVER_UNIT" then
        if not IsControlKeyDown() then return end
        TriggerScan("mouseover", IsShiftKeyDown(), false)

    elseif event == "PLAYER_TARGET_CHANGED" then
        if not IsControlKeyDown() then return end
        if not UnitExists("target") then return end
        TriggerScan("target", IsShiftKeyDown(), true)

    elseif event == "INSPECT_READY" then
        local inspectedGUID = (...)
        if inspectState ~= "INSPECTING" then return end
        if inspectedGUID ~= inspectGUID then return end

        -- Defer to clean execution context to prevent taint spreading to Blizzard UI
        local savedGUID = inspectGUID
        local savedUnit = inspectUnit
        C_Timer.After(0, function()
            if inspectState ~= "INSPECTING" or inspectGUID ~= savedGUID then return end

            -- Re-find unit by GUID, fall back to original token
            local freshUnit = FindUnitByGUID(savedGUID) or savedUnit
            inspectUnit = freshUnit

            inspectState = "COMPARING"
            local ok, err = pcall(SetAchievementComparisonUnit, inspectUnit)
            if not ok then
                ns.SK_ReportError("PvPTitles:SetAchievementComparison", err)
                TryRetry(savedGUID, "could not load achievements")
            end
        end)

    elseif event == "INSPECT_ACHIEVEMENT_READY" then
        if inspectState ~= "COMPARING" then return end

        -- Defer to clean execution context to prevent taint spreading to Blizzard UI
        local savedGUID = inspectGUID
        local savedUnit = inspectUnit
        local savedKey = inspectPlayerKey
        local savedColor = inspectColorName
        C_Timer.After(0, function()
            if inspectState ~= "COMPARING" or inspectGUID ~= savedGUID then return end

            -- Re-find unit by GUID, fall back to original token
            local freshUnit = FindUnitByGUID(savedGUID) or savedUnit
            inspectUnit = freshUnit

            inspectState = "READING"
            local playerKey = savedKey or GetPlayerKey(inspectUnit) or "Unknown"
            local ok, result, reason = pcall(ReadAchievements, inspectUnit, savedGUID)

            if ok and result then
                failedGUIDs[savedGUID] = nil
                inspectRetries = 0
                -- Try to display on nameplate
                local _, plate = FindUnitByGUID(savedGUID)
                if not plate then
                    plate = C_NamePlate.GetNamePlateForUnit(inspectUnit)
                end
                if not plate then
                    plate = C_NamePlate.GetNamePlateForUnit("mouseover")
                end
                if plate then
                    UpdateDisplay(plate, savedGUID)
                else
                    ShowNameplateHint(inspectUnit)
                end
                -- Print results to chat
                PrintTitleSummary(playerKey, result)
                printedGUIDs[savedGUID] = true
                ResetInspect()
                ProcessPending()
            else
                TryRetry(savedGUID, (savedColor or playerKey) .. " — failed to read achievements")
            end
        end)

    elseif event == "NAME_PLATE_UNIT_ADDED" then
        if not SKToolsDB or not SKToolsDB.pvpTitlesShowCached then return end
        if SKToolsDB.pvpTitlesHideBadges then return end
        if SKToolsDB.pvpTitlesHideInCombat and InCombatLockdown() then return end
        local unit = (...)
        if not UnitIsPlayer(unit) then return end

        -- pcall: UnitGUID/table lookups may hit tainted values in instanced PvP
        local ok, guid = pcall(UnitGUID, unit)
        if not ok or not guid then return end

        local cacheOk, cached = pcall(GetCachedData, guid)
        if not cacheOk or not cached then return end

        local plate = C_NamePlate.GetNamePlateForUnit(unit)
        if plate then
            pcall(UpdateDisplay, plate, guid)
        end

    elseif event == "NAME_PLATE_UNIT_REMOVED" then
        local unit = (...)
        local plate = C_NamePlate.GetNamePlateForUnit(unit)
        if plate and pvpTitleOverlays[plate] then
            pvpTitleOverlays[plate]:Hide()
            pvpTitleOverlays[plate] = nil
        end

    elseif event == "PLAYER_REGEN_DISABLED" then
        if SKToolsDB and SKToolsDB.pvpTitlesHideInCombat then
            for _, overlay in pairs(pvpTitleOverlays) do overlay:Hide() end
        end

    elseif event == "PLAYER_REGEN_ENABLED" then
        if SKToolsDB and SKToolsDB.pvpTitlesHideInCombat and SKToolsDB.pvpTitlesShowCached then
            -- Re-show cached badges on visible nameplates
            for _, plate in ipairs(C_NamePlate.GetNamePlates()) do
                local unit = plate.namePlateUnitToken
                if unit and UnitIsPlayer(unit) then
                    -- pcall: may hit tainted GUIDs in instanced PvP
                    local ok, guid = pcall(UnitGUID, unit)
                    if ok and guid then
                        local cacheOk, cached = pcall(GetCachedData, guid)
                        if cacheOk and cached then
                            pcall(UpdateDisplay, plate, guid)
                        end
                    end
                end
            end
        end

    elseif event == "PLAYER_ENTERING_WORLD" then
        -- Hide overlays and cancel inspects on zone transitions
        for _, overlay in pairs(pvpTitleOverlays) do overlay:Hide() end
        ResetInspect()
        pendingGUID = nil
        if pendingTimer then pendingTimer:Cancel(); pendingTimer = nil end
        wipe(printedGUIDs)
    end
end

-----------------------------
-- Feature Toggle
-----------------------------
function ns.SetPvPTitles(enabled)
    if enabled then
        pvpTitleEventFrame:RegisterEvent("UPDATE_MOUSEOVER_UNIT")
        pvpTitleEventFrame:RegisterEvent("PLAYER_TARGET_CHANGED")
        pvpTitleEventFrame:RegisterEvent("INSPECT_READY")
        pvpTitleEventFrame:RegisterEvent("INSPECT_ACHIEVEMENT_READY")
        pvpTitleEventFrame:RegisterEvent("NAME_PLATE_UNIT_ADDED")
        pvpTitleEventFrame:RegisterEvent("NAME_PLATE_UNIT_REMOVED")
        pvpTitleEventFrame:RegisterEvent("PLAYER_REGEN_DISABLED")
        pvpTitleEventFrame:RegisterEvent("PLAYER_REGEN_ENABLED")
        pvpTitleEventFrame:RegisterEvent("PLAYER_ENTERING_WORLD")
        pvpTitleEventFrame:SetScript("OnEvent", function(self, event, ...)
            local ok, err = pcall(OnEvent, event, ...)
            if not ok then ns.SK_ReportError("PvPTitles:" .. event, err) end
        end)
        -- Show cached data on nameplates that already exist (e.g. after /reload)
        if SKToolsDB and SKToolsDB.pvpTitlesShowCached then
            for _, plate in ipairs(C_NamePlate.GetNamePlates()) do
                local unit = plate.namePlateUnitToken
                if unit and UnitIsPlayer(unit) then
                    -- pcall: may hit tainted GUIDs in instanced PvP
                    local ok, guid = pcall(UnitGUID, unit)
                    if ok and guid then
                        local cacheOk, cached = pcall(GetCachedData, guid)
                        if cacheOk and cached then
                            pcall(UpdateDisplay, plate, guid)
                        end
                    end
                end
            end
        end
    else
        pvpTitleEventFrame:UnregisterAllEvents()
        pvpTitleEventFrame:SetScript("OnEvent", nil)
        ResetInspect()
        pendingGUID = nil
        for _, overlay in pairs(pvpTitleOverlays) do overlay:Hide() end
    end
end

-----------------------------
-- Season Progress UI
-----------------------------
local BAR_WIDTH = 200
local BAR_HEIGHT = 14

local function CreateProgressRow(parent, label, r, g, b)
    local row = CreateFrame("Frame", nil, parent)
    row:SetHeight(38)

    -- Label
    local text = row:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
    text:SetPoint("TOPLEFT", row, "TOPLEFT", 0, 0)
    text:SetTextColor(r, g, b)
    text:SetText(label)

    -- Progress bar track
    local bar = CreateFrame("Frame", nil, row, "BackdropTemplate")
    bar:SetSize(BAR_WIDTH, BAR_HEIGHT)
    bar:SetPoint("TOPLEFT", text, "BOTTOMLEFT", 0, -4)
    bar:SetBackdrop(ns.BACKDROP_CONTROL)
    bar:SetBackdropColor(0.08, 0.08, 0.10, 1.0)
    bar:SetBackdropBorderColor(0.20, 0.20, 0.24, 0.30)

    -- Fill texture
    local fill = bar:CreateTexture(nil, "ARTWORK")
    fill:SetPoint("TOPLEFT", bar, "TOPLEFT", 2, -2)
    fill:SetHeight(BAR_HEIGHT - 4)
    fill:SetWidth(1)
    fill:SetColorTexture(r, g, b, 0.8)
    fill:Hide()

    -- Count text (right of bar)
    local count = row:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
    count:SetPoint("LEFT", bar, "RIGHT", 8, 0)
    count:SetTextColor(0.85, 0.85, 0.87)

    -- Track button (anchored right of count text)
    local trackBtn = ns.CreateThemedButton(row, "Track", 50, 20, "secondary")
    trackBtn:SetPoint("LEFT", count, "RIGHT", 10, 0)
    trackBtn:Hide()

    -- Completed text (hidden by default, replaces bar area)
    local completed = row:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
    completed:SetPoint("TOPLEFT", text, "BOTTOMLEFT", 0, -4)
    completed:SetTextColor(0.2, 1.0, 0.4)
    completed:SetText("Completed!")
    completed:Hide()

    row.bar = bar
    row.fill = fill
    row.count = count
    row.completed = completed
    row.trackBtn = trackBtn

    function row:Update(achievementId)
        if not achievementId or achievementId == 0 then
            self:Hide()
            return
        end

        local ok, _, _, _, achCompleted = pcall(GetAchievementInfo, achievementId)
        if not ok then
            self:Hide()
            return
        end

        -- Read criteria for win count
        local cOk, _, _, _, quantity, reqQuantity = pcall(GetAchievementCriteriaInfo, achievementId, 1)
        local wins = (cOk and tonumber(quantity)) or 0
        local needed = (cOk and tonumber(reqQuantity)) or 50

        if achCompleted then
            self.bar:Hide()
            self.fill:Hide()
            self.count:Hide()
            self.completed:Show()
            self.trackBtn:Hide()
        else
            self.bar:Show()
            self.count:Show()
            self.completed:Hide()
            self.trackBtn:Show()

            local pct = (needed > 0) and (wins / needed) or 0
            pct = math.min(1, math.max(0, pct))
            if pct > 0 then
                self.fill:SetWidth(math.max(1, (BAR_WIDTH - 4) * pct))
                self.fill:Show()
            else
                self.fill:Hide()
            end

            self.count:SetText(wins .. " / " .. needed)

            self.trackBtn:SetScript("OnClick", function()
                pcall(C_ContentTracking.ToggleTracking, 2, achievementId, 2)
            end)
        end

        self:Show()
    end

    return row
end

local function BuildSeasonProgress(parent, anchor)
    local header = ns.AddSectionHeader(parent, anchor, "Season Progress", false)
    local card = ns.CreateSectionCard(parent)
    card:SetPoint("TOPLEFT", header, "TOPLEFT", -8, 8)

    -- "No active season" message (hidden by default, shown if season == 0)
    local noSeason = parent:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
    noSeason:SetPoint("TOPLEFT", header, "BOTTOMLEFT", 2, -8)
    noSeason:SetTextColor(0.50, 0.50, 0.55)
    noSeason:SetText("No active PvP season.")
    noSeason:Hide()

    -- Create all 3 rows upfront (hidden by default, shown when season is active)
    local gladRow = CreateProgressRow(parent, "Gladiator (3v3)", 0.2, 0.53, 1.0)
    gladRow:SetPoint("TOPLEFT", header, "BOTTOMLEFT", 2, -8)
    gladRow:SetPoint("RIGHT", parent, "RIGHT", -20, 0)
    gladRow:Hide()

    local legendRow = CreateProgressRow(parent, "Legend (Shuffle)", 0.2, 1.0, 0.6)
    legendRow:SetPoint("TOPLEFT", gladRow, "BOTTOMLEFT", 0, -4)
    legendRow:SetPoint("RIGHT", parent, "RIGHT", -20, 0)
    legendRow:Hide()

    local strategistRow = CreateProgressRow(parent, "Strategist (Blitz)", 0.67, 0.33, 1.0)
    strategistRow:SetPoint("TOPLEFT", legendRow, "BOTTOMLEFT", 0, -4)
    strategistRow:SetPoint("RIGHT", parent, "RIGHT", -20, 0)
    strategistRow:Hide()

    -- Refresh function: reads current season and updates all rows
    local function Refresh()
        local season = GetCurrentArenaSeason()
        if not season or season == 0 then
            noSeason:Show()
            gladRow:Hide()
            legendRow:Hide()
            strategistRow:Hide()
            card:SetPoint("BOTTOM", noSeason, "BOTTOM", 0, -8)
            return
        end

        noSeason:Hide()

        local gladId = SEASON_GLAD[season] or 0
        local legendId = SEASON_LEGEND[season] or 0
        local strategistId = SEASON_STRATEGIST[season] or 0

        gladRow:Update(gladId)
        legendRow:Update(legendId)
        strategistRow:Update(strategistId)

        -- Find last visible row for card sizing
        local lastVisible = header
        if strategistRow:IsShown() then lastVisible = strategistRow
        elseif legendRow:IsShown() then lastVisible = legendRow
        elseif gladRow:IsShown() then lastVisible = gladRow
        end
        card:SetPoint("BOTTOM", lastVisible, "BOTTOM", 0, -8)
    end

    -- Initial refresh
    Refresh()

    -- Re-refresh whenever the panel becomes visible (catches season changes, win updates)
    local scrollParent = parent:GetParent()
    if scrollParent then
        local oldOnShow = scrollParent:GetScript("OnShow")
        scrollParent:SetScript("OnShow", function(self, ...)
            if oldOnShow then oldOnShow(self, ...) end
            Refresh()
        end)
    end

    card:SetPoint("RIGHT", parent, "RIGHT", -8, 0)
    -- Return strategistRow as last element (always exists, even if hidden — for anchoring next sections)
    return strategistRow
end

-----------------------------
-- Settings Builder (Main Panel)
-----------------------------
function ns.BuildPvPTitleSettings(content, anchor)
    local header = ns.AddSectionHeader(content, anchor, "PvP Scanner", false)
    local card = ns.CreateSectionCard(content)
    card:SetPoint("TOPLEFT", header, "TOPLEFT", -8, 8)

    -- Tutorial text
    local tutorial = content:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
    tutorial:SetPoint("TOPLEFT", header, "BOTTOMLEFT", 2, -6)
    tutorial:SetPoint("RIGHT", content, "RIGHT", -20, 0)
    tutorial:SetJustifyH("LEFT")
    tutorial:SetSpacing(3)
    tutorial:SetTextColor(0.55, 0.55, 0.6)
    tutorial:SetText(
        "Scans nearby players for PvP achievements and displays badges above their nameplates.\n" ..
        "|cffBBBBBBCtrl + Click|r  Scan a player (most reliable)\n" ..
        "|cffBBBBBBCtrl + Mouseover|r  Scan a player\n" ..
        "|cffBBBBBBCtrl + Shift|r  Force re-scan (click or mouseover)\n" ..
        "Results are cached permanently so previously scanned players show badges instantly."
    )

    local AddCB, GetLast, SetLast = ns.MakeCheckboxFactory(content, tutorial)

    AddCB("PvPTitles", "pvpTitles", "Enable PvP Scanner",
        "Scan players for R1, Gladiator, Duelist, Rival, and other PvP achievements via Ctrl+mouseover.",
        ns.SetPvPTitles)

    local parentCB = _G["SKToolsPvPTitlesCB"]
    local parentAnchor = GetLast()

    -- Sub-toggles (indented children of Enable PvP Scanner)
    local SUB_INDENT = 24
    local SUB_DEFS = {
        { name = "PvPTitlesCached",    key = "pvpTitlesShowCached",   label = "Show Known Titles Automatically",
          desc = "Display cached PvP title badges above nameplates without requiring Ctrl+hover for previously inspected players." },
        { name = "PvPTitlesScanAnim",  key = "pvpTitlesScanAnim",     label = "Scan Animation",
          desc = "Show the scope lock-on animation on nameplates when scanning a player." },
        { name = "PvPTitlesHideCombat", key = "pvpTitlesHideInCombat", label = "Hide Badges in Combat",
          desc = "Hide all PvP title badges above nameplates while in combat. Badges reappear when combat ends." },
        { name = "PvPTitlesHideBadges", key = "pvpTitlesHideBadges",   label = "Always Hide Badges",
          desc = "Scanning still works and results print to chat, but no badges are displayed above nameplates." },
    }
    local subToggles = {}
    local lastSub = parentAnchor
    for i, def in ipairs(SUB_DEFS) do
        local cb = ns.CreateToggleSwitch(content, "SKTools" .. def.name .. "CB")
        cb:SetPoint("TOPLEFT", lastSub, "BOTTOMLEFT", i == 1 and SUB_INDENT or 0, -8)
        cb.Text:SetText(def.label)
        cb.Text:SetTextColor(0.8, 0.8, 0.8)
        cb:SetChecked(SKToolsDB[def.key])
        cb:SetScript("OnClick", function(self)
            SKToolsDB[def.key] = self:GetChecked()
            if def.key == "pvpTitlesHideBadges" and self:GetChecked() then
                for _, overlay in pairs(pvpTitleOverlays) do overlay:Hide() end
            end
            if ns.RefreshMainNavStatus then ns.RefreshMainNavStatus() end
        end)
        local desc = content:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        desc:SetPoint("TOPLEFT", cb.Text, "BOTTOMLEFT", 0, -2)
        desc:SetPoint("RIGHT", content, "RIGHT", -16, 0)
        desc:SetJustifyH("LEFT")
        desc:SetTextColor(0.55, 0.55, 0.6)
        desc:SetText(def.desc)
        subToggles[#subToggles + 1] = cb
        local spacer = CreateFrame("Frame", nil, content)
        spacer:SetSize(1, 1)
        spacer:SetPoint("LEFT", cb, "LEFT", 0, 0)
        spacer:SetPoint("TOP", desc, "BOTTOM", 0, 0)
        lastSub = spacer
    end

    -- Connecting line from parent to sub-toggles
    local connLine = content:CreateTexture(nil, "ARTWORK")
    connLine:SetWidth(1)
    connLine:SetPoint("TOP", subToggles[1], "TOPLEFT", -8, 4)
    connLine:SetPoint("BOTTOM", subToggles[#subToggles], "LEFT", -8, 0)
    connLine:SetColorTexture(ns.COLORS.divider[1], ns.COLORS.divider[2], ns.COLORS.divider[3], 0.30)

    -- Enable/disable sub-toggles based on parent
    local function UpdatePvPChildState()
        local enabled = parentCB:GetChecked()
        for _, cb in ipairs(subToggles) do cb:SetEnabled(enabled) end
    end
    parentCB:HookScript("OnClick", UpdatePvPChildState)
    ns.RefreshMainPvPChildState = UpdatePvPChildState
    UpdatePvPChildState()

    -- Return to parent's left edge
    local subReturnSpacer = CreateFrame("Frame", nil, content)
    subReturnSpacer:SetSize(1, 1)
    subReturnSpacer:SetPoint("LEFT", parentCB, "LEFT", 0, 0)
    subReturnSpacer:SetPoint("TOP", lastSub, "TOP", 0, 0)
    SetLast(subReturnSpacer)

    -- Clear Cache button
    local clearBtn = ns.CreateThemedButton(content, "Clear Title Cache", 130, 22, "danger")
    clearBtn:SetPoint("TOPLEFT", GetLast(), "BOTTOMLEFT", 0, -8)
    clearBtn:SetScript("OnClick", function()
        if SKToolsPvPTitlesDB then
            wipe(SKToolsPvPTitlesDB.players)
            wipe(SKToolsPvPTitlesDB.guidToKey)
            wipe(guidToKey)
            wipe(printedGUIDs)
            for _, overlay in pairs(pvpTitleOverlays) do overlay:Hide() end
            print("|cff00E5EESKTools:|r PvP title cache cleared.")
        end
    end)
    SetLast(clearBtn)

    card:SetPoint("BOTTOM", GetLast(), "BOTTOM", 0, -8)
    card:SetPoint("RIGHT", content, "RIGHT", -8, 0)

    -- Season Progress section
    local progressLast = BuildSeasonProgress(content, GetLast())
    return progressLast
end

-----------------------------
-- Settings Builder (Combat Panel)
-----------------------------
function ns.BuildPvPTitleCombatSettings(content, anchor, csSyncControls)
    local header = ns.AddSectionHeader(content, anchor, "PvP Scanner", false)
    local card = ns.CreateSectionCard(content)
    card:SetPoint("TOPLEFT", header, "TOPLEFT", -8, 8)

    -- Tutorial text
    local tutorial = content:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
    tutorial:SetPoint("TOPLEFT", header, "BOTTOMLEFT", 2, -6)
    tutorial:SetPoint("RIGHT", content, "RIGHT", -20, 0)
    tutorial:SetJustifyH("LEFT")
    tutorial:SetSpacing(3)
    tutorial:SetTextColor(0.55, 0.55, 0.6)
    tutorial:SetText(
        "Scans nearby players for PvP achievements and displays badges above their nameplates.\n" ..
        "|cffBBBBBBCtrl + Click|r  Scan a player (most reliable)\n" ..
        "|cffBBBBBBCtrl + Mouseover|r  Scan a player\n" ..
        "|cffBBBBBBCtrl + Shift|r  Force re-scan (click or mouseover)\n" ..
        "Results are cached permanently so previously scanned players show badges instantly."
    )

    local AddCB, GetLast, SetLast = ns.MakeCombatCBFactory(content, tutorial, csSyncControls)

    local csParentCB = AddCB("PvPTitles", "pvpTitles", "Enable PvP Scanner",
        "Scan players for R1, Gladiator, Duelist, Rival, and other PvP achievements via Ctrl+mouseover.",
        ns.SetPvPTitles)

    local csParentAnchor = GetLast()

    -- Sub-toggles (indented children of Enable PvP Scanner)
    local CS_SUB_INDENT = 24
    local CS_SUB_DEFS = {
        { name = "PvPTitlesCached",    key = "pvpTitlesShowCached",   label = "Show Known Titles Automatically",
          desc = "Display cached PvP title badges above nameplates without requiring Ctrl+hover for previously inspected players." },
        { name = "PvPTitlesScanAnim",  key = "pvpTitlesScanAnim",     label = "Scan Animation",
          desc = "Show the scope lock-on animation on nameplates when scanning a player." },
        { name = "PvPTitlesHideCombat", key = "pvpTitlesHideInCombat", label = "Hide Badges in Combat",
          desc = "Hide all PvP title badges above nameplates while in combat. Badges reappear when combat ends." },
        { name = "PvPTitlesHideBadges", key = "pvpTitlesHideBadges",   label = "Always Hide Badges",
          desc = "Scanning still works and results print to chat, but no badges are displayed above nameplates." },
    }
    local csSubToggles = {}
    local csLastSub = csParentAnchor
    for i, def in ipairs(CS_SUB_DEFS) do
        local cb = ns.CreateToggleSwitch(content, "SKToolsCS_" .. def.name .. "CB")
        cb:SetPoint("TOPLEFT", csLastSub, "BOTTOMLEFT", i == 1 and CS_SUB_INDENT or 0, -8)
        cb.Text:SetText(def.label)
        cb.Text:SetTextColor(0.8, 0.8, 0.8)
        cb:SetChecked(SKToolsDB[def.key])
        cb:SetScript("OnClick", function(self)
            SKToolsDB[def.key] = self:GetChecked()
            if def.key == "pvpTitlesHideBadges" and self:GetChecked() then
                for _, overlay in pairs(pvpTitleOverlays) do overlay:Hide() end
            end
            if ns.RefreshCombatNavStatus then ns.RefreshCombatNavStatus() end
        end)
        local desc = content:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        desc:SetPoint("TOPLEFT", cb.Text, "BOTTOMLEFT", 0, -2)
        desc:SetPoint("RIGHT", content, "RIGHT", -16, 0)
        desc:SetJustifyH("LEFT")
        desc:SetTextColor(0.55, 0.55, 0.6)
        desc:SetText(def.desc)
        table.insert(csSyncControls, { type = "checkbox", widget = cb, key = def.key })
        csSubToggles[#csSubToggles + 1] = cb
        local spacer = CreateFrame("Frame", nil, content)
        spacer:SetSize(1, 1)
        spacer:SetPoint("LEFT", cb, "LEFT", 0, 0)
        spacer:SetPoint("TOP", desc, "BOTTOM", 0, 0)
        csLastSub = spacer
    end

    -- Connecting line
    local csConnLine = content:CreateTexture(nil, "ARTWORK")
    csConnLine:SetWidth(1)
    csConnLine:SetPoint("TOP", csSubToggles[1], "TOPLEFT", -8, 4)
    csConnLine:SetPoint("BOTTOM", csSubToggles[#csSubToggles], "LEFT", -8, 0)
    csConnLine:SetColorTexture(ns.COLORS.divider[1], ns.COLORS.divider[2], ns.COLORS.divider[3], 0.30)

    -- Enable/disable sub-toggles based on parent
    local function UpdateCSPvPChildState()
        local enabled = csParentCB:GetChecked()
        for _, cb in ipairs(csSubToggles) do cb:SetEnabled(enabled) end
    end
    csParentCB:HookScript("OnClick", UpdateCSPvPChildState)
    table.insert(csSyncControls, { refresh = UpdateCSPvPChildState })
    UpdateCSPvPChildState()

    -- Return to parent's left edge
    local csSubReturn = CreateFrame("Frame", nil, content)
    csSubReturn:SetSize(1, 1)
    csSubReturn:SetPoint("LEFT", csParentCB, "LEFT", 0, 0)
    csSubReturn:SetPoint("TOP", csLastSub, "TOP", 0, 0)
    SetLast(csSubReturn)

    -- Clear Cache button
    local clearBtn = ns.CreateThemedButton(content, "Clear Title Cache", 130, 22, "danger")
    clearBtn:SetPoint("TOPLEFT", GetLast(), "BOTTOMLEFT", 0, -8)
    clearBtn:SetScript("OnClick", function()
        if SKToolsPvPTitlesDB then
            wipe(SKToolsPvPTitlesDB.players)
            wipe(SKToolsPvPTitlesDB.guidToKey)
            wipe(guidToKey)
            wipe(printedGUIDs)
            for _, overlay in pairs(pvpTitleOverlays) do overlay:Hide() end
            print("|cff00E5EESKTools:|r PvP title cache cleared.")
        end
    end)
    SetLast(clearBtn)

    card:SetPoint("BOTTOM", GetLast(), "BOTTOM", 0, -8)
    card:SetPoint("RIGHT", content, "RIGHT", -8, 0)

    -- Season Progress section
    local progressLast = BuildSeasonProgress(content, GetLast())
    return progressLast
end
