-- SKTools PvP Match History
-- Tracks all PvP match results with a standalone UI

local _, ns = ...

local MAX_MATCHES = 5000
local ROW_HEIGHT = 26
local VISIBLE_ROWS = 22
local HEADER_HEIGHT = 28
local CYAN = { r = 0, g = 0.898, b = 0.933 }  -- #00E5EE

local sessionWins, sessionLosses, sessionRating = 0, 0, 0
local historyFrame, scrollFrame, rows, filterButtons, viewButtons
local activeFilter = "All"
local activeView = "matches"  -- "matches", "winrates", "analysis", "players", "graph"
local sortColumn = "timestamp"   -- default sort by date
local sortAscending = false      -- newest first by default
local ShowView, RefreshStatsPanel  -- forward declarations
local RefreshGraphPanel  -- forward declarations
local RefreshAnalysisPanel  -- forward declaration
local RefreshPlayersPanel   -- forward declaration
local UpdateFilterLabel  -- forward declaration
local playersSortCol = "lastSeen"  -- default sort column for Players tab
local playersSortAsc = false       -- descending by default (most recent first)
local playersSearchText = ""       -- search filter for Players tab
-- analysisCompFilter removed — "My Group" now works through spec pills
local activeCharFilter = nil       -- nil = all characters, or "Name-Realm"
local graphSpecFilter = nil        -- nil = all specs, or spec name like "Elemental"
local graphDateFilter = nil        -- nil = all time, or key like "today","week","month","season","custom"
local graphDateCustomStart = nil   -- custom date range start (epoch)
local graphDateCustomEnd = nil     -- custom date range end (epoch)
local graphHiddenModes = {}        -- { [modeKey] = true } for toggled-off legend entries
local playersCacheDirty = true     -- when true, re-aggregate player stats from scratch
local playersCachedFilter = nil    -- charFilter the cache was built with
local playersCachedStats = nil     -- aggregated playerStats hash
local winratesSortCol = "total"    -- win rates sort column
local winratesSortAsc = false      -- win rates sort direction
local compSpecFilters = {}         -- { ["Arms WARRIOR"] = true, ... } active spec pill filters
local analysisSpecFilters = {}     -- { ["Arms WARRIOR"] = true, ... } analysis tab spec pill filters
local analysisSubTab = "kills"     -- "kills" or "deaths"
local analysisSortCol = "games"    -- default sort column for Analysis tab
local analysisSortAsc = false      -- descending by default (most games first)
local compCacheDirty = true        -- when true, re-aggregate comp stats from scratch
local compCachedFilter = nil       -- filter the cache was built with
local compCachedCharFilter = nil   -- charFilter the cache was built with
local compCachedList = nil         -- cached raw compList (pre-pill-filter, pre-sort)

-- Helper: get current player's "Name-Realm" string
local function GetPlayerFullName()
    local name, realm = UnitFullName("player")
    if not name then return nil end
    if not realm or realm == "" then realm = GetNormalizedRealmName() or "" end
    return name .. "-" .. realm
end

-- Arena death tracking (polls party + arena units — works within Midnight API)
local arenaStartTime = nil
local arenaDeaths = {}  -- array of { name, class, unit, time, team }
local arenaDeathTicker = nil

-- Class color fallback
local CLASS_COLORS = {
    WARRIOR     = { r = 0.78, g = 0.61, b = 0.43 },
    PALADIN     = { r = 0.96, g = 0.55, b = 0.73 },
    HUNTER      = { r = 0.67, g = 0.83, b = 0.45 },
    ROGUE       = { r = 1.00, g = 0.96, b = 0.41 },
    PRIEST      = { r = 1.00, g = 1.00, b = 1.00 },
    DEATHKNIGHT = { r = 0.77, g = 0.12, b = 0.23 },
    SHAMAN      = { r = 0.00, g = 0.44, b = 0.87 },
    MAGE        = { r = 0.25, g = 0.78, b = 0.92 },
    WARLOCK     = { r = 0.53, g = 0.53, b = 0.93 },
    MONK        = { r = 0.00, g = 1.00, b = 0.60 },
    DRUID       = { r = 1.00, g = 0.49, b = 0.04 },
    DEMONHUNTER = { r = 0.64, g = 0.19, b = 0.79 },
    EVOKER      = { r = 0.20, g = 0.58, b = 0.50 },
}

local function GetClassColor(classFile)
    if RAID_CLASS_COLORS and RAID_CLASS_COLORS[classFile] then
        local c = RAID_CLASS_COLORS[classFile]
        return c.r, c.g, c.b
    end
    local c = CLASS_COLORS[classFile]
    if c then return c.r, c.g, c.b end
    return 0.5, 0.5, 0.5
end

local function GetClassColorHex(classFile)
    local r, g, b = GetClassColor(classFile)
    return string.format("%02x%02x%02x", r * 255, g * 255, b * 255)
end

-- Class icon atlas names (WoW built-in atlas textures)
local CLASS_ICON_ATLAS = {
    WARRIOR     = "classicon-warrior",
    PALADIN     = "classicon-paladin",
    HUNTER      = "classicon-hunter",
    ROGUE       = "classicon-rogue",
    PRIEST      = "classicon-priest",
    DEATHKNIGHT = "classicon-deathknight",
    SHAMAN      = "classicon-shaman",
    MAGE        = "classicon-mage",
    WARLOCK     = "classicon-warlock",
    MONK        = "classicon-monk",
    DRUID       = "classicon-druid",
    DEMONHUNTER = "classicon-demonhunter",
    EVOKER      = "classicon-evoker",
}

-- Get inline atlas icon string for embedding in font strings
local function ClassIconString(classFile, size)
    size = size or 14
    local atlas = CLASS_ICON_ATLAS[classFile]
    if atlas then
        return CreateAtlasMarkup(atlas, size, size)
    end
    return ""
end

-- Spec icon cache: classToken|specName → icon fileID
local specIconCache
local function EnsureSpecIconCache()
    if specIconCache then return end
    specIconCache = {}
    for classID = 1, GetNumClasses() do
        local _, classFile = GetClassInfo(classID)
        if classFile then
            local numSpecs = GetNumSpecializationsForClassID(classID)
            for specIdx = 1, numSpecs do
                local _, specName, _, specIcon = GetSpecializationInfoForClassID(classID, specIdx)
                if specName and specIcon then
                    specIconCache[classFile .. "|" .. specName] = specIcon
                end
            end
        end
    end
end

local function SpecIconString(classToken, specName, size)
    if not classToken or not specName or specName == "" then return "" end
    size = size or 14
    EnsureSpecIconCache()
    local icon = specIconCache[classToken .. "|" .. specName]
    if icon then
        return "|T" .. icon .. ":" .. size .. ":" .. size .. "|t"
    end
    return ""
end

-- Race icon support
-- Build runtime lookup: raceName → clientFileString using C_CreatureInfo
local raceNameToClientFile
local function EnsureRaceCache()
    if raceNameToClientFile then return end
    raceNameToClientFile = {}
    if not C_CreatureInfo or not C_CreatureInfo.GetRaceInfo then return end
    for raceID = 1, 100 do
        local info = C_CreatureInfo.GetRaceInfo(raceID)
        if info and info.clientFileString then
            local cfs = info.clientFileString
            local low = cfs:lower()
            -- Map by raceName (localized)
            if info.raceName then
                raceNameToClientFile[info.raceName] = low
            end
            -- Map by clientFileString (both casings)
            raceNameToClientFile[cfs] = low
            raceNameToClientFile[low] = low
        end
    end
end

-- Allied/compound race names use shortened atlas keys
local RACE_SHORT_KEY = {
    ["highmountaintauren"]  = "highmountain",
    ["lightforgeddraenei"]  = "lightforged",
    ["magharorc"]           = "maghar",
    ["kultiran"]            = "kultianhuman",
    ["zandalaritroll"]      = "zandalari",
    ["earthendwarf"]        = "earthen",
}

-- Try multiple atlas formats at runtime, cache which one works
local raceAtlasCache = {}
local RACE_ATLAS_FORMATS = {
    "raceicon128-%s%s",
    "raceicon-%s-%s",
    "raceicon-%s%s",
    "raceicon128-%s-%s",
    "raceicon64-%s%s",
    "raceicon64-%s-%s",
}

local function FindRaceAtlas(raceKey, gender)
    local cacheKey = raceKey .. "|" .. gender
    if raceAtlasCache[cacheKey] ~= nil then
        return raceAtlasCache[cacheKey]
    end
    if C_Texture and C_Texture.GetAtlasInfo then
        for _, fmt in ipairs(RACE_ATLAS_FORMATS) do
            local atlas = string.format(fmt, raceKey, gender)
            if C_Texture.GetAtlasInfo(atlas) then
                raceAtlasCache[cacheKey] = atlas
                return atlas
            end
        end
    end
    raceAtlasCache[cacheKey] = false
    return false
end

local function RaceIconString(raceFile, raceName, sex, size)
    size = size or 20
    EnsureRaceCache()
    local gender = (sex == 3) and "female" or "male"

    -- Build list of keys to try (most specific → least specific)
    local keysToTry = {}
    -- 1. Stored raceFile from GetPlayerInfoByGUID (e.g. "scourge")
    if raceFile and raceFile ~= "" then
        keysToTry[#keysToTry + 1] = raceFile
        -- Also try allied race short name
        local short = RACE_SHORT_KEY[raceFile]
        if short then keysToTry[#keysToTry + 1] = short end
    end
    -- 2. C_CreatureInfo lookup by raceName (e.g. "Undead" → "scourge")
    if raceName and raceName ~= "" then
        local fromCreatureInfo = raceNameToClientFile[raceName]
        if fromCreatureInfo then
            keysToTry[#keysToTry + 1] = fromCreatureInfo
            local short = RACE_SHORT_KEY[fromCreatureInfo]
            if short then keysToTry[#keysToTry + 1] = short end
        end
        -- 3. Direct lowercase+strip of raceName (e.g. "Undead" → "undead")
        local direct = raceName:lower():gsub("%s+", ""):gsub("'", "")
        keysToTry[#keysToTry + 1] = direct
        local short = RACE_SHORT_KEY[direct]
        if short then keysToTry[#keysToTry + 1] = short end
    end

    -- Deduplicate and try each key
    local tried = {}
    for _, key in ipairs(keysToTry) do
        if not tried[key] then
            tried[key] = true
            local atlas = FindRaceAtlas(key, gender)
            if atlas then
                return CreateAtlasMarkup(atlas, size, size)
            end
        end
    end
    return ""
end

-----------------------------
-- Mode Detection
-----------------------------
local BRACKET_MODES = {
    [0] = "arena2v2",
    [1] = "arena3v3",
    [2] = "rbg",
    [3] = "shuffle",
    [4] = "blitz",
}

local MODE_LABELS = {
    arena2v2  = "2v2",
    arena3v3  = "3v3",
    shuffle   = "Shuffle",
    blitz     = "Blitz",
    rbg       = "RBG",
    randomBG  = "Random BG",
    epicBG    = "Epic BG",
    skirmish  = "Skirmish",
}

local FILTER_MAP = {
    All      = nil,
    ["2v2"]  = { arena2v2 = true },
    ["3v3"]  = { arena3v3 = true },
    Shuffle  = { shuffle = true },
    Blitz    = { blitz = true },
    RBG      = { rbg = true },
    Skirm    = { skirmish = true },
    BG       = { randomBG = true },
    ["Epic BG"] = { epicBG = true },
}

local function DetectMode()
    local bracket = C_PvP.GetActiveMatchBracket()

    if IsActiveBattlefieldArena() then
        -- Only specific brackets are rated arena
        if bracket == 0 then return "arena2v2", true end
        if bracket == 1 then return "arena3v3", true end
        if bracket == 3 then return "shuffle", true end
        -- Any other bracket in arena = skirmish (unrated)
        return "skirmish", false
    end

    -- Non-arena: rated BG brackets
    if bracket == 2 then return "rbg", true end
    if bracket == 4 then return "blitz", true end

    -- Unrated BG: distinguish by player count
    local numScores = GetNumBattlefieldScores()
    if numScores and numScores > 30 then
        return "epicBG", false
    end
    return "randomBG", false
end

-----------------------------
-- Data Collection
-----------------------------
local function CollectMatchData()
    if not SKToolsArenaDB then return false end

    local mode, isRated = DetectMode()
    -- GetRealZoneText returns the actual BG/arena map name (e.g. "Warsong Gulch")
    -- GetInstanceInfo can return parent zone names instead
    local mapName = GetRealZoneText()
    if not mapName or mapName == "" then
        local mapID = C_Map and C_Map.GetBestMapForUnit and C_Map.GetBestMapForUnit("player")
        if mapID then
            local info = C_Map.GetMapInfo(mapID)
            if info and info.name and info.name ~= "" then mapName = info.name end
        end
    end
    if not mapName or mapName == "" then
        mapName = select(1, GetInstanceInfo()) or "Unknown"
    end

    -- Find player in scoreboard
    local playerName = UnitName("player")
    local playerData, playerFaction
    local numScores = GetNumBattlefieldScores()
    for i = 1, numScores do
        local info = C_PvP.GetScoreInfo(i)
        if info and info.name then
            local name = info.name:match("^([^-]+)")
            if name == playerName then
                playerData = info
                playerFaction = info.faction  -- 0 = Horde, 1 = Alliance
                break
            end
        end
    end

    if not playerData then return false end

    -- Win detection
    local won = false
    if IsActiveBattlefieldArena() then
        local winner = GetBattlefieldWinner()
        local arenaFaction = GetBattlefieldArenaFaction()
        if winner and arenaFaction then
            won = (winner == arenaFaction)
        end
    else
        local winner = GetBattlefieldWinner()
        if winner and playerFaction then
            won = (winner == playerFaction)
        end
    end

    -- Team composition
    -- For BGs (randomBG/epicBG/rbg): skip full rosters to save storage
    local playerTeam, enemyTeam
    local isBGMode = (mode == "randomBG" or mode == "epicBG" or mode == "rbg")
    if isBGMode then
        playerTeam = {}
        enemyTeam = {}
    else
        playerTeam = {}
        enemyTeam = {}
        for i = 1, numScores do
            local info = C_PvP.GetScoreInfo(i)
            if info then
                local entrySex, entryRaceFile
                if info.guid then
                    local _, _, _, englishRace, s = GetPlayerInfoByGUID(info.guid)
                    entrySex = s
                    if englishRace and englishRace ~= "" then
                        entryRaceFile = englishRace:lower():gsub("%s+", ""):gsub("'", "")
                    end
                end
                local entry = {
                    class = info.classToken or "UNKNOWN",
                    spec = info.talentSpec or "",
                    name = info.name or nil,
                    race = info.raceName or nil,
                    raceFile = entryRaceFile,
                    sex = entrySex,
                    kills = info.killingBlows or 0,
                    deaths = info.deaths or 0,
                    damage = info.damageDone or 0,
                    healing = info.healingDone or 0,
                    rating = info.rating or nil,
                    ratingChange = info.ratingChange or nil,
                    mmr = info.prematchMMR or nil,
                    honorLevel = info.honorLevel or nil,
                }
                if info.faction == playerFaction then
                    table.insert(playerTeam, entry)
                else
                    table.insert(enemyTeam, entry)
                end
            end
        end
    end

    -- BG-specific objective stats (flags captured, bases assaulted, etc.)
    local bgStats
    if playerData.stats and #playerData.stats > 0 and mode ~= "shuffle" then
        bgStats = {}
        for _, stat in ipairs(playerData.stats) do
            if stat.name and stat.pvpStatValue and stat.pvpStatValue > 0 then
                bgStats[#bgStats + 1] = { name = stat.name, value = stat.pvpStatValue }
            end
        end
        if #bgStats == 0 then bgStats = nil end
    end

    -- Build death info from arena death polling
    -- (Midnight 12.0: scoreboard deaths field returns 0; use UnitIsDeadOrGhost polling)
    local deathLog
    if mode == "arena2v2" or mode == "arena3v3" or mode == "shuffle" or mode == "skirmish" then
        deathLog = {}

        for _, d in ipairs(arenaDeaths) do
            deathLog[#deathLog + 1] = {
                name = d.name,
                class = d.class,
                spec = d.spec or "",
                team = d.team,
                deaths = 1,
                time = d.time,
            }
        end
    end

    -- Reset death tracker state
    arenaDeaths = {}
    arenaStartTime = nil
    if arenaDeathTicker then
        arenaDeathTicker:Cancel()
        arenaDeathTicker = nil
    end


    -- Solo Shuffle round tracking
    local roundWins, roundLosses
    if mode == "shuffle" and playerData.stats then
        for _, stat in ipairs(playerData.stats) do
            local name = stat.name or (stat.pvpStatID and tostring(stat.pvpStatID)) or ""
            if name:lower():find("round") and name:lower():find("won") then
                roundWins = stat.pvpStatValue or 0
            elseif name:lower():find("round") and name:lower():find("lost") then
                roundLosses = stat.pvpStatValue or 0
            end
        end
        -- If we only found one, derive the other from 6 total (may be slightly off with leavers, but still useful)
        if roundWins and not roundLosses then
            roundLosses = 6 - roundWins
        elseif roundLosses and not roundWins then
            roundWins = 6 - roundLosses
        end
    end

    -- Rating info
    local rating, ratingChange, mmr, enemyMmr
    if isRated and playerData then
        rating = playerData.rating or 0
        ratingChange = playerData.ratingChange or 0
        mmr = playerData.prematchMMR or 0
        -- Team-level MMR from GetBattlefieldTeamInfo (same approach as REFlex)
        -- Returns: teamName, oldTeamRating, newTeamRating, teamMMR
        if GetBattlefieldTeamInfo then
            local arenaFaction = GetBattlefieldArenaFaction()
            if arenaFaction then
                -- Arena: arenaFaction is the player's team index (0 or 1)
                local _, _, _, playerTeamMmr = GetBattlefieldTeamInfo(arenaFaction)
                local _, _, _, enemyTeamMmr = GetBattlefieldTeamInfo(1 - arenaFaction)
                if playerTeamMmr and playerTeamMmr > 0 then
                    mmr = playerTeamMmr  -- team MMR is more accurate than personal prematchMMR
                end
                if enemyTeamMmr and enemyTeamMmr > 0 then
                    enemyMmr = enemyTeamMmr
                end
            else
                -- Rated BG/Blitz: use playerFaction (0=Horde, 1=Alliance)
                if playerFaction then
                    local _, _, _, playerTeamMmr = GetBattlefieldTeamInfo(playerFaction)
                    local _, _, _, enemyTeamMmr = GetBattlefieldTeamInfo(1 - playerFaction)
                    if playerTeamMmr and playerTeamMmr > 0 then
                        mmr = playerTeamMmr
                    end
                    if enemyTeamMmr and enemyTeamMmr > 0 then
                        enemyMmr = enemyTeamMmr
                    end
                end
            end
        end
    end

    -- Match duration (ms -> seconds)
    local duration
    if GetBattlefieldInstanceRunTime then
        local ms = GetBattlefieldInstanceRunTime()
        if ms and ms > 0 then
            duration = math.floor(ms / 1000)
        end
    end

    -- Build match record
    local now = time()
    local match = {
        timestamp = now,
        date = date("%m/%d", now),
        timeStr = date("%H:%M", now),
        mode = mode,
        mapName = mapName,
        isRated = isRated,
        won = won,
        rating = rating,
        ratingChange = ratingChange,
        mmr = mmr,
        kills = playerData.killingBlows or 0,
        deaths = playerData.deaths or 0,
        honorableKills = playerData.honorableKills or 0,
        damage = playerData.damageDone or 0,
        healing = playerData.healingDone or 0,
        honor = playerData.honorGained or 0,
        playerTeam = playerTeam,
        enemyTeam = enemyTeam,
        roundWins = roundWins,
        roundLosses = roundLosses,
        duration = duration,
        deathLog = deathLog,
        bgStats = bgStats,
        player = GetPlayerFullName(),
        enemyMmr = enemyMmr,
    }

    -- Store
    table.insert(SKToolsArenaDB.matches, 1, match)

    -- Prune
    while #SKToolsArenaDB.matches > MAX_MATCHES do
        table.remove(SKToolsArenaDB.matches)
    end
    playersCacheDirty = true
    compCacheDirty = true

    -- Update session stats
    if won then
        sessionWins = sessionWins + 1
    else
        sessionLosses = sessionLosses + 1
    end
    if ratingChange then
        sessionRating = sessionRating + ratingChange
    end

    -- Refresh UI if open
    if historyFrame and historyFrame:IsShown() then
        SKPvPHistory_RefreshUI()
    end

    local resultStr
    if mode == "shuffle" and roundWins then
        local rl = roundLosses or (6 - roundWins)
        local color = (roundWins > rl) and "|cff00ff00" or (roundWins == rl) and "|cffbbbbbb" or "|cffff4444"
        resultStr = color .. roundWins .. "-" .. rl .. "|r"
    else
        resultStr = won and "|cff00ff00WIN|r" or "|cffff4444LOSS|r"
    end
    local ratingStr = ""
    if ratingChange then
        local sign = ratingChange >= 0 and "+" or ""
        ratingStr = " (" .. sign .. ratingChange .. " → " .. (rating or "?") .. ")"
    end
    print("|cff00E5EESKTools:|r PvP match recorded: " .. resultStr .. " " .. (MODE_LABELS[mode] or mode) .. " on " .. mapName .. ratingStr)
    -- ShowMatchToast(match)  -- disabled: causing lua errors
    return true
end

local dataFrame = CreateFrame("Frame")
dataFrame:RegisterEvent("PVP_MATCH_COMPLETE")
dataFrame:SetScript("OnEvent", function()
    if SKToolsDB and SKToolsDB.pvpHistory == false then return end
    -- Try at 0.2s for fast capture, retry at 0.5s if scoreboard wasn't ready
    C_Timer.After(0.2, function()
        local ok, result = pcall(CollectMatchData)
        if not ok then
            print("|cff00E5EESKTools|r |cffFF4444error|r [CollectMatchData]: " .. tostring(result))
        elseif not result then
            C_Timer.After(0.3, function()
                local ok2, err2 = pcall(CollectMatchData)
                if not ok2 then
                    print("|cff00E5EESKTools|r |cffFF4444error|r [CollectMatchData retry]: " .. tostring(err2))
                end
            end)
        end
    end)
end)

-----------------------------
-- Arena Friendly Death Tracker
-----------------------------
local deathTrackerFrame = CreateFrame("Frame")
deathTrackerFrame:RegisterEvent("PLAYER_ENTERING_WORLD")
deathTrackerFrame:SetScript("OnEvent", function(self, event)
    if event == "PLAYER_ENTERING_WORLD" then
        local _, instanceType = IsInInstance()
        if instanceType == "arena" and not (SKToolsDB and SKToolsDB.pvpHistory == false) then
            arenaStartTime = GetTime()
            arenaDeaths = {}
            local deadUnits = {}

            if arenaDeathTicker then arenaDeathTicker:Cancel() end
            arenaDeathTicker = C_Timer.NewTicker(1.0, function()
                local units = { "player", "party1", "party2", "arena1", "arena2", "arena3" }
                for _, unit in ipairs(units) do
                    if not deadUnits[unit] then
                        local ok, exists = pcall(UnitExists, unit)
                        if ok and exists then
                            local ok2, isDead = pcall(UnitIsDeadOrGhost, unit)
                            if ok2 and isDead then
                                deadUnits[unit] = true
                                local elapsed = arenaStartTime and (GetTime() - arenaStartTime) or 0
                                -- Safely extract values (enemy unit info may be secret)
                                local name, classFile, spec
                                local team = unit:match("^arena") and "enemy" or "player"

                                -- Name: pcall + test usability (secret values error on string.len)
                                local okN, rawName = pcall(UnitName, unit)
                                if okN and rawName then
                                    local okStr = pcall(string.len, rawName)
                                    name = okStr and rawName or nil
                                end
                                name = name or "Unknown"

                                -- Class token
                                local okC, _, rawClass = pcall(UnitClass, unit)
                                if okC and rawClass then
                                    local okStr = pcall(string.len, rawClass)
                                    classFile = okStr and rawClass or nil
                                end
                                classFile = classFile or "UNKNOWN"

                                -- Spec: friendly via GetSpecialization, enemy via arena index
                                spec = ""
                                if team == "enemy" then
                                    local idx = tonumber(unit:match("arena(%d)"))
                                    if idx and GetArenaOpponentSpec then
                                        local okS, specID = pcall(GetArenaOpponentSpec, idx)
                                        if okS and specID then
                                            local okNum = pcall(function() return specID > 0 end)
                                            if okNum and specID > 0 and GetSpecializationInfoByID then
                                                local okI, _, specName = pcall(GetSpecializationInfoByID, specID)
                                                if okI and type(specName) == "string" and specName ~= "" then
                                                    spec = specName
                                                end
                                            end
                                        end
                                    end
                                else
                                    if unit == "player" and GetSpecialization and GetSpecializationInfo then
                                        local okS, specIdx = pcall(GetSpecialization)
                                        if okS and specIdx then
                                            local okI, _, specName = pcall(GetSpecializationInfo, specIdx)
                                            if okI and type(specName) == "string" and specName ~= "" then spec = specName end
                                        end
                                    elseif GetInspectSpecialization and GetSpecializationInfoByID then
                                        local okS, specID = pcall(GetInspectSpecialization, unit)
                                        if okS and specID and specID > 0 then
                                            local okI, _, specName = pcall(GetSpecializationInfoByID, specID)
                                            if okI and type(specName) == "string" and specName ~= "" then spec = specName end
                                        end
                                    end
                                end

                                arenaDeaths[#arenaDeaths + 1] = {
                                    name = name,
                                    class = classFile,
                                    spec = spec,
                                    unit = unit,
                                    time = elapsed,
                                    team = team,
                                }
                            end
                        end
                    end
                end
            end)
        else
            if arenaDeathTicker then
                arenaDeathTicker:Cancel()
                arenaDeathTicker = nil
            end
            -- Don't reset arenaDeaths — CollectMatchData needs them
        end
    end
end)

-----------------------------
-- Filtered Data
-----------------------------
-----------------------------
-- Custom Match Tooltip (proper columnar layout)
-----------------------------
local FormatNumber, ResolveSpec, IsHealerSpec  -- forward declarations (defined after filter code)
local matchTooltip
local MT_ROW_H = 20
local MT_WIDTH = 460
local MT_MAX_ROWS = 35
local MT_ICON_SZ = 26
local MT_PAD = 6
-- Column X offsets
local MT_X_ICON = 8
local MT_X_NAME = 68
local MT_X_RATING = 260
local MT_X_CHG = 266
local MT_X_DMG = 375
local MT_X_HEAL = 440

local function CreateMTRow(parent, y)
    local r = {}
    r.icons = parent:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
    r.icons:SetPoint("LEFT", parent, "TOPLEFT", MT_X_ICON, y)
    r.icons:SetJustifyH("LEFT")

    r.name = parent:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
    r.name:SetPoint("LEFT", parent, "TOPLEFT", MT_X_NAME, y)
    r.name:SetWidth(195)
    r.name:SetJustifyH("LEFT")
    r.name:SetWordWrap(false)

    r.rating = parent:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
    r.rating:SetPoint("RIGHT", parent, "TOPLEFT", MT_X_RATING, y)
    r.rating:SetJustifyH("RIGHT")

    r.chg = parent:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
    r.chg:SetPoint("LEFT", parent, "TOPLEFT", MT_X_CHG, y)
    r.chg:SetJustifyH("LEFT")

    r.dmg = parent:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
    r.dmg:SetPoint("RIGHT", parent, "TOPLEFT", MT_X_DMG, y)
    r.dmg:SetJustifyH("RIGHT")

    r.heal = parent:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
    r.heal:SetPoint("RIGHT", parent, "TOPLEFT", MT_X_HEAL, y)
    r.heal:SetJustifyH("RIGHT")

    -- Full-width line (headers, separators, combat text)
    r.full = parent:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
    r.full:SetPoint("LEFT", parent, "TOPLEFT", MT_X_ICON, y)
    r.full:SetJustifyH("LEFT")
    r.fullR = parent:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
    r.fullR:SetPoint("RIGHT", parent, "TOPRIGHT", -MT_PAD, y)
    r.fullR:SetJustifyH("RIGHT")
    return r
end

local function ClearMTRow(r)
    r.icons:SetText("") r.name:SetText("") r.rating:SetText("")
    r.chg:SetText("") r.dmg:SetText("") r.heal:SetText("")
    r.full:SetText("") r.fullR:SetText("")
    r.full:SetFontObject(GameFontHighlight)
    r.full:SetTextColor(1, 1, 1)
    r.icons:Hide() r.name:Hide() r.rating:Hide()
    r.chg:Hide() r.dmg:Hide() r.heal:Hide()
    r.full:Hide() r.fullR:Hide()
end

local function ShowMTCols(r)
    r.icons:Show() r.name:Show() r.rating:Show()
    r.chg:Show() r.dmg:Show() r.heal:Show()
    r.full:Hide() r.fullR:Hide()
end

local function ShowMTFull(r)
    r.icons:Hide() r.name:Hide() r.rating:Hide()
    r.chg:Hide() r.dmg:Hide() r.heal:Hide()
    r.full:Show() r.fullR:Show()
end

local function EnsureMatchTooltip()
    if matchTooltip then return matchTooltip end
    local f = CreateFrame("Frame", "SKMatchTooltip", UIParent, "BackdropTemplate")
    f:SetFrameStrata("TOOLTIP")
    f:SetClampedToScreen(true)
    f:SetBackdrop({
        bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
        edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
        tile = true, tileSize = 16, edgeSize = 16,
        insets = { left = 4, right = 4, top = 4, bottom = 4 },
    })
    f:SetBackdropColor(0.04, 0.04, 0.07, 0.95)
    f:SetBackdropBorderColor(0.25, 0.25, 0.3, 0.8)
    f:Hide()
    f.rows = {}
    for i = 1, MT_MAX_ROWS do
        local y = -(MT_PAD + (i - 1) * MT_ROW_H)
        f.rows[i] = CreateMTRow(f, y)
    end
    matchTooltip = f
    return f
end

local function ShowMatchTooltip(anchor, m, kills, deaths, hks, deathLogText)
    -- No tooltip for BGs/Epic BGs (no team rosters, stats already visible in row)
    if m.mode == "randomBG" or m.mode == "epicBG" then return end

    local f = EnsureMatchTooltip()
    for i = 1, MT_MAX_ROWS do ClearMTRow(f.rows[i]) end

    local rowIdx = 0
    local function Next()
        rowIdx = rowIdx + 1
        return rowIdx <= MT_MAX_ROWS and f.rows[rowIdx] or nil
    end

    local isRated = m.isRated
    local isShuffle = (m.mode == "shuffle")
    local isBlitz = (m.mode == "blitz")
    local myShort = m.player and m.player:match("^([^%-]+)")
    local function IsMe(p)
        if not myShort or not p.name then return false end
        return (p.name:match("^([^%-]+)") or p.name) == myShort
    end

    -- Team header row
    local function AddHeader(label, hexColor, mmr)
        local r = Next(); if not r then return end
        ShowMTFull(r)
        r.full:SetText("|cff" .. hexColor .. label .. "|r")
        r.full:SetFontObject(GameFontNormalLarge)
        if mmr and mmr > 0 then
            r.fullR:SetText("|cff66ccddMMR " .. mmr .. "|r")
        end
    end

    -- Player row
    local function AddPlayer(p, isYou, showHealer)
        local r = Next(); if not r then return end
        ShowMTCols(r)
        local hex = GetClassColorHex(p.class or "UNKNOWN")
        local spec = ResolveSpec(p.spec)
        if spec == "Unknown" then spec = "" end

        local raceI = RaceIconString(p.raceFile, p.race, p.sex, MT_ICON_SZ)
        local specI = SpecIconString(p.class, spec, MT_ICON_SZ)
        if specI == "" then specI = ClassIconString(p.class or "UNKNOWN", MT_ICON_SZ) end
        r.icons:SetText(raceI .. " " .. specI)

        local short = p.name and (p.name:match("^([^-]+)") or p.name) or ""
        local nameStr
        if short ~= "" then
            nameStr = "|cff" .. hex .. short .. "|r"
        else
            nameStr = "|cff" .. hex .. (spec ~= "" and spec or (p.class or "?")) .. "|r"
        end
        if isYou then nameStr = nameStr .. " |cffFFD100*|r" end
        if showHealer and IsHealerSpec(p.spec) then nameStr = nameStr .. " |cff55bbaa(H)|r" end
        r.name:SetText(nameStr)

        if isRated and p.rating and p.rating > 0 then
            r.rating:SetText("|cffbbbbbb" .. p.rating .. "|r")
        end
        if isRated and p.ratingChange and p.ratingChange ~= 0 then
            local sign = p.ratingChange >= 0 and "+" or ""
            local clr = p.ratingChange >= 0 and "44dd44" or "ee5555"
            r.chg:SetText("|cff" .. clr .. sign .. p.ratingChange .. "|r")
        end

        r.dmg:SetText(FormatNumber(p.damage or 0))
        r.heal:SetText(FormatNumber(p.healing or 0))
    end

    -- Totals row (inline with team, no extra spacing)
    local function AddTotals(team)
        local tD, tH = 0, 0
        for _, p in ipairs(team) do
            tD = tD + (p.damage or 0)
            tH = tH + (p.healing or 0)
        end
        local r = Next(); if not r then return end
        ShowMTCols(r)
        r.name:SetText("|cffFFD100Total|r")
        r.dmg:SetText("|cffFFD100" .. FormatNumber(tD) .. "|r")
        r.heal:SetText("|cffFFD100" .. FormatNumber(tH) .. "|r")
    end

    -- Column labels row
    local function AddColHeaders()
        local r = Next(); if not r then return end
        ShowMTCols(r)
        if isRated then r.rating:SetText("|cff555555Rating|r") end
        r.dmg:SetText("|cff555555Dmg|r")
        r.heal:SetText("|cff555555Heal|r")
    end

    local function AddTeam(team, label, hexColor, mmr, showHealer)
        if not team or #team == 0 then return end
        AddHeader(label, hexColor, mmr)
        AddColHeaders()
        for _, p in ipairs(team) do AddPlayer(p, IsMe(p), showHealer) end
        AddTotals(team)
    end

    -- Build content
    if isShuffle then
        local all = {}
        if m.playerTeam then for _, p in ipairs(m.playerTeam) do all[#all+1] = p end end
        if m.enemyTeam then for _, p in ipairs(m.enemyTeam) do all[#all+1] = p end end
        AddHeader("Lobby", "00E5EE", m.mmr)
        AddColHeaders()
        for _, p in ipairs(all) do AddPlayer(p, IsMe(p), true) end
        AddTotals(all)
    else
        AddTeam(m.playerTeam, "Your Team", "44dd44", m.mmr, isBlitz)
        AddTeam(m.enemyTeam, "Enemy Team", "ee5555", m.enemyMmr, isBlitz)
    end

    -- Combat line
    if hks and hks > 0 then
        local r = Next(); if r then
            ShowMTFull(r)
            r.full:SetText("|cff00E5EECombat|r  |cff888888KB: " .. (kills or 0) .. "   Deaths: " .. (deaths or 0) .. "   HK: " .. hks .. "|r")
        end
    end

    -- Death log
    if deathLogText then
        local r = Next(); if r then ShowMTFull(r); r.full:SetText("|cff00E5EEDeath Log|r") end
        for line in deathLogText:gmatch("[^\n]+") do
            r = Next(); if r then ShowMTFull(r); r.full:SetText(line) end
        end
    end

    local totalH = MT_PAD * 2 + rowIdx * MT_ROW_H
    f:SetSize(MT_WIDTH, totalH)
    f:ClearAllPoints()
    local cx, cy = GetCursorPosition()
    local scale = f:GetEffectiveScale()
    cx, cy = cx / scale, cy / scale
    -- Position tooltip above-right of cursor, clamped to screen
    f:SetPoint("BOTTOMLEFT", UIParent, "BOTTOMLEFT", cx + 12, cy + 8)
    f:Show()
end

local function HideMatchTooltip()
    if matchTooltip then matchTooltip:Hide() end
    GameTooltip:Hide()
end

-- Sort comparators keyed by column name
local SORT_FUNCS = {
    timestamp = function(a, b) return (a.timestamp or 0) < (b.timestamp or 0) end,
    won       = function(a, b) return (a.won and 1 or 0) < (b.won and 1 or 0) end,
    mode      = function(a, b) return (a.mode or "") < (b.mode or "") end,
    mapName   = function(a, b) return (a.mapName or "") < (b.mapName or "") end,
    rating    = function(a, b) return (a.rating or 0) < (b.rating or 0) end,
    mmr       = function(a, b) return (a.mmr or 0) < (b.mmr or 0) end,
    kills     = function(a, b) return (a.kills or 0) < (b.kills or 0) end,
    damage    = function(a, b) return (a.damage or 0) < (b.damage or 0) end,
    healing   = function(a, b) return (a.healing or 0) < (b.healing or 0) end,
    duration  = function(a, b) return (a.duration or 0) < (b.duration or 0) end,
}

local function GetFilteredMatches()
    local matches = SKToolsArenaDB and SKToolsArenaDB.matches or {}
    local filterModes = FILTER_MAP[activeFilter]

    local filtered = {}
    for _, m in ipairs(matches) do
        local modeOK = not filterModes or filterModes[m.mode]
        local charOK = not activeCharFilter or m.player == activeCharFilter
        if modeOK and charOK then
            filtered[#filtered + 1] = m
        end
    end

    -- Apply sort
    local cmp = SORT_FUNCS[sortColumn]
    if cmp then
        local asc = sortAscending
        table.sort(filtered, function(a, b)
            if asc then return cmp(a, b) else return cmp(b, a) end
        end)
    end

    return filtered
end

-----------------------------
-- Format Helpers
-----------------------------
FormatNumber = function(n)
    if not n then return "0" end
    if n >= 1000000 then
        return string.format("%.1fM", n / 1000000)
    elseif n >= 1000 then
        return string.format("%.1fK", n / 1000)
    end
    return tostring(n)
end

local function FormatDuration(seconds)
    if not seconds or seconds <= 0 then return "|cff888888—|r" end
    local m = math.floor(seconds / 60)
    local s = seconds % 60
    return string.format("%d:%02d", m, s)
end

local function CompIconString(team, iconSize)
    if not team or #team == 0 then return "" end
    iconSize = iconSize or 20
    local parts = {}
    for _, p in ipairs(team) do
        local icon = ClassIconString(p.class or "UNKNOWN", iconSize)
        if icon ~= "" then
            parts[#parts + 1] = icon
        else
            local hex = GetClassColorHex(p.class or "UNKNOWN")
            parts[#parts + 1] = "|cff" .. hex .. (p.class or "?"):sub(1, 1) .. "|r"
        end
    end
    local sep = iconSize and iconSize <= 14 and " " or ""
    return table.concat(parts, sep)
end

-- Resolve spec: handles both string names and numeric specIDs
ResolveSpec = function(spec)
    if not spec or spec == "" then return "Unknown" end
    if type(spec) == "number" then
        local _, name = GetSpecializationInfoByID(spec)
        return name or ("Spec" .. spec)
    end
    return spec
end

local HEALER_SPECS = {
    ["Restoration"] = true, ["Holy"] = true, ["Discipline"] = true,
    ["Mistweaver"] = true, ["Preservation"] = true,
}
IsHealerSpec = function(spec)
    return HEALER_SPECS[ResolveSpec(spec)] or false
end

local function CompTooltipText(team)
    if not team or #team == 0 then return nil end
    local lines = {}
    for _, p in ipairs(team) do
        local spec = ResolveSpec(p.spec)
        local class = p.class or "UNKNOWN"
        local prettyClass = class:sub(1, 1) .. class:sub(2):lower()
        prettyClass = prettyClass:gsub("knight", " Knight"):gsub("hunter", " Hunter")
        local hex = GetClassColorHex(class)
        lines[#lines + 1] = "|cff" .. hex .. spec .. " " .. prettyClass .. "|r"
    end
    return table.concat(lines, "\n")
end

-- Generate a stable key for a team composition (sorted class tokens)
local function GetCompKey(team)
    if not team or #team == 0 then return nil end
    local classes = {}
    for _, p in ipairs(team) do
        classes[#classes + 1] = p.class or "UNKNOWN"
    end
    table.sort(classes)
    return table.concat(classes, "/")
end

-- Generate icon string label for a comp key
local function GetCompLabel(team, iconSize)
    if not team or #team == 0 then return "" end
    local sorted = {}
    for _, p in ipairs(team) do sorted[#sorted + 1] = p end
    table.sort(sorted, function(a, b) return (a.class or "") < (b.class or "") end)
    return CompIconString(sorted, iconSize)
end

-- Generate a stable key for a team composition including specs
local function GetSpecCompKey(team)
    if not team or #team == 0 then return nil end
    local specs = {}
    for _, p in ipairs(team) do
        local spec = ResolveSpec(p.spec)
        specs[#specs + 1] = spec .. " " .. (p.class or "UNKNOWN")
    end
    table.sort(specs)
    return table.concat(specs, "/")
end

-- Generate a label with spec names and class icons
local function GetSpecCompLabel(team)
    if not team or #team == 0 then return "" end
    local sorted = {}
    for _, p in ipairs(team) do sorted[#sorted + 1] = p end
    table.sort(sorted, function(a, b)
        local sa = ResolveSpec(a.spec) .. (a.class or "")
        local sb = ResolveSpec(b.spec) .. (b.class or "")
        return sa < sb
    end)
    local parts = {}
    for _, p in ipairs(sorted) do
        local hex = GetClassColorHex(p.class or "UNKNOWN")
        local spec = ResolveSpec(p.spec)
        if spec == "Unknown" then spec = "?" end
        local icon = ClassIconString(p.class or "UNKNOWN", 18)
        parts[#parts + 1] = icon .. "|cff" .. hex .. spec .. "|r"
    end
    return table.concat(parts, " ")
end

-----------------------------
-- Post-Match Toast
-----------------------------
local toastFrame

local function CreateToastFrame()
    if toastFrame then return end

    local f = CreateFrame("Frame", "SKToolsPvPToastFrame", UIParent, "BackdropTemplate")
    f:SetSize(300, 80)
    f:SetPoint("TOP", UIParent, "TOP", 0, -100)
    f:SetFrameStrata("FULLSCREEN_DIALOG")
    f:SetBackdrop({
        bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
        edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
        tile = true, tileSize = 16, edgeSize = 16,
        insets = { left = 4, right = 4, top = 4, bottom = 4 },
    })
    f:SetBackdropColor(0.05, 0.05, 0.08, 0.95)
    f:SetBackdropBorderColor(0.3, 0.3, 0.35, 1)
    f:EnableMouse(false)
    f:Hide()

    -- Cyan accent line
    local accent = f:CreateTexture(nil, "ARTWORK")
    accent:SetColorTexture(CYAN.r, CYAN.g, CYAN.b, 0.7)
    accent:SetPoint("TOPLEFT", f, "TOPLEFT", 5, -4)
    accent:SetPoint("TOPRIGHT", f, "TOPRIGHT", -5, -4)
    accent:SetHeight(2)

    -- Result text (large)
    f.resultText = f:CreateFontString(nil, "OVERLAY", "GameFontNormalLarge")
    f.resultText:SetPoint("TOP", 0, -12)

    -- Mode + map line
    f.infoText = f:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
    f.infoText:SetPoint("TOP", f.resultText, "BOTTOM", 0, -4)

    -- Rating change line
    f.ratingText = f:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
    f.ratingText:SetPoint("TOP", f.infoText, "BOTTOM", 0, -4)

    toastFrame = f
end

local function ShowMatchToast(match)
    CreateToastFrame()

    -- Result
    local resultStr
    if match.mode == "shuffle" and match.roundWins then
        local color = match.won and "|cff00ff00" or "|cffff4444"
        resultStr = color .. match.roundWins .. "-" .. (match.roundLosses or (6 - match.roundWins)) .. "|r"
    else
        resultStr = match.won and "|cff00ff00VICTORY|r" or "|cffff4444DEFEAT|r"
    end
    toastFrame.resultText:SetText(resultStr)

    -- Mode + map
    local modeLabel = MODE_LABELS[match.mode] or match.mode
    toastFrame.infoText:SetText(modeLabel .. "  |cff888888—|r  " .. (match.mapName or "Unknown"))

    -- Rating
    if match.isRated and match.ratingChange then
        local sign = match.ratingChange >= 0 and "+" or ""
        local color = match.ratingChange >= 0 and "|cff00ff00" or "|cffff4444"
        toastFrame.ratingText:SetText(color .. sign .. match.ratingChange .. "|r  →  " .. (match.rating or "?") .. " CR")
        toastFrame.ratingText:Show()
    else
        toastFrame.ratingText:Hide()
    end

    -- Color the border based on result
    if match.won then
        toastFrame:SetBackdropBorderColor(0, 0.6, 0, 0.8)
    else
        toastFrame:SetBackdropBorderColor(0.6, 0, 0, 0.8)
    end

    -- Show with fade in
    ns.FadeIn(toastFrame, 0.3)

    -- Schedule fade out after 4.5s (total ~5s visible)
    C_Timer.After(4.5, function()
        ns.FadeOut(toastFrame, 0.5)
    end)
end

-----------------------------
-- Resolve spec from death log entry (used by both Analysis + Match Stats)
-----------------------------
local function ResolveDeathSpec(d, matchData)
    -- If spec is already a valid string name (not numeric), use it
    if d.spec and d.spec ~= "" and not tonumber(d.spec) then
        return d.spec
    end
    -- Try resolving a numeric spec ID directly (old data stored IDs due to pcall bug)
    local numericSpec = tonumber(d.spec)
    if numericSpec and numericSpec > 0 and GetSpecializationInfoByID then
        local ok, _, name = pcall(GetSpecializationInfoByID, numericSpec)
        if ok and type(name) == "string" and name ~= "" then
            return name
        end
    end
    -- Fall back to cross-reference with scoreboard team data by class
    local teamData = d.team == "enemy" and matchData.enemyTeam or matchData.playerTeam
    if teamData and d.class and d.class ~= "UNKNOWN" then
        for _, p in ipairs(teamData) do
            if p.class == d.class and p.spec and p.spec ~= "" then
                return p.spec
            end
        end
    end
    return nil
end

-----------------------------
-- Matchup Stats (for Arena Notes integration)
-----------------------------
local function GetMatchupStats(playerTeam, enemyTeam)
    local matches = SKToolsArenaDB and SKToolsArenaDB.matches or {}
    local pKey = GetSpecCompKey(playerTeam)
    local eKey = GetSpecCompKey(enemyTeam)
    if not pKey or not eKey then return nil end

    local stats = {
        wins = 0, losses = 0,
        killTarget = {},    -- specKey -> { count, totalTime, timeCount }
        deathTarget = {},   -- specKey -> { count, totalTime, timeCount }
    }

    for _, m in ipairs(matches) do
        if m.playerTeam and m.enemyTeam then
            local mPKey = GetSpecCompKey(m.playerTeam)
            local mEKey = GetSpecCompKey(m.enemyTeam)

            if mPKey == pKey and mEKey == eKey then
                if m.won then
                    stats.wins = stats.wins + 1
                    -- First enemy death = kill target
                    if m.deathLog then
                        for _, d in ipairs(m.deathLog) do
                            if d.team == "enemy" and d.class and d.class ~= "UNKNOWN" then
                                local spec = ResolveDeathSpec(d, m) or "Unknown"
                                local sk = spec .. " " .. d.class
                                if not stats.killTarget[sk] then
                                    stats.killTarget[sk] = { count = 0, totalTime = 0, timeCount = 0 }
                                end
                                local e = stats.killTarget[sk]
                                e.count = e.count + 1
                                if d.time then e.totalTime = e.totalTime + d.time; e.timeCount = e.timeCount + 1 end
                                break
                            end
                        end
                    end
                else
                    stats.losses = stats.losses + 1
                    -- First player death = who they target
                    if m.deathLog then
                        for _, d in ipairs(m.deathLog) do
                            if d.team == "player" and d.class and d.class ~= "UNKNOWN" then
                                local spec = ResolveDeathSpec(d, m) or "Unknown"
                                local sk = spec .. " " .. d.class
                                if not stats.deathTarget[sk] then
                                    stats.deathTarget[sk] = { count = 0, totalTime = 0, timeCount = 0 }
                                end
                                local e = stats.deathTarget[sk]
                                e.count = e.count + 1
                                if d.time then e.totalTime = e.totalTime + d.time; e.timeCount = e.timeCount + 1 end
                                break
                            end
                        end
                    end
                end
            end
        end
    end

    local total = stats.wins + stats.losses
    if total == 0 then return nil end
    stats.total = total
    stats.winPct = stats.wins / total * 100
    return stats
end

-- Export functions for cross-file access (ArenaNotes)
ns.GetSpecCompKey = GetSpecCompKey
ns.GetSpecCompLabel = GetSpecCompLabel
ns.GetMatchupStats = GetMatchupStats

-----------------------------
-- Export Match History
-----------------------------
local exportFrame

local function ShowExportFrame()
    local filtered = GetFilteredMatches()
    if #filtered == 0 then
        print("|cff00E5EESKTools:|r No matches to export.")
        return
    end

    if not exportFrame then
        local f = CreateFrame("Frame", "SKToolsExportFrame", UIParent, "BackdropTemplate")
        f:SetSize(620, 400)
        f:SetPoint("CENTER")
        f:SetMovable(true)
        f:EnableMouse(true)
        f:SetClampedToScreen(true)
        f:SetFrameStrata("FULLSCREEN_DIALOG")
        f:SetBackdrop({
            bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
            edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
            tile = true, tileSize = 16, edgeSize = 16,
            insets = { left = 4, right = 4, top = 4, bottom = 4 },
        })
        f:SetBackdropColor(0.05, 0.05, 0.08, 0.95)
        f:SetBackdropBorderColor(0.3, 0.3, 0.35, 1)

        -- Drag to move
        local drag = CreateFrame("Frame", nil, f)
        drag:SetPoint("TOPLEFT", 0, 0)
        drag:SetPoint("TOPRIGHT", 0, 0)
        drag:SetHeight(30)
        drag:EnableMouse(true)
        drag:RegisterForDrag("LeftButton")
        drag:SetScript("OnDragStart", function() f:StartMoving() end)
        drag:SetScript("OnDragStop", function() f:StopMovingOrSizing() end)

        f:SetScript("OnKeyDown", function(self, key)
            if key == "ESCAPE" then pcall(self.SetPropagateKeyboardInput, self, false); self:Hide()
            else pcall(self.SetPropagateKeyboardInput, self, true) end
        end)
        pcall(f.EnableKeyboard, f, true)

        local accent = f:CreateTexture(nil, "ARTWORK")
        accent:SetColorTexture(0, 0.898, 0.933, 0.7)
        accent:SetPoint("TOPLEFT", 5, -4)
        accent:SetPoint("TOPRIGHT", -5, -4)
        accent:SetHeight(2)

        local title = f:CreateFontString(nil, "OVERLAY", "GameFontNormalLarge")
        title:SetPoint("TOPLEFT", 12, -10)
        title:SetText("|cff00E5EEExport Match History|r")

        local closeBtn = CreateFrame("Button", nil, f, "UIPanelCloseButton")
        closeBtn:SetPoint("TOPRIGHT", -2, -2)
        closeBtn:SetScript("OnClick", function() f:Hide() end)

        local hint = f:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
        hint:SetPoint("TOPLEFT", 12, -30)
        hint:SetText("Press Ctrl+A then Ctrl+C to copy")

        local scrollFrame = CreateFrame("ScrollFrame", "SKToolsExportScroll", f, "UIPanelScrollFrameTemplate")
        scrollFrame:SetPoint("TOPLEFT", 10, -48)
        scrollFrame:SetPoint("BOTTOMRIGHT", -30, 10)

        local editBox = CreateFrame("EditBox", "SKToolsExportEditBox", scrollFrame)
        editBox:SetMultiLine(true)
        editBox:SetAutoFocus(false)
        editBox:SetFontObject("ChatFontNormal")
        editBox:SetWidth(560)
        editBox:SetScript("OnEscapePressed", function(self) self:ClearFocus(); f:Hide() end)
        scrollFrame:SetScrollChild(editBox)

        scrollFrame:SetScript("OnSizeChanged", function(self, w)
            editBox:SetWidth(w)
        end)

        f.editBox = editBox
        exportFrame = f
    end

    -- Format data as tab-separated text
    local lines = {}
    lines[#lines + 1] = "Result\tMode\tMap\tRating\tMMR\tK/D\tDamage\tHealing\tDuration\tDate"

    for _, m in ipairs(filtered) do
        local result
        if m.mode == "shuffle" and m.roundWins then
            result = m.roundWins .. "-" .. (m.roundLosses or (6 - m.roundWins))
        else
            result = m.won and "W" or "L"
        end

        local modeLabel = MODE_LABELS[m.mode] or m.mode

        local ratingStr = ""
        if m.isRated and m.ratingChange then
            local sign = m.ratingChange >= 0 and "+" or ""
            ratingStr = sign .. m.ratingChange .. " (" .. (m.rating or "?") .. ")"
        end

        local mmrStr = (m.isRated and m.mmr) and tostring(m.mmr) or ""
        local hkStr = (m.honorableKills and m.honorableKills > 0) and (" (" .. m.honorableKills .. " HKs)") or ""
        local kd = (m.kills or 0) .. "/" .. (m.deaths or 0) .. hkStr
        local dmg = FormatNumber(m.damage)
        local heal = FormatNumber(m.healing)
        local dur = ""
        if m.duration and m.duration > 0 then
            local mins = math.floor(m.duration / 60)
            local secs = m.duration % 60
            dur = string.format("%d:%02d", mins, secs)
        end
        local dateStr = (m.date or "") .. " " .. (m.timeStr or "")

        lines[#lines + 1] = table.concat({result, modeLabel, m.mapName or "", ratingStr, mmrStr, kd, dmg, heal, dur, dateStr}, "\t")
    end

    local text = table.concat(lines, "\n")
    exportFrame.editBox:SetText(text)
    exportFrame:Show()
    exportFrame.editBox:HighlightText()
    exportFrame.editBox:SetFocus()
end

-----------------------------
-- UI Creation
-----------------------------
local function CreateHistoryFrame()
    if historyFrame then return end

    -- Main frame
    local f = CreateFrame("Frame", "SKToolsPvPHistoryFrame", UIParent, "BackdropTemplate")
    f:SetSize(1260, 34 + 30 + 26 + 28 + HEADER_HEIGHT + (ROW_HEIGHT * VISIBLE_ROWS) + 16)
    f:SetPoint("CENTER")
    f:SetMovable(true)
    f:EnableMouse(true)
    f:SetClampedToScreen(true)
    f:SetFrameStrata("DIALOG")
    f:SetBackdrop({
        bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
        edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
        tile = true, tileSize = 16, edgeSize = 16,
        insets = { left = 4, right = 4, top = 4, bottom = 4 },
    })
    f:SetBackdropColor(0.05, 0.05, 0.08, 0.95)
    f:SetBackdropBorderColor(0.3, 0.3, 0.35, 1)
    f:Hide()
    f:SetToplevel(true)
    f:SetScript("OnShow", function(self) self:Raise() end)

    -- Cyan accent line at top
    local accentLine = f:CreateTexture(nil, "ARTWORK")
    accentLine:SetColorTexture(CYAN.r, CYAN.g, CYAN.b, 0.7)
    accentLine:SetPoint("TOPLEFT", f, "TOPLEFT", 5, -4)
    accentLine:SetPoint("TOPRIGHT", f, "TOPRIGHT", -5, -4)
    accentLine:SetHeight(2)

    -- ESC to close (own handler to avoid UISpecialFrames taint)
    f:SetScript("OnKeyDown", function(self, key)
        if key == "ESCAPE" then pcall(self.SetPropagateKeyboardInput, self, false); self:Hide()
        else pcall(self.SetPropagateKeyboardInput, self, true) end
    end)
    pcall(f.EnableKeyboard, f, true)

    -- Title bar drag
    local titleBar = CreateFrame("Frame", nil, f)
    titleBar:SetPoint("TOPLEFT", 4, -6)
    titleBar:SetPoint("TOPRIGHT", -4, -6)
    titleBar:SetHeight(28)
    titleBar:EnableMouse(true)
    titleBar:RegisterForDrag("LeftButton")
    titleBar:SetScript("OnDragStart", function() f:StartMoving() end)
    titleBar:SetScript("OnDragStop", function()
        f:StopMovingOrSizing()
        local point, _, relPoint, x, y = f:GetPoint()
        SKToolsArenaDB.framePos = { point = point, relPoint = relPoint, x = x, y = y }
    end)

    -- Title bar background gradient
    local titleBg = titleBar:CreateTexture(nil, "BACKGROUND")
    titleBg:SetAllPoints()
    titleBg:SetColorTexture(0.08, 0.12, 0.14, 0.8)

    -- Title text
    local title = titleBar:CreateFontString(nil, "OVERLAY", "GameFontNormalLarge")
    title:SetPoint("LEFT", 10, 0)
    title:SetText("|cff00E5EESK|r|cffFF4444PvP|r Match History")

    -- Close button
    local closeBtn = CreateFrame("Button", nil, titleBar, "UIPanelCloseButton")
    closeBtn:SetPoint("TOPRIGHT", f, "TOPRIGHT", -2, -2)
    closeBtn:SetScript("OnClick", function() f:Hide() end)

    -- Title separator
    local titleSep = f:CreateTexture(nil, "ARTWORK")
    titleSep:SetPoint("TOPLEFT", titleBar, "BOTTOMLEFT", 0, 0)
    titleSep:SetPoint("TOPRIGHT", titleBar, "BOTTOMRIGHT", 0, 0)
    titleSep:SetHeight(1)
    titleSep:SetColorTexture(CYAN.r, CYAN.g, CYAN.b, 0.3)

    -- Summary bar
    local summary = f:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
    summary:SetPoint("TOPLEFT", titleBar, "BOTTOMLEFT", 10, -6)
    summary:SetJustifyH("LEFT")
    f.summary = summary

    -- Filter buttons
    local filterY = -64
    local filters = { "All", "2v2", "3v3", "Shuffle", "Blitz", "RBG", "Skirm", "BG", "Epic BG" }
    filterButtons = {}
    local prevBtn
    for i, name in ipairs(filters) do
        local btnW = (#name > 4) and (10 + #name * 7) or 52
        local btn = ns.CreateThemedButton(f, name, btnW, 22, "secondary")
        btn.text = btn.label
        btn.filterName = name

        if i == 1 then
            btn:SetPoint("TOPLEFT", f, "TOPLEFT", 12, filterY)
        else
            btn:SetPoint("LEFT", prevBtn, "RIGHT", 4, 0)
        end

        btn:SetScript("OnClick", function(self)
            activeFilter = self.filterName
            compSpecFilters = {}
            analysisSpecFilters = {}
            for _, b in ipairs(filterButtons) do
                local isActive = (b.filterName == activeFilter)
                if isActive then
                    b:SetBackdropColor(CYAN.r * 0.3, CYAN.g * 0.3, CYAN.b * 0.3, 1)
                    b:SetBackdropBorderColor(CYAN.r, CYAN.g, CYAN.b, 0.8)
                    b.text:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
                else
                    b:SetBackdropColor(0.1, 0.1, 0.12, 0.8)
                    b:SetBackdropBorderColor(0.3, 0.3, 0.35, 0.6)
                    b.text:SetTextColor(0.7, 0.7, 0.7)
                end
            end
            if activeView == "winrates" or activeView == "analysis" or activeView == "players" then
                RefreshStatsPanel()
            elseif activeView == "graph" then
                RefreshGraphPanel()
            end
            UpdateFilterLabel()
            SKPvPHistory_RefreshUI()
        end)

        btn:SetScript("OnEnter", function(self)
            if self.filterName ~= activeFilter then
                self:SetBackdropColor(0.15, 0.15, 0.18, 0.9)
                self.text:SetTextColor(1, 1, 1)
            end
        end)
        btn:SetScript("OnLeave", function(self)
            if self.filterName ~= activeFilter then
                self:SetBackdropColor(0.1, 0.1, 0.12, 0.8)
                self.text:SetTextColor(0.7, 0.7, 0.7)
            end
        end)

        filterButtons[#filterButtons + 1] = btn
        prevBtn = btn
    end

    -- View buttons (row below filters)
    local viewY = filterY - 26
    local views = {
        { key = "matches",   label = "Matches" },
        { key = "winrates",  label = "Win Rates" },
        { key = "analysis",  label = "Analysis" },
        { key = "players",   label = "Players" },
        { key = "graph",     label = "Graph" },
    }
    viewButtons = {}
    local prevViewBtn
    for i, v in ipairs(views) do
        local btnW = math.max(52, 10 + #v.label * 7)
        local btn = ns.CreateThemedButton(f, v.label, btnW, 22, "secondary")
        btn.text = btn.label
        btn.viewKey = v.key

        if i == 1 then
            btn:SetPoint("TOPLEFT", f, "TOPLEFT", 12, viewY)
        else
            btn:SetPoint("LEFT", prevViewBtn, "RIGHT", 4, 0)
        end

        btn:SetScript("OnClick", function() ShowView(v.key) end)
        btn:SetScript("OnEnter", function(self)
            if activeView ~= self.viewKey then
                self:SetBackdropColor(0.15, 0.15, 0.18, 0.9)
                self.text:SetTextColor(1, 1, 1)
            end
        end)
        btn:SetScript("OnLeave", function(self)
            if activeView ~= self.viewKey then
                self:SetBackdropColor(0.1, 0.1, 0.12, 0.8)
                self.text:SetTextColor(0.7, 0.7, 0.7)
            end
        end)

        viewButtons[#viewButtons + 1] = btn
        prevViewBtn = btn
    end
    -- Highlight default view
    viewButtons[1]:SetBackdropColor(CYAN.r * 0.3, CYAN.g * 0.3, CYAN.b * 0.3, 1)
    viewButtons[1]:SetBackdropBorderColor(CYAN.r, CYAN.g, CYAN.b, 0.8)
    viewButtons[1].text:SetTextColor(CYAN.r, CYAN.g, CYAN.b)

    -- Character filter dropdown (right side of view row)
    local charBtn = CreateFrame("Button", nil, f, "BackdropTemplate")
    charBtn:SetSize(170, 22)
    charBtn:SetBackdrop({
        bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
        edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
        tile = true, tileSize = 8, edgeSize = 10,
        insets = { left = 2, right = 2, top = 2, bottom = 2 },
    })
    charBtn:SetBackdropColor(0.1, 0.1, 0.12, 0.8)
    charBtn:SetBackdropBorderColor(0.3, 0.3, 0.35, 0.6)
    charBtn:SetPoint("RIGHT", f, "TOPRIGHT", -14, viewY + 11)

    local charBtnText = charBtn:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
    charBtnText:SetPoint("LEFT", charBtn, "LEFT", 6, 0)
    charBtnText:SetPoint("RIGHT", charBtn, "RIGHT", -14, 0)
    charBtnText:SetJustifyH("LEFT")
    charBtnText:SetText("All Characters")
    charBtnText:SetTextColor(0.7, 0.7, 0.7)
    charBtn.text = charBtnText

    -- Dropdown arrow
    local charArrow = charBtn:CreateTexture(nil, "OVERLAY")
    charArrow:SetSize(10, 10)
    charArrow:SetPoint("RIGHT", charBtn, "RIGHT", -4, 0)
    charArrow:SetAtlas("arrow-down-minor")
    charArrow:SetVertexColor(0.6, 0.6, 0.6)

    charBtn:SetScript("OnEnter", function(self)
        self:SetBackdropColor(0.15, 0.15, 0.18, 0.9)
        charBtnText:SetTextColor(1, 1, 1)
    end)
    charBtn:SetScript("OnLeave", function(self)
        if not activeCharFilter then
            self:SetBackdropColor(0.1, 0.1, 0.12, 0.8)
            charBtnText:SetTextColor(0.7, 0.7, 0.7)
        else
            self:SetBackdropColor(CYAN.r * 0.3, CYAN.g * 0.3, CYAN.b * 0.3, 1)
            charBtnText:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
        end
    end)

    local function UpdateCharButton()
        if activeCharFilter then
            local shortName = activeCharFilter:match("^([^%-]+)") or activeCharFilter
            charBtnText:SetText(shortName)
            charBtn:SetBackdropColor(CYAN.r * 0.3, CYAN.g * 0.3, CYAN.b * 0.3, 1)
            charBtn:SetBackdropBorderColor(CYAN.r, CYAN.g, CYAN.b, 0.8)
            charBtnText:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
        else
            charBtnText:SetText("All Characters")
            charBtn:SetBackdropColor(0.1, 0.1, 0.12, 0.8)
            charBtn:SetBackdropBorderColor(0.3, 0.3, 0.35, 0.6)
            charBtnText:SetTextColor(0.7, 0.7, 0.7)
        end
    end

    -- Dropdown menu frame
    local charMenu = CreateFrame("Frame", "SKToolsCharFilterMenu", f, "BackdropTemplate")
    charMenu:SetBackdrop({
        bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
        edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
        tile = true, tileSize = 8, edgeSize = 10,
        insets = { left = 2, right = 2, top = 2, bottom = 2 },
    })
    charMenu:SetBackdropColor(0.08, 0.08, 0.1, 0.95)
    charMenu:SetBackdropBorderColor(0.3, 0.3, 0.35, 0.8)
    charMenu:SetFrameStrata("DIALOG")
    charMenu:Hide()

    local charMenuItems = {}

    local function RefreshCharMenu()
        -- Collect unique character names
        local chars = {}
        local seen = {}
        local matches = SKToolsArenaDB and SKToolsArenaDB.matches or {}
        for _, m in ipairs(matches) do
            if m.player and not seen[m.player] then
                seen[m.player] = true
                chars[#chars + 1] = m.player
            end
        end
        table.sort(chars)

        -- Clear old items
        for _, item in ipairs(charMenuItems) do
            item:Hide()
        end
        charMenuItems = {}

        -- "All Characters" option
        local allItems = { { label = "All Characters", value = nil } }
        for _, c in ipairs(chars) do
            allItems[#allItems + 1] = { label = c, value = c }
        end

        local itemHeight = 20
        charMenu:SetSize(charBtn:GetWidth(), 4 + #allItems * itemHeight + 4)
        charMenu:SetPoint("TOP", charBtn, "BOTTOM", 0, -2)

        for i, opt in ipairs(allItems) do
            local item = CreateFrame("Button", nil, charMenu)
            item:SetSize(charBtn:GetWidth() - 8, itemHeight)
            item:SetPoint("TOPLEFT", charMenu, "TOPLEFT", 4, -4 - (i - 1) * itemHeight)

            local itemText = item:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
            itemText:SetPoint("LEFT", 4, 0)
            itemText:SetText(opt.label)

            local isActive = (opt.value == activeCharFilter)
            if isActive then
                itemText:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
            else
                itemText:SetTextColor(0.8, 0.8, 0.8)
            end

            item:SetScript("OnEnter", function()
                if opt.value ~= activeCharFilter then
                    itemText:SetTextColor(1, 1, 1)
                end
                item:SetBackdropColor(0.2, 0.2, 0.25, 1)
            end)
            item:SetScript("OnLeave", function()
                if opt.value ~= activeCharFilter then
                    itemText:SetTextColor(0.8, 0.8, 0.8)
                end
            end)
            item:SetScript("OnClick", function()
                activeCharFilter = opt.value
                compSpecFilters = {}
                analysisSpecFilters = {}
                charMenu:Hide()
                UpdateCharButton()
                UpdateFilterLabel()
                if activeView == "winrates" or activeView == "analysis" or activeView == "players" then
                    RefreshStatsPanel()
                elseif activeView == "graph" then
                    RefreshGraphPanel()
                end
                SKPvPHistory_RefreshUI()
            end)

            charMenuItems[#charMenuItems + 1] = item
        end
    end

    charBtn:SetScript("OnClick", function()
        if charMenu:IsShown() then
            charMenu:Hide()
        else
            RefreshCharMenu()
            charMenu:Show()
        end
    end)

    -- Hide dropdown when clicking elsewhere
    charMenu:SetScript("OnShow", function()
        charMenu:SetScript("OnUpdate", function(self, dt)
            local ok, err = pcall(function()
                if not charBtn:IsMouseOver() and not charMenu:IsMouseOver() then
                    local elapsed = (charMenu.hideTimer or 0) + dt
                    charMenu.hideTimer = elapsed
                    if elapsed > 0.3 then
                        charMenu:Hide()
                        charMenu.hideTimer = 0
                    end
                else
                    charMenu.hideTimer = 0
                end
            end)
            if not ok then self:Hide() end
        end)
    end)
    charMenu:SetScript("OnHide", function()
        charMenu:SetScript("OnUpdate", nil)
        charMenu.hideTimer = 0
    end)

    f.charBtn = charBtn
    f.UpdateCharButton = UpdateCharButton

    -- Clear button
    local clearBtn = ns.CreateThemedButton(f, "Clear", 52, 22, "danger")
    clearBtn:SetPoint("RIGHT", f, "TOPRIGHT", -14, filterY + 11)
    clearBtn:SetScript("OnClick", function()
        StaticPopup_Show("SKTOOLS_CLEAR_HISTORY")
    end)

    StaticPopupDialogs["SKTOOLS_CLEAR_HISTORY"] = {
        text = "Clear all PvP match history?",
        button1 = "Yes",
        button2 = "No",
        OnAccept = function()
            if SKToolsArenaDB then
                SKToolsArenaDB.matches = {}
            end
            playersCacheDirty = true
            compCacheDirty = true
            sessionWins, sessionLosses, sessionRating = 0, 0, 0
            SKPvPHistory_RefreshUI()
            print("|cff00E5EESKTools:|r Match history cleared.")
        end,
        timeout = 0,
        whileDead = true,
        hideOnEscape = true,
    }

    -- Reset Session button (left of Clear)
    local resetBtn = ns.CreateThemedButton(f, "Reset", 60, 22, "secondary")
    resetBtn:SetPoint("RIGHT", clearBtn, "LEFT", -6, 0)

    resetBtn:HookScript("OnEnter", function(self)
        GameTooltip:SetOwner(self, "ANCHOR_TOP")
        GameTooltip:AddLine("Reset Session Stats")
        GameTooltip:AddLine("Resets session W/L and rating without clearing history.", 1, 1, 1, true)
        GameTooltip:Show()
    end)
    resetBtn:HookScript("OnLeave", function()
        GameTooltip:Hide()
    end)
    StaticPopupDialogs["SKTOOLS_RESET_SESSION"] = {
        text = "Reset session stats?\n\nThis will zero out your session wins, losses, and rating change. Your match history will not be affected.",
        button1 = "Reset",
        button2 = "Cancel",
        OnAccept = function()
            sessionWins, sessionLosses, sessionRating = 0, 0, 0
            SKPvPHistory_RefreshUI()
            print("|cff00E5EESKTools:|r Session stats reset.")
        end,
        timeout = 0,
        whileDead = true,
        hideOnEscape = true,
    }

    resetBtn:SetScript("OnClick", function()
        StaticPopup_Show("SKTOOLS_RESET_SESSION")
    end)

    -- Export button (left of Reset)
    local exportBtn = ns.CreateThemedButton(f, "Export", 52, 22, "secondary")
    exportBtn:SetPoint("RIGHT", resetBtn, "LEFT", -6, 0)
    exportBtn:SetScript("OnClick", function() ShowExportFrame() end)

    -- Column header
    local contentY = viewY - 26
    local headerFrame = CreateFrame("Frame", nil, f, "BackdropTemplate")
    headerFrame:SetPoint("TOPLEFT", f, "TOPLEFT", 4, contentY)
    headerFrame:SetPoint("TOPRIGHT", f, "TOPRIGHT", -4, contentY)
    headerFrame:SetHeight(HEADER_HEIGHT)
    headerFrame:SetBackdrop({
        bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
        edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
        tile = true, tileSize = 8, edgeSize = 10,
        insets = { left = 2, right = 2, top = 2, bottom = 2 },
    })
    headerFrame:SetBackdropColor(0.08, 0.1, 0.14, 0.9)
    headerFrame:SetBackdropBorderColor(0.25, 0.25, 0.3, 0.6)
    local headerBg = headerFrame
    f.headerFrame = headerFrame

    local colDefs = {
        { text = "Result",   width = 50,  x = 12,  sortKey = "won" },
        { text = "Mode",     width = 60,  x = 62,  sortKey = "mode" },
        { text = "Map",      width = 150, x = 130, sortKey = "mapName" },
        { text = "Rating",   width = 80,  x = 290, sortKey = "rating" },
        { text = "MMR",      width = 90,  x = 375, sortKey = "mmr" },
        { text = "K/D",      width = 80,  x = 470, sortKey = "kills" },
        { text = "Damage",   width = 65,  x = 555, sortKey = "damage" },
        { text = "Healing",  width = 65,  x = 625, sortKey = "healing" },
        { text = "Comp",     width = 150, x = 700 },
        { text = "Duration", width = 60,  x = 890, sortKey = "duration" },
        { text = "Date",     width = 100, x = 960, sortKey = "timestamp" },
    }
    f.colDefs = colDefs
    f.headerButtons = {}

    for _, col in ipairs(colDefs) do
        if col.sortKey then
            local btn = CreateFrame("Button", nil, headerBg)
            btn:SetPoint("TOPLEFT", headerBg, "TOPLEFT", col.x, 0)
            btn:SetSize(col.width, HEADER_HEIGHT)

            btn.label = btn:CreateFontString(nil, "OVERLAY")
            btn.label:SetFont(STANDARD_TEXT_FONT, 12, "OUTLINE")
            btn.label:SetPoint("LEFT", 0, 0)
            btn.label:SetWidth(col.width)
            btn.label:SetJustifyH("LEFT")
            btn.label:SetTextColor(CYAN.r, CYAN.g, CYAN.b)

            btn.arrow = btn:CreateTexture(nil, "OVERLAY")
            btn.arrow:SetSize(10, 10)
            btn.arrow:SetTexture("Interface\\Buttons\\UI-SortArrow")
            btn.arrow:Hide()

            btn.sortKey = col.sortKey
            btn.colText = col.text

            local function UpdateArrow(hb, show, ascending)
                if show then
                    hb.arrow:ClearAllPoints()
                    hb.arrow:SetPoint("LEFT", hb.label, "LEFT", hb.label:GetStringWidth() + 1, 0)
                    if ascending then
                        hb.arrow:SetTexCoord(0, 1, 0, 1)
                    else
                        hb.arrow:SetTexCoord(0, 1, 1, 0)
                    end
                    hb.arrow:SetVertexColor(1, 1, 1, 0.8)
                    hb.arrow:Show()
                else
                    hb.arrow:Hide()
                end
            end

            local function RefreshHeaderLabels()
                for _, hb in ipairs(f.headerButtons) do
                    if hb.sortKey == sortColumn then
                        hb.label:SetText(hb.colText)
                        hb.label:SetTextColor(1, 1, 1)
                        UpdateArrow(hb, true, sortAscending)
                    else
                        hb.label:SetText(hb.colText)
                        hb.label:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
                        UpdateArrow(hb, false)
                    end
                end
            end

            btn:SetScript("OnClick", function(self)
                if sortColumn == self.sortKey then
                    sortAscending = not sortAscending
                else
                    sortColumn = self.sortKey
                    sortAscending = (self.sortKey ~= "timestamp")
                end
                RefreshHeaderLabels()
                SKPvPHistory_RefreshUI()
            end)

            btn:SetScript("OnEnter", function(self)
                if self.sortKey ~= sortColumn then
                    self.label:SetTextColor(1, 1, 1)
                end
            end)
            btn:SetScript("OnLeave", function(self)
                if self.sortKey ~= sortColumn then
                    self.label:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
                end
            end)

            -- Initial state
            if col.sortKey == sortColumn then
                btn.label:SetText(col.text)
                btn.label:SetTextColor(1, 1, 1)
                UpdateArrow(btn, true, sortAscending)
            else
                btn.label:SetText(col.text)
            end

            f.headerButtons[#f.headerButtons + 1] = btn
        else
            -- Non-sortable header (Comp)
            local header = headerBg:CreateFontString(nil, "OVERLAY")
            header:SetFont(STANDARD_TEXT_FONT, 12, "OUTLINE")
            header:SetPoint("TOPLEFT", headerBg, "TOPLEFT", col.x, -7)
            header:SetWidth(col.width)
            header:SetJustifyH("LEFT")
            header:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
            header:SetText(col.text)
        end
    end

    -- Scroll area
    local scrollArea = CreateFrame("Frame", nil, f)
    scrollArea:SetPoint("TOPLEFT", headerBg, "BOTTOMLEFT", 0, 0)
    scrollArea:SetPoint("BOTTOMRIGHT", f, "BOTTOMRIGHT", -18, 6)

    -- Row frames (virtual scroll pool)
    rows = {}
    for i = 1, VISIBLE_ROWS do
        local row = CreateFrame("Frame", nil, scrollArea, "BackdropTemplate")
        row:SetHeight(ROW_HEIGHT)
        row:SetPoint("TOPLEFT", scrollArea, "TOPLEFT", 0, -((i - 1) * ROW_HEIGHT))
        row:SetPoint("RIGHT", scrollArea, "RIGHT", 0, 0)

        row:SetBackdrop({
            bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
        })
        if i % 2 == 0 then
            row:SetBackdropColor(0.1, 0.1, 0.14, 0.5)
        else
            row:SetBackdropColor(0.06, 0.06, 0.09, 0.3)
        end
        row.evenRow = (i % 2 == 0)

        -- Result
        row.result = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
        row.result:SetPoint("LEFT", 12, 0)
        row.result:SetWidth(50)
        row.result:SetJustifyH("LEFT")

        -- Mode
        row.mode = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
        row.mode:SetPoint("LEFT", 62, 0)
        row.mode:SetWidth(60)
        row.mode:SetJustifyH("LEFT")

        -- Map
        row.mapName = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
        row.mapName:SetPoint("LEFT", 130, 0)
        row.mapName:SetWidth(150)
        row.mapName:SetJustifyH("LEFT")

        -- Rating
        row.rating = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
        row.rating:SetPoint("LEFT", 290, 0)
        row.rating:SetWidth(80)
        row.rating:SetJustifyH("LEFT")

        -- MMR
        row.mmr = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
        row.mmr:SetPoint("LEFT", 375, 0)
        row.mmr:SetWidth(90)
        row.mmr:SetJustifyH("LEFT")

        -- K/D
        row.kd = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
        row.kd:SetPoint("LEFT", 470, 0)
        row.kd:SetWidth(80)
        row.kd:SetJustifyH("LEFT")

        -- Damage
        row.dmg = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
        row.dmg:SetPoint("LEFT", 555, 0)
        row.dmg:SetWidth(65)
        row.dmg:SetJustifyH("LEFT")

        -- Healing
        row.healing = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
        row.healing:SetPoint("LEFT", 625, 0)
        row.healing:SetWidth(65)
        row.healing:SetJustifyH("LEFT")

        -- Comp
        row.comp = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
        row.comp:SetPoint("LEFT", 700, 0)
        row.comp:SetWidth(180)
        row.comp:SetJustifyH("LEFT")

        -- Duration
        row.duration = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
        row.duration:SetPoint("LEFT", 890, 0)
        row.duration:SetWidth(60)
        row.duration:SetJustifyH("LEFT")

        -- Date
        row.time = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
        row.time:SetPoint("LEFT", 960, 0)
        row.time:SetWidth(100)
        row.time:SetJustifyH("LEFT")

        -- Hover highlight + tooltip on hover for comp details
        row:EnableMouse(true)
        row:SetScript("OnEnter", function(self)
            self:SetBackdropColor(CYAN.r * 0.12, CYAN.g * 0.12, CYAN.b * 0.12, 0.6)
            local m = self.tooltipMatch
            if not m then return end
            local ok, err = pcall(ShowMatchTooltip, self, m, self.tooltipKills, self.tooltipDeaths, self.tooltipHKs, self.deathLogText)
            if not ok then print("|cffFF4444SKTools tooltip error:|r " .. tostring(err)) end
        end)
        row:SetScript("OnLeave", function(self)
            if self.evenRow then
                self:SetBackdropColor(0.1, 0.1, 0.14, 0.5)
            else
                self:SetBackdropColor(0.06, 0.06, 0.09, 0.3)
            end
            HideMatchTooltip()
        end)

        rows[i] = row
    end

    -- Scrollbar
    local slider = CreateFrame("Slider", nil, f, "BackdropTemplate")
    slider:SetPoint("TOPRIGHT", scrollArea, "TOPRIGHT", 16, 0)
    slider:SetPoint("BOTTOMRIGHT", scrollArea, "BOTTOMRIGHT", 16, 0)
    slider:SetWidth(16)
    slider:SetOrientation("VERTICAL")
    slider:SetBackdrop({
        bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
        edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
        tile = true, tileSize = 8, edgeSize = 10,
        insets = { left = 2, right = 2, top = 2, bottom = 2 },
    })
    slider:SetBackdropColor(0.05, 0.05, 0.08, 0.7)
    slider:SetBackdropBorderColor(0.25, 0.25, 0.3, 0.5)

    local thumb = slider:CreateTexture(nil, "OVERLAY")
    thumb:SetColorTexture(CYAN.r, CYAN.g, CYAN.b, 0.5)
    thumb:SetSize(12, 40)
    slider:SetThumbTexture(thumb)

    slider:SetMinMaxValues(0, 1)
    slider:SetValue(0)
    slider:SetValueStep(1)

    slider:SetScript("OnValueChanged", function(self, value)
        SKPvPHistory_UpdateRows(math.floor(value))
    end)

    f.slider = slider
    f.scrollArea = scrollArea

    -- Mouse wheel scroll
    scrollArea:EnableMouseWheel(true)
    scrollArea:SetScript("OnMouseWheel", function(self, delta)
        local cur = slider:GetValue()
        slider:SetValue(cur - delta * 3)
    end)

    -- Stats panel (hidden by default, shown when Stats button clicked)
    local statsPanel = CreateFrame("Frame", nil, f, "BackdropTemplate")
    statsPanel:SetPoint("TOPLEFT", f, "TOPLEFT", 4, contentY)
    statsPanel:SetPoint("BOTTOMRIGHT", f, "BOTTOMRIGHT", -4, 6)
    statsPanel:SetBackdrop({
        bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
        edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
        edgeSize = 14,
        insets = { left = 3, right = 3, top = 3, bottom = 3 },
    })
    statsPanel:SetBackdropColor(0.08, 0.08, 0.1, 1)
    statsPanel:SetBackdropBorderColor(0.3, 0.3, 0.35, 0.8)
    statsPanel:Hide()
    f.statsPanel = statsPanel

    -- Stats content containers (no sub-tab bar — view buttons handle switching)
    local winRatesContainer = CreateFrame("Frame", nil, statsPanel)
    winRatesContainer:SetPoint("TOPLEFT", statsPanel, "TOPLEFT", 0, 0)
    winRatesContainer:SetPoint("BOTTOMRIGHT", statsPanel, "BOTTOMRIGHT", 0, 0)
    statsPanel.winRatesContainer = winRatesContainer

    local analysisContainer = CreateFrame("Frame", nil, statsPanel)
    analysisContainer:SetPoint("TOPLEFT", statsPanel, "TOPLEFT", 0, 0)
    analysisContainer:SetPoint("BOTTOMRIGHT", statsPanel, "BOTTOMRIGHT", 0, 0)
    analysisContainer:Hide()
    statsPanel.analysisContainer = analysisContainer

    local playersContainer = CreateFrame("Frame", nil, statsPanel)
    playersContainer:SetPoint("TOPLEFT", statsPanel, "TOPLEFT", 0, 0)
    playersContainer:SetPoint("BOTTOMRIGHT", statsPanel, "BOTTOMRIGHT", 0, 0)
    playersContainer:Hide()
    statsPanel.playersContainer = playersContainer

    local allStatsContainers = {
        winrates = winRatesContainer,
        analysis = analysisContainer,
        players  = playersContainer,
    }
    statsPanel.allStatsContainers = allStatsContainers

    -- Filter context label (shown in stats/graph views when filter is not "All")
    local statsFilterLabel = statsPanel:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
    statsFilterLabel:SetPoint("TOPRIGHT", statsPanel, "TOPRIGHT", -12, -8)
    statsFilterLabel:SetTextColor(CYAN.r, CYAN.g, CYAN.b, 0.7)
    statsPanel.filterLabel = statsFilterLabel

    local STAT_ROW_HEIGHT = 24

    -- Helper: create a scrollable stats section
    local function CreateStatsSection(parent, anchor, width, titleText)
        local section = CreateFrame("Frame", nil, parent)
        if anchor == "LEFT" then
            section:SetPoint("TOPLEFT", 0, 0)
            section:SetPoint("BOTTOMLEFT", 0, 0)
            section:SetWidth(width)
        else
            section:SetPoint("TOPLEFT", parent.compSection, "TOPRIGHT", 10, 0)
            section:SetPoint("BOTTOMRIGHT", parent, "BOTTOMRIGHT", 0, 0)
        end

        local title = section:CreateFontString(nil, "OVERLAY", "GameFontNormal")
        title:SetPoint("TOPLEFT", 10, -8)
        title:SetText(titleText)

        -- Scroll frame
        local scroll = CreateFrame("ScrollFrame", nil, section)
        scroll:SetPoint("TOPLEFT", 0, -26)
        scroll:SetPoint("BOTTOMRIGHT", 0, 0)

        local content = CreateFrame("Frame", nil, scroll)
        content:SetWidth(width - 10)
        content:SetHeight(1)  -- Will be resized dynamically
        scroll:SetScrollChild(content)

        scroll:EnableMouseWheel(true)
        scroll:SetScript("OnMouseWheel", function(self, delta)
            local cur = self:GetVerticalScroll()
            local maxScroll = math.max(0, content:GetHeight() - self:GetHeight())
            local newVal = math.max(0, math.min(maxScroll, cur - delta * STAT_ROW_HEIGHT * 3))
            self:SetVerticalScroll(newVal)
        end)

        -- No data label
        local noData = content:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
        noData:SetPoint("TOPLEFT", 10, -20)
        noData:SetText("|cff888888No data for this filter.|r")
        noData:Hide()

        return {
            section = section,
            scroll = scroll,
            content = content,
            noData = noData,
            rows = {},
        }
    end

    -- Comp section (viewport-based, no ScrollFrame)
    local compSection = CreateFrame("Frame", nil, winRatesContainer)
    compSection:SetPoint("TOPLEFT", 0, 0)
    compSection:SetPoint("BOTTOMLEFT", 0, 0)
    compSection:SetWidth(600)
    compSection:SetClipsChildren(true)
    winRatesContainer.compSection = compSection

    local compTitle = compSection:CreateFontString(nil, "OVERLAY", "GameFontNormal")
    compTitle:SetPoint("TOPLEFT", 10, -8)
    compTitle:SetText("|cff00E5EEWin Rate vs Enemy Comps|r")

    -- Spec pill container (between title and column headers)
    local pillContainer = CreateFrame("Frame", nil, compSection)
    pillContainer:SetPoint("TOPLEFT", compSection, "TOPLEFT", 6, -24)
    pillContainer:SetPoint("RIGHT", compSection, "RIGHT", -20, 0)
    pillContainer:SetHeight(1)
    local specPills = {}

    local compNoData = compSection:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
    compNoData:SetPoint("TOPLEFT", 10, -62)
    compNoData:SetText("|cff888888No data for this filter.|r")
    compNoData:Hide()

    -- Column headers for comp section
    local compHeaders = CreateFrame("Frame", nil, compSection)
    compHeaders:SetPoint("TOPLEFT", 0, -26)
    compHeaders:SetPoint("RIGHT", compSection, "RIGHT")
    compHeaders:SetHeight(14)

    local compColDefs = {
        { x = 14,  w = 390, t = "Comp",   sortKey = "comp" },
        { x = 406, w = 70,  t = "Record", sortKey = "record" },
        { x = 480, w = 45,  t = "Win%",   sortKey = "winpct" },
        { x = 528, w = 50,  t = "Games",  sortKey = "total" },
    }
    local compColButtons = {}

    for _, col in ipairs(compColDefs) do
        local btn = CreateFrame("Button", nil, compHeaders)
        btn:SetPoint("LEFT", col.x, 0)
        btn:SetSize(col.w, 14)
        btn.sortKey = col.sortKey

        local fs = btn:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
        fs:SetPoint("LEFT", 0, 0)
        fs:SetJustifyH("LEFT")
        btn.label = fs

        local arrow = btn:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
        arrow:SetPoint("LEFT", fs, "RIGHT", 2, 0)
        btn.arrow = arrow

        local function UpdateColLabel()
            local active = (winratesSortCol == col.sortKey)
            if active then
                fs:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
                arrow:SetText(winratesSortAsc and " ^" or " v")
                arrow:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
            else
                fs:SetTextColor(0.5, 0.5, 0.5)
                arrow:SetText("")
            end
            fs:SetText(col.t)
        end
        btn.UpdateColLabel = UpdateColLabel
        UpdateColLabel()

        btn:SetScript("OnClick", function()
            if winratesSortCol == col.sortKey then
                winratesSortAsc = not winratesSortAsc
            else
                winratesSortCol = col.sortKey
                winratesSortAsc = (col.sortKey == "comp")
            end
            for _, b in ipairs(compColButtons) do b.UpdateColLabel() end
            RefreshStatsPanel()
        end)
        btn:SetScript("OnEnter", function()
            if winratesSortCol ~= col.sortKey then fs:SetTextColor(1, 1, 1) end
        end)
        btn:SetScript("OnLeave", function()
            if winratesSortCol ~= col.sortKey then fs:SetTextColor(0.5, 0.5, 0.5) end
        end)

        compColButtons[#compColButtons + 1] = btn
    end
    statsPanel.compColButtons = compColButtons

    -- Pre-create comp row pool (visible count calculated dynamically)
    local COMP_ROW_POOL = 25
    local compRows = {}
    for i = 1, COMP_ROW_POOL do
        local row = CreateFrame("Frame", nil, compSection, "BackdropTemplate")
        row:SetHeight(STAT_ROW_HEIGHT)
        row:SetPoint("TOPLEFT", compSection, "TOPLEFT", 6, -42 - ((i - 1) * STAT_ROW_HEIGHT))
        row:SetPoint("TOPRIGHT", compSection, "TOPRIGHT", -20, -42 - ((i - 1) * STAT_ROW_HEIGHT))
        row:SetBackdrop({ bgFile = "Interface\\Tooltips\\UI-Tooltip-Background" })
        row:SetBackdropColor(0.06, 0.06, 0.09, 0.3)

        row.fillBar = row:CreateTexture(nil, "BACKGROUND", nil, 1)
        row.fillBar:SetPoint("TOPLEFT", 0, 0)
        row.fillBar:SetPoint("BOTTOMLEFT", 0, 0)
        row.fillBar:SetWidth(1)
        row.fillBar:SetColorTexture(0, 0.5, 0, 0.15)

        row.compText = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
        row.compText:SetPoint("LEFT", 8, 0)
        row.compText:SetWidth(390)
        row.compText:SetJustifyH("LEFT")

        row.record = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
        row.record:SetPoint("LEFT", 400, 0)
        row.record:SetWidth(70)
        row.record:SetJustifyH("LEFT")

        row.winPct = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
        row.winPct:SetPoint("LEFT", 474, 0)
        row.winPct:SetWidth(45)
        row.winPct:SetJustifyH("LEFT")

        row.games = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
        row.games:SetPoint("LEFT", 522, 0)
        row.games:SetWidth(50)
        row.games:SetJustifyH("LEFT")

        row:EnableMouse(true)
        row:SetScript("OnEnter", function(self)
            self:SetBackdropColor(CYAN.r * 0.12, CYAN.g * 0.12, CYAN.b * 0.12, 0.6)
        end)
        row:SetScript("OnLeave", function(self)
            self:SetBackdropColor(0.06, 0.06, 0.09, 0.3)
        end)

        compRows[i] = row
    end

    -- Comp scrollbar (slider controls row offset)
    local compSlider = CreateFrame("Slider", nil, compSection, "BackdropTemplate")
    compSlider:SetPoint("TOPRIGHT", compSection, "TOPRIGHT", 0, -42)
    compSlider:SetPoint("BOTTOMRIGHT", compSection, "BOTTOMRIGHT", 0, 0)
    compSlider:SetWidth(14)
    compSlider:SetOrientation("VERTICAL")
    compSlider:SetBackdrop({
        bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
        edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
        tile = true, tileSize = 8, edgeSize = 10,
        insets = { left = 2, right = 2, top = 2, bottom = 2 },
    })
    compSlider:SetBackdropColor(0.05, 0.05, 0.08, 0.7)
    compSlider:SetBackdropBorderColor(0.25, 0.25, 0.3, 0.5)

    local compThumb = compSlider:CreateTexture(nil, "OVERLAY")
    compThumb:SetColorTexture(CYAN.r, CYAN.g, CYAN.b, 0.5)
    compThumb:SetSize(10, 40)
    compSlider:SetThumbTexture(compThumb)

    compSlider:SetMinMaxValues(0, 1)
    compSlider:SetValue(0)
    statsPanel.compSlider = compSlider

    -- Store comp data for viewport updates
    statsPanel.compRows = compRows
    statsPanel.compList = {}
    statsPanel.compNoData = compNoData
    statsPanel.visibleCompRows = 19

    -- Update visible comp rows from offset
    local function UpdateCompRows(offset)
        offset = math.floor(offset + 0.5)
        local list = statsPanel.compList
        local visibleCount = statsPanel.visibleCompRows or 19
        for i = 1, COMP_ROW_POOL do
            local row = compRows[i]
            if i <= visibleCount then
                local data = list[offset + i]
                if data then
                    row:Show()
                    row.compText:SetText(GetSpecCompLabel(data.team))
                    local wColor = data.wins > 0 and "|cff00ff00" or "|cffbbbbbb"
                    local lColor = data.losses > 0 and "|cffff4444" or "|cffbbbbbb"
                    row.record:SetText(wColor .. data.wins .. "W|r - " .. lColor .. data.losses .. "L|r")
                    local pct = data.total > 0 and (data.wins / data.total * 100) or 0
                    local pctColor = pct > 50 and "|cff00ff00" or (pct < 50 and "|cffff4444" or "|cffbbbbbb")
                    row.winPct:SetText(pctColor .. string.format("%.0f%%", pct) .. "|r")
                    row.games:SetText("|cff888888" .. data.total .. "|r")
                    local rowWidth = row:GetWidth()
                    if rowWidth and rowWidth > 0 then
                        row.fillBar:SetWidth(math.max(1, rowWidth * (pct / 100)))
                    end
                    row.fillBar:SetColorTexture(0, pct > 50 and 0.5 or 0.1, 0, 0.15)
                else
                    row:Hide()
                end
            else
                row:Hide()
            end
        end
    end
    statsPanel.UpdateCompRows = UpdateCompRows

    compSlider:SetScript("OnValueChanged", function(self, value)
        UpdateCompRows(value)
    end)

    compSection:EnableMouseWheel(true)
    compSection:SetScript("OnMouseWheel", function(self, delta)
        local cur = compSlider:GetValue()
        local _, maxVal = compSlider:GetMinMaxValues()
        local newVal = math.max(0, math.min(maxVal, cur - delta * 3))
        compSlider:SetValue(newVal)
    end)

    -- Rebuild spec pill filter buttons from comp data
    local PILL_HEIGHT = 20
    local PILL_HPAD = 4
    local PILL_VPAD = 3

    local function RebuildSpecPills(rawCompList)
        -- Collect unique specs
        local uniqueSpecs = {}
        local seen = {}
        for _, entry in ipairs(rawCompList) do
            for _, p in ipairs(entry.team) do
                local spec = ResolveSpec(p.spec)
                local cls = p.class or "UNKNOWN"
                local key = spec .. " " .. cls
                if not seen[key] then
                    seen[key] = true
                    uniqueSpecs[#uniqueSpecs + 1] = { spec = spec, class = cls, key = key }
                end
            end
        end
        table.sort(uniqueSpecs, function(a, b)
            if a.class ~= b.class then return a.class < b.class end
            return a.spec < b.spec
        end)

        -- Remove stale filters
        for key in pairs(compSpecFilters) do
            if not seen[key] then compSpecFilters[key] = nil end
        end

        local containerWidth = pillContainer:GetWidth()
        if containerWidth < 50 then containerWidth = 500 end
        local curX, curRow, pillIdx = 0, 0, 0

        for _, si in ipairs(uniqueSpecs) do
            pillIdx = pillIdx + 1
            local pill = specPills[pillIdx]
            if not pill then
                pill = CreateFrame("Button", nil, pillContainer, "BackdropTemplate")
                pill:SetHeight(PILL_HEIGHT)
                pill:SetBackdrop({
                    bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
                    edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
                    tile = true, tileSize = 8, edgeSize = 10,
                    insets = { left = 2, right = 2, top = 2, bottom = 2 },
                })
                pill.icon = pill:CreateTexture(nil, "ARTWORK")
                pill.icon:SetSize(14, 14)
                pill.icon:SetPoint("LEFT", 4, 0)
                pill.label = pill:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
                pill.label:SetPoint("LEFT", pill.icon, "RIGHT", 3, 0)
                pill.label:SetJustifyH("LEFT")
                pill:SetScript("OnClick", function(self)
                    if compSpecFilters[self.specKey] then
                        compSpecFilters[self.specKey] = nil
                    else
                        compSpecFilters[self.specKey] = true
                    end
                    RefreshStatsPanel()
                end)
                pill:SetScript("OnEnter", function(self)
                    if not compSpecFilters[self.specKey] then
                        self:SetBackdropColor(0.18, 0.18, 0.22, 0.9)
                    end
                end)
                pill:SetScript("OnLeave", function(self)
                    if not compSpecFilters[self.specKey] then
                        self:SetBackdropColor(0.12, 0.12, 0.15, 0.7)
                    end
                end)
                specPills[pillIdx] = pill
            end

            pill.specKey = si.key
            local atlas = CLASS_ICON_ATLAS[si.class]
            if atlas then pill.icon:SetAtlas(atlas) else pill.icon:SetTexture(nil) end
            local hex = GetClassColorHex(si.class)
            pill.label:SetText("|cff" .. hex .. si.spec .. "|r")
            pill.label:SetWidth(0)
            local textW = pill.label:GetStringWidth() or 40
            local pillW = 4 + 14 + 3 + textW + 6
            pill:SetWidth(pillW)

            -- Flow layout
            if curX + pillW > containerWidth and curX > 0 then
                curRow = curRow + 1
                curX = 0
            end
            pill:ClearAllPoints()
            pill:SetPoint("TOPLEFT", pillContainer, "TOPLEFT", curX, -(curRow * (PILL_HEIGHT + PILL_VPAD)))
            curX = curX + pillW + PILL_HPAD

            -- Active vs inactive style
            if compSpecFilters[si.key] then
                pill:SetBackdropColor(CYAN.r * 0.25, CYAN.g * 0.25, CYAN.b * 0.25, 1)
                pill:SetBackdropBorderColor(CYAN.r, CYAN.g, CYAN.b, 0.8)
            else
                pill:SetBackdropColor(0.12, 0.12, 0.15, 0.7)
                pill:SetBackdropBorderColor(0.25, 0.25, 0.3, 0.4)
            end
            pill:Show()
        end

        -- Hide unused pills
        for i = pillIdx + 1, #specPills do specPills[i]:Hide() end

        -- Set container height
        local numRows = curRow + 1
        local totalH = pillIdx > 0 and (numRows * PILL_HEIGHT + (numRows - 1) * PILL_VPAD) or 0
        pillContainer:SetHeight(totalH)
        return totalH
    end
    statsPanel.RebuildSpecPills = RebuildSpecPills

    -- Reposition headers, rows, slider after pill area changes
    local function RepositionCompLayout(pillAreaHeight)
        local headersY = pillAreaHeight > 0 and (-24 - pillAreaHeight - 6) or -26
        local rowsStartY = headersY - 16

        compHeaders:ClearAllPoints()
        compHeaders:SetPoint("TOPLEFT", compSection, "TOPLEFT", 0, headersY)
        compHeaders:SetPoint("RIGHT", compSection, "RIGHT")

        for i = 1, COMP_ROW_POOL do
            compRows[i]:ClearAllPoints()
            compRows[i]:SetPoint("TOPLEFT", compSection, "TOPLEFT", 6, rowsStartY - ((i - 1) * STAT_ROW_HEIGHT))
            compRows[i]:SetPoint("TOPRIGHT", compSection, "TOPRIGHT", -20, rowsStartY - ((i - 1) * STAT_ROW_HEIGHT))
        end

        compSlider:ClearAllPoints()
        compSlider:SetPoint("TOPRIGHT", compSection, "TOPRIGHT", 0, rowsStartY)
        compSlider:SetPoint("BOTTOMRIGHT", compSection, "BOTTOMRIGHT", 0, 0)

        -- Recalculate visible rows
        local sectionH = compSection:GetHeight()
        if sectionH and sectionH > 0 then
            local available = sectionH + rowsStartY  -- rowsStartY is negative
            statsPanel.visibleCompRows = math.max(1, math.floor(available / STAT_ROW_HEIGHT))
        else
            statsPanel.visibleCompRows = 19
        end

        compNoData:ClearAllPoints()
        compNoData:SetPoint("TOPLEFT", 10, rowsStartY - 20)
    end
    statsPanel.RepositionCompLayout = RepositionCompLayout

    -- Divider line between sections
    local divider = winRatesContainer:CreateTexture(nil, "ARTWORK")
    divider:SetPoint("TOP", winRatesContainer.compSection, "TOPRIGHT", 5, -8)
    divider:SetPoint("BOTTOM", winRatesContainer.compSection, "BOTTOMRIGHT", 5, 8)
    divider:SetWidth(1)
    divider:SetColorTexture(CYAN.r, CYAN.g, CYAN.b, 0.2)

    local mapSec = CreateStatsSection(winRatesContainer, "RIGHT", 390, "|cff00E5EEWin Rate by Map|r")
    statsPanel.map = mapSec

    -- Helper: ensure enough stat rows exist in a section, creating new ones as needed
    local function EnsureStatRows(sec, count, rowType)
        while #sec.rows < count do
            local i = #sec.rows + 1
            local row = CreateFrame("Frame", nil, sec.content, "BackdropTemplate")
            row:SetHeight(STAT_ROW_HEIGHT)
            row:SetPoint("TOPLEFT", sec.content, "TOPLEFT", 6, -((i - 1) * STAT_ROW_HEIGHT))
            row:SetPoint("RIGHT", sec.content, "RIGHT", -6, 0)
            row:SetBackdrop({ bgFile = "Interface\\Tooltips\\UI-Tooltip-Background" })
            row:SetBackdropColor(0.06, 0.06, 0.09, 0.3)

            -- Win% fill bar
            row.fillBar = row:CreateTexture(nil, "BACKGROUND", nil, 1)
            row.fillBar:SetPoint("TOPLEFT", 0, 0)
            row.fillBar:SetPoint("BOTTOMLEFT", 0, 0)
            row.fillBar:SetWidth(1)
            row.fillBar:SetColorTexture(0, 0.5, 0, 0.15)

            if rowType == "comp" then
                row.compText = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
                row.compText:SetPoint("LEFT", 8, 0)
                row.compText:SetWidth(380)
                row.compText:SetJustifyH("LEFT")
            else
                row.mapName = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
                row.mapName:SetPoint("LEFT", 8, 0)
                row.mapName:SetWidth(200)
                row.mapName:SetJustifyH("LEFT")
            end

            row.record = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
            row.record:SetPoint("LEFT", rowType == "comp" and 395 or 215, 0)
            row.record:SetWidth(80)
            row.record:SetJustifyH("LEFT")

            row.winPct = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
            row.winPct:SetPoint("LEFT", rowType == "comp" and 480 or 300, 0)
            row.winPct:SetWidth(50)
            row.winPct:SetJustifyH("LEFT")

            if rowType == "comp" then
                row.games = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
                row.games:SetPoint("LEFT", 535, 0)
                row.games:SetWidth(80)
                row.games:SetJustifyH("LEFT")
            end

            row:EnableMouse(true)
            row:SetScript("OnEnter", function(self)
                self:SetBackdropColor(CYAN.r * 0.12, CYAN.g * 0.12, CYAN.b * 0.12, 0.6)
                if self.tooltipText then
                    GameTooltip:SetOwner(self, "ANCHOR_RIGHT")
                    GameTooltip:AddLine("|cff00E5EEEnemy Composition|r")
                    for line in self.tooltipText:gmatch("[^\n]+") do
                        GameTooltip:AddLine(line)
                    end
                    GameTooltip:Show()
                end
            end)
            row:SetScript("OnLeave", function(self)
                self:SetBackdropColor(0.06, 0.06, 0.09, 0.3)
                GameTooltip:Hide()
            end)

            sec.rows[i] = row
        end
    end
    statsPanel.EnsureStatRows = EnsureStatRows
    statsPanel.STAT_ROW_HEIGHT = STAT_ROW_HEIGHT

    --------------------------------
    -- Analysis Panel (inside analysisContainer)
    --------------------------------
    -- Fixed header area (does not scroll)
    local analysisFixedHeader = CreateFrame("Frame", nil, analysisContainer)
    analysisFixedHeader:SetPoint("TOPLEFT", 0, 0)
    analysisFixedHeader:SetPoint("TOPRIGHT", 0, 0)
    analysisFixedHeader:SetHeight(30)  -- base height; updated after pill rebuild

    -- Scrollable area below the fixed header
    local analysisScroll = CreateFrame("ScrollFrame", nil, analysisContainer)
    analysisScroll:SetPoint("TOPLEFT", analysisFixedHeader, "BOTTOMLEFT", 0, 0)
    analysisScroll:SetPoint("BOTTOMRIGHT", 0, 0)

    local analysisContent = CreateFrame("Frame", nil, analysisScroll)
    analysisContent:SetWidth(1)  -- will be set dynamically
    analysisContent:SetHeight(1)
    analysisScroll:SetScrollChild(analysisContent)

    analysisScroll:EnableMouseWheel(true)
    analysisScroll:SetScript("OnMouseWheel", function(self, delta)
        local cur = self:GetVerticalScroll()
        local maxScroll = math.max(0, analysisContent:GetHeight() - self:GetHeight())
        local newVal = math.max(0, math.min(maxScroll, cur - delta * 60))
        self:SetVerticalScroll(newVal)
    end)

    -- Set content width on show
    analysisContainer:SetScript("OnShow", function(self)
        analysisContent:SetWidth(self:GetWidth() - 10)
    end)

    local analysisNoData = analysisContent:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
    analysisNoData:SetPoint("TOPLEFT", 15, -10)
    analysisNoData:SetText("|cff888888No arena matches with death data for this filter.|r")
    analysisNoData:Hide()

    statsPanel.analysisScroll = analysisScroll
    statsPanel.analysisContent = analysisContent
    statsPanel.analysisNoData = analysisNoData
    statsPanel.analysisFixedHeader = analysisFixedHeader

    -- Pre-create analysis section headers (in fixed header, not scroll content)
    local killHeader = analysisFixedHeader:CreateFontString(nil, "OVERLAY", "GameFontNormal")
    killHeader:SetPoint("TOPLEFT", 10, -10)
    killHeader:SetText("|cff00E5EEKill & Death Analysis|r")
    statsPanel.killHeader = killHeader

    -- "My Group" comp filter button (in fixed header)
    local compFilterBtn = CreateFrame("Button", nil, analysisFixedHeader, "BackdropTemplate")
    compFilterBtn:SetSize(90, 20)
    compFilterBtn:SetPoint("LEFT", killHeader, "RIGHT", 12, 0)
    compFilterBtn:SetBackdrop({
        bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
        edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
        tile = true, tileSize = 8, edgeSize = 10,
        insets = { left = 2, right = 2, top = 2, bottom = 2 },
    })
    compFilterBtn:SetBackdropColor(0.1, 0.1, 0.12, 0.8)
    compFilterBtn:SetBackdropBorderColor(0.3, 0.3, 0.35, 0.6)
    local compFilterText = compFilterBtn:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
    compFilterText:SetPoint("CENTER")
    compFilterText:SetText("My Group")
    compFilterText:SetTextColor(0.7, 0.7, 0.7)
    compFilterBtn.text = compFilterText

    local analysisSpecPills  -- forward-declare for ReadGroupAndSelectPills
    local function ReadGroupAndSelectPills()
        local members = {}  -- { spec = "Arms", class = "WARRIOR" } or { spec = nil, class = "WARRIOR" }
        -- Player (always reliable)
        local specID = GetSpecialization()
        local playerSpec
        if specID then
            local _, name = GetSpecializationInfo(specID)
            playerSpec = name
        end
        local _, playerCls = UnitClass("player")
        members[#members + 1] = { spec = playerSpec, class = playerCls or "UNKNOWN" }
        -- Party members
        for i = 1, 4 do
            local unit = "party" .. i
            if UnitExists(unit) then
                local pSpec
                local pSpecID = GetInspectSpecialization and GetInspectSpecialization(unit)
                if pSpecID and pSpecID > 0 then
                    local _, sName = GetSpecializationInfoByID(pSpecID)
                    pSpec = sName
                end
                local _, pCls = UnitClass(unit)
                members[#members + 1] = { spec = pSpec, class = pCls or "UNKNOWN" }
            end
        end
        if #members < 2 then return nil end

        -- Match each member to a pill key
        local selectedKeys = {}
        for _, m in ipairs(members) do
            if m.spec then
                -- Exact spec+class key
                selectedKeys[#selectedKeys + 1] = m.spec .. " " .. m.class
            else
                -- Spec unknown — find a pill matching just this class
                for _, pill in ipairs(analysisSpecPills) do
                    if pill:IsShown() and pill.specKey and pill.specKey:match("%s" .. m.class .. "$") then
                        selectedKeys[#selectedKeys + 1] = pill.specKey
                        break
                    end
                end
            end
        end
        return selectedKeys
    end

    compFilterBtn:SetScript("OnClick", function()
        -- Request inspect data for party members
        for i = 1, 4 do
            local unit = "party" .. i
            if UnitExists(unit) and CanInspect(unit) then
                NotifyInspect(unit)
            end
        end
        -- Short delay to let inspect data arrive, then select pills
        C_Timer.After(0.3, function()
            local selectedKeys = ReadGroupAndSelectPills()
            if not selectedKeys then
                print("|cff00E5EESKTools:|r Join a group to use this filter.")
                return
            end
            -- Toggle: if all are already selected, clear; otherwise set
            local allSelected = #selectedKeys > 0
            for _, key in ipairs(selectedKeys) do
                if not analysisSpecFilters[key] then allSelected = false; break end
            end
            analysisSpecFilters = {}
            if not allSelected then
                for _, key in ipairs(selectedKeys) do
                    analysisSpecFilters[key] = true
                end
            end
            RefreshAnalysisPanel()
        end)
    end)
    compFilterBtn:SetScript("OnEnter", function(self)
        self:SetBackdropColor(0.15, 0.15, 0.18, 0.9)
        compFilterText:SetTextColor(1, 1, 1)
        GameTooltip:SetOwner(self, "ANCHOR_TOP")
        GameTooltip:AddLine("Select pills for your current group comp")
        GameTooltip:Show()
    end)
    compFilterBtn:SetScript("OnLeave", function(self)
        self:SetBackdropColor(0.1, 0.1, 0.12, 0.8)
        compFilterText:SetTextColor(0.7, 0.7, 0.7)
        GameTooltip:Hide()
    end)
    statsPanel.compFilterBtn = compFilterBtn

    -- Sub-tab buttons: "Your Kills" / "Deaths"
    local analysisSubTabs = {}
    local subTabDefs = {
        { key = "kills",  label = "Wins" },
        { key = "deaths", label = "Losses" },
    }
    local prevSubTab
    for i, def in ipairs(subTabDefs) do
        local btnW = math.max(60, 10 + #def.label * 7)
        local btn = ns.CreateThemedButton(analysisFixedHeader, def.label, btnW, 20, "secondary")
        btn.text = btn.label
        btn.subKey = def.key
        if i == 1 then
            btn:SetPoint("TOPLEFT", analysisFixedHeader, "TOPLEFT", 10, -28)
        else
            btn:SetPoint("LEFT", prevSubTab, "RIGHT", 4, 0)
        end
        btn:SetScript("OnClick", function()
            analysisSubTab = def.key
            for _, b in ipairs(analysisSubTabs) do
                if b.subKey == analysisSubTab then
                    b:SetBackdropColor(CYAN.r * 0.3, CYAN.g * 0.3, CYAN.b * 0.3, 1)
                    b:SetBackdropBorderColor(CYAN.r, CYAN.g, CYAN.b, 0.8)
                    b.text:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
                else
                    b:SetBackdropColor(0.1, 0.1, 0.12, 0.8)
                    b:SetBackdropBorderColor(0.3, 0.3, 0.35, 0.6)
                    b.text:SetTextColor(0.7, 0.7, 0.7)
                end
            end
            RefreshAnalysisPanel()
        end)
        analysisSubTabs[#analysisSubTabs + 1] = btn
        prevSubTab = btn
    end
    -- Highlight default sub-tab
    analysisSubTabs[1]:SetBackdropColor(CYAN.r * 0.3, CYAN.g * 0.3, CYAN.b * 0.3, 1)
    analysisSubTabs[1]:SetBackdropBorderColor(CYAN.r, CYAN.g, CYAN.b, 0.8)
    analysisSubTabs[1].text:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
    statsPanel.analysisSubTabs = analysisSubTabs

    -- Spec pill container for analysis tab (in fixed header, not scroll content)
    local analysisPillContainer = CreateFrame("Frame", nil, analysisFixedHeader)
    analysisPillContainer:SetPoint("TOPLEFT", analysisFixedHeader, "TOPLEFT", 6, -52)
    analysisPillContainer:SetPoint("RIGHT", analysisFixedHeader, "RIGHT", -20, 0)
    analysisPillContainer:SetHeight(1)
    analysisSpecPills = {}
    local A_PILL_MAX = 3

    local A_PILL_HEIGHT = 20
    local A_PILL_HPAD = 4
    local A_PILL_VPAD = 3

    local function CountActiveAnalysisPills()
        local n = 0
        for _ in pairs(analysisSpecFilters) do n = n + 1 end
        return n
    end

    local function RebuildAnalysisPills(killList, lossList)
        -- Collect unique specs from all teams (flat lists)
        local uniqueSpecs = {}
        local seen = {}
        local function CollectFromList(list)
            for _, entry in ipairs(list) do
                if entry.playerTeam then
                    for _, p in ipairs(entry.playerTeam) do
                        local spec = ResolveSpec(p.spec)
                        local cls = p.class or "UNKNOWN"
                        local key = spec .. " " .. cls
                        if not seen[key] and spec ~= "Unknown" then
                            seen[key] = true
                            uniqueSpecs[#uniqueSpecs + 1] = { spec = spec, class = cls, key = key }
                        end
                    end
                end
                if entry.enemyTeam then
                    for _, p in ipairs(entry.enemyTeam) do
                        local spec = ResolveSpec(p.spec)
                        local cls = p.class or "UNKNOWN"
                        local key = spec .. " " .. cls
                        if not seen[key] and spec ~= "Unknown" then
                            seen[key] = true
                            uniqueSpecs[#uniqueSpecs + 1] = { spec = spec, class = cls, key = key }
                        end
                    end
                end
            end
        end
        CollectFromList(killList)
        CollectFromList(lossList)
        table.sort(uniqueSpecs, function(a, b)
            if a.class ~= b.class then return a.class < b.class end
            return a.spec < b.spec
        end)

        -- Remove stale filters
        for key in pairs(analysisSpecFilters) do
            if not seen[key] then analysisSpecFilters[key] = nil end
        end

        local containerWidth = analysisPillContainer:GetWidth()
        if containerWidth < 50 then containerWidth = 500 end
        local curX, curRow, pillIdx = 0, 0, 0

        for _, si in ipairs(uniqueSpecs) do
            pillIdx = pillIdx + 1
            local pill = analysisSpecPills[pillIdx]
            if not pill then
                pill = CreateFrame("Button", nil, analysisPillContainer, "BackdropTemplate")
                pill:SetHeight(A_PILL_HEIGHT)
                pill:SetBackdrop({
                    bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
                    edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
                    tile = true, tileSize = 8, edgeSize = 10,
                    insets = { left = 2, right = 2, top = 2, bottom = 2 },
                })
                pill.icon = pill:CreateTexture(nil, "ARTWORK")
                pill.icon:SetSize(14, 14)
                pill.icon:SetPoint("LEFT", 4, 0)
                pill.label = pill:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
                pill.label:SetPoint("LEFT", pill.icon, "RIGHT", 3, 0)
                pill.label:SetJustifyH("LEFT")
                pill:SetScript("OnClick", function(self)
                    if analysisSpecFilters[self.specKey] then
                        analysisSpecFilters[self.specKey] = nil
                    elseif CountActiveAnalysisPills() < A_PILL_MAX then
                        analysisSpecFilters[self.specKey] = true
                    end
                    RefreshAnalysisPanel()
                end)
                pill:SetScript("OnEnter", function(self)
                    if not analysisSpecFilters[self.specKey] then
                        self:SetBackdropColor(0.18, 0.18, 0.22, 0.9)
                    end
                end)
                pill:SetScript("OnLeave", function(self)
                    if not analysisSpecFilters[self.specKey] then
                        self:SetBackdropColor(0.12, 0.12, 0.15, 0.7)
                    end
                end)
                analysisSpecPills[pillIdx] = pill
            end

            pill.specKey = si.key
            local atlas = CLASS_ICON_ATLAS[si.class]
            if atlas then pill.icon:SetAtlas(atlas) else pill.icon:SetTexture(nil) end
            local hex = GetClassColorHex(si.class)
            pill.label:SetText("|cff" .. hex .. si.spec .. "|r")
            pill.label:SetWidth(0)
            local textW = pill.label:GetStringWidth() or 40
            local pillW = 4 + 14 + 3 + textW + 6
            pill:SetWidth(pillW)

            -- Flow layout
            if curX + pillW > containerWidth and curX > 0 then
                curRow = curRow + 1
                curX = 0
            end
            pill:ClearAllPoints()
            pill:SetPoint("TOPLEFT", analysisPillContainer, "TOPLEFT", curX, -(curRow * (A_PILL_HEIGHT + A_PILL_VPAD)))
            curX = curX + pillW + A_PILL_HPAD

            -- Active vs inactive style
            if analysisSpecFilters[si.key] then
                pill:SetBackdropColor(CYAN.r * 0.25, CYAN.g * 0.25, CYAN.b * 0.25, 1)
                pill:SetBackdropBorderColor(CYAN.r, CYAN.g, CYAN.b, 0.8)
            else
                pill:SetBackdropColor(0.12, 0.12, 0.15, 0.7)
                pill:SetBackdropBorderColor(0.25, 0.25, 0.3, 0.4)
            end
            pill:Show()
        end

        -- Hide unused pills
        for i = pillIdx + 1, #analysisSpecPills do analysisSpecPills[i]:Hide() end

        -- Set container height and update fixed header height
        local numRows = curRow + 1
        local totalH = pillIdx > 0 and (numRows * A_PILL_HEIGHT + (numRows - 1) * A_PILL_VPAD) or 0
        analysisPillContainer:SetHeight(totalH)
        -- Layout: title(28) + sub-tabs(24) + pills + gap(6) + colHeaders(16) + gap(4)
        local headerH = 52 + totalH + (totalH > 0 and 6 or 0) + 20
        analysisFixedHeader:SetHeight(headerH)
        -- Reposition column headers at bottom of fixed header
        if statsPanel.analysisColContainer then
            statsPanel.analysisColContainer:ClearAllPoints()
            statsPanel.analysisColContainer:SetPoint("BOTTOMLEFT", analysisFixedHeader, "BOTTOMLEFT", 0, 4)
            statsPanel.analysisColContainer:SetPoint("BOTTOMRIGHT", analysisFixedHeader, "BOTTOMRIGHT", -16, 4)
        end
        return totalH
    end
    statsPanel.RebuildAnalysisPills = RebuildAnalysisPills

    -- Sortable column headers for analysis matchup rows
    local analysisColContainer = CreateFrame("Frame", nil, analysisFixedHeader)
    analysisColContainer:SetPoint("BOTTOMLEFT", analysisFixedHeader, "BOTTOMLEFT", 0, 4)
    analysisColContainer:SetPoint("BOTTOMRIGHT", analysisFixedHeader, "BOTTOMRIGHT", -16, 4)
    analysisColContainer:SetHeight(14)
    statsPanel.analysisColContainer = analysisColContainer

    -- Column positions use RIGHT-relative anchoring to spread across full width
    -- Enemy comp takes left ~45%, kill target middle ~20%, stats right ~35%
    local analysisColDefs = {
        { x = 20,  w = 300, t = "vs Enemy Comp", sortKey = "enemyComp" },
        { x = 480, w = 180, t = "Kill Target",   sortKey = "target" },
        { x = 670, w = 80,  t = "Rate",          sortKey = "rate" },
        { x = 760, w = 80,  t = "Avg Time",      sortKey = "avgTime" },
        { x = 850, w = 80,  t = "Games",         sortKey = "games" },
    }
    local analysisColButtons = {}
    for _, col in ipairs(analysisColDefs) do
        local btn = CreateFrame("Button", nil, analysisColContainer)
        btn:SetPoint("LEFT", col.x, 0)
        btn:SetSize(col.w, 14)
        btn.sortKey = col.sortKey

        local fs = btn:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
        fs:SetPoint("LEFT", 0, 0)
        fs:SetJustifyH("LEFT")
        btn.label = fs

        local arrow = btn:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
        arrow:SetPoint("LEFT", fs, "RIGHT", 2, 0)
        btn.arrow = arrow

        local function UpdateLabel()
            local active = (analysisSortCol == col.sortKey)
            if active then
                fs:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
                arrow:SetText(analysisSortAsc and " ^" or " v")
                arrow:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
            else
                fs:SetTextColor(0.5, 0.5, 0.5)
                arrow:SetText("")
            end
            -- Dynamic column text for Kill Target / Targeted
            if col.sortKey == "target" then
                fs:SetText(analysisSubTab == "kills" and "Kill Target" or "Targeted")
            else
                fs:SetText(col.t)
            end
        end
        btn.UpdateLabel = UpdateLabel
        UpdateLabel()

        btn:SetScript("OnClick", function()
            if analysisSortCol == col.sortKey then
                analysisSortAsc = not analysisSortAsc
            else
                analysisSortCol = col.sortKey
                analysisSortAsc = (col.sortKey == "enemyComp" or col.sortKey == "target")
            end
            for _, b in ipairs(analysisColButtons) do b.UpdateLabel() end
            RefreshAnalysisPanel()
        end)
        btn:SetScript("OnEnter", function()
            if analysisSortCol ~= col.sortKey then fs:SetTextColor(1, 1, 1) end
        end)
        btn:SetScript("OnLeave", function()
            if analysisSortCol ~= col.sortKey then fs:SetTextColor(0.5, 0.5, 0.5) end
        end)

        analysisColButtons[#analysisColButtons + 1] = btn
    end
    statsPanel.analysisColButtons = analysisColButtons

    statsPanel.killRows = {}  -- mixed heading + matchup frames

    --------------------------------
    -- Players (Scouting) Panel — Virtual Scroll
    --------------------------------
    local PLAYERS_ROW_TOP = -30  -- where column headers start
    local PLAYERS_DATA_TOP = -46 -- where data rows start
    local PLAYER_ROW_HEIGHT = 24
    local PLAYER_ROW_POOL = 30

    playersContainer:SetClipsChildren(true)

    local playersNoData = playersContainer:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
    playersNoData:SetPoint("TOPLEFT", 15, -40)
    playersNoData:SetText("|cff888888No arena/shuffle/skirmish matches with player data yet.|r")
    playersNoData:Hide()
    statsPanel.playersNoData = playersNoData

    -- Header row: title + search box
    local playersHeader = playersContainer:CreateFontString(nil, "OVERLAY", "GameFontNormal")
    playersHeader:SetPoint("TOPLEFT", 10, -8)
    playersHeader:SetText("|cff00E5EEEnemy Players Encountered|r")
    statsPanel.playersHeader = playersHeader

    local searchBox = CreateFrame("EditBox", "SKToolsPlayerSearch", playersContainer, "InputBoxTemplate")
    searchBox:SetSize(180, 20)
    searchBox:SetPoint("TOPRIGHT", playersContainer, "TOPRIGHT", -28, -6)
    searchBox:SetAutoFocus(false)
    searchBox:SetFontObject("GameFontHighlightSmall")
    searchBox:SetTextInsets(4, 18, 0, 0)

    local searchPlaceholder = searchBox:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
    searchPlaceholder:SetPoint("LEFT", 6, 0)
    searchPlaceholder:SetText("|cff666666Search players...|r")
    searchBox.placeholder = searchPlaceholder

    local clearSearchBtn = CreateFrame("Button", nil, searchBox)
    clearSearchBtn:SetSize(14, 14)
    clearSearchBtn:SetPoint("RIGHT", -2, 0)
    clearSearchBtn:SetNormalFontObject("GameFontHighlightSmall")
    clearSearchBtn:SetText("|cff999999x|r")
    clearSearchBtn:Hide()
    clearSearchBtn:SetScript("OnClick", function()
        searchBox:SetText("")
        searchBox:ClearFocus()
        playersSearchText = ""
        clearSearchBtn:Hide()
        searchPlaceholder:Show()
        RefreshPlayersPanel()
    end)

    searchBox:SetScript("OnTextChanged", function(self)
        local text = self:GetText()
        playersSearchText = text:lower()
        searchPlaceholder:SetShown(text == "")
        clearSearchBtn:SetShown(text ~= "")
        RefreshPlayersPanel()
    end)
    searchBox:SetScript("OnEscapePressed", function(self)
        self:ClearFocus()
    end)
    searchBox:SetScript("OnEnterPressed", function(self)
        self:ClearFocus()
    end)
    statsPanel.playersSearchBox = searchBox

    -- Sortable column headers
    local colHeaderContainer = CreateFrame("Frame", nil, playersContainer)
    colHeaderContainer:SetPoint("TOPLEFT", 0, PLAYERS_ROW_TOP)
    colHeaderContainer:SetPoint("RIGHT", playersContainer, "RIGHT", -16, 0)
    colHeaderContainer:SetHeight(14)

    local colDefs = {
        { x = 8,   w = 220, t = "Player",    sortKey = "name" },
        { x = 235, w = 150, t = "Spec",      sortKey = "spec" },
        { x = 390, w = 120, t = "Arena W/L", sortKey = "arenaWins" },
        { x = 515, w = 80,  t = "Shuffles",  sortKey = "shuffleCount" },
        { x = 595, w = 70,  t = "Skirm W/L", sortKey = "skirmWins" },
        { x = 670, w = 100, t = "Last Seen", sortKey = "lastSeen" },
    }
    local colButtons = {}
    for _, col in ipairs(colDefs) do
        local btn = CreateFrame("Button", nil, colHeaderContainer)
        btn:SetPoint("LEFT", col.x, 0)
        btn:SetSize(col.w, 14)
        btn.sortKey = col.sortKey

        local fs = btn:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
        fs:SetPoint("LEFT", 0, 0)
        fs:SetJustifyH("LEFT")
        btn.label = fs

        local arrow = btn:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
        arrow:SetPoint("LEFT", fs, "RIGHT", 2, 0)
        btn.arrow = arrow

        local function UpdateLabel()
            local active = (playersSortCol == col.sortKey)
            if active then
                fs:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
                fs:SetText(col.t)
                arrow:SetText(playersSortAsc and " ^" or " v")
                arrow:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
            else
                fs:SetTextColor(0.5, 0.5, 0.5)
                fs:SetText(col.t)
                arrow:SetText("")
            end
        end
        btn.UpdateLabel = UpdateLabel
        UpdateLabel()

        btn:SetScript("OnClick", function()
            if playersSortCol == col.sortKey then
                playersSortAsc = not playersSortAsc
            else
                playersSortCol = col.sortKey
                playersSortAsc = (col.sortKey == "name" or col.sortKey == "spec")
            end
            for _, b in ipairs(colButtons) do b.UpdateLabel() end
            RefreshPlayersPanel()
        end)
        btn:SetScript("OnEnter", function()
            if playersSortCol ~= col.sortKey then
                fs:SetTextColor(1, 1, 1)
            end
        end)
        btn:SetScript("OnLeave", function()
            if playersSortCol ~= col.sortKey then
                fs:SetTextColor(0.5, 0.5, 0.5)
            end
        end)

        colButtons[#colButtons + 1] = btn
    end
    statsPanel.playersColHeaders = colHeaderContainer
    statsPanel.playersColButtons = colButtons

    -- Pre-create fixed row pool
    local playerRows = {}
    for i = 1, PLAYER_ROW_POOL do
        local row = CreateFrame("Frame", nil, playersContainer, "BackdropTemplate")
        row:SetHeight(PLAYER_ROW_HEIGHT)
        row:SetPoint("TOPLEFT", playersContainer, "TOPLEFT", 6, PLAYERS_DATA_TOP - ((i - 1) * PLAYER_ROW_HEIGHT))
        row:SetPoint("TOPRIGHT", playersContainer, "TOPRIGHT", -20, PLAYERS_DATA_TOP - ((i - 1) * PLAYER_ROW_HEIGHT))
        row:SetBackdrop({ bgFile = "Interface\\Tooltips\\UI-Tooltip-Background" })
        row:SetBackdropColor(0.06, 0.06, 0.09, 0.3)

        row.fillBar = row:CreateTexture(nil, "BACKGROUND", nil, 1)
        row.fillBar:SetPoint("TOPLEFT", 0, 0)
        row.fillBar:SetPoint("BOTTOMLEFT", 0, 0)
        row.fillBar:SetWidth(1)
        row.fillBar:SetColorTexture(0, 0.5, 0, 0.15)

        row.nameText = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
        row.nameText:SetPoint("LEFT", 8, 0)
        row.nameText:SetWidth(220)
        row.nameText:SetJustifyH("LEFT")

        row.specText = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
        row.specText:SetPoint("LEFT", 235, 0)
        row.specText:SetWidth(150)
        row.specText:SetJustifyH("LEFT")

        row.arenaRecord = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
        row.arenaRecord:SetPoint("LEFT", 390, 0)
        row.arenaRecord:SetWidth(120)
        row.arenaRecord:SetJustifyH("LEFT")

        row.shuffleText = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
        row.shuffleText:SetPoint("LEFT", 515, 0)
        row.shuffleText:SetWidth(80)
        row.shuffleText:SetJustifyH("LEFT")

        row.skirmText = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
        row.skirmText:SetPoint("LEFT", 595, 0)
        row.skirmText:SetWidth(70)
        row.skirmText:SetJustifyH("LEFT")

        row.lastSeen = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
        row.lastSeen:SetPoint("LEFT", 670, 0)
        row.lastSeen:SetWidth(100)
        row.lastSeen:SetJustifyH("LEFT")

        row:EnableMouse(true)
        row:SetScript("OnEnter", function(self)
            self:SetBackdropColor(CYAN.r * 0.12, CYAN.g * 0.12, CYAN.b * 0.12, 0.6)
        end)
        row:SetScript("OnLeave", function(self)
            self:SetBackdropColor(0.06, 0.06, 0.09, 0.3)
        end)
        row:Hide()

        playerRows[i] = row
    end
    statsPanel.playerRows = playerRows

    -- Scrollbar (integer offset-based)
    local playersSlider = CreateFrame("Slider", nil, playersContainer, "BackdropTemplate")
    playersSlider:SetPoint("TOPRIGHT", playersContainer, "TOPRIGHT", 0, PLAYERS_DATA_TOP)
    playersSlider:SetPoint("BOTTOMRIGHT", playersContainer, "BOTTOMRIGHT", 0, 0)
    playersSlider:SetWidth(14)
    playersSlider:SetOrientation("VERTICAL")
    playersSlider:SetBackdrop({
        bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
        edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
        tile = true, tileSize = 8, edgeSize = 10,
        insets = { left = 2, right = 2, top = 2, bottom = 2 },
    })
    playersSlider:SetBackdropColor(0.05, 0.05, 0.08, 0.7)
    playersSlider:SetBackdropBorderColor(0.25, 0.25, 0.3, 0.5)

    local playersThumb = playersSlider:CreateTexture(nil, "OVERLAY")
    playersThumb:SetColorTexture(CYAN.r, CYAN.g, CYAN.b, 0.5)
    playersThumb:SetSize(10, 40)
    playersSlider:SetThumbTexture(playersThumb)

    playersSlider:SetMinMaxValues(0, 1)
    playersSlider:SetValue(0)
    playersSlider:Hide()
    statsPanel.playersSlider = playersSlider

    -- Visible rows calculated from container height
    local function GetVisiblePlayerRows()
        local h = playersContainer:GetHeight()
        if h and h > 0 then
            return math.max(1, math.floor((h + PLAYERS_DATA_TOP) / PLAYER_ROW_HEIGHT))
        end
        return 22
    end

    -- UpdatePlayerRows: populate visible rows from offset into playerList
    local function UpdatePlayerRows(offset)
        offset = math.floor(offset + 0.5)
        local list = statsPanel.playerList or {}
        local myRealm = GetNormalizedRealmName() or ""
        local visibleCount = GetVisiblePlayerRows()

        local function FormatPlayerName(fullName)
            local pName, pRealm = fullName:match("^([^-]+)-(.+)$")
            if not pName then return fullName end
            if pRealm == myRealm then return pName end
            return fullName
        end

        for i = 1, PLAYER_ROW_POOL do
            local row = playerRows[i]
            if i <= visibleCount then
                local data = list[offset + i]
                if data then
                    row:Show()
                    local hex = GetClassColorHex(data.class)
                    row.nameText:SetText("|cff" .. hex .. FormatPlayerName(data.name) .. "|r")

                    local icon = ClassIconString(data.class, 16)
                    local specLabel = (data.spec ~= "") and data.spec or "?"
                    row.specText:SetText(icon .. " |cff" .. hex .. specLabel .. "|r")

                    -- Arena W/L
                    local arenaTotal = data.arenaWins + data.arenaLosses
                    if arenaTotal > 0 then
                        local wC = data.arenaWins > 0 and "|cff00ff00" or "|cffbbbbbb"
                        local lC = data.arenaLosses > 0 and "|cffff4444" or "|cffbbbbbb"
                        local pct = data.arenaWins / arenaTotal * 100
                        local pctC = pct > 50 and "|cff00ff00" or (pct < 50 and "|cffff4444" or "|cffbbbbbb")
                        row.arenaRecord:SetText(wC .. data.arenaWins .. "W|r - " .. lC .. data.arenaLosses .. "L|r  " .. pctC .. string.format("%.0f%%", pct) .. "|r")

                        local rowWidth = row:GetWidth()
                        if rowWidth and rowWidth > 0 then
                            row.fillBar:SetWidth(math.max(1, rowWidth * (pct / 100)))
                        end
                        row.fillBar:SetColorTexture(0, pct >= 50 and 0.5 or 0.1, 0, 0.15)
                    else
                        row.arenaRecord:SetText("|cff888888--|r")
                        row.fillBar:SetWidth(1)
                        row.fillBar:SetColorTexture(0, 0, 0, 0)
                    end

                    -- Shuffle
                    if data.shuffleCount > 0 then
                        row.shuffleText:SetText("|cffbbbbbb" .. data.shuffleCount .. (data.shuffleCount == 1 and " game" or " games") .. "|r")
                    else
                        row.shuffleText:SetText("|cff888888--|r")
                    end

                    -- Skirmish W/L
                    local skirmTotal = data.skirmWins + data.skirmLosses
                    if skirmTotal > 0 then
                        local wC = data.skirmWins > 0 and "|cff00ff00" or "|cffbbbbbb"
                        local lC = data.skirmLosses > 0 and "|cffff4444" or "|cffbbbbbb"
                        row.skirmText:SetText(wC .. data.skirmWins .. "W|r-" .. lC .. data.skirmLosses .. "L|r")
                    else
                        row.skirmText:SetText("|cff888888--|r")
                    end

                    -- Last seen
                    if data.lastSeen > 0 then
                        local fmt = (date("%Y", data.lastSeen) ~= date("%Y")) and "%m/%d/%y" or "%m/%d %H:%M"
                        row.lastSeen:SetText("|cff888888" .. date(fmt, data.lastSeen) .. "|r")
                    else
                        row.lastSeen:SetText("")
                    end
                else
                    row:Hide()
                end
            else
                row:Hide()
            end
        end
    end
    statsPanel.UpdatePlayerRows = UpdatePlayerRows

    playersSlider:SetScript("OnValueChanged", function(self, value)
        UpdatePlayerRows(value)
    end)

    playersContainer:EnableMouseWheel(true)
    playersContainer:SetScript("OnMouseWheel", function(self, delta)
        local cur = playersSlider:GetValue()
        local _, maxVal = playersSlider:GetMinMaxValues()
        local newVal = math.max(0, math.min(maxVal, cur - delta * 3))
        playersSlider:SetValue(newVal)
    end)

    --------------------------------
    -- Graph Panel
    --------------------------------
    local GRAPH_PAD_LEFT = 55
    local GRAPH_PAD_RIGHT = 15
    local GRAPH_PAD_TOP = 62
    local GRAPH_PAD_BOTTOM = 30

    local graphPanel = CreateFrame("Frame", nil, f, "BackdropTemplate")
    graphPanel:SetPoint("TOPLEFT", f, "TOPLEFT", 4, contentY)
    graphPanel:SetPoint("BOTTOMRIGHT", f, "BOTTOMRIGHT", -4, 6)
    graphPanel:SetBackdrop({
        bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
        edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
        edgeSize = 14,
        insets = { left = 3, right = 3, top = 3, bottom = 3 },
    })
    graphPanel:SetBackdropColor(0.08, 0.08, 0.1, 1)
    graphPanel:SetBackdropBorderColor(0.3, 0.3, 0.35, 0.8)
    graphPanel:Hide()
    f.graphPanel = graphPanel

    graphPanel.title = graphPanel:CreateFontString(nil, "OVERLAY", "GameFontNormal")
    graphPanel.title:SetPoint("TOPLEFT", 10, -8)
    graphPanel.title:SetText("|cff00E5EERating Over Time|r")

    graphPanel.filterLabel = graphPanel:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
    graphPanel.filterLabel:SetPoint("TOPRIGHT", graphPanel, "TOPRIGHT", -12, -8)
    graphPanel.filterLabel:SetTextColor(CYAN.r, CYAN.g, CYAN.b, 0.7)

    graphPanel.noData = graphPanel:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
    graphPanel.noData:SetPoint("CENTER")
    graphPanel.noData:SetText("|cff888888No rated matches for this filter.|r")
    graphPanel.noData:Hide()

    -- Graph dropdown helper
    local function CreateGraphDropdown(parent, width, anchor, anchorTo, anchorPt, ox, oy)
        local btn = CreateFrame("Button", nil, parent, "BackdropTemplate")
        btn:SetSize(width, 20)
        btn:SetBackdrop({
            bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
            edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
            tile = true, tileSize = 8, edgeSize = 10,
            insets = { left = 2, right = 2, top = 2, bottom = 2 },
        })
        btn:SetBackdropColor(0.1, 0.1, 0.12, 0.8)
        btn:SetBackdropBorderColor(0.3, 0.3, 0.35, 0.6)
        btn:SetPoint(anchor, anchorTo, anchorPt, ox, oy)

        local text = btn:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
        text:SetPoint("LEFT", 6, 0)
        text:SetPoint("RIGHT", -14, 0)
        text:SetJustifyH("LEFT")
        btn.text = text

        local arrow = btn:CreateTexture(nil, "OVERLAY")
        arrow:SetSize(8, 8)
        arrow:SetPoint("RIGHT", -3, 0)
        arrow:SetAtlas("arrow-down-minor")
        arrow:SetVertexColor(0.6, 0.6, 0.6)

        local menu = CreateFrame("Frame", nil, btn, "BackdropTemplate")
        menu:SetBackdrop({
            bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
            edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
            tile = true, tileSize = 8, edgeSize = 10,
            insets = { left = 2, right = 2, top = 2, bottom = 2 },
        })
        menu:SetBackdropColor(0.08, 0.08, 0.1, 0.95)
        menu:SetBackdropBorderColor(0.3, 0.3, 0.35, 0.8)
        menu:SetFrameStrata("DIALOG")
        menu:Hide()
        btn.menu = menu

        local menuItems = {}
        btn.menuItems = menuItems

        btn:SetScript("OnEnter", function(self)
            self:SetBackdropColor(0.15, 0.15, 0.18, 0.9)
            self.text:SetTextColor(1, 1, 1)
        end)
        btn:SetScript("OnLeave", function(self)
            if not self.isActive then
                self:SetBackdropColor(0.1, 0.1, 0.12, 0.8)
                self.text:SetTextColor(0.7, 0.7, 0.7)
            end
        end)
        btn:SetScript("OnClick", function()
            if menu:IsShown() then menu:Hide() else
                if btn.PopulateMenu then btn:PopulateMenu() end
                menu:Show()
            end
        end)

        menu:SetScript("OnShow", function()
            menu:SetScript("OnUpdate", function(self)
                local ok, err = pcall(function()
                    if not btn:IsMouseOver() and not self:IsMouseOver() then
                        local elapsed = (self.hideTimer or 0) + 0.016
                        self.hideTimer = elapsed
                        if elapsed > 0.3 then self:Hide(); self.hideTimer = 0 end
                    else
                        self.hideTimer = 0
                    end
                end)
                if not ok then self:Hide() end
            end)
        end)
        menu:SetScript("OnHide", function(self) self:SetScript("OnUpdate", nil); self.hideTimer = 0 end)

        function btn:SetActiveState(active)
            self.isActive = active
            if active then
                self:SetBackdropColor(CYAN.r * 0.3, CYAN.g * 0.3, CYAN.b * 0.3, 1)
                self:SetBackdropBorderColor(CYAN.r, CYAN.g, CYAN.b, 0.8)
                self.text:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
            else
                self:SetBackdropColor(0.1, 0.1, 0.12, 0.8)
                self:SetBackdropBorderColor(0.3, 0.3, 0.35, 0.6)
                self.text:SetTextColor(0.7, 0.7, 0.7)
            end
        end

        function btn:BuildMenu(options, currentValue, onSelect)
            for _, item in ipairs(menuItems) do item:Hide() end
            wipe(menuItems)
            local itemH = 18
            menu:SetSize(btn:GetWidth(), 4 + #options * itemH + 4)
            menu:SetPoint("TOP", btn, "BOTTOM", 0, -2)
            for i, opt in ipairs(options) do
                local item = CreateFrame("Button", nil, menu)
                item:SetSize(btn:GetWidth() - 8, itemH)
                item:SetPoint("TOPLEFT", menu, "TOPLEFT", 4, -4 - (i - 1) * itemH)
                local itemText = item:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
                itemText:SetPoint("LEFT", 4, 0)
                itemText:SetText(opt.label)
                if opt.value == currentValue then
                    itemText:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
                else
                    itemText:SetTextColor(0.8, 0.8, 0.8)
                end
                item:SetScript("OnEnter", function() if opt.value ~= currentValue then itemText:SetTextColor(1, 1, 1) end end)
                item:SetScript("OnLeave", function() if opt.value ~= currentValue then itemText:SetTextColor(0.8, 0.8, 0.8) end end)
                item:SetScript("OnClick", function()
                    menu:Hide()
                    onSelect(opt.value, opt.label)
                end)
                menuItems[#menuItems + 1] = item
            end
        end

        return btn
    end

    -- Graph filter dropdowns (below legend row)
    local graphDropdownY = -40

    -- Spec filter dropdown
    local specDD = CreateGraphDropdown(graphPanel, 120, "TOPLEFT", graphPanel, "TOPLEFT", 10, graphDropdownY)
    specDD.text:SetText("All Specs")
    specDD.text:SetTextColor(0.7, 0.7, 0.7)
    graphPanel.specDD = specDD

    function specDD:PopulateMenu()
        -- Collect unique specs the player has played from match data
        local specs = {}
        local seen = {}
        local matches = SKToolsArenaDB and SKToolsArenaDB.matches or {}
        for _, m in ipairs(matches) do
            if m.isRated and m.playerTeam then
                local shortName = m.player and m.player:match("^([^%-]+)")
                if shortName then
                    for _, p in ipairs(m.playerTeam) do
                        if p.spec and p.spec ~= "" and p.name and p.name:find(shortName, 1, true) then
                            local resolved = ResolveSpec(p.spec)
                            if resolved ~= "Unknown" and not seen[resolved] then
                                seen[resolved] = true
                                specs[#specs + 1] = resolved
                            end
                            break
                        end
                    end
                end
            end
        end
        table.sort(specs)
        local opts = { { label = "All Specs", value = nil } }
        for _, s in ipairs(specs) do
            opts[#opts + 1] = { label = s, value = s }
        end
        self:BuildMenu(opts, graphSpecFilter, function(val, label)
            graphSpecFilter = val
            self.text:SetText(label)
            self:SetActiveState(val ~= nil)
            RefreshGraphPanel()
        end)
    end

    -- Date filter dropdown
    local dateDD = CreateGraphDropdown(graphPanel, 110, "LEFT", specDD, "RIGHT", 4, 0)
    dateDD.text:SetText("All Time")
    dateDD.text:SetTextColor(0.7, 0.7, 0.7)
    graphPanel.dateDD = dateDD

    -- Custom date range edit boxes (initially hidden)
    local customDateFrame = CreateFrame("Frame", nil, graphPanel, "BackdropTemplate")
    customDateFrame:SetSize(240, 24)
    customDateFrame:SetPoint("LEFT", dateDD, "RIGHT", 6, 0)
    customDateFrame:Hide()
    graphPanel.customDateFrame = customDateFrame

    local fromLabel = customDateFrame:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
    fromLabel:SetPoint("LEFT", 0, 0)
    fromLabel:SetText("|cff888888From:|r")

    local fromBox = CreateFrame("EditBox", nil, customDateFrame, "BackdropTemplate")
    fromBox:SetSize(68, 18)
    fromBox:SetPoint("LEFT", fromLabel, "RIGHT", 4, 0)
    fromBox:SetBackdrop({
        bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
        edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
        tile = true, tileSize = 8, edgeSize = 10,
        insets = { left = 2, right = 2, top = 2, bottom = 2 },
    })
    fromBox:SetBackdropColor(0.05, 0.05, 0.07, 0.9)
    fromBox:SetBackdropBorderColor(0.3, 0.3, 0.35, 0.6)
    fromBox:SetFontObject("GameFontHighlightSmall")
    fromBox:SetTextInsets(4, 4, 0, 0)
    fromBox:SetAutoFocus(false)
    fromBox:SetMaxLetters(10)
    graphPanel.fromBox = fromBox

    local toLabel = customDateFrame:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
    toLabel:SetPoint("LEFT", fromBox, "RIGHT", 6, 0)
    toLabel:SetText("|cff888888To:|r")

    local toBox = CreateFrame("EditBox", nil, customDateFrame, "BackdropTemplate")
    toBox:SetSize(68, 18)
    toBox:SetPoint("LEFT", toLabel, "RIGHT", 4, 0)
    toBox:SetBackdrop({
        bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
        edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
        tile = true, tileSize = 8, edgeSize = 10,
        insets = { left = 2, right = 2, top = 2, bottom = 2 },
    })
    toBox:SetBackdropColor(0.05, 0.05, 0.07, 0.9)
    toBox:SetBackdropBorderColor(0.3, 0.3, 0.35, 0.6)
    toBox:SetFontObject("GameFontHighlightSmall")
    toBox:SetTextInsets(4, 4, 0, 0)
    toBox:SetAutoFocus(false)
    toBox:SetMaxLetters(10)
    graphPanel.toBox = toBox

    -- Parse date string like "1/15", "01/15/26", "1/15/2026"
    local function ParseDateStr(str)
        if not str or str == "" then return nil end
        local m, d, y = str:match("^(%d+)/(%d+)/(%d+)$")
        if not m then
            m, d = str:match("^(%d+)/(%d+)$")
        end
        if not m or not d then return nil end
        m, d = tonumber(m), tonumber(d)
        if not m or not d or m < 1 or m > 12 or d < 1 or d > 31 then return nil end
        if y then
            y = tonumber(y)
            if y < 100 then y = y + 2000 end
        else
            y = tonumber(date("%Y"))
        end
        return time({ year = y, month = m, day = d, hour = 0, min = 0, sec = 0 })
    end

    local function ApplyCustomDateRange()
        local startTs = ParseDateStr(fromBox:GetText())
        local endTs = ParseDateStr(toBox:GetText())
        if endTs then endTs = endTs + 86400 - 1 end  -- end of day
        graphDateCustomStart = startTs
        graphDateCustomEnd = endTs
        if startTs or endTs then
            graphDateFilter = "custom"
            local label = (fromBox:GetText() ~= "" and fromBox:GetText() or "...") .. " - " .. (toBox:GetText() ~= "" and toBox:GetText() or "...")
            dateDD.text:SetText(label)
            dateDD:SetActiveState(true)
        end
        RefreshGraphPanel()
    end

    fromBox:SetScript("OnEnterPressed", function(self) self:ClearFocus(); ApplyCustomDateRange() end)
    fromBox:SetScript("OnEscapePressed", function(self) self:ClearFocus() end)
    toBox:SetScript("OnEnterPressed", function(self) self:ClearFocus(); ApplyCustomDateRange() end)
    toBox:SetScript("OnEscapePressed", function(self) self:ClearFocus() end)

    function dateDD:PopulateMenu()
        local opts = {
            { label = "All Time", value = nil },
            { label = "Today", value = "today" },
            { label = "This Week", value = "week" },
            { label = "This Month", value = "month" },
            { label = "Last 3 Months", value = "3months" },
            { label = "Custom...", value = "custom" },
        }
        self:BuildMenu(opts, graphDateFilter, function(val, label)
            if val == "custom" then
                graphDateFilter = "custom"
                customDateFrame:Show()
                fromBox:SetText("")
                toBox:SetText("")
                graphDateCustomStart = nil
                graphDateCustomEnd = nil
                self.text:SetText("Custom...")
                self:SetActiveState(true)
                fromBox:SetFocus()
            else
                graphDateFilter = val
                graphDateCustomStart = nil
                graphDateCustomEnd = nil
                customDateFrame:Hide()
                self.text:SetText(label)
                self:SetActiveState(val ~= nil)
                RefreshGraphPanel()
            end
        end)
    end

    -- Plot area (the actual drawing region within padding)
    local plotArea = CreateFrame("Frame", nil, graphPanel)
    plotArea:SetPoint("TOPLEFT", GRAPH_PAD_LEFT, -GRAPH_PAD_TOP)
    plotArea:SetPoint("BOTTOMRIGHT", -GRAPH_PAD_RIGHT, GRAPH_PAD_BOTTOM)
    graphPanel.plotArea = plotArea

    -- Texture/element pools (grow on demand)
    graphPanel.gridLines = {}
    graphPanel.yLabels = {}
    graphPanel.xLabels = {}
    graphPanel.lineSegments = {}
    graphPanel.dataPoints = {}
    graphPanel.legendEntries = {}

    -- Pool accessors
    graphPanel.GetGridLine = function(index)
        if not graphPanel.gridLines[index] then
            local tex = plotArea:CreateTexture(nil, "BACKGROUND")
            tex:SetColorTexture(1, 1, 1, 0.08)
            graphPanel.gridLines[index] = tex
        end
        return graphPanel.gridLines[index]
    end

    graphPanel.GetYLabel = function(index)
        if not graphPanel.yLabels[index] then
            local fs = graphPanel:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
            fs:SetJustifyH("RIGHT")
            graphPanel.yLabels[index] = fs
        end
        return graphPanel.yLabels[index]
    end

    graphPanel.GetXLabel = function(index)
        if not graphPanel.xLabels[index] then
            local fs = graphPanel:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
            fs:SetJustifyH("CENTER")
            graphPanel.xLabels[index] = fs
        end
        return graphPanel.xLabels[index]
    end

    graphPanel.GetLineSegment = function(index)
        if not graphPanel.lineSegments[index] then
            local tex = plotArea:CreateTexture(nil, "ARTWORK")
            tex:SetColorTexture(1, 1, 1, 1)
            graphPanel.lineSegments[index] = tex
        end
        return graphPanel.lineSegments[index]
    end

    graphPanel.GetDataPoint = function(index)
        if not graphPanel.dataPoints[index] then
            local dot = plotArea:CreateTexture(nil, "OVERLAY")
            dot:SetColorTexture(1, 1, 1, 1)
            dot:SetSize(6, 6)

            local btn = CreateFrame("Button", nil, plotArea)
            btn:SetSize(14, 14)
            btn.dot = dot
            btn:SetScript("OnEnter", function(self)
                if self.matchData then
                    GameTooltip:SetOwner(self, "ANCHOR_RIGHT")
                    local m = self.matchData
                    local modeLabel = MODE_LABELS[m.mode] or m.mode
                    GameTooltip:AddLine("|cff00E5EE" .. modeLabel .. "|r")
                    local resultStr = m.won and "|cff00ff00Win|r" or "|cffff4444Loss|r"
                    GameTooltip:AddLine(resultStr)
                    if m.rating then
                        local sign = (m.ratingChange or 0) >= 0 and "+" or ""
                        local color = (m.ratingChange or 0) >= 0 and "|cff00ff00" or "|cffff4444"
                        GameTooltip:AddLine("Rating: " .. m.rating .. " (" .. color .. sign .. (m.ratingChange or 0) .. "|r)")
                    end
                    if m.mmr then
                        local mmrText = "MMR: " .. m.mmr
                        if m.enemyMmr then mmrText = mmrText .. "  vs  " .. m.enemyMmr end
                        GameTooltip:AddLine(mmrText, 0.7, 0.7, 0.7)
                    end
                    GameTooltip:AddLine((m.date or "") .. " " .. (m.timeStr or ""), 0.5, 0.5, 0.5)
                    if m.mapName then
                        GameTooltip:AddLine(m.mapName, 0.5, 0.5, 0.5)
                    end
                    GameTooltip:Show()
                end
            end)
            btn:SetScript("OnLeave", function() GameTooltip:Hide() end)
            graphPanel.dataPoints[index] = { dot = dot, btn = btn }
        end
        return graphPanel.dataPoints[index]
    end

    graphPanel.GetLegendEntry = function(index)
        if not graphPanel.legendEntries[index] then
            local btn = CreateFrame("Button", nil, graphPanel)
            btn:SetSize(60, 14)
            local swatch = btn:CreateTexture(nil, "OVERLAY")
            swatch:SetSize(10, 10)
            swatch:SetPoint("LEFT", 0, 0)
            local label = btn:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
            label:SetPoint("LEFT", swatch, "RIGHT", 4, 0)
            btn:SetScript("OnClick", function(self)
                if self.modeKey then
                    graphHiddenModes[self.modeKey] = not graphHiddenModes[self.modeKey]
                    RefreshGraphPanel()
                end
            end)
            btn:SetScript("OnEnter", function(self)
                self.label:SetTextColor(1, 1, 1)
            end)
            btn:SetScript("OnLeave", function(self)
                if self.modeKey and graphHiddenModes[self.modeKey] then
                    self.label:SetTextColor(0.4, 0.4, 0.4)
                end
            end)
            graphPanel.legendEntries[index] = { btn = btn, swatch = swatch, label = label }
        end
        return graphPanel.legendEntries[index]
    end

    -- DrawLine helper: simulates a line with a rotated texture
    graphPanel.DrawLine = function(tex, x1, y1, x2, y2, thickness, r, g, b, a)
        local dx = x2 - x1
        local dy = y2 - y1
        local length = math.sqrt(dx * dx + dy * dy)
        if length < 0.1 then
            tex:Hide()
            return
        end
        local angle = math.atan2(dy, dx)
        local cx = (x1 + x2) / 2
        local cy = (y1 + y2) / 2
        tex:ClearAllPoints()
        tex:SetSize(length, thickness)
        tex:SetPoint("CENTER", plotArea, "BOTTOMLEFT", cx, cy)
        tex:SetRotation(angle)
        tex:SetColorTexture(r, g, b, a)
        tex:Show()
    end

    historyFrame = f
end

-----------------------------
-- View Toggling (List / Stats / Graph)
-----------------------------
UpdateFilterLabel = function()
    if not historyFrame then return end
    local parts = {}
    if activeFilter ~= "All" then parts[#parts + 1] = activeFilter end
    if activeCharFilter then
        local shortName = activeCharFilter:match("^([^%-]+)") or activeCharFilter
        parts[#parts + 1] = shortName
    end
    local text = #parts > 0 and ("Showing: " .. table.concat(parts, " | ")) or ""
    if historyFrame.statsPanel and historyFrame.statsPanel.filterLabel then
        historyFrame.statsPanel.filterLabel:SetText(text)
    end
    if historyFrame.graphPanel and historyFrame.graphPanel.filterLabel then
        historyFrame.graphPanel.filterLabel:SetText(text)
    end
end

ShowView = function(viewKey)
    if not historyFrame then return end
    activeView = viewKey

    -- Hide everything
    historyFrame.headerFrame:Hide()
    historyFrame.scrollArea:Hide()
    historyFrame.slider:Hide()
    historyFrame.statsPanel:Hide()
    if historyFrame.graphPanel then historyFrame.graphPanel:Hide() end

    -- Show the right panels
    local isStats = (viewKey == "winrates" or viewKey == "analysis" or viewKey == "players")
    if viewKey == "matches" then
        historyFrame.headerFrame:Show()
        historyFrame.scrollArea:Show()
        historyFrame.slider:Show()
    elseif isStats then
        historyFrame.statsPanel:Show()
        -- Show the right container, hide others
        local containers = historyFrame.statsPanel.allStatsContainers
        for k, c in pairs(containers) do
            if k == viewKey then c:Show() else c:Hide() end
        end
        RefreshStatsPanel()
    elseif viewKey == "graph" then
        historyFrame.graphPanel:Show()
        RefreshGraphPanel()
    end

    -- Update view button highlights
    for _, btn in ipairs(viewButtons) do
        if btn.viewKey == viewKey then
            btn:SetBackdropColor(CYAN.r * 0.3, CYAN.g * 0.3, CYAN.b * 0.3, 1)
            btn:SetBackdropBorderColor(CYAN.r, CYAN.g, CYAN.b, 0.8)
            btn.text:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
        else
            btn:SetBackdropColor(0.1, 0.1, 0.12, 0.8)
            btn:SetBackdropBorderColor(0.3, 0.3, 0.35, 0.6)
            btn.text:SetTextColor(0.7, 0.7, 0.7)
        end
    end

    UpdateFilterLabel()
    if viewKey == "matches" then SKPvPHistory_RefreshUI() end
end

-----------------------------
-- Update Rows
-----------------------------
function SKPvPHistory_UpdateRows(offset)
    local filtered = GetFilteredMatches()
    offset = offset or 0

    for i = 1, VISIBLE_ROWS do
        local row = rows[i]
        local dataIdx = offset + i
        local match = filtered[dataIdx]

        if match then
            row:Show()

            -- Result
            if match.mode == "shuffle" and match.roundWins then
                local rl = match.roundLosses or (6 - match.roundWins)
                local color = (match.roundWins > rl) and "|cff00ff00" or (match.roundWins == rl) and "|cffbbbbbb" or "|cffff4444"
                row.result:SetText(color .. match.roundWins .. "-" .. rl .. "|r")
            elseif match.won then
                row.result:SetText("|cff00ff00W|r")
            else
                row.result:SetText("|cffff4444L|r")
            end

            -- Mode
            row.mode:SetText(MODE_LABELS[match.mode] or match.mode)

            -- Map
            row.mapName:SetText(match.mapName or "")

            -- Rating
            if match.isRated and match.ratingChange then
                local sign = match.ratingChange >= 0 and "+" or ""
                local color = match.ratingChange >= 0 and "|cff00ff00" or "|cffff4444"
                row.rating:SetText((match.rating or "") .. " " .. color .. sign .. match.ratingChange .. "|r")
            else
                row.rating:SetText("|cff888888—|r")
            end

            -- MMR (yours / enemy)
            if match.isRated and match.mmr then
                local mmrStr = "|cffbbbbbb" .. match.mmr .. "|r"
                if match.enemyMmr then
                    mmrStr = mmrStr .. "|cff666666/|r|cffbbbbbb" .. match.enemyMmr .. "|r"
                end
                row.mmr:SetText(mmrStr)
            else
                row.mmr:SetText("|cff888888—|r")
            end

            -- K/D (HKs)
            local kdText = (match.kills or 0) .. "/" .. (match.deaths or 0)
            if (match.honorableKills or 0) > 0 then
                kdText = kdText .. " |cff888888(" .. match.honorableKills .. ")|r"
            end
            row.kd:SetText(kdText)

            -- Damage
            row.dmg:SetText(FormatNumber(match.damage))

            -- Healing
            row.healing:SetText(FormatNumber(match.healing))

            -- Comp (adjust icon size for large teams)
            if match.playerTeam and match.enemyTeam then
                local totalPlayers = #match.playerTeam + #match.enemyTeam
                local isBG = (match.mode == "randomBG" or match.mode == "epicBG" or match.mode == "rbg")
                local isShuffle = (match.mode == "shuffle")
                if isBG then
                    -- BGs: too many players for icons, skip
                    row.comp:SetText("")
                elseif totalPlayers > 6 and not isShuffle then
                    -- Blitz: two rows with colored team markers
                    local pt = GetCompLabel(match.playerTeam, 14)
                    local et = GetCompLabel(match.enemyTeam, 14)
                    row.comp:SetText("|cff00cc00>|r " .. pt .. "\n" .. "|cffcc0000>|r " .. et)
                elseif isShuffle then
                    -- Shuffle: show all 6 specs in lobby as one group
                    local allPlayers = {}
                    for _, p in ipairs(match.playerTeam) do allPlayers[#allPlayers + 1] = p end
                    for _, p in ipairs(match.enemyTeam) do allPlayers[#allPlayers + 1] = p end
                    row.comp:SetText(GetCompLabel(allPlayers))
                else
                    -- Arena (2v2/3v3): normal display
                    local pt = GetCompLabel(match.playerTeam)
                    local et = GetCompLabel(match.enemyTeam)
                    if pt ~= "" and et ~= "" then
                        row.comp:SetText(pt .. " |cff888888v|r " .. et)
                    else
                        row.comp:SetText("")
                    end
                end
            else
                row.comp:SetText("")
            end

            -- Death log tooltip data (scoreboard + friendly timestamps)
            if match.deathLog and #match.deathLog > 0 then
                local ok, text = pcall(function()
                    local lines = {}
                    for _, d in ipairs(match.deathLog) do
                        local icon = ClassIconString(d.class or "UNKNOWN", 14)
                        local hex = GetClassColorHex(d.class or "UNKNOWN")
                        local spec = ResolveDeathSpec(d, match) or "?"
                        local teamTag = d.team == "player" and "|cff00ff00Ally|r" or "|cffff4444Enemy|r"
                        local deathCount = (d.deaths and d.deaths > 1) and (" x" .. d.deaths) or ""
                        local timeStr = ""
                        if d.time then
                            local m = math.floor(d.time / 60)
                            local s = d.time % 60
                            timeStr = "  |cff888888@ " .. string.format("%d:%04.1f", m, s) .. "|r"
                        end
                        lines[#lines + 1] = icon .. " |cff" .. hex .. spec .. "|r " .. teamTag .. deathCount .. timeStr
                    end
                    return table.concat(lines, "\n")
                end)
                row.deathLogText = ok and text or nil
            else
                row.deathLogText = nil
            end

            -- Combat stats for tooltip
            row.tooltipKills = match.kills or 0
            row.tooltipDeaths = match.deaths or 0
            row.tooltipHKs = match.honorableKills or 0
            row.tooltipDamage = match.damage or 0
            row.tooltipHealing = match.healing or 0

            -- Rating info for tooltip
            row.tooltipMatch = match

            -- Duration
            row.duration:SetText(FormatDuration(match.duration))

            -- Date (show year if not current year)
            local dateStr = match.date or ""
            local timeStr = match.timeStr or ""
            if match.timestamp then
                local matchYear = date("%Y", match.timestamp)
                local currentYear = date("%Y")
                if matchYear ~= currentYear then
                    dateStr = date("%m/%d/%y", match.timestamp)
                end
            end
            row.time:SetText("|cffDDDDDD" .. dateStr .. "|r  |cff888888" .. timeStr .. "|r")
        else
            row:Hide()
            row.deathLogText = nil
            row.tooltipMatch = nil
        end
    end
end

-----------------------------
-- Refresh Stats Panel
-----------------------------
RefreshStatsPanel = function()
    if not historyFrame or not historyFrame.statsPanel then return end
    local panel = historyFrame.statsPanel

    -- Refresh the active view
    if activeView == "analysis" then
        if RefreshAnalysisPanel then RefreshAnalysisPanel() end
        return
    elseif activeView == "players" then
        if RefreshPlayersPanel then RefreshPlayersPanel() end
        return
    end

    local filtered = GetFilteredMatches()
    local rowH = panel.STAT_ROW_HEIGHT
    local mapSec = panel.map

    -- Aggregate comp stats (cached — only recompute when data or filters change)
    local compList
    if compCacheDirty or compCachedFilter ~= activeFilter or compCachedCharFilter ~= activeCharFilter then
        local skipCompModes = { shuffle=true, rbg=true, blitz=true, randomBG=true, epicBG=true }
        local compStats = {}
        local compTeams = {}
        for _, m in ipairs(filtered) do
            if m.enemyTeam and #m.enemyTeam > 0 and not skipCompModes[m.mode] then
                local key = GetSpecCompKey(m.enemyTeam)
                if key then
                    if not compStats[key] then
                        compStats[key] = { wins = 0, losses = 0 }
                        compTeams[key] = m.enemyTeam
                    end
                    if m.won then
                        compStats[key].wins = compStats[key].wins + 1
                    else
                        compStats[key].losses = compStats[key].losses + 1
                    end
                end
            end
        end
        local list = {}
        for key, stats in pairs(compStats) do
            list[#list + 1] = {
                key = key,
                team = compTeams[key],
                wins = stats.wins,
                losses = stats.losses,
                total = stats.wins + stats.losses,
            }
        end
        compCachedList = list
        compCachedFilter = activeFilter
        compCachedCharFilter = activeCharFilter
        compCacheDirty = false
    end
    compList = {}
    for i, v in ipairs(compCachedList) do compList[i] = v end

    -- Rebuild spec pills from raw data, reposition layout
    local pillH = panel.RebuildSpecPills(compList)
    panel.RepositionCompLayout(pillH)

    -- Apply spec pill filter (AND logic: comp must contain ALL selected specs)
    if next(compSpecFilters) then
        local filteredComps = {}
        for _, entry in ipairs(compList) do
            local compSpecs = {}
            for _, p in ipairs(entry.team) do
                compSpecs[ResolveSpec(p.spec) .. " " .. (p.class or "UNKNOWN")] = true
            end
            local allMatch = true
            for specKey in pairs(compSpecFilters) do
                if not compSpecs[specKey] then allMatch = false; break end
            end
            if allMatch then filteredComps[#filteredComps + 1] = entry end
        end
        compList = filteredComps
    end

    local wrCol = winratesSortCol
    local wrAsc = winratesSortAsc
    table.sort(compList, function(a, b)
        local va, vb
        if wrCol == "comp" then
            va, vb = a.key:lower(), b.key:lower()
        elseif wrCol == "record" then
            va, vb = a.wins, b.wins
        elseif wrCol == "winpct" then
            va = a.total > 0 and (a.wins / a.total) or -1
            vb = b.total > 0 and (b.wins / b.total) or -1
        else -- "total"
            va, vb = a.total, b.total
        end
        if va ~= vb then
            if wrAsc then return va < vb else return va > vb end
        end
        return a.total > b.total
    end)

    -- Store comp list and update viewport
    panel.compList = compList
    panel.compNoData:SetShown(#compList == 0)
    local visibleRows = panel.visibleCompRows or 19
    local maxOffset = math.max(0, #compList - visibleRows)
    panel.compSlider:SetMinMaxValues(0, maxOffset)
    panel.compSlider:SetValue(0)
    panel.compSlider:SetShown(maxOffset > 0)
    panel.UpdateCompRows(0)

    -- Aggregate map stats
    local mapStats = {}
    for _, m in ipairs(filtered) do
        local map = m.mapName or "Unknown"
        if not mapStats[map] then
            mapStats[map] = { wins = 0, losses = 0 }
        end
        if m.won then
            mapStats[map].wins = mapStats[map].wins + 1
        else
            mapStats[map].losses = mapStats[map].losses + 1
        end
    end

    local mapList = {}
    for name, stats in pairs(mapStats) do
        mapList[#mapList + 1] = {
            name = name,
            wins = stats.wins,
            losses = stats.losses,
            total = stats.wins + stats.losses,
        }
    end
    table.sort(mapList, function(a, b) return a.total > b.total end)

    -- Ensure enough map rows exist
    panel.EnsureStatRows(mapSec, #mapList, "map")

    -- Populate map rows
    mapSec.noData:SetShown(#mapList == 0)
    for i = 1, #mapSec.rows do
        local row = mapSec.rows[i]
        local data = mapList[i]
        if data then
            row:Show()
            row.mapName:SetText(data.name)
            local wColor = data.wins > 0 and "|cff00ff00" or "|cffbbbbbb"
            local lColor = data.losses > 0 and "|cffff4444" or "|cffbbbbbb"
            row.record:SetText(wColor .. data.wins .. "W|r - " .. lColor .. data.losses .. "L|r")
            local pct = data.total > 0 and (data.wins / data.total * 100) or 0
            local pctColor = pct > 50 and "|cff00ff00" or (pct < 50 and "|cffff4444" or "|cffbbbbbb")
            row.winPct:SetText(pctColor .. string.format("%.0f%%", pct) .. "|r")
            local rowWidth = row:GetWidth()
            if rowWidth and rowWidth > 0 then
                row.fillBar:SetWidth(math.max(1, rowWidth * (pct / 100)))
            end
            row.fillBar:SetColorTexture(0, pct > 50 and 0.5 or 0.1, 0, 0.15)
        else
            row:Hide()
        end
    end
    mapSec.content:SetHeight(math.max(1, #mapList * rowH))
    mapSec.scroll:SetVerticalScroll(0)
end

-----------------------------
-- Refresh Analysis Panel
-----------------------------
RefreshAnalysisPanel = function()
    if not historyFrame or not historyFrame.statsPanel then return end
    local panel = historyFrame.statsPanel
    -- (compFilterBtn is always visible, no separate state to update)

    local filtered = GetFilteredMatches()

    -- Aggregate kill targets and deaths-in-losses, keyed by playerComp|enemyComp
    local killTargets = {}   -- combinedKey -> { specKey -> { count, totalTime, timeCount } }
    local killPlayer = {}    -- combinedKey -> playerTeam
    local killEnemy = {}     -- combinedKey -> enemyTeam
    local killTotals = {}    -- combinedKey -> total wins
    local lossDeath = {}     -- combinedKey -> { specKey -> { count, totalTime, timeCount } }
    local lossPlayer = {}    -- combinedKey -> playerTeam
    local lossEnemy = {}     -- combinedKey -> enemyTeam
    local lossTotals = {}    -- combinedKey -> total losses
    local hasData = false

    for _, m in ipairs(filtered) do
        local hasTeams = m.deathLog and #m.deathLog > 0 and m.enemyTeam and #m.enemyTeam > 0 and m.playerTeam and #m.playerTeam > 0
        if not hasTeams then -- skip
        elseif m.won then
            -- Kill targets — first enemy death is the kill target
            local pKey = GetSpecCompKey(m.playerTeam)
            local eKey = GetSpecCompKey(m.enemyTeam)
            if pKey and eKey then
                local cKey = pKey .. "|" .. eKey
                for _, d in ipairs(m.deathLog) do
                    if d.team == "enemy" and d.class and d.class ~= "UNKNOWN" then
                        local resolvedSpec = ResolveDeathSpec(d, m)
                        if resolvedSpec then
                            hasData = true
                            local specKey = resolvedSpec .. " " .. d.class
                            if not killTargets[cKey] then
                                killTargets[cKey] = {}
                                killPlayer[cKey] = m.playerTeam
                                killEnemy[cKey] = m.enemyTeam
                                killTotals[cKey] = 0
                            end
                            killTotals[cKey] = killTotals[cKey] + 1
                            if not killTargets[cKey][specKey] then
                                killTargets[cKey][specKey] = { count = 0, totalTime = 0, timeCount = 0 }
                            end
                            local entry = killTargets[cKey][specKey]
                            entry.count = entry.count + 1
                            if d.time then
                                entry.totalTime = entry.totalTime + d.time
                                entry.timeCount = entry.timeCount + 1
                            end
                        end
                        break  -- only first enemy death
                    end
                end
            end
        else
            -- Deaths in losses — first player-team death is who was targeted
            local pKey = GetSpecCompKey(m.playerTeam)
            local eKey = GetSpecCompKey(m.enemyTeam)
            if pKey and eKey then
                local cKey = pKey .. "|" .. eKey
                for _, d in ipairs(m.deathLog) do
                    if d.team == "player" and d.class and d.class ~= "UNKNOWN" then
                        local resolvedSpec = ResolveDeathSpec(d, m)
                        if resolvedSpec then
                            hasData = true
                            local specKey = resolvedSpec .. " " .. d.class
                            if not lossDeath[cKey] then
                                lossDeath[cKey] = {}
                                lossPlayer[cKey] = m.playerTeam
                                lossEnemy[cKey] = m.enemyTeam
                                lossTotals[cKey] = 0
                            end
                            lossTotals[cKey] = lossTotals[cKey] + 1
                            if not lossDeath[cKey][specKey] then
                                lossDeath[cKey][specKey] = { count = 0, totalTime = 0, timeCount = 0 }
                            end
                            local entry = lossDeath[cKey][specKey]
                            entry.count = entry.count + 1
                            if d.time then
                                entry.totalTime = entry.totalTime + d.time
                                entry.timeCount = entry.timeCount + 1
                            end
                        end
                        break  -- only first player-team death
                    end
                end
            end
        end
    end

    -- Show/hide no data message
    panel.analysisNoData:SetShown(not hasData)
    panel.killHeader:SetShown(hasData)
    if panel.analysisColContainer then panel.analysisColContainer:SetShown(hasData) end

    if not hasData then
        panel.analysisContent:SetHeight(50)
        for _, row in ipairs(panel.killRows) do row:Hide() end
        if panel.RebuildAnalysisPills then panel.RebuildAnalysisPills({}, {}) end
        return
    end

    -- Build kill list sorted by player comp then total games
    local killList = {}
    for cKey, specs in pairs(killTargets) do
        local specList = {}
        for specKey, entry in pairs(specs) do
            specList[#specList + 1] = {
                specKey = specKey,
                count = entry.count,
                avgTime = entry.timeCount > 0 and (entry.totalTime / entry.timeCount) or nil,
            }
        end
        table.sort(specList, function(a, b) return a.count > b.count end)
        killList[#killList + 1] = {
            playerTeam = killPlayer[cKey],
            enemyTeam = killEnemy[cKey],
            playerCompKey = GetSpecCompKey(killPlayer[cKey]),
            total = killTotals[cKey],
            specs = specList,
        }
    end
    table.sort(killList, function(a, b)
        if a.playerCompKey ~= b.playerCompKey then return a.playerCompKey < b.playerCompKey end
        return a.total > b.total
    end)

    -- Build loss list sorted by player comp then total losses
    local lossList = {}
    for cKey, specs in pairs(lossDeath) do
        local specList = {}
        for specKey, entry in pairs(specs) do
            specList[#specList + 1] = {
                specKey = specKey,
                count = entry.count,
                avgTime = entry.timeCount > 0 and (entry.totalTime / entry.timeCount) or nil,
            }
        end
        table.sort(specList, function(a, b) return a.count > b.count end)
        lossList[#lossList + 1] = {
            playerTeam = lossPlayer[cKey],
            enemyTeam = lossEnemy[cKey],
            playerCompKey = GetSpecCompKey(lossPlayer[cKey]),
            total = lossTotals[cKey],
            specs = specList,
        }
    end
    table.sort(lossList, function(a, b)
        if a.playerCompKey ~= b.playerCompKey then return a.playerCompKey < b.playerCompKey end
        return a.total > b.total
    end)

    -- Rebuild spec pills (pass flat lists)
    local pillH = 0
    if panel.RebuildAnalysisPills then
        pillH = panel.RebuildAnalysisPills(killList, lossList)
    end

    -- Filter by spec pills if any active (AND logic: team must contain ALL selected specs)
    local hasActivePills = next(analysisSpecFilters)
    if hasActivePills then
        local function TeamContainsAllFilters(team)
            if not team then return false end
            local teamSpecs = {}
            for _, p in ipairs(team) do
                local spec = ResolveSpec(p.spec)
                local cls = p.class or "UNKNOWN"
                teamSpecs[spec .. " " .. cls] = true
            end
            for specKey in pairs(analysisSpecFilters) do
                if not teamSpecs[specKey] then return false end
            end
            return true
        end
        local function FilterList(list)
            local filtered = {}
            for _, entry in ipairs(list) do
                if TeamContainsAllFilters(entry.playerTeam) or TeamContainsAllFilters(entry.enemyTeam) then
                    filtered[#filtered + 1] = entry
                end
            end
            return filtered
        end
        killList = FilterList(killList)
        lossList = FilterList(lossList)
    end

    -- Select data based on active sub-tab
    local displayList = (analysisSubTab == "kills") and killList or lossList
    local isKills = (analysisSubTab == "kills")

    -- Sort matchups by selected column
    local col = analysisSortCol
    local asc = analysisSortAsc
    -- Group by player comp first, then sort within groups
    local function GroupByPlayerComp(list)
        local groups = {}
        local groupOrder = {}
        for _, data in ipairs(list) do
            local pck = data.playerCompKey or "?"
            if not groups[pck] then
                groups[pck] = { playerTeam = data.playerTeam, matchups = {} }
                groupOrder[#groupOrder + 1] = pck
            end
            groups[pck].matchups[#groups[pck].matchups + 1] = data
        end
        local result = {}
        for _, pck in ipairs(groupOrder) do
            result[#result + 1] = groups[pck]
        end
        return result
    end

    -- Sort the display list by column within each group
    local function SortMatchups(matchups)
        table.sort(matchups, function(a, b)
            local va, vb
            if col == "enemyComp" then
                va = GetSpecCompKey(a.enemyTeam) or ""
                vb = GetSpecCompKey(b.enemyTeam) or ""
            elseif col == "target" then
                va = (a.specs and a.specs[1]) and a.specs[1].specKey or ""
                vb = (b.specs and b.specs[1]) and b.specs[1].specKey or ""
            elseif col == "rate" then
                va = (a.specs and a.specs[1]) and (a.specs[1].count / a.total) or 0
                vb = (b.specs and b.specs[1]) and (b.specs[1].count / b.total) or 0
            elseif col == "avgTime" then
                va = (a.specs and a.specs[1] and a.specs[1].avgTime) or 9999
                vb = (b.specs and b.specs[1] and b.specs[1].avgTime) or 9999
            else -- "games" or default
                va, vb = a.total, b.total
            end
            if va ~= vb then
                if asc then return va < vb else return va > vb end
            end
            return a.total > b.total  -- tiebreaker
        end)
    end

    local displayGroups = GroupByPlayerComp(displayList)
    for _, g in ipairs(displayGroups) do
        SortMatchups(g.matchups)
    end

    -- Update column header labels
    if panel.analysisColButtons then
        for _, btn in ipairs(panel.analysisColButtons) do btn.UpdateLabel() end
    end

    -- Count total rows needed (1 heading per group + matchups)
    local totalRowCount = 0
    for _, g in ipairs(displayGroups) do totalRowCount = totalRowCount + 1 + #g.matchups end

    -- Row factory: single-line rows with columns spread across full width
    local HEADING_HEIGHT = 24
    local MATCHUP_HEIGHT = 24
    local function EnsureAnalysisRows(rows, count)
        while #rows < count do
            local i = #rows + 1
            local row = CreateFrame("Frame", nil, panel.analysisContent, "BackdropTemplate")
            row:SetBackdrop({ bgFile = "Interface\\Tooltips\\UI-Tooltip-Background" })

            -- Left accent bar for heading rows
            row.accent = row:CreateTexture(nil, "ARTWORK")
            row.accent:SetPoint("TOPLEFT", 0, 0)
            row.accent:SetPoint("BOTTOMLEFT", 0, 0)
            row.accent:SetWidth(3)
            row.accent:SetColorTexture(CYAN.r, CYAN.g, CYAN.b, 0.7)
            row.accent:Hide()

            row.headingLabel = row:CreateFontString(nil, "OVERLAY", "GameFontNormal")
            row.headingLabel:SetPoint("LEFT", 8, 0)
            row.headingLabel:SetJustifyH("LEFT")

            -- Columns spread across full width (positions match analysisColDefs)
            row.enemyComp = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
            row.enemyComp:SetPoint("LEFT", 20, 0)
            row.enemyComp:SetWidth(450)
            row.enemyComp:SetJustifyH("LEFT")

            -- Subtle highlight behind kill target area
            row.targetHighlight = row:CreateTexture(nil, "BACKGROUND", nil, 1)
            row.targetHighlight:SetPoint("TOPLEFT", row, "TOPLEFT", 474, 0)
            row.targetHighlight:SetPoint("BOTTOMLEFT", row, "BOTTOMLEFT", 474, 0)
            row.targetHighlight:SetWidth(190)
            row.targetHighlight:SetColorTexture(1, 1, 1, 0.03)
            row.targetHighlight:Hide()

            row.targetSpec = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
            row.targetSpec:SetPoint("LEFT", 480, 0)
            row.targetSpec:SetWidth(180)
            row.targetSpec:SetJustifyH("LEFT")

            row.rateText = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
            row.rateText:SetPoint("LEFT", 670, 0)
            row.rateText:SetWidth(80)
            row.rateText:SetJustifyH("LEFT")

            row.avgTimeText = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
            row.avgTimeText:SetPoint("LEFT", 760, 0)
            row.avgTimeText:SetWidth(80)
            row.avgTimeText:SetJustifyH("LEFT")

            row.gamesText = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
            row.gamesText:SetPoint("LEFT", 850, 0)
            row.gamesText:SetWidth(80)
            row.gamesText:SetJustifyH("LEFT")

            row:EnableMouse(true)
            row:EnableMouseWheel(true)
            row:SetScript("OnMouseWheel", function(_, delta)
                local scroll = panel.analysisScroll
                local cur = scroll:GetVerticalScroll()
                local maxScroll = math.max(0, panel.analysisContent:GetHeight() - scroll:GetHeight())
                scroll:SetVerticalScroll(math.max(0, math.min(maxScroll, cur - delta * 60)))
            end)
            row:SetScript("OnEnter", function(self)
                if not self.isHeading then
                    local r, g, b = self.baseR or 0.06, self.baseG or 0.06, self.baseB or 0.09
                    self:SetBackdropColor(r + 0.06, g + 0.06, b + 0.06, 0.6)
                    -- Tooltip for secondary kill targets
                    if self.tooltipText then
                        GameTooltip:SetOwner(self, "ANCHOR_TOP")
                        GameTooltip:AddLine(self.tooltipTitle or "Target Breakdown", 1, 1, 1)
                        for _, line in ipairs(self.tooltipLines or {}) do
                            GameTooltip:AddLine(line, nil, nil, nil, true)
                        end
                        GameTooltip:Show()
                    end
                end
            end)
            row:SetScript("OnLeave", function(self)
                if self.isHeading then
                    self:SetBackdropColor(CYAN.r * 0.05, CYAN.g * 0.05, CYAN.b * 0.05, 0.35)
                else
                    self:SetBackdropColor(self.baseR or 0.06, self.baseG or 0.06, self.baseB or 0.09, 0.3)
                end
                GameTooltip:Hide()
            end)

            rows[i] = row
        end
    end

    local function ConfigHeading(row, playerTeam)
        row.isHeading = true
        row:SetHeight(HEADING_HEIGHT)
        row:SetBackdropColor(CYAN.r * 0.05, CYAN.g * 0.05, CYAN.b * 0.05, 0.35)
        row.headingLabel:SetText("Playing as  " .. GetSpecCompLabel(playerTeam))
        row.headingLabel:Show()
        row.enemyComp:Hide()
        row.targetSpec:Hide()
        row.rateText:Hide()
        row.avgTimeText:Hide()
        row.gamesText:Hide()
        if row.targetHighlight then row.targetHighlight:Hide() end
        row.accent:Show()
        row.tooltipText = nil
        row.tooltipLines = nil
    end

    local function ConfigMatchupRow(row, data, rowIndex)
        row.isHeading = false
        row:SetHeight(MATCHUP_HEIGHT)
        local alt = (rowIndex or 0) % 2 == 0
        local r, g, b = alt and 0.08 or 0.05, alt and 0.08 or 0.05, alt and 0.11 or 0.08
        row.baseR, row.baseG, row.baseB = r, g, b
        row:SetBackdropColor(r, g, b, 0.3)
        row.accent:Hide()
        row.headingLabel:Hide()
        if row.targetHighlight then
            row.targetHighlight:SetColorTexture(isKills and 0 or 0.4, isKills and 0.3 or 0, 0, 0.08)
            row.targetHighlight:Show()
        end

        -- Col 1: Enemy comp (generous width)
        row.enemyComp:SetText("vs  " .. GetSpecCompLabel(data.enemyTeam))
        row.enemyComp:Show()

        -- Col 2: Kill target (primary spec) — skull icon + class icon + bright spec name
        local skull = "|TInterface\\TargetingFrame\\UI-TargetingFrame-Skull:16:16:0:-1|t "
        if data.specs and data.specs[1] then
            local s = data.specs[1]
            local spec, class = s.specKey:match("^(.+) (%u+)$")
            local icon = class and ClassIconString(class, 14) or ""
            local hex = class and GetClassColorHex(class) or "bbbbbb"
            row.targetSpec:SetText(skull .. icon .. " |cff" .. hex .. (spec or s.specKey) .. "|r")
        else
            row.targetSpec:SetText(skull .. "|cff888888?|r")
        end
        row.targetSpec:Show()

        -- Col 3: Rate
        if data.specs and data.specs[1] then
            local pct = data.specs[1].count / data.total * 100
            row.rateText:SetText(string.format("%.0f%%", pct))
        else
            row.rateText:SetText("--")
        end
        row.rateText:Show()

        -- Col 4: Avg time
        if data.specs and data.specs[1] and data.specs[1].avgTime then
            local t = data.specs[1].avgTime
            local mins = math.floor(t / 60)
            local secs = math.floor(t % 60)
            row.avgTimeText:SetText(string.format("%d:%02d", mins, secs))
        else
            row.avgTimeText:SetText("|cff666666--|r")
        end
        row.avgTimeText:Show()

        -- Col 5: Games count
        local gColor = isKills and "|cff00ff00" or "|cffff4444"
        row.gamesText:SetText(gColor .. data.total .. "|r")
        row.gamesText:Show()

        -- Tooltip for secondary targets (when multiple specs are killed)
        if data.specs and #data.specs > 1 then
            row.tooltipText = true
            row.tooltipTitle = isKills and "Kill Target Breakdown" or "Death Target Breakdown"
            local lines = {}
            for _, s in ipairs(data.specs) do
                local spec, class = s.specKey:match("^(.+) (%u+)$")
                local icon = class and ClassIconString(class, 14) or ""
                local hex = class and GetClassColorHex(class) or "bbbbbb"
                local pct = s.count / data.total * 100
                local timeStr = ""
                if s.avgTime then
                    local mins = math.floor(s.avgTime / 60)
                    local secs = math.floor(s.avgTime % 60)
                    timeStr = "  @ " .. string.format("%d:%02d", mins, secs)
                end
                lines[#lines + 1] = icon .. " |cff" .. hex .. (spec or s.specKey) .. "|r  " .. string.format("%.0f%%", pct) .. " (" .. s.count .. ")" .. timeStr
            end
            row.tooltipLines = lines
        else
            row.tooltipText = nil
            row.tooltipLines = nil
        end
    end

    -- Ensure enough rows
    EnsureAnalysisRows(panel.killRows, totalRowCount)

    -- Populate rows (grouped by player comp, sub-tab determines wins vs losses)
    local ri = 0
    local matchIdx = 0
    for _, group in ipairs(displayGroups) do
        -- Heading row
        ri = ri + 1
        local hRow = panel.killRows[ri]
        hRow:ClearAllPoints()
        hRow:SetPoint("LEFT", panel.analysisContent, "LEFT", 6, 0)
        hRow:SetPoint("RIGHT", panel.analysisContent, "RIGHT", -6, 0)
        if ri == 1 then
            hRow:SetPoint("TOP", panel.analysisContent, "TOP", 0, -6)
        else
            hRow:SetPoint("TOP", panel.killRows[ri - 1], "BOTTOM", 0, -8)
        end
        ConfigHeading(hRow, group.playerTeam)
        hRow:Show()
        matchIdx = 0

        -- Matchup rows
        for _, data in ipairs(group.matchups) do
            ri = ri + 1
            matchIdx = matchIdx + 1
            local mRow = panel.killRows[ri]
            mRow:ClearAllPoints()
            mRow:SetPoint("LEFT", panel.analysisContent, "LEFT", 16, 0)
            mRow:SetPoint("RIGHT", panel.analysisContent, "RIGHT", -6, 0)
            mRow:SetPoint("TOP", panel.killRows[ri - 1], "BOTTOM", 0, -1)
            ConfigMatchupRow(mRow, data, matchIdx)
            mRow:Show()
        end
    end
    for i = ri + 1, #panel.killRows do
        panel.killRows[i]:Hide()
    end

    -- Update content height for scroll
    local totalHeight = 10
    for gi, g in ipairs(displayGroups) do
        totalHeight = totalHeight + HEADING_HEIGHT + 1
        totalHeight = totalHeight + (#g.matchups * (MATCHUP_HEIGHT + 1))
        if gi < #displayGroups then totalHeight = totalHeight + 8 end
    end
    totalHeight = totalHeight + 30
    panel.analysisContent:SetHeight(math.max(1, totalHeight))
    panel.analysisScroll:SetVerticalScroll(0)
end

-----------------------------
-- Players (Scouting) Panel
-----------------------------
RefreshPlayersPanel = function()
    if not historyFrame or not historyFrame.statsPanel then return end
    local panel = historyFrame.statsPanel

    -- Aggregate player stats (cached — only rebuild when matches change)
    if playersCacheDirty or playersCachedFilter ~= activeCharFilter then
        local matches = SKToolsArenaDB and SKToolsArenaDB.matches or {}
        local playerStats = {}

        for _, m in ipairs(matches) do
            if m.enemyTeam and (not activeCharFilter or m.player == activeCharFilter) then
                local isArena = (m.mode == "arena2v2" or m.mode == "arena3v3")
                local isShuffle = (m.mode == "shuffle")
                local isSkirm = (m.mode == "skirmish")

                if isArena or isShuffle or isSkirm then
                    for _, p in ipairs(m.enemyTeam) do
                        if p.name then
                            local key = p.name
                            if not playerStats[key] then
                                playerStats[key] = {
                                    name = p.name,
                                    class = p.class or "UNKNOWN",
                                    spec = p.spec or "",
                                    arenaWins = 0,
                                    arenaLosses = 0,
                                    shuffleCount = 0,
                                    skirmWins = 0,
                                    skirmLosses = 0,
                                    lastSeen = 0,
                                }
                            end

                            local ps = playerStats[key]
                            if p.spec and p.spec ~= "" then ps.spec = ResolveSpec(p.spec) end
                            if p.class and p.class ~= "UNKNOWN" then ps.class = p.class end
                            if m.timestamp and m.timestamp > ps.lastSeen then ps.lastSeen = m.timestamp end

                            if isArena then
                                if m.won then
                                    ps.arenaWins = ps.arenaWins + 1
                                else
                                    ps.arenaLosses = ps.arenaLosses + 1
                                end
                            elseif isShuffle then
                                ps.shuffleCount = ps.shuffleCount + 1
                            elseif isSkirm then
                                if m.won then
                                    ps.skirmWins = ps.skirmWins + 1
                                else
                                    ps.skirmLosses = ps.skirmLosses + 1
                                end
                            end
                        end
                    end
                end
            end
        end

        -- Pre-compute totalGames for each player
        for _, ps in pairs(playerStats) do
            ps.totalGames = ps.arenaWins + ps.arenaLosses + ps.shuffleCount + ps.skirmWins + ps.skirmLosses
        end
        playersCachedStats = playerStats
        playersCachedFilter = activeCharFilter
        playersCacheDirty = false
    end

    -- Build list from cache, apply search filter
    local playerList = {}
    for _, stats in pairs(playersCachedStats) do
        if playersSearchText == "" or stats.name:lower():find(playersSearchText, 1, true) or stats.spec:lower():find(playersSearchText, 1, true) or stats.class:lower():find(playersSearchText, 1, true) then
            playerList[#playerList + 1] = stats
        end
    end

    -- Sort by selected column
    local col = playersSortCol
    local asc = playersSortAsc
    table.sort(playerList, function(a, b)
        local va, vb
        if col == "name" then
            va, vb = a.name:lower(), b.name:lower()
        elseif col == "spec" then
            va, vb = a.spec:lower(), b.spec:lower()
        elseif col == "arenaWins" then
            local aTotal = a.arenaWins + a.arenaLosses
            local bTotal = b.arenaWins + b.arenaLosses
            va = aTotal > 0 and (a.arenaWins / aTotal) or -1
            vb = bTotal > 0 and (b.arenaWins / bTotal) or -1
        elseif col == "shuffleCount" then
            va, vb = a.shuffleCount, b.shuffleCount
        elseif col == "skirmWins" then
            local aTotal = a.skirmWins + a.skirmLosses
            local bTotal = b.skirmWins + b.skirmLosses
            va = aTotal > 0 and (a.skirmWins / aTotal) or -1
            vb = bTotal > 0 and (b.skirmWins / bTotal) or -1
        elseif col == "lastSeen" then
            va, vb = a.lastSeen, b.lastSeen
        else
            va, vb = a.totalGames, b.totalGames
        end
        if va ~= vb then
            if asc then return va < vb else return va > vb end
        end
        return a.totalGames > b.totalGames
    end)

    -- Show/hide no data
    local hasData = #playerList > 0
    panel.playersNoData:SetShown(not hasData)
    panel.playersHeader:SetShown(hasData)
    panel.playersColHeaders:SetShown(hasData)

    if not hasData then
        for _, row in ipairs(panel.playerRows) do row:Hide() end
        panel.playersSlider:Hide()
        return
    end

    -- Store sorted list and update slider
    panel.playerList = playerList
    local maxOffset = math.max(0, #playerList - (#panel.playerRows))
    panel.playersSlider:SetMinMaxValues(0, maxOffset)
    panel.playersSlider:SetValue(0)
    panel.playersSlider:SetShown(maxOffset > 0)
    panel.UpdatePlayerRows(0)
end

-----------------------------
-- Rating Graph
-----------------------------
local BRACKET_COLORS = {
    arena2v2 = { 0.40, 0.85, 1.00 },   -- light blue
    arena3v3 = { 1.00, 0.80, 0.20 },   -- gold
    shuffle  = { 0.00, 0.90, 0.45 },   -- green
    blitz    = { 1.00, 0.45, 0.20 },   -- orange
    rbg      = { 0.85, 0.30, 0.90 },   -- purple
}

local function GetBracketColor(mode)
    local c = BRACKET_COLORS[mode]
    if c then return c[1], c[2], c[3] end
    return CYAN.r, CYAN.g, CYAN.b
end

RefreshGraphPanel = function()
    if not historyFrame or not historyFrame.graphPanel then return end
    local panel = historyFrame.graphPanel
    local plotArea = panel.plotArea

    -- Hide all pooled elements
    for _, tex in ipairs(panel.gridLines) do tex:Hide() end
    for _, fs in ipairs(panel.yLabels) do fs:Hide() end
    for _, fs in ipairs(panel.xLabels) do fs:Hide() end
    for _, tex in ipairs(panel.lineSegments) do tex:Hide() end
    for _, dp in ipairs(panel.dataPoints) do dp.dot:Hide(); dp.btn:Hide() end
    for _, le in ipairs(panel.legendEntries) do le.btn:Hide() end
    panel.noData:Hide()

    -- Compute date filter cutoff
    local dateCutoff = nil
    local dateEnd = nil
    if graphDateFilter then
        local now = time()
        if graphDateFilter == "today" then
            local t = date("*t", now)
            dateCutoff = time({ year = t.year, month = t.month, day = t.day, hour = 0, min = 0, sec = 0 })
        elseif graphDateFilter == "week" then
            local t = date("*t", now)
            local wday = t.wday == 1 and 7 or (t.wday - 1)  -- Convert Sun=1..Sat=7 to Mon=1..Sun=7
            dateCutoff = time({ year = t.year, month = t.month, day = t.day, hour = 0, min = 0, sec = 0 }) - (wday - 1) * 86400
        elseif graphDateFilter == "month" then
            local t = date("*t", now)
            dateCutoff = time({ year = t.year, month = t.month, day = 1, hour = 0, min = 0, sec = 0 })
        elseif graphDateFilter == "3months" then
            dateCutoff = now - 90 * 86400
        elseif graphDateFilter == "custom" then
            dateCutoff = graphDateCustomStart
            dateEnd = graphDateCustomEnd
        end
    end

    -- Helper: find the player's spec in a match
    local function GetPlayerSpec(m)
        if not m.playerTeam or not m.player then return nil end
        local shortName = m.player:match("^([^%-]+)")
        if not shortName then return nil end
        for _, p in ipairs(m.playerTeam) do
            if p.name and p.name:find(shortName, 1, true) then
                return ResolveSpec(p.spec)
            end
        end
        return nil
    end

    -- Gather rated matches, grouped by mode (chronological = oldest first)
    local filtered = GetFilteredMatches()
    local modeData = {}
    local modeOrder = {}
    local globalMinTime = math.huge
    local globalMaxTime = -math.huge

    for i = #filtered, 1, -1 do
        local m = filtered[i]
        if m.isRated and m.rating and m.timestamp then
            -- Graph date filter
            if dateCutoff and m.timestamp < dateCutoff then
                -- skip
            elseif dateEnd and m.timestamp > dateEnd then
                -- skip
            -- Graph spec filter
            elseif graphSpecFilter and GetPlayerSpec(m) ~= graphSpecFilter then
                -- skip
            else
                local mode = m.mode
                if not modeData[mode] then
                    modeData[mode] = {}
                    modeOrder[#modeOrder + 1] = mode
                end
                modeData[mode][#modeData[mode] + 1] = m
                if m.timestamp < globalMinTime then globalMinTime = m.timestamp end
                if m.timestamp > globalMaxTime then globalMaxTime = m.timestamp end
            end
        end
    end

    -- Check for data
    local totalPoints = 0
    for _, matches in pairs(modeData) do
        totalPoints = totalPoints + #matches
    end
    if totalPoints == 0 then
        panel.noData:Show()
        return
    end

    -- Time range
    local timeRange = globalMaxTime - globalMinTime
    if timeRange < 86400 then  -- at least 1 day range
        globalMinTime = globalMinTime - 43200
        globalMaxTime = globalMaxTime + 43200
        timeRange = globalMaxTime - globalMinTime
    end

    -- Axis scaling: global min/max rating across all brackets
    local globalMinRating = math.huge
    local globalMaxRating = -math.huge

    for _, mode in ipairs(modeOrder) do
        local matches = modeData[mode]
        for _, m in ipairs(matches) do
            if m.rating < globalMinRating then globalMinRating = m.rating end
            if m.rating > globalMaxRating then globalMaxRating = m.rating end
        end
    end

    -- Ensure minimum range and add padding
    local ratingRange = globalMaxRating - globalMinRating
    if ratingRange < 50 then
        local mid = (globalMaxRating + globalMinRating) / 2
        globalMinRating = mid - 25
        globalMaxRating = mid + 25
        ratingRange = 50
    end
    local padding = ratingRange * 0.1
    globalMinRating = globalMinRating - padding
    globalMaxRating = globalMaxRating + padding

    -- Round to nice numbers
    globalMinRating = math.floor(globalMinRating / 25) * 25
    globalMaxRating = math.ceil(globalMaxRating / 25) * 25
    ratingRange = globalMaxRating - globalMinRating

    -- Plot dimensions
    local plotWidth = plotArea:GetWidth()
    local plotHeight = plotArea:GetHeight()
    if plotWidth < 1 or plotHeight < 1 then return end

    local function RatingToY(rating)
        return ((rating - globalMinRating) / ratingRange) * plotHeight
    end

    local function TimeToX(ts)
        return ((ts - globalMinTime) / timeRange) * plotWidth
    end

    -- Grid step
    local gridStep
    if ratingRange <= 100 then gridStep = 25
    elseif ratingRange <= 250 then gridStep = 50
    elseif ratingRange <= 600 then gridStep = 100
    elseif ratingRange <= 1200 then gridStep = 200
    else gridStep = 250
    end

    local gridIdx = 0

    -- Horizontal grid lines + Y-axis labels
    local firstGrid = math.ceil(globalMinRating / gridStep) * gridStep
    for rating = firstGrid, globalMaxRating, gridStep do
        gridIdx = gridIdx + 1
        local y = RatingToY(rating)

        local line = panel.GetGridLine(gridIdx)
        line:ClearAllPoints()
        line:SetRotation(0)
        line:SetHeight(1)
        line:SetPoint("LEFT", plotArea, "BOTTOMLEFT", 0, y)
        line:SetPoint("RIGHT", plotArea, "BOTTOMRIGHT", 0, y)
        line:SetColorTexture(1, 1, 1, 0.08)
        line:Show()

        local label = panel.GetYLabel(gridIdx)
        label:ClearAllPoints()
        label:SetPoint("RIGHT", plotArea, "BOTTOMLEFT", -4, y)
        label:SetText(tostring(math.floor(rating)))
        label:SetTextColor(0.6, 0.6, 0.6)
        label:Show()
    end

    -- Vertical grid lines + X-axis date labels
    local xGridCount = math.min(8, math.max(2, math.floor(plotWidth / 120)))
    for i = 1, xGridCount do
        gridIdx = gridIdx + 1
        local frac = xGridCount > 1 and ((i - 1) / (xGridCount - 1)) or 0
        local x = frac * plotWidth
        local ts = globalMinTime + frac * timeRange

        local line = panel.GetGridLine(gridIdx)
        line:ClearAllPoints()
        line:SetRotation(0)
        line:SetWidth(1)
        line:SetPoint("BOTTOM", plotArea, "BOTTOMLEFT", x, 0)
        line:SetPoint("TOP", plotArea, "TOPLEFT", x, 0)
        line:SetColorTexture(1, 1, 1, 0.05)
        line:Show()

        -- Date label
        local dateFmt = (date("%Y", ts) ~= date("%Y")) and "%m/%d/%y" or "%m/%d"
        local xLabel = panel.GetXLabel(i)
        xLabel:ClearAllPoints()
        xLabel:SetPoint("TOP", plotArea, "BOTTOMLEFT", x, -2)
        xLabel:SetText(date(dateFmt, ts))
        xLabel:SetTextColor(0.5, 0.5, 0.5)
        xLabel:Show()
    end

    -- Plot lines and data points
    local segIdx = 0
    local dpIdx = 0

    for _, mode in ipairs(modeOrder) do
        if graphHiddenModes[mode] then
            -- skip drawing hidden modes
        else
        local matches = modeData[mode]
        local lr, lg, lb = GetBracketColor(mode)

        for i, m in ipairs(matches) do
            local x = TimeToX(m.timestamp)
            local y = RatingToY(m.rating)

            -- Line segment from previous point
            if i > 1 then
                local prevM = matches[i - 1]
                local px = TimeToX(prevM.timestamp)
                local py = RatingToY(prevM.rating)

                segIdx = segIdx + 1
                local seg = panel.GetLineSegment(segIdx)
                panel.DrawLine(seg, px, py, x, y, 1.5, lr, lg, lb, 0.8)
            end

            -- Data point dot (green=win, red=loss)
            dpIdx = dpIdx + 1
            local dp = panel.GetDataPoint(dpIdx)
            local dr, dg, db
            if m.won then
                dr, dg, db = 0, 1, 0
            else
                dr, dg, db = 1, 0.27, 0.27
            end
            dp.dot:ClearAllPoints()
            dp.dot:SetSize(6, 6)
            dp.dot:SetPoint("CENTER", plotArea, "BOTTOMLEFT", x, y)
            dp.dot:SetColorTexture(dr, dg, db, 1)
            dp.dot:Show()

            dp.btn:ClearAllPoints()
            dp.btn:SetPoint("CENTER", plotArea, "BOTTOMLEFT", x, y)
            dp.btn.matchData = m
            dp.btn:Show()
        end
        end -- else (not hidden)
    end

    -- Legend (positioned below title, left-aligned, clickable toggles)
    if #modeOrder >= 1 then
        local legendX = 10
        for i, mode in ipairs(modeOrder) do
            local entry = panel.GetLegendEntry(i)
            local lr, lg, lb = GetBracketColor(mode)
            local modeLabel = MODE_LABELS[mode] or mode
            local hidden = graphHiddenModes[mode]

            entry.btn.modeKey = mode
            entry.swatch:SetColorTexture(lr, lg, lb, hidden and 0.3 or 1)

            entry.label:SetText(modeLabel)
            if hidden then
                entry.label:SetTextColor(0.4, 0.4, 0.4)
            else
                entry.label:SetTextColor(1, 1, 1)
            end

            entry.btn:SetSize(entry.label:GetStringWidth() + 18, 14)
            entry.btn:ClearAllPoints()
            entry.btn:SetPoint("TOPLEFT", panel, "TOPLEFT", legendX, -22)
            entry.btn:Show()

            legendX = legendX + entry.btn:GetWidth() + 8
        end
    end
end

-----------------------------
-- Refresh UI
-----------------------------
function SKPvPHistory_RefreshUI()
    if not historyFrame or not historyFrame:IsShown() then return end

    -- Update summary
    local matches = SKToolsArenaDB and SKToolsArenaDB.matches or {}
    local totalW, totalL = 0, 0
    for _, m in ipairs(matches) do
        if m.won then totalW = totalW + 1 else totalL = totalL + 1 end
    end
    local totalPct = (totalW + totalL) > 0 and string.format("%.1f%%", totalW / (totalW + totalL) * 100) or "0%"

    local sessionStr = string.format("Session: |cff00ff00%dW|r-|cffff4444%dL|r", sessionWins, sessionLosses)
    if sessionRating ~= 0 then
        local sign = sessionRating >= 0 and "+" or ""
        local color = sessionRating >= 0 and "|cff00ff00" or "|cffff4444"
        sessionStr = sessionStr .. "  " .. color .. sign .. sessionRating .. " rating|r"
    end
    local overallStr = string.format("Overall: |cff00ff00%dW|r-|cffff4444%dL|r (%s)", totalW, totalL, totalPct)

    historyFrame.summary:SetText(sessionStr .. "    " .. overallStr)

    -- Update filter button states
    if filterButtons then
        for _, b in ipairs(filterButtons) do
            local isActive = (b.filterName == activeFilter)
            if isActive then
                b:SetBackdropColor(CYAN.r * 0.3, CYAN.g * 0.3, CYAN.b * 0.3, 1)
                b:SetBackdropBorderColor(CYAN.r, CYAN.g, CYAN.b, 0.8)
                b.text:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
            else
                b:SetBackdropColor(0.1, 0.1, 0.12, 0.8)
                b:SetBackdropBorderColor(0.3, 0.3, 0.35, 0.6)
                b.text:SetTextColor(0.7, 0.7, 0.7)
            end
        end
    end

    -- Refresh the active view
    if activeView == "winrates" or activeView == "analysis" or activeView == "players" then
        RefreshStatsPanel()
    elseif activeView == "graph" then
        RefreshGraphPanel()
    else
        -- Update scroll range
        local filtered = GetFilteredMatches()
        local maxScroll = math.max(0, #filtered - VISIBLE_ROWS)
        historyFrame.slider:SetMinMaxValues(0, maxScroll)
        historyFrame.slider:SetValue(0)

        SKPvPHistory_UpdateRows(0)
    end
end

-----------------------------
-- Toggle
-----------------------------
function SKPvPHistory_Toggle()
    CreateHistoryFrame()

    if historyFrame:IsShown() then
        historyFrame:Hide()
    else
        -- Restore saved position
        if SKToolsArenaDB and SKToolsArenaDB.framePos then
            local pos = SKToolsArenaDB.framePos
            historyFrame:ClearAllPoints()
            historyFrame:SetPoint(pos.point, UIParent, pos.relPoint, pos.x, pos.y)
        end
        historyFrame:Show()
        ns.FadeIn(historyFrame, 0.15)
        ShowView("matches")
    end
end

-----------------------------
-- REFlex data import
-----------------------------
-- Map ID remap (same as REFlex uses)
local REFLEX_MAP_REMAP = { [968]=566, [998]=1035, [1681]=2107, [2197]=30 }

local function ImportREFlexData(manual)
    -- Guard: REFlex must be loaded with data
    if type(REFlexDatabase) ~= "table" then
        if manual then
            print("|cff00E5EESKTools:|r REFlex not found. Make sure the REFlex addon is installed and enabled.")
        end
        return
    end
    if #REFlexDatabase == 0 then
        if manual then
            print("|cff00E5EESKTools:|r REFlex database is empty — nothing to import.")
        end
        return
    end
    if not SKToolsArenaDB then return end

    local playerName = GetPlayerFullName()
    if not playerName then
        if manual then
            print("|cff00E5EESKTools:|r Could not determine player name.")
        end
        return
    end

    -- Already imported for this character? (skip silently for auto, allow manual override)
    if SKToolsArenaDB.reflexImported and SKToolsArenaDB.reflexImported[playerName] then
        if not manual then return end
        -- Manual reimport: remove old reflex entries first
        local removed = 0
        local matches = SKToolsArenaDB.matches
        for i = #matches, 1, -1 do
            if matches[i].source == "reflex" and matches[i].player == playerName then
                table.remove(matches, i)
                removed = removed + 1
            end
        end
        if removed > 0 then
            playersCacheDirty = true
            compCacheDirty = true
        end
        SKToolsArenaDB.reflexImported[playerName] = nil
    end

    -- Build a timestamp set for dedup
    local existingTimestamps = {}
    for _, m in ipairs(SKToolsArenaDB.matches) do
        if m.player == playerName and m.timestamp then
            existingTimestamps[m.timestamp] = true
        end
    end

    local imported = 0
    for _, r in ipairs(REFlexDatabase) do
        if not r.Hidden and r.Time and r.Players then
            -- Skip if we already have this match
            if not existingTimestamps[r.Time] then
                -- Determine mode
                local mode, isRated
                if r.isSoloShuffle then
                    mode = "shuffle"
                    isRated = true
                elseif r.isArena then
                    if r.isRated then
                        if (r.PlayersNum or 0) <= 4 then
                            mode = "arena2v2"
                        else
                            mode = "arena3v3"
                        end
                        isRated = true
                    else
                        mode = "skirmish"
                        isRated = false
                    end
                else
                    if r.isRated then
                        -- Distinguish RBG (10v10) from Blitz (8v8) by team size
                        local teamCount = 0
                        if r.Players then
                            for _, p in ipairs(r.Players) do
                                if type(p) == "table" and p[6] == r.PlayerSide then
                                    teamCount = teamCount + 1
                                end
                            end
                        end
                        mode = (teamCount > 0 and teamCount <= 8) and "blitz" or "rbg"
                        isRated = true
                    elseif (r.PlayersNum or 0) > 30 then
                        mode = "epicBG"
                        isRated = false
                    else
                        mode = "randomBG"
                        isRated = false
                    end
                end

                -- Resolve map name: Reflex stores instance map IDs, use GetRealZoneText (not C_Map)
                local mapID = r.Map
                if mapID and REFLEX_MAP_REMAP[mapID] then
                    mapID = REFLEX_MAP_REMAP[mapID]
                end
                local mapName = "Unknown"
                if mapID then
                    local name = GetRealZoneText(mapID)
                    if name and name ~= "" then mapName = name end
                end

                -- Build teams from Players array
                local playerTeam, enemyTeam = {}, {}
                local myStats = nil  -- player's own stats from scoreboard
                local playerSide = r.PlayerSide  -- 0=Horde, 1=Alliance

                for idx, p in ipairs(r.Players) do
                    if type(p) == "table" then
                        local entry = {
                            class = p[9] or "UNKNOWN",
                            spec = p[16] or "",
                            name = p[1] or nil,
                            race = p[7] or nil,
                            kills = p[2] or 0,
                            deaths = p[4] or 0,
                            damage = p[10] or 0,
                            healing = p[11] or 0,
                            rating = (p[12] and p[12] ~= 0) and p[12] or nil,
                            ratingChange = (p[13] and p[13] ~= 0) and p[13] or nil,
                            honorLevel = p[17] or nil,
                        }
                        if p[6] == playerSide then
                            table.insert(playerTeam, entry)
                        else
                            table.insert(enemyTeam, entry)
                        end
                        if idx == r.PlayerNum then
                            myStats = p
                        end
                    end
                end

                -- Player's own stats
                local kills = myStats and (myStats[2] or 0) or 0
                local deaths = myStats and (myStats[4] or 0) or 0
                local honorableKills = myStats and (myStats[3] or 0) or 0
                local damage = myStats and (myStats[10] or 0) or 0
                local healing = myStats and (myStats[11] or 0) or 0
                local honor = myStats and (myStats[5] or 0) or 0

                -- Rating from per-player data; MMR from TeamData
                local rating, ratingChange, mmr, enemyMmr
                if isRated and myStats then
                    -- Per-player rating and ratingChange from scoreboard
                    if myStats[12] and myStats[12] ~= 0 then
                        rating = myStats[12]
                    end
                    -- ratingChange: [13] is unreliable for shuffle (returns garbage values >100)
                    if myStats[13] and myStats[13] ~= 0 then
                        local rc = myStats[13]
                        if rc >= -100 and rc <= 100 then
                            ratingChange = rc
                        end
                    end
                    -- Team MMR from TeamData (index [4] is MMR; [2] and [3] are deprecated/-1 values)
                    if r.TeamData then
                        local teamIdx = (playerSide == 0) and 1 or 2
                        local enemyIdx = (playerSide == 0) and 2 or 1
                        local td = r.TeamData[teamIdx]
                        if td and td[4] and td[4] > 0 then
                            mmr = td[4]
                        end
                        local etd = r.TeamData[enemyIdx]
                        if etd and etd[4] and etd[4] > 0 then
                            enemyMmr = etd[4]
                        end
                    end
                end

                -- Solo shuffle round tracking
                local roundWins, roundLosses
                if mode == "shuffle" and r.PlayerStats and r.PlayerStats[1] then
                    roundWins = r.PlayerStats[1]
                    roundLosses = 6 - roundWins
                end

                -- Won?
                local won = (r.Winner == playerSide)

                -- Build match record
                local ts = r.Time
                local match = {
                    timestamp = ts,
                    date = date("%m/%d", ts),
                    timeStr = date("%H:%M", ts),
                    mode = mode,
                    mapName = mapName,
                    isRated = isRated,
                    won = won,
                    rating = rating,
                    ratingChange = ratingChange,
                    mmr = mmr,
                    kills = kills,
                    deaths = deaths,
                    honorableKills = honorableKills,
                    damage = damage,
                    healing = healing,
                    honor = honor,
                    playerTeam = playerTeam,
                    enemyTeam = enemyTeam,
                    roundWins = roundWins,
                    roundLosses = roundLosses,
                    duration = r.Duration or 0,
                    player = playerName,
                    enemyMmr = enemyMmr,
                    source = "reflex",
                }

                table.insert(SKToolsArenaDB.matches, match)
                imported = imported + 1
            end
        end
    end

    if imported > 0 then
        -- Sort all matches by timestamp descending (newest first)
        table.sort(SKToolsArenaDB.matches, function(a, b) return a.timestamp > b.timestamp end)

        -- Prune to MAX_MATCHES
        while #SKToolsArenaDB.matches > MAX_MATCHES do
            table.remove(SKToolsArenaDB.matches)
        end
        playersCacheDirty = true
        compCacheDirty = true
    end

    -- Mark as imported for this character
    if not SKToolsArenaDB.reflexImported then
        SKToolsArenaDB.reflexImported = {}
    end
    SKToolsArenaDB.reflexImported[playerName] = true

    if manual then
        if imported > 0 then
            local shortName = playerName:match("^([^%-]+)") or playerName
            print("|cff00ff00SKTools:|r Imported " .. imported .. " matches from REFlex for " .. shortName .. ".")
        else
            print("|cff00E5EESKTools:|r No new matches to import from REFlex.")
        end
    end

    -- Refresh UI if open
    if imported > 0 and historyFrame and historyFrame:IsShown() then
        ShowView(activeView)
    end
end

-- Expose for settings button
ns.ImportREFlexData = function() ImportREFlexData(true) end

-----------------------------
-- Init saved variables
-----------------------------
local initFrame = CreateFrame("Frame")
initFrame:RegisterEvent("ADDON_LOADED")
initFrame:SetScript("OnEvent", function(self, event, addon)
    if addon ~= "SKTools" then return end

    if not SKToolsArenaDB then
        SKToolsArenaDB = {}
    end
    if not SKToolsArenaDB.matches then
        SKToolsArenaDB.matches = {}
    end
    if not SKToolsArenaDB.reflexImported then
        SKToolsArenaDB.reflexImported = {}
    end

    -- Defer REFlex import to ensure REFlex SavedVariables are loaded
    C_Timer.After(2, ImportREFlexData)

    self:UnregisterEvent("ADDON_LOADED")
end)
