-- SKTools PvE History
-- Tracks M+ dungeons, raid lockouts, and dungeon runs with boss timestamps

local _, ns = ...

local MAX_MATCHES = 5000
local ROW_HEIGHT = 26
local VISIBLE_ROWS = 22
local HEADER_HEIGHT = 28
local CYAN = { r = 0, g = 0.898, b = 0.933 }  -- #00E5EE
local GREEN = { r = 0, g = 0.8, b = 0.4 }     -- PvE accent

local sessionMplus = { completed = 0, timed = 0, depleted = 0, ratingChange = 0 }
local sessionRaid = { kills = 0, wipes = 0 }
local sessionDungeon = { completed = 0 }

local historyFrame, rows, filterButtons, viewButtons
local activeFilter = "All"
local activeView = "matches"
local sortColumn = "timestamp"
local sortAscending = false
local activeCharFilter = nil
local ShowView, RefreshGraphPanel  -- forward declarations

-- Active run tracking
local activeRun = nil           -- pointer to current run record
local activeRunIndex = nil      -- index in matches[] if resuming raid lockout
local encounterStartTimes = {}  -- [encounterID] = GetTime()
local runStartTime = nil        -- GetTime() for elapsed calculations
local mplusDeathCount = 0
local pendingRaidLookup = false -- waiting for UPDATE_INSTANCE_INFO

-- Save/restore active run for reload/disconnect recovery
-- PLAYER_LOGOUT fires on /reload and /quit (not on crash/disconnect)
-- So pendingRun covers reloads; provisional activeRun covers disconnects
local function SavePendingRun()
    if not SKToolsPvEDB or not activeRun then
        if SKToolsPvEDB then SKToolsPvEDB.pendingRun = nil; SKToolsPvEDB.pendingRunElapsed = nil end
        return
    end
    if activeRun.contentType == "raid" then return end  -- raids use lockoutId recovery
    -- For dungeons, calculate total elapsed time to persist across reload
    local totalElapsed = 0
    if activeRun.contentType == "dungeon" then
        totalElapsed = (activeRun.duration or 0)
        if runStartTime then
            totalElapsed = totalElapsed + (GetTime() - runStartTime)
        end
    end
    SKToolsPvEDB.pendingRun = activeRun
    SKToolsPvEDB.pendingRunElapsed = totalElapsed
end

local function ClearPendingRun()
    if SKToolsPvEDB then
        SKToolsPvEDB.pendingRun = nil
        SKToolsPvEDB.pendingRunElapsed = nil
    end
end

-----------------------------
-- Helpers
-----------------------------
local function GetPlayerFullName()
    local name, realm = UnitFullName("player")
    if not name then return nil end
    if not realm or realm == "" then realm = GetNormalizedRealmName() or "" end
    return name .. "-" .. realm
end

local CLASS_COLORS = {
    WARRIOR     = { r = 0.78, g = 0.61, b = 0.43 },
    PALADIN     = { r = 0.96, g = 0.55, b = 0.73 },
    HUNTER      = { r = 0.67, g = 0.83, b = 0.45 },
    ROGUE       = { r = 1.00, g = 0.96, b = 0.41 },
    PRIEST      = { r = 1.00, g = 1.00, b = 1.00 },
    DEATHKNIGHT = { r = 0.77, g = 0.12, b = 0.23 },
    SHAMAN      = { r = 0.00, g = 0.44, b = 0.87 },
    MAGE        = { r = 0.25, g = 0.78, b = 0.92 },
    WARLOCK     = { r = 0.53, g = 0.53, b = 0.93 },
    MONK        = { r = 0.00, g = 1.00, b = 0.60 },
    DRUID       = { r = 1.00, g = 0.49, b = 0.04 },
    DEMONHUNTER = { r = 0.64, g = 0.19, b = 0.79 },
    EVOKER      = { r = 0.20, g = 0.58, b = 0.50 },
}

local function GetClassColor(classFile)
    if RAID_CLASS_COLORS and RAID_CLASS_COLORS[classFile] then
        local c = RAID_CLASS_COLORS[classFile]
        return c.r, c.g, c.b
    end
    local c = CLASS_COLORS[classFile]
    if c then return c.r, c.g, c.b end
    return 0.5, 0.5, 0.5
end

local function GetClassColorHex(classFile)
    local r, g, b = GetClassColor(classFile)
    return string.format("%02x%02x%02x", r * 255, g * 255, b * 255)
end

local CLASS_ICON_ATLAS = {
    WARRIOR     = "classicon-warrior",
    PALADIN     = "classicon-paladin",
    HUNTER      = "classicon-hunter",
    ROGUE       = "classicon-rogue",
    PRIEST      = "classicon-priest",
    DEATHKNIGHT = "classicon-deathknight",
    SHAMAN      = "classicon-shaman",
    MAGE        = "classicon-mage",
    WARLOCK     = "classicon-warlock",
    MONK        = "classicon-monk",
    DRUID       = "classicon-druid",
    DEMONHUNTER = "classicon-demonhunter",
    EVOKER      = "classicon-evoker",
}

local function ClassIconString(classFile, size)
    size = size or 14
    local atlas = CLASS_ICON_ATLAS[classFile]
    if atlas then
        return CreateAtlasMarkup(atlas, size, size)
    end
    return ""
end

-- Race icon support (mirrors PvPHistory)
local raceNameToClientFile
local function EnsureRaceCache()
    if raceNameToClientFile then return end
    raceNameToClientFile = {}
    if not C_CreatureInfo or not C_CreatureInfo.GetRaceInfo then return end
    for raceID = 1, 100 do
        local info = C_CreatureInfo.GetRaceInfo(raceID)
        if info and info.clientFileString then
            local low = info.clientFileString:lower()
            if info.raceName then raceNameToClientFile[info.raceName] = low end
            raceNameToClientFile[info.clientFileString] = low
            raceNameToClientFile[low] = low
        end
    end
end

local RACE_SHORT_KEY = {
    ["highmountaintauren"]  = "highmountain",
    ["lightforgeddraenei"]  = "lightforged",
    ["magharorc"]           = "maghar",
    ["kultiran"]            = "kultianhuman",
    ["zandalaritroll"]      = "zandalari",
    ["earthendwarf"]        = "earthen",
}

local raceAtlasCache = {}
local RACE_ATLAS_FORMATS = {
    "raceicon128-%s%s", "raceicon-%s-%s", "raceicon-%s%s",
    "raceicon128-%s-%s", "raceicon64-%s%s", "raceicon64-%s-%s",
}

local function FindRaceAtlas(raceKey, gender)
    local cacheKey = raceKey .. "|" .. gender
    if raceAtlasCache[cacheKey] ~= nil then return raceAtlasCache[cacheKey] end
    if C_Texture and C_Texture.GetAtlasInfo then
        for _, fmt in ipairs(RACE_ATLAS_FORMATS) do
            local atlas = string.format(fmt, raceKey, gender)
            if C_Texture.GetAtlasInfo(atlas) then
                raceAtlasCache[cacheKey] = atlas
                return atlas
            end
        end
    end
    raceAtlasCache[cacheKey] = false
    return false
end

local function RaceIconString(raceFile, raceName, sex, size)
    size = size or 16
    EnsureRaceCache()
    local gender = (sex == 3) and "female" or "male"
    local keysToTry = {}
    if raceFile and raceFile ~= "" then
        keysToTry[#keysToTry + 1] = raceFile
        local short = RACE_SHORT_KEY[raceFile]
        if short then keysToTry[#keysToTry + 1] = short end
    end
    if raceName and raceName ~= "" then
        local fromCache = raceNameToClientFile[raceName]
        if fromCache then
            keysToTry[#keysToTry + 1] = fromCache
            local short = RACE_SHORT_KEY[fromCache]
            if short then keysToTry[#keysToTry + 1] = short end
        end
        local direct = raceName:lower():gsub("%s+", ""):gsub("'", "")
        keysToTry[#keysToTry + 1] = direct
        local short = RACE_SHORT_KEY[direct]
        if short then keysToTry[#keysToTry + 1] = short end
    end
    local tried = {}
    for _, key in ipairs(keysToTry) do
        if not tried[key] then
            tried[key] = true
            local atlas = FindRaceAtlas(key, gender)
            if atlas then return CreateAtlasMarkup(atlas, size, size) end
        end
    end
    return ""
end

local function FormatNumber(n)
    if not n or n == 0 then return "0" end
    if n >= 1000000 then
        return string.format("%.1fM", n / 1000000)
    elseif n >= 1000 then
        return string.format("%.1fK", n / 1000)
    end
    return tostring(n)
end

local function FormatDuration(seconds)
    if not seconds or seconds <= 0 then return "--" end
    if seconds >= 3600 then
        local h = math.floor(seconds / 3600)
        local m = math.floor((seconds % 3600) / 60)
        return string.format("%d:%02d:%02d", h, m, seconds % 60)
    end
    return string.format("%d:%02d", math.floor(seconds / 60), seconds % 60)
end

-----------------------------
-- Difficulty Mapping
-----------------------------
local DIFFICULTY_SHORT = {
    [1]  = "N",      -- Normal dungeon
    [2]  = "H",      -- Heroic dungeon
    [8]  = "M+",     -- Mythic Keystone
    [23] = "M0",     -- Mythic dungeon
    [14] = "N",      -- Normal raid
    [15] = "H",      -- Heroic raid
    [16] = "M",      -- Mythic raid
    [17] = "LFR",    -- LFR
}

local CONTENT_TYPE_LABELS = {
    mythicplus = "M+",
    raid       = "Raid",
    dungeon    = "Dungeon",
}

local FILTER_MAP = {
    All      = nil,
    ["M+"]   = { mythicplus = true },
    Raid     = { raid = true },
    Dungeon  = { dungeon = true },
}

-----------------------------
-- Group Roster Collection
-----------------------------
local function CollectGroupRoster()
    local group = {}
    local numMembers = GetNumGroupMembers()
    local function GetUnitRaceInfo(unit)
        local raceName, raceFile = UnitRace(unit)
        local sex = UnitSex(unit)  -- 2=male, 3=female
        return raceName, raceFile and raceFile:lower() or nil, sex
    end

    if numMembers == 0 then
        -- Solo, just add the player
        local name, realm = UnitFullName("player")
        local _, classFile = UnitClass("player")
        local role = UnitGroupRolesAssigned("player") or "NONE"
        local raceName, raceFile, sex = GetUnitRaceInfo("player")
        group[1] = {
            name = (name or "Unknown") .. "-" .. (realm and realm ~= "" and realm or GetNormalizedRealmName() or ""),
            class = classFile or "UNKNOWN",
            role = role,
            race = raceName,
            raceFile = raceFile,
            sex = sex,
        }
        return group
    end

    local isRaid = IsInRaid()
    for i = 1, numMembers do
        local unit = isRaid and ("raid" .. i) or (i < numMembers and ("party" .. i) or "player")
        -- In a 5-player party, party1-4 + player; in raid, raid1-N includes player
        if not isRaid and i == numMembers then unit = "player" end
        if UnitExists(unit) then
            local name, realm = UnitFullName(unit)
            local _, classFile = UnitClass(unit)
            local role = UnitGroupRolesAssigned(unit) or "NONE"
            local raceName, raceFile, sex = GetUnitRaceInfo(unit)
            group[#group + 1] = {
                name = (name or "Unknown") .. "-" .. (realm and realm ~= "" and realm or GetNormalizedRealmName() or ""),
                class = classFile or "UNKNOWN",
                role = role,
                race = raceName,
                raceFile = raceFile,
                sex = sex,
            }
        end
    end
    return group
end

local function CompIconString(party, iconSize)
    if not party or #party == 0 then return "" end
    iconSize = iconSize or 14
    local icons = {}
    for _, p in ipairs(party) do
        icons[#icons + 1] = ClassIconString(p.class, iconSize)
    end
    return table.concat(icons, " ")
end

-----------------------------
-- Filtered Data
-----------------------------
local SORT_FUNCS = {
    timestamp   = function(a, b) return (a.timestamp or 0) < (b.timestamp or 0) end,
    success     = function(a, b) return (a.completed and 1 or 0) < (b.completed and 1 or 0) end,
    contentType = function(a, b) return (a.contentType or "") < (b.contentType or "") end,
    instanceName = function(a, b) return (a.instanceName or "") < (b.instanceName or "") end,
    keyLevel    = function(a, b)
        local aVal = a.keyLevel or a.difficultyID or 0
        local bVal = b.keyLevel or b.difficultyID or 0
        return aVal < bVal
    end,
    duration    = function(a, b) return (a.duration or 0) < (b.duration or 0) end,
    deaths      = function(a, b) return (a.deaths or 0) < (b.deaths or 0) end,
    ratingChange = function(a, b) return (a.ratingChange or 0) < (b.ratingChange or 0) end,
}

local function GetFilteredMatches()
    local matches = SKToolsPvEDB and SKToolsPvEDB.matches or {}
    local filterTypes = FILTER_MAP[activeFilter]

    local filtered = {}
    for _, m in ipairs(matches) do
        local typeOK = not filterTypes or filterTypes[m.contentType]
        local charOK = not activeCharFilter or m.player == activeCharFilter
        if typeOK and charOK then
            filtered[#filtered + 1] = m
        end
    end

    local cmp = SORT_FUNCS[sortColumn]
    if cmp then
        local asc = sortAscending
        table.sort(filtered, function(a, b)
            if asc then return cmp(a, b) else return cmp(b, a) end
        end)
    end

    return filtered
end

-----------------------------
-- Data Collection
-----------------------------
local function GetBossKillCount(run, encounterID)
    local kills, wipes = 0, 0
    if run.bosses then
        for _, b in ipairs(run.bosses) do
            if b.encounterID == encounterID then
                if b.success then kills = kills + 1 else wipes = wipes + 1 end
            end
        end
    end
    return kills, wipes
end

local function GetRunBossProgress(run)
    if not run or not run.bosses then return 0, 0 end
    local killed = {}
    for _, b in ipairs(run.bosses) do
        if b.success then killed[b.encounterID] = true end
    end
    local count = 0
    for _ in pairs(killed) do count = count + 1 end
    local total = run.totalEncounters or count
    return count, total
end

local function FinalizeRun(run)
    if not run then return end
    -- Calculate duration for non-M+ (M+ gets duration from completion info)
    if run.contentType ~= "mythicplus" and runStartTime then
        run.duration = (run.duration or 0) + (GetTime() - runStartTime)
    end
    -- Insert into matches if not already there (raids are inserted on first entry)
    if not activeRunIndex then
        table.insert(SKToolsPvEDB.matches, 1, run)
        -- Prune
        while #SKToolsPvEDB.matches > MAX_MATCHES do
            table.remove(SKToolsPvEDB.matches)
        end
    end
    -- Set dungeon party to last boss kill group
    if run.contentType == "dungeon" and run.bosses then
        for i = #run.bosses, 1, -1 do
            if run.bosses[i].success and run.bosses[i].group then
                run.party = run.bosses[i].group
                break
            end
        end
    end
    ClearPendingRun()
    activeRun = nil
    activeRunIndex = nil
    runStartTime = nil
    encounterStartTimes = {}
end

local function RecordBossEncounter(encounterID, encounterName, success, fightDuration)
    if not activeRun then return end
    if not activeRun.bosses then activeRun.bosses = {} end

    local entry = {
        encounterID = encounterID,
        encounterName = encounterName,
        timestamp = time(),
        success = success,
        fightDuration = fightDuration or 0,
    }

    -- Group snapshot on kills for raids and dungeons
    if success and activeRun.contentType ~= "mythicplus" then
        entry.group = CollectGroupRoster()
    end

    activeRun.bosses[#activeRun.bosses + 1] = entry

    -- Refresh UI if open
    if historyFrame and historyFrame:IsShown() then
        SKPvEHistory_RefreshUI()
    end
end

-- Find existing raid lockout record in saved data
local function FindRaidByLockoutId(lockoutId)
    if not lockoutId or not SKToolsPvEDB or not SKToolsPvEDB.matches then return nil, nil end
    for i, m in ipairs(SKToolsPvEDB.matches) do
        if m.contentType == "raid" and m.lockoutId == lockoutId then
            return m, i
        end
    end
    return nil, nil
end

-- Get lockout ID for current raid instance
local function GetCurrentRaidLockoutId(instanceName, difficultyID)
    local numSaved = GetNumSavedInstances()
    for i = 1, numSaved do
        local name, id, _, diffID, locked, _, _, isRaid = GetSavedInstanceInfo(i)
        if isRaid and name == instanceName and diffID == difficultyID and locked then
            return tostring(id)
        end
    end
    return nil
end

-----------------------------
-- Event Handling
-----------------------------
local dataFrame = CreateFrame("Frame")
dataFrame:RegisterEvent("CHALLENGE_MODE_START")
dataFrame:RegisterEvent("CHALLENGE_MODE_COMPLETED")
dataFrame:RegisterEvent("CHALLENGE_MODE_DEATH_COUNT_UPDATED")
dataFrame:RegisterEvent("ENCOUNTER_START")
dataFrame:RegisterEvent("ENCOUNTER_END")
dataFrame:RegisterEvent("PLAYER_ENTERING_WORLD")
dataFrame:RegisterEvent("SCENARIO_COMPLETED")
dataFrame:RegisterEvent("UPDATE_INSTANCE_INFO")
dataFrame:RegisterEvent("PLAYER_LOGOUT")

local function PvEDataEventHandler(event, ...)
    if SKToolsDB and SKToolsDB.pveHistory == false then return end

    if event == "CHALLENGE_MODE_START" then
        local mapID = C_ChallengeMode.GetActiveChallengeMapID()
        if not mapID then return end

        local name = C_ChallengeMode.GetMapUIInfo(mapID)
        local level, affixIDs = C_ChallengeMode.GetActiveKeystoneInfo()

        local now = time()
        activeRun = {
            timestamp = now,
            date = date("%m/%d", now),
            timeStr = date("%H:%M", now),
            player = GetPlayerFullName(),
            contentType = "mythicplus",
            instanceName = name or "Unknown",
            difficultyID = 8,
            difficultyName = "Mythic Keystone",
            completed = false,
            bosses = {},
            keyLevel = level or 0,
            affixIDs = affixIDs or {},
            deaths = 0,
            mapChallengeModeID = mapID,
        }
        activeRunIndex = nil
        runStartTime = GetTime()
        mplusDeathCount = 0
        encounterStartTimes = {}

    elseif event == "CHALLENGE_MODE_DEATH_COUNT_UPDATED" then
        if activeRun and activeRun.contentType == "mythicplus" then
            local ok, count = pcall(C_ChallengeMode.GetDeathCount)
            if ok and count then
                activeRun.deaths = count
                mplusDeathCount = count
            end
        end

    elseif event == "CHALLENGE_MODE_COMPLETED" then
        if not activeRun or activeRun.contentType ~= "mythicplus" then return end

        local function CollectMplusData()
            local ok, mapID, level, elapsedTime, onTime, keystoneUpgrade, practiceRun,
                  oldScore, newScore, isMapRecord, isAffixRecord = pcall(C_ChallengeMode.GetCompletionInfo)
            if not ok or not mapID then return false end

            local name, _, timeLimit = C_ChallengeMode.GetMapUIInfo(mapID)
            activeRun.instanceName = name or activeRun.instanceName
            activeRun.keyLevel = level or activeRun.keyLevel
            activeRun.timeLimit = timeLimit and (timeLimit / 1000) or 0
            activeRun.duration = elapsedTime and (elapsedTime / 1000) or (runStartTime and (GetTime() - runStartTime) or 0)
            activeRun.onTime = onTime or false
            activeRun.keystoneUpgradeLevels = keystoneUpgrade or 0
            activeRun.completed = true
            activeRun.oldRating = oldScore or 0
            activeRun.newRating = newScore or 0
            activeRun.ratingChange = (newScore or 0) - (oldScore or 0)
            activeRun.isMapRecord = isMapRecord or false
            activeRun.isAffixRecord = isAffixRecord or false
            activeRun.party = CollectGroupRoster()

            -- Update death count one final time
            local dok, dcount = pcall(C_ChallengeMode.GetDeathCount)
            if dok and dcount then activeRun.deaths = dcount end

            -- Session stats
            sessionMplus.completed = sessionMplus.completed + 1
            if activeRun.onTime then
                sessionMplus.timed = sessionMplus.timed + 1
            else
                sessionMplus.depleted = sessionMplus.depleted + 1
            end
            sessionMplus.ratingChange = sessionMplus.ratingChange + (activeRun.ratingChange or 0)

            -- Print confirmation
            local resultStr
            if activeRun.onTime then
                resultStr = "|cff00ff00Timed"
                if (activeRun.keystoneUpgradeLevels or 0) > 0 then
                    resultStr = resultStr .. " +" .. activeRun.keystoneUpgradeLevels
                end
                resultStr = resultStr .. "|r"
            else
                resultStr = "|cffff4444Depleted|r"
            end
            local ratingStr = ""
            if activeRun.ratingChange and activeRun.ratingChange ~= 0 then
                local sign = activeRun.ratingChange >= 0 and "+" or ""
                ratingStr = " (" .. sign .. activeRun.ratingChange .. " -> " .. (activeRun.newRating or "?") .. ")"
            end
            print("|cff00E5EESKTools:|r M+ recorded: " .. resultStr .. " +" .. (activeRun.keyLevel or "?") .. " " .. (activeRun.instanceName or "") .. ratingStr)

            FinalizeRun(activeRun)
            return true
        end

        C_Timer.After(0.5, function()
            local ok, result = pcall(CollectMplusData)
            if not ok or not result then
                C_Timer.After(1.0, function()
                    pcall(CollectMplusData)
                end)
            end
        end)

    elseif event == "ENCOUNTER_START" then
        local encounterID = ...
        encounterStartTimes[encounterID] = GetTime()

    elseif event == "ENCOUNTER_END" then
        local encounterID, encounterName, difficultyID, groupSize, success = ...
        if not activeRun then return end

        -- Skip encounters inside M+ (we track M+ as a whole, not per-boss via ENCOUNTER_END)
        -- But we DO want boss timestamps — record kills, skip wipes for M+
        local fightDuration = 0
        if encounterStartTimes[encounterID] then
            fightDuration = GetTime() - encounterStartTimes[encounterID]
            encounterStartTimes[encounterID] = nil
        end

        -- For M+, only record kills (not wipes) to keep boss timeline clean
        if activeRun.contentType == "mythicplus" then
            if success == 1 then
                RecordBossEncounter(encounterID, encounterName, true, fightDuration)
            end
            return
        end

        -- For raids and dungeons, record both kills and wipes
        RecordBossEncounter(encounterID, encounterName, success == 1, fightDuration)

        -- Raid: check if all bosses cleared after a kill
        if activeRun.contentType == "raid" and success == 1 then
            C_Timer.After(0.5, function()
                if not activeRun or activeRun.contentType ~= "raid" then return end
                local ok, lockTimeLeft, _, encountersTotal, encountersComplete = pcall(GetInstanceLockTimeRemaining)
                if ok and encountersTotal and encountersComplete and encountersTotal > 0 then
                    activeRun.totalEncounters = encountersTotal
                    if encountersComplete >= encountersTotal then
                        activeRun.completed = true
                        print("|cff00E5EESKTools:|r Raid complete! " .. (activeRun.instanceName or ""))
                    end
                end
            end)

            -- Update session stats
            sessionRaid.kills = sessionRaid.kills + 1
        elseif activeRun.contentType == "raid" and success ~= 1 then
            sessionRaid.wipes = sessionRaid.wipes + 1
        end

        -- Refresh UI if open
        if historyFrame and historyFrame:IsShown() then
            SKPvEHistory_RefreshUI()
        end

    elseif event == "SCENARIO_COMPLETED" then
        -- Dungeon completion (non-M+)
        if activeRun and activeRun.contentType == "dungeon" then
            activeRun.completed = true
            sessionDungeon.completed = sessionDungeon.completed + 1
            print("|cff00E5EESKTools:|r Dungeon complete: " .. (activeRun.instanceName or ""))
            FinalizeRun(activeRun)
        end

    elseif event == "UPDATE_INSTANCE_INFO" then
        -- Called after RequestRaidInfo() — now we can look up the lockout ID
        if not pendingRaidLookup then return end
        pendingRaidLookup = false

        if not activeRun or activeRun.contentType ~= "raid" then return end

        local lockoutId = GetCurrentRaidLockoutId(activeRun.instanceName, activeRun.difficultyID)
        if lockoutId then
            -- Check for existing record
            local existing, idx = FindRaidByLockoutId(lockoutId)
            if existing then
                -- Resume existing lockout
                activeRun = existing
                activeRunIndex = idx
                activeRun.visits = activeRun.visits or {}
                activeRun.visits[#activeRun.visits + 1] = { enterTime = time() }
                runStartTime = GetTime()
                return
            end
            activeRun.lockoutId = lockoutId
        end

        -- New raid — get total encounters
        local ok, lockTimeLeft, _, encountersTotal, encountersComplete = pcall(GetInstanceLockTimeRemaining)
        if ok and encountersTotal then
            activeRun.totalEncounters = encountersTotal
        end

        -- Insert into matches
        table.insert(SKToolsPvEDB.matches, 1, activeRun)
        activeRunIndex = 1
        -- Prune
        while #SKToolsPvEDB.matches > MAX_MATCHES do
            table.remove(SKToolsPvEDB.matches)
        end

    elseif event == "PLAYER_LOGOUT" then
        SavePendingRun()

    elseif event == "PLAYER_ENTERING_WORLD" then
        local isLogin, isReload = ...

        local inInstance, instanceType = IsInInstance()

        if not inInstance then
            -- Left an instance — clear any pending run
            ClearPendingRun()
            if activeRun then
                -- Close out the visit for raids
                if activeRun.contentType == "raid" and activeRun.visits then
                    local lastVisit = activeRun.visits[#activeRun.visits]
                    if lastVisit and not lastVisit.leaveTime then
                        lastVisit.leaveTime = time()
                    end
                end
                -- Finalize non-raid runs (raids persist across visits)
                if activeRun.contentType ~= "raid" then
                    FinalizeRun(activeRun)
                else
                    -- Raid: just clear tracking, record stays in matches
                    activeRun = nil
                    activeRunIndex = nil
                    runStartTime = nil
                    encounterStartTimes = {}
                end
            end
            return
        end

        -- We're in an instance
        local instanceName, instType, difficultyID, difficultyName, maxPlayers,
              _, _, instanceID = GetInstanceInfo()

        -- Skip PvP instances
        if instType == "pvp" or instType == "arena" then return end

        -- Skip if we already have an active run for this instance
        if activeRun then return end

        local now = time()

        if instType == "party" then
            local mplusMapID = C_ChallengeMode.GetActiveChallengeMapID()

            -- Check for pending run from reload (preserves pre-reload boss data)
            local pending = SKToolsPvEDB and SKToolsPvEDB.pendingRun
            if pending then
                local currentPlayer = GetPlayerFullName()
                if pending.player == currentPlayer then
                    if mplusMapID and pending.contentType == "mythicplus" then
                        -- Restore M+ run from reload
                        activeRun = pending
                        activeRunIndex = nil
                        runStartTime = GetTime()
                        encounterStartTimes = {}
                        mplusDeathCount = pending.deaths or 0
                        local dok, dcount = pcall(C_ChallengeMode.GetDeathCount)
                        if dok and dcount then activeRun.deaths = dcount; mplusDeathCount = dcount end
                        ClearPendingRun()
                        return
                    elseif not mplusMapID and pending.contentType == "dungeon" then
                        -- Restore dungeon run from reload
                        activeRun = pending
                        activeRun.duration = SKToolsPvEDB.pendingRunElapsed or 0
                        activeRunIndex = nil
                        runStartTime = GetTime()
                        encounterStartTimes = {}
                        ClearPendingRun()
                        return
                    end
                end
                ClearPendingRun()
            end

            -- No pending run restored
            if mplusMapID then
                -- M+ key is active but no pending run (disconnect/crash recovery)
                -- Create provisional run so CHALLENGE_MODE_COMPLETED can fill it in
                local name = C_ChallengeMode.GetMapUIInfo(mplusMapID)
                local level, affixIDs = C_ChallengeMode.GetActiveKeystoneInfo()
                activeRun = {
                    timestamp = now,
                    date = date("%m/%d", now),
                    timeStr = date("%H:%M", now),
                    player = GetPlayerFullName(),
                    contentType = "mythicplus",
                    instanceName = name or "Unknown",
                    difficultyID = 8,
                    difficultyName = "Mythic Keystone",
                    completed = false,
                    bosses = {},
                    keyLevel = level or 0,
                    affixIDs = affixIDs or {},
                    deaths = 0,
                    mapChallengeModeID = mplusMapID,
                }
                activeRunIndex = nil
                runStartTime = GetTime()
                mplusDeathCount = 0
                encounterStartTimes = {}
                -- Grab current death count
                local dok, dcount = pcall(C_ChallengeMode.GetDeathCount)
                if dok and dcount then activeRun.deaths = dcount; mplusDeathCount = dcount end
                return
            end

            -- Non-M+ dungeon
            activeRun = {
                timestamp = now,
                date = date("%m/%d", now),
                timeStr = date("%H:%M", now),
                player = GetPlayerFullName(),
                contentType = "dungeon",
                instanceName = instanceName or "Unknown",
                difficultyID = difficultyID,
                difficultyName = difficultyName or "Unknown",
                completed = false,
                bosses = {},
                duration = 0,
                party = nil,  -- set on finalize from last boss kill group
            }
            activeRunIndex = nil
            runStartTime = GetTime()
            encounterStartTimes = {}

        elseif instType == "raid" then
            -- Create provisional raid record
            activeRun = {
                timestamp = now,
                date = date("%m/%d", now),
                timeStr = date("%H:%M", now),
                player = GetPlayerFullName(),
                contentType = "raid",
                instanceName = instanceName or "Unknown",
                difficultyID = difficultyID,
                difficultyName = difficultyName or "Unknown",
                completed = false,
                bosses = {},
                groupSize = maxPlayers or 0,
                totalEncounters = 0,
                visits = { { enterTime = now } },
                lockoutId = nil,
            }
            activeRunIndex = nil
            runStartTime = GetTime()
            encounterStartTimes = {}

            -- Request lockout info — will be handled by UPDATE_INSTANCE_INFO
            pendingRaidLookup = true
            RequestRaidInfo()
        end
    end
end

dataFrame:SetScript("OnEvent", function(self, event, ...)
    local ok, err = pcall(PvEDataEventHandler, event, ...)
    if not ok then ns.SK_ReportError("PvE:" .. event, err) end
end)

-----------------------------
-- Row Update
-----------------------------
function SKPvEHistory_UpdateRows(offset)
    if not rows then return end
    local filtered = GetFilteredMatches()

    for i = 1, VISIBLE_ROWS do
        local row = rows[i]
        local m = filtered[offset + i]

        if m then
            row:Show()

            -- Result
            if m.contentType == "mythicplus" then
                if m.onTime then
                    local upgradeStr = ""
                    if (m.keystoneUpgradeLevels or 0) > 0 then
                        upgradeStr = " +" .. m.keystoneUpgradeLevels
                    end
                    row.result:SetText("|cff00ff00Timed" .. upgradeStr .. "|r")
                else
                    row.result:SetText("|cffff4444Depleted|r")
                end
            elseif m.contentType == "raid" then
                local killed, total = GetRunBossProgress(m)
                if m.completed then
                    row.result:SetText("|cff00ff00Complete|r")
                elseif killed > 0 then
                    row.result:SetText("|cffffff00" .. killed .. "/" .. total .. "|r")
                else
                    row.result:SetText("|cff888888--" .. "|r")
                end
            else
                if m.completed then
                    row.result:SetText("|cff00ff00Complete|r")
                else
                    local killed, total = GetRunBossProgress(m)
                    if killed > 0 then
                        row.result:SetText("|cffffff00" .. killed .. "/" .. total .. "|r")
                    else
                        row.result:SetText("|cffff4444Incomplete|r")
                    end
                end
            end

            -- Type
            row.contentType:SetText("|cff888888" .. (CONTENT_TYPE_LABELS[m.contentType] or "?") .. "|r")

            -- Instance
            row.instanceName:SetText(m.instanceName or "")

            -- Level/Difficulty
            if m.contentType == "mythicplus" then
                row.level:SetText("|cffFFD100+" .. (m.keyLevel or "?") .. "|r")
            else
                row.level:SetText(DIFFICULTY_SHORT[m.difficultyID] or m.difficultyName or "?")
            end

            -- Time
            local dur = m.duration
            if m.contentType == "raid" and m.visits then
                -- Calculate total raid time from visits
                dur = 0
                for _, v in ipairs(m.visits) do
                    if v.enterTime and v.leaveTime then
                        dur = dur + (v.leaveTime - v.enterTime)
                    elseif v.enterTime then
                        dur = dur + (time() - v.enterTime)  -- ongoing visit
                    end
                end
            end
            row.duration:SetText(FormatDuration(dur))

            -- Deaths (M+ only)
            if m.contentType == "mythicplus" and m.deaths then
                row.deaths:SetText(m.deaths > 0 and ("|cffff4444" .. m.deaths .. "|r") or "0")
            else
                row.deaths:SetText("|cff888888--|r")
            end

            -- Rating (M+ only)
            if m.contentType == "mythicplus" and m.ratingChange then
                if m.ratingChange > 0 then
                    row.rating:SetText("|cff00ff00+" .. m.ratingChange .. "|r")
                elseif m.ratingChange < 0 then
                    row.rating:SetText("|cffff4444" .. m.ratingChange .. "|r")
                else
                    row.rating:SetText("|cff888888+0|r")
                end
            else
                row.rating:SetText("|cff888888--|r")
            end

            -- Party / Group
            local party = m.party
            if party and #party > 0 and #party <= 5 then
                row.party:SetText(CompIconString(party, 20))
            elseif m.contentType == "raid" then
                -- Show player count from most recent boss kill group
                local rosterSize = 0
                if m.bosses then
                    for bi = #m.bosses, 1, -1 do
                        if m.bosses[bi].success and m.bosses[bi].group then
                            rosterSize = #m.bosses[bi].group
                            break
                        end
                    end
                end
                if rosterSize > 0 then
                    row.party:SetText("|cff888888" .. rosterSize .. " players|r")
                else
                    row.party:SetText("")
                end
            else
                row.party:SetText("")
            end

            -- Date (show year if not current year, match PvP history styling)
            local dateStr = m.date or ""
            local timeStr = m.timeStr or ""
            if m.timestamp then
                local matchYear = date("%Y", m.timestamp)
                local currentYear = date("%Y")
                if matchYear ~= currentYear then
                    dateStr = date("%m/%d/%y", m.timestamp)
                end
            end
            row.dateText:SetText("|cffDDDDDD" .. dateStr .. "|r  |cff888888" .. timeStr .. "|r")

            -- Store match data for tooltip
            row.matchData = m
        else
            row:Hide()
            row.matchData = nil
        end
    end
end

-----------------------------
-- Refresh
-----------------------------
function SKPvEHistory_RefreshUI()
    if not historyFrame or not historyFrame:IsShown() then return end

    local filtered = GetFilteredMatches()

    -- Update scrollbar
    local slider = historyFrame.slider
    local maxScroll = math.max(0, #filtered - VISIBLE_ROWS)
    slider:SetMinMaxValues(0, maxScroll)
    if slider:GetValue() > maxScroll then
        slider:SetValue(maxScroll)
    end

    SKPvEHistory_UpdateRows(math.floor(slider:GetValue()))

    -- Update summary
    local matches = SKToolsPvEDB and SKToolsPvEDB.matches or {}
    local totalMplus, totalRaidKills, totalDungeons = 0, 0, 0
    for _, m in ipairs(matches) do
        if m.contentType == "mythicplus" then totalMplus = totalMplus + 1
        elseif m.contentType == "raid" then totalRaidKills = totalRaidKills + 1
        elseif m.contentType == "dungeon" then totalDungeons = totalDungeons + 1
        end
    end

    local sessionStr = ""
    if sessionMplus.completed > 0 then
        sessionStr = string.format("Session: %d M+ (%d timed)", sessionMplus.completed, sessionMplus.timed)
        if sessionMplus.ratingChange ~= 0 then
            local sign = sessionMplus.ratingChange >= 0 and "+" or ""
            local col = sessionMplus.ratingChange >= 0 and "|cff00ff00" or "|cffff4444"
            sessionStr = sessionStr .. " " .. col .. sign .. sessionMplus.ratingChange .. "|r"
        end
    end
    if sessionRaid.kills > 0 or sessionRaid.wipes > 0 then
        if sessionStr ~= "" then sessionStr = sessionStr .. "  |  " end
        sessionStr = sessionStr .. "|cff00ff00" .. sessionRaid.kills .. " kills|r |cffff4444" .. sessionRaid.wipes .. " wipes|r"
    end

    local overallStr = string.format("Overall: %d M+ | %d raids | %d dungeons", totalMplus, totalRaidKills, totalDungeons)
    if sessionStr ~= "" then
        historyFrame.summary:SetText(sessionStr .. "    " .. overallStr)
    else
        historyFrame.summary:SetText(overallStr)
    end
end

-----------------------------
-- UI Creation
-----------------------------
local function CreateHistoryFrame()
    if historyFrame then return end

    local f = CreateFrame("Frame", "SKToolsPvEHistoryFrame", UIParent, "BackdropTemplate")
    f:SetSize(1240, 34 + 30 + 26 + 28 + HEADER_HEIGHT + (ROW_HEIGHT * VISIBLE_ROWS) + 16)
    f:SetPoint("CENTER")
    f:SetMovable(true)
    f:EnableMouse(true)
    f:SetClampedToScreen(true)
    f:SetFrameStrata("DIALOG")
    f:SetBackdrop({
        bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
        edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
        tile = true, tileSize = 16, edgeSize = 16,
        insets = { left = 4, right = 4, top = 4, bottom = 4 },
    })
    f:SetBackdropColor(0.05, 0.05, 0.08, 0.95)
    f:SetBackdropBorderColor(0.3, 0.3, 0.35, 1)
    f:Hide()
    f:SetToplevel(true)
    f:SetScript("OnShow", function(self) self:Raise() end)

    -- Green accent line at top (PvE = green)
    local accentLine = f:CreateTexture(nil, "ARTWORK")
    accentLine:SetColorTexture(GREEN.r, GREEN.g, GREEN.b, 0.7)
    accentLine:SetPoint("TOPLEFT", f, "TOPLEFT", 5, -4)
    accentLine:SetPoint("TOPRIGHT", f, "TOPRIGHT", -5, -4)
    accentLine:SetHeight(2)

    -- ESC to close (pcall to avoid taint in instances)
    f:SetScript("OnKeyDown", function(self, key)
        if key == "ESCAPE" then pcall(self.SetPropagateKeyboardInput, self, false); self:Hide()
        else pcall(self.SetPropagateKeyboardInput, self, true) end
    end)
    pcall(f.EnableKeyboard, f, true)

    -- Title bar drag
    local titleBar = CreateFrame("Frame", nil, f)
    titleBar:SetPoint("TOPLEFT", 4, -6)
    titleBar:SetPoint("TOPRIGHT", -4, -6)
    titleBar:SetHeight(28)
    titleBar:EnableMouse(true)
    titleBar:RegisterForDrag("LeftButton")
    titleBar:SetScript("OnDragStart", function() f:StartMoving() end)
    titleBar:SetScript("OnDragStop", function()
        f:StopMovingOrSizing()
        local point, _, relPoint, x, y = f:GetPoint()
        SKToolsPvEDB.framePos = { point = point, relPoint = relPoint, x = x, y = y }
    end)

    local titleBg = titleBar:CreateTexture(nil, "BACKGROUND")
    titleBg:SetAllPoints()
    titleBg:SetColorTexture(0.08, 0.12, 0.14, 0.8)

    local title = titleBar:CreateFontString(nil, "OVERLAY", "GameFontNormalLarge")
    title:SetPoint("LEFT", 10, 0)
    title:SetText("|cff00E5EESK|r|cff00cc66PvE|r History")

    local closeBtn = CreateFrame("Button", nil, titleBar, "UIPanelCloseButton")
    closeBtn:SetPoint("TOPRIGHT", f, "TOPRIGHT", -2, -2)
    closeBtn:SetScript("OnClick", function() f:Hide() end)

    local titleSep = f:CreateTexture(nil, "ARTWORK")
    titleSep:SetPoint("TOPLEFT", titleBar, "BOTTOMLEFT", 0, 0)
    titleSep:SetPoint("TOPRIGHT", titleBar, "BOTTOMRIGHT", 0, 0)
    titleSep:SetHeight(1)
    titleSep:SetColorTexture(GREEN.r, GREEN.g, GREEN.b, 0.3)

    -- Summary bar
    local summary = f:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
    summary:SetPoint("TOPLEFT", titleBar, "BOTTOMLEFT", 10, -6)
    summary:SetJustifyH("LEFT")
    f.summary = summary

    -- Filter buttons
    local UpdateViewButtonVisibility  -- forward declaration
    local filterY = -64
    local filters = { "All", "M+", "Raid", "Dungeon" }
    filterButtons = {}
    local prevBtn
    for i, name in ipairs(filters) do
        local btnW = math.max(52, 10 + #name * 7)
        local btn = ns.CreateThemedButton(f, name, btnW, 22, "secondary")
        btn.filterName = name
        btn.text = btn.label

        if i == 1 then
            btn:SetPoint("TOPLEFT", f, "TOPLEFT", 12, filterY)
        else
            btn:SetPoint("LEFT", prevBtn, "RIGHT", 4, 0)
        end

        btn:SetScript("OnClick", function(self)
            activeFilter = self.filterName
            for _, b in ipairs(filterButtons) do
                local isActive = (b.filterName == activeFilter)
                if isActive then
                    b:SetBackdropColor(CYAN.r * 0.3, CYAN.g * 0.3, CYAN.b * 0.3, 1)
                    b:SetBackdropBorderColor(CYAN.r, CYAN.g, CYAN.b, 0.8)
                    b.text:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
                else
                    b:SetBackdropColor(0.1, 0.1, 0.12, 0.8)
                    b:SetBackdropBorderColor(0.3, 0.3, 0.35, 0.6)
                    b.text:SetTextColor(0.7, 0.7, 0.7)
                end
            end
            -- Graph only available for M+; switch back to matches if leaving M+
            if activeFilter ~= "M+" and activeView == "graph" then
                ShowView("matches")
            end
            UpdateViewButtonVisibility()
            if activeView == "graph" then
                RefreshGraphPanel()
            end
            SKPvEHistory_RefreshUI()
        end)

        btn:SetScript("OnEnter", function(self)
            if self.filterName ~= activeFilter then
                self:SetBackdropColor(0.15, 0.15, 0.18, 0.9)
                self.text:SetTextColor(1, 1, 1)
            end
        end)
        btn:SetScript("OnLeave", function(self)
            if self.filterName ~= activeFilter then
                self:SetBackdropColor(0.1, 0.1, 0.12, 0.8)
                self.text:SetTextColor(0.7, 0.7, 0.7)
            end
        end)

        filterButtons[#filterButtons + 1] = btn
        prevBtn = btn
    end
    -- Highlight default filter
    filterButtons[1]:SetBackdropColor(CYAN.r * 0.3, CYAN.g * 0.3, CYAN.b * 0.3, 1)
    filterButtons[1]:SetBackdropBorderColor(CYAN.r, CYAN.g, CYAN.b, 0.8)
    filterButtons[1].text:SetTextColor(CYAN.r, CYAN.g, CYAN.b)

    -- View buttons
    local viewY = filterY - 26
    local views = {
        { key = "matches", label = "Matches" },
        { key = "graph",   label = "Graph" },
    }
    viewButtons = {}
    local prevViewBtn
    for i, v in ipairs(views) do
        local btnW = math.max(52, 10 + #v.label * 7)
        local btn = ns.CreateThemedButton(f, v.label, btnW, 22, "secondary")
        btn.text = btn.label
        btn.viewKey = v.key

        if i == 1 then
            btn:SetPoint("TOPLEFT", f, "TOPLEFT", 12, viewY)
        else
            btn:SetPoint("LEFT", prevViewBtn, "RIGHT", 4, 0)
        end

        btn:SetScript("OnClick", function() ShowView(v.key) end)
        btn:SetScript("OnEnter", function(self)
            if activeView ~= self.viewKey then
                self:SetBackdropColor(0.15, 0.15, 0.18, 0.9)
                self.text:SetTextColor(1, 1, 1)
            end
        end)
        btn:SetScript("OnLeave", function(self)
            if activeView ~= self.viewKey then
                self:SetBackdropColor(0.1, 0.1, 0.12, 0.8)
                self.text:SetTextColor(0.7, 0.7, 0.7)
            end
        end)

        viewButtons[#viewButtons + 1] = btn
        prevViewBtn = btn
    end
    viewButtons[1]:SetBackdropColor(CYAN.r * 0.3, CYAN.g * 0.3, CYAN.b * 0.3, 1)
    viewButtons[1]:SetBackdropBorderColor(CYAN.r, CYAN.g, CYAN.b, 0.8)
    viewButtons[1].text:SetTextColor(CYAN.r, CYAN.g, CYAN.b)

    -- Graph button only visible when M+ filter is active
    UpdateViewButtonVisibility = function()
        for _, btn in ipairs(viewButtons) do
            if btn.viewKey == "graph" then
                btn:SetShown(activeFilter == "M+")
            end
        end
    end
    UpdateViewButtonVisibility()

    -- Character filter dropdown
    local charBtn = CreateFrame("Button", nil, f, "BackdropTemplate")
    charBtn:SetSize(170, 22)
    charBtn:SetBackdrop({
        bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
        edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
        tile = true, tileSize = 8, edgeSize = 10,
        insets = { left = 2, right = 2, top = 2, bottom = 2 },
    })
    charBtn:SetBackdropColor(0.1, 0.1, 0.12, 0.8)
    charBtn:SetBackdropBorderColor(0.3, 0.3, 0.35, 0.6)
    charBtn:SetPoint("RIGHT", f, "TOPRIGHT", -14, viewY + 11)

    local charBtnText = charBtn:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
    charBtnText:SetPoint("LEFT", charBtn, "LEFT", 6, 0)
    charBtnText:SetPoint("RIGHT", charBtn, "RIGHT", -14, 0)
    charBtnText:SetJustifyH("LEFT")
    charBtnText:SetText("All Characters")
    charBtnText:SetTextColor(0.7, 0.7, 0.7)
    charBtn.text = charBtnText

    local charArrow = charBtn:CreateTexture(nil, "OVERLAY")
    charArrow:SetSize(10, 10)
    charArrow:SetPoint("RIGHT", charBtn, "RIGHT", -4, 0)
    charArrow:SetAtlas("arrow-down-minor")
    charArrow:SetVertexColor(0.6, 0.6, 0.6)

    charBtn:SetScript("OnEnter", function(self)
        self:SetBackdropColor(0.15, 0.15, 0.18, 0.9)
        charBtnText:SetTextColor(1, 1, 1)
    end)
    charBtn:SetScript("OnLeave", function(self)
        if not activeCharFilter then
            self:SetBackdropColor(0.1, 0.1, 0.12, 0.8)
            charBtnText:SetTextColor(0.7, 0.7, 0.7)
        else
            self:SetBackdropColor(CYAN.r * 0.3, CYAN.g * 0.3, CYAN.b * 0.3, 1)
            charBtnText:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
        end
    end)

    local function UpdateCharButton()
        if activeCharFilter then
            local shortName = activeCharFilter:match("^([^%-]+)") or activeCharFilter
            charBtnText:SetText(shortName)
            charBtn:SetBackdropColor(CYAN.r * 0.3, CYAN.g * 0.3, CYAN.b * 0.3, 1)
            charBtn:SetBackdropBorderColor(CYAN.r, CYAN.g, CYAN.b, 0.8)
            charBtnText:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
        else
            charBtnText:SetText("All Characters")
            charBtn:SetBackdropColor(0.1, 0.1, 0.12, 0.8)
            charBtn:SetBackdropBorderColor(0.3, 0.3, 0.35, 0.6)
            charBtnText:SetTextColor(0.7, 0.7, 0.7)
        end
    end

    local charMenu = CreateFrame("Frame", "SKToolsPvECharFilterMenu", f, "BackdropTemplate")
    charMenu:SetBackdrop({
        bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
        edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
        tile = true, tileSize = 8, edgeSize = 10,
        insets = { left = 2, right = 2, top = 2, bottom = 2 },
    })
    charMenu:SetBackdropColor(0.08, 0.08, 0.1, 0.95)
    charMenu:SetBackdropBorderColor(0.3, 0.3, 0.35, 0.8)
    charMenu:SetFrameStrata("DIALOG")
    charMenu:Hide()

    local charMenuItems = {}

    local function RefreshCharMenu()
        local chars, seen = {}, {}
        local matches = SKToolsPvEDB and SKToolsPvEDB.matches or {}
        for _, m in ipairs(matches) do
            if m.player and not seen[m.player] then
                seen[m.player] = true
                chars[#chars + 1] = m.player
            end
        end
        table.sort(chars)

        for _, item in ipairs(charMenuItems) do item:Hide() end
        charMenuItems = {}

        local allItems = { { label = "All Characters", value = nil } }
        for _, c in ipairs(chars) do
            allItems[#allItems + 1] = { label = c, value = c }
        end

        local itemHeight = 20
        charMenu:SetSize(charBtn:GetWidth(), 4 + #allItems * itemHeight + 4)
        charMenu:SetPoint("TOP", charBtn, "BOTTOM", 0, -2)

        for i, opt in ipairs(allItems) do
            local item = CreateFrame("Button", nil, charMenu)
            item:SetSize(charBtn:GetWidth() - 8, itemHeight)
            item:SetPoint("TOPLEFT", charMenu, "TOPLEFT", 4, -4 - (i - 1) * itemHeight)
            local itemText = item:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
            itemText:SetPoint("LEFT", 4, 0)
            local shortLabel = opt.label
            if opt.value then shortLabel = opt.value:match("^([^%-]+)") or opt.value end
            itemText:SetText(shortLabel)
            if opt.value == activeCharFilter then
                itemText:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
            else
                itemText:SetTextColor(0.8, 0.8, 0.8)
            end
            item:SetScript("OnEnter", function() if opt.value ~= activeCharFilter then itemText:SetTextColor(1, 1, 1) end end)
            item:SetScript("OnLeave", function() if opt.value ~= activeCharFilter then itemText:SetTextColor(0.8, 0.8, 0.8) end end)
            item:SetScript("OnClick", function()
                charMenu:Hide()
                activeCharFilter = opt.value
                UpdateCharButton()
                if activeView == "graph" then RefreshGraphPanel() end
                SKPvEHistory_RefreshUI()
            end)
            charMenuItems[#charMenuItems + 1] = item
        end
    end

    charBtn:SetScript("OnClick", function()
        if charMenu:IsShown() then charMenu:Hide() else RefreshCharMenu(); charMenu:Show() end
    end)
    charMenu:SetScript("OnShow", function()
        charMenu:SetScript("OnUpdate", function(self)
            local ok, err = pcall(function()
                if not charBtn:IsMouseOver() and not self:IsMouseOver() then
                    local elapsed = (self.hideTimer or 0) + 0.016
                    self.hideTimer = elapsed
                    if elapsed > 0.3 then self:Hide(); self.hideTimer = 0 end
                else
                    self.hideTimer = 0
                end
            end)
            if not ok then self:Hide() end
        end)
    end)
    charMenu:SetScript("OnHide", function(self) self:SetScript("OnUpdate", nil); self.hideTimer = 0 end)

    -- Clear button
    local clearBtn = ns.CreateThemedButton(f, "Clear", 52, 22, "danger")
    clearBtn:SetPoint("RIGHT", f, "TOPRIGHT", -14, filterY + 11)

    StaticPopupDialogs["SKTOOLS_CLEAR_PVE_HISTORY"] = {
        text = "Clear all PvE history?",
        button1 = "Yes",
        button2 = "No",
        OnAccept = function()
            if SKToolsPvEDB then SKToolsPvEDB.matches = {} end
            sessionMplus = { completed = 0, timed = 0, depleted = 0, ratingChange = 0 }
            sessionRaid = { kills = 0, wipes = 0 }
            sessionDungeon = { completed = 0 }
            SKPvEHistory_RefreshUI()
            print("|cff00E5EESKTools:|r PvE history cleared.")
        end,
        timeout = 0, whileDead = true, hideOnEscape = true,
    }
    clearBtn:SetScript("OnClick", function() StaticPopup_Show("SKTOOLS_CLEAR_PVE_HISTORY") end)

    -- Reset Session button
    local resetBtn = ns.CreateThemedButton(f, "Reset", 60, 22, "secondary")
    resetBtn:SetPoint("RIGHT", clearBtn, "LEFT", -6, 0)

    resetBtn:HookScript("OnEnter", function(self)
        GameTooltip:SetOwner(self, "ANCHOR_TOP")
        GameTooltip:AddLine("Reset Session Stats")
        GameTooltip:AddLine("Resets session counters without clearing history.", 1, 1, 1, true)
        GameTooltip:Show()
    end)
    resetBtn:HookScript("OnLeave", function()
        GameTooltip:Hide()
    end)

    StaticPopupDialogs["SKTOOLS_RESET_PVE_SESSION"] = {
        text = "Reset PvE session stats?",
        button1 = "Reset",
        button2 = "Cancel",
        OnAccept = function()
            sessionMplus = { completed = 0, timed = 0, depleted = 0, ratingChange = 0 }
            sessionRaid = { kills = 0, wipes = 0 }
            sessionDungeon = { completed = 0 }
            SKPvEHistory_RefreshUI()
            print("|cff00E5EESKTools:|r PvE session stats reset.")
        end,
        timeout = 0, whileDead = true, hideOnEscape = true,
    }
    resetBtn:SetScript("OnClick", function() StaticPopup_Show("SKTOOLS_RESET_PVE_SESSION") end)

    -- Column headers
    local contentY = viewY - 26
    local headerFrame = CreateFrame("Frame", nil, f, "BackdropTemplate")
    headerFrame:SetPoint("TOPLEFT", f, "TOPLEFT", 4, contentY)
    headerFrame:SetPoint("TOPRIGHT", f, "TOPRIGHT", -4, contentY)
    headerFrame:SetHeight(HEADER_HEIGHT)
    headerFrame:SetBackdrop({
        bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
        edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
        tile = true, tileSize = 8, edgeSize = 10,
        insets = { left = 2, right = 2, top = 2, bottom = 2 },
    })
    headerFrame:SetBackdropColor(0.08, 0.1, 0.14, 0.9)
    headerFrame:SetBackdropBorderColor(0.25, 0.25, 0.3, 0.6)
    f.headerFrame = headerFrame

    local colDefs = {
        { text = "Result",   width = 82,  x = 12,  sortKey = "success" },
        { text = "Type",     width = 68,  x = 98,  sortKey = "contentType" },
        { text = "Instance", width = 280, x = 172, sortKey = "instanceName" },
        { text = "Level",    width = 55,  x = 458, sortKey = "keyLevel" },
        { text = "Time",     width = 85,  x = 520, sortKey = "duration" },
        { text = "Deaths",   width = 55,  x = 612, sortKey = "deaths" },
        { text = "Rating",   width = 80,  x = 674, sortKey = "ratingChange" },
        { text = "Party",    width = 220, x = 760 },
        { text = "Date",     width = 140, x = 990, sortKey = "timestamp" },
    }
    f.colDefs = colDefs
    f.headerButtons = {}

    for _, col in ipairs(colDefs) do
        if col.sortKey then
            local btn = CreateFrame("Button", nil, headerFrame)
            btn:SetPoint("TOPLEFT", headerFrame, "TOPLEFT", col.x, 0)
            btn:SetSize(col.width, HEADER_HEIGHT)

            btn.label = btn:CreateFontString(nil, "OVERLAY")
            btn.label:SetFont(STANDARD_TEXT_FONT, 12, "OUTLINE")
            btn.label:SetPoint("LEFT", 0, 0)
            btn.label:SetWidth(col.width)
            btn.label:SetJustifyH("LEFT")
            btn.label:SetTextColor(CYAN.r, CYAN.g, CYAN.b)

            btn.arrow = btn:CreateTexture(nil, "OVERLAY")
            btn.arrow:SetSize(10, 10)
            btn.arrow:SetTexture("Interface\\Buttons\\UI-SortArrow")
            btn.arrow:Hide()

            btn.sortKey = col.sortKey
            btn.colText = col.text

            local function UpdateArrow(hb, show, ascending)
                if show then
                    hb.arrow:ClearAllPoints()
                    hb.arrow:SetPoint("LEFT", hb.label, "LEFT", hb.label:GetStringWidth() + 1, 0)
                    if ascending then
                        hb.arrow:SetTexCoord(0, 1, 0, 1)
                    else
                        hb.arrow:SetTexCoord(0, 1, 1, 0)
                    end
                    hb.arrow:SetVertexColor(1, 1, 1, 0.8)
                    hb.arrow:Show()
                else
                    hb.arrow:Hide()
                end
            end

            local function RefreshHeaderLabels()
                for _, hb in ipairs(f.headerButtons) do
                    if hb.sortKey == sortColumn then
                        hb.label:SetText(hb.colText)
                        hb.label:SetTextColor(1, 1, 1)
                        UpdateArrow(hb, true, sortAscending)
                    else
                        hb.label:SetText(hb.colText)
                        hb.label:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
                        UpdateArrow(hb, false)
                    end
                end
            end

            btn:SetScript("OnClick", function(self)
                if sortColumn == self.sortKey then
                    sortAscending = not sortAscending
                else
                    sortColumn = self.sortKey
                    sortAscending = (self.sortKey ~= "timestamp")
                end
                RefreshHeaderLabels()
                SKPvEHistory_RefreshUI()
            end)

            btn:SetScript("OnEnter", function(self)
                if self.sortKey ~= sortColumn then self.label:SetTextColor(1, 1, 1) end
            end)
            btn:SetScript("OnLeave", function(self)
                if self.sortKey ~= sortColumn then self.label:SetTextColor(CYAN.r, CYAN.g, CYAN.b) end
            end)

            if col.sortKey == sortColumn then
                btn.label:SetText(col.text)
                btn.label:SetTextColor(1, 1, 1)
                UpdateArrow(btn, true, sortAscending)
            else
                btn.label:SetText(col.text)
            end

            f.headerButtons[#f.headerButtons + 1] = btn
        else
            local header = headerFrame:CreateFontString(nil, "OVERLAY")
            header:SetFont(STANDARD_TEXT_FONT, 12, "OUTLINE")
            header:SetPoint("TOPLEFT", headerFrame, "TOPLEFT", col.x, -7)
            header:SetWidth(col.width)
            header:SetJustifyH("LEFT")
            header:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
            header:SetText(col.text)
        end
    end

    -- Scroll area
    local scrollArea = CreateFrame("Frame", nil, f)
    scrollArea:SetPoint("TOPLEFT", headerFrame, "BOTTOMLEFT", 0, 0)
    scrollArea:SetPoint("BOTTOMRIGHT", f, "BOTTOMRIGHT", -18, 6)

    -- Row frames
    rows = {}
    for i = 1, VISIBLE_ROWS do
        local row = CreateFrame("Frame", nil, scrollArea, "BackdropTemplate")
        row:SetHeight(ROW_HEIGHT)
        row:SetPoint("TOPLEFT", scrollArea, "TOPLEFT", 0, -((i - 1) * ROW_HEIGHT))
        row:SetPoint("RIGHT", scrollArea, "RIGHT", 0, 0)
        row:SetBackdrop({ bgFile = "Interface\\Tooltips\\UI-Tooltip-Background" })
        if i % 2 == 0 then
            row:SetBackdropColor(0.1, 0.1, 0.14, 0.5)
        else
            row:SetBackdropColor(0.06, 0.06, 0.09, 0.3)
        end
        row.evenRow = (i % 2 == 0)

        row.result = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
        row.result:SetPoint("LEFT", 12, 0)
        row.result:SetWidth(82)
        row.result:SetJustifyH("LEFT")

        row.contentType = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
        row.contentType:SetPoint("LEFT", 98, 0)
        row.contentType:SetWidth(68)
        row.contentType:SetJustifyH("LEFT")

        row.instanceName = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
        row.instanceName:SetPoint("LEFT", 172, 0)
        row.instanceName:SetWidth(280)
        row.instanceName:SetJustifyH("LEFT")

        row.level = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
        row.level:SetPoint("LEFT", 458, 0)
        row.level:SetWidth(55)
        row.level:SetJustifyH("LEFT")

        row.duration = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
        row.duration:SetPoint("LEFT", 520, 0)
        row.duration:SetWidth(85)
        row.duration:SetJustifyH("LEFT")

        row.deaths = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
        row.deaths:SetPoint("LEFT", 612, 0)
        row.deaths:SetWidth(55)
        row.deaths:SetJustifyH("LEFT")

        row.rating = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
        row.rating:SetPoint("LEFT", 674, 0)
        row.rating:SetWidth(80)
        row.rating:SetJustifyH("LEFT")

        row.party = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
        row.party:SetPoint("LEFT", 760, 0)
        row.party:SetWidth(220)
        row.party:SetJustifyH("LEFT")

        row.dateText = row:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
        row.dateText:SetPoint("LEFT", 990, 0)
        row.dateText:SetWidth(140)
        row.dateText:SetJustifyH("LEFT")

        -- Tooltip
        row:EnableMouse(true)
        row:SetScript("OnEnter", function(self)
            self:SetBackdropColor(CYAN.r * 0.12, CYAN.g * 0.12, CYAN.b * 0.12, 0.6)
            local m = self.matchData
            if not m then return end

            -- Anchor tooltip at cursor position
            GameTooltip:SetOwner(UIParent, "ANCHOR_NONE")
            local cx, cy = GetCursorPosition()
            local scale = GameTooltip:GetEffectiveScale()
            GameTooltip:SetPoint("BOTTOMLEFT", UIParent, "BOTTOMLEFT", cx / scale + 12, cy / scale + 8)

            -- Helper: format a player line with race + class icons and role
            local ICON_SZ = 16
            local function AddPlayerLine(p)
                local hex = GetClassColorHex(p.class)
                local raceI = RaceIconString(p.raceFile, p.race, p.sex, ICON_SZ)
                local classI = ClassIconString(p.class, ICON_SZ)
                local icons = raceI ~= "" and (raceI .. " " .. classI) or ("  " .. classI)
                local short = p.name and (p.name:match("^([^%-]+)") or p.name) or "Unknown"
                local roleStr = ""
                if p.role == "TANK" then roleStr = " |cff888888(Tank)|r"
                elseif p.role == "HEALER" then roleStr = " |cff888888(Healer)|r"
                end
                GameTooltip:AddLine("  " .. icons .. " |cff" .. hex .. short .. "|r" .. roleStr)
            end

            if m.contentType == "mythicplus" then
                GameTooltip:AddLine("|cff00E5EE+" .. (m.keyLevel or "?") .. " " .. (m.instanceName or "") .. "|r")
                local resultStr = m.onTime and "|cff00ff00Timed|r" or "|cffff4444Depleted|r"
                GameTooltip:AddLine(resultStr)
                if m.timeLimit and m.duration then
                    local diff = m.duration - m.timeLimit
                    local sign = diff > 0 and "+" or "-"
                    local col = diff > 0 and "|cffff4444" or "|cff00ff00"
                    GameTooltip:AddDoubleLine("Time", FormatDuration(m.duration) .. " / " .. FormatDuration(m.timeLimit) .. " (" .. col .. sign .. FormatDuration(math.abs(diff)) .. "|r)", 0.8, 0.8, 0.8, 1, 1, 1)
                end
                if m.deaths then
                    GameTooltip:AddDoubleLine("Deaths", m.deaths, 0.8, 0.8, 0.8, 1, 1, 1)
                end
                if m.ratingChange then
                    local sign = m.ratingChange >= 0 and "+" or ""
                    local col = m.ratingChange >= 0 and "|cff00ff00" or "|cffff4444"
                    GameTooltip:AddDoubleLine("Rating", col .. sign .. m.ratingChange .. "|r  " .. (m.newRating or "?"), 0.8, 0.8, 0.8, 1, 1, 1)
                end
                -- Affixes
                if m.affixIDs and #m.affixIDs > 0 then
                    local affixNames = {}
                    for _, id in ipairs(m.affixIDs) do
                        local name = C_ChallengeMode.GetAffixInfo(id)
                        if name then affixNames[#affixNames + 1] = name end
                    end
                    if #affixNames > 0 then
                        GameTooltip:AddLine("Affixes: " .. table.concat(affixNames, ", "), 0.7, 0.7, 0.7, true)
                    end
                end
                -- Boss timeline
                if m.bosses and #m.bosses > 0 then
                    GameTooltip:AddLine(" ")
                    GameTooltip:AddLine("|cff00E5EEBoss Timeline|r")
                    for _, b in ipairs(m.bosses) do
                        if b.success then
                            local elapsed = b.timestamp - m.timestamp
                            GameTooltip:AddLine("  " .. b.encounterName .. " |cff888888at " .. FormatDuration(elapsed) .. " (" .. FormatDuration(b.fightDuration) .. " fight)|r")
                        end
                    end
                end
                -- Party
                if m.party and #m.party > 0 then
                    GameTooltip:AddLine(" ")
                    GameTooltip:AddLine("|cff00E5EEParty|r")
                    for _, p in ipairs(m.party) do AddPlayerLine(p) end
                end

            elseif m.contentType == "raid" then
                GameTooltip:AddLine("|cff00E5EE" .. (m.instanceName or "") .. " " .. (m.difficultyName or "") .. "|r")
                local killed, total = GetRunBossProgress(m)
                GameTooltip:AddLine(killed .. "/" .. total .. " bosses", 0.8, 0.8, 0.8)
                -- Visits
                if m.visits and #m.visits > 0 then
                    GameTooltip:AddLine(" ")
                    local totalTime = 0
                    for vi, v in ipairs(m.visits) do
                        local vDur = (v.leaveTime or time()) - v.enterTime
                        totalTime = totalTime + vDur
                        local vDate = date("%m/%d %H:%M", v.enterTime)
                        GameTooltip:AddLine("  Session " .. vi .. ": " .. vDate .. " (" .. FormatDuration(vDur) .. ")", 0.7, 0.7, 0.7)
                    end
                    GameTooltip:AddLine("  Total: " .. FormatDuration(totalTime), 0.9, 0.9, 0.9)
                end
                -- Boss timeline
                if m.bosses and #m.bosses > 0 then
                    GameTooltip:AddLine(" ")
                    GameTooltip:AddLine("|cff00E5EEBoss Timeline|r")
                    -- Group by encounter for pull count
                    local encounterPulls = {}
                    for _, b in ipairs(m.bosses) do
                        if not encounterPulls[b.encounterID] then
                            encounterPulls[b.encounterID] = { kills = 0, wipes = 0 }
                        end
                        if b.success then
                            encounterPulls[b.encounterID].kills = encounterPulls[b.encounterID].kills + 1
                        else
                            encounterPulls[b.encounterID].wipes = encounterPulls[b.encounterID].wipes + 1
                        end
                    end
                    -- Show kills with pull count
                    local shown = {}
                    for _, b in ipairs(m.bosses) do
                        if b.success and not shown[b.encounterID] then
                            shown[b.encounterID] = true
                            local pulls = encounterPulls[b.encounterID]
                            local totalPulls = pulls.kills + pulls.wipes
                            local pullStr = totalPulls > 1 and (" |cffff4444" .. totalPulls .. " pulls|r") or ""
                            local timeStr = date("%H:%M", b.timestamp)
                            local groupSize = b.group and #b.group or 0
                            local groupStr = groupSize > 0 and (" |cff888888(" .. groupSize .. " players)|r") or ""
                            GameTooltip:AddLine("  |cff00ff00" .. b.encounterName .. "|r" .. pullStr .. " |cff888888at " .. timeStr .. " (" .. FormatDuration(b.fightDuration) .. ")|r" .. groupStr)
                        end
                    end
                    -- Show bosses that only have wipes
                    for _, b in ipairs(m.bosses) do
                        if not b.success and not shown[b.encounterID] then
                            shown[b.encounterID] = true
                            local pulls = encounterPulls[b.encounterID]
                            GameTooltip:AddLine("  |cffff4444" .. b.encounterName .. "|r |cff888888(" .. pulls.wipes .. " wipes)|r")
                        end
                    end
                end
                -- Raid roster from most recent boss kill
                local latestGroup
                if m.bosses then
                    for i = #m.bosses, 1, -1 do
                        if m.bosses[i].success and m.bosses[i].group and #m.bosses[i].group > 0 then
                            latestGroup = m.bosses[i].group
                            break
                        end
                    end
                end
                if latestGroup then
                    -- Sort by role: Tanks, Healers, DPS
                    local tanks, healers, dps = {}, {}, {}
                    for _, p in ipairs(latestGroup) do
                        if p.role == "TANK" then tanks[#tanks + 1] = p
                        elseif p.role == "HEALER" then healers[#healers + 1] = p
                        else dps[#dps + 1] = p end
                    end
                    GameTooltip:AddLine(" ")
                    GameTooltip:AddLine("|cff00E5EERaid Roster|r (" .. #latestGroup .. " players)")
                    if #tanks > 0 then
                        GameTooltip:AddLine("|cff888888Tanks:|r")
                        for _, p in ipairs(tanks) do AddPlayerLine(p) end
                    end
                    if #healers > 0 then
                        GameTooltip:AddLine("|cff888888Healers:|r")
                        for _, p in ipairs(healers) do AddPlayerLine(p) end
                    end
                    if #dps > 0 then
                        GameTooltip:AddLine("|cff888888DPS:|r")
                        for _, p in ipairs(dps) do AddPlayerLine(p) end
                    end
                end

            else -- dungeon
                GameTooltip:AddLine("|cff00E5EE" .. (m.instanceName or "") .. " " .. (m.difficultyName or "") .. "|r")
                local resultStr = m.completed and "|cff00ff00Complete|r" or "|cffff4444Incomplete|r"
                GameTooltip:AddLine(resultStr)
                if m.duration then
                    GameTooltip:AddDoubleLine("Duration", FormatDuration(m.duration), 0.8, 0.8, 0.8, 1, 1, 1)
                end
                -- Boss timeline
                if m.bosses and #m.bosses > 0 then
                    GameTooltip:AddLine(" ")
                    GameTooltip:AddLine("|cff00E5EEBoss Timeline|r")
                    for _, b in ipairs(m.bosses) do
                        if b.success then
                            local elapsed = b.timestamp - m.timestamp
                            GameTooltip:AddLine("  " .. b.encounterName .. " |cff888888at " .. FormatDuration(elapsed) .. " (" .. FormatDuration(b.fightDuration) .. " fight)|r")
                        end
                    end
                end
                -- Party
                if m.party and #m.party > 0 then
                    GameTooltip:AddLine(" ")
                    GameTooltip:AddLine("|cff00E5EEParty|r")
                    for _, p in ipairs(m.party) do AddPlayerLine(p) end
                end
            end

            GameTooltip:Show()
        end)
        row:SetScript("OnLeave", function(self)
            if self.evenRow then
                self:SetBackdropColor(0.1, 0.1, 0.14, 0.5)
            else
                self:SetBackdropColor(0.06, 0.06, 0.09, 0.3)
            end
            GameTooltip:Hide()
        end)

        rows[i] = row
    end

    -- Scrollbar
    local slider = CreateFrame("Slider", nil, f, "BackdropTemplate")
    slider:SetPoint("TOPRIGHT", scrollArea, "TOPRIGHT", 16, 0)
    slider:SetPoint("BOTTOMRIGHT", scrollArea, "BOTTOMRIGHT", 16, 0)
    slider:SetWidth(16)
    slider:SetOrientation("VERTICAL")
    slider:SetBackdrop({
        bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
        edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
        tile = true, tileSize = 8, edgeSize = 10,
        insets = { left = 2, right = 2, top = 2, bottom = 2 },
    })
    slider:SetBackdropColor(0.05, 0.05, 0.08, 0.7)
    slider:SetBackdropBorderColor(0.25, 0.25, 0.3, 0.5)

    local thumb = slider:CreateTexture(nil, "OVERLAY")
    thumb:SetColorTexture(CYAN.r, CYAN.g, CYAN.b, 0.5)
    thumb:SetSize(12, 40)
    slider:SetThumbTexture(thumb)

    slider:SetMinMaxValues(0, 1)
    slider:SetValue(0)
    slider:SetValueStep(1)
    slider:SetScript("OnValueChanged", function(self, value)
        SKPvEHistory_UpdateRows(math.floor(value))
    end)
    f.slider = slider
    f.scrollArea = scrollArea

    scrollArea:EnableMouseWheel(true)
    scrollArea:SetScript("OnMouseWheel", function(self, delta)
        local cur = slider:GetValue()
        slider:SetValue(cur - delta * 3)
    end)

    ---------------------
    -- Graph Panel
    ---------------------
    local GRAPH_PAD_LEFT = 55
    local GRAPH_PAD_RIGHT = 15
    local GRAPH_PAD_TOP = 30
    local GRAPH_PAD_BOTTOM = 30

    local graphPanel = CreateFrame("Frame", nil, f, "BackdropTemplate")
    graphPanel:SetPoint("TOPLEFT", f, "TOPLEFT", 4, contentY)
    graphPanel:SetPoint("BOTTOMRIGHT", f, "BOTTOMRIGHT", -4, 6)
    graphPanel:SetBackdrop({
        bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
        edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
        edgeSize = 14,
        insets = { left = 3, right = 3, top = 3, bottom = 3 },
    })
    graphPanel:SetBackdropColor(0.08, 0.08, 0.1, 1)
    graphPanel:SetBackdropBorderColor(0.3, 0.3, 0.35, 0.8)
    graphPanel:Hide()
    f.graphPanel = graphPanel

    graphPanel.title = graphPanel:CreateFontString(nil, "OVERLAY", "GameFontNormal")
    graphPanel.title:SetPoint("TOPLEFT", 10, -8)
    graphPanel.title:SetText("|cff00E5EEM+ Rating Over Time|r")

    graphPanel.noData = graphPanel:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
    graphPanel.noData:SetPoint("CENTER")
    graphPanel.noData:SetText("|cff888888No M+ runs with rating data for this filter.|r")
    graphPanel.noData:Hide()

    local plotArea = CreateFrame("Frame", nil, graphPanel)
    plotArea:SetPoint("TOPLEFT", GRAPH_PAD_LEFT, -GRAPH_PAD_TOP)
    plotArea:SetPoint("BOTTOMRIGHT", -GRAPH_PAD_RIGHT, GRAPH_PAD_BOTTOM)
    graphPanel.plotArea = plotArea

    graphPanel.gridLines = {}
    graphPanel.yLabels = {}
    graphPanel.xLabels = {}
    graphPanel.lineSegments = {}
    graphPanel.dataPoints = {}

    graphPanel.GetGridLine = function(index)
        if not graphPanel.gridLines[index] then
            local tex = plotArea:CreateTexture(nil, "BACKGROUND")
            tex:SetColorTexture(1, 1, 1, 0.08)
            graphPanel.gridLines[index] = tex
        end
        return graphPanel.gridLines[index]
    end

    graphPanel.GetYLabel = function(index)
        if not graphPanel.yLabels[index] then
            local fs = graphPanel:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
            fs:SetJustifyH("RIGHT")
            graphPanel.yLabels[index] = fs
        end
        return graphPanel.yLabels[index]
    end

    graphPanel.GetXLabel = function(index)
        if not graphPanel.xLabels[index] then
            local fs = graphPanel:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
            fs:SetJustifyH("CENTER")
            graphPanel.xLabels[index] = fs
        end
        return graphPanel.xLabels[index]
    end

    graphPanel.GetLineSegment = function(index)
        if not graphPanel.lineSegments[index] then
            local tex = plotArea:CreateTexture(nil, "ARTWORK")
            tex:SetColorTexture(1, 1, 1, 1)
            graphPanel.lineSegments[index] = tex
        end
        return graphPanel.lineSegments[index]
    end

    graphPanel.GetDataPoint = function(index)
        if not graphPanel.dataPoints[index] then
            local dot = plotArea:CreateTexture(nil, "OVERLAY")
            dot:SetColorTexture(1, 1, 1, 1)
            dot:SetSize(6, 6)

            local btn = CreateFrame("Button", nil, plotArea)
            btn:SetSize(14, 14)
            btn.dot = dot
            btn:SetScript("OnEnter", function(self)
                if self.matchData then
                    GameTooltip:SetOwner(UIParent, "ANCHOR_NONE")
                    local cx, cy = GetCursorPosition()
                    local sc = GameTooltip:GetEffectiveScale()
                    GameTooltip:SetPoint("BOTTOMLEFT", UIParent, "BOTTOMLEFT", cx / sc + 12, cy / sc + 8)
                    local m = self.matchData
                    GameTooltip:AddLine("|cff00E5EE+" .. (m.keyLevel or "?") .. " " .. (m.instanceName or "") .. "|r")
                    local resultStr = m.onTime and "|cff00ff00Timed|r" or "|cffff4444Depleted|r"
                    GameTooltip:AddLine(resultStr)
                    if m.newRating then
                        local sign = (m.ratingChange or 0) >= 0 and "+" or ""
                        local col = (m.ratingChange or 0) >= 0 and "|cff00ff00" or "|cffff4444"
                        GameTooltip:AddLine("Rating: " .. m.newRating .. " (" .. col .. sign .. (m.ratingChange or 0) .. "|r)")
                    end
                    GameTooltip:AddLine((m.date or "") .. " " .. (m.timeStr or ""), 0.5, 0.5, 0.5)
                    GameTooltip:Show()
                end
            end)
            btn:SetScript("OnLeave", function() GameTooltip:Hide() end)
            graphPanel.dataPoints[index] = { dot = dot, btn = btn }
        end
        return graphPanel.dataPoints[index]
    end

    graphPanel.DrawLine = function(tex, x1, y1, x2, y2, thickness, r, g, b, a)
        local dx = x2 - x1
        local dy = y2 - y1
        local length = math.sqrt(dx * dx + dy * dy)
        if length < 0.1 then tex:Hide(); return end
        local angle = math.atan2(dy, dx)
        local cx = (x1 + x2) / 2
        local cy = (y1 + y2) / 2
        tex:ClearAllPoints()
        tex:SetSize(length, thickness)
        tex:SetPoint("CENTER", plotArea, "BOTTOMLEFT", cx, cy)
        tex:SetRotation(angle)
        tex:SetColorTexture(r, g, b, a)
        tex:Show()
    end

    -- ShowView
    ShowView = function(viewKey)
        activeView = viewKey
        for _, btn in ipairs(viewButtons) do
            if btn.viewKey == viewKey then
                btn:SetBackdropColor(CYAN.r * 0.3, CYAN.g * 0.3, CYAN.b * 0.3, 1)
                btn:SetBackdropBorderColor(CYAN.r, CYAN.g, CYAN.b, 0.8)
                btn.text:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
            else
                btn:SetBackdropColor(0.1, 0.1, 0.12, 0.8)
                btn:SetBackdropBorderColor(0.3, 0.3, 0.35, 0.6)
                btn.text:SetTextColor(0.7, 0.7, 0.7)
            end
        end

        if viewKey == "matches" then
            headerFrame:Show()
            scrollArea:Show()
            slider:Show()
            graphPanel:Hide()
            SKPvEHistory_RefreshUI()
        elseif viewKey == "graph" then
            headerFrame:Hide()
            scrollArea:Hide()
            slider:Hide()
            graphPanel:Show()
            RefreshGraphPanel()
        end
    end

    -- RefreshGraphPanel
    RefreshGraphPanel = function()
        if not historyFrame or not historyFrame.graphPanel then return end
        local panel = historyFrame.graphPanel

        -- Hide all pooled elements
        for _, tex in ipairs(panel.gridLines) do tex:Hide() end
        for _, fs in ipairs(panel.yLabels) do fs:Hide() end
        for _, fs in ipairs(panel.xLabels) do fs:Hide() end
        for _, tex in ipairs(panel.lineSegments) do tex:Hide() end
        for _, dp in ipairs(panel.dataPoints) do dp.dot:Hide(); dp.btn:Hide() end
        panel.noData:Hide()

        -- Collect M+ runs with rating data, sorted chronologically
        local mplusRuns = {}
        local filtered = GetFilteredMatches()
        for _, m in ipairs(filtered) do
            if m.contentType == "mythicplus" and m.newRating and m.newRating > 0 then
                mplusRuns[#mplusRuns + 1] = m
            end
        end
        -- Sort oldest first
        table.sort(mplusRuns, function(a, b) return (a.timestamp or 0) < (b.timestamp or 0) end)

        if #mplusRuns == 0 then
            panel.noData:Show()
            return
        end

        local pArea = panel.plotArea
        local plotWidth = pArea:GetWidth()
        local plotHeight = pArea:GetHeight()
        if plotWidth <= 0 or plotHeight <= 0 then return end

        -- Find rating range
        local minRating, maxRating = math.huge, -math.huge
        for _, m in ipairs(mplusRuns) do
            if m.newRating < minRating then minRating = m.newRating end
            if m.newRating > maxRating then maxRating = m.newRating end
        end
        local ratingRange = math.max(50, maxRating - minRating)
        local ratingPad = ratingRange * 0.1
        minRating = math.max(0, minRating - ratingPad)
        maxRating = maxRating + ratingPad

        -- Axes helpers
        local function IndexToX(idx) return (idx - 1) / math.max(1, #mplusRuns - 1) * plotWidth end
        local function RatingToY(r) return (r - minRating) / (maxRating - minRating) * plotHeight end

        -- Grid lines (horizontal)
        local gridCount = 5
        local ratingStep = (maxRating - minRating) / gridCount
        for gi = 0, gridCount do
            local r = minRating + gi * ratingStep
            local y = RatingToY(r)

            local idx = gi + 1
            local line = panel.GetGridLine(idx)
            line:ClearAllPoints()
            line:SetHeight(1)
            line:SetPoint("LEFT", pArea, "BOTTOMLEFT", 0, y)
            line:SetPoint("RIGHT", pArea, "BOTTOMRIGHT", 0, y)
            line:Show()

            local label = panel.GetYLabel(idx)
            label:ClearAllPoints()
            label:SetPoint("RIGHT", pArea, "BOTTOMLEFT", -4, y)
            label:SetText(string.format("%.0f", r))
            label:SetTextColor(0.5, 0.5, 0.5)
            label:Show()
        end

        -- X-axis labels (dates)
        local xLabelCount = math.min(8, #mplusRuns)
        local xStep = math.max(1, math.floor(#mplusRuns / xLabelCount))
        local xlIdx = 0
        for i = 1, #mplusRuns, xStep do
            xlIdx = xlIdx + 1
            local m = mplusRuns[i]
            local x = IndexToX(i)

            local xLabel = panel.GetXLabel(xlIdx)
            xLabel:ClearAllPoints()
            xLabel:SetPoint("TOP", pArea, "BOTTOMLEFT", x, -2)
            xLabel:SetText(date("%m/%d", m.timestamp))
            xLabel:SetTextColor(0.5, 0.5, 0.5)
            xLabel:Show()
        end

        -- Plot line and data points
        local segIdx = 0
        local dpIdx = 0
        local lineR, lineG, lineB = GREEN.r, GREEN.g, GREEN.b

        for i, m in ipairs(mplusRuns) do
            local x = IndexToX(i)
            local y = RatingToY(m.newRating)

            -- Line segment from previous point
            if i > 1 then
                local prevM = mplusRuns[i - 1]
                local px = IndexToX(i - 1)
                local py = RatingToY(prevM.newRating)
                segIdx = segIdx + 1
                local seg = panel.GetLineSegment(segIdx)
                panel.DrawLine(seg, px, py, x, y, 1.5, lineR, lineG, lineB, 0.8)
            end

            -- Data point (green=timed, red=depleted)
            dpIdx = dpIdx + 1
            local dp = panel.GetDataPoint(dpIdx)
            local dr, dg, db
            if m.onTime then
                dr, dg, db = 0, 1, 0
            else
                dr, dg, db = 1, 0.27, 0.27
            end
            dp.dot:ClearAllPoints()
            dp.dot:SetSize(6, 6)
            dp.dot:SetPoint("CENTER", pArea, "BOTTOMLEFT", x, y)
            dp.dot:SetColorTexture(dr, dg, db, 1)
            dp.dot:Show()

            dp.btn:ClearAllPoints()
            dp.btn:SetPoint("CENTER", pArea, "BOTTOMLEFT", x, y)
            dp.btn.matchData = m
            dp.btn:Show()
        end
    end

    historyFrame = f
end

-----------------------------
-- Toggle
-----------------------------
function SKPvEHistory_Toggle()
    CreateHistoryFrame()

    if historyFrame:IsShown() then
        historyFrame:Hide()
    else
        if SKToolsPvEDB and SKToolsPvEDB.framePos then
            local pos = SKToolsPvEDB.framePos
            historyFrame:ClearAllPoints()
            historyFrame:SetPoint(pos.point, UIParent, pos.relPoint, pos.x, pos.y)
        end
        historyFrame:Show()
        ns.FadeIn(historyFrame, 0.15)
        ShowView("matches")
    end
end

-----------------------------
-- Settings Builders
-----------------------------
function ns.BuildPvEHistorySettings(content, anchor)
    local AddCB, GetLast, SetLast = ns.MakeCheckboxFactory(content, anchor)

    AddCB("PvEHistory", "pveHistory", "PvE History",
        "Track M+ dungeons, raids, and dungeon runs. View with /pvehistory.",
        nil)

    do
        local openBtn = ns.CreateThemedButton(content, "Open PvE History", 130, 22, "secondary")
        openBtn:SetPoint("TOPLEFT", GetLast(), "BOTTOMLEFT", 0, -4)
        openBtn:SetScript("OnClick", function() SKPvEHistory_Toggle() end)
        SetLast(openBtn)
    end

    return GetLast()
end

function ns.BuildPvEHistoryCombatSettings(content, anchor, csSyncControls)
    -- Not standalone tab — added to Arena tab content
end

-----------------------------
-- Initialization
-----------------------------
local initFrame = CreateFrame("Frame")
initFrame:RegisterEvent("ADDON_LOADED")
initFrame:SetScript("OnEvent", function(self, event, addon)
    if addon ~= "SKTools" then return end

    if not SKToolsPvEDB then
        SKToolsPvEDB = {}
    end
    if not SKToolsPvEDB.matches then
        SKToolsPvEDB.matches = {}
    end

end)
