-- SKTools Nameplates
-- 6 indicator styles, target borders, NP event frame
-- + Nameplates settings tab (both main & combat)

local _, ns = ...

-----------------------------
-- Nameplate Indicators
-----------------------------
local NP_THICKNESS = 2
local NP_TARGET_THICKNESS = 4
local NP_GAP = 1
local DEFAULT_STYLES = { [1] = true }  -- cached to avoid per-call allocation
local GLOW_BACKDROP = {
    bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
    edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
    tile = true, tileSize = 16, edgeSize = 12,
    insets = { left = 2, right = 2, top = 2, bottom = 2 },
}
local GROUP_FRAME_CONTAINERS = { "CompactPartyFrame", "CompactRaidFrameContainer" }

local function CreateTex(parent, w, h)
    local tex = parent:CreateTexture(nil, "OVERLAY")
    tex:SetColorTexture(1, 1, 1, 1)
    tex:SetSize(w, h)
    return tex
end

-- Style 1: Crosses (original)
local function BuildCrosses(f)
    local t, o, n, g, c = NP_THICKNESS, 18, 8, NP_GAP, 6
    local lines = {}
    -- Top-left
    lines[1] = CreateTex(f, o, t); lines[1]:SetPoint("BOTTOMRIGHT", f, "TOPLEFT", -g, g)
    lines[2] = CreateTex(f, t, o); lines[2]:SetPoint("BOTTOMRIGHT", f, "TOPLEFT", -g, g + t)
    lines[3] = CreateTex(f, n, t); lines[3]:SetPoint("BOTTOMLEFT", f, "TOPLEFT", -g, g)
    -- Top-right
    lines[4] = CreateTex(f, o, t); lines[4]:SetPoint("BOTTOMLEFT", f, "TOPRIGHT", g, g)
    lines[5] = CreateTex(f, t, o); lines[5]:SetPoint("BOTTOMLEFT", f, "TOPRIGHT", g, g + t)
    lines[6] = CreateTex(f, n, t); lines[6]:SetPoint("BOTTOMRIGHT", f, "TOPRIGHT", g, g)
    -- Bottom-left
    lines[7] = CreateTex(f, o, t); lines[7]:SetPoint("TOPRIGHT", f, "BOTTOMLEFT", -g, -g)
    lines[8] = CreateTex(f, t, o); lines[8]:SetPoint("TOPRIGHT", f, "BOTTOMLEFT", -g, -g - t)
    lines[9] = CreateTex(f, n, t); lines[9]:SetPoint("TOPLEFT", f, "BOTTOMLEFT", -g, -g)
    -- Bottom-right
    lines[10] = CreateTex(f, o, t); lines[10]:SetPoint("TOPLEFT", f, "BOTTOMRIGHT", g, -g)
    lines[11] = CreateTex(f, t, o); lines[11]:SetPoint("TOPLEFT", f, "BOTTOMRIGHT", g, -g - t)
    lines[12] = CreateTex(f, n, t); lines[12]:SetPoint("TOPRIGHT", f, "BOTTOMRIGHT", g, -g)
    -- Corner fills
    local corners = {}
    corners[1] = CreateTex(f, c, c); corners[1]:SetPoint("BOTTOMRIGHT", f, "TOPLEFT", -g + 1, g - 1)
    corners[2] = CreateTex(f, c, c); corners[2]:SetPoint("BOTTOMLEFT", f, "TOPRIGHT", g - 1, g - 1)
    corners[3] = CreateTex(f, c, c); corners[3]:SetPoint("TOPRIGHT", f, "BOTTOMLEFT", -g + 1, -g + 1)
    corners[4] = CreateTex(f, c, c); corners[4]:SetPoint("TOPLEFT", f, "BOTTOMRIGHT", g - 1, -g + 1)
    f.parts = lines
    f.corners = corners
    f.SetColors = function(self, lr, lg, lb, cr, cg, cb, a)
        for _, p in ipairs(self.parts) do p:SetVertexColor(lr, lg, lb, a) end
        for _, p in ipairs(self.corners) do p:SetVertexColor(cr, cg, cb, a) end
    end
    f.SetScale = function(self, isTarget)
        local thick = isTarget and NP_TARGET_THICKNESS or NP_THICKNESS
        for i, p in ipairs(self.parts) do
            local pos = (i - 1) % 3
            if pos == 1 then p:SetWidth(thick) else p:SetHeight(thick) end
        end
    end
end

-- Style 3: Brackets (L-shaped corners framing the healthbar)
local function BuildBrackets(f)
    local t = NP_THICKNESS
    local armH = 8
    local armV = 6
    local g = NP_GAP
    local parts = {}
    -- Top-left
    parts[1] = CreateTex(f, armH, t)
    parts[1]:SetPoint("BOTTOMLEFT", f, "TOPLEFT", -(g + t), g)
    parts[2] = CreateTex(f, t, armV)
    parts[2]:SetPoint("TOPLEFT", f, "TOPLEFT", -(g + t), g + t)
    -- Top-right
    parts[3] = CreateTex(f, armH, t)
    parts[3]:SetPoint("BOTTOMRIGHT", f, "TOPRIGHT", g + t, g)
    parts[4] = CreateTex(f, t, armV)
    parts[4]:SetPoint("TOPRIGHT", f, "TOPRIGHT", g + t, g + t)
    -- Bottom-left
    parts[5] = CreateTex(f, armH, t)
    parts[5]:SetPoint("TOPLEFT", f, "BOTTOMLEFT", -(g + t), -g)
    parts[6] = CreateTex(f, t, armV)
    parts[6]:SetPoint("BOTTOMLEFT", f, "BOTTOMLEFT", -(g + t), -(g + t))
    -- Bottom-right
    parts[7] = CreateTex(f, armH, t)
    parts[7]:SetPoint("TOPRIGHT", f, "BOTTOMRIGHT", g + t, -g)
    parts[8] = CreateTex(f, t, armV)
    parts[8]:SetPoint("BOTTOMRIGHT", f, "BOTTOMRIGHT", g + t, -(g + t))
    f.parts = parts
    f.SetColors = function(self, lr, lg, lb, _, _, _, a)
        for _, p in ipairs(self.parts) do p:SetVertexColor(lr, lg, lb, a) end
    end
    f.SetScale = function(self, isTarget)
        local thick = isTarget and NP_TARGET_THICKNESS or NP_THICKNESS
        local h = isTarget and 12 or 8
        local v = isTarget and 8 or 6
        for i, p in ipairs(self.parts) do
            if i % 2 == 1 then p:SetSize(h, thick) else p:SetSize(thick, v) end
        end
    end
end

-- Style 4: Bars (horizontal lines above and below the healthbar)
local function BuildBars(f)
    local t = NP_THICKNESS
    local g = 2
    local parts = {}
    -- Top bar (spans full healthbar width)
    parts[1] = CreateTex(f, 1, t)
    parts[1]:SetPoint("BOTTOMLEFT", f, "TOPLEFT", 0, g)
    parts[1]:SetPoint("BOTTOMRIGHT", f, "TOPRIGHT", 0, g)
    -- Bottom bar
    parts[2] = CreateTex(f, 1, t)
    parts[2]:SetPoint("TOPLEFT", f, "BOTTOMLEFT", 0, -g)
    parts[2]:SetPoint("TOPRIGHT", f, "BOTTOMRIGHT", 0, -g)
    f.parts = parts
    f.SetColors = function(self, lr, lg, lb, _, _, _, a)
        for _, p in ipairs(self.parts) do p:SetVertexColor(lr, lg, lb, a) end
    end
    f.SetScale = function(self, isTarget)
        local thick = isTarget and NP_TARGET_THICKNESS or NP_THICKNESS
        for _, p in ipairs(self.parts) do p:SetHeight(thick) end
    end
end

-- Style 4: Glow Border (rounded two-layer glow behind the healthbar)
local function BuildGlow(f)
    local parts = {}
    -- Outer layer: soft halo
    parts[1] = CreateFrame("Frame", nil, f, "BackdropTemplate")
    parts[1]:SetPoint("TOPLEFT", f, -6, 6)
    parts[1]:SetPoint("BOTTOMRIGHT", f, 6, -6)
    parts[1]:SetBackdrop(GLOW_BACKDROP)
    parts[1]:SetFrameLevel(f:GetFrameLevel())
    -- Inner layer: crisp colored border
    parts[2] = CreateFrame("Frame", nil, f, "BackdropTemplate")
    parts[2]:SetPoint("TOPLEFT", f, -2, 2)
    parts[2]:SetPoint("BOTTOMRIGHT", f, 2, -2)
    parts[2]:SetBackdrop(GLOW_BACKDROP)
    parts[2]:SetFrameLevel(f:GetFrameLevel())
    f.parts = parts
    f.SetColors = function(self, lr, lg, lb, _, _, _, a)
        local m = self._glowMult or 1.0
        self.parts[1]:SetBackdropColor(lr, lg, lb, a * 0.45 * m)
        self.parts[1]:SetBackdropBorderColor(lr, lg, lb, a * 0.6 * m)
        self.parts[2]:SetBackdropColor(lr, lg, lb, a * 0.3 * m)
        self.parts[2]:SetBackdropBorderColor(lr, lg, lb, a * 0.9 * m)
    end
    f.SetScale = function(self, isTarget)
        self._glowMult = isTarget and 1.4 or 1.0
        local outerPad = isTarget and 7 or 6
        local innerPad = isTarget and 3 or 2
        self.parts[1]:ClearAllPoints()
        self.parts[1]:SetPoint("TOPLEFT", self, -outerPad, outerPad)
        self.parts[1]:SetPoint("BOTTOMRIGHT", self, outerPad, -outerPad)
        self.parts[2]:ClearAllPoints()
        self.parts[2]:SetPoint("TOPLEFT", self, -innerPad, innerPad)
        self.parts[2]:SetPoint("BOTTOMRIGHT", self, innerPad, -innerPad)
    end
end

-- Style 5: Corner Triangles (filled triangles at each corner pointing outward)
local function BuildTriangles(f)
    local size = 6
    local g = 1
    local parts = {}
    for i, corner in ipairs({
        {"BOTTOMRIGHT", "TOPLEFT", -g, g},
        {"BOTTOMLEFT", "TOPRIGHT", g, g},
        {"TOPRIGHT", "BOTTOMLEFT", -g, -g},
        {"TOPLEFT", "BOTTOMRIGHT", g, -g},
    }) do
        parts[i] = CreateTex(f, size, size)
        parts[i]:SetPoint(corner[1], f, corner[2], corner[3], corner[4])
        parts[i]:SetRotation(math.rad(45))
    end
    f.parts = parts
    f.SetColors = function(self, lr, lg, lb, _, _, _, a)
        for _, p in ipairs(self.parts) do p:SetVertexColor(lr, lg, lb, a) end
    end
    f.SetScale = function(self, isTarget)
        local s = isTarget and 10 or 6
        for _, p in ipairs(self.parts) do p:SetSize(s, s) end
    end
end

-- Style 6: Crosshair (targeting reticle: lines above/below/left/right of center)
local function BuildCrosshair(f)
    local t = NP_THICKNESS
    local armLen = 10
    local g = 2
    local parts = {}
    -- Top
    parts[1] = CreateTex(f, t, armLen)
    parts[1]:SetPoint("BOTTOM", f, "TOP", 0, g)
    -- Bottom
    parts[2] = CreateTex(f, t, armLen)
    parts[2]:SetPoint("TOP", f, "BOTTOM", 0, -g)
    -- Left
    parts[3] = CreateTex(f, armLen, t)
    parts[3]:SetPoint("RIGHT", f, "LEFT", -g, 0)
    -- Right
    parts[4] = CreateTex(f, armLen, t)
    parts[4]:SetPoint("LEFT", f, "RIGHT", g, 0)
    f.parts = parts
    f.SetColors = function(self, lr, lg, lb, _, _, _, a)
        for _, p in ipairs(self.parts) do p:SetVertexColor(lr, lg, lb, a) end
    end
    f.SetScale = function(self, isTarget)
        local thick = isTarget and NP_TARGET_THICKNESS or NP_THICKNESS
        local len = isTarget and 14 or 10
        parts[1]:SetWidth(thick); parts[1]:SetHeight(len)
        parts[2]:SetWidth(thick); parts[2]:SetHeight(len)
        parts[3]:SetWidth(len); parts[3]:SetHeight(thick)
        parts[4]:SetWidth(len); parts[4]:SetHeight(thick)
    end
end

local styleBuildFuncs = {
    BuildGlow, BuildCrosses, BuildBrackets,
    BuildBars, BuildTriangles, BuildCrosshair,
}

-- Nameplate indicator overlays (keyed by plate base frame, avoids secure UnitFrame)
local npIndicators = {}
local npUnitMap = {}  -- plate -> unit token (our own mapping, no secure access)
local npLastTargetPlate = nil  -- cached target plate for incremental refresh
local npTargetBorderFrame = nil
local partyTargetBorderFrame = nil

-- Expose for external use
ns.npIndicators = npIndicators
ns.npUnitMap = npUnitMap

local function EnsureIndicators(plate)
    local styles = SKToolsDB.namePlateStyles or DEFAULT_STYLES
    local existing = npIndicators[plate]
    if not existing then
        existing = {}
        npIndicators[plate] = existing
    end

    -- Remove indicators for deactivated styles
    for style, ind in pairs(existing) do
        if not styles[style] then
            ind:Hide()
            existing[style] = nil
        end
    end

    -- Parent directly to the healthbar for exact positioning
    -- pcall: other addons can taint UnitFrame/healthBar properties
    local parent = plate
    pcall(function()
        local uf = plate.UnitFrame
        if uf then
            local hb = uf.healthBar or uf.HealthBar
            if hb then parent = hb end
        end
    end)

    -- Create indicators for newly active styles
    for style, enabled in pairs(styles) do
        if enabled and not existing[style] then
            local f = CreateFrame("Frame", nil, parent)
            f:SetAllPoints()
            local ok, lvl = pcall(parent.GetFrameLevel, parent)
            lvl = (ok and lvl) or 0
            if style == 1 then
                f:SetFrameLevel(math.max(lvl - 1, 0))
            else
                f:SetFrameLevel(lvl + 2)
            end
            local buildOk = pcall(styleBuildFuncs[style], f)
            if buildOk then
                f.skStyle = style
                existing[style] = f
            else
                f:Hide()
            end
        end
    end
    return existing
end

local function UpdateNameplateIndicator(plate, unit)
    if not plate or not unit then return end

    if not SKToolsDB or not SKToolsDB.namePlates then
        -- Hide any existing indicators without creating new ones
        local existing = npIndicators[plate]
        if existing then
            for _, ind in pairs(existing) do ind:Hide() end
        end
        return
    end

    local indicators = EnsureIndicators(plate)

    local okR, reaction = pcall(UnitReaction, unit, "player")
    if not okR then reaction = nil end
    local okP, isPlayer = pcall(UnitIsPlayer, unit)
    if not okP then isPlayer = false end
    -- Use cached plate ref instead of UnitIsUnit to avoid secret value taint in instanced PvP
    local isTarget = (plate == npLastTargetPlate)

    local useTargetColor = isTarget and SKToolsDB.targetHighlight

    for _, ind in pairs(indicators) do
        ind:SetScale(isTarget)

        if reaction and reaction <= 4 then
            if useTargetColor then
                if isPlayer then
                    ind:SetColors(0.7, 0, 0, 0.7, 0, 0, 0.95)
                else
                    ind:SetColors(0.4, 0, 0.55, 0.4, 0, 0.55, 0.95)
                end
            elseif isPlayer then
                ind:SetColors(1, 0, 0, 1, 0, 0, 0.9)
            else
                ind:SetColors(0.6, 0, 0.8, 1, 0, 0, 0.9)
            end
            ind:Show()
        elseif reaction then
            if useTargetColor then
                if isPlayer then
                    ind:SetColors(0, 1, 0, 0, 1, 0, 0.95)
                else
                    ind:SetColors(0.1, 0.25, 0.7, 0.1, 0.25, 0.7, 0.95)
                end
            elseif isPlayer then
                ind:SetColors(0, 1, 0, 0, 1, 0, 0.9)
            else
                ind:SetColors(0.2, 0.4, 1, 0, 1, 0, 0.9)
            end
            ind:Show()
        else
            ind:Hide()
        end
    end
end
ns.UpdateNameplateIndicator = UpdateNameplateIndicator

-----------------------------
-- Target Border Overlays
-----------------------------
local function GetTHConfig()
    return SKToolsDB and SKToolsDB.targetHighlightConfig or ns.defaults.targetHighlightConfig
end

local function GetTargetBorderColor()
    local cfg = GetTHConfig()
    local d = ns.defaults.targetHighlightConfig
    if cfg.enemyColor and UnitExists("target") then
        local reaction = UnitReaction("target", "player")
        if reaction and reaction <= 4 then
            return cfg.enemyColorR or d.enemyColorR, cfg.enemyColorG or d.enemyColorG, cfg.enemyColorB or d.enemyColorB
        end
    end
    return cfg.colorR or d.colorR, cfg.colorG or d.colorG, cfg.colorB or d.colorB
end

local function RecolorBorderFrame(frame, r, g, b)
    if not frame or not frame.colorTextures then return end
    for _, tex in ipairs(frame.colorTextures) do
        if tex.SetColorTexture then tex:SetColorTexture(r, g, b, 1) end
    end
end

local function EnsureNPTargetCrosshair()
    if npTargetBorderFrame then return npTargetBorderFrame end
    local cfg = GetTHConfig()
    local thick = cfg.crosshairThickness or NP_TARGET_THICKNESS
    local armLen = math.floor(thick * 3.5 + 0.5)
    local gap = 2

    local f = CreateFrame("Frame", nil, UIParent)
    f:SetSize(1, 1)
    f:SetFrameStrata("HIGH")
    f:SetFrameLevel(10)

    local function MakeArm(w, h)
        local tex = f:CreateTexture(nil, "OVERLAY")
        tex:SetColorTexture(cfg.colorR, cfg.colorG, cfg.colorB, 1)
        tex:SetSize(w, h)
        return tex
    end

    local armTop = MakeArm(thick, armLen)
    local armBottom = MakeArm(thick, armLen)
    local armLeft = MakeArm(armLen, thick)
    local armRight = MakeArm(armLen, thick)
    f.arms = { top = armTop, bottom = armBottom, left = armLeft, right = armRight }
    f.colorTextures = { armTop, armBottom, armLeft, armRight }
    f.finalGap = gap
    f.arms.top:SetPoint("BOTTOM", f, "TOP", 0, gap)
    f.arms.bottom:SetPoint("TOP", f, "BOTTOM", 0, -gap)
    f.arms.left:SetPoint("RIGHT", f, "LEFT", -gap, 0)
    f.arms.right:SetPoint("LEFT", f, "RIGHT", gap, 0)

    f:Hide()
    npTargetBorderFrame = f
    return f
end

local function EnsureNPTargetBorderBox()
    if npTargetBorderFrame then return npTargetBorderFrame end
    local cfg = GetTHConfig()
    local t = cfg.crosshairThickness or NP_TARGET_THICKNESS
    local r, g, b = cfg.colorR, cfg.colorG, cfg.colorB

    local f = CreateFrame("Frame", nil, UIParent)
    f:SetFrameStrata("HIGH")
    f:SetFrameLevel(10)

    local function MakeEdge()
        local tex = f:CreateTexture(nil, "OVERLAY")
        tex:SetColorTexture(r, g, b, 1)
        return tex
    end
    local top = MakeEdge()
    top:SetPoint("TOPLEFT", f, "TOPLEFT", 0, 0)
    top:SetPoint("TOPRIGHT", f, "TOPRIGHT", 0, 0)
    top:SetHeight(t)
    local bottom = MakeEdge()
    bottom:SetPoint("BOTTOMLEFT", f, "BOTTOMLEFT", 0, 0)
    bottom:SetPoint("BOTTOMRIGHT", f, "BOTTOMRIGHT", 0, 0)
    bottom:SetHeight(t)
    local left = MakeEdge()
    left:SetPoint("TOPLEFT", f, "TOPLEFT", 0, 0)
    left:SetPoint("BOTTOMLEFT", f, "BOTTOMLEFT", 0, 0)
    left:SetWidth(t)
    local right = MakeEdge()
    right:SetPoint("TOPRIGHT", f, "TOPRIGHT", 0, 0)
    right:SetPoint("BOTTOMRIGHT", f, "BOTTOMRIGHT", 0, 0)
    right:SetWidth(t)
    f.colorTextures = { top, bottom, left, right }

    f:Hide()
    npTargetBorderFrame = f
    return f
end

local function EnsureNPTarget()
    local cfg = GetTHConfig()
    local style = cfg.npTargetStyle or 1
    if style == 2 then
        return EnsureNPTargetBorderBox()
    else
        return EnsureNPTargetCrosshair()
    end
end

local function EnsurePartyTargetBorder()
    if partyTargetBorderFrame then return partyTargetBorderFrame end
    local cfg = GetTHConfig()
    local t = cfg.thickness
    local r, g, b = cfg.colorR, cfg.colorG, cfg.colorB

    local f = CreateFrame("Frame", nil, UIParent)
    f:SetFrameStrata("HIGH")
    f:SetFrameLevel(50)
    f:SetIgnoreParentAlpha(true)

    local function MakeEdge()
        local tex = f:CreateTexture(nil, "OVERLAY")
        tex:SetColorTexture(r, g, b, 1)
        return tex
    end
    -- Edges sit on the outside of the frame
    local top = MakeEdge()
    top:SetPoint("TOPLEFT", f, "TOPLEFT", -t, t)
    top:SetPoint("TOPRIGHT", f, "TOPRIGHT", t, t)
    top:SetHeight(t)
    local bottom = MakeEdge()
    bottom:SetPoint("BOTTOMLEFT", f, "BOTTOMLEFT", -t, -t)
    bottom:SetPoint("BOTTOMRIGHT", f, "BOTTOMRIGHT", t, -t)
    bottom:SetHeight(t)
    local left = MakeEdge()
    left:SetPoint("TOPLEFT", f, "TOPLEFT", -t, t)
    left:SetPoint("BOTTOMLEFT", f, "BOTTOMLEFT", -t, -t)
    left:SetWidth(t)
    local right = MakeEdge()
    right:SetPoint("TOPRIGHT", f, "TOPRIGHT", t, t)
    right:SetPoint("BOTTOMRIGHT", f, "BOTTOMRIGHT", t, -t)
    right:SetWidth(t)
    f.colorTextures = { top, bottom, left, right }

    f:Hide()
    partyTargetBorderFrame = f
    return f
end

-- Check .unit without spreading taint: issecurevariable reads metadata, not the value
local function SafeUnitMatch(f)
    if not f or not f:IsVisible() then return false end
    if not issecurevariable(f, "unit") then return false end  -- tainted, skip
    return f.unit and UnitIsUnit(f.unit, "target")
end

-- Helper: search children via varargs to avoid table allocation from {GetChildren()}
local function FindTargetChild(...)
    for i = 1, select("#", ...) do
        local child = select(i, ...)
        if SafeUnitMatch(child) then return child end
    end
end

local function FindGroupFrameForTarget()
    if not UnitExists("target") then return nil end
    for i = 1, 5 do
        local f = _G["CompactPartyFrameMember" .. i]
        if SafeUnitMatch(f) then return f end
    end
    for i = 1, 40 do
        local f = _G["CompactRaidFrame" .. i]
        if SafeUnitMatch(f) then return f end
    end
    for _, cname in ipairs(GROUP_FRAME_CONTAINERS) do
        local c = _G[cname]
        if c and c:IsVisible() and c.GetChildren then
            local found = FindTargetChild(c:GetChildren())
            if found then return found end
        end
    end
    -- Arena enemy frames
    for i = 1, 5 do
        local f = _G["CompactArenaFrameMember" .. i]
        if SafeUnitMatch(f) then return f end
    end
    local arenaContainer = _G["CompactArenaFrame"]
    if arenaContainer and arenaContainer:IsVisible() and arenaContainer.GetChildren then
        local found = FindTargetChild(arenaContainer:GetChildren())
        if found then return found end
    end
    return nil
end

-- Snap-in animation: border starts expanded and condenses onto the target
-- Kept as OnUpdate (only 1 frame animating at a time, lightweight)
local function AnimateBorderSnapIn(border, anchor, finalPad, duration, extraPad)
    local startPad = finalPad + extraPad
    local elapsed = 0
    local padDiff = finalPad - startPad  -- pre-compute
    border:ClearAllPoints()
    border:SetPoint("TOPLEFT", anchor, "TOPLEFT", -startPad, startPad)
    border:SetPoint("BOTTOMRIGHT", anchor, "BOTTOMRIGHT", startPad, -startPad)
    border:SetAlpha(0.4)
    border:Show()
    border:SetScript("OnUpdate", function(self, dt)
        local ok, err = pcall(function()
        if not anchor or not anchor:IsVisible() then
            self:SetScript("OnUpdate", nil)
            self:Hide()
            return
        end
        elapsed = elapsed + dt
        local t = elapsed / duration
        if t >= 1 then
            self:ClearAllPoints()
            self:SetPoint("TOPLEFT", anchor, "TOPLEFT", -finalPad, finalPad)
            self:SetPoint("BOTTOMRIGHT", anchor, "BOTTOMRIGHT", finalPad, -finalPad)
            self:SetAlpha(1)
            self:SetScript("OnUpdate", nil)
            return
        end
        local ease = 1 - (1 - t) * (1 - t)
        local pad = startPad + padDiff * ease
        self:ClearAllPoints()
        self:SetPoint("TOPLEFT", anchor, "TOPLEFT", -pad, pad)
        self:SetPoint("BOTTOMRIGHT", anchor, "BOTTOMRIGHT", pad, -pad)
        self:SetAlpha(0.4 + 0.6 * ease)
        end)
        if not ok then self:SetScript("OnUpdate", nil); self:Hide() end
    end)
end

local function UpdatePartyTargetBorder()
    if not SKToolsDB or not SKToolsDB.targetHighlight then
        if partyTargetBorderFrame then partyTargetBorderFrame:Hide() end
        return
    end
    local cfg = GetTHConfig()
    if not cfg.partyHighlight then
        if partyTargetBorderFrame then partyTargetBorderFrame:Hide() end
        return
    end
    local border = EnsurePartyTargetBorder()
    local frame = FindGroupFrameForTarget()
    if frame then
        -- SetParent on CompactArenaFrame children can trigger internal SetFromScale errors
        local ok = pcall(function()
            border:SetParent(frame)
            border:SetFrameLevel(frame:GetFrameLevel() + 10)
        end)
        if not ok then border:Hide(); return end
        local partySpeed = cfg.animSpeed * 0.57
        local partyDist = math.floor(cfg.startDistance * 0.67 + 0.5)
        AnimateBorderSnapIn(border, frame, 0, partySpeed, partyDist)
        RecolorBorderFrame(border, GetTargetBorderColor())
    else
        border:SetScript("OnUpdate", nil)
        if border._snapAnim then border._snapAnim:Stop() end
        border:Hide()
    end
end
ns.UpdatePartyTargetBorder = UpdatePartyTargetBorder

local function UpdateNPTargetBorder(newPlate)
    if not SKToolsDB or not SKToolsDB.targetHighlight then
        if npTargetBorderFrame then npTargetBorderFrame:Hide() end
        return
    end
    local cfg = GetTHConfig()
    if (cfg.npTargetStyle or 1) == 0 then
        if npTargetBorderFrame then npTargetBorderFrame:Hide() end
        return
    end
    local frame = EnsureNPTarget()
    if newPlate then
        local anchor = newPlate
        pcall(function()
            local uf = newPlate.UnitFrame
            if uf then
                local hb = uf.healthBar or uf.HealthBar
                if hb then anchor = hb end
            end
        end)
        frame:SetParent(anchor)
        local ok, lvl = pcall(anchor.GetFrameLevel, anchor)
        frame:SetFrameLevel((ok and lvl or 0) + 5)
        frame:ClearAllPoints()

        local style = cfg.npTargetStyle or 1
        if style == 1 and frame.arms then
            -- Crosshair: snap arms inward (OnUpdate — only 1 target animates at a time)
            frame:SetAllPoints(anchor)
            local finalGap = frame.finalGap
            local startGap = finalGap + cfg.startDistance
            local elapsed = 0
            local dur = cfg.animSpeed
            local arms = frame.arms

            for _, arm in pairs(arms) do arm:ClearAllPoints() end
            arms.top:SetPoint("BOTTOM", frame, "TOP", 0, startGap)
            arms.bottom:SetPoint("TOP", frame, "BOTTOM", 0, -startGap)
            arms.left:SetPoint("RIGHT", frame, "LEFT", -startGap, 0)
            arms.right:SetPoint("LEFT", frame, "RIGHT", startGap, 0)

            frame:SetAlpha(0.4)
            frame:Show()
            frame:SetScript("OnUpdate", function(self, dt)
                local ok, err = pcall(function()
                if not anchor or not anchor:IsVisible() then
                    self:SetScript("OnUpdate", nil)
                    self:Hide()
                    return
                end
                elapsed = elapsed + dt
                local t = elapsed / dur
                if t >= 1 then
                    t = 1
                    self:SetScript("OnUpdate", nil)
                end
                local ease = 1 - (1 - t) * (1 - t)
                local gap = startGap + (finalGap - startGap) * ease
                for _, arm in pairs(arms) do arm:ClearAllPoints() end
                arms.top:SetPoint("BOTTOM", frame, "TOP", 0, gap)
                arms.bottom:SetPoint("TOP", frame, "BOTTOM", 0, -gap)
                arms.left:SetPoint("RIGHT", frame, "LEFT", -gap, 0)
                arms.right:SetPoint("LEFT", frame, "RIGHT", gap, 0)
                self:SetAlpha(0.4 + 0.6 * ease)
                end)
                if not ok then self:SetScript("OnUpdate", nil); self:Hide() end
            end)
        else
            -- Border box: snap inward
            AnimateBorderSnapIn(frame, anchor, 10, cfg.animSpeed, cfg.startDistance)
        end
        RecolorBorderFrame(frame, GetTargetBorderColor())
    else
        -- Stop all animations and hide
        frame:SetScript("OnUpdate", nil)
        frame:Hide()
    end
end
ns.UpdateNPTargetBorder = UpdateNPTargetBorder

local function RecreateTargetBorders()
    if npTargetBorderFrame then npTargetBorderFrame:Hide(); npTargetBorderFrame = nil end
    if partyTargetBorderFrame then partyTargetBorderFrame:Hide(); partyTargetBorderFrame = nil end
    if SKToolsDB and SKToolsDB.targetHighlight then
        pcall(UpdateNPTargetBorder, C_NamePlate.GetNamePlateForUnit("target"))
        pcall(UpdatePartyTargetBorder)
    end
end
ns.RecreateTargetBorders = RecreateTargetBorders

-- Event-driven nameplate indicators (no secure frame hooks)
-- Events registered by SetNameplates() during init, not at file load
local npEventFrame = CreateFrame("Frame")
-- Single pcall wraps the entire handler to avoid per-call pcall overhead
local function npEventHandler(event, ...)
    if event == "NAME_PLATE_UNIT_ADDED" then
        local unit = (...)
        local plate = C_NamePlate.GetNamePlateForUnit(unit)
        if plate then
            npUnitMap[plate] = unit
            UpdateNameplateIndicator(plate, unit)
            if UnitIsUnit(unit, "target") then
                npLastTargetPlate = plate
                UpdateNPTargetBorder(plate)
            end
        end
    elseif event == "NAME_PLATE_UNIT_REMOVED" then
        local unit = (...)
        local plate = C_NamePlate.GetNamePlateForUnit(unit)
        if plate then
            npUnitMap[plate] = nil
            if npIndicators[plate] then
                for _, ind in pairs(npIndicators[plate]) do ind:Hide() end
                npIndicators[plate] = nil
            end
            if plate == npLastTargetPlate then
                if npTargetBorderFrame then npTargetBorderFrame:Hide() end
            end
        end
    elseif event == "PLAYER_TARGET_CHANGED" then
        local oldPlate = npLastTargetPlate
        if oldPlate and npUnitMap[oldPlate] then
            UpdateNameplateIndicator(oldPlate, npUnitMap[oldPlate])
        end
        local newPlate = C_NamePlate.GetNamePlateForUnit("target")
        if newPlate and npUnitMap[newPlate] and newPlate ~= oldPlate then
            UpdateNameplateIndicator(newPlate, npUnitMap[newPlate])
        end
        npLastTargetPlate = newPlate
        UpdateNPTargetBorder(newPlate)
        UpdatePartyTargetBorder()
    elseif event == "GROUP_ROSTER_UPDATE" then
        UpdatePartyTargetBorder()
    end
end
npEventFrame:SetScript("OnEvent", function(self, event, ...)
    local ok, err = pcall(npEventHandler, event, ...)
    if not ok then ns.SK_ReportError("NP:" .. event, err) end
end)

local function ForceResetAllIndicators()
    for plate, indicators in pairs(npIndicators) do
        for _, ind in pairs(indicators) do ind:Hide() end
        npIndicators[plate] = nil
    end
    -- Re-apply to all tracked plates
    for plate, unit in pairs(npUnitMap) do
        UpdateNameplateIndicator(plate, unit)
    end
end
ns.ForceResetAllIndicators = ForceResetAllIndicators

-----------------------------
-- Feature Toggle (register/unregister events)
-----------------------------
local function SetNameplates()
    local needEvents = (SKToolsDB and SKToolsDB.namePlates) or (SKToolsDB and SKToolsDB.targetHighlight)
    if needEvents then
        npEventFrame:RegisterEvent("NAME_PLATE_UNIT_ADDED")
        npEventFrame:RegisterEvent("NAME_PLATE_UNIT_REMOVED")
        npEventFrame:RegisterEvent("PLAYER_TARGET_CHANGED")
        npEventFrame:RegisterEvent("GROUP_ROSTER_UPDATE")
        -- Repopulate npUnitMap from all currently visible nameplates
        for _, plate in ipairs(C_NamePlate.GetNamePlates()) do
            local unit = plate.namePlateUnitToken
            if unit and not npUnitMap[plate] then
                npUnitMap[plate] = unit
            end
        end
        -- Refresh existing plates
        for plate, unit in pairs(npUnitMap) do
            pcall(UpdateNameplateIndicator, plate, unit)
        end
        if SKToolsDB.targetHighlight then
            pcall(UpdateNPTargetBorder, C_NamePlate.GetNamePlateForUnit("target"))
            pcall(UpdatePartyTargetBorder)
        end
    else
        npEventFrame:UnregisterAllEvents()
        -- Destroy all indicator frames to free rendering resources
        for plate, indicators in pairs(npIndicators) do
            for _, ind in pairs(indicators) do
                ind:Hide()
                if ind.parts then
                    for _, part in pairs(ind.parts) do
                        if part.SetBackdrop then part:SetBackdrop(nil) end
                        part:Hide()
                    end
                end
                if ind.corners then
                    for _, c in pairs(ind.corners) do c:Hide() end
                end
            end
        end
        wipe(npIndicators)
        wipe(npUnitMap)
        npLastTargetPlate = nil
        if npTargetBorderFrame then
            npTargetBorderFrame:SetScript("OnUpdate", nil)
            if npTargetBorderFrame._snapAnim then npTargetBorderFrame._snapAnim:Stop() end
            npTargetBorderFrame:Hide()
        end
        if partyTargetBorderFrame then
            partyTargetBorderFrame:SetScript("OnUpdate", nil)
            if partyTargetBorderFrame._snapAnim then partyTargetBorderFrame._snapAnim:Stop() end
            partyTargetBorderFrame:Hide()
        end
    end
end
ns.SetNameplates = SetNameplates

-----------------------------
-- Nameplates Settings Builder (main panel)
-----------------------------
function ns.BuildNameplateSettings(content, anchor)
    local NAMEPLATE_STYLES = ns.NAMEPLATE_STYLES
    local npCard = ns.CreateSectionCard(content)
    npCard:SetPoint("TOPLEFT", anchor, "BOTTOMLEFT", -4, -12)
    local AddCB, GetLast, SetLast = ns.MakeCheckboxFactory(content, anchor)

    AddCB("NamePlates", "namePlates", "Nameplate Indicators",
        "Adds colored indicators around nameplates. Red for enemies, green for allies, purple for enemy NPCs.",
        function(enabled)
            SetNameplates()
        end)

    -- Nameplate style selector (multi-select toggle buttons)
    do
        local lastEl = GetLast()
        local styleLabel = content:CreateFontString(nil, "ARTWORK", "GameFontNormal")
        styleLabel:SetPoint("TOPLEFT", lastEl, "BOTTOMLEFT", 0, -12)
        styleLabel:SetText("Indicator Styles  |cff888888(select multiple)|r")
        styleLabel:SetTextColor(0.9, 0.9, 0.9)

        local styleBtns = {}
        local prevStyleBtn

        local function IsStyleActive(idx)
            local styles = SKToolsDB.namePlateStyles or DEFAULT_STYLES
            return styles[idx] == true
        end

        local function RefreshStyleButtons()
            for _, b in ipairs(styleBtns) do
                if IsStyleActive(b.styleIndex) then
                    b:SetBackdropColor(0, ns.CYAN.g * 0.15, ns.CYAN.b * 0.15, 1)
                    b:SetBackdropBorderColor(ns.CYAN.r, ns.CYAN.g, ns.CYAN.b, 0.6)
                    b.text:SetTextColor(ns.CYAN.r, ns.CYAN.g, ns.CYAN.b)
                else
                    b:SetBackdropColor(0.1, 0.1, 0.12, 0.8)
                    b:SetBackdropBorderColor(0.3, 0.3, 0.35, 0.6)
                    b.text:SetTextColor(0.7, 0.7, 0.7)
                end
            end
        end

        for i, style in ipairs(NAMEPLATE_STYLES) do
            local btn = CreateFrame("Button", nil, content, "BackdropTemplate")
            btn:SetSize(68, 26)
            btn:SetBackdrop(ns.BACKDROP_PILL)
            if i == 1 then
                btn:SetPoint("TOPLEFT", styleLabel, "BOTTOMLEFT", 0, -6)
            else
                btn:SetPoint("LEFT", prevStyleBtn, "RIGHT", 4, 0)
            end

            btn.text = btn:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
            btn.text:SetPoint("CENTER")
            btn.text:SetText(style.label)
            btn.styleIndex = i

            btn:SetScript("OnClick", function(self)
                if not SKToolsDB.namePlateStyles then
                    SKToolsDB.namePlateStyles = { [1] = true }
                end
                local styles = SKToolsDB.namePlateStyles
                if styles[self.styleIndex] then
                    -- Don't allow deselecting the last active style
                    local count = 0
                    for _, v in pairs(styles) do if v then count = count + 1 end end
                    if count <= 1 then return end
                    styles[self.styleIndex] = nil
                else
                    styles[self.styleIndex] = true
                end
                RefreshStyleButtons()
                ForceResetAllIndicators()
            end)
            btn:SetScript("OnEnter", function(self)
                if not IsStyleActive(self.styleIndex) then
                    self:SetBackdropColor(0.15, 0.15, 0.18, 0.9)
                    self.text:SetTextColor(1, 1, 1)
                end
            end)
            btn:SetScript("OnLeave", function(self)
                if not IsStyleActive(self.styleIndex) then
                    self:SetBackdropColor(0.1, 0.1, 0.12, 0.8)
                    self.text:SetTextColor(0.7, 0.7, 0.7)
                end
            end)

            styleBtns[#styleBtns + 1] = btn
            prevStyleBtn = btn
        end
        RefreshStyleButtons()

        local styleSpacer = content:CreateFontString(nil, "ARTWORK")
        styleSpacer:SetPoint("TOPLEFT", styleBtns[1], "BOTTOMLEFT", 0, -4)
        styleSpacer:SetHeight(1)
        SetLast(styleSpacer)
    end

    -- Close Nameplate Indicators card
    npCard:SetPoint("BOTTOM", GetLast(), "BOTTOM", 0, -8)
    npCard:SetPoint("RIGHT", content, "RIGHT", -8, 0)

    -- Target Highlighting section
    local thHeader = ns.AddSectionHeader(content, GetLast(), "Target Highlighting", false)
    local thCard = ns.CreateSectionCard(content)
    thCard:SetPoint("TOPLEFT", thHeader, "TOPLEFT", -8, 8)

    do
        local defaults = ns.defaults
        local cfg = SKToolsDB.targetHighlightConfig
        local thChildren = {}

        local thCB = ns.CreateToggleSwitch(content, "SKToolsTargetHighlightCB")
        thCB:SetPoint("TOPLEFT", thHeader, "BOTTOMLEFT", 0, -12)
        thCB.Text:SetText("Target Highlighting")
        thCB.Text:SetTextColor(0.9, 0.9, 0.9)
        thCB:SetChecked(SKToolsDB.targetHighlight)
        thCB:SetScript("OnClick", function(self)
            SKToolsDB.targetHighlight = self:GetChecked()
            SetNameplates()
            ForceResetAllIndicators()
        end)

        -- Party Frame Highlight toggle (child of Target Highlighting)
        local partyCB = ns.CreateToggleSwitch(content)
        partyCB:SetPoint("TOPLEFT", thCB, "BOTTOMLEFT", 16, -6)
        partyCB.Text:SetText("Party Frame Highlight")
        partyCB.Text:SetFontObject("GameFontHighlightSmall")
        partyCB.Text:SetTextColor(0.9, 0.9, 0.9)
        partyCB:SetChecked(cfg.partyHighlight)
        partyCB:SetScript("OnClick", function(self)
            cfg.partyHighlight = self:GetChecked()
            RecreateTargetBorders()
        end)
        thChildren[#thChildren + 1] = partyCB

        -- Nameplate target style selector
        local npStyleLabel = content:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        npStyleLabel:SetPoint("TOPLEFT", partyCB, "BOTTOMLEFT", -16, -10)
        npStyleLabel:SetText("Nameplate Style")
        npStyleLabel:SetTextColor(0.9, 0.9, 0.9)
        thChildren[#thChildren + 1] = npStyleLabel

        local NP_TARGET_STYLES = { {label = "Crosshair", idx = 1}, {label = "Border", idx = 2}, {label = "None", idx = 0} }
        local npTStyleBtns = {}
        local prevNpTBtn

        local function RefreshNpTStyleButtons()
            local cur = cfg.npTargetStyle or 1
            for _, b in ipairs(npTStyleBtns) do
                if b.styleIdx == cur then
                    b:SetBackdropColor(0, ns.CYAN.g * 0.15, ns.CYAN.b * 0.15, 1)
                    b:SetBackdropBorderColor(ns.CYAN.r, ns.CYAN.g, ns.CYAN.b, 0.6)
                    b.text:SetTextColor(ns.CYAN.r, ns.CYAN.g, ns.CYAN.b)
                else
                    b:SetBackdropColor(0.1, 0.1, 0.12, 0.8)
                    b:SetBackdropBorderColor(0.3, 0.3, 0.35, 0.6)
                    b.text:SetTextColor(0.7, 0.7, 0.7)
                end
            end
        end

        for i, style in ipairs(NP_TARGET_STYLES) do
            local btn = CreateFrame("Button", nil, content, "BackdropTemplate")
            btn:SetSize(84, 26)
            btn:SetBackdrop(ns.BACKDROP_PILL)
            if i == 1 then
                btn:SetPoint("TOPLEFT", npStyleLabel, "BOTTOMLEFT", 0, -6)
            else
                btn:SetPoint("LEFT", prevNpTBtn, "RIGHT", 4, 0)
            end
            btn.text = btn:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
            btn.text:SetPoint("CENTER")
            btn.text:SetText(style.label)
            btn.styleIdx = style.idx
            btn:SetScript("OnClick", function(self)
                cfg.npTargetStyle = self.styleIdx
                RefreshNpTStyleButtons()
                RecreateTargetBorders()
            end)
            npTStyleBtns[#npTStyleBtns + 1] = btn
            thChildren[#thChildren + 1] = btn
            prevNpTBtn = btn
        end
        RefreshNpTStyleButtons()

        -- Color swatch
        local colorLabel = content:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        colorLabel:SetPoint("TOPLEFT", npTStyleBtns[1], "BOTTOMLEFT", 0, -8)
        colorLabel:SetText("Color")
        colorLabel:SetTextColor(0.9, 0.9, 0.9)
        thChildren[#thChildren + 1] = colorLabel

        local colorSwatch = CreateFrame("Button", nil, content, "BackdropTemplate")
        colorSwatch:SetSize(24, 24)
        colorSwatch:SetPoint("LEFT", colorLabel, "RIGHT", 8, 0)
        colorSwatch:SetBackdrop(ns.BACKDROP_CONTROL)
        colorSwatch:SetBackdropBorderColor(0.3, 0.3, 0.35, 0.8)
        local swatchBg = colorSwatch:CreateTexture(nil, "BACKGROUND")
        swatchBg:SetPoint("TOPLEFT", 3, -3)
        swatchBg:SetPoint("BOTTOMRIGHT", -3, 3)
        swatchBg:SetColorTexture(0, 0, 0, 1)
        local swatchTex = colorSwatch:CreateTexture(nil, "ARTWORK")
        swatchTex:SetPoint("TOPLEFT", 3, -3)
        swatchTex:SetPoint("BOTTOMRIGHT", -3, 3)
        swatchTex:SetColorTexture(cfg.colorR, cfg.colorG, cfg.colorB, 1)
        colorSwatch.tex = swatchTex
        colorSwatch:SetScript("OnEnter", function(self) self:SetBackdropBorderColor(ns.CYAN.r, ns.CYAN.g, ns.CYAN.b, 0.8) end)
        colorSwatch:SetScript("OnLeave", function(self) self:SetBackdropBorderColor(0.3, 0.3, 0.35, 0.8) end)
        colorSwatch:HookScript("OnClick", function() end) -- ensure hover scripts aren't overridden
        colorSwatch:SetScript("OnClick", function()
            local prev = { r = cfg.colorR, g = cfg.colorG, b = cfg.colorB }
            local function SetColor(r, g, b)
                cfg.colorR, cfg.colorG, cfg.colorB = r, g, b
                swatchTex:SetColorTexture(r, g, b, 1)
                RecreateTargetBorders()
            end
            ColorPickerFrame:SetupColorPickerAndShow({
                r = cfg.colorR, g = cfg.colorG, b = cfg.colorB,
                swatchFunc = function()
                    local r, g, b = ColorPickerFrame:GetColorRGB()
                    SetColor(r, g, b)
                end,
                cancelFunc = function()
                    SetColor(prev.r, prev.g, prev.b)
                end,
            })
        end)
        thChildren[#thChildren + 1] = colorSwatch

        -- Enemy color toggle + swatch
        local enemyCB = ns.CreateToggleSwitch(content)
        enemyCB:SetPoint("TOPLEFT", colorLabel, "BOTTOMLEFT", 0, -8)
        enemyCB.Text:SetText("Separate Enemy Color")
        enemyCB.Text:SetFontObject("GameFontHighlightSmall")
        enemyCB.Text:SetTextColor(0.9, 0.9, 0.9)
        enemyCB:SetChecked(cfg.enemyColor)
        thChildren[#thChildren + 1] = enemyCB

        local enemyColorLabel = content:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        enemyColorLabel:SetPoint("TOPLEFT", enemyCB, "BOTTOMLEFT", 0, -6)
        enemyColorLabel:SetText("Enemy Color")
        enemyColorLabel:SetTextColor(0.9, 0.9, 0.9)
        thChildren[#thChildren + 1] = enemyColorLabel

        local enemySwatch = CreateFrame("Button", nil, content, "BackdropTemplate")
        enemySwatch:SetSize(24, 24)
        enemySwatch:SetPoint("LEFT", enemyColorLabel, "RIGHT", 8, 0)
        enemySwatch:SetBackdrop(ns.BACKDROP_CONTROL)
        enemySwatch:SetBackdropBorderColor(0.3, 0.3, 0.35, 0.8)
        local eBg = enemySwatch:CreateTexture(nil, "BACKGROUND")
        eBg:SetPoint("TOPLEFT", 3, -3)
        eBg:SetPoint("BOTTOMRIGHT", -3, 3)
        eBg:SetColorTexture(0, 0, 0, 1)
        local eSwatchTex = enemySwatch:CreateTexture(nil, "ARTWORK")
        eSwatchTex:SetPoint("TOPLEFT", 3, -3)
        eSwatchTex:SetPoint("BOTTOMRIGHT", -3, 3)
        eSwatchTex:SetColorTexture(cfg.enemyColorR, cfg.enemyColorG, cfg.enemyColorB, 1)
        enemySwatch.tex = eSwatchTex
        enemySwatch:SetScript("OnEnter", function(self) self:SetBackdropBorderColor(ns.CYAN.r, ns.CYAN.g, ns.CYAN.b, 0.8) end)
        enemySwatch:SetScript("OnLeave", function(self) self:SetBackdropBorderColor(0.3, 0.3, 0.35, 0.8) end)
        enemySwatch:SetScript("OnClick", function()
            if not cfg.enemyColor then return end
            local prev = { r = cfg.enemyColorR, g = cfg.enemyColorG, b = cfg.enemyColorB }
            local function SetEColor(r, g, b)
                cfg.enemyColorR, cfg.enemyColorG, cfg.enemyColorB = r, g, b
                eSwatchTex:SetColorTexture(r, g, b, 1)
                RecreateTargetBorders()
            end
            ColorPickerFrame:SetupColorPickerAndShow({
                r = cfg.enemyColorR, g = cfg.enemyColorG, b = cfg.enemyColorB,
                swatchFunc = function()
                    local r, g, b = ColorPickerFrame:GetColorRGB()
                    SetEColor(r, g, b)
                end,
                cancelFunc = function() SetEColor(prev.r, prev.g, prev.b) end,
            })
        end)
        thChildren[#thChildren + 1] = enemySwatch

        local function UpdateEnemyColorState()
            local on = cfg.enemyColor
            enemyColorLabel:SetAlpha(on and 1.0 or 0.4)
            enemySwatch:SetAlpha(on and 1.0 or 0.4)
        end
        enemyCB:SetScript("OnClick", function(self)
            cfg.enemyColor = self:GetChecked()
            UpdateEnemyColorState()
            RecreateTargetBorders()
        end)
        UpdateEnemyColorState()

        -- Helper to make a labeled slider
        local function MakeTHSlider(label, anchorTo, min, max, step, initial, format, onChange)
            local lbl = content:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
            lbl:SetPoint("TOPLEFT", anchorTo, "BOTTOMLEFT", 0, -14)
            lbl:SetText(label)
            lbl:SetTextColor(0.9, 0.9, 0.9)
            thChildren[#thChildren + 1] = lbl

            local minus = ns.CreateThemedButton(content, "-", 24, 24, "secondary")
            minus:SetPoint("TOPLEFT", lbl, "BOTTOMLEFT", 0, -12)
            thChildren[#thChildren + 1] = minus

            local sl = CreateFrame("Slider", nil, content, "OptionsSliderTemplate")
            sl:SetPoint("LEFT", minus, "RIGHT", 4, 0)
            sl:SetWidth(140)
            sl:SetMinMaxValues(min, max)
            sl:SetValueStep(step)
            sl:SetObeyStepOnDrag(true)
            sl:SetValue(initial)
            ns.StyleSlider(sl)
            sl.Low:SetText(string.format(format, min))
            sl.High:SetText(string.format(format, max))
            sl.Text:SetText(string.format(format, initial))
            sl:SetScript("OnValueChanged", function(self, value)
                value = math.floor(value / step + 0.5) * step
                self.Text:SetText(string.format(format, value))
                onChange(value)
            end)
            thChildren[#thChildren + 1] = sl

            local plus = ns.CreateThemedButton(content, "+", 24, 24, "secondary")
            plus:SetPoint("LEFT", sl, "RIGHT", 4, 0)
            thChildren[#thChildren + 1] = plus

            minus:SetScript("OnClick", function()
                local val = sl:GetValue() - step
                if val >= min then sl:SetValue(val) end
            end)
            plus:SetScript("OnClick", function()
                local val = sl:GetValue() + step
                if val <= max then sl:SetValue(val) end
            end)

            local spacer = content:CreateFontString(nil, "ARTWORK")
            spacer:SetPoint("TOPLEFT", minus, "BOTTOMLEFT", 0, -4)
            spacer:SetHeight(1)

            return spacer, sl
        end

        local speedAnchor, speedSl = MakeTHSlider("Animation Speed", enemyColorLabel, 0.1, 1.0, 0.05, cfg.animSpeed, "%.2fs", function(v)
            cfg.animSpeed = v
        end)

        local distAnchor, distSl = MakeTHSlider("Start Distance", speedAnchor, 10, 250, 10, cfg.startDistance, "%dpx", function(v)
            cfg.startDistance = v
        end)

        local chThickAnchor, chThickSl = MakeTHSlider("Nameplate Thickness", distAnchor, 2, 10, 1, cfg.crosshairThickness or 4, "%dpx", function(v)
            cfg.crosshairThickness = v
            RecreateTargetBorders()
        end)

        local thickAnchor, thickSl = MakeTHSlider("Party Border Thickness", chThickAnchor, 2, 10, 1, cfg.thickness, "%dpx", function(v)
            cfg.thickness = v
            RecreateTargetBorders()
        end)

        -- Reset to Defaults button
        local resetBtn = ns.CreateThemedButton(content, "Reset to Defaults", 130, 24, "danger")
        resetBtn:SetPoint("TOPLEFT", thickAnchor, "BOTTOMLEFT", 0, -14)
        resetBtn:SetScript("OnClick", function()
            local d = defaults.targetHighlightConfig
            cfg.colorR, cfg.colorG, cfg.colorB = d.colorR, d.colorG, d.colorB
            cfg.enemyColor = d.enemyColor
            cfg.enemyColorR, cfg.enemyColorG, cfg.enemyColorB = d.enemyColorR, d.enemyColorG, d.enemyColorB
            cfg.animSpeed = d.animSpeed
            cfg.startDistance = d.startDistance
            cfg.crosshairThickness = d.crosshairThickness
            cfg.thickness = d.thickness
            cfg.npTargetStyle = d.npTargetStyle
            cfg.partyHighlight = d.partyHighlight
            -- Update all controls
            swatchTex:SetColorTexture(cfg.colorR, cfg.colorG, cfg.colorB, 1)
            enemyCB:SetChecked(cfg.enemyColor)
            eSwatchTex:SetColorTexture(cfg.enemyColorR, cfg.enemyColorG, cfg.enemyColorB, 1)
            UpdateEnemyColorState()
            speedSl:SetValue(cfg.animSpeed)
            distSl:SetValue(cfg.startDistance)
            chThickSl:SetValue(cfg.crosshairThickness)
            thickSl:SetValue(cfg.thickness)
            partyCB:SetChecked(cfg.partyHighlight)
            RefreshNpTStyleButtons()
            RecreateTargetBorders()
        end)
        thChildren[#thChildren + 1] = resetBtn

        -- Grey-out config controls when Target Highlighting is off
        local function UpdateTHChildState()
            local on = SKToolsDB.targetHighlight
            local alpha = on and 1.0 or 0.4
            for _, child in ipairs(thChildren) do
                if child.SetEnabled then child:SetEnabled(on) end
                child:SetAlpha(alpha)
            end
            if on then UpdateEnemyColorState() end
        end

        thCB:HookScript("OnClick", function() UpdateTHChildState() end)
        UpdateTHChildState()

        thCard:SetPoint("BOTTOM", resetBtn, "BOTTOM", 0, -8)
        thCard:SetPoint("RIGHT", content, "RIGHT", -8, 0)
        SetLast(resetBtn)
    end
end


-----------------------------
-- Nameplates Combat Settings Builder
-----------------------------
function ns.BuildNameplateCombatSettings(content, csSyncControls)
    local NAMEPLATE_STYLES = ns.NAMEPLATE_STYLES
    local defaults = ns.defaults

    local npAnchor = ns.CreateTabTitle(content, "Nameplates", "Nameplate indicator styles and target highlighting.")

    local npCard = ns.CreateSectionCard(content)
    npCard:SetPoint("TOPLEFT", npAnchor, "BOTTOMLEFT", -4, -12)
    local AddCB, GetLast, SetLast = ns.MakeCombatCBFactory(content, npAnchor, csSyncControls)

    local npParentCB = AddCB("NamePlates", "namePlates", "Nameplate Indicators",
        "Adds colored indicators around nameplates. Red for enemies, green for allies, purple for enemy NPCs.",
        function(enabled)
            pcall(SetNameplates)
        end)

    local csStyleBtns = {}
    do
        local lastEl = GetLast()
        local styleLabel = content:CreateFontString(nil, "ARTWORK", "GameFontNormal")
        styleLabel:SetPoint("TOPLEFT", lastEl, "BOTTOMLEFT", 0, -12)
        styleLabel:SetText("Indicator Styles  |cff888888(select multiple)|r")
        styleLabel:SetTextColor(0.9, 0.9, 0.9)
        local prevBtn
        local function RefreshCSStyleButtons()
            local styles = SKToolsDB.namePlateStyles or DEFAULT_STYLES
            for _, b in ipairs(csStyleBtns) do
                if styles[b.styleIndex] then
                    b:SetBackdropColor(0, ns.CYAN.g * 0.15, ns.CYAN.b * 0.15, 1)
                    b:SetBackdropBorderColor(ns.CYAN.r, ns.CYAN.g, ns.CYAN.b, 0.6)
                    b.text:SetTextColor(ns.CYAN.r, ns.CYAN.g, ns.CYAN.b)
                else
                    b:SetBackdropColor(0.1, 0.1, 0.12, 0.8)
                    b:SetBackdropBorderColor(0.3, 0.3, 0.35, 0.6)
                    b.text:SetTextColor(0.7, 0.7, 0.7)
                end
            end
        end
        for i, style in ipairs(NAMEPLATE_STYLES) do
            local btn = CreateFrame("Button", nil, content, "BackdropTemplate")
            btn:SetSize(68, 26)
            btn:SetBackdrop(ns.BACKDROP_PILL)
            if i == 1 then btn:SetPoint("TOPLEFT", styleLabel, "BOTTOMLEFT", 0, -6)
            else btn:SetPoint("LEFT", prevBtn, "RIGHT", 4, 0) end
            btn.text = btn:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
            btn.text:SetPoint("CENTER")
            btn.text:SetText(style.label)
            btn.styleIndex = i
            btn:SetScript("OnClick", function(self)
                if not SKToolsDB.namePlateStyles then SKToolsDB.namePlateStyles = { [1] = true } end
                local styles = SKToolsDB.namePlateStyles
                if styles[self.styleIndex] then
                    local count = 0
                    for _, v in pairs(styles) do if v then count = count + 1 end end
                    if count <= 1 then return end
                    styles[self.styleIndex] = nil
                else styles[self.styleIndex] = true end
                RefreshCSStyleButtons()
                pcall(ForceResetAllIndicators)
            end)
            btn:SetScript("OnEnter", function(self)
                local styles = SKToolsDB.namePlateStyles or {}
                if not styles[self.styleIndex] then
                    self:SetBackdropColor(0.15, 0.15, 0.18, 0.9)
                    self.text:SetTextColor(1, 1, 1)
                end
            end)
            btn:SetScript("OnLeave", function(self)
                local styles = SKToolsDB.namePlateStyles or {}
                if not styles[self.styleIndex] then
                    self:SetBackdropColor(0.1, 0.1, 0.12, 0.8)
                    self.text:SetTextColor(0.7, 0.7, 0.7)
                end
            end)
            csStyleBtns[#csStyleBtns + 1] = btn
            prevBtn = btn
        end
        RefreshCSStyleButtons()
        table.insert(csSyncControls, { type = "custom", refresh = RefreshCSStyleButtons })
        local styleSpacer = content:CreateFontString(nil, "ARTWORK")
        styleSpacer:SetPoint("TOPLEFT", csStyleBtns[1], "BOTTOMLEFT", 0, -4)
        styleSpacer:SetHeight(1)
        SetLast(styleSpacer)
    end

    -- Close Nameplate Indicators card
    npCard:SetPoint("BOTTOM", GetLast(), "BOTTOM", 0, -8)
    npCard:SetPoint("RIGHT", content, "RIGHT", -8, 0)

    -- Target Highlighting section
    local csThHeader = ns.AddSectionHeader(content, GetLast(), "Target Highlighting", false)
    local csThCard = ns.CreateSectionCard(content)
    csThCard:SetPoint("TOPLEFT", csThHeader, "TOPLEFT", -8, 8)

    do
        local cfg = SKToolsDB.targetHighlightConfig
        local thChildren = {}

        local thCB = ns.CreateToggleSwitch(content, "SKToolsCS_TargetHighlight")
        thCB:SetPoint("TOPLEFT", csThHeader, "BOTTOMLEFT", 0, -12)
        thCB.Text:SetText("Target Highlighting")
        thCB.Text:SetTextColor(0.9, 0.9, 0.9)
        thCB:SetChecked(SKToolsDB.targetHighlight)
        thCB:SetScript("OnClick", function(self)
            SKToolsDB.targetHighlight = self:GetChecked()
            pcall(SetNameplates)
            pcall(ForceResetAllIndicators)
        end)
        table.insert(csSyncControls, { type = "checkbox", widget = thCB, key = "targetHighlight" })

        -- Party Frame Highlight toggle (child of Target Highlighting)
        local csPartyCB = ns.CreateToggleSwitch(content)
        csPartyCB:SetPoint("TOPLEFT", thCB, "BOTTOMLEFT", 16, -6)
        csPartyCB.Text:SetText("Party Frame Highlight")
        csPartyCB.Text:SetFontObject("GameFontHighlightSmall")
        csPartyCB.Text:SetTextColor(0.9, 0.9, 0.9)
        csPartyCB:SetChecked(cfg.partyHighlight)
        csPartyCB:SetScript("OnClick", function(self)
            cfg.partyHighlight = self:GetChecked()
            pcall(RecreateTargetBorders)
        end)
        thChildren[#thChildren + 1] = csPartyCB

        -- Nameplate target style selector
        local npStyleLabel = content:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        npStyleLabel:SetPoint("TOPLEFT", csPartyCB, "BOTTOMLEFT", -16, -10)
        npStyleLabel:SetText("Nameplate Style")
        npStyleLabel:SetTextColor(0.9, 0.9, 0.9)
        thChildren[#thChildren + 1] = npStyleLabel

        local CS_NP_TARGET_STYLES = { {label = "Crosshair", idx = 1}, {label = "Border", idx = 2}, {label = "None", idx = 0} }
        local csNpTStyleBtns = {}
        local prevCsNpTBtn

        local function RefreshCsNpTStyleButtons()
            local cur = cfg.npTargetStyle or 1
            for _, b in ipairs(csNpTStyleBtns) do
                if b.styleIdx == cur then
                    b:SetBackdropColor(0, ns.CYAN.g * 0.15, ns.CYAN.b * 0.15, 1)
                    b:SetBackdropBorderColor(ns.CYAN.r, ns.CYAN.g, ns.CYAN.b, 0.6)
                    b.text:SetTextColor(ns.CYAN.r, ns.CYAN.g, ns.CYAN.b)
                else
                    b:SetBackdropColor(0.1, 0.1, 0.12, 0.8)
                    b:SetBackdropBorderColor(0.3, 0.3, 0.35, 0.6)
                    b.text:SetTextColor(0.7, 0.7, 0.7)
                end
            end
        end

        for i, style in ipairs(CS_NP_TARGET_STYLES) do
            local btn = CreateFrame("Button", nil, content, "BackdropTemplate")
            btn:SetSize(84, 26)
            btn:SetBackdrop(ns.BACKDROP_PILL)
            if i == 1 then
                btn:SetPoint("TOPLEFT", npStyleLabel, "BOTTOMLEFT", 0, -6)
            else
                btn:SetPoint("LEFT", prevCsNpTBtn, "RIGHT", 4, 0)
            end
            btn.text = btn:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
            btn.text:SetPoint("CENTER")
            btn.text:SetText(style.label)
            btn.styleIdx = style.idx
            btn:SetScript("OnClick", function(self)
                cfg.npTargetStyle = self.styleIdx
                RefreshCsNpTStyleButtons()
                pcall(RecreateTargetBorders)
            end)
            csNpTStyleBtns[#csNpTStyleBtns + 1] = btn
            thChildren[#thChildren + 1] = btn
            prevCsNpTBtn = btn
        end
        RefreshCsNpTStyleButtons()

        -- Color swatch
        local colorLabel = content:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        colorLabel:SetPoint("TOPLEFT", csNpTStyleBtns[1], "BOTTOMLEFT", 0, -8)
        colorLabel:SetText("Color")
        colorLabel:SetTextColor(0.9, 0.9, 0.9)
        thChildren[#thChildren + 1] = colorLabel

        local colorSwatch = CreateFrame("Button", nil, content, "BackdropTemplate")
        colorSwatch:SetSize(24, 24)
        colorSwatch:SetPoint("LEFT", colorLabel, "RIGHT", 8, 0)
        colorSwatch:SetBackdrop(ns.BACKDROP_CONTROL)
        colorSwatch:SetBackdropBorderColor(0.3, 0.3, 0.35, 0.8)
        local swatchBg = colorSwatch:CreateTexture(nil, "BACKGROUND")
        swatchBg:SetPoint("TOPLEFT", 3, -3)
        swatchBg:SetPoint("BOTTOMRIGHT", -3, 3)
        swatchBg:SetColorTexture(0, 0, 0, 1)
        local swatchTex = colorSwatch:CreateTexture(nil, "ARTWORK")
        swatchTex:SetPoint("TOPLEFT", 3, -3)
        swatchTex:SetPoint("BOTTOMRIGHT", -3, 3)
        swatchTex:SetColorTexture(cfg.colorR, cfg.colorG, cfg.colorB, 1)
        colorSwatch.tex = swatchTex
        colorSwatch:SetScript("OnClick", function()
            if InCombatLockdown() then
                print("|cff00E5EESKTools|r Color picker unavailable in combat.")
                return
            end
            local prev = { r = cfg.colorR, g = cfg.colorG, b = cfg.colorB }
            local function SetColor(r, g, b)
                cfg.colorR, cfg.colorG, cfg.colorB = r, g, b
                swatchTex:SetColorTexture(r, g, b, 1)
                pcall(RecreateTargetBorders)
            end
            ColorPickerFrame:SetupColorPickerAndShow({
                r = cfg.colorR, g = cfg.colorG, b = cfg.colorB,
                swatchFunc = function()
                    local r, g, b = ColorPickerFrame:GetColorRGB()
                    SetColor(r, g, b)
                end,
                cancelFunc = function() SetColor(prev.r, prev.g, prev.b) end,
            })
        end)
        colorSwatch:SetScript("OnEnter", function(self)
            self:SetBackdropBorderColor(ns.CYAN.r, ns.CYAN.g, ns.CYAN.b, 0.8)
            if InCombatLockdown() then
                GameTooltip:SetOwner(self, "ANCHOR_RIGHT")
                GameTooltip:SetText("Color picker unavailable in combat", 1, 0.8, 0)
                GameTooltip:Show()
            end
        end)
        colorSwatch:SetScript("OnLeave", function(self)
            self:SetBackdropBorderColor(0.3, 0.3, 0.35, 0.8)
            GameTooltip:Hide()
        end)
        thChildren[#thChildren + 1] = colorSwatch

        -- Enemy color toggle + swatch
        local csEnemyCB = ns.CreateToggleSwitch(content)
        csEnemyCB:SetPoint("TOPLEFT", colorLabel, "BOTTOMLEFT", 0, -8)
        csEnemyCB.Text:SetText("Separate Enemy Color")
        csEnemyCB.Text:SetFontObject("GameFontHighlightSmall")
        csEnemyCB.Text:SetTextColor(0.9, 0.9, 0.9)
        csEnemyCB:SetChecked(cfg.enemyColor)
        thChildren[#thChildren + 1] = csEnemyCB

        local csEnemyColorLabel = content:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        csEnemyColorLabel:SetPoint("TOPLEFT", csEnemyCB, "BOTTOMLEFT", 0, -6)
        csEnemyColorLabel:SetText("Enemy Color")
        csEnemyColorLabel:SetTextColor(0.9, 0.9, 0.9)
        thChildren[#thChildren + 1] = csEnemyColorLabel

        local csEnemySwatch = CreateFrame("Button", nil, content, "BackdropTemplate")
        csEnemySwatch:SetSize(24, 24)
        csEnemySwatch:SetPoint("LEFT", csEnemyColorLabel, "RIGHT", 8, 0)
        csEnemySwatch:SetBackdrop(ns.BACKDROP_CONTROL)
        csEnemySwatch:SetBackdropBorderColor(0.3, 0.3, 0.35, 0.8)
        local csEBg = csEnemySwatch:CreateTexture(nil, "BACKGROUND")
        csEBg:SetPoint("TOPLEFT", 3, -3)
        csEBg:SetPoint("BOTTOMRIGHT", -3, 3)
        csEBg:SetColorTexture(0, 0, 0, 1)
        local csESwatchTex = csEnemySwatch:CreateTexture(nil, "ARTWORK")
        csESwatchTex:SetPoint("TOPLEFT", 3, -3)
        csESwatchTex:SetPoint("BOTTOMRIGHT", -3, 3)
        csESwatchTex:SetColorTexture(cfg.enemyColorR, cfg.enemyColorG, cfg.enemyColorB, 1)
        csEnemySwatch.tex = csESwatchTex
        csEnemySwatch:SetScript("OnClick", function()
            if InCombatLockdown() then
                print("|cff00E5EESKTools|r Color picker unavailable in combat.")
                return
            end
            if not cfg.enemyColor then return end
            local prev = { r = cfg.enemyColorR, g = cfg.enemyColorG, b = cfg.enemyColorB }
            local function SetEColor(r, g, b)
                cfg.enemyColorR, cfg.enemyColorG, cfg.enemyColorB = r, g, b
                csESwatchTex:SetColorTexture(r, g, b, 1)
                pcall(RecreateTargetBorders)
            end
            ColorPickerFrame:SetupColorPickerAndShow({
                r = cfg.enemyColorR, g = cfg.enemyColorG, b = cfg.enemyColorB,
                swatchFunc = function()
                    local r, g, b = ColorPickerFrame:GetColorRGB()
                    SetEColor(r, g, b)
                end,
                cancelFunc = function() SetEColor(prev.r, prev.g, prev.b) end,
            })
        end)
        csEnemySwatch:SetScript("OnEnter", function(self)
            self:SetBackdropBorderColor(ns.CYAN.r, ns.CYAN.g, ns.CYAN.b, 0.8)
            if InCombatLockdown() then
                GameTooltip:SetOwner(self, "ANCHOR_RIGHT")
                GameTooltip:SetText("Color picker unavailable in combat", 1, 0.8, 0)
                GameTooltip:Show()
            end
        end)
        csEnemySwatch:SetScript("OnLeave", function(self)
            self:SetBackdropBorderColor(0.3, 0.3, 0.35, 0.8)
            GameTooltip:Hide()
        end)
        thChildren[#thChildren + 1] = csEnemySwatch

        local function UpdateCSEnemyColorState()
            local on = cfg.enemyColor
            csEnemyColorLabel:SetAlpha(on and 1.0 or 0.4)
            csEnemySwatch:SetAlpha(on and 1.0 or 0.4)
        end
        csEnemyCB:SetScript("OnClick", function(self)
            cfg.enemyColor = self:GetChecked()
            UpdateCSEnemyColorState()
            pcall(RecreateTargetBorders)
        end)
        UpdateCSEnemyColorState()

        -- Slider helper with +/- buttons
        local function MakeTHSlider(label, anchorTo, min, max, step, initial, fmt, onChange)
            local lbl = content:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
            lbl:SetPoint("TOPLEFT", anchorTo, "BOTTOMLEFT", 0, -14)
            lbl:SetText(label)
            lbl:SetTextColor(0.9, 0.9, 0.9)
            thChildren[#thChildren + 1] = lbl
            local minus = ns.CreateThemedButton(content, "-", 24, 24, "secondary")
            minus:SetPoint("TOPLEFT", lbl, "BOTTOMLEFT", 0, -12)
            thChildren[#thChildren + 1] = minus
            local sl = CreateFrame("Slider", nil, content, "OptionsSliderTemplate")
            sl:SetPoint("LEFT", minus, "RIGHT", 4, 0)
            sl:SetWidth(140)
            sl:SetMinMaxValues(min, max)
            sl:SetValueStep(step)
            sl:SetObeyStepOnDrag(true)
            sl:SetValue(initial)
            ns.StyleSlider(sl)
            sl.Low:SetText(string.format(fmt, min))
            sl.High:SetText(string.format(fmt, max))
            sl.Text:SetText(string.format(fmt, initial))
            sl:SetScript("OnValueChanged", function(self, value)
                value = math.floor(value / step + 0.5) * step
                self.Text:SetText(string.format(fmt, value))
                onChange(value)
            end)
            thChildren[#thChildren + 1] = sl
            local plus = ns.CreateThemedButton(content, "+", 24, 24, "secondary")
            plus:SetPoint("LEFT", sl, "RIGHT", 4, 0)
            thChildren[#thChildren + 1] = plus
            minus:SetScript("OnClick", function()
                local val = sl:GetValue() - step
                if val >= min then sl:SetValue(val) end
            end)
            plus:SetScript("OnClick", function()
                local val = sl:GetValue() + step
                if val <= max then sl:SetValue(val) end
            end)
            local spacer = content:CreateFontString(nil, "ARTWORK")
            spacer:SetPoint("TOPLEFT", minus, "BOTTOMLEFT", 0, -4)
            spacer:SetHeight(1)
            return spacer, sl
        end

        local speedAnchor, speedSl = MakeTHSlider("Animation Speed", csEnemyColorLabel, 0.1, 1.0, 0.05, cfg.animSpeed, "%.2fs", function(v) cfg.animSpeed = v end)
        local distAnchor, distSl = MakeTHSlider("Start Distance", speedAnchor, 10, 250, 10, cfg.startDistance, "%dpx", function(v) cfg.startDistance = v end)
        local chThickAnchor, chThickSl = MakeTHSlider("Nameplate Thickness", distAnchor, 2, 10, 1, cfg.crosshairThickness or 4, "%dpx", function(v)
            cfg.crosshairThickness = v; pcall(RecreateTargetBorders)
        end)
        local thickAnchor, thickSl = MakeTHSlider("Party Border Thickness", chThickAnchor, 2, 10, 1, cfg.thickness, "%dpx", function(v)
            cfg.thickness = v; pcall(RecreateTargetBorders)
        end)

        local resetBtn = ns.CreateThemedButton(content, "Reset to Defaults", 130, 24, "danger")
        resetBtn:SetPoint("TOPLEFT", thickAnchor, "BOTTOMLEFT", 0, -14)
        resetBtn:SetScript("OnClick", function()
            local d = defaults.targetHighlightConfig
            cfg.colorR, cfg.colorG, cfg.colorB = d.colorR, d.colorG, d.colorB
            cfg.enemyColor = d.enemyColor
            cfg.enemyColorR, cfg.enemyColorG, cfg.enemyColorB = d.enemyColorR, d.enemyColorG, d.enemyColorB
            cfg.animSpeed = d.animSpeed
            cfg.startDistance = d.startDistance
            cfg.crosshairThickness = d.crosshairThickness
            cfg.thickness = d.thickness
            cfg.npTargetStyle = d.npTargetStyle
            cfg.partyHighlight = d.partyHighlight
            swatchTex:SetColorTexture(cfg.colorR, cfg.colorG, cfg.colorB, 1)
            csEnemyCB:SetChecked(cfg.enemyColor)
            csESwatchTex:SetColorTexture(cfg.enemyColorR, cfg.enemyColorG, cfg.enemyColorB, 1)
            UpdateCSEnemyColorState()
            speedSl:SetValue(cfg.animSpeed)
            distSl:SetValue(cfg.startDistance)
            chThickSl:SetValue(cfg.crosshairThickness)
            thickSl:SetValue(cfg.thickness)
            csPartyCB:SetChecked(cfg.partyHighlight)
            RefreshCsNpTStyleButtons()
            pcall(RecreateTargetBorders)
        end)
        thChildren[#thChildren + 1] = resetBtn

        local function UpdateCSTHChildState()
            local on = SKToolsDB.targetHighlight
            local alpha = on and 1.0 or 0.4
            for _, child in ipairs(thChildren) do
                if child.SetEnabled then child:SetEnabled(on) end
                child:SetAlpha(alpha)
            end
            if on then UpdateCSEnemyColorState() end
        end
        thCB:HookScript("OnClick", UpdateCSTHChildState)
        table.insert(csSyncControls, { type = "custom", refresh = function()
            swatchTex:SetColorTexture(cfg.colorR, cfg.colorG, cfg.colorB, 1)
            csEnemyCB:SetChecked(cfg.enemyColor)
            csESwatchTex:SetColorTexture(cfg.enemyColorR, cfg.enemyColorG, cfg.enemyColorB, 1)
            UpdateCSEnemyColorState()
            speedSl:SetValue(cfg.animSpeed)
            distSl:SetValue(cfg.startDistance)
            chThickSl:SetValue(cfg.crosshairThickness or 4)
            thickSl:SetValue(cfg.thickness)
            csPartyCB:SetChecked(cfg.partyHighlight)
            RefreshCsNpTStyleButtons()
            UpdateCSTHChildState()
        end })
        csThCard:SetPoint("BOTTOM", resetBtn, "BOTTOM", 0, -8)
        csThCard:SetPoint("RIGHT", content, "RIGHT", -8, 0)
        SetLast(resetBtn)
    end
end
