-- SKTools by Silverkitty

local _, ns = ...

-----------------------------
-- Minimap Button
-----------------------------
local minimapButton

local function CreateMinimapButton()
    local SIZE = 34
    local BORDER = 2
    local INNER = SIZE - BORDER * 2
    local ICON_SIZE = 24

    local btn = CreateFrame("Button", "SKToolsMinimapButton", UIParent)
    btn:SetSize(SIZE, SIZE)
    btn:SetFrameStrata("MEDIUM")
    btn:SetFrameLevel(8)
    btn:SetMovable(true)
    btn:SetClampedToScreen(true)

    -- Circular masks
    local outerMask = btn:CreateMaskTexture()
    outerMask:SetAllPoints()
    outerMask:SetTexture("Interface\\CHARACTERFRAME\\TempPortraitAlphaMask", "CLAMPTOBLACKADDITIVE", "CLAMPTOBLACKADDITIVE")

    local innerMask = btn:CreateMaskTexture()
    innerMask:SetSize(INNER, INNER)
    innerMask:SetPoint("CENTER")
    innerMask:SetTexture("Interface\\CHARACTERFRAME\\TempPortraitAlphaMask", "CLAMPTOBLACKADDITIVE", "CLAMPTOBLACKADDITIVE")

    -- Border ring (full-size gray circle, visible as 2px ring)
    local ring = btn:CreateTexture(nil, "BACKGROUND")
    ring:SetAllPoints()
    ring:SetColorTexture(0.45, 0.45, 0.45, 1)
    ring:AddMaskTexture(outerMask)

    -- Dark fill (inner circle)
    local fill = btn:CreateTexture(nil, "BORDER")
    fill:SetSize(INNER, INNER)
    fill:SetPoint("CENTER")
    fill:SetColorTexture(0.07, 0.07, 0.07, 1)
    fill:AddMaskTexture(innerMask)

    -- Icon (circular masked)
    local icon = btn:CreateTexture(nil, "ARTWORK")
    icon:SetSize(ICON_SIZE, ICON_SIZE)
    icon:SetPoint("CENTER")
    icon:SetTexture("Interface\\Icons\\Achievement_Featsofstrength_Gladiator_08")
    icon:SetTexCoord(0.08, 0.92, 0.08, 0.92)
    icon:AddMaskTexture(innerMask)

    -- Hover highlight
    local highlight = btn:CreateTexture(nil, "HIGHLIGHT")
    highlight:SetAllPoints()
    highlight:SetColorTexture(1, 1, 1, 0.1)
    highlight:AddMaskTexture(outerMask)

    -- Position: next to LFG eye by default, or saved position
    local pos = SKToolsDB.buttonPos
    if pos then
        btn:SetPoint("BOTTOMRIGHT", UIParent, "BOTTOMRIGHT", pos.x, pos.y)
    elseif QueueStatusButton then
        btn:SetPoint("RIGHT", QueueStatusButton, "LEFT", -2, 0)
    else
        btn:SetPoint("BOTTOMRIGHT", UIParent, "BOTTOMRIGHT", -260, 5)
    end

    -- Free drag (saves position relative to screen bottom-right)
    btn:RegisterForDrag("LeftButton")
    btn:SetScript("OnDragStart", function(self) self:StartMoving() end)
    btn:SetScript("OnDragStop", function(self)
        self:StopMovingOrSizing()
        local xOff = self:GetRight() - UIParent:GetRight()
        local yOff = self:GetBottom()
        SKToolsDB.buttonPos = { x = xOff, y = yOff }
        self:ClearAllPoints()
        self:SetPoint("BOTTOMRIGHT", UIParent, "BOTTOMRIGHT", xOff, yOff)
    end)

    btn:RegisterForClicks("LeftButtonUp", "RightButtonUp", "MiddleButtonUp")
    btn:SetScript("OnClick", function(self, button)
        if button == "RightButton" then
            SlashCmdList["SKTOOLS"]("")
        elseif button == "MiddleButton" then
            ns.ExecuteButtonAction(SKToolsDB.buttonMiddleClick or "notes")
        elseif IsShiftKeyDown() then
            ns.ExecuteButtonAction(SKToolsDB.buttonShiftClick or "pveHistory")
        else
            ns.ExecuteButtonAction(SKToolsDB.buttonLeftClick or "pvpHistory")
        end
    end)

    btn:SetScript("OnEnter", function(self)
        GameTooltip:SetOwner(self, "ANCHOR_LEFT")
        GameTooltip:AddLine("SKTools")

        local modeLabels = {
            arena2v2 = "2v2 Arena", arena3v3 = "3v3 Arena", shuffle = "Shuffle",
            blitz = "Blitz", rbg = "RBG", randomBG = "Battleground",
            epicBG = "Epic BG", skirmish = "Skirmish",
        }

        local matches = SKToolsArenaDB and SKToolsArenaDB.matches
        if matches and #matches > 0 then
            GameTooltip:AddLine(" ")
            GameTooltip:AddLine("Recent Matches", 0.63, 0.63, 0.63)

            local count = math.min(10, #matches)
            local wins, losses = 0, 0

            for i = 1, count do
                local m = matches[i]

                local result
                if m.mode == "shuffle" and m.roundWins then
                    local color = m.won and "|cff00ff00" or "|cffff4444"
                    result = color .. m.roundWins .. "-" .. (m.roundLosses or (6 - m.roundWins)) .. "|r"
                elseif m.won then
                    result = "|cff00ff00W|r"
                else
                    result = "|cffff4444L|r"
                end

                local modeLabel = modeLabels[m.mode] or m.mode or "?"

                local rightStr
                if m.isRated and m.ratingChange then
                    local sign = m.ratingChange >= 0 and "+" or ""
                    local rColor = m.ratingChange >= 0 and "|cff00ff00" or "|cffff4444"
                    rightStr = rColor .. sign .. m.ratingChange .. "|r"
                elseif m.mode == "skirmish" then
                    rightStr = ""
                else
                    local hkSuffix = (m.honorableKills and m.honorableKills > 0) and (" (" .. m.honorableKills .. ")" ) or ""
                    rightStr = "|cff888888" .. (m.kills or 0) .. "/" .. (m.deaths or 0) .. hkSuffix .. "|r"
                end

                GameTooltip:AddDoubleLine(
                    result .. "  " .. modeLabel,
                    rightStr,
                    1, 1, 1,
                    1, 1, 1
                )

                if m.won then wins = wins + 1 else losses = losses + 1 end
            end

            GameTooltip:AddLine(" ")
            local summaryColor = wins >= losses and "|cff00ff00" or "|cffff4444"
            GameTooltip:AddLine(
                summaryColor .. wins .. "W|r |cff888888-|r |cffff4444" .. losses .. "L|r" ..
                "  |cff888888(last " .. count .. ")|r",
                1, 1, 1
            )
        end

        GameTooltip:AddLine(" ")
        local function ActionLabel(key)
            for _, opt in ipairs(ns.BUTTON_ACTIONS) do
                if opt.value == key then return opt.label end
            end
            return key
        end
        GameTooltip:AddLine("|cffffff00Left-click:|r " .. ActionLabel(SKToolsDB.buttonLeftClick or "pvpHistory"), 1, 1, 1)
        GameTooltip:AddLine("|cffffff00Shift-click:|r " .. ActionLabel(SKToolsDB.buttonShiftClick or "pveHistory"), 1, 1, 1)
        GameTooltip:AddLine("|cffffff00Middle-click:|r " .. ActionLabel(SKToolsDB.buttonMiddleClick or "notes"), 1, 1, 1)
        GameTooltip:AddLine("|cffffff00Right-click:|r Settings", 1, 1, 1)
        GameTooltip:AddLine("Drag to reposition", 0.7, 0.7, 0.7)
        GameTooltip:Show()
    end)
    btn:SetScript("OnLeave", function()
        GameTooltip:Hide()
    end)

    minimapButton = btn

    if not SKToolsDB.showMinimapButton then
        btn:Hide()
    end
end

-----------------------------
-- Settings Panel
-----------------------------
local skMainCategory
local skMainOuter
local skMainBuilt = false
local combatSettingsFrame
local CreateSettingsPanel  -- forward declare

local function RegisterSettingsCategory()
    skMainOuter = CreateFrame("Frame")
    skMainOuter.name = "SKTools"
    skMainCategory = Settings.RegisterCanvasLayoutCategory(skMainOuter, "SKTools")
    Settings.RegisterAddOnCategory(skMainCategory)
    skMainOuter:SetScript("OnShow", function(self)
        if not skMainBuilt then
            skMainBuilt = true
            CreateSettingsPanel()
            -- Re-open on next frame so Blizzard re-layouts with the built content
            C_Timer.After(0, function()
                if skMainCategory then
                    Settings.OpenToCategory(skMainCategory:GetID())
                end
            end)
        end
        if ns.RefreshMainNavStatus then ns.RefreshMainNavStatus() end
        if ns.RefreshMainNotesChildState then ns.RefreshMainNotesChildState() end
    end)
end

CreateSettingsPanel = function()
    local MakeTabContent = ns.MakeTabContent
    local MakeCheckboxFactory = ns.MakeCheckboxFactory

    local outer = skMainOuter

    --------------------------------
    -- Side Nav + Content (shared helper)
    --------------------------------
    local _, contentContainer, contentPanels, SelectTab, mainNavItems = ns.BuildSideNav(
        outer,
        { outer, 0, 0 },    -- navAnchorTop
        { outer, 0, 0 },    -- navAnchorBottom
        { outer, 176, 0 },  -- contentTL
        { outer, 0, 0 }     -- contentBR
    )

    --------------------------------
    -- General Tab
    --------------------------------
    do
        local genSF, genContent = MakeTabContent(outer, contentContainer)
        contentPanels["general"] = genSF

        local genAnchor = ns.CreateTabTitle(genContent, "SKTools", "Quality-of-life tools for PvE and PvP.")

        -- SKTools Button section (card-wrapped)
        local btnHeader = ns.AddSectionHeader(genContent, genAnchor, "SKTools Button", false)
        local btnCard = ns.CreateSectionCard(genContent)
        btnCard:SetPoint("TOPLEFT", btnHeader, "TOPLEFT", -8, 8)

        local AddCB, GetLast, SetLast = MakeCheckboxFactory(genContent, btnHeader)

        AddCB("MinimapBtn", "showMinimapButton", "Show SKTools Button",
            "Shows a button near the action bars. Left-click for match history, right-click for settings. Also: /history",
            function(enabled)
                if minimapButton then
                    if enabled then minimapButton:Show() else minimapButton:Hide() end
                end
            end)

        do
            local lastEl = GetLast()
            local resetBtn = ns.CreateThemedButton(genContent, "Reset Icon Position", 150, 24, "secondary")
            resetBtn:SetPoint("TOPLEFT", lastEl, "BOTTOMLEFT", 0, -8)
            resetBtn:SetScript("OnClick", function()
                SKToolsDB.buttonPos = nil
                if minimapButton then
                    minimapButton:ClearAllPoints()
                    if QueueStatusButton then
                        minimapButton:SetPoint("RIGHT", QueueStatusButton, "LEFT", -2, 0)
                    else
                        minimapButton:SetPoint("BOTTOMRIGHT", UIParent, "BOTTOMRIGHT", -260, 5)
                    end
                end
            end)
            SetLast(resetBtn)
        end

        -- Button click action dropdowns
        do
            local clickLabel = genContent:CreateFontString(nil, "ARTWORK", "GameFontNormal")
            clickLabel:SetPoint("TOPLEFT", GetLast(), "BOTTOMLEFT", 0, -14)
            clickLabel:SetText("Button Click Actions")
            clickLabel:SetTextColor(0.7, 0.7, 0.7)

            local actions = ns.BUTTON_ACTIONS

            local DD_LEFT = 90  -- align all dropdowns to same X offset

            local leftLabel = genContent:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
            leftLabel:SetPoint("TOPLEFT", clickLabel, "BOTTOMLEFT", 0, -10)
            leftLabel:SetText("Left-click")
            leftLabel:SetTextColor(0.55, 0.55, 0.6)
            local leftDD = ns.CreateThemedDropdown(genContent, 140, actions, SKToolsDB.buttonLeftClick, function(val)
                SKToolsDB.buttonLeftClick = val
            end)
            leftDD:SetPoint("LEFT", clickLabel, "LEFT", DD_LEFT, 0)
            leftDD:SetPoint("TOP", leftLabel, "TOP", 0, 0)

            local shiftLabel = genContent:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
            shiftLabel:SetPoint("LEFT", clickLabel, "LEFT", 0, 0)
            shiftLabel:SetPoint("TOP", leftDD, "BOTTOM", 0, -6)
            shiftLabel:SetText("Shift-click")
            shiftLabel:SetTextColor(0.55, 0.55, 0.6)
            local shiftDD = ns.CreateThemedDropdown(genContent, 140, actions, SKToolsDB.buttonShiftClick, function(val)
                SKToolsDB.buttonShiftClick = val
            end)
            shiftDD:SetPoint("LEFT", clickLabel, "LEFT", DD_LEFT, 0)
            shiftDD:SetPoint("TOP", shiftLabel, "TOP", 0, 0)

            local midLabel = genContent:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
            midLabel:SetPoint("LEFT", clickLabel, "LEFT", 0, 0)
            midLabel:SetPoint("TOP", shiftDD, "BOTTOM", 0, -6)
            midLabel:SetText("Middle-click")
            midLabel:SetTextColor(0.55, 0.55, 0.6)
            local midDD = ns.CreateThemedDropdown(genContent, 140, actions, SKToolsDB.buttonMiddleClick, function(val)
                SKToolsDB.buttonMiddleClick = val
            end)
            midDD:SetPoint("LEFT", clickLabel, "LEFT", DD_LEFT, 0)
            midDD:SetPoint("TOP", midLabel, "TOP", 0, 0)

            SetLast(midDD)
        end

        btnCard:SetPoint("BOTTOM", GetLast(), "BOTTOM", 0, -8)
        btnCard:SetPoint("RIGHT", genContent, "RIGHT", -8, 0)

        local generalLast = ns.BuildGeneralSettings(genContent, GetLast())
        SetLast(generalLast)
        local tweaksLast = ns.BuildTweaksSettings(genContent, GetLast())
        SetLast(tweaksLast)
    end

    --------------------------------
    -- Stats Tab
    --------------------------------
    do
        local statsSF, statsContent = MakeTabContent(outer, contentContainer)
        contentPanels["stats"] = statsSF

        local statsAnchor = ns.CreateTabTitle(statsContent, "Stat Tracker", "Secondary stat diminishing returns and information.")
        ns.BuildStatSettings(statsContent, statsAnchor)
    end

    --------------------------------
    -- Nameplates Tab
    --------------------------------
    do
        local npSF, npContent = MakeTabContent(outer, contentContainer)
        contentPanels["nameplates"] = npSF

        local npAnchor = ns.CreateTabTitle(npContent, "Nameplates", "Nameplate indicator styles and target highlighting.")
        ns.BuildNameplateSettings(npContent, npAnchor)
    end

    --------------------------------
    -- History Tab
    --------------------------------
    do
        local histSF, histContent = MakeTabContent(outer, contentContainer)
        contentPanels["history"] = histSF

        local histAnchor = ns.CreateTabTitle(histContent, "History", "PvP and PvE match history tracking.")

        local AddCB, GetLast, SetLast = MakeCheckboxFactory(histContent, histAnchor)

        AddCB("PvPHistory", "pvpHistory", "PvP Match History",
            "Automatically records arena and battleground results. View with /history or left-click the SKTools button.",
            nil)

        do
            local btnRow = CreateFrame("Frame", nil, histContent)
            btnRow:SetSize(300, 22)
            btnRow:SetPoint("TOPLEFT", GetLast(), "BOTTOMLEFT", 0, -4)

            local openBtn = ns.CreateThemedButton(btnRow, "Open History", 110, 22, "secondary")
            openBtn:SetPoint("LEFT", 0, 0)
            openBtn:SetScript("OnClick", function()
                SKPvPHistory_Toggle()
            end)

            local importBtn = ns.CreateThemedButton(btnRow, "Import from REFlex", 140, 22, "secondary")
            importBtn:SetPoint("LEFT", openBtn, "RIGHT", 6, 0)
            importBtn:SetScript("OnClick", function()
                ns.ImportREFlexData()
            end)

            SetLast(btnRow)
        end

        AddCB("PvEHistory", "pveHistory", "PvE History",
            "Track M+ dungeons, raids, and dungeon runs. View with /pvehistory.",
            nil)

        do
            local openBtn = ns.CreateThemedButton(histContent, "Open PvE History", 130, 22, "secondary")
            openBtn:SetPoint("TOPLEFT", GetLast(), "BOTTOMLEFT", 0, -4)
            openBtn:SetScript("OnClick", function()
                SKPvEHistory_Toggle()
            end)
            SetLast(openBtn)
        end
    end

    --------------------------------
    -- Battleground Tab
    --------------------------------
    do
        local bgSF, bgContent = MakeTabContent(outer, contentContainer)
        contentPanels["battleground"] = bgSF

        local bgAnchor = ns.CreateTabTitle(bgContent, "Battle Intel", "Capture timers and score prediction for battlegrounds.")
        ns.BuildBattleIntelSettings(bgContent, bgAnchor)
    end

    --------------------------------
    -- Notes Tab
    --------------------------------
    do
        local notesSF, notesContent = MakeTabContent(outer, contentContainer)
        contentPanels["notes"] = notesSF

        local notesTabAnchor = ns.CreateTabTitle(notesContent, "Notes", "Scouting notes for arenas, battlegrounds, dungeons, and raids.")

        local AddCB, GetLast, SetLast = MakeCheckboxFactory(notesContent, notesTabAnchor)

        AddCB("ArenaNotes", "arenaNotes", "Notes",
            "Enable the notes system. Browse all notes with /notes.",
            function(enabled)
                if enabled then SKArenaNotesEnable() else SKArenaNotesDisable() end
            end)

        local notesCB = _G["SKToolsArenaNotesCB"]
        local notesAnchor = GetLast()

        -- Sub-toggles (indented children of Notes)
        local SUB_INDENT = 24
        local SUB_DEFS = {
            { name = "NotesArena",   key = "arenaNotesArena",   label = "Auto-show in Arenas",        desc = "Show notes when entering an arena match." },
            { name = "NotesBG",      key = "arenaNotesBG",      label = "Auto-show in Battlegrounds", desc = "Show notes when entering a battleground." },
            { name = "NotesDungeon", key = "arenaNotesDungeon", label = "Auto-show in Dungeons",      desc = "Show notes when entering a dungeon." },
            { name = "NotesRaid",    key = "arenaNotesRaid",    label = "Auto-show in Raids",         desc = "Show notes when entering a raid." },
        }
        local subToggles = {}
        local lastSub = notesAnchor
        for i, def in ipairs(SUB_DEFS) do
            local cb = ns.CreateToggleSwitch(notesContent, "SKTools" .. def.name .. "CB")
            cb:SetPoint("TOPLEFT", lastSub, "BOTTOMLEFT", i == 1 and SUB_INDENT or 0, -8)
            cb.Text:SetText(def.label)
            cb.Text:SetTextColor(0.8, 0.8, 0.8)
            cb:SetChecked(SKToolsDB[def.key])
            cb:SetScript("OnClick", function(self)
                SKToolsDB[def.key] = self:GetChecked()
                if ns.RefreshMainNavStatus then ns.RefreshMainNavStatus() end
            end)
            local desc = notesContent:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
            desc:SetPoint("TOPLEFT", cb.Text, "BOTTOMLEFT", 0, -2)
            desc:SetPoint("RIGHT", notesContent, "RIGHT", -16, 0)
            desc:SetJustifyH("LEFT")
            desc:SetTextColor(0.55, 0.55, 0.6)
            desc:SetText(def.desc)
            subToggles[#subToggles + 1] = cb
            local spacer = CreateFrame("Frame", nil, notesContent)
            spacer:SetSize(1, 1)
            spacer:SetPoint("LEFT", cb, "LEFT", 0, 0)
            spacer:SetPoint("TOP", desc, "BOTTOM", 0, 0)
            lastSub = spacer
        end

        -- Connecting line from parent to sub-toggles
        local connLine = notesContent:CreateTexture(nil, "ARTWORK")
        connLine:SetWidth(1)
        connLine:SetPoint("TOP", subToggles[1], "TOPLEFT", -8, 4)
        connLine:SetPoint("BOTTOM", subToggles[#subToggles], "LEFT", -8, 0)
        connLine:SetColorTexture(ns.COLORS.divider[1], ns.COLORS.divider[2], ns.COLORS.divider[3], 0.30)

        -- Enable/disable sub-toggles based on Notes parent
        local function UpdateNotesChildState()
            local enabled = notesCB:GetChecked()
            for _, cb in ipairs(subToggles) do
                cb:SetEnabled(enabled)
            end
        end
        notesCB:HookScript("OnClick", UpdateNotesChildState)
        ns.RefreshMainNotesChildState = UpdateNotesChildState
        UpdateNotesChildState()

        -- Return to parent's left edge for the Open Notes button
        local subReturnSpacer = CreateFrame("Frame", nil, notesContent)
        subReturnSpacer:SetSize(1, 1)
        subReturnSpacer:SetPoint("LEFT", notesCB, "LEFT", 0, 0)
        subReturnSpacer:SetPoint("TOP", lastSub, "TOP", 0, 0)
        SetLast(subReturnSpacer)

        do
            local openBtn = ns.CreateThemedButton(notesContent, "Open Notes", 100, 22, "secondary")
            openBtn:SetPoint("TOPLEFT", GetLast(), "BOTTOMLEFT", 0, -8)
            openBtn:SetScript("OnClick", function()
                C_Timer.After(0, SKArenaNotes_ToggleBrowser)
            end)
            SetLast(openBtn)
        end
    end

    --------------------------------
    -- PvP Scanner Tab
    --------------------------------
    do
        local titlesSF, titlesContent = MakeTabContent(outer, contentContainer)
        contentPanels["pvptitles"] = titlesSF

        local titlesAnchor = ns.CreateTabTitle(titlesContent, "PvP Scanner", "Scan nearby players for PvP achievements.")
        ns.BuildPvPTitleSettings(titlesContent, titlesAnchor)
    end

    --------------------------------
    -- Macros Tab
    --------------------------------
    do
        local macSF, macContent = MakeTabContent(outer, contentContainer)
        contentPanels["macros"] = macSF

        local macAnchor = ns.CreateTabTitle(macContent, "Smart Macros", "Auto-replace role placeholders with party member names whenever your group changes.")
        ns.BuildMacroSettings(macContent, macAnchor)
    end

    --------------------------------
    -- Profile Tab
    --------------------------------
    do
        local profSF, profContent = MakeTabContent(outer, contentContainer)
        contentPanels["profile"] = profSF

        local profAnchor = ns.CreateTabTitle(profContent, "Profile", "Character and account statistics.")
        ns.BuildProfileSettings(profContent, profAnchor)
    end

    --------------------------------
    -- Shaman Tab (class-gated)
    --------------------------------
    if select(2, UnitClass("player")) == "SHAMAN" then
        local shamSF, shamContent = MakeTabContent(outer, contentContainer)
        contentPanels["shaman"] = shamSF

        local shamAnchor = ns.CreateTabTitle(shamContent, "Totem Tracking", "")
        ns.BuildShamanSettings(shamContent, shamAnchor)
    end

    -- Nav status dots
    local NAV_STATUS_KEYS = {
        general      = {"fastLoot", "chatCopy", "autoVendor", "autoRepair", "pvpTabTarget", "queueTimer", "classColorHealth"},
        stats        = {"statTooltips", "statItemTooltips", "statCompareTooltips"},
        nameplates   = {"namePlates", "targetHighlight"},
        history      = {"pvpHistory", "pveHistory"},
        battleground = {"battleIntel"},
        notes        = {"arenaNotes", "arenaNotesArena", "arenaNotesBG", "arenaNotesDungeon", "arenaNotesRaid"},
        pvptitles    = {"pvpTitles"},
        macros       = {"smartMacros"},
        profile      = true,  -- always on
        shaman       = {"totemTracking"},
    }
    local function RefreshMainNavStatus()
        if not SKToolsDB then return end
        for _, item in ipairs(mainNavItems) do
            local keys = NAV_STATUS_KEYS[item.key]
            if keys == true then
                -- Always on (e.g. Profile)
                item.statusDot:SetColorTexture(ns.CYAN.r, ns.CYAN.g, ns.CYAN.b, 0.5)
                item.statusDot:Show()
            elseif keys then
                local anyActive = false
                for _, k in ipairs(keys) do
                    if SKToolsDB[k] then anyActive = true; break end
                end
                if anyActive then
                    item.statusDot:SetColorTexture(ns.CYAN.r, ns.CYAN.g, ns.CYAN.b, 0.5)
                else
                    item.statusDot:SetColorTexture(0.4, 0.4, 0.4, 0.4)
                end
                item.statusDot:Show()
            end
        end
    end
    ns.RefreshMainNavStatus = RefreshMainNavStatus

    -- Default to General tab
    SelectTab("general")
    RefreshMainNavStatus()

end -- end CreateSettingsPanel

-----------------------------
-- Combat Settings Frame
-----------------------------
local function BuildCombatSettingsFrame()
    local MakeTabContent = ns.MakeTabContent
    local MakeCombatCBFactory = ns.MakeCombatCBFactory
    local CYAN = ns.CYAN

    local CC = ns.COLORS

    local f = CreateFrame("Frame", "SKToolsCombatSettings", UIParent, "BackdropTemplate")
    f:SetSize(820, 720)
    f:SetPoint("CENTER")
    f:SetMovable(true)
    f:EnableMouse(true)
    f:SetClampedToScreen(true)
    f:RegisterForDrag("LeftButton")
    f:SetScript("OnDragStart", f.StartMoving)
    f:SetScript("OnDragStop", f.StopMovingOrSizing)
    f:SetFrameStrata("DIALOG")
    f:SetBackdrop(ns.BACKDROP_PANEL)
    f:SetBackdropColor(CC.bgFrame[1], CC.bgFrame[2], CC.bgFrame[3], CC.bgFrame[4])
    f:SetBackdropBorderColor(CC.border[1], CC.border[2], CC.border[3], 0.80)
    f:Hide()
    tinsert(UISpecialFrames, "SKToolsCombatSettings")

    ns.CreateDropShadow(f, 8)
    ns.CreateCyanAccentLine(f)

    -- Close button (VS Code style: invisible until hovered, red tint on hover)
    local closeBtn = CreateFrame("Button", nil, f, "BackdropTemplate")
    closeBtn:SetSize(24, 24)
    closeBtn:SetPoint("TOPRIGHT", -6, -6)
    closeBtn:SetBackdrop(ns.BACKDROP_CONTROL)
    closeBtn:SetBackdropColor(0, 0, 0, 0)
    closeBtn:SetBackdropBorderColor(0, 0, 0, 0)
    local closeLbl = closeBtn:CreateFontString(nil, "OVERLAY", "GameFontNormal")
    closeLbl:SetPoint("CENTER", 0, 0)
    closeLbl:SetText("x")
    closeLbl:SetTextColor(CC.textMuted[1], CC.textMuted[2], CC.textMuted[3])
    closeBtn:SetScript("OnClick", function() f:Hide() end)
    closeBtn:SetScript("OnEnter", function(self)
        self:SetBackdropColor(0.60, 0.15, 0.15, 0.6)
        self:SetBackdropBorderColor(0.60, 0.20, 0.20, 0.5)
        closeLbl:SetTextColor(1, 1, 1)
    end)
    closeBtn:SetScript("OnLeave", function(self)
        self:SetBackdropColor(0, 0, 0, 0)
        self:SetBackdropBorderColor(0, 0, 0, 0)
        closeLbl:SetTextColor(CC.textMuted[1], CC.textMuted[2], CC.textMuted[3])
    end)

    --------------------------------
    -- Side Nav + Content (shared helper)
    --------------------------------
    local _, contentContainer, csContentPanels, SelectTab, csNavItems = ns.BuildSideNav(
        f,
        { f, 4, -4 },      -- navAnchorTop
        { f, 4, 4 },        -- navAnchorBottom
        { f, 176, -4 },     -- contentTL
        { f, -8, 8 }        -- contentBR
    )

    local csSyncControls = {}

    --------------------------------
    -- General Tab
    --------------------------------
    local genSF, genContent = MakeTabContent(f, contentContainer)
    csContentPanels["general"] = genSF

    local genAnchor = ns.CreateTabTitle(genContent, "SKTools", "Quality-of-life tools for PvE and PvP.")

    -- SKTools Button section (card-wrapped)
    local csBtnHeader = ns.AddSectionHeader(genContent, genAnchor, "SKTools Button", false)
    local csBtnCard = ns.CreateSectionCard(genContent)
    csBtnCard:SetPoint("TOPLEFT", csBtnHeader, "TOPLEFT", -8, 8)

    local AddCB, GetLast, SetLast = MakeCombatCBFactory(genContent, csBtnHeader, csSyncControls)

    AddCB("MinimapBtn", "showMinimapButton", "Show SKTools Button",
        "Shows a button near the action bars. Left-click for match history, right-click for settings. Also: /history",
        function(enabled)
            if minimapButton then
                if enabled then minimapButton:Show() else minimapButton:Hide() end
            end
        end)

    do
        local lastEl = GetLast()
        local resetBtn = ns.CreateThemedButton(genContent, "Reset Icon Position", 150, 24, "secondary")
        resetBtn:SetPoint("TOPLEFT", lastEl, "BOTTOMLEFT", 0, -8)
        resetBtn:SetScript("OnClick", function()
            SKToolsDB.buttonPos = nil
            if minimapButton then
                minimapButton:ClearAllPoints()
                if QueueStatusButton then
                    minimapButton:SetPoint("RIGHT", QueueStatusButton, "LEFT", -2, 0)
                else
                    minimapButton:SetPoint("BOTTOMRIGHT", UIParent, "BOTTOMRIGHT", -260, 5)
                end
            end
        end)
        SetLast(resetBtn)
    end

    -- Button click action dropdowns
    do
        local clickLabel = genContent:CreateFontString(nil, "ARTWORK", "GameFontNormal")
        clickLabel:SetPoint("TOPLEFT", GetLast(), "BOTTOMLEFT", 0, -14)
        clickLabel:SetText("Button Click Actions")
        clickLabel:SetTextColor(0.7, 0.7, 0.7)

        local actions = ns.BUTTON_ACTIONS

        local CS_DD_LEFT = 90

        local leftLabel = genContent:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        leftLabel:SetPoint("TOPLEFT", clickLabel, "BOTTOMLEFT", 0, -10)
        leftLabel:SetText("Left-click")
        leftLabel:SetTextColor(0.55, 0.55, 0.6)
        local leftDD = ns.CreateThemedDropdown(genContent, 140, actions, SKToolsDB.buttonLeftClick, function(val)
            SKToolsDB.buttonLeftClick = val
        end)
        leftDD:SetPoint("LEFT", clickLabel, "LEFT", CS_DD_LEFT, 0)
        leftDD:SetPoint("TOP", leftLabel, "TOP", 0, 0)

        local shiftLabel = genContent:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        shiftLabel:SetPoint("LEFT", clickLabel, "LEFT", 0, 0)
        shiftLabel:SetPoint("TOP", leftDD, "BOTTOM", 0, -6)
        shiftLabel:SetText("Shift-click")
        shiftLabel:SetTextColor(0.55, 0.55, 0.6)
        local shiftDD = ns.CreateThemedDropdown(genContent, 140, actions, SKToolsDB.buttonShiftClick, function(val)
            SKToolsDB.buttonShiftClick = val
        end)
        shiftDD:SetPoint("LEFT", clickLabel, "LEFT", CS_DD_LEFT, 0)
        shiftDD:SetPoint("TOP", shiftLabel, "TOP", 0, 0)

        local midLabel = genContent:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        midLabel:SetPoint("LEFT", clickLabel, "LEFT", 0, 0)
        midLabel:SetPoint("TOP", shiftDD, "BOTTOM", 0, -6)
        midLabel:SetText("Middle-click")
        midLabel:SetTextColor(0.55, 0.55, 0.6)
        local midDD = ns.CreateThemedDropdown(genContent, 140, actions, SKToolsDB.buttonMiddleClick, function(val)
            SKToolsDB.buttonMiddleClick = val
        end)
        midDD:SetPoint("LEFT", clickLabel, "LEFT", CS_DD_LEFT, 0)
        midDD:SetPoint("TOP", midLabel, "TOP", 0, 0)

        table.insert(csSyncControls, { type = "custom", refresh = function()
            leftDD:SetValue(SKToolsDB.buttonLeftClick)
            shiftDD:SetValue(SKToolsDB.buttonShiftClick)
            midDD:SetValue(SKToolsDB.buttonMiddleClick)
        end })

        SetLast(midDD)
    end

    csBtnCard:SetPoint("BOTTOM", GetLast(), "BOTTOM", 0, -8)
    csBtnCard:SetPoint("RIGHT", genContent, "RIGHT", -8, 0)

    -- General + Tweaks sections (from General.lua)
    local generalLast = ns.BuildGeneralCombatSettings(genContent, GetLast(), csSyncControls)
    local tweaksLast = ns.BuildTweaksCombatSettings(genContent, generalLast, csSyncControls)

    --------------------------------
    -- Stats Tab
    --------------------------------
    do
        local statsSF, statsContent = MakeTabContent(f, contentContainer)
        csContentPanels["stats"] = statsSF

        local statsAnchor = ns.CreateTabTitle(statsContent, "Stat Tracker", "Secondary stat diminishing returns and information.")
        ns.BuildStatCombatSettings(statsContent, statsAnchor, csSyncControls)
    end

    --------------------------------
    -- Nameplates Tab
    --------------------------------
    do
        local npSF, npCont = MakeTabContent(f, contentContainer)
        csContentPanels["nameplates"] = npSF
        ns.BuildNameplateCombatSettings(npCont, csSyncControls)
    end

    --------------------------------
    -- History Tab
    --------------------------------
    do
        local histSF, histContent = MakeTabContent(f, contentContainer)
        csContentPanels["history"] = histSF

        local histAnchor = ns.CreateTabTitle(histContent, "History", "PvP and PvE match history tracking.")

        AddCB, GetLast, SetLast = MakeCombatCBFactory(histContent, histAnchor, csSyncControls)

        AddCB("PvPHistory", "pvpHistory", "PvP Match History",
            "Automatically records arena and battleground results. View with /history or left-click the SKTools button.", nil)

        do
            local btnRow = CreateFrame("Frame", nil, histContent)
            btnRow:SetSize(300, 22)
            btnRow:SetPoint("TOPLEFT", GetLast(), "BOTTOMLEFT", 0, -4)

            local openHistBtn = ns.CreateThemedButton(btnRow, "Open History", 110, 22, "secondary")
            openHistBtn:SetPoint("LEFT", 0, 0)
            openHistBtn:SetScript("OnClick", function()
                SKPvPHistory_Toggle()
            end)

            local importBtn = ns.CreateThemedButton(btnRow, "Import from REFlex", 140, 22, "secondary")
            importBtn:SetPoint("LEFT", openHistBtn, "RIGHT", 6, 0)
            importBtn:SetScript("OnClick", function()
                ns.ImportREFlexData()
            end)

            SetLast(btnRow)
        end

        AddCB("PvEHistory", "pveHistory", "PvE History",
            "Track M+ dungeons, raids, and dungeon runs. View with /pvehistory.", nil)

        do
            local openPvEBtn = ns.CreateThemedButton(histContent, "Open PvE History", 130, 22, "secondary")
            openPvEBtn:SetPoint("TOPLEFT", GetLast(), "BOTTOMLEFT", 0, -4)
            openPvEBtn:SetScript("OnClick", function()
                SKPvEHistory_Toggle()
            end)
            SetLast(openPvEBtn)
        end
    end

    --------------------------------
    -- Battleground Tab
    --------------------------------
    do
        local bgSF, bgContent = MakeTabContent(f, contentContainer)
        csContentPanels["battleground"] = bgSF
        ns.BuildBattleIntelCombatSettings(bgContent, csSyncControls)
    end

    --------------------------------
    -- Notes Tab
    --------------------------------
    do
        local notesSF, notesContent = MakeTabContent(f, contentContainer)
        csContentPanels["notes"] = notesSF

        local notesTabAnchor = ns.CreateTabTitle(notesContent, "Notes", "Scouting notes for arenas, battlegrounds, dungeons, and raids.")

        AddCB, GetLast, SetLast = MakeCombatCBFactory(notesContent, notesTabAnchor, csSyncControls)
        local csNotesCB = AddCB("ArenaNotes", "arenaNotes", "Notes",
            "Enable the notes system. Browse all notes with /notes.",
            function(enabled) if enabled then SKArenaNotesEnable() else SKArenaNotesDisable() end end)

        local csNotesAnchor = GetLast()

        -- Sub-toggles (indented children of Notes)
        local CS_SUB_INDENT = 24
        local CS_SUB_DEFS = {
            { name = "NotesArena",   key = "arenaNotesArena",   label = "Auto-show in Arenas",        desc = "Show notes when entering an arena match." },
            { name = "NotesBG",      key = "arenaNotesBG",      label = "Auto-show in Battlegrounds", desc = "Show notes when entering a battleground." },
            { name = "NotesDungeon", key = "arenaNotesDungeon", label = "Auto-show in Dungeons",      desc = "Show notes when entering a dungeon." },
            { name = "NotesRaid",    key = "arenaNotesRaid",    label = "Auto-show in Raids",         desc = "Show notes when entering a raid." },
        }
        local csSubToggles = {}
        local csLastSub = csNotesAnchor
        for i, def in ipairs(CS_SUB_DEFS) do
            local cb = ns.CreateToggleSwitch(notesContent, "SKToolsCS_" .. def.name .. "CB")
            cb:SetPoint("TOPLEFT", csLastSub, "BOTTOMLEFT", i == 1 and CS_SUB_INDENT or 0, -8)
            cb.Text:SetText(def.label)
            cb.Text:SetTextColor(0.8, 0.8, 0.8)
            cb:SetChecked(SKToolsDB[def.key])
            cb:SetScript("OnClick", function(self)
                SKToolsDB[def.key] = self:GetChecked()
                if ns.RefreshCombatNavStatus then ns.RefreshCombatNavStatus() end
            end)
            local desc = notesContent:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
            desc:SetPoint("TOPLEFT", cb.Text, "BOTTOMLEFT", 0, -2)
            desc:SetPoint("RIGHT", notesContent, "RIGHT", -16, 0)
            desc:SetJustifyH("LEFT")
            desc:SetTextColor(0.55, 0.55, 0.6)
            desc:SetText(def.desc)
            table.insert(csSyncControls, { type = "checkbox", widget = cb, key = def.key })
            csSubToggles[#csSubToggles + 1] = cb
            local spacer = CreateFrame("Frame", nil, notesContent)
            spacer:SetSize(1, 1)
            spacer:SetPoint("LEFT", cb, "LEFT", 0, 0)
            spacer:SetPoint("TOP", desc, "BOTTOM", 0, 0)
            csLastSub = spacer
        end

        -- Connecting line
        local csConnLine = notesContent:CreateTexture(nil, "ARTWORK")
        csConnLine:SetWidth(1)
        csConnLine:SetPoint("TOP", csSubToggles[1], "TOPLEFT", -8, 4)
        csConnLine:SetPoint("BOTTOM", csSubToggles[#csSubToggles], "LEFT", -8, 0)
        csConnLine:SetColorTexture(ns.COLORS.divider[1], ns.COLORS.divider[2], ns.COLORS.divider[3], 0.30)

        -- Enable/disable sub-toggles based on Notes parent
        local function UpdateCSNotesChildState()
            local enabled = csNotesCB:GetChecked()
            for _, cb in ipairs(csSubToggles) do
                cb:SetEnabled(enabled)
            end
        end
        csNotesCB:HookScript("OnClick", UpdateCSNotesChildState)
        table.insert(csSyncControls, { refresh = UpdateCSNotesChildState })
        UpdateCSNotesChildState()

        -- Return to parent's left edge
        local csSubReturn = CreateFrame("Frame", nil, notesContent)
        csSubReturn:SetSize(1, 1)
        csSubReturn:SetPoint("LEFT", csNotesCB, "LEFT", 0, 0)
        csSubReturn:SetPoint("TOP", csLastSub, "TOP", 0, 0)
        SetLast(csSubReturn)

        do
            local openNotesBtn = ns.CreateThemedButton(notesContent, "Open Notes", 100, 22, "secondary")
            openNotesBtn:SetPoint("TOPLEFT", GetLast(), "BOTTOMLEFT", 0, -8)
            openNotesBtn:SetScript("OnClick", function()
                C_Timer.After(0, SKArenaNotes_ToggleBrowser)
            end)
            SetLast(openNotesBtn)
        end
    end

    --------------------------------
    -- PvP Scanner Tab
    --------------------------------
    do
        local titlesSF, titlesContent = MakeTabContent(f, contentContainer)
        csContentPanels["pvptitles"] = titlesSF

        local titlesAnchor = ns.CreateTabTitle(titlesContent, "PvP Scanner", "Scan nearby players for PvP achievements.")
        ns.BuildPvPTitleCombatSettings(titlesContent, titlesAnchor, csSyncControls)
    end

    --------------------------------
    -- Macros Tab
    --------------------------------
    do
        local macSF, macContent = MakeTabContent(f, contentContainer)
        csContentPanels["macros"] = macSF
        ns.BuildMacroCombatSettings(macContent, csSyncControls)
    end

    --------------------------------
    -- Profile Tab
    --------------------------------
    do
        local profSF, profContent = MakeTabContent(f, contentContainer)
        csContentPanels["profile"] = profSF
        ns.BuildProfileCombatSettings(profContent, csSyncControls)
    end

    --------------------------------
    -- Shaman Tab (class-gated)
    --------------------------------
    if select(2, UnitClass("player")) == "SHAMAN" then
        local shamSF, shamContent = MakeTabContent(f, contentContainer)
        csContentPanels["shaman"] = shamSF
        ns.BuildShamanCombatSettings(shamContent, csSyncControls)
    end

    -- Nav status dots
    local CS_NAV_STATUS_KEYS = {
        general      = {"fastLoot", "chatCopy", "autoVendor", "autoRepair", "pvpTabTarget", "queueTimer", "classColorHealth"},
        stats        = {"statTooltips", "statItemTooltips", "statCompareTooltips"},
        nameplates   = {"namePlates", "targetHighlight"},
        history      = {"pvpHistory", "pveHistory"},
        battleground = {"battleIntel"},
        notes        = {"arenaNotes", "arenaNotesArena", "arenaNotesBG", "arenaNotesDungeon", "arenaNotesRaid"},
        pvptitles    = {"pvpTitles"},
        macros       = {"smartMacros"},
        profile      = true,  -- always on
        shaman       = {"totemTracking"},
    }
    local function RefreshCombatNavStatus()
        if not SKToolsDB then return end
        for _, item in ipairs(csNavItems or {}) do
            local keys = CS_NAV_STATUS_KEYS[item.key]
            if keys == true then
                item.statusDot:SetColorTexture(ns.CYAN.r, ns.CYAN.g, ns.CYAN.b, 0.5)
                item.statusDot:Show()
            elseif keys then
                local anyActive = false
                for _, k in ipairs(keys) do
                    if SKToolsDB[k] then anyActive = true; break end
                end
                if anyActive then
                    item.statusDot:SetColorTexture(ns.CYAN.r, ns.CYAN.g, ns.CYAN.b, 0.5)
                else
                    item.statusDot:SetColorTexture(0.4, 0.4, 0.4, 0.4)
                end
                item.statusDot:Show()
            end
        end
    end
    ns.RefreshCombatNavStatus = RefreshCombatNavStatus

    -- OnShow: refresh all controls from current state
    f:SetToplevel(true)
    f:SetScript("OnShow", function(self)
        self:Raise()
        for _, ctrl in ipairs(csSyncControls) do
            if ctrl.refresh then
                ctrl.refresh()
            elseif ctrl.type == "checkbox" and ctrl.key then
                ctrl.widget:SetChecked(SKToolsDB[ctrl.key])
            end
        end
        RefreshCombatNavStatus()
    end)

    -- Force first tab selection
    SelectTab("general")
    RefreshCombatNavStatus()
    combatSettingsFrame = f
end

local function ShowCombatSettings()
    if not combatSettingsFrame then
        BuildCombatSettingsFrame()
    end
    if combatSettingsFrame:IsShown() then
        combatSettingsFrame:Hide()
    else
        combatSettingsFrame:Show()
        ns.FadeIn(combatSettingsFrame, 0.2)
    end
end

-----------------------------
-- Slash Commands
-----------------------------
SLASH_SKTOOLS1 = "/sk"
SLASH_SKTOOLS2 = "/sktools"
do -- scope block
    SlashCmdList["SKTOOLS"] = function(msg)
        ShowCombatSettings()
    end

SLASH_SKHISTORY1 = "/history"
SlashCmdList["SKHISTORY"] = function()
    SKPvPHistory_Toggle()
end

SLASH_SKNOTES1 = "/notes"
SlashCmdList["SKNOTES"] = function()
    C_Timer.After(0, SKArenaNotes_ToggleBrowser)
end

SLASH_SKPVEHISTORY1 = "/pvehistory"
SlashCmdList["SKPVEHISTORY"] = function()
    SKPvEHistory_Toggle()
end

SLASH_SKSTATS1 = "/stats"
SlashCmdList["SKSTATS"] = function()
    ns.StatsOverview_Toggle()
end

SLASH_SKPROFILE1 = "/profile"
SlashCmdList["SKPROFILE"] = function()
    ns.Profile_Toggle()
end

SLASH_SKBILOG1 = "/bilog"
SlashCmdList["SKBILOG"] = function(msg)
    ns.BattleIntel_DumpLog(msg and msg:trim() or nil)
end

end -- end scope block

-----------------------------
-- Initialization
-----------------------------
local initFrame = CreateFrame("Frame")
initFrame:RegisterEvent("ADDON_LOADED")
initFrame:SetScript("OnEvent", function(self, event, addon)
    if addon ~= "SKTools" then return end

    local defaults = ns.defaults

    -- Initialize saved variables with defaults
    if not SKToolsDB then
        SKToolsDB = {}
    end
    for k, v in pairs(defaults) do
        if SKToolsDB[k] == nil then
            SKToolsDB[k] = v
        end
    end
    -- Merge new keys into sub-tables (totemTrackSpells, targetHighlightConfig)
    for _, subKey in ipairs({"totemTrackSpells", "targetHighlightConfig"}) do
        if defaults[subKey] and SKToolsDB[subKey] then
            for k, v in pairs(defaults[subKey]) do
                if SKToolsDB[subKey][k] == nil then
                    SKToolsDB[subKey][k] = v
                end
            end
        end
    end

    -- Initialize BI calibration log
    if not SKToolsBILogDB then SKToolsBILogDB = {} end

    -- Restore cached totem durations from SavedVariables
    ns.Totem_RestoreCache()

    -- Apply settings
    ns.SetFastLoot(SKToolsDB.fastLoot)
    ns.SetChatCopy(SKToolsDB.chatCopy)
    ns.SetPvpTabTarget(SKToolsDB.pvpTabTarget)
    ns.SetQueueTimer(SKToolsDB.queueTimer)
    ns.SetHideKeybinds(SKToolsDB.hideKeybinds)
    ns.SetHideMacroNames(SKToolsDB.hideMacroNames)
    ns.SetAutoVendor(SKToolsDB.autoVendor)
    ns.SetAutoRepair(SKToolsDB.autoRepair)
    ns.SetTotemTracking(SKToolsDB.totemTracking)
    ns.SetSmartMacros(SKToolsDB.smartMacros)
    ns.SetStatTracking(SKToolsDB.statTooltips or SKToolsDB.statItemTooltips or SKToolsDB.statCompareTooltips)
    ns.SetClassColorHealth(SKToolsDB.classColorHealth)
    ns.SetBattleIntel(SKToolsDB.battleIntel)
    ns.SetNameplates()
    if SKToolsDB.arenaNotes then SKArenaNotesEnable() end
    if not SKToolsPvPTitlesDB then
        SKToolsPvPTitlesDB = { version = 1, players = {}, guidToKey = {} }
    end
    if not SKToolsProfileDB then
        SKToolsProfileDB = {}
    end
    ns.PvPTitles_RestoreCache()
    ns.SetPvPTitles(SKToolsDB.pvpTitles)
    C_Timer.After(0, function()
        CreateMinimapButton()
        RegisterSettingsCategory()
    end)

    self:UnregisterEvent("ADDON_LOADED")
end)
