-- SKTools General Features
-- FastLoot, ArenaNumbers, PvpTabTarget, AutoVendor, AutoRepair, ChatCopy
-- + General/Tweaks settings sections (both main & combat)

local _, ns = ...

-----------------------------
-- Fast Loot
-----------------------------
local delay = 0
local DEBOUNCE_INTERVAL = 0.3

local lootFrame = CreateFrame("Frame")

local function FastLoot()
    if GetTime() - delay >= DEBOUNCE_INTERVAL then
        delay = GetTime()
        if GetCVarBool("autoLootDefault") ~= IsModifiedClick("AUTOLOOTTOGGLE") then
            for i = GetNumLootItems(), 1, -1 do
                LootSlot(i)
            end
            delay = GetTime()
        end
    end
end

local function SetFastLoot(enabled)
    if enabled then
        lootFrame:RegisterEvent("LOOT_READY")
        lootFrame:SetScript("OnEvent", FastLoot)
    else
        lootFrame:UnregisterEvent("LOOT_READY")
        lootFrame:SetScript("OnEvent", nil)
    end
end
ns.SetFastLoot = SetFastLoot


-----------------------------
-- Arena Nameplate Numbers
-----------------------------
local arenaFrame = CreateFrame("Frame")
local arenaOverlays = {}       -- plate -> overlay frame (single overlay per plate)
local arenaKnownPlates = {}    -- plate -> nameplate unit token
local arenaGUIDMap = {}        -- GUID -> arena number

-- Build GUID -> arena number mapping from arena unit tokens
local function BuildArenaGUIDMap()
    for i = 1, 5 do
        local ok, guid = pcall(UnitGUID, "arena" .. i)
        if ok and guid then
            arenaGUIDMap[guid] = i
        end
    end
end

-- Discover arena mapping from target (guaranteed non-secret since "target" is allowlisted)
local function DiscoverFromTarget()
    if not UnitExists("target") then return end
    local ok, guid = pcall(UnitGUID, "target")
    if not ok or not guid then return end
    for i = 1, 5 do
        local ok, isMatch = pcall(UnitIsUnit, "target", "arena" .. i)
        if ok and isMatch then
            arenaGUIDMap[guid] = i
            return
        end
    end
end

local function GetOrCreateArenaOverlay(plate)
    if arenaOverlays[plate] then return arenaOverlays[plate] end

    local overlay = CreateFrame("Frame", nil, plate)
    overlay:SetAllPoints()
    overlay:SetFrameLevel(plate:GetFrameLevel() + 10)

    overlay.text = overlay:CreateFontString(nil, "OVERLAY")
    overlay.text:SetFont(STANDARD_TEXT_FONT, 14, "OUTLINE")
    overlay.text:SetPoint("BOTTOM", plate, "TOP", 0, 0)
    overlay.text:SetTextColor(1, 1, 0)

    overlay:Hide()
    arenaOverlays[plate] = overlay
    return overlay
end

local function UpdateArenaNameplate(unit)
    if not unit then return end
    local plate = C_NamePlate.GetNamePlateForUnit(unit)
    if not plate then return end

    local overlay = GetOrCreateArenaOverlay(plate)

    -- Look up arena number by GUID (player GUIDs are non-secret in Midnight)
    local guid = UnitGUID(unit)
    local arenaNum = guid and arenaGUIDMap[guid]

    if arenaNum then
        overlay.text:SetText(arenaNum)
        overlay:Show()
    else
        overlay:Hide()
    end
end

local function RefreshAllArenaNameplates()
    if not IsActiveBattlefieldArena() then return end
    for plate, unit in pairs(arenaKnownPlates) do
        local ok, err = pcall(UpdateArenaNameplate, unit)
        if not ok then ns.SK_ReportError("ArenaRefresh", err) end
    end
end

local function SetArenaNumbers(enabled)
    if enabled then
        arenaFrame:RegisterEvent("NAME_PLATE_UNIT_ADDED")
        arenaFrame:RegisterEvent("NAME_PLATE_UNIT_REMOVED")
        arenaFrame:RegisterEvent("PLAYER_ENTERING_WORLD")
        arenaFrame:RegisterEvent("ARENA_OPPONENT_UPDATE")
        arenaFrame:RegisterEvent("PLAYER_TARGET_CHANGED")
        arenaFrame:RegisterEvent("PLAYER_FOCUS_CHANGED")
        arenaFrame:SetScript("OnEvent", function(self, event, ...)
            if not SKToolsDB or not SKToolsDB.arenaNumbers then return end
            if not IsActiveBattlefieldArena() then return end
            local ok, err
            if event == "NAME_PLATE_UNIT_ADDED" then
                local unit = (...)
                local plate = unit and C_NamePlate.GetNamePlateForUnit(unit)
                if plate then
                    arenaKnownPlates[plate] = unit
                end
                ok, err = pcall(UpdateArenaNameplate, unit)
            elseif event == "NAME_PLATE_UNIT_REMOVED" then
                local unit = (...)
                if unit then
                    local plate = C_NamePlate.GetNamePlateForUnit(unit)
                    if plate then
                        arenaKnownPlates[plate] = nil
                        if arenaOverlays[plate] then
                            arenaOverlays[plate]:Hide()
                        end
                    end
                end
            elseif event == "ARENA_OPPONENT_UPDATE" then
                BuildArenaGUIDMap()
                ok, err = pcall(RefreshAllArenaNameplates)
            elseif event == "PLAYER_TARGET_CHANGED" or event == "PLAYER_FOCUS_CHANGED" then
                -- UnitIsUnit("target/focus", "arena"..i) is non-secret
                pcall(DiscoverFromTarget)
                ok, err = pcall(RefreshAllArenaNameplates)
            else
                ok, err = pcall(RefreshAllArenaNameplates)
            end
            if ok == false then ns.SK_ReportError("ArenaNumbers:" .. event, err) end
        end)
    else
        arenaFrame:UnregisterAllEvents()
        arenaFrame:SetScript("OnEvent", nil)
        wipe(arenaKnownPlates)
        wipe(arenaGUIDMap)
    end
end
ns.SetArenaNumbers = SetArenaNumbers

-----------------------------
-- PvP Tab Targeting
-----------------------------
local tabFrame = CreateFrame("Frame")

local function IsInPvP()
    local _, ZoneType = IsInInstance()
    return ZoneType == "arena" or ZoneType == "pvp"
end

local function IsTabTargetingPlayersOnly()
    return GetBindingAction("tab") == "TARGETNEARESTENEMYPLAYER"
end

local function UpdateTabTargeting()
    if not SKToolsDB.pvpTabTarget then return end
    if InCombatLockdown() then
        tabFrame:RegisterEvent("PLAYER_REGEN_ENABLED")
        return
    end
    tabFrame:UnregisterEvent("PLAYER_REGEN_ENABLED")
    if IsInPvP() and not IsTabTargetingPlayersOnly() then
        print('Tab target Players')
        SetBinding("TAB", "TARGETNEARESTENEMYPLAYER")
    elseif not IsInPvP() and IsTabTargetingPlayersOnly() then
        print('Tab target Monsters (All)')
        SetBinding("TAB", "TARGETNEARESTENEMY")
    end
end

local function SetPvpTabTarget(enabled)
    if enabled then
        tabFrame:RegisterEvent("PLAYER_ENTERING_WORLD")
        tabFrame:RegisterEvent("ZONE_CHANGED_NEW_AREA")
        tabFrame:RegisterEvent("PLAYER_FLAGS_CHANGED")
        tabFrame:SetScript("OnEvent", function()
            UpdateTabTargeting()
        end)
    else
        tabFrame:UnregisterAllEvents()
        tabFrame:SetScript("OnEvent", nil)
        if not InCombatLockdown() and IsTabTargetingPlayersOnly() then
            SetBinding("TAB", "TARGETNEARESTENEMY")
        end
    end
end
ns.SetPvpTabTarget = SetPvpTabTarget

-----------------------------
-- PvP Queue Timer
-----------------------------
local queueTimerFrame
local queueTimerHooked = false
local queueTimerOrigText

local function QueueTimer_Start(self, index)
    if not SKToolsDB.queueTimer then return end

    local enterButton = self.enterButton
    if not enterButton then return end

    if not queueTimerOrigText then
        queueTimerOrigText = enterButton:GetText()
    end

    if not queueTimerFrame then
        queueTimerFrame = CreateFrame("Frame", nil, self)
    end
    queueTimerFrame:Show()

    queueTimerFrame:SetScript("OnUpdate", function(self)
        local ok, err = pcall(function()
            local secs = GetBattlefieldPortExpiration(index)
            if not secs or secs <= 0 then
                enterButton:SetText(queueTimerOrigText or "Enter")
                self:Hide()
                return
            end
            local timeStr = string.format("%d:%02d", math.floor(secs / 60), math.floor(secs % 60))
            enterButton:SetText((queueTimerOrigText or "Enter") .. "  " .. timeStr)
        end)
        if not ok then self:Hide(); ns.SK_ReportError("QueueTimer", err) end
    end)
end

local function SetQueueTimer(enabled)
    if enabled and not queueTimerHooked then
        hooksecurefunc("PVPReadyDialog_Display", QueueTimer_Start)
        queueTimerHooked = true
    end
end
ns.SetQueueTimer = SetQueueTimer

-----------------------------
-- Hide Action Bar Text
-----------------------------
local ALL_BAR_PREFIXES = {
    "ActionButton", "MultiBarBottomLeftButton", "MultiBarBottomRightButton",
    "MultiBarLeftButton", "MultiBarRightButton",
    "MultiBar5Button", "MultiBar6Button", "MultiBar7Button", "MultiBar8Button",
}

local noop = function() end
local origHotKeyShow = {}
local origNameShow = {}

local function HideBarText_Apply()
    local hideHK = SKToolsDB.hideKeybinds
    local hideMN = SKToolsDB.hideMacroNames
    for _, prefix in ipairs(ALL_BAR_PREFIXES) do
        for i = 1, 12 do
            local btn = _G[prefix .. i]
            if btn then
                local hk = btn.HotKey
                if hk then
                    if hideHK then
                        if not origHotKeyShow[hk] then origHotKeyShow[hk] = hk.Show end
                        hk:Hide()
                        hk.Show = noop
                    elseif origHotKeyShow[hk] then
                        hk.Show = origHotKeyShow[hk]
                        origHotKeyShow[hk] = nil
                        hk:Show()
                    end
                end
                local nm = btn.Name
                if nm then
                    if hideMN then
                        if not origNameShow[nm] then origNameShow[nm] = nm.Show end
                        nm:Hide()
                        nm.Show = noop
                    elseif origNameShow[nm] then
                        nm.Show = origNameShow[nm]
                        origNameShow[nm] = nil
                        nm:Show()
                    end
                end
            end
        end
    end
end

local function SetHideKeybinds(enabled)
    HideBarText_Apply()
end
ns.SetHideKeybinds = SetHideKeybinds

local function SetHideMacroNames(enabled)
    HideBarText_Apply()
end
ns.SetHideMacroNames = SetHideMacroNames

-----------------------------
-- Auto Vendor Grey Items
-----------------------------
local vendorFrame = CreateFrame("Frame")

local function SellGreyItems()
    if not SKToolsDB.autoVendor then return end
    local totalEarned = 0
    for bag = 0, 4 do
        for slot = 1, C_Container.GetContainerNumSlots(bag) do
            local info = C_Container.GetContainerItemInfo(bag, slot)
            if info and info.quality == Enum.ItemQuality.Poor and not info.hasNoValue then
                C_Container.UseContainerItem(bag, slot)
                totalEarned = totalEarned + (info.stackCount * (select(11, C_Item.GetItemInfo(info.itemID)) or 0))
            end
        end
    end
    if totalEarned > 0 then
        print("|cff00E5EESKTools:|r Sold grey items for " .. GetCoinTextureString(totalEarned))
    end
end

local function SetAutoVendor(enabled)
    if enabled then
        vendorFrame:RegisterEvent("MERCHANT_SHOW")
        vendorFrame:SetScript("OnEvent", SellGreyItems)
    else
        vendorFrame:UnregisterEvent("MERCHANT_SHOW")
        vendorFrame:SetScript("OnEvent", nil)
    end
end
ns.SetAutoVendor = SetAutoVendor

-----------------------------
-- Auto Repair
-----------------------------
local repairFrame = CreateFrame("Frame")

local function AutoRepair()
    if not SKToolsDB.autoRepair then return end
    if not CanMerchantRepair() then return end

    local cost, canRepair = GetRepairAllCost()
    if not canRepair or cost == 0 then return end

    local useGuild = SKToolsDB.useGuildRepair and IsInGuild() and CanGuildBankRepair() and GetGuildBankWithdrawMoney() >= cost
    RepairAllItems(useGuild)

    local source = useGuild and "guild bank" or "personal funds"
    print("|cff00E5EESKTools:|r Repaired all items for " .. GetCoinTextureString(cost) .. " (" .. source .. ")")
end

local function SetAutoRepair(enabled)
    if enabled then
        repairFrame:RegisterEvent("MERCHANT_SHOW")
        repairFrame:SetScript("OnEvent", AutoRepair)
    else
        repairFrame:UnregisterEvent("MERCHANT_SHOW")
        repairFrame:SetScript("OnEvent", nil)
    end
end
ns.SetAutoRepair = SetAutoRepair

-----------------------------
-- Chat Copy (Shift-click chat to copy text)
-----------------------------
local skChatCopyFrame = nil
local chatBuffers = {}  -- [chatFrameName] = circular buffer of plain text lines
local CHAT_BUFFER_MAX = 500

local function StripChatFormatting(text)
    if type(text) ~= "string" then return "" end
    local ok, result = pcall(function()
        text = text:gsub("|K.-|k", "***")         -- BNet scrambled names
        text = text:gsub("|c%x%x%x%x%x%x%x%x", "") -- color start
        text = text:gsub("|r", "")                 -- color reset
        text = text:gsub("|H.-|h(.-)|h", "%1")    -- hyperlinks → link text
        text = text:gsub("|T.-|t", "")             -- textures
        text = text:gsub("|A.-|a", "")             -- atlases
        text = text:gsub("|n", "\n")               -- newlines
        return text:trim()
    end)
    return ok and result or ""
end

local function ShowChatCopy(chatFrame)
    if not skChatCopyFrame then
        local f = CreateFrame("Frame", "SKToolsChatCopyFrame", UIParent, "BackdropTemplate")
        f:SetBackdrop({ bgFile = "Interface\\DialogFrame\\UI-DialogBox-Background",
            edgeFile = "Interface\\DialogFrame\\UI-DialogBox-Border",
            edgeSize = 16, insets = { left = 4, right = 4, top = 4, bottom = 4 } })
        f:SetBackdropColor(0.05, 0.05, 0.1, 0.95)
        f:SetFrameStrata("DIALOG")
        f:SetFrameLevel(100)
        f:EnableMouse(true)

        local close = CreateFrame("Button", nil, f, "UIPanelCloseButton")
        close:SetPoint("TOPRIGHT", -2, -2)

        local scroll = CreateFrame("ScrollFrame", nil, f, "UIPanelScrollFrameTemplate")
        scroll:SetPoint("TOPLEFT", 12, -10)
        scroll:SetPoint("BOTTOMRIGHT", -30, 10)

        local edit = CreateFrame("EditBox", nil, scroll)
        edit:SetMultiLine(true)
        edit:SetAutoFocus(false)
        edit:SetFontObject(ChatFontNormal)
        edit:SetWidth(440)
        edit:SetScript("OnEscapePressed", function() f:Hide() end)
        scroll:SetScrollChild(edit)
        f.edit = edit
        f.scroll = scroll

        tinsert(UISpecialFrames, "SKToolsChatCopyFrame")
        skChatCopyFrame = f
    end

    local f = skChatCopyFrame
    f:ClearAllPoints()
    f:SetPoint("TOPLEFT", chatFrame, "TOPLEFT", 0, 0)
    f:SetPoint("BOTTOMRIGHT", chatFrame, "BOTTOMRIGHT", 0, 0)
    f.edit:SetWidth(chatFrame:GetWidth() - 50)

    local buf = chatBuffers[chatFrame:GetName()] or {}
    local text = table.concat(buf, "\n")
    f.edit:SetText(text)
    f.edit:SetCursorPosition(#text)
    f:Show()
    f.edit:SetFocus()
    -- Scroll to bottom after layout
    C_Timer.After(0, function()
        f.scroll:SetVerticalScroll(f.scroll:GetVerticalScrollRange())
    end)
end

local chatCopyHooked = false
local function SetChatCopy(enabled)
    if not enabled then return end
    if chatCopyHooked then return end
    chatCopyHooked = true
    C_Timer.After(0, function()
        for i = 1, 10 do
            local cf = _G["ChatFrame" .. i]
            if cf then
                local name = cf:GetName()
                chatBuffers[name] = chatBuffers[name] or {}

                -- Buffer messages as they arrive — deferred to avoid tainting chat execution context
                hooksecurefunc(cf, "AddMessage", function(self, text)
                    C_Timer.After(0, function()
                        pcall(function()
                            if SKToolsDB and SKToolsDB.chatCopy and text and type(text) == "string" then
                                local buf = chatBuffers[name]
                                buf[#buf + 1] = StripChatFormatting(text)
                                if #buf > CHAT_BUFFER_MAX then
                                    table.remove(buf, 1)
                                end
                            end
                        end)
                    end)
                end)

                -- Shift-click to open
                cf:HookScript("OnMouseUp", function(self, button)
                    if IsShiftKeyDown() and button == "LeftButton" and SKToolsDB and SKToolsDB.chatCopy then
                        ShowChatCopy(self)
                    end
                end)
            end
        end
    end)
end
ns.SetChatCopy = SetChatCopy


-----------------------------
-- Class Colored Health Bars
-----------------------------
local classColorFrame = CreateFrame("Frame")
local classColorBars = {}
local classColorRecoloring = false

local function ClassColor_GetColor(unitToken)
    local ok, _, classFile = pcall(UnitClass, unitToken)
    if not ok or not classFile then return nil end
    if issecretvalue and issecretvalue(classFile) then return nil end
    local color = RAID_CLASS_COLORS[classFile]
    if not color then return nil end
    return color.r, color.g, color.b
end

local function ClassColor_ApplyToBar(bar, unitToken)
    if classColorRecoloring then return end
    if not bar or not unitToken then return end
    local okE, exists = pcall(UnitExists, unitToken)
    if not okE or not exists then return end
    classColorRecoloring = true
    local applyOk, applyErr = pcall(function()
        local okP, isPlayer = pcall(UnitIsPlayer, unitToken)
        if okP and isPlayer then
            local r, g, b = ClassColor_GetColor(unitToken)
            if r then
                local tex = bar:GetStatusBarTexture()
                if tex then tex:SetDesaturated(true) end
                bar:SetStatusBarColor(r, g, b)
                return
            end
        end
        -- NPC, secret value, or no class color — restore defaults
        local tex = bar:GetStatusBarTexture()
        if tex then tex:SetDesaturated(false) end
        bar:SetStatusBarColor(0, 1, 0)
    end)
    classColorRecoloring = false
    if not applyOk then ns.SK_ReportError("ClassColorHP:Apply", applyErr) end
end

local function ClassColor_FindHealthBar(chain)
    local current = _G[chain[1]]
    if not current then return nil end
    for i = 2, #chain do
        current = current[chain[i]]
        if not current then return nil end
    end
    return current
end

local HEALTH_BAR_PATHS = {
    { unit = "player",
      chain = {"PlayerFrame","PlayerFrameContent","PlayerFrameContentMain",
               "HealthBarsContainer","HealthBar"} },
    { unit = "target",
      chain = {"TargetFrame","TargetFrameContent","TargetFrameContentMain",
               "HealthBarsContainer","HealthBar"} },
    { unit = "focus",
      chain = {"FocusFrame","TargetFrameContent","TargetFrameContentMain",
               "HealthBarsContainer","HealthBar"} },
}

local function ClassColor_Setup()
    if #classColorBars > 0 then return end
    for _, def in ipairs(HEALTH_BAR_PATHS) do
        local ok, bar = pcall(ClassColor_FindHealthBar, def.chain)
        if ok and bar and bar.SetStatusBarColor then
            classColorBars[#classColorBars + 1] = { bar = bar, unit = def.unit }
            hooksecurefunc(bar, "SetStatusBarColor", function()
                if SKToolsDB and SKToolsDB.classColorHealth then
                    ClassColor_ApplyToBar(bar, def.unit)
                end
            end)
            ClassColor_ApplyToBar(bar, def.unit)
        end
    end
end

local function ClassColor_RefreshAll()
    for _, entry in ipairs(classColorBars) do
        ClassColor_ApplyToBar(entry.bar, entry.unit)
    end
end

local function SetClassColorHealth(enabled)
    if enabled then
        classColorFrame:RegisterEvent("PLAYER_LOGIN")
        classColorFrame:RegisterEvent("PLAYER_ENTERING_WORLD")
        classColorFrame:RegisterEvent("PLAYER_TARGET_CHANGED")
        classColorFrame:RegisterEvent("PLAYER_FOCUS_CHANGED")
        classColorFrame:SetScript("OnEvent", function(self, event)
            local ok, err = pcall(function()
                if event == "PLAYER_LOGIN" or event == "PLAYER_ENTERING_WORLD" then
                    ClassColor_Setup()
                    ClassColor_RefreshAll()
                elseif event == "PLAYER_TARGET_CHANGED" then
                    for _, entry in ipairs(classColorBars) do
                        if entry.unit == "target" then
                            ClassColor_ApplyToBar(entry.bar, "target")
                        end
                    end
                elseif event == "PLAYER_FOCUS_CHANGED" then
                    for _, entry in ipairs(classColorBars) do
                        if entry.unit == "focus" then
                            ClassColor_ApplyToBar(entry.bar, "focus")
                        end
                    end
                end
            end)
            if not ok then ns.SK_ReportError("ClassColorHP", err) end
        end)
        pcall(ClassColor_Setup)
        pcall(ClassColor_RefreshAll)
    else
        classColorFrame:UnregisterAllEvents()
        classColorFrame:SetScript("OnEvent", nil)
        -- Restore default colors on all hooked bars
        for _, entry in ipairs(classColorBars) do
            pcall(function()
                local tex = entry.bar:GetStatusBarTexture()
                if tex then tex:SetDesaturated(false) end
                entry.bar:SetStatusBarColor(0, 1, 0)
            end)
        end
    end
end
ns.SetClassColorHealth = SetClassColorHealth


-----------------------------
-- General Settings Builder (main panel)
-----------------------------
function ns.BuildGeneralSettings(panel, anchor)
    local generalHeader = ns.AddSectionHeader(panel, anchor, "General", false)
    local generalCard = ns.CreateSectionCard(panel)
    generalCard:SetPoint("TOPLEFT", generalHeader, "TOPLEFT", -8, 8)
    local AddCB, GetLast, SetLast = ns.MakeCheckboxFactory(panel, generalHeader)

    AddCB("FastLoot", "fastLoot", "Fast Loot",
        "Automatically loots all items from a corpse instantly.",
        SetFastLoot)

    AddCB("ChatCopy", "chatCopy", "Chat Copy",
        "Shift-click any chat window to open a copyable text overlay.",
        nil)

    AddCB("AutoVendor", "autoVendor", "Auto Vendor Grey Items",
        "Automatically sells all grey (vendor trash) items when you open a merchant.",
        SetAutoVendor)

    -- Auto Repair with inline guild bank sub-option
    do
        local lastElement = GetLast()
        local cb = ns.CreateToggleSwitch(panel, "SKToolsAutoRepairCB")
        cb:SetPoint("TOPLEFT", lastElement, "BOTTOMLEFT", 0, -12)
        cb.Text:SetText("Auto Repair")
        cb.Text:SetTextColor(0.9, 0.9, 0.9)
        cb:SetChecked(SKToolsDB.autoRepair)

        local desc = panel:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        desc:SetPoint("TOPLEFT", cb.Text, "BOTTOMLEFT", 0, -2)
        desc:SetPoint("RIGHT", panel, "RIGHT", -16, 0)
        desc:SetJustifyH("LEFT")
        desc:SetTextColor(0.55, 0.55, 0.6)
        desc:SetText("Automatically repairs all gear when you open a merchant.")

        local guildCb = ns.CreateToggleSwitch(panel, "SKToolsGuildRepairCB")
        guildCb:SetPoint("TOPLEFT", desc, "BOTTOMLEFT", 0, -6)
        guildCb.Text:SetText("Use guild bank funds first")
        guildCb.Text:SetFontObject("GameFontHighlightSmall")
        guildCb:SetChecked(SKToolsDB.useGuildRepair)
        guildCb:SetEnabled(SKToolsDB.autoRepair)

        cb:SetScript("OnClick", function(self)
            local checked = self:GetChecked()
            SKToolsDB.autoRepair = checked
            SetAutoRepair(checked)
            guildCb:SetEnabled(checked)
            if ns.RefreshMainNavStatus then ns.RefreshMainNavStatus() end
        end)

        guildCb:SetScript("OnClick", function(self)
            SKToolsDB.useGuildRepair = self:GetChecked()
        end)

        local spacer = CreateFrame("Frame", nil, panel)
        spacer:SetSize(1, 1)
        spacer:SetPoint("LEFT", cb, "LEFT", 0, 0)
        spacer:SetPoint("TOP", guildCb, "BOTTOM", 0, 0)
        SetLast(spacer)
    end

    AddCB("PvpTab", "pvpTabTarget", "PvP Tab Targeting",
        "Switches Tab to target players only in arenas and battlegrounds, and reverts back outside.",
        SetPvpTabTarget)

    AddCB("QueueTimer", "queueTimer", "PvP Queue Timer",
        "Shows a countdown timer on the queue pop dialog when a battleground or arena queue is ready.",
        SetQueueTimer)

    AddCB("ClassColorHP", "classColorHealth", "Class Colored Health Bars",
        "Colors player, target, and focus health bars based on class color.",
        SetClassColorHealth)

    AddCB("TrackPlayed", "trackPlayedTime", "Track /played Time",
        "Requests and stores total played time on each login. Displayed in your Character Profile.",
        nil)

    generalCard:SetPoint("BOTTOM", GetLast(), "BOTTOM", 0, -8)
    generalCard:SetPoint("RIGHT", panel, "RIGHT", -8, 0)
    return GetLast()
end

-----------------------------
-- Tweaks Settings Builder (main panel)
-----------------------------
function ns.BuildTweaksSettings(panel, anchor)
    local tweaksHeader = ns.AddSectionHeader(panel, anchor, "Tweaks", false)
    local tweaksCard = ns.CreateSectionCard(panel)
    tweaksCard:SetPoint("TOPLEFT", tweaksHeader, "TOPLEFT", -8, 8)
    local lastEl = tweaksHeader

    -- Spell Queue Window
    do
        local sqwLabel = panel:CreateFontString(nil, "ARTWORK", "GameFontNormal")
        sqwLabel:SetPoint("TOPLEFT", tweaksHeader, "BOTTOMLEFT", 0, -16)
        sqwLabel:SetText("Spell Queue Window")
        sqwLabel:SetTextColor(0.85, 0.85, 0.87)

        local sqwDesc = panel:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        sqwDesc:SetPoint("TOPLEFT", sqwLabel, "BOTTOMLEFT", 0, -2)
        sqwDesc:SetTextColor(0.55, 0.55, 0.6)
        sqwDesc:SetText("How early you can queue your next spell (in ms). Default: 400")

        -- Minus button
        local sqwMinus = ns.CreateThemedButton(panel, "-", 24, 24, "secondary")
        sqwMinus:SetPoint("TOPLEFT", sqwDesc, "BOTTOMLEFT", 0, -20)

        local slider = CreateFrame("Slider", "SKToolsSQWSlider", panel, "OptionsSliderTemplate")
        slider:SetPoint("LEFT", sqwMinus, "RIGHT", 4, 0)
        slider:SetWidth(200)
        slider:SetMinMaxValues(0, 400)
        slider:SetValueStep(1)
        slider:SetObeyStepOnDrag(true)
        ns.StyleSlider(slider)

        local currentVal = tonumber(GetCVar("SpellQueueWindow")) or 400
        slider:SetValue(currentVal)

        _G["SKToolsSQWSliderLow"]:SetText("0")
        _G["SKToolsSQWSliderHigh"]:SetText("400")
        _G["SKToolsSQWSliderText"]:SetText(currentVal .. " ms")

        -- Plus button
        local sqwPlus = ns.CreateThemedButton(panel, "+", 24, 24, "secondary")
        sqwPlus:SetPoint("LEFT", slider, "RIGHT", 4, 0)

        local editBox = ns.CreateThemedEditBox(panel, 50, 22)
        editBox:SetPoint("LEFT", sqwPlus, "RIGHT", 12, 0)
        editBox:SetNumeric(true)
        editBox:SetMaxLetters(3)
        editBox:SetText(tostring(currentVal))

        local msLabel = panel:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        msLabel:SetPoint("LEFT", editBox, "RIGHT", 4, 0)
        msLabel:SetTextColor(0.55, 0.55, 0.6)
        msLabel:SetText("ms")

        slider:SetScript("OnValueChanged", function(self, value)
            value = math.floor(value + 0.5)
            SetCVar("SpellQueueWindow", value)
            _G["SKToolsSQWSliderText"]:SetText(value .. " ms")
            editBox:SetText(tostring(value))
        end)

        sqwMinus:SetScript("OnClick", function()
            local val = slider:GetValue() - 1
            if val >= 0 then slider:SetValue(val) end
        end)
        sqwPlus:SetScript("OnClick", function()
            local val = slider:GetValue() + 1
            if val <= 400 then slider:SetValue(val) end
        end)

        editBox:SetScript("OnEnterPressed", function(self)
            local val = tonumber(self:GetText()) or 400
            val = math.max(0, math.min(400, val))
            slider:SetValue(val)
            self:ClearFocus()
        end)
        editBox:SetScript("OnEscapePressed", function(self) self:ClearFocus() end)

        -- Spacer below slider row to account for Low/High labels
        local sqwSpacer = panel:CreateFontString(nil, "ARTWORK")
        sqwSpacer:SetPoint("TOPLEFT", sqwMinus, "BOTTOMLEFT", 0, -4)
        sqwSpacer:SetHeight(1)

        lastEl = sqwSpacer
    end

    -- Bag Sort Direction
    do
        local cb = ns.CreateToggleSwitch(panel, "SKToolsBagSortCB")
        cb:SetPoint("TOPLEFT", lastEl, "BOTTOMLEFT", 0, -16)
        cb.Text:SetText("Sort Bags Right to Left")
        cb.Text:SetTextColor(0.9, 0.9, 0.9)
        cb:SetChecked(C_Container.GetSortBagsRightToLeft())
        cb:SetScript("OnClick", function(self)
            C_Container.SetSortBagsRightToLeft(self:GetChecked())
        end)
        cb:HookScript("OnShow", function(self)
            self:SetChecked(C_Container.GetSortBagsRightToLeft())
        end)
        local desc = panel:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        desc:SetPoint("TOPLEFT", cb.Text, "BOTTOMLEFT", 0, -2)
        desc:SetPoint("RIGHT", panel, "RIGHT", -16, 0)
        desc:SetJustifyH("LEFT")
        desc:SetTextColor(0.55, 0.55, 0.6)
        desc:SetText("Changes bag sorting direction. Uncheck for left to right.")
        local spacer = CreateFrame("Frame", nil, panel)
        spacer:SetSize(1, 1)
        spacer:SetPoint("LEFT", cb, "LEFT", 0, 0)
        spacer:SetPoint("TOP", desc, "BOTTOM", 0, 0)
        lastEl = spacer
    end

    -- Sharpen Graphics
    do
        local cb = ns.CreateToggleSwitch(panel, "SKToolsSharpenCB")
        cb:SetPoint("TOPLEFT", lastEl, "BOTTOMLEFT", 0, -12)
        cb.Text:SetText("Sharpen Graphics")
        cb.Text:SetTextColor(0.9, 0.9, 0.9)
        cb:SetChecked(GetCVarBool("ResampleAlwaysSharpen"))
        cb:SetScript("OnClick", function(self)
            SetCVar("ResampleAlwaysSharpen", self:GetChecked() and "1" or "0")
        end)
        cb:HookScript("OnShow", function(self)
            self:SetChecked(GetCVarBool("ResampleAlwaysSharpen"))
        end)
        local desc = panel:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        desc:SetPoint("TOPLEFT", cb.Text, "BOTTOMLEFT", 0, -2)
        desc:SetPoint("RIGHT", panel, "RIGHT", -16, 0)
        desc:SetJustifyH("LEFT")
        desc:SetTextColor(0.55, 0.55, 0.6)
        desc:SetText("Toggles ResampleAlwaysSharpen for crisper visuals.")
        local spacer = CreateFrame("Frame", nil, panel)
        spacer:SetSize(1, 1)
        spacer:SetPoint("LEFT", cb, "LEFT", 0, 0)
        spacer:SetPoint("TOP", desc, "BOTTOM", 0, 0)
        lastEl = spacer
    end

    -- Hide Action Bar Keybinds
    do
        local cb = ns.CreateToggleSwitch(panel, "SKToolsHideKeybindsCB")
        cb:SetPoint("TOPLEFT", lastEl, "BOTTOMLEFT", 0, -12)
        cb.Text:SetText("Hide Action Bar Keybinds")
        cb.Text:SetTextColor(0.9, 0.9, 0.9)
        cb:SetChecked(SKToolsDB.hideKeybinds)
        cb:SetScript("OnClick", function(self)
            SKToolsDB.hideKeybinds = self:GetChecked()
            SetHideKeybinds(self:GetChecked())
        end)
        cb:HookScript("OnShow", function(self)
            self:SetChecked(SKToolsDB.hideKeybinds)
        end)
        local desc = panel:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        desc:SetPoint("TOPLEFT", cb.Text, "BOTTOMLEFT", 0, -2)
        desc:SetPoint("RIGHT", panel, "RIGHT", -16, 0)
        desc:SetJustifyH("LEFT")
        desc:SetTextColor(0.55, 0.55, 0.6)
        desc:SetText("Hides keybind text on action bar buttons for a cleaner look.")
        local spacer = CreateFrame("Frame", nil, panel)
        spacer:SetSize(1, 1)
        spacer:SetPoint("LEFT", cb, "LEFT", 0, 0)
        spacer:SetPoint("TOP", desc, "BOTTOM", 0, 0)
        lastEl = spacer
    end

    -- Hide Action Bar Macro Names
    do
        local cb = ns.CreateToggleSwitch(panel, "SKToolsHideMacroNamesCB")
        cb:SetPoint("TOPLEFT", lastEl, "BOTTOMLEFT", 0, -12)
        cb.Text:SetText("Hide Action Bar Macro Names")
        cb.Text:SetTextColor(0.9, 0.9, 0.9)
        cb:SetChecked(SKToolsDB.hideMacroNames)
        cb:SetScript("OnClick", function(self)
            SKToolsDB.hideMacroNames = self:GetChecked()
            SetHideMacroNames(self:GetChecked())
        end)
        cb:HookScript("OnShow", function(self)
            self:SetChecked(SKToolsDB.hideMacroNames)
        end)
        local desc = panel:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        desc:SetPoint("TOPLEFT", cb.Text, "BOTTOMLEFT", 0, -2)
        desc:SetPoint("RIGHT", panel, "RIGHT", -16, 0)
        desc:SetJustifyH("LEFT")
        desc:SetTextColor(0.55, 0.55, 0.6)
        desc:SetText("Hides macro name text on action bar buttons.")
        local spacer = CreateFrame("Frame", nil, panel)
        spacer:SetSize(1, 1)
        spacer:SetPoint("LEFT", cb, "LEFT", 0, 0)
        spacer:SetPoint("TOP", desc, "BOTTOM", 0, 0)
        lastEl = spacer
    end

    tweaksCard:SetPoint("BOTTOM", lastEl, "BOTTOM", 0, -8)
    tweaksCard:SetPoint("RIGHT", panel, "RIGHT", -8, 0)
    return lastEl
end

-----------------------------
-- General Combat Settings Builder
-----------------------------
function ns.BuildGeneralCombatSettings(panel, anchor, csSyncControls)
    local generalHeader = ns.AddSectionHeader(panel, anchor, "General", false)
    local generalCard = ns.CreateSectionCard(panel)
    generalCard:SetPoint("TOPLEFT", generalHeader, "TOPLEFT", -8, 8)
    local AddCB, GetLast, SetLast = ns.MakeCombatCBFactory(panel, generalHeader, csSyncControls)

    AddCB("FastLoot", "fastLoot", "Fast Loot",
        "Automatically loots all items from a corpse instantly.", SetFastLoot)
    AddCB("ChatCopy", "chatCopy", "Chat Copy",
        "Shift-click any chat window to open a copyable text overlay.", nil)
    AddCB("AutoVendor", "autoVendor", "Auto Vendor Grey Items",
        "Automatically sells all grey (vendor trash) items when you open a merchant.", SetAutoVendor)

    do
        local lastElement = GetLast()
        local cb = ns.CreateToggleSwitch(panel, "SKToolsCS_AutoRepair")
        cb:SetPoint("TOPLEFT", lastElement, "BOTTOMLEFT", 0, -12)
        cb.Text:SetText("Auto Repair")
        cb.Text:SetTextColor(0.9, 0.9, 0.9)
        cb:SetChecked(SKToolsDB.autoRepair)
        local desc = panel:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        desc:SetPoint("TOPLEFT", cb.Text, "BOTTOMLEFT", 0, -2)
        desc:SetPoint("RIGHT", panel, "RIGHT", -16, 0)
        desc:SetJustifyH("LEFT")
        desc:SetTextColor(0.55, 0.55, 0.6)
        desc:SetText("Automatically repairs all gear when you open a merchant.")
        local guildCb = ns.CreateToggleSwitch(panel, "SKToolsCS_GuildRepair")
        guildCb:SetPoint("TOPLEFT", desc, "BOTTOMLEFT", 0, -6)
        guildCb.Text:SetText("Use guild bank funds first")
        guildCb.Text:SetFontObject("GameFontHighlightSmall")
        guildCb:SetChecked(SKToolsDB.useGuildRepair)
        guildCb:SetEnabled(SKToolsDB.autoRepair)
        cb:SetScript("OnClick", function(self)
            local checked = self:GetChecked()
            SKToolsDB.autoRepair = checked
            SetAutoRepair(checked)
            guildCb:SetEnabled(checked)
            if ns.RefreshCombatNavStatus then ns.RefreshCombatNavStatus() end
        end)
        guildCb:SetScript("OnClick", function(self) SKToolsDB.useGuildRepair = self:GetChecked() end)
        table.insert(csSyncControls, { type = "custom", refresh = function()
            cb:SetChecked(SKToolsDB.autoRepair)
            guildCb:SetChecked(SKToolsDB.useGuildRepair)
            guildCb:SetEnabled(SKToolsDB.autoRepair)
        end })
        local spacer = CreateFrame("Frame", nil, panel)
        spacer:SetSize(1, 1)
        spacer:SetPoint("LEFT", cb, "LEFT", 0, 0)
        spacer:SetPoint("TOP", guildCb, "BOTTOM", 0, 0)
        SetLast(spacer)
    end

    AddCB("PvpTab", "pvpTabTarget", "PvP Tab Targeting",
        "Switches Tab to target players only in arenas and battlegrounds, and reverts back outside.", SetPvpTabTarget)

    AddCB("QueueTimer", "queueTimer", "PvP Queue Timer",
        "Shows a countdown timer on the queue pop dialog when a battleground or arena queue is ready.",
        SetQueueTimer)

    AddCB("ClassColorHP", "classColorHealth", "Class Colored Health Bars",
        "Colors player, target, and focus health bars based on class color.",
        SetClassColorHealth)

    AddCB("TrackPlayed", "trackPlayedTime", "Track /played Time",
        "Requests and stores total played time on each login. Displayed in your Character Profile.",
        nil)

    generalCard:SetPoint("BOTTOM", GetLast(), "BOTTOM", 0, -8)
    generalCard:SetPoint("RIGHT", panel, "RIGHT", -8, 0)
    return GetLast()
end

-----------------------------
-- Tweaks Combat Settings Builder
-----------------------------
function ns.BuildTweaksCombatSettings(panel, anchor, csSyncControls)
    local tweaksHeader = ns.AddSectionHeader(panel, anchor, "Tweaks", false)
    local tweaksCard = ns.CreateSectionCard(panel)
    tweaksCard:SetPoint("TOPLEFT", tweaksHeader, "TOPLEFT", -8, 8)
    local lastEl = tweaksHeader

    do
        local sqwLabel = panel:CreateFontString(nil, "ARTWORK", "GameFontNormal")
        sqwLabel:SetPoint("TOPLEFT", tweaksHeader, "BOTTOMLEFT", 0, -16)
        sqwLabel:SetText("Spell Queue Window")
        sqwLabel:SetTextColor(0.85, 0.85, 0.87)
        local sqwDesc = panel:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        sqwDesc:SetPoint("TOPLEFT", sqwLabel, "BOTTOMLEFT", 0, -2)
        sqwDesc:SetTextColor(0.55, 0.55, 0.6)
        sqwDesc:SetText("How early you can queue your next spell (in ms). Default: 400")
        local sqwMinus = ns.CreateThemedButton(panel, "-", 24, 24, "secondary")
        sqwMinus:SetPoint("TOPLEFT", sqwDesc, "BOTTOMLEFT", 0, -20)
        local slider = CreateFrame("Slider", "SKToolsCS_SQWSlider", panel, "OptionsSliderTemplate")
        slider:SetPoint("LEFT", sqwMinus, "RIGHT", 4, 0)
        slider:SetWidth(200)
        slider:SetMinMaxValues(0, 400)
        slider:SetValueStep(1)
        slider:SetObeyStepOnDrag(true)
        ns.StyleSlider(slider)
        local currentVal = tonumber(GetCVar("SpellQueueWindow")) or 400
        slider:SetValue(currentVal)
        _G["SKToolsCS_SQWSliderLow"]:SetText("0")
        _G["SKToolsCS_SQWSliderHigh"]:SetText("400")
        _G["SKToolsCS_SQWSliderText"]:SetText(currentVal .. " ms")
        local sqwPlus = ns.CreateThemedButton(panel, "+", 24, 24, "secondary")
        sqwPlus:SetPoint("LEFT", slider, "RIGHT", 4, 0)
        local editBox = ns.CreateThemedEditBox(panel, 50, 22)
        editBox:SetPoint("LEFT", sqwPlus, "RIGHT", 12, 0)
        editBox:SetNumeric(true)
        editBox:SetMaxLetters(3)
        editBox:SetText(tostring(currentVal))
        local msLabel = panel:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        msLabel:SetPoint("LEFT", editBox, "RIGHT", 4, 0)
        msLabel:SetTextColor(0.55, 0.55, 0.6)
        msLabel:SetText("ms")
        slider:SetScript("OnValueChanged", function(self, value)
            value = math.floor(value + 0.5)
            SetCVar("SpellQueueWindow", value)
            _G["SKToolsCS_SQWSliderText"]:SetText(value .. " ms")
            editBox:SetText(tostring(value))
        end)
        sqwMinus:SetScript("OnClick", function()
            local val = slider:GetValue() - 1
            if val >= 0 then slider:SetValue(val) end
        end)
        sqwPlus:SetScript("OnClick", function()
            local val = slider:GetValue() + 1
            if val <= 400 then slider:SetValue(val) end
        end)
        editBox:SetScript("OnEnterPressed", function(self)
            local val = tonumber(self:GetText()) or 400
            val = math.max(0, math.min(400, val))
            slider:SetValue(val)
            self:ClearFocus()
        end)
        editBox:SetScript("OnEscapePressed", function(self) self:ClearFocus() end)
        table.insert(csSyncControls, { type = "custom", refresh = function()
            local v = tonumber(GetCVar("SpellQueueWindow")) or 400
            slider:SetValue(v)
            editBox:SetText(tostring(v))
        end })
        local sqwSpacer = panel:CreateFontString(nil, "ARTWORK")
        sqwSpacer:SetPoint("TOPLEFT", sqwMinus, "BOTTOMLEFT", 0, -4)
        sqwSpacer:SetHeight(1)
        lastEl = sqwSpacer
    end

    do
        local cb = ns.CreateToggleSwitch(panel, "SKToolsCS_BagSort")
        cb:SetPoint("TOPLEFT", lastEl, "BOTTOMLEFT", 0, -16)
        cb.Text:SetText("Sort Bags Right to Left")
        cb.Text:SetTextColor(0.9, 0.9, 0.9)
        cb:SetChecked(C_Container.GetSortBagsRightToLeft())
        cb:SetScript("OnClick", function(self) C_Container.SetSortBagsRightToLeft(self:GetChecked()) end)
        local desc = panel:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        desc:SetPoint("TOPLEFT", cb.Text, "BOTTOMLEFT", 0, -2)
        desc:SetPoint("RIGHT", panel, "RIGHT", -16, 0)
        desc:SetJustifyH("LEFT")
        desc:SetTextColor(0.55, 0.55, 0.6)
        desc:SetText("Changes bag sorting direction. Uncheck for left to right.")
        table.insert(csSyncControls, { type = "custom",
            refresh = function() cb:SetChecked(C_Container.GetSortBagsRightToLeft()) end })
        local spacer = CreateFrame("Frame", nil, panel)
        spacer:SetSize(1, 1)
        spacer:SetPoint("LEFT", cb, "LEFT", 0, 0)
        spacer:SetPoint("TOP", desc, "BOTTOM", 0, 0)
        lastEl = spacer
    end

    do
        local cb = ns.CreateToggleSwitch(panel, "SKToolsCS_Sharpen")
        cb:SetPoint("TOPLEFT", lastEl, "BOTTOMLEFT", 0, -12)
        cb.Text:SetText("Sharpen Graphics")
        cb.Text:SetTextColor(0.9, 0.9, 0.9)
        cb:SetChecked(GetCVarBool("ResampleAlwaysSharpen"))
        cb:SetScript("OnClick", function(self) SetCVar("ResampleAlwaysSharpen", self:GetChecked() and "1" or "0") end)
        local desc = panel:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        desc:SetPoint("TOPLEFT", cb.Text, "BOTTOMLEFT", 0, -2)
        desc:SetPoint("RIGHT", panel, "RIGHT", -16, 0)
        desc:SetJustifyH("LEFT")
        desc:SetTextColor(0.55, 0.55, 0.6)
        desc:SetText("Toggles ResampleAlwaysSharpen for crisper visuals.")
        table.insert(csSyncControls, { type = "custom",
            refresh = function() cb:SetChecked(GetCVarBool("ResampleAlwaysSharpen")) end })
        local spacer = CreateFrame("Frame", nil, panel)
        spacer:SetSize(1, 1)
        spacer:SetPoint("LEFT", cb, "LEFT", 0, 0)
        spacer:SetPoint("TOP", desc, "BOTTOM", 0, 0)
        lastEl = spacer
    end

    -- Hide Action Bar Keybinds
    do
        local cb = ns.CreateToggleSwitch(panel, "SKToolsCS_HideKeybinds")
        cb:SetPoint("TOPLEFT", lastEl, "BOTTOMLEFT", 0, -12)
        cb.Text:SetText("Hide Action Bar Keybinds")
        cb.Text:SetTextColor(0.9, 0.9, 0.9)
        cb:SetChecked(SKToolsDB.hideKeybinds)
        cb:SetScript("OnClick", function(self)
            SKToolsDB.hideKeybinds = self:GetChecked()
            SetHideKeybinds(self:GetChecked())
        end)
        local desc = panel:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        desc:SetPoint("TOPLEFT", cb.Text, "BOTTOMLEFT", 0, -2)
        desc:SetPoint("RIGHT", panel, "RIGHT", -16, 0)
        desc:SetJustifyH("LEFT")
        desc:SetTextColor(0.55, 0.55, 0.6)
        desc:SetText("Hides keybind text on action bar buttons for a cleaner look.")
        table.insert(csSyncControls, { type = "custom",
            refresh = function() cb:SetChecked(SKToolsDB.hideKeybinds) end })
        local spacer = CreateFrame("Frame", nil, panel)
        spacer:SetSize(1, 1)
        spacer:SetPoint("LEFT", cb, "LEFT", 0, 0)
        spacer:SetPoint("TOP", desc, "BOTTOM", 0, 0)
        lastEl = spacer
    end

    -- Hide Action Bar Macro Names
    do
        local cb = ns.CreateToggleSwitch(panel, "SKToolsCS_HideMacroNames")
        cb:SetPoint("TOPLEFT", lastEl, "BOTTOMLEFT", 0, -12)
        cb.Text:SetText("Hide Action Bar Macro Names")
        cb.Text:SetTextColor(0.9, 0.9, 0.9)
        cb:SetChecked(SKToolsDB.hideMacroNames)
        cb:SetScript("OnClick", function(self)
            SKToolsDB.hideMacroNames = self:GetChecked()
            SetHideMacroNames(self:GetChecked())
        end)
        local desc = panel:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        desc:SetPoint("TOPLEFT", cb.Text, "BOTTOMLEFT", 0, -2)
        desc:SetPoint("RIGHT", panel, "RIGHT", -16, 0)
        desc:SetJustifyH("LEFT")
        desc:SetTextColor(0.55, 0.55, 0.6)
        desc:SetText("Hides macro name text on action bar buttons.")
        table.insert(csSyncControls, { type = "custom",
            refresh = function() cb:SetChecked(SKToolsDB.hideMacroNames) end })
        local spacer = CreateFrame("Frame", nil, panel)
        spacer:SetSize(1, 1)
        spacer:SetPoint("LEFT", cb, "LEFT", 0, 0)
        spacer:SetPoint("TOP", desc, "BOTTOM", 0, 0)
        lastEl = spacer
    end

    tweaksCard:SetPoint("BOTTOM", lastEl, "BOTTOM", 0, -8)
    tweaksCard:SetPoint("RIGHT", panel, "RIGHT", -8, 0)
    return lastEl
end
