-- SKTools Core
-- Defaults, constants, shared utilities, settings panel helpers

local _, ns = ...

-----------------------------
-- Defaults
-----------------------------
ns.defaults = {
    fastLoot = true,
    arenaNumbers = false,  -- disabled: Midnight secret values block all approaches
    pvpTabTarget = true,
    queueTimer = true,
    classColorHealth = false,
    trackPlayedTime = true,
    hideKeybinds = false,
    hideMacroNames = false,
    namePlates = true,
    namePlateStyles = { [1] = true, [2] = true },  -- active style indices (Glow + Crosses)
    targetHighlight = true,
    targetHighlightConfig = {
        colorR = 1, colorG = 0.84, colorB = 0,
        enemyColor = false,
        enemyColorR = 1, enemyColorG = 0, enemyColorB = 0,
        animSpeed = 0.35,
        startDistance = 80,
        crosshairThickness = 6,
        thickness = 3,
        npTargetStyle = 1,  -- 1=Crosshair, 2=Border, 0=None
        partyHighlight = true,
    },
    chatCopy = true,
    autoVendor = false,
    autoRepair = false,
    useGuildRepair = false,
    showMinimapButton = true,
    ccPlannerEnabled = false,
    arenaNotes = true,
    arenaNotesArena = true,
    arenaNotesBG = true,
    arenaNotesDungeon = true,
    arenaNotesRaid = true,
    pvpHistory = true,
    pveHistory = true,
    pvpTitles = true,
    pvpTitlesShowCached = true,
    pvpTitlesHideInCombat = true,
    pvpTitlesHideBadges = false,
    pvpTitlesScanAnim = true,
    totemTracking = true,
    totemTrackSpells = {
        groundingTotem = true,
        tremorTotem = true,
        capacitorTotem = true,
        counterstrikeTotem = true,
        staticFieldTotem = true,
        healingTideTotem = true,
        spiritLinkTotem = true,
        windrushTotem = true,
        earthgrabTotem = true,
        poisonCleansingTotem = true,
        healingStreamTotem = true,
        stormstreamTotem = true,
        surgingTotem = true,
        totemOfWrath = true,
    },
    totemButtonSlots = {},  -- [spellID] = { {bar, slot}, ... }
    smartMacros = true,
    smartMacroTemplates = {},  -- [macroIndex] = "template body with @placeholders"
    smartMacroLastResolved = {},  -- [macroIndex] = "last body written by EditMacro"
    smartMacroLastMapping = {},   -- [charPrefix] = { healer=name, tank=name, dps={name,...} }
    smartMacroNames = {},         -- [macroIndex] = "macro name" for index-shift detection
    battleIntel = true,
    battleIntelPos = nil,
    battleIntelShowPrediction = true,
    battleIntelShowCaptures = true,
    battleIntelShowFlags = true,
    statTooltips = false,
    statItemTooltips = false,
    statCompareTooltips = false,
    buttonLeftClick = "pvpHistory",
    buttonMiddleClick = "notes",
    buttonShiftClick = "pveHistory",
}

-----------------------------
-- Button Action Options
-----------------------------
ns.BUTTON_ACTIONS = {
    { value = "pvpHistory", label = "PvP History" },
    { value = "pveHistory", label = "PvE History" },
    { value = "notes",      label = "Notes" },
    { value = "settings",   label = "Settings" },
}

function ns.ExecuteButtonAction(action)
    if action == "pvpHistory" then
        SKPvPHistory_Toggle()
    elseif action == "pveHistory" then
        SKPvEHistory_Toggle()
    elseif action == "notes" then
        C_Timer.After(0, SKArenaNotes_ToggleBrowser)
    elseif action == "settings" then
        SlashCmdList["SKTOOLS"]("")
    end
end

-----------------------------
-- Nameplate Style Constants
-----------------------------
ns.NAMEPLATE_STYLES = {
    { key = "glow",      label = "Glow" },
    { key = "crosses",   label = "Crosses" },
    { key = "brackets",  label = "Brackets" },
    { key = "bars",      label = "Bars" },
    { key = "triangles", label = "Corners" },
    { key = "crosshair", label = "Crosshair" },
}

-----------------------------
-- Error Reporting (rate-limited)
-----------------------------
local _errorTimers = {}
function ns.SK_ReportError(funcName, err)
    local now = GetTime()
    if not _errorTimers[funcName] or (now - _errorTimers[funcName]) > 5 then
        _errorTimers[funcName] = now
        print("|cff00E5EESKTools|r |cffFF4444error|r [" .. funcName .. "]: " .. tostring(err))
    end
end

-----------------------------
-- Shared UI Constants
-----------------------------
ns.CYAN = { r = 0, g = 0.898, b = 0.933 }
local CYAN = ns.CYAN

-----------------------------
-- Clean Dark Color Palette
-----------------------------
ns.COLORS = {
    -- Surface elevation layers (7% → 11% → 16-19% luminance staircase)
    bgNav        = { 0.07, 0.07, 0.09, 0.97 },   -- deepest (sidebar)
    bgFrame      = { 0.09, 0.09, 0.12, 0.97 },   -- frame chrome
    bgContent    = { 0.11, 0.11, 0.14, 0.97 },   -- scrollable content area
    bgCard       = { 0.16, 0.16, 0.19, 1.00 },   -- elevated card surface

    -- Controls
    bgControl    = { 0.08, 0.08, 0.10, 1.00 },
    bgControlHov = { 0.12, 0.12, 0.15, 1.00 },

    -- Borders (barely visible — elevation does the work)
    border      = { 0.20, 0.20, 0.24, 0.30 },
    borderHover = { 0.30, 0.30, 0.35, 0.50 },

    -- Toggle
    toggleOffBg   = { 0.06, 0.06, 0.08, 1.00 },
    toggleOffKnob = { 0.50, 0.50, 0.53 },
    toggleOnBg    = { 0, 0.18, 0.20, 1.00 },

    -- Text hierarchy
    textHeader = { 0.95, 0.95, 0.95 },
    textLabel  = { 0.85, 0.85, 0.87 },
    textDesc   = { 0.50, 0.50, 0.55 },
    textMuted  = { 0.38, 0.38, 0.42 },
    textNav    = { 0.50, 0.50, 0.53 },

    -- Dividers
    divider = { 0.18, 0.18, 0.22, 0.40 },
}
local C = ns.COLORS

-----------------------------
-- Clean Dark Backdrop System
-----------------------------
local THIN_EDGE = "Interface\\Buttons\\WHITE8X8"

ns.BACKDROP_PANEL = {
    bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
    edgeFile = THIN_EDGE, edgeSize = 1,
    insets = { left = 1, right = 1, top = 1, bottom = 1 },
}
ns.BACKDROP_CARD = {
    bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
    edgeFile = THIN_EDGE, edgeSize = 1,
    insets = { left = 1, right = 1, top = 1, bottom = 1 },
}
ns.BACKDROP_CONTROL = {
    bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
    edgeFile = THIN_EDGE, edgeSize = 1,
    insets = { left = 1, right = 1, top = 1, bottom = 1 },
}
ns.BACKDROP_PILL = {
    bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
    edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
    edgeSize = 14,
    insets = { left = 4, right = 4, top = 4, bottom = 4 },
}

-----------------------------
-- Drop Shadow Helper
-----------------------------
function ns.CreateDropShadow(frame, size)
    size = size or 6
    local shadow = CreateFrame("Frame", nil, frame)
    shadow:SetFrameLevel(math.max(frame:GetFrameLevel() - 1, 0))
    shadow:SetPoint("TOPLEFT", -size, size)
    shadow:SetPoint("BOTTOMRIGHT", size, -size)

    local LAYERS = 3
    for i = 1, LAYERS do
        local alpha = 0.25 * (1 - (i - 1) / LAYERS)
        local offset = math.floor(size * i / LAYERS)
        -- Top
        local t = shadow:CreateTexture(nil, "BACKGROUND")
        t:SetColorTexture(0, 0, 0, alpha)
        t:SetPoint("TOPLEFT", frame, "TOPLEFT", -offset, offset)
        t:SetPoint("TOPRIGHT", frame, "TOPRIGHT", offset, offset)
        t:SetHeight(offset)
        -- Bottom
        local b = shadow:CreateTexture(nil, "BACKGROUND")
        b:SetColorTexture(0, 0, 0, alpha)
        b:SetPoint("BOTTOMLEFT", frame, "BOTTOMLEFT", -offset, -offset)
        b:SetPoint("BOTTOMRIGHT", frame, "BOTTOMRIGHT", offset, -offset)
        b:SetHeight(offset)
        -- Left
        local l = shadow:CreateTexture(nil, "BACKGROUND")
        l:SetColorTexture(0, 0, 0, alpha)
        l:SetPoint("TOPLEFT", frame, "TOPLEFT", -offset, 0)
        l:SetPoint("BOTTOMLEFT", frame, "BOTTOMLEFT", -offset, 0)
        l:SetWidth(offset)
        -- Right
        local r = shadow:CreateTexture(nil, "BACKGROUND")
        r:SetColorTexture(0, 0, 0, alpha)
        r:SetPoint("TOPRIGHT", frame, "TOPRIGHT", offset, 0)
        r:SetPoint("BOTTOMRIGHT", frame, "BOTTOMRIGHT", offset, 0)
        r:SetWidth(offset)
    end
    return shadow
end

-----------------------------
-- Toggle Switch Widget
-----------------------------
function ns.CreateToggleSwitch(parent, globalName)
    local TRACK_W, TRACK_H = 44, 22
    local KNOB_SIZE = 18
    local KNOB_OFF_X, KNOB_ON_X = 2, TRACK_W - KNOB_SIZE - 2
    local R = TRACK_H / 2

    local sw = CreateFrame("Button", globalName, parent)
    sw:SetSize(TRACK_W, TRACK_H)
    sw:RegisterForClicks("LeftButtonUp")

    -- Pill-shaped track: left cap (circle) + center fill + right cap (circle)
    local trackCenter = sw:CreateTexture(nil, "BACKGROUND")
    trackCenter:SetPoint("TOPLEFT", R, 0)
    trackCenter:SetPoint("BOTTOMRIGHT", -R, 0)

    local trackLeft = sw:CreateTexture(nil, "BACKGROUND")
    trackLeft:SetSize(TRACK_H, TRACK_H)
    trackLeft:SetPoint("LEFT", 0, 0)
    local maskLeft = sw:CreateMaskTexture()
    maskLeft:SetAllPoints(trackLeft)
    maskLeft:SetTexture("Interface\\CHARACTERFRAME\\TempPortraitAlphaMask",
        "CLAMPTOBLACKADDITIVE", "CLAMPTOBLACKADDITIVE")
    trackLeft:AddMaskTexture(maskLeft)

    local trackRight = sw:CreateTexture(nil, "BACKGROUND")
    trackRight:SetSize(TRACK_H, TRACK_H)
    trackRight:SetPoint("RIGHT", 0, 0)
    local maskRight = sw:CreateMaskTexture()
    maskRight:SetAllPoints(trackRight)
    maskRight:SetTexture("Interface\\CHARACTERFRAME\\TempPortraitAlphaMask",
        "CLAMPTOBLACKADDITIVE", "CLAMPTOBLACKADDITIVE")
    trackRight:AddMaskTexture(maskRight)

    local function SetTrackColor(r, g, b, a)
        trackLeft:SetColorTexture(r, g, b, a)
        trackCenter:SetColorTexture(r, g, b, a)
        trackRight:SetColorTexture(r, g, b, a)
    end

    -- Knob (circular via mask)
    local knob = CreateFrame("Frame", nil, sw)
    knob:SetSize(KNOB_SIZE, KNOB_SIZE)
    knob:SetPoint("LEFT", sw, "LEFT", KNOB_OFF_X, 0)
    knob:SetFrameLevel(sw:GetFrameLevel() + 2)
    local knobMask = knob:CreateMaskTexture()
    knobMask:SetAllPoints()
    knobMask:SetTexture("Interface\\CHARACTERFRAME\\TempPortraitAlphaMask",
        "CLAMPTOBLACKADDITIVE", "CLAMPTOBLACKADDITIVE")
    local knobTex = knob:CreateTexture(nil, "ARTWORK")
    knobTex:SetAllPoints()
    knobTex:SetColorTexture(C.toggleOffKnob[1], C.toggleOffKnob[2], C.toggleOffKnob[3], 1)
    knobTex:AddMaskTexture(knobMask)

    -- Label (positioned right of toggle, matching old checkbox .Text)
    sw.Text = sw:CreateFontString(nil, "OVERLAY", "GameFontNormal")
    sw.Text:SetPoint("LEFT", sw, "RIGHT", 8, 0)

    -- State
    local checked = false
    local currentKnobX = KNOB_OFF_X
    local animTicker = nil
    local isHovered = false

    local function UpdateVisual(animate)
        local targetX
        if checked then
            SetTrackColor(C.toggleOnBg[1], C.toggleOnBg[2], C.toggleOnBg[3], C.toggleOnBg[4])
            knobTex:SetColorTexture(CYAN.r, CYAN.g, CYAN.b, 1)
            targetX = KNOB_ON_X
        else
            if isHovered then
                SetTrackColor(C.bgControlHov[1], C.bgControlHov[2], C.bgControlHov[3], C.bgControlHov[4])
            else
                SetTrackColor(C.toggleOffBg[1], C.toggleOffBg[2], C.toggleOffBg[3], C.toggleOffBg[4])
            end
            knobTex:SetColorTexture(C.toggleOffKnob[1], C.toggleOffKnob[2], C.toggleOffKnob[3], 1)
            targetX = KNOB_OFF_X
        end
        if animate and targetX ~= currentKnobX then
            if animTicker then animTicker:Cancel() end
            animTicker = C_Timer.NewTicker(0.016, function()
                local diff = targetX - currentKnobX
                if math.abs(diff) < 0.5 then
                    currentKnobX = targetX
                    knob:ClearAllPoints()
                    knob:SetPoint("LEFT", sw, "LEFT", currentKnobX, 0)
                    animTicker:Cancel()
                    animTicker = nil
                    return
                end
                currentKnobX = currentKnobX + diff * 0.35
                knob:ClearAllPoints()
                knob:SetPoint("LEFT", sw, "LEFT", currentKnobX, 0)
            end)
        else
            currentKnobX = targetX
            knob:ClearAllPoints()
            knob:SetPoint("LEFT", sw, "LEFT", targetX, 0)
        end
    end

    function sw:GetChecked() return checked end
    function sw:SetChecked(val)
        checked = val and true or false
        UpdateVisual() -- instant (no animation for loading from DB)
    end

    -- OnClick interception: factory calls SetScript("OnClick", handler) AFTER creation.
    -- We store the user handler and call it after toggling internal state.
    local userOnClick = nil
    local origSetScript = sw.SetScript

    origSetScript(sw, "OnClick", function(self)
        checked = not checked
        UpdateVisual(true) -- animated slide
        if userOnClick then userOnClick(self) end
    end)

    sw.SetScript = function(self, scriptType, handler)
        if scriptType == "OnClick" then
            userOnClick = handler
        else
            origSetScript(self, scriptType, handler)
        end
    end

    -- GetScript override: return user handler, not internal handler
    local origGetScript = sw.GetScript
    sw.GetScript = function(self, scriptType)
        if scriptType == "OnClick" then
            return userOnClick
        else
            return origGetScript(self, scriptType)
        end
    end

    -- HookScript support
    local origHookScript = sw.HookScript
    sw.HookScript = function(self, scriptType, handler)
        if scriptType == "OnClick" then
            local prev = userOnClick
            userOnClick = function(s)
                if prev then prev(s) end
                handler(s)
            end
        else
            origHookScript(self, scriptType, handler)
        end
    end

    -- Enabled state
    function sw:SetEnabled(val)
        if val then
            self:EnableMouse(true)
            self:SetAlpha(1.0)
        else
            self:EnableMouse(false)
            self:SetAlpha(0.5)
        end
    end

    -- Hover
    origSetScript(sw, "OnEnter", function(self)
        isHovered = true
        if not checked then
            SetTrackColor(C.bgControlHov[1], C.bgControlHov[2], C.bgControlHov[3], C.bgControlHov[4])
        end
    end)
    origSetScript(sw, "OnLeave", function(self)
        isHovered = false
        UpdateVisual()
    end)

    UpdateVisual()
    return sw
end

-----------------------------
-- Themed Slider Styling
-----------------------------
function ns.StyleSlider(slider)
    local thumb = slider:GetThumbTexture()
    if thumb then
        thumb:SetColorTexture(CYAN.r, CYAN.g, CYAN.b, 0.8)
        thumb:SetSize(12, 20)
    end
    -- Style the track background
    local bg = slider:CreateTexture(nil, "BACKGROUND")
    bg:SetPoint("TOPLEFT", slider, "TOPLEFT", 0, -8)
    bg:SetPoint("BOTTOMRIGHT", slider, "BOTTOMRIGHT", 0, 8)
    bg:SetColorTexture(C.bgControl[1], C.bgControl[2], C.bgControl[3], 0.8)
    -- Style text labels
    local name = slider:GetName()
    if name then
        local low = _G[name .. "Low"]
        local high = _G[name .. "High"]
        local text = _G[name .. "Text"]
        if low then low:SetTextColor(0.5, 0.5, 0.55) end
        if high then high:SetTextColor(0.5, 0.5, 0.55) end
        if text then text:SetTextColor(CYAN.r, CYAN.g, CYAN.b) end
    end
end

-----------------------------
-- Themed EditBox
-----------------------------
function ns.CreateThemedEditBox(parent, width, height)
    local eb = CreateFrame("EditBox", nil, parent, "BackdropTemplate")
    eb:SetSize(width or 50, height or 22)
    eb:SetBackdrop(ns.BACKDROP_CONTROL)
    eb:SetBackdropColor(C.bgControl[1], C.bgControl[2], C.bgControl[3], C.bgControl[4])
    eb:SetBackdropBorderColor(C.border[1], C.border[2], C.border[3], C.border[4])
    eb:SetFontObject("GameFontHighlightSmall")
    eb:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
    eb:SetTextInsets(6, 6, 0, 0)
    eb:SetAutoFocus(false)
    eb:SetScript("OnEditFocusGained", function(self)
        self:SetBackdropBorderColor(CYAN.r, CYAN.g, CYAN.b, 0.7)
    end)
    eb:SetScript("OnEditFocusLost", function(self)
        self:SetBackdropBorderColor(C.border[1], C.border[2], C.border[3], C.border[4])
    end)
    return eb
end

-----------------------------
-- Themed Dropdown
-----------------------------
function ns.CreateThemedDropdown(parent, width, options, selectedValue, onChange)
    -- options = { { value = "key", label = "Display Text" }, ... }
    local dd = CreateFrame("Button", nil, parent, "BackdropTemplate")
    dd:SetSize(width or 140, 24)
    dd:SetBackdrop(ns.BACKDROP_CONTROL)
    dd:SetBackdropColor(C.bgControl[1], C.bgControl[2], C.bgControl[3], C.bgControl[4])
    dd:SetBackdropBorderColor(C.border[1], C.border[2], C.border[3], C.border[4])

    local text = dd:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
    text:SetPoint("LEFT", 8, 0)
    text:SetPoint("RIGHT", -18, 0)
    text:SetJustifyH("LEFT")
    text:SetTextColor(CYAN.r, CYAN.g, CYAN.b)

    local arrow = dd:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
    arrow:SetPoint("RIGHT", -6, 0)
    arrow:SetText("|cff888888v|r")

    dd.options = options
    dd.selectedValue = selectedValue

    local function UpdateText()
        for _, opt in ipairs(dd.options) do
            if opt.value == dd.selectedValue then
                text:SetText(opt.label)
                return
            end
        end
        text:SetText("—")
    end

    function dd:SetValue(val)
        dd.selectedValue = val
        UpdateText()
    end

    -- Popup menu
    local menu = CreateFrame("Frame", nil, dd, "BackdropTemplate")
    menu:SetBackdrop(ns.BACKDROP_CONTROL)
    menu:SetBackdropColor(C.bgCard[1], C.bgCard[2], C.bgCard[3], 0.97)
    menu:SetBackdropBorderColor(C.border[1], C.border[2], C.border[3], 0.80)
    menu:SetFrameStrata("FULLSCREEN_DIALOG")
    menu:SetWidth(width or 140)
    menu:Hide()

    local menuItems = {}
    for i, opt in ipairs(options) do
        local item = CreateFrame("Button", nil, menu)
        item:SetSize((width or 140) - 8, 20)
        item:SetPoint("TOPLEFT", menu, "TOPLEFT", 4, -4 - (i - 1) * 20)
        local itemBg = item:CreateTexture(nil, "BACKGROUND")
        itemBg:SetAllPoints()
        itemBg:SetColorTexture(0, 0, 0, 0)
        local itemText = item:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
        itemText:SetPoint("LEFT", 6, 0)
        itemText:SetText(opt.label)
        itemText:SetTextColor(0.8, 0.8, 0.8)
        item:SetScript("OnEnter", function()
            itemBg:SetColorTexture(CYAN.r * 0.1, CYAN.g * 0.1, CYAN.b * 0.1, 0.8)
            itemText:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
        end)
        item:SetScript("OnLeave", function()
            itemBg:SetColorTexture(0, 0, 0, 0)
            itemText:SetTextColor(0.8, 0.8, 0.8)
        end)
        item:SetScript("OnClick", function()
            dd.selectedValue = opt.value
            UpdateText()
            menu:Hide()
            if onChange then onChange(opt.value) end
        end)
        menuItems[i] = item
    end
    menu:SetHeight(#options * 20 + 8)

    dd:SetScript("OnClick", function()
        if menu:IsShown() then
            menu:Hide()
        else
            menu:ClearAllPoints()
            menu:SetPoint("TOPLEFT", dd, "BOTTOMLEFT", 0, -2)
            menu:Show()
        end
    end)
    dd:SetScript("OnEnter", function(self)
        self:SetBackdropBorderColor(C.borderHover[1], C.borderHover[2], C.borderHover[3], C.borderHover[4])
    end)
    dd:SetScript("OnLeave", function(self)
        self:SetBackdropBorderColor(C.border[1], C.border[2], C.border[3], C.border[4])
    end)

    -- Close menu when clicking elsewhere
    menu:SetScript("OnShow", function()
        menu:SetScript("OnUpdate", function(self)
            local ok, err = pcall(function()
                if not dd:IsMouseOver() and not self:IsMouseOver() then
                    self:Hide()
                end
            end)
            if not ok then self:Hide() end
        end)
    end)
    menu:SetScript("OnHide", function()
        menu:SetScript("OnUpdate", nil)
    end)

    UpdateText()
    return dd
end

-----------------------------
-- Side Nav Builder (shared by Blizzard settings + combat frame)
-----------------------------
-- Icons for each nav tab
local NAV_ICONS = {
    general      = "Interface\\Icons\\INV_Gizmo_GoblingTonkController",
    stats        = "Interface\\Icons\\Spell_ChargePositive",
    nameplates   = "Interface\\Icons\\Ability_Hunter_MasterMarksman",
    history      = "Interface\\Icons\\Achievement_Arena_2v2_7",
    battleground = "Interface\\Icons\\Achievement_BG_winAB",
    notes        = "Interface\\Icons\\INV_Scroll_02",
    pvptitles    = "Interface\\Icons\\INV_Misc_Spyglass_03",
    macros       = "Interface\\Icons\\Trade_Engineering",
    profile      = "Interface\\Icons\\Achievement_Character_Human_Male",
    shaman       = "Interface\\Icons\\ClassIcon_Shaman",
}

local NAV_TOOLTIPS = {
    general      = "Quality-of-life tools for PvE and PvP",
    stats        = "Secondary stat diminishing returns",
    nameplates   = "Nameplate indicators and target highlighting",
    history      = "PvP and PvE match history",
    battleground = "Capture timers and score prediction for battlegrounds",
    notes        = "Scouting notes for arenas, BGs, dungeons, and raids",
    pvptitles    = "Scan players for PvP achievements",
    macros       = "Smart macro role placeholders",
    profile      = "Character and account statistics",
    shaman       = "Totem duration tracking and cooldowns",
}

function ns.BuildSideNav(parentFrame, navAnchorTop, navAnchorBottom, contentTL, contentBR)
    -- navAnchorTop/Bottom: {frame, x, y} tables for nav panel positioning
    -- contentTL/BR: {frame, x, y} for content container positioning

    local navPanel = CreateFrame("Frame", nil, parentFrame)
    navPanel:SetPoint("TOPLEFT", navAnchorTop[1], "TOPLEFT", navAnchorTop[2], navAnchorTop[3])
    navPanel:SetPoint("BOTTOMLEFT", navAnchorBottom[1], "BOTTOMLEFT", navAnchorBottom[2], navAnchorBottom[3])
    navPanel:SetWidth(170)

    local navBg = navPanel:CreateTexture(nil, "BACKGROUND")
    navBg:SetAllPoints()
    navBg:SetColorTexture(C.bgNav[1], C.bgNav[2], C.bgNav[3], C.bgNav[4])

    local navDivider = navPanel:CreateTexture(nil, "ARTWORK")
    navDivider:SetWidth(1)
    navDivider:SetPoint("TOPRIGHT", navPanel, "TOPRIGHT", 0, 0)
    navDivider:SetPoint("BOTTOMRIGHT", navPanel, "BOTTOMRIGHT", 0, 0)
    navDivider:SetColorTexture(C.divider[1], C.divider[2], C.divider[3], C.divider[4])

    -- Top branding
    local brandFrame = CreateFrame("Frame", nil, navPanel)
    brandFrame:SetSize(170, 40)
    brandFrame:SetPoint("TOPLEFT", navPanel, "TOPLEFT", 0, 0)

    local brandText = brandFrame:CreateFontString(nil, "OVERLAY", "GameFontNormalLarge")
    brandText:SetPoint("LEFT", 16, 0)
    brandText:SetText("|cff00E5EESK|r|cff88888CTools|r")

    local brandLine = brandFrame:CreateTexture(nil, "ARTWORK")
    brandLine:SetHeight(1)
    brandLine:SetPoint("BOTTOMLEFT", brandFrame, "BOTTOMLEFT", 8, 0)
    brandLine:SetPoint("BOTTOMRIGHT", brandFrame, "BOTTOMRIGHT", -8, 0)
    brandLine:SetColorTexture(C.divider[1], C.divider[2], C.divider[3], C.divider[4])

    -- Content container
    local contentContainer = CreateFrame("Frame", nil, parentFrame)
    contentContainer:SetPoint("TOPLEFT", contentTL[1], "TOPLEFT", contentTL[2], contentTL[3])
    contentContainer:SetPoint("BOTTOMRIGHT", contentBR[1], "BOTTOMRIGHT", contentBR[2], contentBR[3])
    contentContainer:EnableMouseWheel(true)
    local contentBgTex = contentContainer:CreateTexture(nil, "BACKGROUND")
    contentBgTex:SetAllPoints()
    contentBgTex:SetColorTexture(C.bgContent[1], C.bgContent[2], C.bgContent[3], C.bgContent[4])

    -- Nav items
    local TAB_DEFS = {
        { key = "general",    label = "General" },
        { key = "stats",      label = "Stats" },
        { key = "nameplates", label = "Nameplates" },
        { key = "history",      label = "History" },
        { key = "battleground", label = "Battle Intel" },
        { key = "notes",        label = "Notes" },
        { key = "pvptitles",  label = "PvP Scanner" },
        { key = "macros",     label = "Macros" },
        { key = "profile",   label = "Profile" },
    }
    if select(2, UnitClass("player")) == "SHAMAN" then
        table.insert(TAB_DEFS, { key = "shaman", label = "Shaman" })
    end

    local navItems = {}
    local contentPanels = {}
    local activeTabKey = nil

    local function SelectTab(key)
        if activeTabKey == key then return end
        for _, item in ipairs(navItems) do
            if item.key == key then
                item.bg:SetColorTexture(0, CYAN.g * 0.04, CYAN.b * 0.04, 0.8)
                item.text:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
                item.icon:SetVertexColor(1, 1, 1)
                item.icon:SetDesaturated(false)
                item.activeBar:SetColorTexture(CYAN.r, CYAN.g, CYAN.b, 1)
                item.activeBar:Show()
                item.isActive = true
            else
                item.bg:SetColorTexture(0, 0, 0, 0)
                item.text:SetTextColor(C.textNav[1], C.textNav[2], C.textNav[3])
                item.icon:SetVertexColor(0.45, 0.45, 0.48)
                item.icon:SetDesaturated(true)
                item.activeBar:Hide()
                item.isActive = false
            end
        end
        for k, p in pairs(contentPanels) do
            if k == key then
                p:SetAlpha(0)
                p:Show()
                -- Slide in from right: start offset, animate to 0
                local child = p:GetScrollChild()
                if child then
                    child:ClearAllPoints()
                    child:SetPoint("TOPLEFT", 15, 0)
                end
                ns.FadeIn(p, 0.18)
                -- Slide child back to origin
                if child then
                    C_Timer.After(0.001, function()
                        child:ClearAllPoints()
                        child:SetPoint("TOPLEFT", 0, 0)
                    end)
                end
            else
                p:Hide()
            end
        end
        activeTabKey = key
    end

    for i, def in ipairs(TAB_DEFS) do
        local item = CreateFrame("Button", nil, navPanel)
        item:SetSize(170, 42)
        if i == 1 then
            item:SetPoint("TOPLEFT", brandFrame, "BOTTOMLEFT", 0, 0)
        else
            item:SetPoint("TOPLEFT", navItems[i - 1], "BOTTOMLEFT", 0, 0)
        end
        item.key = def.key
        item.isActive = false

        local bg = item:CreateTexture(nil, "BACKGROUND")
        bg:SetPoint("TOPLEFT", 4, -1)
        bg:SetPoint("BOTTOMRIGHT", 0, 1)
        bg:SetColorTexture(0, 0, 0, 0)
        item.bg = bg

        local barTex = item:CreateTexture(nil, "ARTWORK")
        barTex:SetWidth(3)
        barTex:SetPoint("TOPLEFT", item, "TOPLEFT", 0, 0)
        barTex:SetPoint("BOTTOMLEFT", item, "BOTTOMLEFT", 0, 0)
        barTex:SetColorTexture(1, 1, 1, 1)
        barTex:Hide()
        item.activeBar = barTex

        -- Icon (circular via mask)
        local icon = item:CreateTexture(nil, "OVERLAY")
        icon:SetSize(32, 32)
        icon:SetPoint("LEFT", 12, 0)
        icon:SetTexture(NAV_ICONS[def.key] or "Interface\\Icons\\INV_Misc_QuestionMark")
        icon:SetVertexColor(0.5, 0.5, 0.5)
        icon:SetDesaturated(true)
        local iconMask = item:CreateMaskTexture()
        iconMask:SetSize(28, 28)
        iconMask:SetPoint("CENTER", icon, "CENTER")
        iconMask:SetTexture("Interface\\CHARACTERFRAME\\TempPortraitAlphaMask",
            "CLAMPTOBLACKADDITIVE", "CLAMPTOBLACKADDITIVE")
        icon:AddMaskTexture(iconMask)
        item.icon = icon

        local text = item:CreateFontString(nil, "OVERLAY", "GameFontNormal")
        -- Adjust text position for larger icon
        text:SetPoint("LEFT", icon, "RIGHT", 8, 0)
        text:SetText(def.label)
        text:SetTextColor(C.textNav[1], C.textNav[2], C.textNav[3])
        item.text = text

        -- Status dot (right side of nav item)
        local statusDot = item:CreateTexture(nil, "OVERLAY")
        statusDot:SetSize(8, 8)
        statusDot:SetPoint("RIGHT", item, "RIGHT", -12, 0)
        local dotMask = item:CreateMaskTexture()
        dotMask:SetAllPoints(statusDot)
        dotMask:SetTexture("Interface\\CHARACTERFRAME\\TempPortraitAlphaMask",
            "CLAMPTOBLACKADDITIVE", "CLAMPTOBLACKADDITIVE")
        statusDot:SetColorTexture(0.4, 0.4, 0.4, 0.4)
        statusDot:AddMaskTexture(dotMask)
        item.statusDot = statusDot

        if i > 1 then
            local sep = item:CreateTexture(nil, "ARTWORK")
            sep:SetHeight(1)
            sep:SetPoint("TOPLEFT", item, "TOPLEFT", 8, 0)
            sep:SetPoint("TOPRIGHT", item, "TOPRIGHT", -8, 0)
            sep:SetColorTexture(C.divider[1], C.divider[2], C.divider[3], 0.30)
        end

        item:SetScript("OnEnter", function(self)
            if not self.isActive then
                self.bg:SetColorTexture(0.11, 0.11, 0.14, 0.6)
                self.text:SetTextColor(C.textLabel[1], C.textLabel[2], C.textLabel[3])
                self.icon:SetVertexColor(0.75, 0.75, 0.78)
                self.icon:SetDesaturated(false)
            end
            GameTooltip:SetOwner(self, "ANCHOR_RIGHT")
            GameTooltip:SetText(def.label, 1, 1, 1)
            if NAV_TOOLTIPS[def.key] then
                GameTooltip:AddLine(NAV_TOOLTIPS[def.key], 0.7, 0.7, 0.7, true)
            end
            GameTooltip:Show()
        end)
        item:SetScript("OnLeave", function(self)
            if not self.isActive then
                self.bg:SetColorTexture(0, 0, 0, 0)
                self.text:SetTextColor(C.textNav[1], C.textNav[2], C.textNav[3])
                self.icon:SetVertexColor(0.45, 0.45, 0.48)
                self.icon:SetDesaturated(true)
            end
            GameTooltip:Hide()
        end)
        item:SetScript("OnClick", function() SelectTab(def.key) end)

        navItems[#navItems + 1] = item
    end

    -- Version footer
    local footerLine = navPanel:CreateTexture(nil, "ARTWORK")
    footerLine:SetHeight(1)
    footerLine:SetPoint("BOTTOMLEFT", navPanel, "BOTTOMLEFT", 8, 20)
    footerLine:SetPoint("BOTTOMRIGHT", navPanel, "BOTTOMRIGHT", -8, 20)
    footerLine:SetColorTexture(C.divider[1], C.divider[2], C.divider[3], 0.30)

    local version = C_AddOns and C_AddOns.GetAddOnMetadata("SKTools", "Version") or "?"
    local versionText = navPanel:CreateFontString(nil, "OVERLAY", "GameFontDisableSmall")
    versionText:SetPoint("BOTTOM", navPanel, "BOTTOM", 0, 6)
    local cm = C.textMuted
    versionText:SetText(string.format("|cff%02x%02x%02xv%s|r", cm[1]*255, cm[2]*255, cm[3]*255, version))

    return navPanel, contentContainer, contentPanels, SelectTab, navItems
end

-----------------------------
-- Themed Button
-----------------------------
function ns.CreateThemedButton(parent, text, width, height, style)
    width = width or 100
    height = height or 24
    style = style or "secondary"

    local btn = CreateFrame("Button", nil, parent, "BackdropTemplate")
    btn:SetSize(width, height)
    btn:SetBackdrop(ns.BACKDROP_CONTROL)

    btn.label = btn:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
    btn.label:SetPoint("CENTER")
    btn.label:SetText(text)

    local normalBg, normalBorder, normalText
    local hoverBg, hoverBorder, hoverText

    if style == "primary" then
        normalBg     = { 0, CYAN.g * 0.15, CYAN.b * 0.15, 1.0 }
        normalBorder = { CYAN.r, CYAN.g, CYAN.b, 0.35 }
        normalText   = { CYAN.r, CYAN.g, CYAN.b }
        hoverBg      = { 0, CYAN.g * 0.22, CYAN.b * 0.22, 1.0 }
        hoverBorder  = { CYAN.r, CYAN.g, CYAN.b, 0.60 }
        hoverText    = { 1, 1, 1 }
    elseif style == "danger" then
        normalBg     = { 0.18, 0.06, 0.06, 1.0 }
        normalBorder = { 0.50, 0.15, 0.15, 0.40 }
        normalText   = { 0.90, 0.40, 0.40 }
        hoverBg      = { 0.25, 0.08, 0.08, 1.0 }
        hoverBorder  = { 0.70, 0.20, 0.20, 0.60 }
        hoverText    = { 1, 0.5, 0.5 }
    else -- secondary
        normalBg     = { 0.14, 0.14, 0.17, 1.0 }
        normalBorder = { 0.24, 0.24, 0.28, 0.40 }
        normalText   = { C.textLabel[1], C.textLabel[2], C.textLabel[3] }
        hoverBg      = { 0.18, 0.18, 0.21, 1.0 }
        hoverBorder  = { 0.30, 0.30, 0.35, 0.60 }
        hoverText    = { CYAN.r, CYAN.g, CYAN.b }
    end

    btn:SetBackdropColor(unpack(normalBg))
    btn:SetBackdropBorderColor(unpack(normalBorder))
    btn.label:SetTextColor(unpack(normalText))

    btn:SetScript("OnEnter", function(self)
        self:SetBackdropColor(unpack(hoverBg))
        self:SetBackdropBorderColor(unpack(hoverBorder))
        self.label:SetTextColor(unpack(hoverText))
    end)
    btn:SetScript("OnLeave", function(self)
        self:SetBackdropColor(unpack(normalBg))
        self:SetBackdropBorderColor(unpack(normalBorder))
        self.label:SetTextColor(unpack(normalText))
    end)

    btn.SetText = function(self, t) self.label:SetText(t) end
    btn.GetText = function(self) return self.label:GetText() end

    return btn
end

-----------------------------
-- Scrollbar Styling
-----------------------------
function ns.StyleScrollBar(scrollFrame)
    local scrollBar = scrollFrame.ScrollBar
    if not scrollBar then return end
    scrollBar:SetWidth(12)
    if scrollBar.Background then scrollBar.Background:SetAlpha(0) end
    if scrollBar.NineSlice then scrollBar.NineSlice:SetAlpha(0) end
    local thumb = scrollBar.ThumbTexture or scrollBar.thumbTexture
    if thumb then
        thumb:SetColorTexture(CYAN.r, CYAN.g, CYAN.b, 0.25)
        thumb:SetWidth(8)
    end
    if scrollBar.ScrollUpButton then scrollBar.ScrollUpButton:SetAlpha(0); scrollBar.ScrollUpButton:SetHeight(1) end
    if scrollBar.ScrollDownButton then scrollBar.ScrollDownButton:SetAlpha(0); scrollBar.ScrollDownButton:SetHeight(1) end
end

-----------------------------
-- Cyan Accent Line
-----------------------------
function ns.CreateCyanAccentLine(parent)
    local line = parent:CreateTexture(nil, "OVERLAY")
    line:SetHeight(2)
    line:SetPoint("TOPLEFT", parent, "TOPLEFT", 1, -1)
    line:SetPoint("TOPRIGHT", parent, "TOPRIGHT", -1, -1)
    line:SetColorTexture(CYAN.r, CYAN.g, CYAN.b, 0.8)
    return line
end

-----------------------------
-- Fade Helpers
-----------------------------
-----------------------------
-- Smooth Scrolling
-----------------------------
function ns.AttachSmoothScroll(scrollFrame, wheelParent, scrollSpeed)
    scrollSpeed = scrollSpeed or 60
    local targetScroll = 0
    local currentScroll = 0
    local isScrolling = false
    local ticker = nil
    local LERP_SPEED = 12

    local function StopScrolling()
        isScrolling = false
        if ticker then ticker:Cancel(); ticker = nil end
    end

    local function StartScrolling()
        if isScrolling then return end
        isScrolling = true
        currentScroll = scrollFrame:GetVerticalScroll()
        ticker = C_Timer.NewTicker(0.016, function()
            if not scrollFrame or not scrollFrame:IsVisible() then
                StopScrolling()
                return
            end
            local diff = targetScroll - currentScroll
            if math.abs(diff) < 0.5 then
                scrollFrame:SetVerticalScroll(targetScroll)
                currentScroll = targetScroll
                StopScrolling()
                return
            end
            currentScroll = currentScroll + diff * math.min(1, LERP_SPEED * 0.016)
            scrollFrame:SetVerticalScroll(currentScroll)
        end)
    end

    scrollFrame:EnableMouseWheel(false)
    local sfBar = scrollFrame.ScrollBar
    if sfBar then sfBar:EnableMouseWheel(false) end

    wheelParent:HookScript("OnMouseWheel", function(_, delta)
        if not scrollFrame:IsVisible() then return end
        local maxScroll = scrollFrame:GetVerticalScrollRange()
        if maxScroll <= 0 then return end
        if not isScrolling then
            currentScroll = scrollFrame:GetVerticalScroll()
            targetScroll = currentScroll
        end
        targetScroll = math.max(0, math.min(targetScroll - delta * scrollSpeed, maxScroll))
        StartScrolling()
    end)
end

-----------------------------
-- Fade Helpers
-----------------------------
function ns.FadeIn(frame, duration)
    if not frame then return end
    duration = duration or 0.15
    if frame._fadeTicker then frame._fadeTicker:Cancel() end
    frame:SetAlpha(0)
    frame:Show()
    local elapsed = 0
    frame._fadeTicker = C_Timer.NewTicker(0.016, function(ticker)
        elapsed = elapsed + 0.016
        local pct = math.min(elapsed / duration, 1)
        frame:SetAlpha(pct)
        if pct >= 1 then
            ticker:Cancel()
            frame._fadeTicker = nil
        end
    end)
end

function ns.FadeOut(frame, duration, onFinish)
    if not frame then return end
    duration = duration or 0.15
    if frame._fadeTicker then frame._fadeTicker:Cancel() end
    local startAlpha = frame:GetAlpha()
    local elapsed = 0
    frame._fadeTicker = C_Timer.NewTicker(0.016, function(ticker)
        elapsed = elapsed + 0.016
        local pct = math.min(elapsed / duration, 1)
        frame:SetAlpha(startAlpha * (1 - pct))
        if pct >= 1 then
            ticker:Cancel()
            frame._fadeTicker = nil
            frame:Hide()
            if onFinish then onFinish() end
        end
    end)
end

-----------------------------
-- Tab Content Title
-----------------------------
function ns.CreateTabTitle(content, titleText, subtitleText)
    local title = content:CreateFontString(nil, "ARTWORK", "GameFontNormalLarge")
    title:SetPoint("TOPLEFT", 12, -16)
    title:SetText(titleText)
    title:SetTextColor(CYAN.r, CYAN.g, CYAN.b)

    local anchor = title
    if subtitleText and subtitleText ~= "" then
        local sub = content:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        sub:SetPoint("TOPLEFT", title, "BOTTOMLEFT", 0, -4)
        sub:SetText(subtitleText)
        sub:SetTextColor(C.textDesc[1], C.textDesc[2], C.textDesc[3])
        anchor = sub
    end
    return anchor
end

-----------------------------
-- Section Header Helper
-----------------------------
function ns.AddSectionHeader(panel, anchor, text, isFirst)
    local header = panel:CreateFontString(nil, "ARTWORK", "GameFontNormalLarge")
    header:SetPoint("LEFT", panel, "LEFT", 12, 0)
    header:SetPoint("TOP", anchor, "BOTTOM", 0, isFirst and -28 or -40)
    header:SetText(text)
    header:SetTextColor(C.textHeader[1], C.textHeader[2], C.textHeader[3])
    return header
end

-----------------------------
-- Section Card Helper
-----------------------------
function ns.CreateSectionCard(parent)
    local card = CreateFrame("Frame", nil, parent, "BackdropTemplate")
    card:SetBackdrop(ns.BACKDROP_CARD)
    card:SetBackdropColor(0.16, 0.16, 0.20, 1.0)
    card:SetBackdropBorderColor(0.35, 0.35, 0.40, 0.8)

    card:SetFrameLevel(math.max(parent:GetFrameLevel() - 1, 0))
    return card
end

-----------------------------
-- Scrollable Panel Helper
-----------------------------
function ns.CreateScrollablePanel(titleText, subtitleText, contentHeight)
    local outer = CreateFrame("Frame")
    outer.name = titleText

    local sf = CreateFrame("ScrollFrame", nil, outer, "UIPanelScrollFrameTemplate")
    sf:SetPoint("TOPLEFT", 0, 0)
    sf:SetPoint("BOTTOMRIGHT", -26, 10)

    local content = CreateFrame("Frame")
    content:SetWidth(600)
    content:SetHeight((contentHeight or 1400) + 30)
    sf:SetScrollChild(content)
    sf:SetScript("OnSizeChanged", function(self, w) content:SetWidth(w) end)
    outer:EnableMouseWheel(true)
    ns.AttachSmoothScroll(sf, outer)

    local title = content:CreateFontString(nil, "ARTWORK", "GameFontNormalLarge")
    title:SetPoint("TOPLEFT", 16, -16)
    title:SetText(titleText)

    local anchor = title
    if subtitleText then
        local subtitle = content:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        subtitle:SetPoint("TOPLEFT", title, "BOTTOMLEFT", 0, -8)
        subtitle:SetText(subtitleText)
        anchor = subtitle
    end

    return outer, content, anchor
end

-----------------------------
-- Main Settings Checkbox Factory
-----------------------------
function ns.MakeCheckboxFactory(targetPanel, startAnchor)
    local lastEl = startAnchor
    local isFirst = true
    local function AddCheckbox(name, key, label, description, onToggle)
        local cb = ns.CreateToggleSwitch(targetPanel, "SKTools" .. name .. "CB")
        cb:SetPoint("TOPLEFT", lastEl, "BOTTOMLEFT", 0, isFirst and -18 or -16)
        cb.Text:SetText(label)
        cb.Text:SetTextColor(0.9, 0.9, 0.9)
        cb:SetChecked(SKToolsDB[key])
        cb:SetScript("OnClick", function(self)
            SKToolsDB[key] = self:GetChecked()
            if onToggle then onToggle(SKToolsDB[key]) end
            if ns.RefreshMainNavStatus then ns.RefreshMainNavStatus() end
        end)

        local desc = targetPanel:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        desc:SetPoint("TOPLEFT", cb.Text, "BOTTOMLEFT", 0, -2)
        desc:SetPoint("RIGHT", targetPanel, "RIGHT", -16, 0)
        desc:SetJustifyH("LEFT")
        desc:SetTextColor(0.55, 0.55, 0.6)
        desc:SetText(description)
        -- Spacer to reset left alignment (desc is indented under label text)
        local spacer = CreateFrame("Frame", nil, targetPanel)
        spacer:SetSize(1, 1)
        spacer:SetPoint("LEFT", cb, "LEFT", 0, 0)
        spacer:SetPoint("TOP", desc, "BOTTOM", 0, 0)
        lastEl = spacer
        isFirst = false
    end
    local function GetLast() return lastEl end
    local function SetLast(el) lastEl = el end
    return AddCheckbox, GetLast, SetLast
end

-----------------------------
-- Combat Settings Tab Content Helper
-----------------------------
function ns.MakeTabContent(parentFrame, contentContainer)
    local sf = CreateFrame("ScrollFrame", nil, parentFrame, "UIPanelScrollFrameTemplate")
    if contentContainer then
        sf:SetPoint("TOPLEFT", contentContainer, "TOPLEFT", 0, 0)
        sf:SetPoint("BOTTOMRIGHT", contentContainer, "BOTTOMRIGHT", -22, 0)
    else
        sf:SetPoint("TOPLEFT", 8, -66)
        sf:SetPoint("BOTTOMRIGHT", -30, 8)
    end
    local content = CreateFrame("Frame")
    content:SetWidth(440)
    content:SetHeight(900)
    sf:SetScrollChild(content)
    sf:SetScript("OnSizeChanged", function(self, w) content:SetWidth(w - 4) end)
    ns.StyleScrollBar(sf)
    local wheelParent = contentContainer or parentFrame
    ns.AttachSmoothScroll(sf, wheelParent)
    sf:Hide()
    return sf, content
end

-----------------------------
-- Combat Settings Checkbox Factory
-----------------------------
function ns.MakeCombatCBFactory(targetPanel, startAnchor, csSyncControls)
    local lastEl = startAnchor
    local isFirst = true
    local function AddCB(name, key, label, description, onToggle)
        local cb = ns.CreateToggleSwitch(targetPanel, "SKToolsCS_" .. name)
        cb:SetPoint("TOPLEFT", lastEl, "BOTTOMLEFT", 0, isFirst and -18 or -16)
        cb.Text:SetText(label)
        cb.Text:SetTextColor(0.9, 0.9, 0.9)
        cb:SetChecked(SKToolsDB[key])
        cb:SetScript("OnClick", function(self)
            SKToolsDB[key] = self:GetChecked()
            if onToggle then onToggle(SKToolsDB[key]) end
            if ns.RefreshCombatNavStatus then ns.RefreshCombatNavStatus() end
        end)

        local desc = targetPanel:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        desc:SetPoint("TOPLEFT", cb.Text, "BOTTOMLEFT", 0, -2)
        desc:SetPoint("RIGHT", targetPanel, "RIGHT", -16, 0)
        desc:SetJustifyH("LEFT")
        desc:SetTextColor(0.55, 0.55, 0.6)
        desc:SetText(description)
        table.insert(csSyncControls, { type = "checkbox", widget = cb, key = key })
        -- Spacer to reset left alignment (desc is indented under label text)
        local spacer = CreateFrame("Frame", nil, targetPanel)
        spacer:SetSize(1, 1)
        spacer:SetPoint("LEFT", cb, "LEFT", 0, 0)
        spacer:SetPoint("TOP", desc, "BOTTOM", 0, 0)
        lastEl = spacer
        isFirst = false
        return cb
    end
    local function GetLast() return lastEl end
    local function SetLast(el) lastEl = el; isFirst = false end
    return AddCB, GetLast, SetLast
end
