-- SKTools Character Profile
-- Passively collects character statistics on login, displays per-character and account-wide data

local _, ns = ...

-----------------------------
-- Constants
-----------------------------
local CYAN = ns.CYAN
local C = ns.COLORS

-- Expansion sort order for profession tiers (newest first)
local EXPANSION_SORT = {
    ["Khaz Algar"] = 1, ["Dragon Isles"] = 2, ["Shadowlands"] = 3,
    ["Kul Tiran"] = 4, ["Zandalari"] = 4,
    ["Legion"] = 5, ["Draenor"] = 6, ["Pandaria"] = 7,
    ["Cataclysm"] = 8, ["Northrend"] = 9, ["Outland"] = 10,
}

-- Secondary profession tier skill line IDs (not returned by GetAllProfessionTradeSkillLines)
-- Source: warcraft.wiki.gg/wiki/TradeSkillLineID
local SECONDARY_TIER_LINES = {
    -- Cooking (base 185)
    2548, 2547, 2546, 2545, 2544, 2543, 2542, 2541, 2752, 2824,
    -- Fishing (base 356)
    2592, 2591, 2590, 2589, 2588, 2587, 2586, 2585, 2754, 2826,
}

-- PvP bracket indices (verified: wowinterface.com/forums/showthread.php?t=60009)
local BRACKETS = {
    { idx = 1, label = "2v2" },
    { idx = 2, label = "3v3" },
    { idx = 4, label = "RBG" },
    { idx = 7, label = "Shuffle" },
    { idx = 9, label = "Blitz" },
}

-- All-time highest personal rating stat IDs (from Wowhead)
local HIGHEST_RATING_STATS = {
    [1] = 370,  -- Highest 2v2 personal rating
    [2] = 595,  -- Highest 3v3 personal rating
}

-- GetStatistic IDs grouped for display
-- Source: https://www.wowhead.com/achievements/character-statistics/player-vs-player/rated-arenas
local STAT_GROUPS = {
    { header = "Combat", winRates = true, stats = {
        { key = "arenasPlayed",     id = 838,   label = "Arenas Played" },
        { key = "arenasWon",        id = 837,   label = "Arenas Won" },
        { key = "blitzPlayed",      id = 40199, label = "Blitz Played" },
        { key = "blitzWon",         id = 40201, label = "Blitz Won" },
        { key = "duelsWon",         id = 319,   label = "Duels Won" },
        { key = "duelsLost",        id = 320,   label = "Duels Lost" },
    }},
    { header = "Exploration", stats = {
        { key = "questsCompleted",  id = 98,    label = "Quests Completed" },
        { key = "dailyQuests",      id = 97,    label = "Daily Quests" },
        { key = "creaturesKilled",  id = 1197,  label = "Creatures Killed" },
        { key = "totalDeaths",      id = 60,    label = "Deaths" },
    }},
    { header = "Economy", stats = {
        { key = "auctionsPosted",   id = 329,   label = "Auctions Posted" },
    }},
    { header = "Travel", stats = {
        { key = "flightPathsTaken", id = 349,   label = "Flight Paths" },
        { key = "hearthstoneUses",  id = 353,   label = "Hearthstone Uses" },
    }},
}

-- Flat STAT_DEFS derived from groups (for backward-compatible data collection)
local STAT_DEFS = {}
for _, group in ipairs(STAT_GROUPS) do
    for _, def in ipairs(group.stats) do
        STAT_DEFS[#STAT_DEFS + 1] = def
    end
end

-- Class display order + colors (for account view bar)
local CLASS_ORDER = {
    "WARRIOR", "PALADIN", "HUNTER", "ROGUE", "PRIEST",
    "DEATHKNIGHT", "SHAMAN", "MAGE", "WARLOCK", "MONK",
    "DRUID", "DEMONHUNTER", "EVOKER",
}

local CLASS_ICONS = {
    WARRIOR      = "Interface\\Icons\\ClassIcon_Warrior",
    PALADIN      = "Interface\\Icons\\ClassIcon_Paladin",
    HUNTER       = "Interface\\Icons\\ClassIcon_Hunter",
    ROGUE        = "Interface\\Icons\\ClassIcon_Rogue",
    PRIEST       = "Interface\\Icons\\ClassIcon_Priest",
    DEATHKNIGHT  = "Interface\\Icons\\ClassIcon_DeathKnight",
    SHAMAN       = "Interface\\Icons\\ClassIcon_Shaman",
    MAGE         = "Interface\\Icons\\ClassIcon_Mage",
    WARLOCK      = "Interface\\Icons\\ClassIcon_Warlock",
    MONK         = "Interface\\Icons\\ClassIcon_Monk",
    DRUID        = "Interface\\Icons\\ClassIcon_Druid",
    DEMONHUNTER  = "Interface\\Icons\\ClassIcon_DemonHunter",
    EVOKER       = "Interface\\Icons\\ClassIcon_Evoker",
}

-----------------------------
-- Helpers
-----------------------------
local function GetCharKey()
    local name, realm = UnitFullName("player")
    realm = realm and realm ~= "" and realm or GetRealmName()
    return name .. "-" .. realm, name, realm
end

local function FormatPlayedTime(seconds)
    if not seconds or seconds == 0 then return "0 minutes" end
    local days = math.floor(seconds / 86400)
    local hours = math.floor((seconds % 86400) / 3600)
    local mins = math.floor((seconds % 3600) / 60)
    if days > 0 then
        return string.format("%d days, %d hours", days, hours)
    elseif hours > 0 then
        return string.format("%d hours, %d minutes", hours, mins)
    else
        return string.format("%d minutes", mins)
    end
end

local function FormatPlayedShort(seconds)
    if not seconds or seconds == 0 then return "0m" end
    local days = math.floor(seconds / 86400)
    local hours = math.floor((seconds % 86400) / 3600)
    if days > 0 then
        return string.format("%dd %dh", days, hours)
    else
        return string.format("%dh %dm", hours, math.floor((seconds % 3600) / 60))
    end
end

local function SafeStat(id)
    local val = GetStatistic(id)
    if not val or val == "--" or val == "" then return 0 end
    -- Remove commas and gold suffix for numeric parsing
    val = val:gsub(",", ""):gsub(" gold", ""):gsub("g", "")
    return tonumber(val) or 0
end

local function FormatNumber(n)
    if not n or n == 0 then return "0" end
    return BreakUpLargeNumbers(math.floor(n))
end

local function GetClassColor(classFile)
    local color = RAID_CLASS_COLORS[classFile]
    if color then return color.r, color.g, color.b end
    return 0.7, 0.7, 0.7
end

local function GetClassHex(classFile)
    local r, g, b = GetClassColor(classFile)
    return string.format("%02x%02x%02x", r * 255, g * 255, b * 255)
end

-----------------------------
-- PvP Badge Self-Scan
-----------------------------

-- Achievement ID → short season label mapping
local ACHIEVEMENT_LABELS = {
    -- R1 (3v3) — 35 achievements
    [418] = "TBC1", [419] = "TBC2", [420] = "TBC3",
    [3336] = "WOTLK1", [3436] = "WOTLK2", [3758] = "WOTLK3", [4599] = "WOTLK4",
    [6002] = "CATA1", [6124] = "CATA2", [6938] = "CATA3",
    [8214] = "MOP1", [8791] = "MOP2", [8643] = "MOP3", [8666] = "MOP4",
    [9232] = "WOD1", [10096] = "WOD2", [10097] = "WOD3",
    [11012] = "LEG1", [11014] = "LEG2", [11037] = "LEG3", [11062] = "LEG4",
    [12010] = "LEG5", [12134] = "LEG6", [12185] = "LEG7",
    [12945] = "BFA1", [13200] = "BFA2", [13630] = "BFA3", [13957] = "BFA4",
    [14690] = "SL1", [14973] = "SL2", [15353] = "SL3", [15606] = "SL4",
    [15951] = "DF1", [17764] = "DF2", [19132] = "DF3", [19454] = "DF4",
    [40380] = "TWW1", [41354] = "TWW2", [42036] = "TWW3",
    [61180] = "MN1",

    -- Glad — only IDs not already in R1 array (2091 excluded: generic "Gladiator" isn't season-specific)
    [8644] = "MOP3", [8667] = "MOP4",
    [9239] = "WOD1", [10098] = "WOD2", [10110] = "WOD3",
    [11011] = "LEG1", [11013] = "LEG2", [11038] = "LEG3", [11061] = "LEG4",
    [12045] = "LEG5", [12167] = "LEG6", [12168] = "LEG7",
    [12961] = "BFA1", [13212] = "BFA2", [13647] = "BFA3", [13967] = "BFA4",
    [14689] = "SL1", [14972] = "SL2", [15352] = "SL3", [15605] = "SL4",
    [15957] = "DF1", [17740] = "DF2", [19091] = "DF3", [19490] = "DF4",
    [40393] = "TWW1", [41032] = "TWW2", [41049] = "TWW3",
    [61188] = "MN1",

    -- Solo Shuffle R1
    [16734] = "DF1", [17767] = "DF2", [19131] = "DF3", [19453] = "DF4",
    [40381] = "TWW1", [41355] = "TWW2", [42033] = "TWW3",
    [61179] = "MN1",

    -- Blitz R1 (both faction variants share the same label)
    [40234] = "TWW1", [40235] = "TWW1",
    [41356] = "TWW2", [41357] = "TWW2",
    [42034] = "TWW3", [42035] = "TWW3",
    [61177] = "MN1",  [61178] = "MN1",
}

-- Character Milestone Definitions
-- overrideIcon: use instead of achievement icon (achievement icons render poorly in circular badges)
-- countText: shown bottom-right on badge (like PvP title count badges)
local MS_PVP_ICON   = 132147  -- ability_dualwield (crossed swords)
local MS_MOUNT_ICON = "Interface\\Icons\\Ability_Mount_RidingHorse"
local MS_PET_ICON   = 656580  -- Silver Tabby Cat
local MS_ACHV_ICON  = "Interface\\Icons\\Achievement_Quests_Completed_08"
local MS_AOTC_ICON  = 463446
local MS_ANNIV_ICON = 133783  -- celebration icon
local MILESTONE_DEFS = {
    -- PvP Career (generic milestones — per-season titles come from ownBadges)
    { id = 397,   label = "Step Into the Arena",  cat = "pvp", color = {0.6, 0.6, 0.6}, overrideIcon = MS_PVP_ICON },
    { id = 2090,  label = "Challenger",            cat = "pvp", color = {0.6, 0.6, 0.6}, overrideIcon = MS_PVP_ICON },
    { id = 2093,  label = "Rival",                 cat = "pvp", color = {0, 0.8, 0.4},   overrideIcon = MS_PVP_ICON },
    { id = 2092,  label = "Duelist",               cat = "pvp", color = {0.2, 0.7, 0.9}, overrideIcon = MS_PVP_ICON },
    { id = 1174,  label = "Arena Master",          cat = "pvp", color = {1, 0.5, 0},     overrideIcon = 236537 },
    { id = 5867,  label = "Flawless Victory",      cat = "pvp", color = {1, 0.84, 0},    overrideIcon = 236537 },
    -- Collections — mount milestones
    { id = 2143,                label = "50 Mounts",   cat = "collection", overrideIcon = MS_MOUNT_ICON, countText = "50" },
    { id = 2536,  alt = 2537,  label = "100 Mounts",  cat = "collection", overrideIcon = MS_MOUNT_ICON, countText = "100" },
    { id = 7860,  alt = 7862,  label = "150 Mounts",  cat = "collection", overrideIcon = MS_MOUNT_ICON, countText = "150" },
    { id = 8302,  alt = 8304,  label = "200 Mounts",  cat = "collection", overrideIcon = MS_MOUNT_ICON, countText = "200" },
    { id = 9598,  alt = 9599,  label = "250 Mounts",  cat = "collection", overrideIcon = MS_MOUNT_ICON, countText = "250" },
    { id = 10355, alt = 10356, label = "300 Mounts",  cat = "collection", overrideIcon = MS_MOUNT_ICON, countText = "300" },
    { id = 12931, alt = 12932, label = "350 Mounts",  cat = "collection", overrideIcon = MS_MOUNT_ICON, countText = "350" },
    { id = 12933, alt = 12934, label = "400 Mounts",  cat = "collection", overrideIcon = MS_MOUNT_ICON, countText = "400" },
    { id = 15833, alt = 15834, label = "500 Mounts",  cat = "collection", overrideIcon = MS_MOUNT_ICON, countText = "500" },
    { id = 62096, alt = 62103, label = "600 Mounts",  cat = "collection", overrideIcon = MS_MOUNT_ICON, countText = "600" },
    -- Collections — pet milestones (full chain)
    { id = 1017,  label = "First Companion Pet",   cat = "collection", overrideIcon = MS_PET_ICON, countText = "1" },
    { id = 15,    label = "15 Pets",                cat = "collection", overrideIcon = MS_PET_ICON, countText = "15" },
    { id = 1248,  label = "25 Pets",                cat = "collection", overrideIcon = MS_PET_ICON, countText = "25" },
    { id = 1250,  label = "50 Pets",                cat = "collection", overrideIcon = MS_PET_ICON, countText = "50" },
    { id = 2516,  label = "75 Pets",                cat = "collection", overrideIcon = MS_PET_ICON, countText = "75" },
    { id = 5876,  label = "100 Pets",               cat = "collection", overrideIcon = MS_PET_ICON, countText = "100" },
    { id = 5877,  label = "125 Pets",               cat = "collection", overrideIcon = MS_PET_ICON, countText = "125" },
    { id = 5875,  label = "150 Pets",               cat = "collection", overrideIcon = MS_PET_ICON, countText = "150" },
    { id = 7500,  label = "250 Pets",               cat = "collection", overrideIcon = MS_PET_ICON, countText = "250" },
    { id = 7501,  label = "400 Pets",               cat = "collection", overrideIcon = MS_PET_ICON, countText = "400" },
    { id = 9643,  label = "600 Pets",               cat = "collection", overrideIcon = MS_PET_ICON, countText = "600" },
    { id = 12992, label = "800 Pets",               cat = "collection", overrideIcon = MS_PET_ICON, countText = "800" },
    { id = 12958, label = "1000 Pets",              cat = "collection", overrideIcon = MS_PET_ICON, countText = "1000" },
    { id = 15641, label = "1200 Pets",              cat = "collection", overrideIcon = MS_PET_ICON, countText = "1200" },
    { id = 15642, label = "1400 Pets",              cat = "collection", overrideIcon = MS_PET_ICON, countText = "1400" },
    { id = 15643, label = "1600 Pets",              cat = "collection", overrideIcon = MS_PET_ICON, countText = "1600" },
    { id = 15644, label = "1800 Pets",              cat = "collection", overrideIcon = MS_PET_ICON, countText = "1800" },
    -- General / Prestige
    { id = 4496,  label = "It's Over 9000!",        cat = "general", color = {0.1, 0.9, 0.3}, overrideIcon = MS_ACHV_ICON, countText = "9000" },
    { id = 2144,  label = "Long Strange Trip",       cat = "general", color = {0.85, 0.3, 0.7}, overrideIcon = MS_ACHV_ICON },
    { id = 7520,  alt = 7521, label = "Loremaster",  cat = "general", color = {0.9, 0.7, 0.3}, overrideIcon = MS_ACHV_ICON },
    { id = 2336,  label = "The Insane",              cat = "general", color = {0.7, 0.15, 0.3}, overrideIcon = MS_ACHV_ICON },
    -- Ahead of the Curve (Heroic raid completion — per tier, raid-specific icons)
    { id = 6954,  label = "AotC: MV",       cat = "aotc", overrideIcon = 625910 },   -- Mogu'shan Vaults
    { id = 8246,  label = "AotC: HoF",      cat = "aotc", overrideIcon = 624006 },   -- Heart of Fear
    { id = 8248,  label = "AotC: ToES",     cat = "aotc", overrideIcon = 627685 },   -- Terrace of Endless Spring
    { id = 8249,  label = "AotC: ToT",      cat = "aotc", overrideIcon = 840303 },   -- Throne of Thunder
    { id = 8398,  alt = 8399, label = "AotC: SoO", cat = "aotc", overrideIcon = 456571 },  -- Siege of Orgrimmar
    { id = 9441,  label = "AotC: HM",       cat = "aotc", overrideIcon = 1030796 },  -- Highmaul
    { id = 9444,  label = "AotC: BRF",      cat = "aotc", overrideIcon = 1005700 },  -- Blackrock Foundry
    { id = 10044, label = "AotC: HFC",      cat = "aotc", overrideIcon = 1113431 },  -- Hellfire Citadel
    { id = 11194, label = "AotC: EN",       cat = "aotc", overrideIcon = 1413871 },  -- Emerald Nightmare
    { id = 11581, label = "AotC: ToV",      cat = "aotc", overrideIcon = 1530370 },  -- Trial of Valor
    { id = 11195, label = "AotC: NH",       cat = "aotc", overrideIcon = 1413856 },  -- The Nighthold
    { id = 11874, label = "AotC: ToS",      cat = "aotc", overrideIcon = 1546412 },  -- Tomb of Sargeras
    { id = 12110, label = "AotC: Antorus",  cat = "aotc", overrideIcon = 1711335 },  -- Antorus
    { id = 12536, label = "AotC: Uldir",    cat = "aotc", overrideIcon = 2032223 },  -- Uldir
    { id = 13322, label = "AotC: BoD",      cat = "aotc", overrideIcon = 2484334 },  -- Battle of Dazar'alor
    { id = 13418, label = "AotC: CoS",      cat = "aotc", overrideIcon = 2828147 },  -- Crucible of Storms
    { id = 13784, label = "AotC: TEP",      cat = "aotc", overrideIcon = 3012068 },  -- The Eternal Palace
    { id = 14068, label = "AotC: Ny'alotha", cat = "aotc", overrideIcon = 3194610 }, -- Ny'alotha
    { id = 14460, label = "AotC: Nathria",  cat = "aotc", overrideIcon = 3670321 },  -- Castle Nathria
    { id = 15134, label = "AotC: SoD",      cat = "aotc", overrideIcon = 4062738 },  -- Sanctum of Domination
    { id = 15470, label = "AotC: SotFO",    cat = "aotc", overrideIcon = 4254080 },  -- Sepulcher of the First Ones
    { id = 17107, label = "AotC: VotI",     cat = "aotc", overrideIcon = 4630364 },  -- Vault of the Incarnates
    { id = 18253, label = "AotC: Aberrus",  cat = "aotc", overrideIcon = 5161748 },  -- Aberrus
    { id = 19350, label = "AotC: Amirdrassil", cat = "aotc", overrideIcon = 5342925 }, -- Amirdrassil
    { id = 40253, label = "AotC: NaP",      cat = "aotc", overrideIcon = 5779391 },  -- Nerub-ar Palace
    { id = 41298, label = "AotC: LoU",      cat = "aotc", overrideIcon = 6392621 },  -- Liberation of Undermine
    { id = 41624, label = "AotC: MFO",      cat = "aotc", overrideIcon = 6922083 },  -- Manaforge Omega
    -- Cutting Edge (Mythic raid final boss — per tier, same raid icons as AOTC)
    { id = 7485,  label = "CE: MV",        cat = "ce", overrideIcon = 625910 },   -- Mogu'shan Vaults
    { id = 7486,  label = "CE: HoF",       cat = "ce", overrideIcon = 624006 },   -- Heart of Fear
    { id = 7487,  label = "CE: ToES",      cat = "ce", overrideIcon = 627685 },   -- Terrace of Endless Spring
    { id = 8238,  label = "CE: ToT",       cat = "ce", overrideIcon = 840303 },   -- Throne of Thunder
    { id = 8401,  alt = 8400, label = "CE: SoO", cat = "ce", overrideIcon = 456571 },  -- Siege of Orgrimmar (25/10)
    { id = 9442,  label = "CE: HM",        cat = "ce", overrideIcon = 1030796 },  -- Highmaul
    { id = 9443,  label = "CE: BRF",       cat = "ce", overrideIcon = 1005700 },  -- Blackrock Foundry
    { id = 10045, label = "CE: HFC",       cat = "ce", overrideIcon = 1113431 },  -- Hellfire Citadel
    { id = 11191, label = "CE: EN",        cat = "ce", overrideIcon = 1413871 },  -- Emerald Nightmare
    { id = 11580, label = "CE: ToV",       cat = "ce", overrideIcon = 1530370 },  -- Trial of Valor
    { id = 11192, label = "CE: NH",        cat = "ce", overrideIcon = 1413856 },  -- The Nighthold
    { id = 11875, label = "CE: ToS",       cat = "ce", overrideIcon = 1546412 },  -- Tomb of Sargeras
    { id = 12111, label = "CE: Antorus",   cat = "ce", overrideIcon = 1711335 },  -- Antorus
    { id = 12535, label = "CE: Uldir",     cat = "ce", overrideIcon = 2032223 },  -- Uldir
    { id = 13323, label = "CE: BoD",       cat = "ce", overrideIcon = 2484334 },  -- Battle of Dazar'alor
    { id = 13419, label = "CE: CoS",       cat = "ce", overrideIcon = 2828147 },  -- Crucible of Storms
    { id = 13785, label = "CE: TEP",       cat = "ce", overrideIcon = 3012068 },  -- The Eternal Palace
    { id = 14069, label = "CE: Ny'alotha", cat = "ce", overrideIcon = 3194610 },  -- Ny'alotha
    { id = 14461, label = "CE: Nathria",   cat = "ce", overrideIcon = 3670321 },  -- Castle Nathria
    { id = 15135, label = "CE: SoD",       cat = "ce", overrideIcon = 4062738 },  -- Sanctum of Domination
    { id = 15471, label = "CE: SotFO",     cat = "ce", overrideIcon = 4254080 },  -- Sepulcher of the First Ones
    { id = 17108, label = "CE: VotI",      cat = "ce", overrideIcon = 4630364 },  -- Vault of the Incarnates
    { id = 18254, label = "CE: Aberrus",   cat = "ce", overrideIcon = 5161748 },  -- Aberrus
    { id = 19351, label = "CE: Amirdrassil", cat = "ce", overrideIcon = 5342925 }, -- Amirdrassil
    { id = 40254, label = "CE: NaP",       cat = "ce", overrideIcon = 5779391 },  -- Nerub-ar Palace
    { id = 41297, label = "CE: LoU",       cat = "ce", overrideIcon = 6392621 },  -- Liberation of Undermine
    { id = 41625, label = "CE: MFO",       cat = "ce", overrideIcon = 6922083 },  -- Manaforge Omega
    -- M+ Dungeon Master (meta: all seasonal M+10 in time as all roles)
    { id = 42148, label = "Enterprising Master", cat = "mplus", color = MPLUS_LEGEND_COLOR, overrideIcon = 255347 },  -- TWW S2
    { id = 61877, label = "Unbound Master",      cat = "mplus", color = MPLUS_LEGEND_COLOR, overrideIcon = 255347 },  -- TWW S3
    -- Delve Solo Achievements (Tier ?? boss before next patch)
    { id = 40433, label = "Solo Zekvir",    cat = "delve", color = {0.9, 0.6, 0.25}, overrideIcon = 5764867 },   -- TWW S1
    { id = 41210, label = "Solo Underpin",  cat = "delve", color = {0.9, 0.6, 0.25}, overrideIcon = 6100921 },   -- TWW S2
    { id = 42190, label = "Solo Ky'veza",   cat = "delve", color = {0.9, 0.6, 0.25}, overrideIcon = 6361900 },   -- TWW S3
    { id = 61799, label = "Solo Nullaeus",  cat = "delve", color = {0.9, 0.6, 0.25}, overrideIcon = 7452103 },   -- Midnight S1
    -- WoW Anniversaries
    { id = 2398,  label = "WoW's 4th Anniversary",   cat = "anniversary", overrideIcon = MS_ANNIV_ICON },
    { id = 4400,  label = "WoW's 5th Anniversary",   cat = "anniversary", overrideIcon = MS_ANNIV_ICON },
    { id = 5512,  label = "WoW's 6th Anniversary",   cat = "anniversary", overrideIcon = MS_ANNIV_ICON },
    { id = 5863,  label = "WoW's 7th Anniversary",   cat = "anniversary", overrideIcon = MS_ANNIV_ICON },
    { id = 6131,  label = "WoW's 8th Anniversary",   cat = "anniversary", overrideIcon = MS_ANNIV_ICON },
    { id = 7853,  label = "WoW's 9th Anniversary",   cat = "anniversary", overrideIcon = MS_ANNIV_ICON },
    { id = 8820,  label = "WoW's 10th Anniversary",  cat = "anniversary", overrideIcon = MS_ANNIV_ICON },
    { id = 10058, label = "WoW's 11th Anniversary",  cat = "anniversary", overrideIcon = MS_ANNIV_ICON },
    { id = 10741, label = "WoW's 12th Anniversary",  cat = "anniversary", overrideIcon = MS_ANNIV_ICON },
    { id = 11848, label = "WoW's 13th Anniversary",  cat = "anniversary", overrideIcon = MS_ANNIV_ICON },
    { id = 12827, label = "WoW's 14th Anniversary",  cat = "anniversary", overrideIcon = MS_ANNIV_ICON },
    { id = 13917, label = "WoW's 15th Anniversary",  cat = "anniversary", overrideIcon = MS_ANNIV_ICON },
    { id = 14271, label = "WoW's 16th Anniversary",  cat = "anniversary", overrideIcon = MS_ANNIV_ICON },
    { id = 14942, label = "WoW's 17th Anniversary",  cat = "anniversary", overrideIcon = MS_ANNIV_ICON },
    { id = 15218, label = "WoW's 18th Anniversary",  cat = "anniversary", overrideIcon = MS_ANNIV_ICON },
    { id = 18702, label = "WoW's 19th Anniversary",  cat = "anniversary", overrideIcon = MS_ANNIV_ICON },
    { id = 41220, label = "WoW's 20th Anniversary",  cat = "anniversary", overrideIcon = MS_ANNIV_ICON },
}

local COLLECTION_COLOR  = { 1, 0.84, 0 }
local GENERAL_COLOR     = { 0.8, 0.8, 0.8 }
local ANNIVERSARY_COLOR = { 0.5, 0.7, 1.0 }
local AOTC_COLOR        = { 0, 0.75, 0.7 }  -- teal
local CE_COLOR          = { 0.9, 0.2, 0.2 }  -- red

-- M+ Keystone top-tier color and label
local MPLUS_LEGEND_COLOR = { 0.85, 0.3, 0.85 }  -- pink/purple

-- Keystone top-tier achievements per season (highest available tier for that season)
local KEYSTONE_TOP_TIERS = {
    -- Shadowlands (Master = top for S1/S2/S4; Hero 3000 = top for S3)
    { id = 14532, label = "SL1" },    -- Keystone Master: Season One (+15)
    { id = 15078, label = "SL2" },    -- Keystone Master: Season Two (2000)
    { id = 15506, label = "SL3" },    -- Keystone Hero: Season Three (3000)
    { id = 15690, label = "SL4" },    -- Keystone Master: Season Four (2000)
    -- Dragonflight (Hero 2500 = top)
    { id = 16650, label = "DF1" },    -- Keystone Hero: Season One (2500)
    { id = 17845, label = "DF2" },    -- Keystone Hero: Season Two (2500)
    { id = 19012, label = "DF3" },    -- Keystone Hero: Season Three (2500)
    { id = 19783, label = "DF4" },    -- Keystone Hero: Season Four (2500)
    -- The War Within (Hero 2500 for S1; Legend 3000 for S2+)
    { id = 20526, label = "TWW1" },   -- Keystone Hero: Season One (2500)
    { id = 40951, label = "TWW2" },   -- Keystone Legend: Season Two (3000)
    { id = 42172, label = "TWW3" },   -- Keystone Legend: Season Three (3000)
}

local function ScanOwnPvPTitles()
    local badges = {}
    local earnedIDs = {}  -- track which IDs were already assigned a badge (R1 > Glad priority)

    -- Helper: scan an achievement array in reverse (newest season first)
    local function ScanReverse(arr, badgeType)
        for i = #arr, 1, -1 do
            local id = arr[i]
            if not earnedIDs[id] and ACHIEVEMENT_LABELS[id] then
                local _, _, _, completed = GetAchievementInfo(id)
                if completed then
                    earnedIDs[id] = true
                    badges[#badges + 1] = { type = badgeType, id = id, label = ACHIEVEMENT_LABELS[id] }
                end
            end
        end
    end

    -- R1 (3v3) — check first (highest priority), newest first
    ScanReverse(ns.R1_ACHIEVEMENTS, "r1")

    -- Solo Shuffle R1, newest first
    ScanReverse(ns.SOLOSHUFFLE_R1_ACHIEVEMENTS, "ssR1")

    -- Blitz R1 (both faction variants — deduplicate by label), newest first
    local blitzLabelsEarned = {}
    for i = #ns.BLITZ_R1_ACHIEVEMENTS, 1, -1 do
        local id = ns.BLITZ_R1_ACHIEVEMENTS[i]
        if not earnedIDs[id] then
            local _, _, _, completed = GetAchievementInfo(id)
            if completed then
                earnedIDs[id] = true
                local lbl = ACHIEVEMENT_LABELS[id]
                if not blitzLabelsEarned[lbl] then
                    blitzLabelsEarned[lbl] = true
                    badges[#badges + 1] = { type = "blitzR1", id = id, label = lbl }
                end
            end
        end
    end

    -- Gladiator — skip any ID already earned as R1, skip seasons where R1 was earned (different IDs same season),
    -- and skip 2091 (generic, not season-specific)
    local r1Labels = {}
    for _, bd in ipairs(badges) do
        if bd.type == "r1" then r1Labels[bd.label] = true end
    end
    for i = #ns.GLAD_ACHIEVEMENTS, 1, -1 do
        local id = ns.GLAD_ACHIEVEMENTS[i]
        local lbl = ACHIEVEMENT_LABELS[id]
        if id ~= 2091 and not earnedIDs[id] and lbl and not r1Labels[lbl] then
            local _, _, _, completed = GetAchievementInfo(id)
            if completed then
                earnedIDs[id] = true
                badges[#badges + 1] = { type = "glad", id = id, label = lbl }
            end
        end
    end

    -- 3K+ check (from personal stats)
    local peak = math.max(SafeStat(370), SafeStat(595))
    if peak >= 3000 then
        badges[#badges + 1] = { type = "threek", id = nil, label = "3k" }
    end

    -- Always check Hero (not just fallback — it's a milestone worth showing alongside other titles)
    local gen = ns.PVP_BADGE_GENERAL
    local _, _, _, heroH = GetAchievementInfo(gen.heroHorde)
    local _, _, _, heroA = GetAchievementInfo(gen.heroAlliance)
    if heroH or heroA then
        badges[#badges + 1] = { type = "hero", id = heroH and gen.heroHorde or gen.heroAlliance, label = "Hero" }
    end

    return badges
end

-----------------------------
-- Leaderboard Lookup
-----------------------------
local BRACKET_TO_LB_KEY = {
    [1] = "2v2",
    [2] = "3v3",
    [7] = "shuffle",
    [9] = "blitz",
}

local leaderboardIndex = nil

local function BuildLeaderboardIndex()
    leaderboardIndex = {}
    if not ns.LEADERBOARD then return end
    for regionKey, brackets in pairs(ns.LEADERBOARD) do
        leaderboardIndex[regionKey] = {}
        for bracketKey, entries in pairs(brackets) do
            local idx = {}
            for rank, entry in ipairs(entries) do
                idx[entry[1]] = rank
            end
            leaderboardIndex[regionKey][bracketKey] = idx
        end
    end
end

function ns.GetLeaderboardRank(nameRealm, bracketIdx)
    local lbKey = BRACKET_TO_LB_KEY[bracketIdx]
    if not lbKey then return nil end

    local region = GetCurrentRegion()
    local regionKey = (region == 3) and "EU" or "US"

    if not leaderboardIndex then
        BuildLeaderboardIndex()
    end

    local regionData = leaderboardIndex[regionKey]
    if not regionData then return nil end
    local bracketData = regionData[lbKey]
    if not bracketData then return nil end

    return bracketData[nameRealm]
end

-----------------------------
-- Leaderboard Peak Lookup
-----------------------------
local BRACKET_TO_PEAK_KEY = {
    [4] = "rbg",
    [7] = "shuffle",
    [9] = "blitz",
}

local peakIndex = nil

local function BuildPeakIndex()
    peakIndex = {}
    if not ns.LEADERBOARD_PEAKS then return end
    for regionKey, brackets in pairs(ns.LEADERBOARD_PEAKS) do
        peakIndex[regionKey] = {}
        for bracketKey, entries in pairs(brackets) do
            local idx = {}
            for _, entry in ipairs(entries) do
                idx[entry[1]] = entry[2]
            end
            peakIndex[regionKey][bracketKey] = idx
        end
    end
end

function ns.GetLeaderboardPeak(nameRealm, bracketIdx)
    local peakKey = BRACKET_TO_PEAK_KEY[bracketIdx]
    if not peakKey then return nil end

    local region = GetCurrentRegion()
    local regionKey = (region == 3) and "EU" or "US"

    if not peakIndex then
        BuildPeakIndex()
    end

    local regionData = peakIndex[regionKey]
    if not regionData then return nil end
    local bracketData = regionData[peakKey]
    if not bracketData then return nil end

    return bracketData[nameRealm]
end

-----------------------------
-- Data Collection Helpers
-----------------------------

-- Match an expansion tier name to its base profession (e.g. "Khaz Algar Enchanting" → "Enchanting")
-- Excludes base lines where displayName == baseName (those have garbage combined data)
local function MatchBaseProfession(displayName, profNames)
    for baseName, _ in pairs(profNames) do
        local suffix = " " .. baseName
        if displayName ~= baseName and displayName:sub(-#suffix) == suffix then
            return baseName
        end
    end
    return nil
end

local function CollectProfessionData()
    local result = {}
    local indices = { GetProfessions() }  -- prof1, prof2, archaeology, fishing, cooking

    -- Collect base profession names and icons
    local profIcons = {}   -- baseName → icon
    local profOrder = {}   -- ordered base names
    for _, idx in ipairs(indices) do
        if idx then
            local name, icon = GetProfessionInfo(idx)
            if name and not profIcons[name] then
                profIcons[name] = icon
                profOrder[#profOrder + 1] = name
            end
        end
    end
    -- Push Archaeology to the end of the list
    for i, name in ipairs(profOrder) do
        if name == "Archaeology" then
            table.remove(profOrder, i)
            profOrder[#profOrder + 1] = name
            break
        end
    end

    -- Gather expansion tiers via C_TradeSkillUI (group by base profession name)
    local profTiers = {}  -- baseName → { tiers }
    if C_TradeSkillUI and C_TradeSkillUI.GetAllProfessionTradeSkillLines
        and C_TradeSkillUI.GetProfessionInfoBySkillLineID then
        local allLines = C_TradeSkillUI.GetAllProfessionTradeSkillLines() or {}
        -- Append secondary profession tiers (Cooking/Fishing) not returned by the API
        for _, extraID in ipairs(SECONDARY_TIER_LINES) do
            allLines[#allLines + 1] = extraID
        end
        for _, lineID in ipairs(allLines) do
            local ok, info = pcall(C_TradeSkillUI.GetProfessionInfoBySkillLineID, lineID)
            if ok and info and info.skillLevel and info.skillLevel > 0 then
                local displayName = info.professionName
                if displayName then
                    -- Match to a base profession by suffix (e.g. "Khaz Algar Enchanting" → "Enchanting")
                    -- Skip base lines where displayName equals baseName (garbage combined data)
                    local baseName = MatchBaseProfession(displayName, profIcons)
                    if baseName then
                        if not profTiers[baseName] then profTiers[baseName] = {} end
                        profTiers[baseName][#profTiers[baseName] + 1] = {
                            name = displayName,
                            skill = info.skillLevel,
                            max = info.maxSkillLevel or info.skillLevel,
                        }
                    end
                end
            end
        end
    end

    -- Preserve previously cached tiers from SavedVariables
    local charKey = GetCharKey()
    local existingSnap = SKToolsProfileDB and SKToolsProfileDB[charKey]
    local existingProfs = existingSnap and existingSnap.professions
    local cachedTiers = {}
    if existingProfs then
        for _, ep in ipairs(existingProfs) do
            if ep.tiers and #ep.tiers > 0 then
                cachedTiers[ep.name] = ep.tiers
            end
        end
    end

    -- Build result: attach tiers if available, fallback to cached tiers or overall skill
    for _, name in ipairs(profOrder) do
        local entry = { name = name, icon = profIcons[name], skill = 0, max = 0 }
        if profTiers[name] and #profTiers[name] > 0 then
            entry.tiers = profTiers[name]
        elseif cachedTiers[name] then
            entry.tiers = cachedTiers[name]
        end
        for _, idx in ipairs(indices) do
            if idx then
                local pName, _, sl, ml = GetProfessionInfo(idx)
                if pName == name then
                    entry.skill = sl or 0
                    entry.max = ml or 0
                    break
                end
            end
        end
        result[#result + 1] = entry
    end
    return #result > 0 and result or nil
end

-- Cache profession tiers when profession window opens (skill data only available then)
do
    local profCacheFrame = CreateFrame("Frame")
    profCacheFrame:RegisterEvent("TRADE_SKILL_DATA_SOURCE_CHANGED")
    profCacheFrame:SetScript("OnEvent", function()
        C_Timer.After(0.5, function()
            if not C_TradeSkillUI or not C_TradeSkillUI.GetAllProfessionTradeSkillLines
                or not C_TradeSkillUI.GetProfessionInfoBySkillLineID then return end
            if not SKToolsProfileDB then return end

            local charKey = GetCharKey()
            local snap = SKToolsProfileDB[charKey]
            if not snap or not snap.professions then return end

            -- Build set of base profession names this character has
            local profNames = {}
            for _, prof in ipairs(snap.professions) do
                profNames[prof.name] = true
            end

            -- Query all tier lines for real skill data
            local tiersByBase = {}
            local allLines = C_TradeSkillUI.GetAllProfessionTradeSkillLines() or {}
            for _, extraID in ipairs(SECONDARY_TIER_LINES) do
                allLines[#allLines + 1] = extraID
            end
            for _, lineID in ipairs(allLines) do
                local ok, info = pcall(C_TradeSkillUI.GetProfessionInfoBySkillLineID, lineID)
                if ok and info and info.professionName and info.skillLevel and info.skillLevel > 0 then
                    local baseName = MatchBaseProfession(info.professionName, profNames)
                    if baseName then
                        if not tiersByBase[baseName] then tiersByBase[baseName] = {} end
                        tiersByBase[baseName][#tiersByBase[baseName] + 1] = {
                            name = info.professionName,
                            skill = info.skillLevel,
                            max = info.maxSkillLevel or info.skillLevel,
                        }
                    end
                end
            end

            -- Update snapshot with cached tiers
            for _, prof in ipairs(snap.professions) do
                if tiersByBase[prof.name] then
                    prof.tiers = tiersByBase[prof.name]
                end
            end
        end)
    end)
end

local function CollectMilestones()
    local entries = {}

    -- Helper: check a single achievement ID
    local function CheckAchievement(id)
        local _, achName, _, completed, month, day, year, _, icon = GetAchievementInfo(id)
        if completed then
            return achName, icon, month, day, year
        end
        return nil
    end

    -- Helper: date comparison (returns true if a is earlier than b)
    local function DateEarlier(aY, aM, aD, bY, bM, bD)
        if (aY or 0) ~= (bY or 0) then return (aY or 0) < (bY or 0) end
        if (aM or 0) ~= (bM or 0) then return (aM or 0) < (bM or 0) end
        return (aD or 0) < (bD or 0)
    end

    -- 1. Scan static milestone achievements
    for _, def in ipairs(MILESTONE_DEFS) do
        -- Check primary ID, then alt ID (faction variant)
        local achId = def.id
        local achName, icon, month, day, year = CheckAchievement(def.id)
        if not achName and def.alt then
            achId = def.alt
            achName, icon, month, day, year = CheckAchievement(def.alt)
        end
        if achName then
            local color = def.color
            if not color then
                if def.cat == "collection" then color = COLLECTION_COLOR
                elseif def.cat == "anniversary" then color = ANNIVERSARY_COLOR
                elseif def.cat == "aotc" then color = AOTC_COLOR
                elseif def.cat == "ce" then color = CE_COLOR
                else color = GENERAL_COLOR end
            end
            entries[#entries + 1] = {
                label = def.label,
                name = achName,
                icon = icon,
                overrideIcon = def.overrideIcon,
                countText = def.countText,
                color = color,
                month = month, day = day, year = year,
                cat = def.cat,
                achId = achId,
            }
        end
    end

    -- 2. Scan Keystone top-tier achievements (highest available per season)
    for _, legend in ipairs(KEYSTONE_TOP_TIERS) do
        local achName, icon, month, day, year = CheckAchievement(legend.id)
        if achName then
            entries[#entries + 1] = {
                label = legend.label,
                name = achName,
                icon = icon,
                color = MPLUS_LEGEND_COLOR,
                month = month, day = day, year = year,
                cat = "mplus",
                mplusTier = legend.label,
                achId = legend.id,
            }
        end
    end

    -- Sort chronologically (oldest first)
    table.sort(entries, function(a, b)
        if (a.year or 0) ~= (b.year or 0) then return (a.year or 0) < (b.year or 0) end
        if (a.month or 0) ~= (b.month or 0) then return (a.month or 0) < (b.month or 0) end
        return (a.day or 0) < (b.day or 0)
    end)

    return #entries > 0 and entries or nil
end

-- Notable title names: unobtainable or extremely rare titles (matched by name, not ID)
-- WoW title IDs are unreliable for hardcoding, so we scan all known titles and filter by name
local NOTABLE_TITLE_NAMES = {
    -- R1 titles from any bracket (unobtainable after each season)
    ["Merciless Gladiator"] = true, ["Vengeful Gladiator"] = true, ["Brutal Gladiator"] = true,
    ["Deadly Gladiator"] = true, ["Furious Gladiator"] = true, ["Relentless Gladiator"] = true,
    ["Wrathful Gladiator"] = true, ["Vicious Gladiator"] = true, ["Ruthless Gladiator"] = true,
    ["Cataclysmic Gladiator"] = true, ["Tyrannical Gladiator"] = true, ["Grievous Gladiator"] = true,
    ["Prideful Gladiator"] = true, ["Primal Gladiator"] = true, ["Wild Gladiator"] = true,
    ["Warmongering Gladiator"] = true, ["Vindictive Gladiator"] = true, ["Fearless Gladiator"] = true,
    ["Cruel Gladiator"] = true, ["Fierce Gladiator"] = true, ["Dominant Gladiator"] = true,
    ["Demonic Gladiator"] = true, ["Dread Gladiator"] = true, ["Sinister Gladiator"] = true,
    ["Notorious Gladiator"] = true, ["Corrupted Gladiator"] = true, ["Sinful Gladiator"] = true,
    ["Unchained Gladiator"] = true, ["Cosmic Gladiator"] = true, ["Eternal Gladiator"] = true,
    ["Crimson Gladiator"] = true, ["Obsidian Gladiator"] = true, ["Verdant Gladiator"] = true,
    ["Draconic Gladiator"] = true, ["Forged Gladiator"] = true, ["Awakened Gladiator"] = true,
    ["Weathered Gladiator"] = true,
    -- Unobtainable titles (source: warcraft.wiki.gg/wiki/Title#Unobtainable_titles)
    ["the Magic Seeker"] = true, ["Conqueror of Naxxramas"] = true, ["Obsidian Slayer"] = true,
    ["Death's Demise"] = true, ["the Celestial Defender"] = true, ["Grand Crusader"] = true,
    ["Tarren Mill Terror"] = true, ["Southshore Slayer"] = true, ["of the Iron Vanguard"] = true,
    ["of the Black Harvest"] = true, ["the Chosen"] = true, ["Azeroth's Champion"] = true,
    ["Fearless Spectator"] = true, ["Veilstrider"] = true, ["Scarab Lord"] = true,
    ["Champion of the Naaru"] = true, ["Hand of A'dal"] = true, ["Vanquisher"] = true,
    ["Patron of War"] = true, ["Renowned Explorer"] = true, ["Plunderlord"] = true,
    ["Arena Master"] = true, ["the Immortal"] = true, ["the Undying"] = true,
    ["Conqueror of Ulduar"] = true, ["Champion of Ulduar"] = true, ["the Argent Defender"] = true,
    ["Brawler"] = true, ["Darkspear Revolutionary"] = true, ["the Hordebreaker"] = true,
    ["Legend of Pandaria"] = true, ["Field Medic"] = true, ["the Undaunted"] = true,
    ["Defender of the Wall"] = true, ["Mogu-Slayer"] = true, ["Flameweaver"] = true,
    ["Scarlet Commander"] = true, ["Darkmaster"] = true, ["Purified Defender"] = true,
    ["Siegebreaker"] = true, ["Stormbrewer"] = true, ["Jade Protector"] = true,
    ["Mistwalker"] = true, ["the Indomitable"] = true, ["the Mine Master"] = true,
    ["Dockmaster"] = true, ["the Soul Preserver"] = true, ["Scion of Rukhmar"] = true,
    ["Spiritwalker"] = true, ["Lord of Blackrock"] = true, ["Lady of Blackrock"] = true,
    ["the Violet Guardian"] = true, ["the Grimrail Suplexer"] = true,
    -- Feats of Strength titles (unobtainable)
    ["Bloodsail Admiral"] = true, ["the Insane"] = true, ["Herald of the Titans"] = true,
    ["the Camel-Hoarder"] = true, ["Predator"] = true, ["the Faceless One"] = true,
    ["Swabbie"] = true, ["Deck Hand"] = true, ["Swashbuckler"] = true,
    ["Buccaneer"] = true, ["First Mate"] = true, ["High Explorer"] = true,
    ["Immortal Spelunker"] = true, ["Landlubber"] = true, ["the Treasured"] = true,
    ["the Mad"] = true, ["Mind-Seeker"] = true,
    -- WoW Remix titles (unobtainable, usable in retail)
    ["Timerunner"] = true, ["Claw of Eternus"] = true, ["Chronoscholar"] = true,
    ["the Infernal"] = true, ["of the Infinite Chaos"] = true,
}

-- Tiered title groups (highest to lowest) — only show the best earned
local TIERED_TITLE_GROUPS = {
    { "Plunderlord", "First Mate", "Buccaneer", "Swashbuckler", "Landlubber", "Deck Hand", "Swabbie" },
}

local function CollectNotableTitles()
    local result = {}
    local found = {}  -- name → true for quick lookup
    -- Scan all title IDs (WoW has ~600+ titles, scan a safe range)
    for titleId = 1, 800 do
        if IsTitleKnown(titleId) then
            local titleName = GetTitleName(titleId)
            if titleName then
                -- Strip formatting tokens
                local clean = titleName:gsub("%%s", ""):gsub("^%s+", ""):gsub("%s+$", ""):gsub(",%s*$", "")
                if clean ~= "" and NOTABLE_TITLE_NAMES[clean] then
                    result[#result + 1] = { id = titleId, name = clean }
                    found[clean] = true
                end
            end
        end
    end
    -- Filter tiered groups: keep only the highest earned
    for _, group in ipairs(TIERED_TITLE_GROUPS) do
        local bestIdx
        for _, name in ipairs(group) do
            if found[name] then bestIdx = name; break end
        end
        if bestIdx then
            -- Remove all group titles except the best
            for i = #result, 1, -1 do
                if found[result[i].name] and result[i].name ~= bestIdx then
                    for _, gName in ipairs(group) do
                        if result[i].name == gName then
                            table.remove(result, i)
                            break
                        end
                    end
                end
            end
        end
    end
    return #result > 0 and result or nil
end

-----------------------------
-- Data Collection
-----------------------------
local pendingSnapshot = nil

local function CollectCharacterData()
    local charKey, name, realm = GetCharKey()
    local className, classFile = UnitClass("player")
    local race = UnitRace("player")
    local level = UnitLevel("player")
    local faction = UnitFactionGroup("player")

    -- Spec
    local specName, specIcon = "", nil
    local specIdx = GetSpecialization()
    if specIdx then
        local _, sName, _, sIcon = GetSpecializationInfo(specIdx)
        specName = sName or ""
        specIcon = sIcon
    end

    -- iLvl
    local avgIlvl, avgIlvlEquipped, avgIlvlPvP = GetAverageItemLevel()

    -- Achievements
    local achievementPoints = GetTotalAchievementPoints() or 0

    -- PvP
    local hks = GetPVPLifetimeStats() or 0
    local existingSnap = SKToolsProfileDB and SKToolsProfileDB[charKey]
    local existingRatings = existingSnap and existingSnap.pvpRatings or {}
    local pvpRatings = {}
    for _, bracket in ipairs(BRACKETS) do
        local rating, seasonBest, _, seasonPlayed, seasonWon = GetPersonalRatedInfo(bracket.idx)
        rating = rating or 0
        seasonBest = seasonBest or 0
        -- All-time best: use GetStatistic if available (2v2/3v3), else track ourselves
        local statId = HIGHEST_RATING_STATS[bracket.idx]
        local allTimeBest = statId and SafeStat(statId) or 0
        -- Also keep our own tracking as fallback (for Shuffle/Blitz/RBG)
        local prev = existingRatings[bracket.idx]
        local trackedBest = prev and prev.allTimeBest or 0
        allTimeBest = math.max(allTimeBest, trackedBest, seasonBest, rating)
        pvpRatings[bracket.idx] = {
            rating = rating,
            seasonBest = seasonBest,
            allTimeBest = allTimeBest,
            played = seasonPlayed or 0,
            won = seasonWon or 0,
        }
    end

    -- M+ data
    local mplusRuns = 0
    local mplusHighestKey = 0
    if C_MythicPlus and C_MythicPlus.GetRunHistory then
        local runs = C_MythicPlus.GetRunHistory(true, false)
        if runs then
            mplusRuns = #runs
            for _, run in ipairs(runs) do
                if run.level and run.level > mplusHighestKey then
                    mplusHighestKey = run.level
                end
            end
        end
    end

    local mplusRating = 0
    if C_ChallengeMode and C_ChallengeMode.GetOverallDungeonScore then
        mplusRating = C_ChallengeMode.GetOverallDungeonScore() or 0
    end

    -- Collections (deferred — journals might not be loaded yet)
    local mountsCollected = 0
    if C_MountJournal and C_MountJournal.GetMountIDs then
        local ids = C_MountJournal.GetMountIDs()
        if ids then
            for _, id in ipairs(ids) do
                local _, _, _, _, _, _, _, _, _, _, isCollected = C_MountJournal.GetMountInfoByID(id)
                if isCollected then mountsCollected = mountsCollected + 1 end
            end
        end
    end

    local petsCollected = 0
    if C_PetJournal and C_PetJournal.GetNumPets then
        local _, numOwned = C_PetJournal.GetNumPets()
        petsCollected = numOwned or 0
    end

    local toysCollected = 0
    if C_ToyBox and C_ToyBox.GetNumLearnedDisplayedToys then
        toysCollected = C_ToyBox.GetNumLearnedDisplayedToys() or 0
    end

    local decorCollected = 0
    if C_HousingCatalog and C_HousingCatalog.GetDecorTotalOwnedCount then
        decorCollected = C_HousingCatalog.GetDecorTotalOwnedCount() or 0
    end
    -- Preserve existing value if API returned 0 (may not be loaded yet)
    if decorCollected == 0 and existingSnap then
        decorCollected = existingSnap.decorCollected or 0
    end

    -- Fun statistics
    local stats = {}
    for _, def in ipairs(STAT_DEFS) do
        stats[def.key] = SafeStat(def.id)
    end

    -- Honor level
    local honorLevel = UnitHonorLevel("player") or 0

    -- Guild name
    local guildName = GetGuildInfo("player") or nil

    -- Gold on hand
    local goldOnHand = math.floor((GetMoney() or 0) / 10000)

    -- Professions, milestones, notable titles
    local professions = CollectProfessionData()
    local milestones = CollectMilestones()
    local notableTitles = CollectNotableTitles()

    pendingSnapshot = {
        name = name,
        realm = realm,
        classFile = classFile,
        className = className,
        race = race,
        level = level,
        specName = specName,
        specIcon = specIcon,
        faction = faction,
        avgItemLevel = avgIlvl or 0,
        avgItemLevelEquipped = avgIlvlEquipped or 0,
        avgItemLevelPvP = avgIlvlPvP or 0,
        achievementPoints = achievementPoints,
        honorableKills = hks,
        honorLevel = honorLevel,
        guildName = guildName,
        pvpRatings = pvpRatings,
        mplusRuns = mplusRuns,
        mplusHighestKey = math.max(mplusHighestKey, existingSnap and existingSnap.mplusHighestKey or 0),
        mplusRating = math.max(mplusRating, existingSnap and existingSnap.mplusRating or 0),
        mountsCollected = mountsCollected,
        petsCollected = petsCollected,
        toysCollected = toysCollected,
        decorCollected = decorCollected,
        goldOnHand = goldOnHand,
        stats = stats,
        professions = professions,
        milestones = milestones,
        notableTitles = notableTitles,
        lastUpdated = time(),
    }

    return charKey
end

local function FinalizeSnapshot(totalPlayed, levelPlayed)
    if not pendingSnapshot then return end
    pendingSnapshot.totalPlayed = totalPlayed or 0
    pendingSnapshot.levelPlayed = levelPlayed or 0

    local charKey = GetCharKey()
    SKToolsProfileDB[charKey] = pendingSnapshot
    pendingSnapshot = nil
end

-----------------------------
-- Event Frame
-----------------------------
local profileFrame = CreateFrame("Frame")
profileFrame:RegisterEvent("PLAYER_ENTERING_WORLD")

profileFrame:SetScript("OnEvent", function(self, event, ...)
    if event == "PLAYER_ENTERING_WORLD" then
        local isLogin, isReload = ...
        if not isLogin and not isReload then return end
        if not SKToolsProfileDB then SKToolsProfileDB = {} end

        -- Collect sync data
        C_Timer.After(2, function()
            CollectCharacterData()
            if SKToolsDB and SKToolsDB.trackPlayedTime ~= false then
                self:RegisterEvent("TIME_PLAYED_MSG")
                RequestTimePlayed()
            else
                FinalizeSnapshot(nil, nil)
            end
        end)

    elseif event == "TIME_PLAYED_MSG" then
        local totalPlayed, levelPlayed = ...
        self:UnregisterEvent("TIME_PLAYED_MSG")
        FinalizeSnapshot(totalPlayed, levelPlayed)
    end
end)

-----------------------------
-- UI Widget Helpers
-----------------------------

-- Stat box: large number on top, label below, optional sub-text
local function CreateStatBox(parent, width, height)
    height = height or 60
    local box = CreateFrame("Frame", nil, parent, "BackdropTemplate")
    box:SetSize(width, height)
    box:SetBackdrop(ns.BACKDROP_CARD)
    box:SetBackdropColor(C.bgCard[1], C.bgCard[2], C.bgCard[3], C.bgCard[4])
    box:SetBackdropBorderColor(C.border[1], C.border[2], C.border[3], C.border[4])

    box.value = box:CreateFontString(nil, "OVERLAY", "GameFontNormalLarge")
    box.value:SetPoint("TOP", 0, -6)
    box.value:SetTextColor(CYAN.r, CYAN.g, CYAN.b)

    box.label = box:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
    box.label:SetPoint("TOP", box.value, "BOTTOM", 0, -2)
    box.label:SetTextColor(C.textDesc[1], C.textDesc[2], C.textDesc[3])

    box.sub = box:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
    box.sub:SetPoint("TOP", box.label, "BOTTOM", 0, -1)
    box.sub:SetTextColor(C.textMuted[1], C.textMuted[2], C.textMuted[3])

    function box:SetData(val, lbl, subText)
        self.value:SetText(val or "")
        self.label:SetText(lbl or "")
        if subText then
            self.sub:SetText(subText)
            self.sub:Show()
        else
            self.sub:SetText("")
            self.sub:Hide()
        end
    end

    return box
end

-- Progress bar with label and count
local function CreateCollectionBar(parent)
    local row = CreateFrame("Frame", nil, parent)
    row:SetHeight(22)

    local icon = row:CreateTexture(nil, "ARTWORK")
    icon:SetSize(18, 18)
    icon:SetPoint("LEFT", 0, 0)
    row.icon = icon

    local lbl = row:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
    lbl:SetPoint("LEFT", icon, "RIGHT", 6, 0)
    lbl:SetTextColor(C.textLabel[1], C.textLabel[2], C.textLabel[3])
    row.label = lbl

    local barBg = CreateFrame("Frame", nil, row, "BackdropTemplate")
    barBg:SetHeight(14)
    barBg:SetPoint("LEFT", row, "LEFT", 100, 0)
    barBg:SetPoint("RIGHT", row, "RIGHT", -60, 0)
    barBg:SetBackdrop(ns.BACKDROP_CONTROL)
    barBg:SetBackdropColor(C.bgControl[1], C.bgControl[2], C.bgControl[3], C.bgControl[4])
    barBg:SetBackdropBorderColor(C.border[1], C.border[2], C.border[3], C.border[4])
    row.barBg = barBg

    local fill = barBg:CreateTexture(nil, "ARTWORK")
    fill:SetPoint("TOPLEFT", 1, -1)
    fill:SetPoint("BOTTOMLEFT", 1, 1)
    fill:SetColorTexture(CYAN.r, CYAN.g, CYAN.b, 0.5)
    row.fill = fill

    local countText = row:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
    countText:SetPoint("RIGHT", row, "RIGHT", 0, 0)
    countText:SetTextColor(C.textLabel[1], C.textLabel[2], C.textLabel[3])
    row.countText = countText

    function row:SetData(labelText, current, iconTex, color, maxVal)
        self.label:SetText(labelText)
        self.countText:SetText(FormatNumber(current))
        if iconTex then self.icon:SetTexture(iconTex) end
        if color then
            self.fill:SetColorTexture(color[1], color[2], color[3], 0.5)
        end
        -- Bar fill proportional to maxVal so bars are relative to each other
        local denominator = maxVal and maxVal > 0 and maxVal or math.max(current, 1)
        C_Timer.After(0, function()
            local barW = self.barBg:GetWidth() - 2
            if barW > 0 then
                local pct = math.min(current / denominator, 1)
                self.fill:SetWidth(math.max(1, pct * barW))
            end
        end)
    end

    return row
end

-- Win rate bar: segmented XP-bar style with notch dividers
local function CreateWinRateBar(parent, labelText, pct, color)
    local row = CreateFrame("Frame", nil, parent)
    row:SetHeight(20)

    local lbl = row:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
    lbl:SetPoint("LEFT", 4, 0)
    lbl:SetText(labelText)
    lbl:SetTextColor(C.textLabel[1], C.textLabel[2], C.textLabel[3])

    local pctText = row:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
    pctText:SetPoint("RIGHT", 0, 0)
    pctText:SetText(string.format("%.1f%%", pct))
    pctText:SetTextColor(CYAN.r, CYAN.g, CYAN.b)

    -- Bar background (dark frame)
    local barBg = CreateFrame("Frame", nil, row, "BackdropTemplate")
    barBg:SetHeight(14)
    barBg:SetPoint("LEFT", row, "LEFT", 120, 0)
    barBg:SetPoint("RIGHT", pctText, "LEFT", -8, 0)
    barBg:SetBackdrop(ns.BACKDROP_CONTROL)
    barBg:SetBackdropColor(C.bgControl[1], C.bgControl[2], C.bgControl[3], C.bgControl[4])
    barBg:SetBackdropBorderColor(C.border[1], C.border[2], C.border[3], C.border[4])

    -- Fill texture (glossy StatusBar texture instead of flat color)
    local fill = barBg:CreateTexture(nil, "ARTWORK")
    fill:SetPoint("TOPLEFT", 1, -1)
    fill:SetPoint("BOTTOMLEFT", 1, 1)
    fill:SetTexture("Interface\\TargetingFrame\\UI-StatusBar")
    fill:SetVertexColor(color[1], color[2], color[3], 0.8)

    -- Deferred: set fill width + draw notch dividers
    C_Timer.After(0, function()
        local barW = barBg:GetWidth() - 2
        if barW <= 0 then return end
        fill:SetWidth(math.max(1, math.floor(barW * pct / 100)))
        -- 9 notch dividers at 10% intervals for segmented look
        for i = 1, 9 do
            local notch = barBg:CreateTexture(nil, "OVERLAY")
            notch:SetSize(1, barBg:GetHeight() - 2)
            notch:SetPoint("LEFT", barBg, "LEFT", 1 + math.floor(barW * i / 10), 0)
            notch:SetColorTexture(0, 0, 0, 0.5)
        end
    end)

    return row
end

-- Superlative row: "Title ... CharName ... Value"
local function CreateSuperlativeRow(parent)
    local row = CreateFrame("Frame", nil, parent)
    row:SetHeight(20)

    local title = row:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
    title:SetPoint("LEFT", 4, 0)
    title:SetTextColor(C.textDesc[1], C.textDesc[2], C.textDesc[3])
    row.title = title

    local nameText = row:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
    nameText:SetPoint("LEFT", row, "LEFT", 120, 0)
    row.nameText = nameText

    local valText = row:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
    valText:SetPoint("RIGHT", row, "RIGHT", -4, 0)
    valText:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
    row.valText = valText

    function row:SetData(titleStr, charName, classFile, value)
        self.title:SetText(titleStr)
        local r, g, b = GetClassColor(classFile or "WARRIOR")
        self.nameText:SetTextColor(r, g, b)
        self.nameText:SetText(charName or "")
        self.valText:SetText(value or "")
    end

    return row
end

-- Roster row: class icon + name + level + ilvl + spec + played + last seen
-- Roster column definitions: shared between headers and data rows
local ROSTER_COLS = {
    { key = "name",     label = "Name",      x = 28,  width = 100, justify = "LEFT" },
    { key = "level",    label = "Lvl",       x = 130, width = 30,  justify = "CENTER" },
    { key = "ilvl",     label = "iLvl",      x = 162, width = 36,  justify = "CENTER" },
    { key = "pvpilvl",  label = "PvP iLvl",  x = 200, width = 46,  justify = "CENTER" },
    { key = "spec",     label = "Spec",      x = 250, width = 75,  justify = "LEFT" },
    { key = "played",   label = "Played",    x = 330, width = 75,  justify = "CENTER" },
    { key = "lastseen", label = "Last Seen",  x = 0,   width = 60,  justify = "RIGHT", anchorRight = -4 },
}

local function CreateRosterRow(parent)
    local row = CreateFrame("Frame", nil, parent)
    row:SetHeight(22)

    local icon = row:CreateTexture(nil, "ARTWORK")
    icon:SetSize(18, 18)
    icon:SetPoint("LEFT", 4, 0)
    local iconMask = row:CreateMaskTexture()
    iconMask:SetSize(16, 16)
    iconMask:SetPoint("CENTER", icon, "CENTER")
    iconMask:SetTexture("Interface\\CHARACTERFRAME\\TempPortraitAlphaMask",
        "CLAMPTOBLACKADDITIVE", "CLAMPTOBLACKADDITIVE")
    icon:AddMaskTexture(iconMask)
    row.icon = icon

    row.cells = {}
    local hidePlayed = SKToolsDB and SKToolsDB.trackPlayedTime == false
    for _, col in ipairs(ROSTER_COLS) do
        if not (col.key == "played" and hidePlayed) then
            local cell = row:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
            if col.anchorRight then
                cell:SetPoint("RIGHT", row, "RIGHT", col.anchorRight, 0)
            else
                cell:SetPoint("LEFT", row, "LEFT", col.x, 0)
            end
            cell:SetWidth(col.width)
            cell:SetJustifyH(col.justify)
            cell:SetTextColor(C.textLabel[1], C.textLabel[2], C.textLabel[3])
            row.cells[col.key] = cell
        end
    end

    function row:SetSnapshot(snap)
        self.icon:SetTexture(CLASS_ICONS[snap.classFile] or "Interface\\Icons\\INV_Misc_QuestionMark")
        local cr, cg, cb = GetClassColor(snap.classFile)
        self.cells.name:SetTextColor(cr, cg, cb)
        self.cells.name:SetText(snap.name or "")
        self.cells.level:SetText(snap.level or "")
        self.cells.ilvl:SetText(string.format("%.0f", snap.avgItemLevelEquipped or 0))
        self.cells.pvpilvl:SetText(string.format("%.0f", snap.avgItemLevelPvP or 0))
        self.cells.spec:SetText(snap.specName or "")
        self.cells.spec:SetTextColor(C.textDesc[1], C.textDesc[2], C.textDesc[3])
        if self.cells.played then self.cells.played:SetText(FormatPlayedShort(snap.totalPlayed)) end
        -- Last seen
        if snap.lastUpdated then
            local ago = time() - snap.lastUpdated
            if ago < 60 then
                self.cells.lastseen:SetText("Just now")
            elseif ago < 3600 then
                self.cells.lastseen:SetText(math.floor(ago / 60) .. "m ago")
            elseif ago < 86400 then
                self.cells.lastseen:SetText(math.floor(ago / 3600) .. "h ago")
            else
                self.cells.lastseen:SetText(math.floor(ago / 86400) .. "d ago")
            end
        else
            self.cells.lastseen:SetText("")
        end
        self.cells.lastseen:SetTextColor(C.textMuted[1], C.textMuted[2], C.textMuted[3])
    end

    return row
end

-----------------------------
-- Character View Builder
-----------------------------
local function BuildCharacterView(content, snap)
    if not snap then return content end

    local r, g, b = GetClassColor(snap.classFile)
    local classHex = GetClassHex(snap.classFile)

    -- Character Header
    local headerCard = ns.CreateSectionCard(content)

    local charIcon = content:CreateTexture(nil, "ARTWORK")
    charIcon:SetSize(48, 48)
    charIcon:SetPoint("TOPLEFT", content, "TOPLEFT", 20, -8)
    charIcon:SetTexture(CLASS_ICONS[snap.classFile] or "Interface\\Icons\\INV_Misc_QuestionMark")
    local iconMask = content:CreateMaskTexture()
    iconMask:SetSize(42, 42)
    iconMask:SetPoint("CENTER", charIcon, "CENTER")
    iconMask:SetTexture("Interface\\CHARACTERFRAME\\TempPortraitAlphaMask",
        "CLAMPTOBLACKADDITIVE", "CLAMPTOBLACKADDITIVE")
    charIcon:AddMaskTexture(iconMask)

    local charName = content:CreateFontString(nil, "ARTWORK", "GameFontNormalLarge")
    charName:SetPoint("LEFT", charIcon, "RIGHT", 10, 8)
    charName:SetText(snap.name)
    charName:SetTextColor(r, g, b)

    -- 3K+ badge next to name (no date available, so shown here instead of timeline)
    local peak3k = math.max(SafeStat(370), SafeStat(595))
    if peak3k >= 3000 then
        local btype = ns.PVP_BADGE_TYPES["threek"]
        if btype then
            local threekBadge = ns.CreateCircularBadge(content, btype.icon,
                btype.ring[1], btype.ring[2], btype.ring[3], 22)
            threekBadge:SetPoint("LEFT", charName, "RIGHT", 6, 0)
            threekBadge.count:SetText("3k")
            threekBadge.count:SetFont(STANDARD_TEXT_FONT, 8, "OUTLINE")
            threekBadge.count:SetTextColor(btype.countColor[1], btype.countColor[2], btype.countColor[3])
            threekBadge.count:Show()
            threekBadge:EnableMouse(true)
            threekBadge:SetScript("OnEnter", function(self)
                GameTooltip:SetOwner(self, "ANCHOR_RIGHT")
                GameTooltip:AddLine("3000+ Rating", 1, 0.84, 0)
                GameTooltip:AddLine(string.format("Peak: %s", FormatNumber(peak3k)), 0.7, 0.7, 0.7)
                GameTooltip:Show()
            end)
            threekBadge:SetScript("OnLeave", function() GameTooltip:Hide() end)
            threekBadge:Show()
        end
    end

    local charSub = content:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
    charSub:SetPoint("TOPLEFT", charName, "BOTTOMLEFT", 0, -6)
    charSub:SetText(string.format("Level %d %s %s — %s",
        snap.level or 0, snap.race or "", snap.className or "", snap.specName or ""))
    charSub:SetTextColor(C.textDesc[1], C.textDesc[2], C.textDesc[3])

    -- Right side stat badges: Honor Level, Item Level, PvP iLvl
    local STAT_BADGE_SZ = 44
    local STAT_BADGE_GAP = 10
    local STAT_LABEL_H = 12  -- label above badge

    -- Helper: create a stat badge (label on top, circular badge below)
    local function MakeStatBadge(value, label, ringR, ringG, ringB, textR, textG, textB)
        local totalH = STAT_LABEL_H + 2 + STAT_BADGE_SZ
        local f = CreateFrame("Frame", nil, content)
        f:SetSize(STAT_BADGE_SZ, totalH)

        -- Label above badge
        local lblText = f:CreateFontString(nil, "OVERLAY")
        lblText:SetFont(STANDARD_TEXT_FONT, 8, "OUTLINE")
        lblText:SetPoint("TOP", f, "TOP", 0, 0)
        lblText:SetText(label)
        lblText:SetTextColor(C.textDesc[1], C.textDesc[2], C.textDesc[3])

        -- Circular background with ring
        local mask = f:CreateMaskTexture()
        mask:SetSize(STAT_BADGE_SZ, STAT_BADGE_SZ)
        mask:SetPoint("BOTTOM")
        mask:SetTexture("Interface\\CHARACTERFRAME\\TempPortraitAlphaMask",
            "CLAMPTOBLACKADDITIVE", "CLAMPTOBLACKADDITIVE")

        local innerSz = STAT_BADGE_SZ - 4
        local innerMask = f:CreateMaskTexture()
        innerMask:SetSize(innerSz, innerSz)
        innerMask:SetPoint("BOTTOM", 0, 2)
        innerMask:SetTexture("Interface\\CHARACTERFRAME\\TempPortraitAlphaMask",
            "CLAMPTOBLACKADDITIVE", "CLAMPTOBLACKADDITIVE")

        local ring = f:CreateTexture(nil, "BACKGROUND")
        ring:SetSize(STAT_BADGE_SZ, STAT_BADGE_SZ)
        ring:SetPoint("BOTTOM")
        ring:SetColorTexture(ringR, ringG, ringB, 0.9)
        ring:AddMaskTexture(mask)

        local fill = f:CreateTexture(nil, "BORDER")
        fill:SetSize(innerSz, innerSz)
        fill:SetPoint("BOTTOM", 0, 2)
        fill:SetColorTexture(0.08, 0.08, 0.1, 0.95)
        fill:AddMaskTexture(innerMask)

        -- Value text centered in badge circle
        local valText = f:CreateFontString(nil, "OVERLAY")
        valText:SetFont(STANDARD_TEXT_FONT, 14, "OUTLINE")
        valText:SetPoint("CENTER", ring, "CENTER", 0, 0)
        valText:SetText(value)
        valText:SetTextColor(textR or 1, textG or 1, textB or 1)

        f:Show()
        return f
    end

    local pvpIlvl = snap.avgItemLevelPvP or 0
    local ilvl = snap.avgItemLevelEquipped or 0

    -- Build stat badges, then vertically center them in header after card height is known
    local pvpIlvlBadge = MakeStatBadge(string.format("%.0f", pvpIlvl), "PvP iLvl",
        0.7, 0.15, 0.15,  0.9, 0.2, 0.2)

    local ilvlBadge = MakeStatBadge(string.format("%.0f", ilvl), "Item Level",
        0.1, 0.5, 0.6,  CYAN.r, CYAN.g, CYAN.b)

    -- Position right-to-left, vertically centered in header card (deferred for height calc)
    local function PositionStatBadges()
        local cardTop = headerCard:GetTop()
        local cardBot = headerCard:GetBottom()
        if not cardTop or not cardBot then return end
        local cardH = cardTop - cardBot
        local badgeH = STAT_LABEL_H + 2 + STAT_BADGE_SZ
        local yOff = -math.max(0, (cardH - badgeH) / 2)

        pvpIlvlBadge:ClearAllPoints()
        pvpIlvlBadge:SetPoint("TOPRIGHT", content, "TOPRIGHT", -16, yOff)
        ilvlBadge:ClearAllPoints()
        ilvlBadge:SetPoint("TOPRIGHT", pvpIlvlBadge, "TOPLEFT", -STAT_BADGE_GAP, 0)
    end

    -- Initial placement (fallback), then deferred for accurate centering
    pvpIlvlBadge:SetPoint("TOPRIGHT", content, "TOPRIGHT", -16, -10)
    ilvlBadge:SetPoint("TOPRIGHT", pvpIlvlBadge, "TOPLEFT", -STAT_BADGE_GAP, 0)
    C_Timer.After(0, PositionStatBadges)

    -- Guild name (green, between subtitle and played time)
    local lastSubLine = charSub
    if snap.guildName and snap.guildName ~= "" then
        local guildText = content:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        guildText:SetPoint("TOPLEFT", charSub, "BOTTOMLEFT", 0, -3)
        guildText:SetText("<" .. snap.guildName .. ">")
        guildText:SetTextColor(0.25, 1, 0.25)
        lastSubLine = guildText
    end

    -- Played time (only if tracking is enabled)
    local headerBottom = lastSubLine
    if SKToolsDB and SKToolsDB.trackPlayedTime ~= false and snap.totalPlayed and snap.totalPlayed > 0 then
        local playedText = content:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        playedText:SetPoint("TOPLEFT", lastSubLine, "BOTTOMLEFT", 0, -4)
        playedText:SetText("Played: " .. FormatPlayedTime(snap.totalPlayed))
        playedText:SetTextColor(C.textLabel[1], C.textLabel[2], C.textLabel[3])
        headerBottom = playedText
    end

    headerCard:SetPoint("TOPLEFT", content, "TOPLEFT", 8, 0)
    headerCard:SetPoint("RIGHT", content, "RIGHT", -8, 0)
    headerCard:SetPoint("BOTTOM", headerBottom, "BOTTOM", 0, -10)

    -- Milestones Section (merged PvP titles + milestones timeline)
    local lastBeforePvP = headerBottom  -- anchor for next section
    do
        local ownBadges = ScanOwnPvPTitles()
        local milestones = snap.milestones

        -- Build merged timeline: combine ownBadges (PvP titles) + milestones chronologically
        local timeline = {}

        -- Add PvP title badges with achievement dates (skip 3k — shown in header instead)
        for _, bd in ipairs(ownBadges) do
            local btype = ns.PVP_BADGE_TYPES[bd.type]
            if btype and bd.type ~= "threek" then
                local month, day, year
                if bd.id then
                    local _, achName, _, _, m, d, y = GetAchievementInfo(bd.id)
                    month, day, year = m, d, y
                    bd.achName = achName
                end
                timeline[#timeline + 1] = {
                    source = "pvp",
                    badgeType = bd.type,
                    btype = btype,
                    bd = bd,
                    month = month, day = day, year = year,
                }
            end
        end

        -- Add milestone entries
        if milestones then
            for _, entry in ipairs(milestones) do
                timeline[#timeline + 1] = {
                    source = "milestone",
                    entry = entry,
                    month = entry.month, day = entry.day, year = entry.year,
                }
            end
        end

        -- Sort chronologically (oldest first)
        table.sort(timeline, function(a, b)
            if (a.year or 0) ~= (b.year or 0) then return (a.year or 0) < (b.year or 0) end
            if (a.month or 0) ~= (b.month or 0) then return (a.month or 0) < (b.month or 0) end
            return (a.day or 0) < (b.day or 0)
        end)

        if #timeline > 0 then
            local msHeader = ns.AddSectionHeader(content, headerBottom, "Milestones", false)

            local msCard = ns.CreateSectionCard(content)
            msCard:SetPoint("TOPLEFT", msHeader, "TOPLEFT", -8, 8)

            local BADGE_SZ = 36
            local BADGE_GAP = 8
            local YEAR_LABEL_H = 12   -- year label above badge
            local BOTTOM_LABEL_H = 14 -- season label / count label below badge
            local ROW_HEIGHT = YEAR_LABEL_H + BADGE_SZ + BOTTOM_LABEL_H + 8
            local PAD = 16
            local LINE_THICK = 2

            -- Create badge frames for each timeline entry
            local allBadges = {}
            for _, item in ipairs(timeline) do
                local badge, tooltipFn

                if item.source == "pvp" then
                    -- PvP title badge: use badge type icon + ring color
                    local btype = item.btype
                    local bd = item.bd
                    badge = ns.CreateCircularBadge(content, btype.icon,
                        btype.ring[1], btype.ring[2], btype.ring[3], BADGE_SZ)

                    -- Season label below badge
                    badge.label:SetText(bd.label or "")
                    badge.label:SetTextColor(btype.countColor[1], btype.countColor[2], btype.countColor[3])
                    badge.label:Show()

                    tooltipFn = function(self)
                        GameTooltip:SetOwner(self, "ANCHOR_RIGHT")
                        local typeName = btype and ns.PVP_BADGE_LABELS and ns.PVP_BADGE_LABELS[bd.type]
                        if bd.achName then
                            GameTooltip:AddLine(bd.achName, 1, 1, 1)
                        elseif bd.type == "threek" then
                            GameTooltip:AddLine("3000+ Rating", 1, 0.84, 0)
                        elseif bd.type == "hero" then
                            GameTooltip:AddLine("Hero of the Horde / Alliance", 1, 1, 1)
                        end
                        if item.month and item.year and item.year > 0 then
                            GameTooltip:AddLine(
                                string.format("%d/%d/%d", item.month, item.day, 2000 + (item.year or 0)),
                                0.7, 0.7, 0.7)
                        end
                        GameTooltip:Show()
                    end
                else
                    -- Milestone badge: use overrideIcon or achievement icon
                    local entry = item.entry
                    local clr = entry.color or GENERAL_COLOR
                    local iconTex = entry.overrideIcon
                    if not iconTex then
                        iconTex = entry.icon
                        if not iconTex or iconTex == 0 then
                            iconTex = 136243  -- question mark fallback
                        end
                    end

                    badge = ns.CreateCircularBadge(content, iconTex,
                        clr[1], clr[2], clr[3], BADGE_SZ)

                    -- Count text on badge (bottom-right)
                    if entry.countText then
                        badge.count:SetText(entry.countText)
                        badge.count:SetTextColor(clr[1], clr[2], clr[3])
                        badge.count:Show()
                    end

                    -- Category label below badge
                    if entry.cat == "aotc" then
                        badge.label:SetText("AOTC")
                        badge.label:SetTextColor(AOTC_COLOR[1], AOTC_COLOR[2], AOTC_COLOR[3])
                        badge.label:Show()
                    elseif entry.cat == "ce" then
                        badge.label:SetText("CE")
                        badge.label:SetTextColor(CE_COLOR[1], CE_COLOR[2], CE_COLOR[3])
                        badge.label:Show()
                    elseif entry.mplusTier then
                        badge.label:SetText(entry.mplusTier)
                        badge.label:SetTextColor(MPLUS_LEGEND_COLOR[1], MPLUS_LEGEND_COLOR[2], MPLUS_LEGEND_COLOR[3])
                        badge.label:Show()
                    elseif entry.cat == "delve" then
                        badge.label:SetText("SOLO")
                        badge.label:SetTextColor(0.9, 0.6, 0.25)
                        badge.label:Show()
                    end

                    tooltipFn = function(self)
                        GameTooltip:SetOwner(self, "ANCHOR_RIGHT")
                        GameTooltip:AddLine(entry.label, clr[1], clr[2], clr[3])
                        if entry.name and entry.name ~= entry.label then
                            GameTooltip:AddLine(entry.name, 1, 1, 1)
                        end
                        if entry.month and entry.year and entry.year > 0 then
                            GameTooltip:AddLine(
                                string.format("%d/%d/%d", entry.month, entry.day, 2000 + (entry.year or 0)),
                                0.7, 0.7, 0.7)
                        end
                        GameTooltip:Show()
                    end
                end

                -- Year label ABOVE badge (separate fontstring, not badge.label)
                local yearLabel = badge:CreateFontString(nil, "OVERLAY")
                yearLabel:SetFont(STANDARD_TEXT_FONT, 9, "OUTLINE")
                yearLabel:SetPoint("BOTTOM", badge, "TOP", 0, 1)
                if item.year and item.year > 0 then
                    yearLabel:SetText("'" .. string.format("%02d", item.year))
                    yearLabel:SetTextColor(C.textMuted[1], C.textMuted[2], C.textMuted[3])
                end
                badge.yearLabel = yearLabel

                -- Tooltip on hover + click to open achievement
                badge:EnableMouse(true)
                badge:SetScript("OnEnter", tooltipFn)
                badge:SetScript("OnLeave", function() GameTooltip:Hide() end)

                local clickAchId = item.source == "pvp" and item.bd.id or (item.entry and item.entry.achId)
                if clickAchId then
                    badge:SetScript("OnMouseUp", function()
                        if OpenAchievementFrameToAchievement then
                            OpenAchievementFrameToAchievement(clickAchId)
                        end
                    end)
                end

                badge:Show()
                allBadges[#allBadges + 1] = badge
            end

            -- Connecting line textures (one per row)
            local msLines = {}

            local function LayoutTimeline()
                local cardWidth = content:GetWidth() - PAD * 2 - 24
                if cardWidth <= 0 then cardWidth = 600 end
                local n = #allBadges
                local cellStep = BADGE_SZ + BADGE_GAP

                -- Determine columns per row
                local maxPerRow = math.floor((cardWidth + BADGE_GAP) / cellStep)
                if maxPerRow < 1 then maxPerRow = 1 end
                local cols = maxPerRow
                if n <= maxPerRow then
                    cols = n
                else
                    local rows = math.ceil(n / cols)
                    local lastRowCount = n - (rows - 1) * cols
                    if lastRowCount < cols * 0.5 and cols > 1 then
                        local tryCols = cols - 1
                        local tryRows = math.ceil(n / tryCols)
                        local tryLast = n - (tryRows - 1) * tryCols
                        if tryRows <= rows + 1 and tryLast >= tryCols * 0.5 then
                            cols = tryCols
                        end
                    end
                end

                local totalRows = math.ceil(n / cols)

                -- Position badges in grid (with year label space above)
                for i, badge in ipairs(allBadges) do
                    local row = math.ceil(i / cols) - 1
                    local col = (i - 1) % cols

                    local rowStart = row * cols + 1
                    local rowEnd = math.min(rowStart + cols - 1, n)
                    local rowCount = rowEnd - rowStart + 1
                    local rowWidth = rowCount * BADGE_SZ + (rowCount - 1) * BADGE_GAP

                    local startX
                    if rowCount < cols then
                        local fullRowWidth = cols * BADGE_SZ + (cols - 1) * BADGE_GAP
                        startX = (cardWidth - fullRowWidth) / 2
                    else
                        startX = (cardWidth - rowWidth) / 2
                    end

                    badge:ClearAllPoints()
                    badge:SetPoint("TOPLEFT", msHeader, "BOTTOMLEFT",
                        PAD + startX + col * cellStep,
                        -(YEAR_LABEL_H + 6) - row * ROW_HEIGHT)
                end

                -- Draw connecting lines per row
                for _, line in ipairs(msLines) do line:Hide() end
                for row = 0, totalRows - 1 do
                    local rowStart = row * cols + 1
                    local rowEnd = math.min(rowStart + cols - 1, n)
                    local rowCount = rowEnd - rowStart + 1
                    if rowCount > 1 then
                        local line = msLines[row + 1]
                        if not line then
                            line = content:CreateTexture(nil, "BACKGROUND")
                            line:SetColorTexture(0.4, 0.4, 0.45, 0.5)
                            msLines[row + 1] = line
                        end
                        local first = allBadges[rowStart]
                        local last = allBadges[rowEnd]
                        line:SetHeight(LINE_THICK)
                        line:SetPoint("LEFT", first, "CENTER", 0, 0)
                        line:SetPoint("RIGHT", last, "CENTER", 0, 0)
                        line:Show()
                    end
                end

                -- Size the card
                local totalH = totalRows * ROW_HEIGHT + YEAR_LABEL_H + 10
                msCard:SetPoint("BOTTOM", msHeader, "BOTTOM", 0, -(totalH + 8))
            end

            LayoutTimeline()
            C_Timer.After(0, LayoutTimeline)

            msCard:SetPoint("RIGHT", content, "RIGHT", -8, 0)
            lastBeforePvP = msCard
        end
    end

    -- PvP Summary (no section header — compact row of HK, 2v2 Peak, 3v3 Peak)
    local boxGap = 6
    local boxH = 66
    local padL, padR = 4, 20
    local pvpTopY = -16  -- gap below previous section

    local function MakePeakBox(parent, bracket)
        local info = snap.pvpRatings and snap.pvpRatings[bracket.idx]
        local rating = info and info.rating or 0
        local best = info and info.seasonBest or 0
        local allTime = info and info.allTimeBest or 0

        local peak = math.max(allTime, best, rating)
        local nameRealm = snap.name .. "-" .. snap.realm
        local lbPeak = ns.GetLeaderboardPeak(nameRealm, bracket.idx)
        if lbPeak and lbPeak > peak then peak = lbPeak end

        local box = CreateFrame("Frame", nil, parent, "BackdropTemplate")
        box:SetSize(10, boxH)
        box:SetBackdrop(ns.BACKDROP_CARD)
        box:SetBackdropColor(C.bgCard[1], C.bgCard[2], C.bgCard[3], C.bgCard[4])
        box:SetBackdropBorderColor(C.border[1], C.border[2], C.border[3], C.border[4])

        local val = box:CreateFontString(nil, "OVERLAY", "GameFontNormalLarge")
        val:SetPoint("CENTER", 0, 6)
        val:SetText(peak > 0 and FormatNumber(peak) or "—")
        val:SetTextColor(CYAN.r, CYAN.g, CYAN.b)

        local lbl = box:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
        lbl:SetPoint("TOP", val, "BOTTOM", 0, -2)
        lbl:SetText(bracket.label .. " Peak")
        lbl:SetTextColor(C.textDesc[1], C.textDesc[2], C.textDesc[3])

        return box
    end

    -- HK box (centered content)
    local hkBox = CreateFrame("Frame", nil, content, "BackdropTemplate")
    hkBox:SetSize(10, boxH)
    hkBox:SetBackdrop(ns.BACKDROP_CARD)
    hkBox:SetBackdropColor(C.bgCard[1], C.bgCard[2], C.bgCard[3], C.bgCard[4])
    hkBox:SetBackdropBorderColor(C.border[1], C.border[2], C.border[3], C.border[4])

    local hkVal = hkBox:CreateFontString(nil, "OVERLAY", "GameFontNormalLarge")
    hkVal:SetPoint("CENTER", 0, 6)
    hkVal:SetText(FormatNumber(snap.honorableKills))
    hkVal:SetTextColor(CYAN.r, CYAN.g, CYAN.b)

    local hkLbl = hkBox:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
    hkLbl:SetPoint("TOP", hkVal, "BOTTOM", 0, -2)
    hkLbl:SetText("Honorable Kills")
    hkLbl:SetTextColor(C.textDesc[1], C.textDesc[2], C.textDesc[3])

    local box2v2 = MakePeakBox(content, BRACKETS[1])
    local box3v3 = MakePeakBox(content, BRACKETS[2])

    -- Wrap card around the boxes
    local pvpCard = ns.CreateSectionCard(content)
    pvpCard:SetPoint("TOPLEFT", content, "TOPLEFT", 8, 0)
    pvpCard:SetPoint("RIGHT", content, "RIGHT", -8, 0)
    pvpCard:SetHeight(boxH + 12)

    -- Defer card vertical position (needs lastBeforePvP to have geometry)
    C_Timer.After(0, function()
        local ref = lastBeforePvP
        if ref and ref.GetBottom and ref:GetBottom() then
            pvpCard:ClearAllPoints()
            pvpCard:SetPoint("TOPLEFT", lastBeforePvP, "BOTTOMLEFT", 0, pvpTopY + 6)
            pvpCard:SetPoint("RIGHT", content, "RIGHT", -8, 0)
            pvpCard:SetHeight(boxH + 12)
        end
    end)

    -- Anchor boxes to lastBeforePvP (not pvpCard, to avoid circular dependency)
    local pvpBoxes = {hkBox, box2v2, box3v3}
    local boxInset = 12  -- inset from content edges to match other section cards
    hkBox:SetPoint("TOPLEFT", lastBeforePvP, "BOTTOMLEFT", boxInset, pvpTopY)
    box2v2:SetPoint("TOPLEFT", lastBeforePvP, "BOTTOMLEFT", boxInset + 210, pvpTopY)
    box3v3:SetPoint("TOPLEFT", lastBeforePvP, "BOTTOMLEFT", boxInset + 420, pvpTopY)

    C_Timer.After(0, function()
        local totalW = content:GetWidth() - boxInset * 2 - 16
        if totalW <= 0 then return end
        local cellW = math.floor((totalW - 2 * boxGap) / 3)
        for i, box in ipairs(pvpBoxes) do
            local xOff = boxInset + (i - 1) * (cellW + boxGap)
            box:ClearAllPoints()
            box:SetPoint("TOPLEFT", lastBeforePvP, "BOTTOMLEFT", xOff, pvpTopY)
            box:SetSize(cellW, boxH)
        end
    end)

    -- Statistics Section (grouped with zebra stripes + win rate bars)
    local statsHeader = ns.AddSectionHeader(content, pvpCard, "Statistics", false)

    local statsCard = ns.CreateSectionCard(content)
    statsCard:SetPoint("TOPLEFT", statsHeader, "TOPLEFT", -8, 8)

    local stats = snap.stats or {}
    local goldOnHand = snap.goldOnHand or 0
    local lastStatRow = statsHeader
    local isFirstGroup = true

    for gi, group in ipairs(STAT_GROUPS) do
        -- Group sub-header
        local subHeader = content:CreateFontString(nil, "ARTWORK", "GameFontNormal")
        if isFirstGroup then
            subHeader:SetPoint("TOPLEFT", lastStatRow, "BOTTOMLEFT", 4, -10)
            isFirstGroup = false
        else
            subHeader:SetPoint("TOPLEFT", lastStatRow, "BOTTOMLEFT", 0, -12)
        end
        subHeader:SetText(group.header)
        subHeader:SetTextColor(1, 1, 1)
        lastStatRow = subHeader

        -- Gold on Hand as first Economy row (special, not from GetStatistic)
        local rowIndex = 0
        if group.header == "Economy" and goldOnHand > 0 then
            rowIndex = rowIndex + 1
            local row = CreateFrame("Frame", nil, content)
            row:SetHeight(18)
            row:SetPoint("TOPLEFT", lastStatRow, "BOTTOMLEFT", 0, -2)
            row:SetPoint("RIGHT", content, "RIGHT", -20, 0)

            -- Zebra stripe
            if rowIndex % 2 == 1 then
                local stripe = row:CreateTexture(nil, "BACKGROUND")
                stripe:SetAllPoints()
                stripe:SetColorTexture(1, 1, 1, 0.03)
            end

            local lbl = row:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
            lbl:SetPoint("LEFT", 4, 0)
            lbl:SetText("Gold on Hand")
            lbl:SetTextColor(C.textLabel[1], C.textLabel[2], C.textLabel[3])

            local val = row:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
            val:SetPoint("RIGHT", 0, 0)
            val:SetText(FormatNumber(goldOnHand) .. "g")
            val:SetTextColor(CYAN.r, CYAN.g, CYAN.b)

            lastStatRow = row
        end

        -- Stat rows in this group
        for si, def in ipairs(group.stats) do
            rowIndex = rowIndex + 1
            local row = CreateFrame("Frame", nil, content)
            row:SetHeight(18)
            row:SetPoint("TOPLEFT", lastStatRow, "BOTTOMLEFT", 0, -2)
            row:SetPoint("RIGHT", content, "RIGHT", -20, 0)

            -- Zebra stripe
            if rowIndex % 2 == 1 then
                local stripe = row:CreateTexture(nil, "BACKGROUND")
                stripe:SetAllPoints()
                stripe:SetColorTexture(1, 1, 1, 0.03)
            end

            local lbl = row:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
            lbl:SetPoint("LEFT", 4, 0)
            lbl:SetText(def.label)
            lbl:SetTextColor(C.textLabel[1], C.textLabel[2], C.textLabel[3])

            local val = row:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
            val:SetPoint("RIGHT", 0, 0)
            local displayVal = FormatNumber(stats[def.key] or 0)
            if def.suffix then displayVal = displayVal .. def.suffix end
            val:SetText(displayVal)
            val:SetTextColor(CYAN.r, CYAN.g, CYAN.b)

            lastStatRow = row
        end

        -- Win rate bars for Combat group (XP-bar style)
        if group.winRates then
            local winRateData = {
                { label = "Arena Win Rate",  played = "arenasPlayed", won = "arenasWon",  color = {0.8, 0.2, 0.2} },
                { label = "Blitz Win Rate",  played = "blitzPlayed",  won = "blitzWon",   color = {0.6, 0.2, 0.8} },
                { label = "Duels Win Rate",  played = nil,            won = "duelsWon",   lost = "duelsLost", color = {0.2, 0.8, 0.3} },
            }

            for _, wr in ipairs(winRateData) do
                local total, wins
                if wr.lost then
                    wins = stats[wr.won] or 0
                    total = wins + (stats[wr.lost] or 0)
                else
                    total = stats[wr.played] or 0
                    wins = stats[wr.won] or 0
                end

                if total >= 10 then
                    local pct = (wins / total) * 100
                    local bar = CreateWinRateBar(content, wr.label, pct, wr.color)
                    bar:SetPoint("TOPLEFT", lastStatRow, "BOTTOMLEFT", 0, -4)
                    bar:SetPoint("RIGHT", content, "RIGHT", -20, 0)
                    lastStatRow = bar
                end
            end
        end
    end

    statsCard:SetPoint("BOTTOM", lastStatRow, "BOTTOM", 0, -10)
    statsCard:SetPoint("RIGHT", content, "RIGHT", -8, 0)

    -- Return bottom element for height calculation
    return statsCard
end

-----------------------------
-- Account View Builder
-----------------------------
local function GetSortedSnapshots()
    local snaps = {}
    for key, snap in pairs(SKToolsProfileDB or {}) do
        snap._key = key
        snaps[#snaps + 1] = snap
    end
    table.sort(snaps, function(a, b)
        return (a.totalPlayed or 0) > (b.totalPlayed or 0)
    end)
    return snaps
end

local function BuildAccountView(content, snaps)
    if #snaps == 0 then
        local noData = content:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        noData:SetPoint("TOPLEFT", 12, -60)
        noData:SetText("No character data yet. Log in on your characters to start collecting!")
        noData:SetTextColor(C.textDesc[1], C.textDesc[2], C.textDesc[3])
        return noData
    end

    -- Aggregate data
    local totalPlayed = 0
    local classTimes = {}  -- classFile → totalSeconds
    local totalHKs = 0
    local totalDeaths = 0
    local totalQuests = 0
    local totalKills = 0
    local totalGold = 0      -- sum of goldOnHand across characters
    local totalArenas = 0
    local totalAchieve = 0
    local maxHonorLevel = 0
    local maxMounts, maxPets, maxToys, maxDecor = 0, 0, 0, 0

    -- Superlatives
    local mostPlayed = { snap = nil, val = 0 }
    local highestIlvl = { snap = nil, val = 0 }
    local mostHKs = { snap = nil, val = 0 }
    local mostDeaths = { snap = nil, val = 0 }
    local mostQuests = { snap = nil, val = 0 }
    local bestDuelist = { snap = nil, val = 0 }
    local mostArenas = { snap = nil, val = 0 }
    local bestMplusRating = { snap = nil, val = 0 }
    local bestMplusKey = { snap = nil, val = 0 }

    -- Best ratings per bracket across all characters
    local bestRatings = {}  -- bracketIdx → { snap, rating }
    for _, bracket in ipairs(BRACKETS) do
        bestRatings[bracket.idx] = { snap = nil, rating = 0 }
    end

    for _, snap in ipairs(snaps) do
        local played = snap.totalPlayed or 0
        totalPlayed = totalPlayed + played
        classTimes[snap.classFile] = (classTimes[snap.classFile] or 0) + played

        totalHKs = totalHKs + (snap.honorableKills or 0)
        totalAchieve = math.max(totalAchieve, snap.achievementPoints or 0)
        maxHonorLevel = math.max(maxHonorLevel, snap.honorLevel or 0)
        maxMounts = math.max(maxMounts, snap.mountsCollected or 0)
        maxPets = math.max(maxPets, snap.petsCollected or 0)
        maxToys = math.max(maxToys, snap.toysCollected or 0)
        maxDecor = math.max(maxDecor, snap.decorCollected or 0)

        totalGold = totalGold + (snap.goldOnHand or 0)

        local stats = snap.stats or {}
        totalDeaths = totalDeaths + (stats.totalDeaths or 0)
        totalQuests = totalQuests + (stats.questsCompleted or 0)
        totalKills = totalKills + (stats.creaturesKilled or 0)
        totalArenas = totalArenas + (stats.arenasPlayed or 0)

        if played > mostPlayed.val then
            mostPlayed = { snap = snap, val = played }
        end
        if (snap.avgItemLevelEquipped or 0) > highestIlvl.val then
            highestIlvl = { snap = snap, val = snap.avgItemLevelEquipped or 0 }
        end
        if (snap.honorableKills or 0) > mostHKs.val then
            mostHKs = { snap = snap, val = snap.honorableKills or 0 }
        end
        if (stats.totalDeaths or 0) > mostDeaths.val then
            mostDeaths = { snap = snap, val = stats.totalDeaths or 0 }
        end
        if (stats.questsCompleted or 0) > mostQuests.val then
            mostQuests = { snap = snap, val = stats.questsCompleted or 0 }
        end
        local duelWins = stats.duelsWon or 0
        if duelWins > bestDuelist.val then
            bestDuelist = { snap = snap, val = duelWins }
        end
        local arenas = stats.arenasPlayed or 0
        if arenas > mostArenas.val then
            mostArenas = { snap = snap, val = arenas }
        end
        if (snap.mplusRating or 0) > bestMplusRating.val then
            bestMplusRating = { snap = snap, val = snap.mplusRating }
        end
        if (snap.mplusHighestKey or 0) > bestMplusKey.val then
            bestMplusKey = { snap = snap, val = snap.mplusHighestKey }
        end

        -- Track best rating per bracket
        local pvpR = snap.pvpRatings or {}
        for _, bracket in ipairs(BRACKETS) do
            local info = pvpR[bracket.idx]
            if info then
                local best = math.max(info.allTimeBest or 0, info.seasonBest or 0, info.rating or 0)
                if best > bestRatings[bracket.idx].rating then
                    bestRatings[bracket.idx] = { snap = snap, rating = best }
                end
            end
        end
    end

    -- Total Played Time header
    local totalHeader = content:CreateFontString(nil, "ARTWORK", "GameFontNormalLarge")
    totalHeader:SetPoint("TOPLEFT", 12, -8)
    local trackPlayed = SKToolsDB and SKToolsDB.trackPlayedTime ~= false
    if trackPlayed and totalPlayed > 0 then
        totalHeader:SetText("Total Played: " .. FormatPlayedTime(totalPlayed))
    else
        totalHeader:SetText("Account Overview")
    end
    totalHeader:SetTextColor(CYAN.r, CYAN.g, CYAN.b)

    -- Honor Level badge (top-right of account view)
    if maxHonorLevel > 0 then
        local honorSz = 44
        local honorLabelH = 12
        local honorF = CreateFrame("Frame", nil, content)
        honorF:SetSize(honorSz, honorLabelH + 2 + honorSz)
        honorF:SetPoint("TOPRIGHT", content, "TOPRIGHT", -16, -6)

        local honorLbl = honorF:CreateFontString(nil, "OVERLAY")
        honorLbl:SetFont(STANDARD_TEXT_FONT, 8, "OUTLINE")
        honorLbl:SetPoint("TOP", honorF, "TOP", 0, 0)
        honorLbl:SetText("Honor Level")
        honorLbl:SetTextColor(C.textDesc[1], C.textDesc[2], C.textDesc[3])

        local hMask = honorF:CreateMaskTexture()
        hMask:SetSize(honorSz, honorSz)
        hMask:SetPoint("BOTTOM")
        hMask:SetTexture("Interface\\CHARACTERFRAME\\TempPortraitAlphaMask",
            "CLAMPTOBLACKADDITIVE", "CLAMPTOBLACKADDITIVE")
        local hInner = honorSz - 4
        local hInnerMask = honorF:CreateMaskTexture()
        hInnerMask:SetSize(hInner, hInner)
        hInnerMask:SetPoint("BOTTOM", 0, 2)
        hInnerMask:SetTexture("Interface\\CHARACTERFRAME\\TempPortraitAlphaMask",
            "CLAMPTOBLACKADDITIVE", "CLAMPTOBLACKADDITIVE")
        local hRing = honorF:CreateTexture(nil, "BACKGROUND")
        hRing:SetSize(honorSz, honorSz)
        hRing:SetPoint("BOTTOM")
        hRing:SetColorTexture(0.6, 0.5, 0.1, 0.9)
        hRing:AddMaskTexture(hMask)
        local hFill = honorF:CreateTexture(nil, "BORDER")
        hFill:SetSize(hInner, hInner)
        hFill:SetPoint("BOTTOM", 0, 2)
        hFill:SetColorTexture(0.08, 0.08, 0.1, 0.95)
        hFill:AddMaskTexture(hInnerMask)
        local hVal = honorF:CreateFontString(nil, "OVERLAY")
        hVal:SetFont(STANDARD_TEXT_FONT, 14, "OUTLINE")
        hVal:SetPoint("CENTER", hRing, "CENTER", 0, 0)
        hVal:SetText(tostring(maxHonorLevel))
        hVal:SetTextColor(1, 0.84, 0)
        honorF:Show()
    end

    local charCount = content:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
    charCount:SetPoint("TOPLEFT", totalHeader, "BOTTOMLEFT", 0, -2)
    charCount:SetText(#snaps .. " character" .. (#snaps ~= 1 and "s" or "") .. " tracked")
    charCount:SetTextColor(C.textDesc[1], C.textDesc[2], C.textDesc[3])

    -- Played Time by Class — stacked bar (only if tracking enabled)
    local lastBeforeHighlights = charCount
    if trackPlayed then
    local barHeader = ns.AddSectionHeader(content, charCount, "Played Time by Class", true)

    local barCard = ns.CreateSectionCard(content)
    barCard:SetPoint("TOPLEFT", barHeader, "TOPLEFT", -8, 8)

    -- Build sorted class data
    local classData = {}
    for _, classFile in ipairs(CLASS_ORDER) do
        if classTimes[classFile] and classTimes[classFile] > 0 then
            classData[#classData + 1] = {
                classFile = classFile,
                time = classTimes[classFile],
                pct = totalPlayed > 0 and (classTimes[classFile] / totalPlayed) or 0,
            }
        end
    end
    table.sort(classData, function(a, b) return a.time > b.time end)

    -- Stacked bar
    local stackBar = CreateFrame("Frame", nil, content, "BackdropTemplate")
    stackBar:SetHeight(20)
    stackBar:SetPoint("TOPLEFT", barHeader, "BOTTOMLEFT", 4, -8)
    stackBar:SetPoint("RIGHT", content, "RIGHT", -20, 0)
    stackBar:SetBackdrop(ns.BACKDROP_CONTROL)
    stackBar:SetBackdropColor(C.bgControl[1], C.bgControl[2], C.bgControl[3], C.bgControl[4])
    stackBar:SetBackdropBorderColor(C.border[1], C.border[2], C.border[3], C.border[4])

    -- Create segments after layout
    C_Timer.After(0, function()
        local barW = stackBar:GetWidth() - 2
        if barW <= 0 then return end
        local xOff = 1
        for _, entry in ipairs(classData) do
            local cr, cg, cb = GetClassColor(entry.classFile)
            local segW = math.max(1, math.floor(barW * entry.pct))
            local seg = stackBar:CreateTexture(nil, "ARTWORK")
            seg:SetPoint("TOPLEFT", xOff, -1)
            seg:SetSize(segW, 18)
            seg:SetColorTexture(cr, cg, cb, 0.8)
            xOff = xOff + segW
        end
    end)

    -- Legend rows
    local lastLegend = stackBar
    for i, entry in ipairs(classData) do
        local row = CreateFrame("Frame", nil, content)
        row:SetHeight(20)
        row:SetPoint("TOPLEFT", lastLegend, "BOTTOMLEFT", i == 1 and 0 or 0, -4)
        row:SetPoint("RIGHT", content, "RIGHT", -20, 0)

        local cIcon = row:CreateTexture(nil, "ARTWORK")
        cIcon:SetSize(16, 16)
        cIcon:SetPoint("LEFT", 0, 0)
        cIcon:SetTexture(CLASS_ICONS[entry.classFile] or "")
        local mask = row:CreateMaskTexture()
        mask:SetSize(16, 16)
        mask:SetPoint("CENTER", cIcon, "CENTER")
        mask:SetTexture("Interface\\CHARACTERFRAME\\TempPortraitAlphaMask",
            "CLAMPTOBLACKADDITIVE", "CLAMPTOBLACKADDITIVE")
        cIcon:AddMaskTexture(mask)

        local cName = row:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        cName:SetPoint("LEFT", cIcon, "RIGHT", 6, 0)
        local cr, cg, cb = GetClassColor(entry.classFile)
        cName:SetTextColor(cr, cg, cb)
        -- Find characters of this class
        local classChars = {}
        for _, snap in ipairs(snaps) do
            if snap.classFile == entry.classFile then
                classChars[#classChars + 1] = snap.name
            end
        end
        cName:SetText(table.concat(classChars, ", "))

        local cTime = row:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        cTime:SetPoint("RIGHT", 0, 0)
        cTime:SetText(FormatPlayedTime(entry.time) .. string.format("  (%.0f%%)", entry.pct * 100))
        cTime:SetTextColor(C.textLabel[1], C.textLabel[2], C.textLabel[3])

        lastLegend = row
    end

    barCard:SetPoint("BOTTOM", lastLegend, "BOTTOM", 0, -8)
    barCard:SetPoint("RIGHT", content, "RIGHT", -8, 0)
    lastBeforeHighlights = barCard
    end -- trackPlayed

    -- Superlatives
    local supHeader = ns.AddSectionHeader(content, lastBeforeHighlights, "Account Highlights", false)

    local supCard = ns.CreateSectionCard(content)
    supCard:SetPoint("TOPLEFT", supHeader, "TOPLEFT", -8, 8)

    local superlatives = {
        { title = "Most Played", data = mostPlayed, fmt = function(s) return FormatPlayedTime(s.val) end },
        { title = "Highest iLvl", data = highestIlvl, fmt = function(s) return string.format("%.0f", s.val) end },
        { title = "Most HKs", data = mostHKs, fmt = function(s) return FormatNumber(s.val) end },
        { title = "Most Deaths", data = mostDeaths, fmt = function(s) return FormatNumber(s.val) end },
        { title = "Most Quests", data = mostQuests, fmt = function(s) return FormatNumber(s.val) end },
        { title = "Best Duelist", data = bestDuelist, fmt = function(s)
            local stats = s.snap and s.snap.stats or {}
            return (stats.duelsWon or 0) .. "W / " .. (stats.duelsLost or 0) .. "L"
        end },
        { title = "Most Arenas", data = mostArenas, fmt = function(s)
            local stats = s.snap and s.snap.stats or {}
            local played = stats.arenasPlayed or 0
            local won = stats.arenasWon or 0
            if played >= 100 then
                return FormatNumber(played) .. " (" .. string.format("%.1f%%", (won / played) * 100) .. " win)"
            end
            return FormatNumber(played)
        end },
        { title = "Best M+ Rating", data = bestMplusRating, fmt = function(s) return FormatNumber(s.val) end },
        { title = "Highest M+ Key", data = bestMplusKey, fmt = function(s) return "+" .. s.val end },
    }

    local lastSup = supHeader
    for i, sup in ipairs(superlatives) do
        if sup.data.snap then
            local row = CreateSuperlativeRow(content)
            row:SetPoint("TOPLEFT", lastSup, "BOTTOMLEFT", i == 1 and 4 or 0, i == 1 and -8 or -2)
            row:SetPoint("RIGHT", content, "RIGHT", -20, 0)
            row:SetData(sup.title, sup.data.snap.name, sup.data.snap.classFile, sup.fmt(sup.data))
            lastSup = row
        end
    end

    supCard:SetPoint("BOTTOM", lastSup, "BOTTOM", 0, -8)
    supCard:SetPoint("RIGHT", content, "RIGHT", -8, 0)

    -- Best Ratings across all characters
    local ratHeader = ns.AddSectionHeader(content, supCard, "Best Ratings", false)

    local ratCard = ns.CreateSectionCard(content)
    ratCard:SetPoint("TOPLEFT", ratHeader, "TOPLEFT", -8, 8)

    local lastRat = ratHeader
    local hasAnyRating = false
    for _, bracket in ipairs(BRACKETS) do
        -- Skip RBG from display (still tracked in data)
        if bracket.idx ~= 4 then
            local best = bestRatings[bracket.idx]
            if best.rating > 0 and best.snap then
                hasAnyRating = true
                local row = CreateSuperlativeRow(content)
                row:SetPoint("TOPLEFT", lastRat, "BOTTOMLEFT", lastRat == ratHeader and 4 or 0, lastRat == ratHeader and -8 or -2)
                row:SetPoint("RIGHT", content, "RIGHT", -20, 0)
                row:SetData(bracket.label, best.snap.name, best.snap.classFile, FormatNumber(best.rating))
                lastRat = row
            end
        end
    end

    -- M+ Rating
    if bestMplusRating.snap and bestMplusRating.val > 0 then
        hasAnyRating = true
        local row = CreateSuperlativeRow(content)
        row:SetPoint("TOPLEFT", lastRat, "BOTTOMLEFT", lastRat == ratHeader and 4 or 0, lastRat == ratHeader and -8 or -2)
        row:SetPoint("RIGHT", content, "RIGHT", -20, 0)
        row:SetData("M+ Rating", bestMplusRating.snap.name, bestMplusRating.snap.classFile, FormatNumber(bestMplusRating.val))
        lastRat = row
    end

    -- Highest M+ Key
    if bestMplusKey.snap and bestMplusKey.val > 0 then
        hasAnyRating = true
        local row = CreateSuperlativeRow(content)
        row:SetPoint("TOPLEFT", lastRat, "BOTTOMLEFT", lastRat == ratHeader and 4 or 0, lastRat == ratHeader and -8 or -2)
        row:SetPoint("RIGHT", content, "RIGHT", -20, 0)
        row:SetData("Highest Key", bestMplusKey.snap.name, bestMplusKey.snap.classFile, "+" .. bestMplusKey.val)
        lastRat = row
    end

    if not hasAnyRating then
        local noRat = content:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        noRat:SetPoint("TOPLEFT", ratHeader, "BOTTOMLEFT", 4, -8)
        noRat:SetText("No rated data yet.")
        noRat:SetTextColor(C.textDesc[1], C.textDesc[2], C.textDesc[3])
        lastRat = noRat
    end

    ratCard:SetPoint("BOTTOM", lastRat, "BOTTOM", 0, -8)
    ratCard:SetPoint("RIGHT", content, "RIGHT", -8, 0)

    -- Account Totals (grouped with zebra stripes)
    local totHeader = ns.AddSectionHeader(content, ratCard, "Account Totals", false)

    local totCard = ns.CreateSectionCard(content)
    totCard:SetPoint("TOPLEFT", totHeader, "TOPLEFT", -8, 8)

    local generalRows = {}
    if trackPlayed and totalPlayed > 0 then
        generalRows[#generalRows + 1] = { label = "Total Played", value = FormatPlayedTime(totalPlayed) }
    end
    generalRows[#generalRows + 1] = { label = "Achievement Points", value = FormatNumber(totalAchieve) }

    local totGroups = {
        { header = "General", rows = generalRows },
        { header = "Combat", rows = {
            { label = "Honorable Kills",     value = FormatNumber(totalHKs) },
            { label = "Total Arenas",        value = FormatNumber(totalArenas) },
            { label = "Creatures Killed",    value = FormatNumber(totalKills) },
            { label = "Total Deaths",        value = FormatNumber(totalDeaths) },
        }},
        { header = "Economy", rows = {
            { label = "Total Gold",          value = FormatNumber(totalGold) .. "g" },
            { label = "Quests Completed",    value = FormatNumber(totalQuests) },
        }},
    }

    local lastTot = totHeader
    local isFirstTotGroup = true
    for _, group in ipairs(totGroups) do
        -- Sub-header
        local subH = content:CreateFontString(nil, "ARTWORK", "GameFontNormal")
        if isFirstTotGroup then
            subH:SetPoint("TOPLEFT", lastTot, "BOTTOMLEFT", 4, -10)
            isFirstTotGroup = false
        else
            subH:SetPoint("TOPLEFT", lastTot, "BOTTOMLEFT", 0, -12)
        end
        subH:SetText(group.header)
        subH:SetTextColor(1, 1, 1)
        lastTot = subH

        for ri, tot in ipairs(group.rows) do
            local row = CreateFrame("Frame", nil, content)
            row:SetHeight(18)
            row:SetPoint("TOPLEFT", lastTot, "BOTTOMLEFT", 0, -2)
            row:SetPoint("RIGHT", content, "RIGHT", -20, 0)

            -- Zebra stripe
            if ri % 2 == 1 then
                local stripe = row:CreateTexture(nil, "BACKGROUND")
                stripe:SetAllPoints()
                stripe:SetColorTexture(1, 1, 1, 0.03)
            end

            local lbl = row:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
            lbl:SetPoint("LEFT", 4, 0)
            lbl:SetText(tot.label)
            lbl:SetTextColor(C.textLabel[1], C.textLabel[2], C.textLabel[3])

            local val = row:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
            val:SetPoint("RIGHT", 0, 0)
            val:SetText(tot.value)
            val:SetTextColor(CYAN.r, CYAN.g, CYAN.b)

            lastTot = row
        end
    end

    totCard:SetPoint("BOTTOM", lastTot, "BOTTOM", 0, -10)
    totCard:SetPoint("RIGHT", content, "RIGHT", -8, 0)

    -- Collections Section (account-wide, uses max across characters)
    local collHeader = ns.AddSectionHeader(content, totCard, "Collections", false)
    local collCard = ns.CreateSectionCard(content)
    collCard:SetPoint("TOPLEFT", collHeader, "TOPLEFT", -8, 8)

    local collMax = math.max(maxMounts, maxPets, maxToys, maxDecor, 1)

    local mountBar = CreateCollectionBar(content)
    mountBar:SetPoint("TOPLEFT", collHeader, "BOTTOMLEFT", 4, -8)
    mountBar:SetPoint("RIGHT", content, "RIGHT", -20, 0)
    mountBar:SetData("Mounts", maxMounts,
        "Interface\\Icons\\Ability_Mount_RidingHorse", {0.2, 0.6, 1}, collMax)

    local petBar = CreateCollectionBar(content)
    petBar:SetPoint("TOPLEFT", mountBar, "BOTTOMLEFT", 0, -4)
    petBar:SetPoint("RIGHT", content, "RIGHT", -20, 0)
    petBar:SetData("Pets", maxPets,
        "Interface\\Icons\\INV_Box_PetCarrier_01", {CYAN.r, CYAN.g, CYAN.b}, collMax)

    local toyBar = CreateCollectionBar(content)
    toyBar:SetPoint("TOPLEFT", petBar, "BOTTOMLEFT", 0, -4)
    toyBar:SetPoint("RIGHT", content, "RIGHT", -20, 0)
    toyBar:SetData("Toys", maxToys,
        "Interface\\Icons\\INV_Misc_Toy_02", {0.9, 0.6, 0.2}, collMax)

    local decorBar = CreateCollectionBar(content)
    decorBar:SetPoint("TOPLEFT", toyBar, "BOTTOMLEFT", 0, -4)
    decorBar:SetPoint("RIGHT", content, "RIGHT", -20, 0)
    decorBar:SetData("Decor", maxDecor,
        "Interface\\Icons\\INV_Misc_Key_14", {0.7, 0.5, 0.9}, collMax)

    collCard:SetPoint("BOTTOM", decorBar, "BOTTOM", 0, -8)
    collCard:SetPoint("RIGHT", content, "RIGHT", -8, 0)

    -- Account Professions (aggregated per-expansion tiers across all characters)
    local lastBeforeRoster = collCard

    -- Helper: extract expansion prefix from tier name by stripping base profession suffix
    local function GetExpPrefix(tierName, baseName)
        if tierName == baseName then return "Classic" end
        local suffix = " " .. baseName
        if #tierName > #suffix and tierName:sub(-#suffix) == suffix then
            return tierName:sub(1, -(#suffix + 1))
        end
        return tierName
    end

    -- Helper: sort order for an expansion prefix (lower = newer)
    local function ExpSortKey(prefix)
        return EXPANSION_SORT[prefix] or 99
    end

    -- Aggregate: profAgg[baseName] = { icon, tiers = { [tierName] = { skill, max, charName, classFile } } }
    local profAgg = {}
    local profNameOrder = {}
    for _, snap in ipairs(snaps) do
        if snap.professions then
            for _, prof in ipairs(snap.professions) do
                if not profAgg[prof.name] then
                    profAgg[prof.name] = { icon = prof.icon, tiers = {} }
                    profNameOrder[#profNameOrder + 1] = prof.name
                end
                local agg = profAgg[prof.name]
                if prof.tiers then
                    for _, tier in ipairs(prof.tiers) do
                        local existing = agg.tiers[tier.name]
                        if not existing or tier.skill > existing.skill then
                            agg.tiers[tier.name] = {
                                name = tier.name, skill = tier.skill, max = tier.max,
                                charName = snap.name, classFile = snap.classFile,
                            }
                        end
                    end
                end
                -- Always track overall (current expansion from GetProfessionInfo)
                if prof.skill and prof.skill > 0 then
                    local existingOvr = agg.overall
                    if not existingOvr or prof.skill > existingOvr.skill then
                        agg.overall = {
                            skill = prof.skill, max = prof.max,
                            charName = snap.name, classFile = snap.classFile,
                        }
                    end
                end
            end
        end
    end

    if #profNameOrder > 0 then
        local profHeader = ns.AddSectionHeader(content, collCard, "Professions", false)
        local profCard = ns.CreateSectionCard(content)
        profCard:SetPoint("TOPLEFT", profHeader, "TOPLEFT", -8, 8)

        -- Use a container inside profCard so expand/collapse triggers relayout
        local profContainer = CreateFrame("Frame", nil, content)
        profContainer:SetPoint("TOPLEFT", profHeader, "BOTTOMLEFT", 4, -8)
        profContainer:SetPoint("RIGHT", content, "RIGHT", -20, 0)

        local LEFT_TIER = 24   -- indented tier rows relative to container

        -- Build all profession blocks
        local allProfBlocks = {}
        for pi, baseName in ipairs(profNameOrder) do
            local agg = profAgg[baseName]

            -- Sort tiers by expansion (newest first)
            local sortedTiers = {}
            for _, tier in pairs(agg.tiers) do
                sortedTiers[#sortedTiers + 1] = tier
            end
            table.sort(sortedTiers, function(a, b)
                return ExpSortKey(GetExpPrefix(a.name, baseName)) < ExpSortKey(GetExpPrefix(b.name, baseName))
            end)

            -- Determine the "current" tier (newest, first in sorted list) and extra tiers
            local currentTier = sortedTiers[1]
            local extraTiers = {}
            for i = 2, #sortedTiers do
                extraTiers[#extraTiers + 1] = sortedTiers[i]
            end

            -- Profession header row: icon + name + current tier skill + expand arrow
            local hdrRow = CreateFrame("Button", nil, profContainer)
            hdrRow:SetHeight(22)
            hdrRow:SetPoint("LEFT", 0, 0)
            hdrRow:SetPoint("RIGHT", 0, 0)

            local icon = hdrRow:CreateTexture(nil, "ARTWORK")
            icon:SetSize(18, 18)
            icon:SetPoint("LEFT", 0, 0)
            if agg.icon then icon:SetTexture(agg.icon) end

            local hdrLbl = hdrRow:CreateFontString(nil, "ARTWORK", "GameFontNormal")
            hdrLbl:SetPoint("LEFT", icon, "RIGHT", 6, 0)
            hdrLbl:SetText(baseName)
            hdrLbl:SetTextColor(1, 1, 1)

            -- Current expansion skill on the right (prefer overall = active expansion from GetProfessionInfo)
            local displayInfo = agg.overall or (currentTier and {
                skill = currentTier.skill, max = currentTier.max,
                charName = currentTier.charName, classFile = currentTier.classFile,
            })
            if displayInfo then
                -- Character name (class-colored)
                if displayInfo.charName then
                    local charLbl = hdrRow:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
                    charLbl:SetPoint("RIGHT", hdrRow, "RIGHT", -80, 0)
                    charLbl:SetText(displayInfo.charName)
                    local cr, cg, cb = GetClassColor(displayInfo.classFile)
                    charLbl:SetTextColor(cr, cg, cb)
                end

                local skillText = hdrRow:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
                skillText:SetPoint("RIGHT", 0, 0)
                skillText:SetText(displayInfo.skill .. " / " .. displayInfo.max)
                skillText:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
            end

            -- Expand arrow (only if there are extra tiers)
            local arrow = nil
            if #extraTiers > 0 then
                arrow = hdrRow:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
                arrow:SetPoint("LEFT", hdrLbl, "RIGHT", 4, 0)
                arrow:SetText("+")
                arrow:SetTextColor(C.textMuted[1], C.textMuted[2], C.textMuted[3])
            end

            -- Extra tier rows (hidden by default)
            local tierRows = {}
            for ti, tier in ipairs(extraTiers) do
                local prefix = GetExpPrefix(tier.name, baseName)

                local row = CreateFrame("Frame", nil, profContainer)
                row:SetHeight(18)
                row:SetPoint("LEFT", LEFT_TIER, 0)
                row:SetPoint("RIGHT", 0, 0)

                -- Zebra stripe
                if ti % 2 == 1 then
                    local stripe = row:CreateTexture(nil, "BACKGROUND")
                    stripe:SetAllPoints()
                    stripe:SetColorTexture(1, 1, 1, 0.03)
                end

                local expLbl = row:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
                expLbl:SetPoint("LEFT", 0, 0)
                expLbl:SetText(prefix)
                expLbl:SetTextColor(C.textLabel[1], C.textLabel[2], C.textLabel[3])

                -- Character name (class-colored)
                local charLbl = row:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
                charLbl:SetPoint("RIGHT", row, "RIGHT", -80, 0)
                charLbl:SetText(tier.charName)
                local cr, cg, cb = GetClassColor(tier.classFile)
                charLbl:SetTextColor(cr, cg, cb)

                local skillText = row:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
                skillText:SetPoint("RIGHT", 0, 0)
                skillText:SetText(tier.skill .. " / " .. tier.max)
                skillText:SetTextColor(CYAN.r, CYAN.g, CYAN.b)

                row:Hide()
                tierRows[#tierRows + 1] = row
            end

            allProfBlocks[#allProfBlocks + 1] = {
                hdrRow = hdrRow,
                tierRows = tierRows,
                arrow = arrow,
                expanded = false,
            }
        end

        -- Layout function: positions all blocks vertically
        local function RelayoutProfs()
            local yOff = 0
            for _, block in ipairs(allProfBlocks) do
                block.hdrRow:ClearAllPoints()
                block.hdrRow:SetPoint("TOPLEFT", profContainer, "TOPLEFT", 0, -yOff)
                block.hdrRow:SetPoint("RIGHT", profContainer, "RIGHT", 0, 0)
                yOff = yOff + 22 + 4  -- header height + gap

                if block.expanded then
                    for _, row in ipairs(block.tierRows) do
                        row:ClearAllPoints()
                        row:SetPoint("TOPLEFT", profContainer, "TOPLEFT", LEFT_TIER, -yOff)
                        row:SetPoint("RIGHT", profContainer, "RIGHT", 0, 0)
                        row:Show()
                        yOff = yOff + 20  -- row height + gap
                    end
                else
                    for _, row in ipairs(block.tierRows) do
                        row:Hide()
                    end
                end
            end
            profContainer:SetHeight(math.max(1, yOff))
        end

        -- Wire up click handlers for expand/collapse
        for _, block in ipairs(allProfBlocks) do
            if #block.tierRows > 0 then
                block.hdrRow:SetScript("OnClick", function()
                    block.expanded = not block.expanded
                    if block.arrow then
                        block.arrow:SetText(block.expanded and "-" or "+")
                    end
                    RelayoutProfs()
                end)
                block.hdrRow:SetScript("OnEnter", function(self)
                    self:GetFontString() -- no-op, just highlight
                end)
            end
        end

        -- Initial layout
        RelayoutProfs()

        local profHint = content:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        profHint:SetPoint("TOPLEFT", profContainer, "BOTTOMLEFT", 0, -6)
        profHint:SetText("Open the Professions pane on a character to log its profession data.")
        profHint:SetTextColor(C.textMuted[1], C.textMuted[2], C.textMuted[3], 0.7)

        profCard:SetPoint("BOTTOM", profHint, "BOTTOM", 0, -8)
        profCard:SetPoint("RIGHT", content, "RIGHT", -8, 0)
        lastBeforeRoster = profCard
    end

    -- Notable Titles (account-wide, merged from all snapshots)
    local titleSet = {}
    local titleList = {}
    for _, snap in ipairs(snaps) do
        if snap.notableTitles then
            for _, t in ipairs(snap.notableTitles) do
                if not titleSet[t.name] then
                    titleSet[t.name] = true
                    titleList[#titleList + 1] = t.name
                end
            end
        end
    end
    if #titleList > 0 then
        local titlesNHeader = ns.AddSectionHeader(content, lastBeforeRoster, "Notable Titles", false)
        local titlesNCard = ns.CreateSectionCard(content)
        titlesNCard:SetPoint("TOPLEFT", titlesNHeader, "TOPLEFT", -8, 8)

        local titlesText = content:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        titlesText:SetPoint("TOPLEFT", titlesNHeader, "BOTTOMLEFT", 4, -8)
        titlesText:SetPoint("RIGHT", content, "RIGHT", -20, 0)
        titlesText:SetText(table.concat(titleList, "  |  "))
        titlesText:SetTextColor(1, 0.84, 0)  -- gold
        titlesText:SetJustifyH("LEFT")
        titlesText:SetWordWrap(true)

        local countText = content:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        countText:SetPoint("TOPLEFT", titlesText, "BOTTOMLEFT", 0, -4)
        countText:SetText(#titleList .. " rare title" .. (#titleList ~= 1 and "s" or "") .. " earned")
        countText:SetTextColor(C.textMuted[1], C.textMuted[2], C.textMuted[3])

        titlesNCard:SetPoint("BOTTOM", countText, "BOTTOM", 0, -8)
        titlesNCard:SetPoint("RIGHT", content, "RIGHT", -8, 0)
        lastBeforeRoster = titlesNCard
    end

    -- Character Roster
    local rosterHeader = ns.AddSectionHeader(content, lastBeforeRoster, "Characters", false)

    local rosterCard = ns.CreateSectionCard(content)
    rosterCard:SetPoint("TOPLEFT", rosterHeader, "TOPLEFT", -8, 8)

    -- Column headers
    local colHeader = CreateFrame("Frame", nil, content)
    colHeader:SetHeight(18)
    colHeader:SetPoint("TOPLEFT", rosterHeader, "BOTTOMLEFT", 4, -8)
    colHeader:SetPoint("RIGHT", content, "RIGHT", -20, 0)

    local hidePlayed = SKToolsDB and SKToolsDB.trackPlayedTime == false
    for _, col in ipairs(ROSTER_COLS) do
        if not (col.key == "played" and hidePlayed) then
            local txt = colHeader:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
            if col.anchorRight then
                txt:SetPoint("RIGHT", colHeader, "RIGHT", col.anchorRight, 0)
            else
                txt:SetPoint("LEFT", colHeader, "LEFT", col.x, 0)
            end
            txt:SetWidth(col.width)
            txt:SetJustifyH(col.justify)
            txt:SetText(col.label)
            txt:SetTextColor(C.textMuted[1], C.textMuted[2], C.textMuted[3])
        end
    end

    local lastRoster = colHeader
    for _, snap in ipairs(snaps) do
        local row = CreateRosterRow(content)
        row:SetPoint("TOPLEFT", lastRoster, "BOTTOMLEFT", 0, -2)
        row:SetPoint("RIGHT", content, "RIGHT", -20, 0)
        row:SetSnapshot(snap)
        lastRoster = row
    end

    local rosterHint = content:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
    rosterHint:SetPoint("TOPLEFT", lastRoster, "BOTTOMLEFT", 0, -6)
    rosterHint:SetText("Log into other characters to track them.")
    rosterHint:SetTextColor(C.textMuted[1], C.textMuted[2], C.textMuted[3], 0.7)

    rosterCard:SetPoint("BOTTOM", rosterHint, "BOTTOM", 0, -8)
    rosterCard:SetPoint("RIGHT", content, "RIGHT", -8, 0)

    return rosterCard
end

-----------------------------
-- Shared Profile Content Builder
-----------------------------
local function BuildProfileContent(content, tabAnchor)
    -- View toggle buttons
    local charBtn = ns.CreateThemedButton(content, "Character", 90, 24, "primary")
    charBtn:SetPoint("TOPLEFT", tabAnchor, "BOTTOMLEFT", 0, -12)

    local acctBtn = ns.CreateThemedButton(content, "Account", 90, 24, "secondary")
    acctBtn:SetPoint("LEFT", charBtn, "RIGHT", 6, 0)

    -- Refresh icon button (top-right, subtle)
    local refreshBtn = CreateFrame("Button", nil, content)
    refreshBtn:SetSize(22, 22)
    refreshBtn:SetPoint("TOPLEFT", acctBtn, "TOPRIGHT", 8, 0)
    local refreshIcon = refreshBtn:CreateTexture(nil, "ARTWORK")
    refreshIcon:SetAllPoints()
    refreshIcon:SetTexture("Interface\\Buttons\\UI-RefreshButton")
    refreshIcon:SetVertexColor(0.5, 0.5, 0.55)
    refreshBtn:SetScript("OnEnter", function()
        refreshIcon:SetVertexColor(CYAN.r, CYAN.g, CYAN.b)
        GameTooltip:SetOwner(refreshBtn, "ANCHOR_RIGHT")
        GameTooltip:AddLine("Refresh profile data", 1, 1, 1)
        GameTooltip:Show()
    end)
    refreshBtn:SetScript("OnLeave", function()
        refreshIcon:SetVertexColor(0.5, 0.5, 0.55)
        GameTooltip:Hide()
    end)

    -- Character selector dropdown
    local selectedCharKey = GetCharKey()  -- default to current character

    local charDropdown = CreateFrame("Button", nil, content, "BackdropTemplate")
    charDropdown:SetSize(160, 24)
    charDropdown:SetPoint("LEFT", refreshBtn, "RIGHT", 8, 0)
    charDropdown:SetBackdrop(ns.BACKDROP_CONTROL)
    charDropdown:SetBackdropColor(0.1, 0.1, 0.12, 0.8)
    charDropdown:SetBackdropBorderColor(0.3, 0.3, 0.35, 0.6)

    local ddText = charDropdown:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
    ddText:SetPoint("LEFT", 8, 0)
    ddText:SetPoint("RIGHT", charDropdown, "RIGHT", -16, 0)
    ddText:SetJustifyH("LEFT")
    charDropdown.text = ddText

    local ddArrow = charDropdown:CreateTexture(nil, "OVERLAY")
    ddArrow:SetSize(10, 10)
    ddArrow:SetPoint("RIGHT", charDropdown, "RIGHT", -4, 0)
    ddArrow:SetAtlas("arrow-down-minor")
    ddArrow:SetVertexColor(0.6, 0.6, 0.6)

    charDropdown:SetScript("OnEnter", function(self)
        self:SetBackdropColor(0.15, 0.15, 0.18, 0.9)
        ddText:SetTextColor(1, 1, 1)
    end)
    charDropdown:SetScript("OnLeave", function(self)
        if not ddMenu or not ddMenu:IsShown() then
            self:SetBackdropColor(0.1, 0.1, 0.12, 0.8)
            local isCurrentChar = (selectedCharKey == GetCharKey())
            ddText:SetTextColor(isCurrentChar and 0.7 or CYAN.r, isCurrentChar and 0.7 or CYAN.g, isCurrentChar and 0.7 or CYAN.b)
        end
    end)

    local function UpdateDropdownText()
        local snap = SKToolsProfileDB and SKToolsProfileDB[selectedCharKey]
        local name = snap and snap.name or selectedCharKey:match("^([^%-]+)") or selectedCharKey
        ddText:SetText(name)
        local isCurrentChar = (selectedCharKey == GetCharKey())
        if isCurrentChar then
            ddText:SetTextColor(0.7, 0.7, 0.7)
            charDropdown:SetBackdropColor(0.1, 0.1, 0.12, 0.8)
            charDropdown:SetBackdropBorderColor(0.3, 0.3, 0.35, 0.6)
        else
            ddText:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
            charDropdown:SetBackdropColor(CYAN.r * 0.15, CYAN.g * 0.15, CYAN.b * 0.15, 1)
            charDropdown:SetBackdropBorderColor(CYAN.r, CYAN.g, CYAN.b, 0.5)
        end
    end

    -- Dropdown menu
    local ddMenu = CreateFrame("Frame", nil, charDropdown, "BackdropTemplate")
    ddMenu:SetBackdrop(ns.BACKDROP_CONTROL)
    ddMenu:SetBackdropColor(0.08, 0.08, 0.1, 0.95)
    ddMenu:SetBackdropBorderColor(0.3, 0.3, 0.35, 0.8)
    ddMenu:SetFrameStrata("DIALOG")
    ddMenu:Hide()

    local ddMenuItems = {}

    local function RefreshDropdownMenu(showViewFn)
        for _, item in ipairs(ddMenuItems) do item:Hide() end
        ddMenuItems = {}

        local snaps = GetSortedSnapshots()
        if #snaps <= 1 then
            charDropdown:Hide()
            return
        end
        charDropdown:Show()

        local itemH = 22
        ddMenu:SetSize(charDropdown:GetWidth(), 4 + #snaps * itemH + 4)
        ddMenu:SetPoint("TOP", charDropdown, "BOTTOM", 0, -2)

        for i, snap in ipairs(snaps) do
            local item = CreateFrame("Button", nil, ddMenu)
            item:SetSize(charDropdown:GetWidth() - 8, itemH)
            item:SetPoint("TOPLEFT", ddMenu, "TOPLEFT", 4, -4 - (i - 1) * itemH)

            local itemIcon = item:CreateTexture(nil, "ARTWORK")
            itemIcon:SetSize(16, 16)
            itemIcon:SetPoint("LEFT", 2, 0)
            itemIcon:SetTexture(CLASS_ICONS[snap.classFile] or "")
            local mask = item:CreateMaskTexture()
            mask:SetSize(14, 14)
            mask:SetPoint("CENTER", itemIcon, "CENTER")
            mask:SetTexture("Interface\\CHARACTERFRAME\\TempPortraitAlphaMask",
                "CLAMPTOBLACKADDITIVE", "CLAMPTOBLACKADDITIVE")
            itemIcon:AddMaskTexture(mask)

            local itemText = item:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
            itemText:SetPoint("LEFT", itemIcon, "RIGHT", 6, 0)

            local isActive = (snap._key == selectedCharKey)
            local cr, cg, cb = GetClassColor(snap.classFile)
            if isActive then
                itemText:SetText(snap.name)
                itemText:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
            else
                itemText:SetText(snap.name)
                itemText:SetTextColor(cr, cg, cb)
            end

            item:SetScript("OnEnter", function()
                if not isActive then itemText:SetTextColor(1, 1, 1) end
            end)
            item:SetScript("OnLeave", function()
                if not isActive then itemText:SetTextColor(cr, cg, cb) end
            end)
            item:SetScript("OnClick", function()
                selectedCharKey = snap._key
                ddMenu:Hide()
                UpdateDropdownText()
                if showViewFn then showViewFn("character") end
            end)

            ddMenuItems[#ddMenuItems + 1] = item
        end
    end

    charDropdown:SetScript("OnClick", function()
        if ddMenu:IsShown() then
            ddMenu:Hide()
        else
            RefreshDropdownMenu(nil)  -- showViewFn set later
            ddMenu:Show()
        end
    end)

    -- Auto-hide dropdown when mouse leaves
    ddMenu:SetScript("OnShow", function()
        ddMenu.hideTimer = 0
        ddMenu:SetScript("OnUpdate", function(self, dt)
            if not charDropdown:IsMouseOver() and not ddMenu:IsMouseOver() then
                self.hideTimer = (self.hideTimer or 0) + dt
                if self.hideTimer > 0.3 then
                    ddMenu:Hide()
                    self.hideTimer = 0
                end
            else
                self.hideTimer = 0
            end
        end)
    end)
    ddMenu:SetScript("OnHide", function(self)
        self:SetScript("OnUpdate", nil)
    end)

    -- Container for view content (cleared and rebuilt on toggle)
    local viewContainer = CreateFrame("Frame", nil, content)
    viewContainer:SetPoint("TOPLEFT", charBtn, "BOTTOMLEFT", 0, -8)
    viewContainer:SetPoint("RIGHT", content, "RIGHT", 0, 0)
    viewContainer:SetHeight(800)

    local currentView = "character"
    local viewChildren = {}

    local function ClearView()
        for _, child in ipairs(viewChildren) do
            child:Hide()
            child:SetParent(nil)
        end
        wipe(viewChildren)
        -- Clear all child frames/fontstrings from viewContainer
        for _, region in ipairs({viewContainer:GetRegions()}) do
            region:Hide()
            region:SetParent(nil)
        end
        for _, child in ipairs({viewContainer:GetChildren()}) do
            child:Hide()
            child:SetParent(nil)
        end
    end

    -- Active/inactive button styles
    local activeBg     = { 0, CYAN.g * 0.15, CYAN.b * 0.15, 1.0 }
    local activeBorder = { CYAN.r, CYAN.g, CYAN.b, 0.35 }
    local activeText   = { CYAN.r, CYAN.g, CYAN.b }
    local inactiveBg     = { 0.14, 0.14, 0.17, 1.0 }
    local inactiveBorder = { 0.24, 0.24, 0.28, 0.40 }
    local inactiveText   = { C.textLabel[1], C.textLabel[2], C.textLabel[3] }

    local function SetButtonActive(btn, active)
        if active then
            btn:SetBackdropColor(unpack(activeBg))
            btn:SetBackdropBorderColor(unpack(activeBorder))
            btn.label:SetTextColor(unpack(activeText))
            btn:SetScript("OnEnter", nil)
            btn:SetScript("OnLeave", nil)
        else
            btn:SetBackdropColor(unpack(inactiveBg))
            btn:SetBackdropBorderColor(unpack(inactiveBorder))
            btn.label:SetTextColor(unpack(inactiveText))
            btn:SetScript("OnEnter", function(self)
                self:SetBackdropColor(0.18, 0.18, 0.21, 1.0)
                self:SetBackdropBorderColor(0.30, 0.30, 0.35, 0.60)
                self.label:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
            end)
            btn:SetScript("OnLeave", function(self)
                self:SetBackdropColor(unpack(inactiveBg))
                self:SetBackdropBorderColor(unpack(inactiveBorder))
                self.label:SetTextColor(unpack(inactiveText))
            end)
        end
    end

    local function ShowView(view)
        ClearView()
        currentView = view
        ddMenu:Hide()

        -- Update button styles
        SetButtonActive(charBtn, view == "character")
        SetButtonActive(acctBtn, view == "account")

        -- Show/hide character dropdown
        if view == "character" then
            local snaps = GetSortedSnapshots()
            if #snaps > 1 then
                charDropdown:Show()
                UpdateDropdownText()
            else
                charDropdown:Hide()
            end
        else
            charDropdown:Hide()
        end

        local bottomElement
        if view == "character" then
            local snap = SKToolsProfileDB and SKToolsProfileDB[selectedCharKey]
            if snap then
                bottomElement = BuildCharacterView(viewContainer, snap)
            else
                local noData = viewContainer:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
                noData:SetPoint("TOPLEFT", 4, -8)
                noData:SetText("No data for this character yet. Please /reload or relog.")
                noData:SetTextColor(C.textDesc[1], C.textDesc[2], C.textDesc[3])
                bottomElement = noData
            end
        else
            local snaps = GetSortedSnapshots()
            bottomElement = BuildAccountView(viewContainer, snaps)
        end

        -- Size the container to fit
        if bottomElement then
            C_Timer.After(0, function()
                local bottom = bottomElement:GetBottom()
                local top = viewContainer:GetTop()
                if bottom and top then
                    viewContainer:SetHeight(top - bottom + 20)
                end
            end)
        end
    end

    -- Wire dropdown's showViewFn now that ShowView is defined
    charDropdown:SetScript("OnClick", function()
        if ddMenu:IsShown() then
            ddMenu:Hide()
        else
            RefreshDropdownMenu(ShowView)
            ddMenu:Show()
        end
    end)

    charBtn:SetScript("OnClick", function()
        selectedCharKey = GetCharKey()  -- reset to current char when switching back
        ShowView("character")
    end)
    acctBtn:SetScript("OnClick", function() ShowView("account") end)

    refreshBtn:SetScript("OnClick", function()
        CollectCharacterData()
        profileFrame:RegisterEvent("TIME_PLAYED_MSG")
        RequestTimePlayed()
        C_Timer.After(1, function()
            ShowView(currentView)
        end)
    end)

    return ShowView, viewContainer
end

-----------------------------
-- Standalone Profile Window
-----------------------------
local profileWindow = nil

local function CreateProfileWindow()
    if profileWindow then return end

    local CC = ns.COLORS

    local f = CreateFrame("Frame", "SKToolsProfileFrame", UIParent, "BackdropTemplate")
    f:SetSize(700, 600)
    f:SetPoint("CENTER")
    f:SetMovable(true)
    f:EnableMouse(true)
    f:SetClampedToScreen(true)
    f:SetFrameStrata("DIALOG")
    f:SetBackdrop(ns.BACKDROP_PANEL)
    f:SetBackdropColor(CC.bgFrame[1], CC.bgFrame[2], CC.bgFrame[3], CC.bgFrame[4])
    f:SetBackdropBorderColor(CC.border[1], CC.border[2], CC.border[3], 0.80)
    f:Hide()
    tinsert(UISpecialFrames, "SKToolsProfileFrame")

    ns.CreateDropShadow(f, 8)
    ns.CreateCyanAccentLine(f)

    -- ESC to close
    f:SetScript("OnKeyDown", function(self, key)
        if key == "ESCAPE" then
            pcall(self.SetPropagateKeyboardInput, self, false)
            self:Hide()
        else
            pcall(self.SetPropagateKeyboardInput, self, true)
        end
    end)
    pcall(f.EnableKeyboard, f, true)

    -- Draggable
    f:RegisterForDrag("LeftButton")
    f:SetScript("OnDragStart", f.StartMoving)
    f:SetScript("OnDragStop", f.StopMovingOrSizing)

    -- Close button
    local closeBtn = CreateFrame("Button", nil, f, "BackdropTemplate")
    closeBtn:SetSize(24, 24)
    closeBtn:SetPoint("TOPRIGHT", -6, -6)
    closeBtn:SetBackdrop(ns.BACKDROP_CONTROL)
    closeBtn:SetBackdropColor(0, 0, 0, 0)
    closeBtn:SetBackdropBorderColor(0, 0, 0, 0)
    local closeLbl = closeBtn:CreateFontString(nil, "OVERLAY", "GameFontNormal")
    closeLbl:SetPoint("CENTER", 0, 0)
    closeLbl:SetText("x")
    closeLbl:SetTextColor(CC.textMuted[1], CC.textMuted[2], CC.textMuted[3])
    closeBtn:SetScript("OnClick", function() f:Hide() end)
    closeBtn:SetScript("OnEnter", function(self)
        self:SetBackdropColor(0.60, 0.15, 0.15, 0.6)
        self:SetBackdropBorderColor(0.60, 0.20, 0.20, 0.5)
        closeLbl:SetTextColor(1, 1, 1)
    end)
    closeBtn:SetScript("OnLeave", function(self)
        self:SetBackdropColor(0, 0, 0, 0)
        self:SetBackdropBorderColor(0, 0, 0, 0)
        closeLbl:SetTextColor(CC.textMuted[1], CC.textMuted[2], CC.textMuted[3])
    end)

    f:SetToplevel(true)
    f:SetScript("OnShow", function(self) self:Raise() end)

    -- Content area
    local contentArea = CreateFrame("Frame", nil, f)
    contentArea:SetPoint("TOPLEFT", 4, -4)
    contentArea:SetPoint("BOTTOMRIGHT", -4, 4)
    local contentBg = contentArea:CreateTexture(nil, "BACKGROUND")
    contentBg:SetAllPoints()
    contentBg:SetColorTexture(CC.bgContent[1], CC.bgContent[2], CC.bgContent[3], CC.bgContent[4])

    -- Scrollable content
    local sf = CreateFrame("ScrollFrame", nil, contentArea, "UIPanelScrollFrameTemplate")
    sf:SetPoint("TOPLEFT", 0, 0)
    sf:SetPoint("BOTTOMRIGHT", -22, 0)
    local content = CreateFrame("Frame")
    content:SetWidth(660)
    content:SetHeight(1200)
    sf:SetScrollChild(content)
    sf:SetScript("OnSizeChanged", function(self, w) content:SetWidth(w - 4) end)
    ns.StyleScrollBar(sf)
    ns.AttachSmoothScroll(sf, contentArea)

    -- Tab title
    local tabAnchor = ns.CreateTabTitle(content, "Profile", "Character and account statistics.")

    -- Build shared content
    local ShowView, viewContainer = BuildProfileContent(content, tabAnchor)

    f:SetScript("OnShow", function()
        ShowView("character")
    end)

    profileWindow = f
end

function ns.Profile_Toggle()
    CreateProfileWindow()
    if profileWindow:IsShown() then
        profileWindow:Hide()
    else
        profileWindow:Show()
        ns.FadeIn(profileWindow, 0.15)
    end
end

-----------------------------
-- Settings Tab Builders
-----------------------------
function ns.BuildProfileSettings(content, anchor)
    -- Open Profile button
    local openBtn = ns.CreateThemedButton(content, "Open Profile Window", 160, 26, "primary")
    openBtn:SetPoint("TOPLEFT", anchor, "BOTTOMLEFT", 0, -16)
    openBtn:SetScript("OnClick", function()
        ns.Profile_Toggle()
    end)

    -- Inline profile view
    local ShowView, viewContainer = BuildProfileContent(content, openBtn)

    -- Show character view when tab becomes visible
    local settingsShown = false
    content:GetParent():HookScript("OnShow", function()
        if not settingsShown then
            settingsShown = true
            ShowView("character")
        else
            ShowView("character")
        end
    end)

    return viewContainer
end

function ns.BuildProfileCombatSettings(content, csSyncControls)
    local anchor = ns.CreateTabTitle(content, "Profile", "Character and account statistics.")

    -- Open Profile button
    local openBtn = ns.CreateThemedButton(content, "Open Profile Window", 160, 26, "primary")
    openBtn:SetPoint("TOPLEFT", anchor, "BOTTOMLEFT", 0, -16)
    openBtn:SetScript("OnClick", function()
        ns.Profile_Toggle()
    end)

    -- Inline profile view
    local ShowView, viewContainer = BuildProfileContent(content, openBtn)

    content:GetParent():HookScript("OnShow", function()
        ShowView("character")
    end)
end
