-- SKTools Battle Intel
-- Real-time battleground intelligence: capture timers, score prediction, flag tracking

local _, ns = ...

-----------------------------
-- BG Configuration
-----------------------------
local BG_MAP_IDS = {
    -- Resource/Node BGs (maxScore read dynamically from widget)
    [93]   = { name = "Arathi Basin",       type = "resource", capTime = 60, widgetID = 1671 },
    [529]  = { name = "Arathi Basin",       type = "resource", capTime = 60, widgetID = 1671 },
    [837]  = { name = "Arathi Basin",       type = "resource", capTime = 60, widgetID = 1671 },  -- Winter
    [1366] = { name = "Arathi Basin",       type = "resource", capTime = 60, widgetID = 1671 },
    [1383] = { name = "Arathi Basin",       type = "resource", capTime = 60, widgetID = 1671 },  -- Jagged
    [1681] = { name = "Arathi Basin",       type = "resource", capTime = 60, widgetID = 1671 },
    [2107] = { name = "Arathi Basin",       type = "resource", capTime = 60, widgetID = 1671 },
    [2177] = { name = "Arathi Basin",       type = "resource", capTime = 60, widgetID = 1671 },
    [275]  = { name = "Battle for Gilneas", type = "resource", capTime = 60, widgetID = 1671 },
    [761]  = { name = "Battle for Gilneas", type = "resource", capTime = 60, widgetID = 1671 },
    [1334] = { name = "Battle for Gilneas", type = "resource", capTime = 60, widgetID = 1671 },
    [519]  = { name = "Deepwind Gorge",     type = "resource", capTime = 60, widgetID = 2074 },
    [1576] = { name = "Deepwind Gorge",     type = "resource", capTime = 60, widgetID = 2074 },
    [2245] = { name = "Deepwind Gorge",     type = "resource", capTime = 60, widgetID = 2074 },
    [2345] = { name = "Deephaul Ravine",    type = "resource", capTime = 0,  widgetID = 5153 },
    [2656] = { name = "Deephaul Ravine",    type = "resource", capTime = 0,  widgetID = 5153 },
    -- Epic BGs (score prediction + capture timers, no gate HP)
    [30]   = { name = "Alterac Valley",    type = "resource", capTime = 242, widgetID = 1671 },
    [91]   = { name = "Alterac Valley",    type = "resource", capTime = 242, widgetID = 1671 },
    [2197] = { name = "Alterac Valley",    type = "resource", capTime = 241, widgetID = 1671 },
    [169]  = { name = "Isle of Conquest",   type = "resource", capTime = 61,  widgetID = 1671 },
    [628]  = { name = "Isle of Conquest",   type = "resource", capTime = 61,  widgetID = 1671 },
    [1335] = { name = "Isle of Conquest",   type = "resource", capTime = 61,  widgetID = 1671 },
    -- Score prediction only (no traditional node captures)
    [423]  = { name = "Silvershard Mines",  type = "resource", capTime = 0, widgetID = 1671 },
    [727]  = { name = "Silvershard Mines",  type = "resource", capTime = 0, widgetID = 1671 },
    [417]  = { name = "Temple of Kotmogu",  type = "resource", capTime = 0, widgetID = 1671 },
    [449]  = { name = "Temple of Kotmogu",  type = "resource", capTime = 0, widgetID = 1671 },
    [998]  = { name = "Temple of Kotmogu",  type = "resource", capTime = 0, widgetID = 1671 },
    [1803] = { name = "Seething Shore",     type = "resource", capTime = 0, widgetID = 1671 },
    -- Hybrid (nodes + center flag)
    [112]  = { name = "Eye of the Storm", type = "hybrid", capTime = 60, widgetID = 1671, flagRespawn = 21 },
    [397]  = { name = "Eye of the Storm", type = "hybrid", capTime = 60, widgetID = 1671, flagRespawn = 21 },
    [566]  = { name = "Eye of the Storm", type = "hybrid", capTime = 60, widgetID = 1671, flagRespawn = 21 },
    [968]  = { name = "Eye of the Storm", type = "hybrid", capTime = 60, widgetID = 1671, flagRespawn = 21 },
    -- CTF (no score widget — wins at 3 flag captures, not resource accumulation)
    [92]   = { name = "Warsong Gulch", type = "ctf", flagRespawn = 12 },
    [489]  = { name = "Warsong Gulch", type = "ctf", flagRespawn = 12 },
    [1339] = { name = "Warsong Gulch", type = "ctf", flagRespawn = 12 },
    [1460] = { name = "Warsong Gulch", type = "ctf", flagRespawn = 12 },
    [2106] = { name = "Warsong Gulch", type = "ctf", flagRespawn = 12 },
    [206]  = { name = "Twin Peaks",    type = "ctf", flagRespawn = 12 },
    [726]  = { name = "Twin Peaks",    type = "ctf", flagRespawn = 12 },
}

-----------------------------
-- Constants
-----------------------------
local ALLIANCE_COLOR = { 0.2, 0.4, 0.8 }
local HORDE_COLOR    = { 0.5, 0.08, 0.08 }
local CONTESTED_COLOR = { 0.85, 0.75, 0.2 }
local BAR_WIDTH = 240
local BAR_HEIGHT = 24
local BAR_SPACING = 3
local SCORE_TICK_MIN_INTERVAL = 0.5

local FACTION_ICONS = {
    Alliance = "Interface\\FriendsFrame\\PlusManz-Alliance",
    Horde    = "Interface\\FriendsFrame\\PlusManz-Horde",
}

-----------------------------
-- State
-----------------------------
local activeConfig = nil
local activeMapID = nil
local isBlitz = false
local scoreTicker = nil
local pendingActivateTimer = nil
local anchorFrame = nil
local predictionBar = nil
local timerBars = {}
local barPool = {}

-- Score tracking
local lastScoreTime = 0
local lastAllyScore = 0
local lastHordeScore = 0
local maxScore = 0

-- Tick calibration (self-calibrating from observed score ticks)
local calibrationTicks = {}    -- { { time, allyScore, hordeScore }, ... } — first few ticks
local calibratedInterval = nil -- seconds between ticks (nil = not yet calibrated)
local calibratedPPT = nil      -- { [basesControlled] = ptsPerTick } observed overrides

-- Node ownership tracking for model-based prediction
local nodeOwnership = {}       -- [nodeName] = { faction = "Alliance"|"Horde", contested = bool }

-- Node capture state
local activeCaptures = {}      -- [nodeName] = { faction, startTime, duration }

-- Flag state (CTF)
local flagState = {}           -- { alliance = { status, carrier }, horde = { status, carrier } }

-----------------------------
-- BG Scoring Tables (node-control BGs only)
-----------------------------
-- Points per tick by bases controlled. Tick interval is ~2s for most BGs.
-- Self-calibration overrides these if observed values differ.
local BG_SCORING = {
    ["Arathi Basin"] = {
        maxBases = 5, tickInterval = 2,
        ptsPerTick      = { [0]=0, [1]=2,  [2]=3,  [3]=4,  [4]=7,  [5]=60 },
        ptsPerTickBlitz  = { [0]=0, [1]=7,  [2]=10, [3]=15, [4]=50, [5]=65 },
    },
    ["Battle for Gilneas"] = {
        maxBases = 3, tickInterval = 2,
        ptsPerTick = { [0]=0, [1]=2, [2]=3, [3]=30 },
    },
    ["Deepwind Gorge"] = {
        maxBases = 5, tickInterval = 2,
        ptsPerTick      = { [0]=0, [1]=2,  [2]=3,  [3]=4,  [4]=7,  [5]=60 },
        ptsPerTickBlitz  = { [0]=0, [1]=7,  [2]=10, [3]=15, [4]=50, [5]=65 },
    },
    ["Eye of the Storm"] = {
        maxBases = 4, tickInterval = 1,
        ptsPerTick = { [0]=0, [1]=1, [2]=2, [3]=5, [4]=10 },
    },
}

-----------------------------
-- Calibration Log (persisted in SKToolsBILogDB)
-----------------------------
-- Logs every observed score tick, capture event, and flag event
-- so we can compare hardcoded values to reality and permanently fix them.
-- Data aggregates across sessions. Use /bilog to dump, /bilog clear to wipe.

local function BILog(category, data)
    if not SKToolsBILogDB then SKToolsBILogDB = {} end
    if not SKToolsBILogDB[category] then SKToolsBILogDB[category] = {} end
    data.time = date("%Y-%m-%d %H:%M:%S")
    data.bg = activeConfig and activeConfig.name or "?"
    data.blitz = isBlitz
    table.insert(SKToolsBILogDB[category], data)
    -- Cap per category to prevent SavedVars bloat
    while #SKToolsBILogDB[category] > 500 do
        table.remove(SKToolsBILogDB[category], 1)
    end
end

local function BILogScoreTick(allyScore, hordeScore, allyBases, hordeBases, tickDelta, interval)
    BILog("ticks", {
        aScore = allyScore, hScore = hordeScore,
        aBases = allyBases, hBases = hordeBases,
        aDelta = tickDelta.ally, hDelta = tickDelta.horde,
        interval = tonumber(string.format("%.2f", interval)),
    })
end

local function BILogCapture(nodeName, faction, duration, source)
    BILog("captures", {
        node = nodeName, faction = faction,
        duration = tonumber(string.format("%.1f", duration)),
        source = source,  -- "chat" or "poi"
    })
end

local function BILogFlag(faction, duration)
    BILog("flags", {
        faction = faction,
        duration = duration,
    })
end

function ns.BattleIntel_DumpLog(arg)
    if not SKToolsBILogDB then
        print("|cff00E5EE[BI Log]|r No data yet.")
        return
    end

    if arg == "clear" then
        SKToolsBILogDB = {}
        print("|cff00E5EE[BI Log]|r Cleared all data.")
        return
    end

    -- Summary mode
    local cats = { "ticks", "captures", "flags" }
    for _, cat in ipairs(cats) do
        local entries = SKToolsBILogDB[cat]
        if entries and #entries > 0 then
            print("|cff00E5EE[BI Log]|r --- " .. cat:upper() .. " (" .. #entries .. " entries) ---")
            if cat == "ticks" then
                -- Aggregate: group by BG+blitz+bases, show avg PPT and interval
                local groups = {}
                for _, e in ipairs(entries) do
                    local key = (e.bg or "?") .. (e.blitz and " (Blitz)" or "")
                    if not groups[key] then groups[key] = {} end
                    table.insert(groups[key], e)
                end
                for key, ticks in pairs(groups) do
                    print("  " .. key .. ": " .. #ticks .. " ticks")
                    -- PPT by base count
                    local pptByBases = {}
                    local intervals = {}
                    for _, t in ipairs(ticks) do
                        if t.aBases and t.aDelta and t.aDelta > 0 then
                            if not pptByBases[t.aBases] then pptByBases[t.aBases] = {} end
                            table.insert(pptByBases[t.aBases], t.aDelta)
                        end
                        if t.hBases and t.hDelta and t.hDelta > 0 then
                            if not pptByBases[t.hBases] then pptByBases[t.hBases] = {} end
                            table.insert(pptByBases[t.hBases], t.hDelta)
                        end
                        if t.interval then table.insert(intervals, t.interval) end
                    end
                    for bases, vals in pairs(pptByBases) do
                        local sum = 0
                        for _, v in ipairs(vals) do sum = sum + v end
                        local avg = sum / #vals
                        print(string.format("    %d bases: avg PPT=%.1f (%d samples)", bases, avg, #vals))
                    end
                    if #intervals > 0 then
                        local sum = 0
                        for _, v in ipairs(intervals) do sum = sum + v end
                        print(string.format("    avg interval=%.2fs (%d samples)", sum / #intervals, #intervals))
                    end
                end
            elseif cat == "captures" then
                -- Group by BG+blitz, show durations
                local groups = {}
                for _, e in ipairs(entries) do
                    local key = (e.bg or "?") .. (e.blitz and " (Blitz)" or "")
                    if not groups[key] then groups[key] = {} end
                    table.insert(groups[key], e)
                end
                for key, caps in pairs(groups) do
                    local durations = {}
                    for _, c in ipairs(caps) do
                        table.insert(durations, c.duration)
                    end
                    table.sort(durations)
                    local median = durations[math.ceil(#durations / 2)] or 0
                    print(string.format("  %s: %d captures, median duration=%.1fs", key, #caps, median))
                end
            elseif cat == "flags" then
                for _, e in ipairs(entries) do
                    print(string.format("  %s %s: %s respawn=%ds", e.time or "", e.bg or "?", e.faction or "?", e.duration or 0))
                end
            end
        end
    end

    if not next(SKToolsBILogDB) then
        print("|cff00E5EE[BI Log]|r No data yet. Queue some BGs!")
    end
end

-----------------------------
-- Bar Pool
-----------------------------
local function AcquireBar()
    for i, bar in ipairs(barPool) do
        if not bar.inUse then
            bar.inUse = true
            bar:Show()
            return bar
        end
    end
    return nil  -- will create new if needed
end

local function ReleaseBar(bar)
    bar.inUse = false
    bar:Hide()
end

local function ReleaseAllBars()
    for _, bar in ipairs(timerBars) do
        ReleaseBar(bar)
    end
    timerBars = {}
end

-----------------------------
-- Timer Bar Creation
-----------------------------
local function CreateBar(parent)
    local CC = ns.COLORS
    local bar = CreateFrame("Frame", nil, parent, "BackdropTemplate")
    bar:SetSize(BAR_WIDTH, BAR_HEIGHT)
    bar:SetBackdrop(ns.BACKDROP_CONTROL)
    bar:SetBackdropColor(CC.bgControl[1], CC.bgControl[2], CC.bgControl[3], 0.9)
    bar:SetBackdropBorderColor(CC.border[1], CC.border[2], CC.border[3], 0.6)

    -- Fill texture (manual width for taint safety — no StatusBar:SetValue)
    local fill = bar:CreateTexture(nil, "BORDER")
    fill:SetPoint("TOPLEFT", 1, -1)
    fill:SetPoint("BOTTOMLEFT", 1, 1)
    fill:SetWidth(1)
    fill:SetColorTexture(unpack(ALLIANCE_COLOR))
    bar.fill = fill

    -- Icon
    local icon = bar:CreateTexture(nil, "ARTWORK")
    icon:SetSize(BAR_HEIGHT - 4, BAR_HEIGHT - 4)
    icon:SetPoint("LEFT", 4, 0)
    icon:SetTexCoord(0.08, 0.92, 0.08, 0.92)
    bar.icon = icon

    -- Label (upgraded to 12pt)
    local label = bar:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
    label:SetPoint("LEFT", icon, "RIGHT", 4, 0)
    label:SetPoint("RIGHT", bar, "RIGHT", -50, 0)
    label:SetJustifyH("LEFT")
    label:SetTextColor(0.9, 0.9, 0.9)
    bar.label = label

    -- Timer text (monospace-like to prevent digit width jitter)
    local timer = bar:CreateFontString(nil, "OVERLAY", "NumberFontNormalSmall")
    timer:SetPoint("RIGHT", -6, 0)
    timer:SetJustifyH("RIGHT")
    timer:SetTextColor(0.9, 0.9, 0.9)
    bar.timer = timer

    bar.inUse = false
    bar:Hide()
    table.insert(barPool, bar)
    return bar
end

local function SetBarFill(bar, fraction)
    local w = math.max(1, math.floor((BAR_WIDTH - 2) * math.min(1, math.max(0, fraction))))
    bar.fill:SetWidth(w)
end

local function SetBarColor(bar, r, g, b)
    bar.fill:SetColorTexture(1, 1, 1)
    bar.fill:SetGradient("HORIZONTAL", CreateColor(r * 0.4, g * 0.4, b * 0.4, 0.6), CreateColor(r, g, b, 0.6))
end

local function SetTimerColor(bar, seconds)
    bar.timer:SetTextColor(0.9, 0.9, 0.9)
end

local function FormatTime(seconds)
    if seconds <= 0 then return "0:00" end
    return string.format("%d:%02d", math.floor(seconds / 60), math.floor(seconds % 60))
end

-- Helper: set prediction bar label + timer consistently
local function SetPredictionText(label, timeText)
    if not predictionBar then return end
    predictionBar.label:SetText(label)
    if timeText then
        predictionBar.timer:SetText(timeText)
        predictionBar.timer:SetTextColor(0.9, 0.9, 0.9)
        predictionBar.timer:Show()
    else
        predictionBar.timer:SetText("")
        predictionBar.timer:Hide()
    end
end

-----------------------------
-- Anchor Frame (movable HUD)
-----------------------------
local DEFAULT_POS = { point = "TOPRIGHT", relPoint = "TOPRIGHT", x = -250, y = -120 }

local function CreateAnchorFrame()
    if anchorFrame then return anchorFrame end

    local CC = ns.COLORS
    local f = CreateFrame("Frame", "SKToolsBattleIntelFrame", UIParent, "BackdropTemplate")
    f:SetSize(BAR_WIDTH + 8, 10)  -- height adjusts dynamically
    f:SetBackdrop(ns.BACKDROP_PANEL)
    f:SetBackdropColor(CC.bgFrame[1], CC.bgFrame[2], CC.bgFrame[3], CC.bgFrame[4])
    f:SetBackdropBorderColor(CC.border[1], CC.border[2], CC.border[3], 0.80)
    f:SetFrameStrata("MEDIUM")
    f:SetFrameLevel(50)
    f:SetClampedToScreen(true)
    f:SetMovable(true)
    f:EnableMouse(true)
    f:RegisterForDrag("LeftButton")

    ns.CreateDropShadow(f, 6)
    ns.CreateCyanAccentLine(f)

    -- Drag behavior
    f:SetScript("OnDragStart", function(self)
        self:StartMoving()
    end)
    f:SetScript("OnDragStop", function(self)
        self:StopMovingOrSizing()
        local point, _, relPoint, x, y = self:GetPoint()
        SKToolsDB.battleIntelPos = { point = point, relPoint = relPoint, x = x, y = y }
    end)

    -- Migrate old default position (was TOP,TOP,0,-120 — collides with BG scoreboard)
    local pos = SKToolsDB.battleIntelPos
    if pos and pos.point == "TOP" and pos.relPoint == "TOP" and pos.x == 0 and pos.y == -120 then
        SKToolsDB.battleIntelPos = nil
        pos = nil
    end

    -- Position from saved vars or default
    if pos then
        f:SetPoint(pos.point, UIParent, pos.relPoint, pos.x, pos.y)
    else
        f:SetPoint(DEFAULT_POS.point, UIParent, DEFAULT_POS.relPoint, DEFAULT_POS.x, DEFAULT_POS.y)
    end

    -- Divider line between prediction bar and timer bars
    local predDivider = f:CreateTexture(nil, "ARTWORK")
    predDivider:SetHeight(1)
    predDivider:SetPoint("LEFT", f, "LEFT", 6, 0)
    predDivider:SetPoint("RIGHT", f, "RIGHT", -6, 0)
    predDivider:SetColorTexture(CC.divider[1], CC.divider[2], CC.divider[3], CC.divider[4])
    predDivider:Hide()
    f.predDivider = predDivider

    -- Prediction bar (always first — NOT in the pool, managed separately)
    local pred = CreateBar(f)
    -- Remove from pool so AcquireBar never hands it out as a timer bar
    for i = #barPool, 1, -1 do
        if barPool[i] == pred then
            table.remove(barPool, i)
            break
        end
    end
    pred:SetPoint("TOPLEFT", f, "TOPLEFT", 4, -4)
    pred.label:SetPoint("LEFT", pred.icon, "RIGHT", 4, 0)
    pred.label:SetPoint("RIGHT", pred, "RIGHT", -50, 0)
    pred:Hide()
    predictionBar = pred

    f:Hide()
    anchorFrame = f
    return f
end

-----------------------------
-- Layout Timer Bars
-----------------------------
local function LayoutBars()
    if not anchorFrame then return end
    local yOff = -4  -- top padding
    local barCount = 0
    local predShown = predictionBar and predictionBar:IsShown()
    local hasTimerBars = false

    -- Prediction bar first
    if predShown then
        predictionBar:SetPoint("TOPLEFT", anchorFrame, "TOPLEFT", 4, yOff)
        yOff = yOff - BAR_HEIGHT - BAR_SPACING
        barCount = barCount + 1
    end

    -- Check if any timer bars are showing
    for _, bar in ipairs(timerBars) do
        if bar:IsShown() then hasTimerBars = true; break end
    end

    -- Divider between prediction and timer bars
    if predShown and hasTimerBars and anchorFrame.predDivider then
        anchorFrame.predDivider:ClearAllPoints()
        anchorFrame.predDivider:SetPoint("LEFT", anchorFrame, "LEFT", 6, 0)
        anchorFrame.predDivider:SetPoint("RIGHT", anchorFrame, "RIGHT", -6, 0)
        anchorFrame.predDivider:SetPoint("TOP", anchorFrame, "TOP", 0, yOff + 1)
        anchorFrame.predDivider:Show()
        yOff = yOff - 3
    elseif anchorFrame.predDivider then
        anchorFrame.predDivider:Hide()
    end

    -- Timer bars
    for _, bar in ipairs(timerBars) do
        if bar:IsShown() then
            bar:SetPoint("TOPLEFT", anchorFrame, "TOPLEFT", 4, yOff)
            yOff = yOff - BAR_HEIGHT - BAR_SPACING
            barCount = barCount + 1
        end
    end

    -- Adjust anchor height
    if barCount > 0 then
        local divExtra = (predShown and hasTimerBars) and 3 or 0
        anchorFrame:SetHeight(4 + barCount * (BAR_HEIGHT + BAR_SPACING) + divExtra + 4)
        anchorFrame:Show()
    else
        anchorFrame:Hide()
    end
end

-----------------------------
-- Score Estimation
-----------------------------
local function ResetScoreState()
    lastScoreTime = 0
    lastAllyScore = 0
    lastHordeScore = 0
    maxScore = 0
    calibrationTicks = {}
    calibratedInterval = nil
    calibratedPPT = nil
    nodeOwnership = {}
end

-- Count uncontested bases for each faction
local function CountBases()
    local ally, horde = 0, 0
    for _, info in pairs(nodeOwnership) do
        if not info.contested then
            if info.faction == "Alliance" then
                ally = ally + 1
            elseif info.faction == "Horde" then
                horde = horde + 1
            end
        end
    end
    return ally, horde
end

-- Get scoring config for the active BG (nil if not a node-control BG)
local function GetScoringConfig()
    if not activeConfig then return nil end
    local scoring = BG_SCORING[activeConfig.name]
    if not scoring then return nil end
    -- In Blitz, use Blitz-specific PPT if available
    if isBlitz and scoring.ptsPerTickBlitz then
        return {
            maxBases = scoring.maxBases,
            tickInterval = scoring.tickInterval,
            ptsPerTick = scoring.ptsPerTickBlitz,
        }
    end
    return scoring
end

-- Calculate time-to-win with optional pending capture projection
local function CalcTimeToWin(allyScore, hordeScore, max, scoring, tickInterval)
    local allyBases, hordeBases = CountBases()
    local ppt = calibratedPPT or scoring.ptsPerTick

    local allyPPT = ppt[allyBases] or 0
    local hordePPT = ppt[hordeBases] or 0
    local interval = calibratedInterval or tickInterval

    -- Rates in points per second
    local allyRate = (interval > 0 and allyPPT > 0) and (allyPPT / interval) or 0
    local hordeRate = (interval > 0 and hordePPT > 0) and (hordePPT / interval) or 0

    -- Check for pending captures that will flip bases
    local now = GetTime()
    local bestFlipTime = nil
    local flipAllyDelta, flipHordeDelta = 0, 0
    for nodeName, cap in pairs(activeCaptures) do
        local remaining = cap.duration - (now - cap.startTime)
        if remaining > 0 then
            local node = nodeOwnership[nodeName]
            if node and node.contested then
                if not bestFlipTime or remaining < bestFlipTime then
                    bestFlipTime = remaining
                end
                -- This base is being captured by cap.faction
                if cap.faction == "Alliance" then
                    flipAllyDelta = flipAllyDelta + 1
                    flipHordeDelta = flipHordeDelta - 1
                elseif cap.faction == "Horde" then
                    flipHordeDelta = flipHordeDelta + 1
                    flipAllyDelta = flipAllyDelta - 1
                end
            end
        end
    end

    local allyTTW, hordeTTW

    if bestFlipTime and (flipAllyDelta ~= 0 or flipHordeDelta ~= 0) then
        -- Phase 1: current rates until flip
        local allyP1 = allyScore + allyRate * bestFlipTime
        local hordeP1 = hordeScore + hordeRate * bestFlipTime

        -- Check if either wins during Phase 1
        if allyRate > 0 and (max - allyScore) / allyRate <= bestFlipTime then
            allyTTW = (max - allyScore) / allyRate
        end
        if hordeRate > 0 and (max - hordeScore) / hordeRate <= bestFlipTime then
            hordeTTW = (max - hordeScore) / hordeRate
        end

        -- Phase 2: new rates after flip
        if not allyTTW or not hordeTTW then
            local newAllyBases = math.max(0, math.min(scoring.maxBases, allyBases + flipAllyDelta))
            local newHordeBases = math.max(0, math.min(scoring.maxBases, hordeBases + flipHordeDelta))
            local newAllyPPT = ppt[newAllyBases] or 0
            local newHordePPT = ppt[newHordeBases] or 0
            local newAllyRate = (interval > 0 and newAllyPPT > 0) and (newAllyPPT / interval) or 0
            local newHordeRate = (interval > 0 and newHordePPT > 0) and (newHordePPT / interval) or 0

            if not allyTTW then
                allyTTW = (newAllyRate > 0) and (bestFlipTime + (max - allyP1) / newAllyRate) or math.huge
            end
            if not hordeTTW then
                hordeTTW = (newHordeRate > 0) and (bestFlipTime + (max - hordeP1) / newHordeRate) or math.huge
            end
        end
    else
        -- Simple case: no pending captures
        allyTTW = (allyRate > 0) and ((max - allyScore) / allyRate) or math.huge
        hordeTTW = (hordeRate > 0) and ((max - hordeScore) / hordeRate) or math.huge
    end

    return allyTTW or math.huge, hordeTTW or math.huge, allyRate, hordeRate
end

local function UpdateScoreEstimation(allyScore, hordeScore, max)
    if not SKToolsDB.battleIntelShowPrediction then
        if predictionBar then predictionBar:Hide() end
        return
    end

    local now = GetTime()
    maxScore = max

    -- Throttle: only process if > 0.5s since last tick
    if (now - lastScoreTime) < SCORE_TICK_MIN_INTERVAL then return end

    local scoring = GetScoringConfig()

    -- Detect actual score ticks (score changed since last reading)
    local scoreChanged = allyScore ~= lastAllyScore or hordeScore ~= lastHordeScore
    if scoreChanged then
        -- Log every observed tick for calibration data
        local allyBases, hordeBases = CountBases()
        local interval = lastScoreTime > 0 and (now - lastScoreTime) or 0
        if lastAllyScore > 0 or lastHordeScore > 0 then
            BILogScoreTick(allyScore, hordeScore, allyBases, hordeBases,
                { ally = allyScore - lastAllyScore, horde = hordeScore - lastHordeScore }, interval)
        end

        -- Self-calibration: collect first ticks to measure interval and PPT
        if scoring and not calibratedInterval then
            table.insert(calibrationTicks, { time = now, ally = allyScore, horde = hordeScore })

            if #calibrationTicks >= 3 then
                -- Measure average interval between ticks
                local totalDT = calibrationTicks[#calibrationTicks].time - calibrationTicks[1].time
                calibratedInterval = totalDT / (#calibrationTicks - 1)

                -- Validate against hardcoded scoring table using observed deltas
                if allyBases > 0 and #calibrationTicks >= 2 then
                    local last = calibrationTicks[#calibrationTicks]
                    local prev = calibrationTicks[#calibrationTicks - 1]
                    local observedPPT = last.ally - prev.ally
                    local expectedPPT = scoring.ptsPerTick[allyBases] or 0
                    if expectedPPT > 0 and observedPPT > 0 and math.abs(observedPPT - expectedPPT) > 1 then
                        -- Scoring table mismatch — use observed values
                        calibratedPPT = {}
                        for k, v in pairs(scoring.ptsPerTick) do calibratedPPT[k] = v end
                        calibratedPPT[allyBases] = observedPPT
                    end
                end
            end
        end
    end

    lastScoreTime = now
    lastAllyScore = allyScore
    lastHordeScore = hordeScore

    if max <= 0 then
        if predictionBar then predictionBar:Hide() end
        LayoutBars()
        return
    end

    if not predictionBar then return end

    -- Non-node BGs (Temple, SSM, Seething Shore, AV, IoC, Deephaul): no model-based prediction
    if not scoring then
        -- Fallback: simple linear extrapolation for non-node BGs
        if #calibrationTicks < 2 then
            predictionBar:Show()
            predictionBar.inUse = true
            SetPredictionText("Calculating...", nil)
            predictionBar.icon:SetTexture(136998)
            SetBarColor(predictionBar, unpack(CONTESTED_COLOR))
            SetBarFill(predictionBar, 0.5)
            LayoutBars()
            return
        end
        local oldest = calibrationTicks[1]
        local dt = now - oldest.time
        if dt < 2 then
            predictionBar:Hide()
            LayoutBars()
            return
        end
        local aRate = (allyScore - oldest.ally) / dt
        local hRate = (hordeScore - oldest.horde) / dt
        local aTTW = (aRate > 0) and ((max - allyScore) / aRate) or math.huge
        local hTTW = (hRate > 0) and ((max - hordeScore) / hRate) or math.huge
        if aTTW == math.huge and hTTW == math.huge then
            predictionBar:Hide()
            LayoutBars()
            return
        end
        predictionBar:Show()
        predictionBar.inUse = true
        if aTTW <= hTTW then
            SetPredictionText("Alliance wins in", "~" .. FormatTime(aTTW))
            SetBarColor(predictionBar, unpack(ALLIANCE_COLOR))
            SetBarFill(predictionBar, allyScore / max)
            predictionBar.icon:SetTexture(FACTION_ICONS.Alliance)
        else
            SetPredictionText("Horde wins in", "~" .. FormatTime(hTTW))
            SetBarColor(predictionBar, unpack(HORDE_COLOR))
            SetBarFill(predictionBar, hordeScore / max)
            predictionBar.icon:SetTexture(FACTION_ICONS.Horde)
        end
        LayoutBars()
        return
    end

    -- Node-control BGs: model-based prediction
    if not calibratedInterval then
        -- Still calibrating
        predictionBar:Show()
        predictionBar.inUse = true
        SetPredictionText("Calculating...", nil)
        predictionBar.icon:SetTexture(136998)
        SetBarColor(predictionBar, unpack(CONTESTED_COLOR))
        SetBarFill(predictionBar, 0.5)
        LayoutBars()
        return
    end

    local allyBases, hordeBases = CountBases()
    if allyBases == 0 and hordeBases == 0 then
        -- All bases contested or no ownership data yet
        predictionBar:Show()
        predictionBar.inUse = true
        SetPredictionText("Contested", nil)
        predictionBar.icon:SetTexture(136998)
        SetBarColor(predictionBar, unpack(CONTESTED_COLOR))
        SetBarFill(predictionBar, 0.5)
        LayoutBars()
        return
    end

    local allyTTW, hordeTTW = CalcTimeToWin(allyScore, hordeScore, max, scoring, scoring.tickInterval)

    if allyTTW == math.huge and hordeTTW == math.huge then
        predictionBar:Show()
        predictionBar.inUse = true
        SetPredictionText("Contested", nil)
        predictionBar.icon:SetTexture(136998)
        SetBarColor(predictionBar, unpack(CONTESTED_COLOR))
        SetBarFill(predictionBar, 0.5)
        LayoutBars()
        return
    end

    predictionBar:Show()
    predictionBar.inUse = true

    if allyTTW <= hordeTTW then
        SetPredictionText("Alliance wins in", FormatTime(allyTTW))
        SetBarColor(predictionBar, unpack(ALLIANCE_COLOR))
        SetBarFill(predictionBar, allyScore / max)
        predictionBar.icon:SetTexture(FACTION_ICONS.Alliance)
    else
        SetPredictionText("Horde wins in", FormatTime(hordeTTW))
        SetBarColor(predictionBar, unpack(HORDE_COLOR))
        SetBarFill(predictionBar, hordeScore / max)
        predictionBar.icon:SetTexture(FACTION_ICONS.Horde)
    end

    LayoutBars()
end

-----------------------------
-- Node Capture Tracking
-----------------------------
local function AddCaptureTimer(nodeName, faction, duration)
    if duration <= 0 then return end

    -- Always track for prediction model
    local now = GetTime()
    activeCaptures[nodeName] = { faction = faction, startTime = now, duration = duration }

    -- Visual bar only if captures toggle is on
    if not SKToolsDB.battleIntelShowCaptures then return end

    -- Remove existing timer bar for this node
    for i = #timerBars, 1, -1 do
        if timerBars[i].nodeName == nodeName then
            ReleaseBar(timerBars[i])
            table.remove(timerBars, i)
        end
    end

    local bar = AcquireBar()
    if not bar then
        bar = CreateBar(anchorFrame)
        bar.inUse = true
        bar:Show()
    end

    bar.nodeName = nodeName
    bar.startTime = now
    bar.duration = duration
    bar.label:SetText(nodeName)

    if faction == "Alliance" then
        SetBarColor(bar, unpack(ALLIANCE_COLOR))
        bar.icon:SetTexture(FACTION_ICONS.Alliance)
    elseif faction == "Horde" then
        SetBarColor(bar, unpack(HORDE_COLOR))
        bar.icon:SetTexture(FACTION_ICONS.Horde)
    else
        SetBarColor(bar, unpack(CONTESTED_COLOR))
        bar.icon:SetTexture(136998)  -- generic timer icon
    end

    bar.timer:Show()
    table.insert(timerBars, bar)
    LayoutBars()
end

local function RemoveCaptureTimer(nodeName)
    for i = #timerBars, 1, -1 do
        if timerBars[i].nodeName == nodeName then
            ReleaseBar(timerBars[i])
            table.remove(timerBars, i)
        end
    end
    activeCaptures[nodeName] = nil
    LayoutBars()
end

-----------------------------
-- Flag Respawn Timers (CTF / Hybrid)
-----------------------------
local function AddFlagRespawnTimer(faction, duration)
    if not SKToolsDB.battleIntelShowFlags then return end
    if not duration or duration <= 0 then return end

    local flagKey = faction .. "_respawn"

    -- Remove existing respawn bar for this faction
    for i = #timerBars, 1, -1 do
        if timerBars[i].flagKey == flagKey then
            ReleaseBar(timerBars[i])
            table.remove(timerBars, i)
        end
    end

    local bar = AcquireBar()
    if not bar then
        bar = CreateBar(anchorFrame)
        bar.inUse = true
        bar:Show()
    end

    bar.flagKey = flagKey
    bar.nodeName = nil
    bar.startTime = GetTime()
    bar.duration = duration
    bar.label:SetText("Flag Respawn")
    bar.timer:Show()
    SetBarFill(bar, 1)

    if faction == "Alliance" then
        SetBarColor(bar, unpack(ALLIANCE_COLOR))
        bar.icon:SetTexture(FACTION_ICONS.Alliance)
    else
        SetBarColor(bar, unpack(HORDE_COLOR))
        bar.icon:SetTexture(FACTION_ICONS.Horde)
    end

    table.insert(timerBars, bar)
    LayoutBars()
end

-----------------------------
-- Chat Message Parsing
-----------------------------
local function ParseChatMessage(event, text)
    if not activeConfig then return end
    local bgType = activeConfig.type

    -- Node captures (resource / hybrid)
    if bgType == "resource" or bgType == "hybrid" then
        local capTime = isBlitz and 30 or (activeConfig.capTime or 0)

        -- "The Alliance/Horde has assaulted the <Node>!" or "claims the <Node>!"
        -- These mean capture STARTED — start a countdown timer, mark node contested
        local faction, node = text:match("The (Alliance) has assaulted the (.+)!")
        if not faction then faction, node = text:match("The (Horde) has assaulted the (.+)!") end
        if not faction then faction, node = text:match("The (Alliance) claims the (.+)!") end
        if not faction then faction, node = text:match("The (Horde) claims the (.+)!") end

        if faction and node then
            -- Mark contested for prediction model
            nodeOwnership[node] = { faction = nodeOwnership[node] and nodeOwnership[node].faction or faction, contested = true }
            if capTime > 0 then
                BILogCapture(node, faction, capTime, "chat")
                AddCaptureTimer(node, faction, capTime)
            end
            return
        end

        -- "The Alliance/Horde has taken the <Node>!" — capture COMPLETED
        local takenFaction, takenNode = text:match("The (Alliance) has taken the (.+)!")
        if not takenFaction then takenFaction, takenNode = text:match("The (Horde) has taken the (.+)!") end
        if takenFaction and takenNode then
            nodeOwnership[takenNode] = { faction = takenFaction, contested = false }
            RemoveCaptureTimer(takenNode)
            return
        end

        -- "has defended the <Node>" — assault repelled, clear contested
        local defNode = text:match("has defended the (.+)")
        if defNode then
            defNode = defNode:gsub("[!.]$", "")
            if nodeOwnership[defNode] then
                nodeOwnership[defNode].contested = false
            end
            RemoveCaptureTimer(defNode)
            return
        end
    end

    -- Flag respawn (CTF / hybrid) — only track captures for respawn timer
    if bgType == "ctf" or bgType == "hybrid" then
        if text:find("captured the") then
            local respawn = activeConfig.flagRespawn or 12
            local faction
            if event == "CHAT_MSG_BG_SYSTEM_ALLIANCE" then
                faction = "Alliance"
            elseif event == "CHAT_MSG_BG_SYSTEM_HORDE" then
                faction = "Horde"
            end
            if faction then
                BILogFlag(faction, respawn)
                AddFlagRespawnTimer(faction, respawn)
            end
            return
        end
    end
end

-----------------------------
-- POI-based Capture Detection
-----------------------------
local function HandleAreaPOIsUpdated()
    if not activeConfig then return end
    if not activeMapID then return end

    local ok, pois = pcall(C_AreaPoiInfo.GetAreaPOIForMap, activeMapID)
    if not ok or type(pois) ~= "table" then return end

    for _, poiID in ipairs(pois) do
        local ok2, info = pcall(C_AreaPoiInfo.GetAreaPOIInfo, activeMapID, poiID)
        if ok2 and info and info.name then
            -- Determine faction from atlas name
            local faction = nil
            local isContested = false
            if info.atlasName then
                if info.atlasName:find("alliance") or info.atlasName:find("Alliance") then
                    faction = "Alliance"
                elseif info.atlasName:find("horde") or info.atlasName:find("Horde") then
                    faction = "Horde"
                end
            end

            -- Check if this POI is a timed capture (contested node)
            local ok3, isTimed = pcall(C_AreaPoiInfo.IsAreaPOITimed, poiID)
            if ok3 and isTimed then
                local ok4, secsLeft = pcall(C_AreaPoiInfo.GetAreaPOISecondsLeft, poiID)
                if ok4 and secsLeft and secsLeft > 0 then
                    isContested = true
                    -- Update capture timer if needed
                    if SKToolsDB.battleIntelShowCaptures and (activeConfig.capTime or 0) > 0 then
                        local existing = activeCaptures[info.name]
                        if not existing or math.abs((existing.startTime + existing.duration - GetTime()) - secsLeft) > 3 then
                            BILogCapture(info.name, faction or "Contested", secsLeft, "poi")
                            AddCaptureTimer(info.name, faction or "Contested", secsLeft)
                        end
                    end
                end
            end

            -- Initialize/update node ownership from POI data (handles mid-game joins)
            if faction and BG_SCORING[activeConfig.name] then
                if not nodeOwnership[info.name] then
                    nodeOwnership[info.name] = { faction = faction, contested = isContested }
                elseif isContested then
                    nodeOwnership[info.name].contested = true
                end
            end
        end
    end
end

-----------------------------
-- Update Ticker
-----------------------------
local function OnTick()
    if not activeConfig then return end
    local now = GetTime()

    -- Update capture timer bars
    for i = #timerBars, 1, -1 do
        local bar = timerBars[i]
        if bar.startTime and bar.duration then
            local elapsed = now - bar.startTime
            local remaining = bar.duration - elapsed
            if remaining <= 0 then
                -- Timer expired
                if bar.nodeName then
                    activeCaptures[bar.nodeName] = nil
                end
                ReleaseBar(bar)
                table.remove(timerBars, i)
            else
                bar.timer:SetText(FormatTime(remaining))
                SetTimerColor(bar, remaining)
                SetBarFill(bar, remaining / bar.duration)
            end
        end
    end

    -- Read score widget
    if activeConfig.widgetID then
        local ok, info = pcall(C_UIWidgetManager.GetDoubleStatusBarWidgetVisualizationInfo, activeConfig.widgetID)
        if ok and info and info.leftBarMax and info.leftBarMax > 0 then
            UpdateScoreEstimation(info.leftBarValue or 0, info.rightBarValue or 0, info.leftBarMax)
        end
    end

    LayoutBars()
end

-----------------------------
-- BG Lifecycle
-----------------------------
local biFrame = CreateFrame("Frame")

local function DeactivateBG()
    activeConfig = nil
    activeMapID = nil
    isBlitz = false
    activeCaptures = {}
    flagState = {}
    ResetScoreState()
    ReleaseAllBars()

    if pendingActivateTimer then
        pendingActivateTimer:Cancel()
        pendingActivateTimer = nil
    end

    -- Clean up unused pool bars to prevent memory creep across BGs
    for i = #barPool, 1, -1 do
        if not barPool[i].inUse then
            barPool[i]:Hide()
            table.remove(barPool, i)
        end
    end

    if predictionBar then
        predictionBar:Hide()
        predictionBar.inUse = false
    end

    if scoreTicker then
        scoreTicker:Cancel()
        scoreTicker = nil
    end

    if anchorFrame then
        anchorFrame:Hide()
    end

    biFrame:UnregisterEvent("UPDATE_UI_WIDGET")
    biFrame:UnregisterEvent("CHAT_MSG_BG_SYSTEM_ALLIANCE")
    biFrame:UnregisterEvent("CHAT_MSG_BG_SYSTEM_HORDE")
    biFrame:UnregisterEvent("CHAT_MSG_BG_SYSTEM_NEUTRAL")
    biFrame:UnregisterEvent("AREA_POIS_UPDATED")
end

local function ActivateBG(config, mapID)
    DeactivateBG()
    activeConfig = config
    activeMapID = mapID

    -- Detect Blitz (solo rated BG) — affects capture timers and scoring rates
    local ok, result = pcall(C_PvP.IsRatedSoloRBG)
    isBlitz = ok and result or false

    CreateAnchorFrame()

    biFrame:RegisterEvent("UPDATE_UI_WIDGET")
    biFrame:RegisterEvent("CHAT_MSG_BG_SYSTEM_ALLIANCE")
    biFrame:RegisterEvent("CHAT_MSG_BG_SYSTEM_HORDE")
    biFrame:RegisterEvent("CHAT_MSG_BG_SYSTEM_NEUTRAL")
    biFrame:RegisterEvent("AREA_POIS_UPDATED")

    -- Start update ticker (0.5s interval)
    scoreTicker = C_Timer.NewTicker(0.5, OnTick)

    -- Initial POI scan
    C_Timer.After(0.5, HandleAreaPOIsUpdated)
end

-----------------------------
-- Event Handler
-----------------------------
biFrame:SetScript("OnEvent", function(self, event, ...)
    if event == "PLAYER_ENTERING_WORLD" then
        local _, instanceType = IsInInstance()
        if instanceType == "pvp" then
            if pendingActivateTimer then pendingActivateTimer:Cancel() end
            pendingActivateTimer = C_Timer.After(1.5, function()
                pendingActivateTimer = nil
                local ok, mapID = pcall(C_Map.GetBestMapForUnit, "player")
                if ok and mapID then
                    local config = BG_MAP_IDS[mapID]
                    if config then
                        ActivateBG(config, mapID)
                    end
                end
            end)
        else
            if activeConfig then
                DeactivateBG()
            end
        end

    elseif event == "UPDATE_UI_WIDGET" then
        -- Widget updates handled by ticker poll, but use this for immediate score updates
        local widgetInfo = ...
        if type(widgetInfo) == "table" and activeConfig and activeConfig.widgetID
            and widgetInfo.widgetID == activeConfig.widgetID then
            local ok, info = pcall(C_UIWidgetManager.GetDoubleStatusBarWidgetVisualizationInfo, widgetInfo.widgetID)
            if ok and info and info.leftBarMax and info.leftBarMax > 0 then
                UpdateScoreEstimation(info.leftBarValue or 0, info.rightBarValue or 0, info.leftBarMax)
            end
        end

    elseif event == "CHAT_MSG_BG_SYSTEM_ALLIANCE"
        or event == "CHAT_MSG_BG_SYSTEM_HORDE"
        or event == "CHAT_MSG_BG_SYSTEM_NEUTRAL" then
        local text = ...
        if text then
            ParseChatMessage(event, text)
        end

    elseif event == "AREA_POIS_UPDATED" then
        HandleAreaPOIsUpdated()
    end
end)

-----------------------------
-- Public API
-----------------------------
function ns.SetBattleIntel(enabled)
    if enabled then
        biFrame:RegisterEvent("PLAYER_ENTERING_WORLD")
        -- Check if already in BG
        local _, instanceType = IsInInstance()
        if instanceType == "pvp" then
            if pendingActivateTimer then pendingActivateTimer:Cancel() end
            pendingActivateTimer = C_Timer.After(1.5, function()
                pendingActivateTimer = nil
                local ok, mapID = pcall(C_Map.GetBestMapForUnit, "player")
                if ok and mapID then
                    local config = BG_MAP_IDS[mapID]
                    if config then
                        ActivateBG(config, mapID)
                    end
                end
            end)
        end
    else
        biFrame:UnregisterEvent("PLAYER_ENTERING_WORLD")
        DeactivateBG()
    end
end

-----------------------------
-- Test Mode (for positioning)
-----------------------------
local function MakeTestBar(label, iconTex, color, timerText, seconds, fillFrac)
    local bar = AcquireBar()
    if not bar then
        bar = CreateBar(anchorFrame)
        bar.inUse = true
        bar:Show()
    end
    bar.label:SetText(label)
    bar.icon:SetTexture(iconTex)
    bar.timer:Show()
    bar.timer:SetText(timerText)
    bar.startTime = GetTime()
    bar.duration = seconds
    SetBarColor(bar, color[1], color[2], color[3])
    SetBarFill(bar, fillFrac)
    SetTimerColor(bar, seconds * fillFrac)
    table.insert(timerBars, bar)
    return bar
end

local function ShowTestBars()
    CreateAnchorFrame()
    DeactivateBG()

    -- Score prediction
    predictionBar:Show()
    predictionBar.inUse = true
    SetPredictionText("Alliance wins in", "3:42")
    predictionBar.icon:SetTexture(FACTION_ICONS.Alliance)
    SetBarColor(predictionBar, unpack(ALLIANCE_COLOR))
    SetBarFill(predictionBar, 0.7)

    -- Node capture: Horde assaulting (mid-timer)
    MakeTestBar("Blacksmith", FACTION_ICONS.Horde, HORDE_COLOR, "0:42", 60, 0.7)

    -- Node capture: Alliance almost done
    MakeTestBar("Stables", FACTION_ICONS.Alliance, ALLIANCE_COLOR, "0:08", 60, 0.13)

    -- Node capture: contested (early)
    MakeTestBar("Lumber Mill", FACTION_ICONS.Horde, HORDE_COLOR, "0:55", 60, 0.92)

    -- Flag respawn timer
    MakeTestBar("Flag Respawn", FACTION_ICONS.Alliance, ALLIANCE_COLOR, "0:07", 12, 0.58)

    LayoutBars()
end

local function HideTestBars()
    ReleaseAllBars()
    if predictionBar then
        predictionBar:Hide()
        predictionBar.inUse = false
    end
    if anchorFrame then anchorFrame:Hide() end
end

ns.BattleIntel_ShowTest = ShowTestBars
ns.BattleIntel_HideTest = HideTestBars

-----------------------------
-- Settings Builders
-----------------------------
function ns.BuildBattleIntelSettings(panel, anchor)
    local header = ns.AddSectionHeader(panel, anchor, "Battle Intel", false)
    local card = ns.CreateSectionCard(panel)
    card:SetPoint("TOPLEFT", header, "TOPLEFT", -8, 8)

    local AddCB, GetLast, SetLast = ns.MakeCheckboxFactory(panel, header)

    AddCB("BattleIntel", "battleIntel", "Battle Intel",
        "Real-time battleground intelligence with capture timers, score prediction, and flag tracking.",
        ns.SetBattleIntel)

    local biParentCB = _G["SKToolsBattleIntelCB"]
    local biParentAnchor = GetLast()

    -- Sub-toggles (indented children of Battle Intel)
    local SUB_INDENT = 24
    local SUB_DEFS = {
        { name = "BIScorePrediction", key = "battleIntelShowPrediction", label = "Score Prediction",
          desc = "Show predicted winner and estimated time to win based on current score rates." },
        { name = "BICaptureTimers",   key = "battleIntelShowCaptures",   label = "Capture Timers",
          desc = "Show countdown bars when nodes or bases are being captured." },
        { name = "BIFlagRespawn",     key = "battleIntelShowFlags",      label = "Flag Respawn Timers",
          desc = "Show flag respawn countdown after a capture in WSG, Twin Peaks, and Eye of the Storm." },
    }
    local subToggles = {}
    local lastSub = biParentAnchor
    for i, def in ipairs(SUB_DEFS) do
        local cb = ns.CreateToggleSwitch(panel, "SKTools" .. def.name .. "CB")
        cb:SetPoint("TOPLEFT", lastSub, "BOTTOMLEFT", i == 1 and SUB_INDENT or 0, -8)
        cb.Text:SetText(def.label)
        cb.Text:SetTextColor(0.8, 0.8, 0.8)
        cb:SetChecked(SKToolsDB[def.key])
        cb:SetScript("OnClick", function(self)
            SKToolsDB[def.key] = self:GetChecked()
            if ns.RefreshMainNavStatus then ns.RefreshMainNavStatus() end
        end)
        local desc = panel:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        desc:SetPoint("TOPLEFT", cb.Text, "BOTTOMLEFT", 0, -2)
        desc:SetPoint("RIGHT", panel, "RIGHT", -16, 0)
        desc:SetJustifyH("LEFT")
        desc:SetTextColor(0.55, 0.55, 0.6)
        desc:SetText(def.desc)
        subToggles[#subToggles + 1] = cb
        local spacer = CreateFrame("Frame", nil, panel)
        spacer:SetSize(1, 1)
        spacer:SetPoint("LEFT", cb, "LEFT", 0, 0)
        spacer:SetPoint("TOP", desc, "BOTTOM", 0, 0)
        lastSub = spacer
    end

    -- Connecting line from parent to sub-toggles
    local connLine = panel:CreateTexture(nil, "ARTWORK")
    connLine:SetWidth(1)
    connLine:SetPoint("TOP", subToggles[1], "TOPLEFT", -8, 4)
    connLine:SetPoint("BOTTOM", subToggles[#subToggles], "LEFT", -8, 0)
    connLine:SetColorTexture(ns.COLORS.divider[1], ns.COLORS.divider[2], ns.COLORS.divider[3], 0.30)

    -- Enable/disable sub-toggles based on parent
    local function UpdateBIChildState()
        local enabled = biParentCB:GetChecked()
        for _, cb in ipairs(subToggles) do cb:SetEnabled(enabled) end
    end
    biParentCB:HookScript("OnClick", UpdateBIChildState)
    ns.RefreshMainBIChildState = UpdateBIChildState
    UpdateBIChildState()

    -- Return to parent's left edge
    local subReturnSpacer = CreateFrame("Frame", nil, panel)
    subReturnSpacer:SetSize(1, 1)
    subReturnSpacer:SetPoint("LEFT", biParentCB, "LEFT", 0, 0)
    subReturnSpacer:SetPoint("TOP", lastSub, "TOP", 0, 0)
    SetLast(subReturnSpacer)

    -- Reset Position button
    do
        local btnRow = CreateFrame("Frame", nil, panel)
        btnRow:SetSize(300, 22)
        btnRow:SetPoint("TOPLEFT", GetLast(), "BOTTOMLEFT", 0, -8)

        local resetBtn = ns.CreateThemedButton(btnRow, "Reset Position", 120, 22, "secondary")
        resetBtn:SetPoint("LEFT", 0, 0)
        resetBtn:SetScript("OnClick", function()
            SKToolsDB.battleIntelPos = nil
            if anchorFrame then
                anchorFrame:ClearAllPoints()
                anchorFrame:SetPoint(DEFAULT_POS.point, UIParent, DEFAULT_POS.relPoint, DEFAULT_POS.x, DEFAULT_POS.y)
            end
        end)

        local testBtn = ns.CreateThemedButton(btnRow, "Test Bars", 90, 22, "secondary")
        testBtn:SetPoint("LEFT", resetBtn, "RIGHT", 6, 0)
        local testShowing = false
        testBtn:SetScript("OnClick", function()
            if testShowing then
                HideTestBars()
                testBtn:SetText("Test Bars")
            else
                ShowTestBars()
                testBtn:SetText("Hide Test")
            end
            testShowing = not testShowing
        end)

        SetLast(btnRow)
    end

    card:SetPoint("BOTTOM", GetLast(), "BOTTOM", 0, -8)
    card:SetPoint("RIGHT", panel, "RIGHT", -8, 0)
    return GetLast()
end

function ns.BuildBattleIntelCombatSettings(panel, csSyncControls)
    local anchor = ns.CreateTabTitle(panel, "Battle Intel", "Real-time battleground intelligence.")

    local header = ns.AddSectionHeader(panel, anchor, "Battle Intel", false)
    local card = ns.CreateSectionCard(panel)
    card:SetPoint("TOPLEFT", header, "TOPLEFT", -8, 8)

    local AddCB, GetLast, SetLast = ns.MakeCombatCBFactory(panel, header, csSyncControls)

    local csBIParentCB = AddCB("BattleIntel", "battleIntel", "Battle Intel",
        "Real-time battleground intelligence with capture timers, score prediction, and flag tracking.",
        ns.SetBattleIntel)

    local csBIParentAnchor = GetLast()

    -- Sub-toggles (indented children of Battle Intel)
    local CS_SUB_INDENT = 24
    local CS_SUB_DEFS = {
        { name = "BIScorePrediction", key = "battleIntelShowPrediction", label = "Score Prediction",
          desc = "Show predicted winner and estimated time to win based on current score rates." },
        { name = "BICaptureTimers",   key = "battleIntelShowCaptures",   label = "Capture Timers",
          desc = "Show countdown bars when nodes or bases are being captured." },
        { name = "BIFlagRespawn",     key = "battleIntelShowFlags",      label = "Flag Respawn Timers",
          desc = "Show flag respawn countdown after a capture in WSG, Twin Peaks, and Eye of the Storm." },
    }
    local csSubToggles = {}
    local csLastSub = csBIParentAnchor
    for i, def in ipairs(CS_SUB_DEFS) do
        local cb = ns.CreateToggleSwitch(panel, "SKToolsCS_" .. def.name .. "CB")
        cb:SetPoint("TOPLEFT", csLastSub, "BOTTOMLEFT", i == 1 and CS_SUB_INDENT or 0, -8)
        cb.Text:SetText(def.label)
        cb.Text:SetTextColor(0.8, 0.8, 0.8)
        cb:SetChecked(SKToolsDB[def.key])
        cb:SetScript("OnClick", function(self)
            SKToolsDB[def.key] = self:GetChecked()
            if ns.RefreshCombatNavStatus then ns.RefreshCombatNavStatus() end
        end)
        local desc = panel:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
        desc:SetPoint("TOPLEFT", cb.Text, "BOTTOMLEFT", 0, -2)
        desc:SetPoint("RIGHT", panel, "RIGHT", -16, 0)
        desc:SetJustifyH("LEFT")
        desc:SetTextColor(0.55, 0.55, 0.6)
        desc:SetText(def.desc)
        table.insert(csSyncControls, { type = "checkbox", widget = cb, key = def.key })
        csSubToggles[#csSubToggles + 1] = cb
        local spacer = CreateFrame("Frame", nil, panel)
        spacer:SetSize(1, 1)
        spacer:SetPoint("LEFT", cb, "LEFT", 0, 0)
        spacer:SetPoint("TOP", desc, "BOTTOM", 0, 0)
        csLastSub = spacer
    end

    -- Connecting line
    local csConnLine = panel:CreateTexture(nil, "ARTWORK")
    csConnLine:SetWidth(1)
    csConnLine:SetPoint("TOP", csSubToggles[1], "TOPLEFT", -8, 4)
    csConnLine:SetPoint("BOTTOM", csSubToggles[#csSubToggles], "LEFT", -8, 0)
    csConnLine:SetColorTexture(ns.COLORS.divider[1], ns.COLORS.divider[2], ns.COLORS.divider[3], 0.30)

    -- Enable/disable sub-toggles based on parent
    local function UpdateCSBIChildState()
        local enabled = csBIParentCB:GetChecked()
        for _, cb in ipairs(csSubToggles) do cb:SetEnabled(enabled) end
    end
    csBIParentCB:HookScript("OnClick", UpdateCSBIChildState)
    table.insert(csSyncControls, { refresh = UpdateCSBIChildState })
    UpdateCSBIChildState()

    -- Return to parent's left edge
    local csSubReturn = CreateFrame("Frame", nil, panel)
    csSubReturn:SetSize(1, 1)
    csSubReturn:SetPoint("LEFT", csBIParentCB, "LEFT", 0, 0)
    csSubReturn:SetPoint("TOP", csLastSub, "TOP", 0, 0)
    SetLast(csSubReturn)

    -- Reset Position button
    do
        local btnRow = CreateFrame("Frame", nil, panel)
        btnRow:SetSize(300, 22)
        btnRow:SetPoint("TOPLEFT", GetLast(), "BOTTOMLEFT", 0, -8)

        local resetBtn = ns.CreateThemedButton(btnRow, "Reset Position", 120, 22, "secondary")
        resetBtn:SetPoint("LEFT", 0, 0)
        resetBtn:SetScript("OnClick", function()
            SKToolsDB.battleIntelPos = nil
            if anchorFrame then
                anchorFrame:ClearAllPoints()
                anchorFrame:SetPoint(DEFAULT_POS.point, UIParent, DEFAULT_POS.relPoint, DEFAULT_POS.x, DEFAULT_POS.y)
            end
        end)

        local testBtn = ns.CreateThemedButton(btnRow, "Test Bars", 90, 22, "secondary")
        testBtn:SetPoint("LEFT", resetBtn, "RIGHT", 6, 0)
        local testShowing = false
        testBtn:SetScript("OnClick", function()
            if testShowing then
                HideTestBars()
                testBtn:SetText("Test Bars")
            else
                ShowTestBars()
                testBtn:SetText("Hide Test")
            end
            testShowing = not testShowing
        end)

        SetLast(btnRow)
    end

    card:SetPoint("BOTTOM", GetLast(), "BOTTOM", 0, -8)
    card:SetPoint("RIGHT", panel, "RIGHT", -8, 0)
    return GetLast()
end
