-- SKTools Arena Notes
-- Scouting report system: layered notes per comp, vs comp, matchup, and per-spec

local _, ns = ...

local CYAN = { r = 0, g = 0.898, b = 0.933 }

-----------------------------
-- Class Data (duplicated from PvPHistory.lua — locals can't be shared)
-----------------------------
local CLASS_COLORS = {
    WARRIOR     = { r = 0.78, g = 0.61, b = 0.43 },
    PALADIN     = { r = 0.96, g = 0.55, b = 0.73 },
    HUNTER      = { r = 0.67, g = 0.83, b = 0.45 },
    ROGUE       = { r = 1.00, g = 0.96, b = 0.41 },
    PRIEST      = { r = 1.00, g = 1.00, b = 1.00 },
    DEATHKNIGHT = { r = 0.77, g = 0.12, b = 0.23 },
    SHAMAN      = { r = 0.00, g = 0.44, b = 0.87 },
    MAGE        = { r = 0.25, g = 0.78, b = 0.92 },
    WARLOCK     = { r = 0.53, g = 0.53, b = 0.93 },
    MONK        = { r = 0.00, g = 1.00, b = 0.60 },
    DRUID       = { r = 1.00, g = 0.49, b = 0.04 },
    DEMONHUNTER = { r = 0.64, g = 0.19, b = 0.79 },
    EVOKER      = { r = 0.20, g = 0.58, b = 0.50 },
}

local CLASS_ICON_ATLAS = {
    WARRIOR     = "classicon-warrior",
    PALADIN     = "classicon-paladin",
    HUNTER      = "classicon-hunter",
    ROGUE       = "classicon-rogue",
    PRIEST      = "classicon-priest",
    DEATHKNIGHT = "classicon-deathknight",
    SHAMAN      = "classicon-shaman",
    MAGE        = "classicon-mage",
    WARLOCK     = "classicon-warlock",
    MONK        = "classicon-monk",
    DRUID       = "classicon-druid",
    DEMONHUNTER = "classicon-demonhunter",
    EVOKER      = "classicon-evoker",
}

local function ClassIconString(classFile, size)
    size = size or 14
    local atlas = CLASS_ICON_ATLAS[classFile]
    if atlas then
        return CreateAtlasMarkup(atlas, size, size)
    end
    return ""
end

local function CompIconString(team, size)
    size = size or 14
    local parts = {}
    for _, p in ipairs(team) do
        local icon = ClassIconString(p.class, size)
        if icon ~= "" then
            parts[#parts + 1] = icon
        end
    end
    return table.concat(parts, " ")
end

local CLASS_DISPLAY_NAMES = {
    WARRIOR     = "Warrior",
    PALADIN     = "Paladin",
    HUNTER      = "Hunter",
    ROGUE       = "Rogue",
    PRIEST      = "Priest",
    DEATHKNIGHT = "Death Knight",
    SHAMAN      = "Shaman",
    MAGE        = "Mage",
    WARLOCK     = "Warlock",
    MONK        = "Monk",
    DRUID       = "Druid",
    DEMONHUNTER = "Demon Hunter",
    EVOKER      = "Evoker",
}

local BG_MAP_NAMES = {
    "Alterac Valley",
    "Arathi Basin",
    "Battle for Gilneas",
    "Deepwind Gorge",
    "Eye of the Storm",
    "Isle of Conquest",
    "Silvershard Mines",
    "Temple of Kotmogu",
    "Twin Peaks",
    "Warsong Gulch",
}

local function GetClassColor(classFile)
    if RAID_CLASS_COLORS and RAID_CLASS_COLORS[classFile] then
        local c = RAID_CLASS_COLORS[classFile]
        return c.r, c.g, c.b
    end
    local c = CLASS_COLORS[classFile]
    if c then return c.r, c.g, c.b end
    return 0.5, 0.5, 0.5
end

local function GetClassColorHex(classFile)
    local r, g, b = GetClassColor(classFile)
    return string.format("%02x%02x%02x", r * 255, g * 255, b * 255)
end

local function ClassDisplayName(classFile)
    return CLASS_DISPLAY_NAMES[classFile] or classFile
end

-----------------------------
-- Key Generation
-----------------------------
local function MakeCompKey(team)
    local entries = {}
    for _, p in ipairs(team) do
        local spec = (p.spec and p.spec ~= "") and p.spec or "Unknown"
        entries[#entries + 1] = spec .. " " .. (p.class or "UNKNOWN")
    end
    table.sort(entries)
    return table.concat(entries, "/")
end

local function MakeMatchupKey(myTeam, enemyTeam)
    return MakeCompKey(myTeam) .. "|" .. MakeCompKey(enemyTeam)
end

local function MakeSpecKey(entry)
    local spec = (entry.spec and entry.spec ~= "") and entry.spec or "Unknown"
    return spec .. " " .. (entry.class or "UNKNOWN")
end

-- Parse a comp key to extract class tokens for icons
local function CompKeyToTeam(key)
    local team = {}
    for token in key:gmatch("[^/]+") do
        local spec, class = token:match("^(.+) (%u+)$")
        team[#team + 1] = { spec = spec or "Unknown", class = class or "UNKNOWN" }
    end
    return team
end

-- Display-friendly versions (MAGE → Mage, DEATHKNIGHT → Death Knight)
local function SpecKeyToDisplay(key)
    local spec, class = key:match("^(.+) (%u+)$")
    if spec and class then
        return spec .. " " .. ClassDisplayName(class)
    end
    return key
end

local function CompKeyToDisplay(key)
    local parts = {}
    for token in key:gmatch("[^/]+") do
        parts[#parts + 1] = SpecKeyToDisplay(token)
    end
    return table.concat(parts, " / ")
end

-----------------------------
-- Team Detection
-----------------------------
local function DetectPlayerTeam()
    local team = {}
    -- Player
    local _, playerClass = UnitClass("player")
    local playerSpec = "Unknown"
    local specIndex = GetSpecialization()
    if specIndex then
        local _, specName = GetSpecializationInfo(specIndex)
        if specName then playerSpec = specName end
    end
    team[#team + 1] = { class = playerClass or "UNKNOWN", spec = playerSpec }

    -- Party members
    for i = 1, 4 do
        local unit = "party" .. i
        if UnitExists(unit) then
            local _, partyClass = UnitClass(unit)
            local partySpec = "Unknown"
            local inspectSpec = GetInspectSpecialization(unit)
            if inspectSpec and inspectSpec > 0 then
                local _, specName, _, _, _, classFile = GetSpecializationInfoByID(inspectSpec)
                if specName then partySpec = specName end
                if classFile then partyClass = classFile end
            end
            team[#team + 1] = { class = partyClass or "UNKNOWN", spec = partySpec }
        end
    end
    return team
end

local function DetectEnemyTeam()
    local team = {}
    for i = 1, 5 do
        local specID = GetArenaOpponentSpec(i)
        if specID and specID > 0 then
            local _, specName, _, _, _, classFile = GetSpecializationInfoByID(specID)
            team[#team + 1] = {
                class = classFile or "UNKNOWN",
                spec = specName or "Unknown",
            }
        end
    end
    return team
end

-----------------------------
-- All Specs (for browser prepopulation & comp builder)
-----------------------------
local allSpecsByClass  -- lazy init
local function GetAllSpecsByClass()
    if allSpecsByClass and #allSpecsByClass > 0 then return allSpecsByClass end
    allSpecsByClass = {}
    for classID = 1, GetNumClasses() do
        local className, classFile = GetClassInfo(classID)
        if classFile then
            local entry = { classFile = classFile, className = className, specs = {} }
            local numSpecs = GetNumSpecializationsForClassID(classID)
            for specIdx = 1, numSpecs do
                local _, specName, _, specIcon = GetSpecializationInfoForClassID(classID, specIdx)
                if specName then
                    entry.specs[#entry.specs + 1] = {
                        name = specName,
                        class = classFile,
                        key = specName .. " " .. classFile,
                        icon = specIcon,
                    }
                end
            end
            if #entry.specs > 0 then
                allSpecsByClass[#allSpecsByClass + 1] = entry
            end
        end
    end
    return allSpecsByClass
end

-- Tank specs (sorted to bottom of the spec list)
local TANK_SPECS = {
    ["Protection WARRIOR"] = true,
    ["Protection PALADIN"] = true,
    ["Blood DEATHKNIGHT"] = true,
    ["Vengeance DEMONHUNTER"] = true,
    ["Brewmaster MONK"] = true,
    ["Guardian DRUID"] = true,
}

local function GetAllSpecKeys()
    local nonTank = {}
    local tank = {}
    for _, classEntry in ipairs(GetAllSpecsByClass()) do
        for _, spec in ipairs(classEntry.specs) do
            if TANK_SPECS[spec.key] then
                tank[#tank + 1] = { key = spec.key, class = classEntry.classFile }
            else
                nonTank[#nonTank + 1] = { key = spec.key, class = classEntry.classFile }
            end
        end
    end
    -- Sort within each group: by class name, then spec name
    local function sortByClass(a, b)
        if a.class ~= b.class then return a.class < b.class end
        return a.key < b.key
    end
    table.sort(nonTank, sortByClass)
    table.sort(tank, sortByClass)
    -- Combine: non-tanks first, then tanks
    local keys = {}
    for _, v in ipairs(nonTank) do keys[#keys + 1] = v.key end
    for _, v in ipairs(tank) do keys[#keys + 1] = v.key end
    return keys
end

-----------------------------
-- State
-----------------------------
local eventFrame
local prepFrame, postMatchFrame, browserFrame
local currentPlayerTeam, currentEnemyTeam
local inPvPInstance = false
local currentBGMapName = nil

-----------------------------
-- DB Access
-----------------------------
local function EnsureDB()
    if not SKToolsNotesDB then
        SKToolsNotesDB = {}
    end
    if not SKToolsNotesDB.compNotes then SKToolsNotesDB.compNotes = {} end
    if not SKToolsNotesDB.vsCompNotes then SKToolsNotesDB.vsCompNotes = {} end
    if not SKToolsNotesDB.matchupNotes then SKToolsNotesDB.matchupNotes = {} end
    if not SKToolsNotesDB.specNotes then SKToolsNotesDB.specNotes = {} end
    if not SKToolsNotesDB.lastModified then SKToolsNotesDB.lastModified = {} end
    if not SKToolsNotesDB.compTitles then SKToolsNotesDB.compTitles = {} end
    if not SKToolsNotesDB.bgNotes then SKToolsNotesDB.bgNotes = {} end
    for _, name in ipairs(BG_MAP_NAMES) do
        if SKToolsNotesDB.bgNotes[name] == nil then
            SKToolsNotesDB.bgNotes[name] = ""
        end
    end
    if not SKToolsNotesDB.dungeonNotes then SKToolsNotesDB.dungeonNotes = {} end
    if not SKToolsNotesDB.raidNotes then SKToolsNotesDB.raidNotes = {} end
    if not SKToolsNotesDB.collapsedSections then SKToolsNotesDB.collapsedSections = {} end
    if SKToolsNotesDB.browserShowPvP == nil then SKToolsNotesDB.browserShowPvP = true end
    if SKToolsNotesDB.browserShowPvE == nil then SKToolsNotesDB.browserShowPvE = true end
    if not SKToolsNotesDB.generalNotes then SKToolsNotesDB.generalNotes = {} end
    -- Migrate old genericNotes into generalNotes
    if SKToolsNotesDB.genericNotes and SKToolsNotesDB.genericNotes ~= "" then
        if not SKToolsNotesDB.generalNotes["General Notes"] or SKToolsNotesDB.generalNotes["General Notes"] == "" then
            SKToolsNotesDB.generalNotes["General Notes"] = SKToolsNotesDB.genericNotes
        end
        SKToolsNotesDB.genericNotes = nil
    end
    -- Always ensure "General Notes" entry exists
    if SKToolsNotesDB.generalNotes["General Notes"] == nil then
        SKToolsNotesDB.generalNotes["General Notes"] = ""
    end
end

-- Record a timestamp when a note is saved
local function TouchNote(noteType, key)
    EnsureDB()
    local ts = time()
    if not SKToolsNotesDB.lastModified[noteType] then
        SKToolsNotesDB.lastModified[noteType] = {}
    end
    SKToolsNotesDB.lastModified[noteType][key] = ts
end

-- Get last modified timestamp for a note
local function GetNoteTimestamp(noteType, key)
    EnsureDB()
    local tbl = SKToolsNotesDB.lastModified[noteType]
    return tbl and tbl[key]
end

-- Remove timestamp when a note is deleted
local function ClearNoteTimestamp(noteType, key)
    EnsureDB()
    local tbl = SKToolsNotesDB.lastModified[noteType]
    if tbl then tbl[key] = nil end
end

-- Format a relative time string ("just now", "2h ago", "3d ago")
local function FormatRelativeTime(ts)
    if not ts then return nil end
    local diff = time() - ts
    if diff < 60 then return "just now" end
    if diff < 3600 then return math.floor(diff / 60) .. "m ago" end
    if diff < 86400 then return math.floor(diff / 3600) .. "h ago" end
    if diff < 2592000 then return math.floor(diff / 86400) .. "d ago" end
    return date("%b %d", ts)
end

-- Get/set comp title (shared between comp, vscomp, and matchup)
local function GetCompTitle(compKey)
    EnsureDB()
    local t = SKToolsNotesDB.compTitles[compKey]
    if t and t ~= "" then return t end
    return nil
end

local function SetCompTitle(compKey, title)
    EnsureDB()
    if title and title ~= "" then
        SKToolsNotesDB.compTitles[compKey] = title
    else
        SKToolsNotesDB.compTitles[compKey] = nil
    end
end

-----------------------------
-- Shared UI Helpers
-----------------------------
local BACKDROP_INFO = {
    bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
    edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
    tile = true, tileSize = 16, edgeSize = 14,
    insets = { left = 4, right = 4, top = 4, bottom = 4 },
}

-- Shared UI helpers from Core.lua
local CreateThemedButton = ns.CreateThemedButton
local StyleScrollBar = ns.StyleScrollBar
local CreateCyanAccentLine = ns.CreateCyanAccentLine
local FadeIn = ns.FadeIn
local FadeOut = ns.FadeOut

local function CreateDragTitleBar(parent, titleText, onClose, onDragStop)
    local titleBar = CreateFrame("Frame", nil, parent)
    titleBar:SetHeight(24)
    titleBar:SetPoint("TOPLEFT", parent, "TOPLEFT", 0, 0)
    titleBar:SetPoint("TOPRIGHT", parent, "TOPRIGHT", 0, 0)
    titleBar:EnableMouse(true)
    titleBar:RegisterForDrag("LeftButton")
    titleBar:SetScript("OnDragStart", function() parent:StartMoving() end)
    titleBar:SetScript("OnDragStop", function()
        parent:StopMovingOrSizing()
        if onDragStop then onDragStop() end
    end)

    local title = titleBar:CreateFontString(nil, "OVERLAY", "GameFontNormal")
    title:SetPoint("LEFT", titleBar, "LEFT", 10, 0)
    title:SetText(titleText)
    title:SetTextColor(CYAN.r, CYAN.g, CYAN.b)

    local closeBtn = CreateFrame("Button", nil, parent, "BackdropTemplate")
    closeBtn:SetSize(22, 22)
    closeBtn:SetPoint("TOPRIGHT", parent, "TOPRIGHT", -5, -4)
    closeBtn:SetBackdrop({
        bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
        edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
        tile = true, tileSize = 8, edgeSize = 8,
        insets = { left = 1, right = 1, top = 1, bottom = 1 },
    })
    closeBtn:SetBackdropColor(0.12, 0.12, 0.14, 0.6)
    closeBtn:SetBackdropBorderColor(0.25, 0.25, 0.3, 0.4)
    local closeLbl = closeBtn:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
    closeLbl:SetPoint("CENTER", 0, 1)
    closeLbl:SetText("x")
    closeLbl:SetTextColor(0.6, 0.6, 0.6)
    closeBtn:SetScript("OnEnter", function(self)
        self:SetBackdropColor(0.5, 0.1, 0.1, 0.8)
        self:SetBackdropBorderColor(0.7, 0.2, 0.2, 0.6)
        closeLbl:SetTextColor(1, 1, 1)
    end)
    closeBtn:SetScript("OnLeave", function(self)
        self:SetBackdropColor(0.12, 0.12, 0.14, 0.6)
        self:SetBackdropBorderColor(0.25, 0.25, 0.3, 0.4)
        closeLbl:SetTextColor(0.6, 0.6, 0.6)
    end)
    closeBtn:SetScript("OnClick", function()
        if onClose then onClose() else parent:Hide() end
    end)
    -- Raise above the title bar so it receives clicks
    closeBtn:SetFrameLevel(titleBar:GetFrameLevel() + 5)

    return titleBar, title
end

-- Create a note EditBox section with label + icons
local function CreateNoteSection(parent, labelText, dbTable, dbKey, numVisibleLines, yOffset, anchorTo)
    local section = CreateFrame("Frame", nil, parent)
    section:SetPoint("TOPLEFT", anchorTo or parent, anchorTo and "BOTTOMLEFT" or "TOPLEFT", 0, yOffset or 0)
    section:SetPoint("RIGHT", parent, "RIGHT", -12, 0)

    -- Label (icon string + text)
    local label = section:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
    label:SetPoint("TOPLEFT", section, "TOPLEFT", 12, 0)
    label:SetJustifyH("LEFT")
    label:SetText(labelText)
    label:SetTextColor(0.85, 0.85, 0.85)
    section.label = label

    -- EditBox container with border (Button so clicks anywhere focus the editbox)
    local boxHeight = math.max(numVisibleLines * 14 + 12, 34)
    local container = CreateFrame("Button", nil, section, "BackdropTemplate")
    container:SetPoint("TOPLEFT", label, "BOTTOMLEFT", 0, -6)
    container:SetPoint("RIGHT", section, "RIGHT", 0, 0)
    container:SetHeight(boxHeight)
    container:SetBackdrop({
        bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
        edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
        tile = true, tileSize = 8, edgeSize = 10,
        insets = { left = 2, right = 2, top = 2, bottom = 2 },
    })
    container:SetBackdropColor(0.04, 0.04, 0.06, 0.95)
    container:SetBackdropBorderColor(CYAN.r * 0.25, CYAN.g * 0.25, CYAN.b * 0.25, 0.5)
    container:RegisterForClicks("LeftButtonUp")
    section.container = container

    -- "Has notes" accent bar (left edge of container)
    local accent = container:CreateTexture(nil, "OVERLAY")
    accent:SetWidth(3)
    accent:SetPoint("TOPLEFT", container, "TOPLEFT", 1, -1)
    accent:SetPoint("BOTTOMLEFT", container, "BOTTOMLEFT", 1, 1)
    accent:SetColorTexture(CYAN.r, CYAN.g, CYAN.b, 0.8)
    accent:Hide()

    -- Scrollframe for the editbox
    local scrollFrame = CreateFrame("ScrollFrame", nil, container, "UIPanelScrollFrameTemplate")
    scrollFrame:SetPoint("TOPLEFT", container, "TOPLEFT", 8, -5)
    scrollFrame:SetPoint("BOTTOMRIGHT", container, "BOTTOMRIGHT", -18, 5)
    StyleScrollBar(scrollFrame)

    local editBox = CreateFrame("EditBox", nil, scrollFrame)
    editBox:SetMultiLine(true)
    editBox:SetAutoFocus(false)
    editBox:SetFontObject("ChatFontNormal")
    editBox:SetWidth(scrollFrame:GetWidth() or 300)
    editBox:SetScript("OnEscapePressed", function(self) self:ClearFocus() end)
    scrollFrame:SetScrollChild(editBox)

    -- Click anywhere in container to focus the editbox
    container:SetScript("OnClick", function() editBox:SetFocus() end)

    -- Placeholder text
    local placeholder = editBox:CreateFontString(nil, "ARTWORK", "GameFontDisableSmall")
    placeholder:SetPoint("TOPLEFT", editBox, "TOPLEFT", 2, -2)
    placeholder:SetText("Add notes...")
    placeholder:SetTextColor(0.45, 0.45, 0.45)
    section.placeholder = placeholder

    -- Update placeholder visibility
    local function UpdatePlaceholder()
        if editBox:GetText() == "" and not editBox:HasFocus() then
            placeholder:Show()
        else
            placeholder:Hide()
        end
    end

    -- Note type (for timestamp tracking), set via SetNote
    local noteType = nil

    -- Save function
    local function SaveNote()
        if dbTable and dbKey then
            local text = editBox:GetText()
            if text == "" then
                dbTable[dbKey] = nil
                if noteType then ClearNoteTimestamp(noteType, dbKey) end
            else
                dbTable[dbKey] = text
                if noteType then TouchNote(noteType, dbKey) end
            end
        end
    end

    -- Debounced save on text change (1s timer)
    local saveTimer = nil
    editBox:SetScript("OnTextChanged", function(self, userInput)
        UpdatePlaceholder()
        if not userInput then return end
        if saveTimer then saveTimer:Cancel() end
        saveTimer = C_Timer.NewTimer(1.0, function()
            SaveNote()
            saveTimer = nil
        end)
    end)

    -- Save on focus lost
    editBox:SetScript("OnEditFocusLost", function(self)
        if saveTimer then saveTimer:Cancel(); saveTimer = nil end
        SaveNote()
        UpdatePlaceholder()
    end)
    editBox:SetScript("OnEditFocusGained", function(self)
        placeholder:Hide()
    end)

    -- Size the editbox width dynamically
    scrollFrame:SetScript("OnSizeChanged", function(self, w)
        editBox:SetWidth(w)
    end)

    section.editBox = editBox
    section.scrollFrame = scrollFrame

    -- Total section height: label + gap + box
    local totalH = label:GetStringHeight() + 6 + boxHeight
    section:SetHeight(totalH)

    -- Populate/update function
    section.SetNote = function(self, tbl, key, placeholderText, nType)
        dbTable = tbl
        dbKey = key
        noteType = nType
        self.currentDbKey = key
        if placeholderText then placeholder:SetText(placeholderText) end
        local existing = tbl and key and tbl[key]
        editBox:SetText(existing or "")
        UpdatePlaceholder()
    end

    section.SetLabel = function(self, text)
        label:SetText(text)
    end

    section.SetLabelColor = function(self, r, g, b)
        label:SetTextColor(r, g, b)
    end

    section.SetHasNotes = function(self, hasNotes)
        if hasNotes then accent:Show() else accent:Hide() end
    end

    section.GetSectionHeight = function(self)
        return (label:GetStringHeight() or 14) + 6 + boxHeight
    end

    UpdatePlaceholder()
    return section
end

-- Forward declaration (CreateBrowserFrame is defined later but referenced in prep frame OnClick)
local CreateBrowserFrame

-----------------------------
-- Prep Frame
-----------------------------
local function SavePrepFramePos()
    if prepFrame then
        local point, _, relPoint, x, y = prepFrame:GetPoint()
        SKToolsNotesDB.prepFramePos = { point = point, relPoint = relPoint, x = x, y = y }
    end
end

local prepNoteSections = {}  -- pool of reusable note sections
local prepCatHeaders = {}    -- pool of category header FontStrings
local prepCatDividers = {}   -- pool of category divider textures
local prepHistoryLines = {}  -- pool of read-only FontStrings for match history

local function CreatePrepFrame()
    if prepFrame then return prepFrame end
    EnsureDB()

    prepFrame = CreateFrame("Frame", "SKToolsArenaPrepNotes", UIParent, "BackdropTemplate")
    prepFrame:SetSize(650, 700)
    prepFrame:SetFrameStrata("DIALOG")
    prepFrame:SetMovable(true)
    prepFrame:SetClampedToScreen(true)
    prepFrame:EnableMouse(true)
    prepFrame:SetBackdrop(BACKDROP_INFO)
    prepFrame:SetBackdropColor(0.06, 0.06, 0.08, 0.95)
    prepFrame:SetBackdropBorderColor(0.2, 0.2, 0.25, 0.9)
    pcall(tinsert, UISpecialFrames, "SKToolsArenaPrepNotes")

    -- Position
    local pos = SKToolsNotesDB.prepFramePos
    if pos then
        prepFrame:SetPoint(pos.point or "RIGHT", UIParent, pos.relPoint or "RIGHT", pos.x or -50, pos.y or 0)
    else
        prepFrame:SetPoint("RIGHT", UIParent, "RIGHT", -50, 0)
    end

    prepFrame:SetToplevel(true)
    prepFrame:SetScript("OnShow", function(self) self:Raise() end)

    CreateCyanAccentLine(prepFrame)
    local _, prepTitleText = CreateDragTitleBar(prepFrame, "Arena Notes", function() prepFrame:Hide() end, SavePrepFramePos)
    prepFrame.titleText = prepTitleText

    -- "All Notes" toggle button (switch to full browser)
    local allNotesBtn = CreateThemedButton(prepFrame, "All Notes", 72, 18, "secondary")
    allNotesBtn:SetPoint("TOPRIGHT", prepFrame, "TOPRIGHT", -30, -5)
    allNotesBtn:SetFrameLevel(allNotesBtn:GetFrameLevel() + 5)
    allNotesBtn:SetScript("OnClick", function()
        prepFrame:Hide()
        CreateBrowserFrame()
        if browserFrame.matchNotesBtn then browserFrame.matchNotesBtn:Show() end
        FadeIn(browserFrame, 0.2)
        SKArenaNotes_RefreshBrowser()
    end)

    -- Scrollable content area
    local scrollFrame = CreateFrame("ScrollFrame", nil, prepFrame, "UIPanelScrollFrameTemplate")
    scrollFrame:SetPoint("TOPLEFT", prepFrame, "TOPLEFT", 6, -30)
    scrollFrame:SetPoint("BOTTOMRIGHT", prepFrame, "BOTTOMRIGHT", -18, 8)
    StyleScrollBar(scrollFrame)

    local content = CreateFrame("Frame", nil, scrollFrame)
    content:SetWidth(610)
    content:SetHeight(1)  -- will be resized dynamically
    scrollFrame:SetScrollChild(content)

    prepFrame.content = content
    prepFrame.scrollFrame = scrollFrame

    -- Mouse wheel scrolling
    prepFrame:EnableMouseWheel(true)
    prepFrame:SetScript("OnMouseWheel", function(self, delta)
        local current = scrollFrame:GetVerticalScroll()
        local maxScroll = scrollFrame:GetVerticalScrollRange()
        scrollFrame:SetVerticalScroll(math.max(0, math.min(current - delta * 30, maxScroll)))
    end)

    prepFrame:Hide()
    return prepFrame
end

-- Sections are progressively revealed based on available data:
--   YOUR COMP:   only when player team specs are fully resolved
--   ENEMY COMP:  only when all enemies detected
--   MATCHUP:     only when BOTH teams fully known
--   ENEMY SPECS: per detected enemy (no dependency on player team)
local function PopulateAndShowPrepNotes(playerTeam, enemyTeam, expectedEnemies, playerTeamComplete)
    EnsureDB()
    CreatePrepFrame()

    -- Restore title in case it was changed by ShowInstancePrepNote
    if prepFrame.titleText then
        prepFrame.titleText:SetText("Arena Notes")
    end

    currentPlayerTeam = playerTeam
    currentEnemyTeam = enemyTeam

    local content = prepFrame.content

    -- Save which section's editbox has focus (by dbKey) so we can restore it after rebuild
    local focusedKey, focusedCursor
    for _, s in ipairs(prepNoteSections) do
        if s.editBox and s.editBox:HasFocus() then
            focusedKey = s.currentDbKey
            focusedCursor = s.editBox:GetCursorPosition()
            break
        end
    end

    -- Hide all pooled elements
    for _, s in ipairs(prepNoteSections) do s:Hide() end
    for _, h in ipairs(prepCatHeaders) do h:Hide() end
    for _, d in ipairs(prepCatDividers) do d:Hide() end
    for _, l in ipairs(prepHistoryLines) do l:Hide() end

    local secIdx, hdrIdx, histLineIdx = 0, 0, 0

    -- Reuse or create a read-only FontString for match history lines
    local function NextHistoryLine(text, yOff, indent)
        histLineIdx = histLineIdx + 1
        local fs = prepHistoryLines[histLineIdx]
        if not fs then
            fs = content:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
            fs:SetJustifyH("LEFT")
            fs:SetWidth(600)
            prepHistoryLines[histLineIdx] = fs
        end
        fs:ClearAllPoints()
        fs:SetPoint("TOPLEFT", content, "TOPLEFT", indent or 16, yOff)
        fs:SetText(text)
        fs:Show()
        return (fs:GetStringHeight() or 14) + 3
    end

    -- Reuse or create a category header + divider
    local function NextHeader(text, yOff)
        hdrIdx = hdrIdx + 1
        local h = prepCatHeaders[hdrIdx]
        if not h then
            h = content:CreateFontString(nil, "OVERLAY", "GameFontNormal")
            h:SetJustifyH("LEFT")
            prepCatHeaders[hdrIdx] = h
        end
        h:ClearAllPoints()
        h:SetPoint("TOPLEFT", content, "TOPLEFT", 12, yOff)
        h:SetText(text)
        h:SetTextColor(CYAN.r, CYAN.g, CYAN.b, 1)
        h:Show()

        local d = prepCatDividers[hdrIdx]
        if not d then
            d = content:CreateTexture(nil, "ARTWORK")
            d:SetHeight(1)
            prepCatDividers[hdrIdx] = d
        end
        d:ClearAllPoints()
        d:SetPoint("TOPLEFT", h, "BOTTOMLEFT", -8, -3)
        d:SetPoint("RIGHT", content, "RIGHT", -12, 0)
        d:SetColorTexture(CYAN.r, CYAN.g, CYAN.b, 0.25)
        d:Show()

        return (h:GetStringHeight() or 14) + 8
    end

    -- Reuse or create a note section
    local function NextSection(labelText, dbTbl, dbKey, yOff, placeholder, labelR, labelG, labelB, nType)
        secIdx = secIdx + 1
        local sec = prepNoteSections[secIdx]
        if not sec then
            sec = CreateNoteSection(content, "", nil, nil, 5, 0, nil)
            prepNoteSections[secIdx] = sec
        end
        sec:ClearAllPoints()
        sec:SetPoint("TOPLEFT", content, "TOPLEFT", 0, yOff)
        sec:SetPoint("RIGHT", content, "RIGHT", -12, 0)
        sec:SetLabel(labelText)
        sec:SetNote(dbTbl, dbKey, placeholder, nType)
        sec:SetLabelColor(labelR or 0.85, labelG or 0.85, labelB or 0.85)
        sec:SetHasNotes(dbTbl and dbKey and dbTbl[dbKey] and dbTbl[dbKey] ~= "" and true or false)
        sec:Show()
        return sec:GetSectionHeight()
    end

    -- Default playerTeamComplete to true for callers with full data (post-match, public API)
    if playerTeamComplete == nil then playerTeamComplete = true end

    local allEnemiesDetected = expectedEnemies and #enemyTeam >= expectedEnemies
    local yOff = -4

    -- 1. YOUR COMP — only when player team specs are fully resolved
    if playerTeamComplete then
        local myKey = MakeCompKey(playerTeam)
        local myIcons = CompIconString(playerTeam, 20)
        local myTitle = GetCompTitle(myKey)
        local myDisplay = myTitle and (myIcons .. "  |cffFFD100" .. myTitle .. "|r") or (myIcons .. "  " .. CompKeyToDisplay(myKey))
        yOff = yOff - NextHeader("YOUR COMP", yOff)
        yOff = yOff - NextSection(
            myDisplay,
            SKToolsNotesDB.compNotes, myKey, yOff,
            "Add notes about your comp...", nil, nil, nil, "comp") - 12
    end

    -- 2. ENEMY COMP — only when full enemy team is known
    if allEnemiesDetected then
        local enemyKey = MakeCompKey(enemyTeam)
        local enemyIcons = CompIconString(enemyTeam, 20)
        local enemyTitle = GetCompTitle(enemyKey)
        local enemyDisplay = enemyTitle and (enemyIcons .. "  |cffFFD100" .. enemyTitle .. "|r") or (enemyIcons .. "  " .. CompKeyToDisplay(enemyKey))
        yOff = yOff - NextHeader("ENEMY COMP", yOff)
        yOff = yOff - NextSection(
            enemyDisplay,
            SKToolsNotesDB.vsCompNotes, enemyKey, yOff,
            "Add notes about facing this comp...", nil, nil, nil, "vscomp") - 12

        -- 3. MATCHUP — only when BOTH teams fully known
        if playerTeamComplete then
            local myKey = MakeCompKey(playerTeam)
            local myIcons = CompIconString(playerTeam, 20)
            local matchupKey = MakeMatchupKey(playerTeam, enemyTeam)
            yOff = yOff - NextHeader("MATCHUP", yOff)
            yOff = yOff - NextSection(
                myIcons .. "  vs  " .. enemyIcons,
                SKToolsNotesDB.matchupNotes, matchupKey, yOff,
                "Add notes about this specific matchup...", nil, nil, nil, "matchup") - 12
        end
    end

    -- 3.5. MATCH HISTORY — show stats from PvP History when both teams known
    if playerTeamComplete and allEnemiesDetected and ns.GetMatchupStats then
        local stats = ns.GetMatchupStats(playerTeam, enemyTeam)
        yOff = yOff - NextHeader("MATCH HISTORY", yOff)
        if stats then
            -- Win rate line
            local winColor = stats.winPct >= 50 and "00ff00" or "ff4444"
            local winLine = string.format(
                "|cff%s%dW - %dL  (%.0f%%)|r  |cff888888(%d games)|r",
                winColor, stats.wins, stats.losses, stats.winPct, stats.total
            )
            yOff = yOff - NextHistoryLine(winLine, yOff, 16)

            -- Kill targets (who you usually kill in wins)
            if stats.wins > 0 then
                local killParts = {}
                local sortedKills = {}
                for sk, entry in pairs(stats.killTarget) do
                    sortedKills[#sortedKills + 1] = { specKey = sk, count = entry.count, totalTime = entry.totalTime, timeCount = entry.timeCount }
                end
                table.sort(sortedKills, function(a, b) return a.count > b.count end)
                for _, k in ipairs(sortedKills) do
                    local pct = k.count / stats.wins * 100
                    local spec, class = k.specKey:match("^(.+) (%u+)$")
                    local icon = class and ClassIconString(class, 16) or ""
                    local hex = class and GetClassColorHex(class) or "bbbbbb"
                    local timeStr = ""
                    if k.timeCount > 0 then
                        local avg = k.totalTime / k.timeCount
                        local mins = math.floor(avg / 60)
                        local secs = math.floor(avg % 60)
                        timeStr = "  |cff888888@ " .. string.format("%d:%02d", mins, secs) .. "|r"
                    end
                    killParts[#killParts + 1] = icon .. "|cff" .. hex .. (spec or k.specKey) .. "|r " .. string.format("%.0f%%", pct) .. timeStr
                end
                if #killParts > 0 then
                    yOff = yOff - NextHistoryLine("|cffaaaaaaYou kill:|r  " .. table.concat(killParts, "  "), yOff, 16)
                end
            end

            -- Death targets (who they usually kill in losses)
            if stats.losses > 0 then
                local deathParts = {}
                local sortedDeaths = {}
                for sk, entry in pairs(stats.deathTarget) do
                    sortedDeaths[#sortedDeaths + 1] = { specKey = sk, count = entry.count, totalTime = entry.totalTime, timeCount = entry.timeCount }
                end
                table.sort(sortedDeaths, function(a, b) return a.count > b.count end)
                for _, d in ipairs(sortedDeaths) do
                    local pct = d.count / stats.losses * 100
                    local spec, class = d.specKey:match("^(.+) (%u+)$")
                    local icon = class and ClassIconString(class, 16) or ""
                    local hex = class and GetClassColorHex(class) or "bbbbbb"
                    local timeStr = ""
                    if d.timeCount > 0 then
                        local avg = d.totalTime / d.timeCount
                        local mins = math.floor(avg / 60)
                        local secs = math.floor(avg % 60)
                        timeStr = "  |cff888888@ " .. string.format("%d:%02d", mins, secs) .. "|r"
                    end
                    deathParts[#deathParts + 1] = icon .. "|cff" .. hex .. (spec or d.specKey) .. "|r " .. string.format("%.0f%%", pct) .. timeStr
                end
                if #deathParts > 0 then
                    yOff = yOff - NextHistoryLine("|cffaaaaaaThey kill:|r  " .. table.concat(deathParts, "  "), yOff, 16)
                end
            end

            yOff = yOff - 8
        else
            yOff = yOff - NextHistoryLine("|cff888888No history vs this comp|r", yOff, 16) - 4
        end
    end

    -- 4. ENEMY SPECS — show per detected enemy with progress indicator
    if #enemyTeam > 0 then
        local specHeaderText = "ENEMY SPECS"
        if not allEnemiesDetected and expectedEnemies then
            specHeaderText = specHeaderText .. "  |cff888888(" .. #enemyTeam .. "/" .. expectedEnemies .. " detected)|r"
        end
        yOff = yOff - NextHeader(specHeaderText, yOff)
        for _, enemy in ipairs(enemyTeam) do
            local specKey = MakeSpecKey(enemy)
            local specIcon = ClassIconString(enemy.class, 20)
            local specDisplay = SpecKeyToDisplay(specKey)
            local r, g, b = GetClassColor(enemy.class)
            yOff = yOff - NextSection(
                specIcon .. "  " .. specDisplay,
                SKToolsNotesDB.specNotes, specKey, yOff,
                "Add notes about " .. specDisplay .. "...",
                r, g, b, "spec") - 10
        end
    elseif expectedEnemies and expectedEnemies > 0 then
        -- No enemies yet — show detecting indicator
        yOff = yOff - NextHeader("ENEMY SPECS  |cff888888(detecting...)|r", yOff)
    end

    -- Set content height and resize frame
    content:SetHeight(math.abs(yOff) + 10)
    local desiredH = math.min(math.abs(yOff) + 44, 1200)
    prepFrame:SetHeight(desiredH)

    -- Restore focus to the editbox that had it before rebuild
    if focusedKey then
        for idx = 1, secIdx do
            local s = prepNoteSections[idx]
            if s and s:IsShown() and s.currentDbKey == focusedKey then
                s.editBox:SetFocus()
                if focusedCursor then
                    s.editBox:SetCursorPosition(focusedCursor)
                end
                break
            end
        end
    end

    if not prepFrame:IsShown() then
        FadeIn(prepFrame, 0.2)
    end
end

-- Show instance note when entering a BG, dungeon, or raid
local function ShowInstancePrepNote(mapName, notesTbl, noteType, titleText)
    EnsureDB()
    CreatePrepFrame()

    local content = prepFrame.content

    -- Hide all pooled elements
    for _, s in ipairs(prepNoteSections) do s:Hide() end
    for _, h in ipairs(prepCatHeaders) do h:Hide() end
    for _, d in ipairs(prepCatDividers) do d:Hide() end
    for _, l in ipairs(prepHistoryLines) do l:Hide() end

    -- Update title
    if prepFrame.titleText then
        prepFrame.titleText:SetText(titleText)
    end

    local secIdx, hdrIdx = 0, 0
    local yOff = -4

    -- Header: map name
    hdrIdx = hdrIdx + 1
    local h = prepCatHeaders[hdrIdx]
    if not h then
        h = content:CreateFontString(nil, "OVERLAY", "GameFontNormal")
        h:SetJustifyH("LEFT")
        prepCatHeaders[hdrIdx] = h
    end
    h:ClearAllPoints()
    h:SetPoint("TOPLEFT", content, "TOPLEFT", 12, yOff)
    h:SetText(mapName)
    h:SetTextColor(CYAN.r, CYAN.g, CYAN.b, 1)
    h:Show()

    local d = prepCatDividers[hdrIdx]
    if not d then
        d = content:CreateTexture(nil, "ARTWORK")
        d:SetHeight(1)
        prepCatDividers[hdrIdx] = d
    end
    d:ClearAllPoints()
    d:SetPoint("TOPLEFT", h, "BOTTOMLEFT", -8, -3)
    d:SetPoint("RIGHT", content, "RIGHT", -12, 0)
    d:SetColorTexture(CYAN.r, CYAN.g, CYAN.b, 0.25)
    d:Show()

    yOff = yOff - ((h:GetStringHeight() or 14) + 8)

    -- Note section
    secIdx = secIdx + 1
    local sec = prepNoteSections[secIdx]
    if not sec then
        sec = CreateNoteSection(content, "", nil, nil, 5, 0, nil)
        prepNoteSections[secIdx] = sec
    end
    sec:ClearAllPoints()
    sec:SetPoint("TOPLEFT", content, "TOPLEFT", 0, yOff)
    sec:SetPoint("RIGHT", content, "RIGHT", -12, 0)
    sec:SetLabel(mapName)
    local placeholder = "Write " .. noteType .. " notes here..."
    sec:SetNote(notesTbl, mapName, placeholder, noteType)
    sec:SetLabelColor(0.85, 0.85, 0.85)
    sec:SetHasNotes(notesTbl[mapName] and notesTbl[mapName] ~= "" and true or false)
    sec:Show()
    yOff = yOff - sec:GetSectionHeight() - 12

    -- Set content height and resize frame
    content:SetHeight(math.abs(yOff) + 10)
    local desiredH = math.min(math.abs(yOff) + 44, 1200)
    prepFrame:SetHeight(desiredH)

    if not prepFrame:IsShown() then
        FadeIn(prepFrame, 0.2)
    end
end

-- Public function (full data known, e.g. post-match review)
function SKArenaNotes_ShowPrepNotes(playerTeam, enemyTeam)
    PopulateAndShowPrepNotes(playerTeam, enemyTeam, #enemyTeam)
end

-----------------------------
-- Post-Match Prompt
-----------------------------
local function CreatePostMatchFrame()
    if postMatchFrame then return postMatchFrame end

    postMatchFrame = CreateFrame("Frame", "SKToolsPostMatchPrompt", UIParent, "BackdropTemplate")
    postMatchFrame:SetSize(400, 110)
    postMatchFrame:SetFrameStrata("FULLSCREEN_DIALOG")
    pcall(tinsert, UISpecialFrames, "SKToolsPostMatchPrompt")

    local pos = SKToolsNotesDB.postMatchFramePos
    if pos then
        postMatchFrame:SetPoint(pos.point or "TOP", UIParent, pos.relPoint or "TOP", pos.x or 0, pos.y or -120)
    else
        postMatchFrame:SetPoint("TOP", UIParent, "TOP", 0, -120)
    end

    postMatchFrame:SetBackdrop(BACKDROP_INFO)
    postMatchFrame:SetBackdropColor(0.06, 0.06, 0.08, 0.95)
    postMatchFrame:SetBackdropBorderColor(0.2, 0.2, 0.25, 0.9)
    postMatchFrame:EnableMouse(true)
    postMatchFrame:SetMovable(true)
    postMatchFrame:SetClampedToScreen(true)
    postMatchFrame:RegisterForDrag("LeftButton")
    postMatchFrame:SetScript("OnDragStart", function(self) self:StartMoving() end)
    postMatchFrame:SetScript("OnDragStop", function(self)
        self:StopMovingOrSizing()
        local point, _, relPoint, x, y = self:GetPoint()
        SKToolsNotesDB.postMatchFramePos = { point = point, relPoint = relPoint, x = x, y = y }
    end)

    CreateCyanAccentLine(postMatchFrame)

    -- Title
    local titleText = postMatchFrame:CreateFontString(nil, "OVERLAY", "GameFontNormal")
    titleText:SetPoint("TOP", postMatchFrame, "TOP", 0, -14)
    titleText:SetText("Add notes about this matchup?")
    titleText:SetTextColor(0.9, 0.9, 0.9)
    postMatchFrame.titleText = titleText

    -- Comp display line
    local compLine = postMatchFrame:CreateFontString(nil, "OVERLAY", "GameFontNormal")
    compLine:SetPoint("TOP", titleText, "BOTTOM", 0, -8)
    compLine:SetTextColor(0.8, 0.8, 0.8)
    postMatchFrame.compLine = compLine

    -- Buttons
    local openBtn = CreateThemedButton(postMatchFrame, "Open Notes", 130, 26, "primary")
    openBtn:SetPoint("BOTTOMLEFT", postMatchFrame, "BOTTOMLEFT", 20, 14)
    openBtn:SetScript("OnClick", function()
        postMatchFrame:Hide()
        if postMatchFrame.matchPlayerTeam and postMatchFrame.matchEnemyTeam then
            PopulateAndShowPrepNotes(postMatchFrame.matchPlayerTeam, postMatchFrame.matchEnemyTeam, #postMatchFrame.matchEnemyTeam)
        end
    end)

    local dismissBtn = CreateThemedButton(postMatchFrame, "Dismiss", 130, 26, "secondary")
    dismissBtn:SetPoint("BOTTOMRIGHT", postMatchFrame, "BOTTOMRIGHT", -20, 14)
    dismissBtn:SetScript("OnClick", function()
        postMatchFrame:Hide()
    end)

    postMatchFrame:Hide()
    return postMatchFrame
end

local postMatchDismissTimer
local function ShowPostMatchPrompt(playerTeam, enemyTeam, won)
    CreatePostMatchFrame()

    postMatchFrame.matchPlayerTeam = playerTeam
    postMatchFrame.matchEnemyTeam = enemyTeam

    -- Color border green/red based on outcome
    if won then
        postMatchFrame:SetBackdropBorderColor(0.1, 0.8, 0.1, 0.9)
    else
        postMatchFrame:SetBackdropBorderColor(0.8, 0.1, 0.1, 0.9)
    end

    -- Update comp line
    local myIcons = CompIconString(playerTeam, 18)
    local enemyIcons = CompIconString(enemyTeam, 18)
    local resultText = won and "|cff00ff00W|r" or "|cffff4444L|r"
    postMatchFrame.compLine:SetText(myIcons .. "  |cff666666vs|r  " .. enemyIcons .. "    " .. resultText)

    FadeIn(postMatchFrame, 0.3)

    -- Auto-dismiss after 30s with fade
    if postMatchDismissTimer then postMatchDismissTimer:Cancel() end
    postMatchDismissTimer = C_Timer.NewTimer(27, function()
        if postMatchFrame:IsShown() then
            FadeOut(postMatchFrame, 3)
        end
    end)
end

-----------------------------
-- Notes Browser
-----------------------------
local browserSelectedKey = nil
local browserSelectedType = nil  -- "comp", "vscomp", "matchup", "spec", "bg", "dungeon", "raid", "general"
local browserSearchText = ""
local browserShowPvP = true   -- persisted in SKToolsNotesDB.browserShowPvP
local browserShowPvE = true   -- persisted in SKToolsNotesDB.browserShowPvE
local browserListButtons = {}
local browserFocusedIdx = nil  -- keyboard focus index within entries list
local browserEntries = {}      -- cached for keyboard navigation
local browserEditMode = false  -- false = view mode (read-only), true = edit mode
local browserSavedText = ""    -- last saved text for revert
local browserShowBuilder = false  -- true = show comp builder, false = show generic notes

local SECTION_MODES = {
    comp = "pvp", vscomp = "pvp", matchup = "pvp", spec = "pvp", bg = "pvp",
    dungeon = "pve", raid = "pve",
    general = "all",
}

local function SaveBrowserFramePos()
    if browserFrame then
        local point, _, relPoint, x, y = browserFrame:GetPoint()
        SKToolsNotesDB.browserFramePos = { point = point, relPoint = relPoint, x = x, y = y }
    end
end

-- Format a comp key with title if available (used in headers and lists)
local function CompDisplayWithTitle(compKey, iconSize)
    iconSize = iconSize or 18
    local team = CompKeyToTeam(compKey)
    local icons = CompIconString(team, iconSize)
    local title = GetCompTitle(compKey)
    if title then
        return icons .. "  " .. title
    end
    return icons .. "  " .. CompKeyToDisplay(compKey)
end

-- Format a key for display in the list
local function FormatKeyForDisplay(key, tabType)
    if tabType == "matchup" then
        local myPart, enemyPart = key:match("^(.+)|(.+)$")
        if myPart and enemyPart then
            local myLabel = CompDisplayWithTitle(myPart, 18)
            local enemyLabel = CompDisplayWithTitle(enemyPart, 18)
            return myLabel .. "  |cff888888vs|r  " .. enemyLabel
        end
    elseif tabType == "spec" then
        local spec, class = key:match("^(.+) (%u+)$")
        if class then
            return ClassIconString(class, 18) .. " " .. SpecKeyToDisplay(key)
        end
    elseif tabType == "bg" or tabType == "dungeon" or tabType == "raid" then
        return key
    elseif tabType == "general" then
        return key
    else
        return CompDisplayWithTitle(key, 18)
    end
    return key
end

-- Build a flat list of all notes across all categories, with section headers
-- Specs are prepopulated with ALL game specs even if no notes exist yet
local function BuildBrowserEntries()
    EnsureDB()
    local entries = {}
    local lower = browserSearchText ~= "" and browserSearchText:lower() or nil
    local isSearching = (lower ~= nil)

    local function isCollapsed(sectionKey)
        return (not isSearching) and SKToolsNotesDB.collapsedSections[sectionKey]
    end
    local function showSection(sectionKey)
        local mode = SECTION_MODES[sectionKey]
        if mode == "all" then return true end
        if mode == "pvp" then return browserShowPvP end
        if mode == "pve" then return browserShowPvE end
        return true
    end
    local function collapseIndicator(sectionKey)
        return isCollapsed(sectionKey) and "|cff888888>|r " or "|cff888888v|r "
    end

    -- Comp categories: only show entries that have notes
    local compCategories = {
        { name = "Your Comps",  tbl = SKToolsNotesDB.compNotes,    tabType = "comp" },
        { name = "Vs Comps",    tbl = SKToolsNotesDB.vsCompNotes,  tabType = "vscomp" },
        { name = "Matchups",    tbl = SKToolsNotesDB.matchupNotes, tabType = "matchup" },
    }

    for _, cat in ipairs(compCategories) do
      if showSection(cat.tabType) then
        -- Count total notes in this category (regardless of search)
        local totalCount = 0
        for _, v in pairs(cat.tbl) do
            if v and v ~= "" then totalCount = totalCount + 1 end
        end

        local collapsed = isCollapsed(cat.tabType)

        local keys = {}
        if not collapsed then
            for k, v in pairs(cat.tbl) do
                if v and v ~= "" then
                    local match = true
                    if lower then
                        match = k:lower():find(lower, 1, true) or v:lower():find(lower, 1, true)
                    end
                    if match then
                        keys[#keys + 1] = k
                    end
                end
            end
            table.sort(keys)
        end

        if totalCount > 0 then
            local headerLabel = collapseIndicator(cat.tabType) .. cat.name .. "  |cff888888(" .. totalCount .. ")|r"
            entries[#entries + 1] = { type = "header", label = headerLabel, sectionKey = cat.tabType }
            for _, k in ipairs(keys) do
                entries[#entries + 1] = {
                    type = "note",
                    key = k,
                    tbl = cat.tbl,
                    tabType = cat.tabType,
                }
            end
        end
      end
    end

    -- Specs: show all specs (collapsed hides them)
    if showSection("spec") then
        local allSpecs = GetAllSpecKeys()
        local specTbl = SKToolsNotesDB.specNotes
        local collapsed = isCollapsed("spec")

        -- Count specs that have notes
        local specWithNotesCount = 0
        for _, key in ipairs(allSpecs) do
            if specTbl[key] and specTbl[key] ~= "" then specWithNotesCount = specWithNotesCount + 1 end
        end

        local specKeys = {}
        if not collapsed then
            for _, key in ipairs(allSpecs) do
                local hasNote = specTbl[key] and specTbl[key] ~= ""
                local match = true
                if lower then
                    match = key:lower():find(lower, 1, true)
                        or (hasNote and specTbl[key]:lower():find(lower, 1, true))
                end
                if match then
                    specKeys[#specKeys + 1] = key
                end
            end
        end

        if #specKeys > 0 or (collapsed and (#allSpecs > 0 or specWithNotesCount > 0)) then
            local headerLabel = collapseIndicator("spec") .. "Specs  |cff888888(" .. specWithNotesCount .. " noted)|r"
            entries[#entries + 1] = { type = "header", label = headerLabel, sectionKey = "spec" }
            for _, k in ipairs(specKeys) do
                entries[#entries + 1] = {
                    type = "note",
                    key = k,
                    tbl = specTbl,
                    tabType = "spec",
                }
            end
        end
    end

    -- Battlegrounds: show ALL BGs (pre-populated), count how many have notes
    if showSection("bg") then
        local bgTbl = SKToolsNotesDB.bgNotes
        local collapsed = isCollapsed("bg")
        local bgKeys = {}
        local bgNotedCount = 0
        for k, v in pairs(bgTbl) do
            if v and v ~= "" then bgNotedCount = bgNotedCount + 1 end
            if not collapsed then
                local match = true
                if lower then
                    match = k:lower():find(lower, 1, true) or (v and v ~= "" and v:lower():find(lower, 1, true))
                end
                if match then bgKeys[#bgKeys + 1] = k end
            end
        end
        table.sort(bgKeys)

        -- Count total entries for header visibility
        local bgTotal = 0
        for _ in pairs(bgTbl) do bgTotal = bgTotal + 1 end

        if #bgKeys > 0 or (collapsed and bgTotal > 0) then
            entries[#entries + 1] = { type = "header", label = collapseIndicator("bg") .. "Battlegrounds  |cff888888(" .. bgNotedCount .. " noted)|r", sectionKey = "bg" }
            for _, k in ipairs(bgKeys) do
                entries[#entries + 1] = {
                    type = "note",
                    key = k,
                    tbl = bgTbl,
                    tabType = "bg",
                }
            end
        end
    end

    -- Dungeons: dynamically populated, show all entries
    if showSection("dungeon") then
        local dungeonTbl = SKToolsNotesDB.dungeonNotes
        local collapsed = isCollapsed("dungeon")
        local dungeonKeys = {}
        local dungeonNotedCount = 0
        for k, v in pairs(dungeonTbl) do
            if v and v ~= "" then dungeonNotedCount = dungeonNotedCount + 1 end
            if not collapsed then
                local match = true
                if lower then
                    match = k:lower():find(lower, 1, true) or (v and v ~= "" and v:lower():find(lower, 1, true))
                end
                if match then dungeonKeys[#dungeonKeys + 1] = k end
            end
        end
        table.sort(dungeonKeys)

        local dungeonTotal = 0
        for _ in pairs(dungeonTbl) do dungeonTotal = dungeonTotal + 1 end

        if #dungeonKeys > 0 or (collapsed and dungeonTotal > 0) then
            entries[#entries + 1] = { type = "header", label = collapseIndicator("dungeon") .. "Dungeons  |cff888888(" .. dungeonNotedCount .. " noted)|r", sectionKey = "dungeon" }
            for _, k in ipairs(dungeonKeys) do
                entries[#entries + 1] = {
                    type = "note",
                    key = k,
                    tbl = dungeonTbl,
                    tabType = "dungeon",
                }
            end
        end
    end

    -- Raids: dynamically populated, show all entries
    if showSection("raid") then
        local raidTbl = SKToolsNotesDB.raidNotes
        local collapsed = isCollapsed("raid")
        local raidKeys = {}
        local raidNotedCount = 0
        for k, v in pairs(raidTbl) do
            if v and v ~= "" then raidNotedCount = raidNotedCount + 1 end
            if not collapsed then
                local match = true
                if lower then
                    match = k:lower():find(lower, 1, true) or (v and v ~= "" and v:lower():find(lower, 1, true))
                end
                if match then raidKeys[#raidKeys + 1] = k end
            end
        end
        table.sort(raidKeys)

        local raidTotal = 0
        for _ in pairs(raidTbl) do raidTotal = raidTotal + 1 end

        if #raidKeys > 0 or (collapsed and raidTotal > 0) then
            entries[#entries + 1] = { type = "header", label = collapseIndicator("raid") .. "Raids  |cff888888(" .. raidNotedCount .. " noted)|r", sectionKey = "raid" }
            for _, k in ipairs(raidKeys) do
                entries[#entries + 1] = {
                    type = "note",
                    key = k,
                    tbl = raidTbl,
                    tabType = "raid",
                }
            end
        end
    end

    -- General: freeform notes with custom titles (always show all, like BGs)
    if showSection("general") then
        local genTbl = SKToolsNotesDB.generalNotes
        local collapsed = isCollapsed("general")
        local genKeys = {}
        local genNotedCount = 0
        for k, v in pairs(genTbl) do
            if v and v ~= "" then genNotedCount = genNotedCount + 1 end
            if not collapsed then
                local match = true
                if lower then
                    match = k:lower():find(lower, 1, true) or (v and v ~= "" and v:lower():find(lower, 1, true))
                end
                if match then genKeys[#genKeys + 1] = k end
            end
        end
        table.sort(genKeys)

        local genTotal = 0
        for _ in pairs(genTbl) do genTotal = genTotal + 1 end

        if #genKeys > 0 or (collapsed and genTotal > 0) then
            entries[#entries + 1] = { type = "header", label = collapseIndicator("general") .. "General  |cff888888(" .. genNotedCount .. " noted)|r", sectionKey = "general" }
            for _, k in ipairs(genKeys) do
                entries[#entries + 1] = {
                    type = "note",
                    key = k,
                    tbl = genTbl,
                    tabType = "general",
                }
            end
        end
    end

    return entries
end

local BROWSER_LIST_WIDTH = 420
local BROWSER_ROW_HEIGHT = 42
local BROWSER_ROW_HEIGHT_COMPACT = 22
local BROWSER_HEADER_HEIGHT = 28
local BROWSER_VISIBLE_ROWS = 30

CreateBrowserFrame = function()
    if browserFrame then return browserFrame end
    EnsureDB()

    browserFrame = CreateFrame("Frame", "SKToolsNotesBrowser", UIParent, "BackdropTemplate")
    browserFrame:SetSize(1100, 750)
    browserFrame:SetFrameStrata("DIALOG")
    browserFrame:SetMovable(true)
    browserFrame:SetClampedToScreen(true)
    browserFrame:EnableMouse(true)
    browserFrame:SetBackdrop(BACKDROP_INFO)
    browserFrame:SetBackdropColor(0.05, 0.05, 0.07, 0.97)
    browserFrame:SetBackdropBorderColor(CYAN.r * 0.15, CYAN.g * 0.15, CYAN.b * 0.15, 0.6)
    pcall(tinsert, UISpecialFrames, "SKToolsNotesBrowser")

    local pos = SKToolsNotesDB.browserFramePos
    if pos then
        browserFrame:SetPoint(pos.point or "CENTER", UIParent, pos.relPoint or "CENTER", pos.x or 0, pos.y or 0)
    else
        browserFrame:SetPoint("CENTER", UIParent, "CENTER", 0, 0)
    end

    CreateCyanAccentLine(browserFrame)
    CreateDragTitleBar(browserFrame, "Arena Notes", function() browserFrame:Hide() end, SaveBrowserFramePos)

    -- "Match Notes" toggle button (switch back to contextual prep notes, only in PvP)
    local matchNotesBtn = CreateThemedButton(browserFrame, "Match Notes", 86, 18, "primary")
    matchNotesBtn:SetPoint("TOPLEFT", browserFrame, "TOPLEFT", 110, -5)
    matchNotesBtn:SetFrameLevel(matchNotesBtn:GetFrameLevel() + 5)
    matchNotesBtn:SetScript("OnClick", function()
        browserFrame:Hide()
        if prepFrame then
            FadeIn(prepFrame, 0.2)
        end
    end)
    matchNotesBtn:Hide()  -- hidden by default, shown when in PvP
    browserFrame.matchNotesBtn = matchNotesBtn

    -- "+ New Note" button (top bar, always visible)
    local newNoteBtn = CreateThemedButton(browserFrame, "+ New Note", 86, 18, "primary")
    newNoteBtn:SetPoint("TOPRIGHT", browserFrame, "TOPRIGHT", -36, -6)
    newNoteBtn:SetFrameLevel(newNoteBtn:GetFrameLevel() + 5)  -- raise above title bar
    newNoteBtn:SetScript("OnClick", function()
        browserSelectedKey = nil
        browserSelectedType = nil
        browserEditMode = false
        browserShowBuilder = true
        -- Reset builder to clean state
        if browserFrame.builderSelected then wipe(browserFrame.builderSelected) end
        if browserFrame.builderMyTeam then wipe(browserFrame.builderMyTeam) end
        browserFrame.builderMatchupPhase = 1
        browserFrame.builderType = "general"
        SKArenaNotes_RefreshBrowser()
    end)

    -- Search box (custom dark-themed)
    local searchContainer = CreateFrame("Frame", nil, browserFrame, "BackdropTemplate")
    searchContainer:SetSize(180, 18)
    searchContainer:SetPoint("RIGHT", newNoteBtn, "LEFT", -8, 0)
    searchContainer:SetBackdrop({
        bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
        edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
        tile = true, tileSize = 8, edgeSize = 10,
        insets = { left = 2, right = 2, top = 2, bottom = 2 },
    })
    searchContainer:SetBackdropColor(0.04, 0.04, 0.06, 0.95)
    searchContainer:SetBackdropBorderColor(0.25, 0.25, 0.3, 0.6)
    searchContainer:SetFrameLevel(searchContainer:GetFrameLevel() + 5)  -- raise above title bar

    local searchIcon = searchContainer:CreateFontString(nil, "OVERLAY", "GameFontDisableSmall")
    searchIcon:SetPoint("LEFT", searchContainer, "LEFT", 6, 0)
    searchIcon:SetText("Search")
    searchIcon:SetTextColor(0.4, 0.4, 0.4)

    local searchBox = CreateFrame("EditBox", "SKToolsNotesSearchBox", searchContainer)
    searchBox:SetPoint("LEFT", searchContainer, "LEFT", 8, 0)
    searchBox:SetHeight(18)
    searchBox:SetAutoFocus(false)
    searchBox:SetFontObject("GameFontHighlightSmall")
    searchBox:SetTextColor(0.85, 0.85, 0.85)
    searchBox:SetScript("OnEscapePressed", function(self) self:ClearFocus() end)

    -- Clear search "x" button (inside search container, right side)
    local clearSearchBtn = CreateFrame("Button", nil, searchContainer)
    clearSearchBtn:SetSize(16, 16)
    clearSearchBtn:SetPoint("RIGHT", searchContainer, "RIGHT", -2, 0)
    local clearSearchLbl = clearSearchBtn:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
    clearSearchLbl:SetPoint("CENTER", 0, 1)
    clearSearchLbl:SetText("x")
    clearSearchLbl:SetTextColor(0.5, 0.5, 0.5)
    clearSearchBtn:SetScript("OnEnter", function() clearSearchLbl:SetTextColor(1, 0.3, 0.3) end)
    clearSearchBtn:SetScript("OnLeave", function() clearSearchLbl:SetTextColor(0.5, 0.5, 0.5) end)
    clearSearchBtn:SetScript("OnClick", function()
        searchBox:SetText("")
        searchBox:ClearFocus()
    end)
    clearSearchBtn:Hide()

    -- Anchor search text to stop before clear button
    searchBox:SetPoint("RIGHT", clearSearchBtn, "LEFT", -2, 0)

    searchBox:SetScript("OnTextChanged", function(self, userInput)
        browserSearchText = self:GetText() or ""
        browserFocusedIdx = nil
        if browserSearchText ~= "" then
            searchIcon:Hide()
            clearSearchBtn:Show()
        else
            clearSearchBtn:Hide()
            if not self:HasFocus() then searchIcon:Show() end
        end
        SKArenaNotes_RefreshBrowser()
    end)
    searchBox:SetScript("OnEditFocusGained", function(self)
        searchIcon:Hide()
        searchContainer:SetBackdropBorderColor(CYAN.r * 0.4, CYAN.g * 0.4, CYAN.b * 0.4, 0.8)
    end)
    searchBox:SetScript("OnEditFocusLost", function(self)
        if self:GetText() == "" then
            searchIcon:Show()
            clearSearchBtn:Hide()
        end
        searchContainer:SetBackdropBorderColor(0.25, 0.25, 0.3, 0.6)
    end)
    browserFrame.searchBox = searchBox

    -- Keyboard navigation (Up/Down/Enter/Escape) — only while browser is visible
    browserFrame:SetToplevel(true)
    browserFrame:SetScript("OnShow", function(self) self:Raise(); pcall(self.EnableKeyboard, self, true) end)
    browserFrame:SetScript("OnHide", function(self) pcall(self.EnableKeyboard, self, false) end)
    pcall(browserFrame.SetPropagateKeyboardInput, browserFrame, true)
    browserFrame:SetScript("OnKeyDown", function(self, key)
        -- Default: let all keys through to the game
        pcall(self.SetPropagateKeyboardInput, self, true)

        -- Don't intercept keys when an editbox has focus (WoW chat, search, editor)
        local focused = GetCurrentKeyBoardFocus()
        if focused then return end

        if key == "UP" or key == "DOWN" then
            local numEntries = #browserEntries
            if numEntries == 0 then return end

            -- Find next note entry (skip headers)
            local dir = key == "DOWN" and 1 or -1
            local start = browserFocusedIdx or (dir == 1 and 0 or numEntries + 1)
            local idx = start
            for _ = 1, numEntries do
                idx = idx + dir
                if idx < 1 then idx = numEntries end
                if idx > numEntries then idx = 1 end
                if browserEntries[idx] and browserEntries[idx].type == "note" then
                    pcall(self.SetPropagateKeyboardInput, self, false)
                    browserFocusedIdx = idx
                    -- Scroll to make focused row visible
                    local offset = FauxScrollFrame_GetOffset(browserFrame.listScroll)
                    if idx <= offset then
                        FauxScrollFrame_SetOffset(browserFrame.listScroll, math.max(0, idx - 1))
                    elseif idx > offset + BROWSER_VISIBLE_ROWS then
                        FauxScrollFrame_SetOffset(browserFrame.listScroll, idx - BROWSER_VISIBLE_ROWS)
                    end
                    SKArenaNotes_RefreshBrowser()
                    return
                end
            end
        elseif key == "ENTER" then
            if browserFocusedIdx and browserEntries[browserFocusedIdx] then
                local entry = browserEntries[browserFocusedIdx]
                if entry.type == "note" then
                    pcall(self.SetPropagateKeyboardInput, self, false)
                    browserSelectedKey = entry.key
                    browserSelectedType = entry.tabType
                    browserEditMode = false
                    browserShowBuilder = false
                    SKArenaNotes_RefreshBrowser()
                end
            end
        elseif key == "ESCAPE" then
            -- ESC always closes the window — reset all state and hide
            pcall(self.SetPropagateKeyboardInput, self, false)
            browserEditMode = false
            browserSelectedKey = nil
            browserSelectedType = nil
            browserShowBuilder = false
            browserFrame:Hide()
        end
    end)
    browserFrame:SetScript("OnKeyUp", function(self, key)
        pcall(self.SetPropagateKeyboardInput, self, true)
    end)

    -- Divider under title bar
    local tabDivider = browserFrame:CreateTexture(nil, "ARTWORK")
    tabDivider:SetHeight(1)
    tabDivider:SetPoint("TOPLEFT", browserFrame, "TOPLEFT", 8, -28)
    tabDivider:SetPoint("TOPRIGHT", browserFrame, "TOPRIGHT", -8, -28)
    tabDivider:SetColorTexture(CYAN.r, CYAN.g, CYAN.b, 0.2)

    -- PvP/PvE view toggle (segmented control above list)
    local toggleContainer = CreateFrame("Frame", nil, browserFrame)
    toggleContainer:SetPoint("TOPLEFT", browserFrame, "TOPLEFT", 8, -34)
    toggleContainer:SetSize(BROWSER_LIST_WIDTH, 24)

    local btnW = math.floor(BROWSER_LIST_WIDTH / 2) - 12
    local toggleBackdrop = {
        bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
        edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
        tile = true, tileSize = 8, edgeSize = 10,
        insets = { left = 2, right = 2, top = 2, bottom = 2 },
    }

    local pvpBtn = CreateFrame("Button", nil, toggleContainer, "BackdropTemplate")
    pvpBtn:SetSize(btnW, 22)
    pvpBtn:SetPoint("LEFT", toggleContainer, "LEFT", 0, 0)
    pvpBtn:SetBackdrop(toggleBackdrop)
    pvpBtn.label = pvpBtn:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
    pvpBtn.label:SetPoint("CENTER")
    pvpBtn.label:SetText("PvP")
    browserFrame.pvpBtn = pvpBtn

    local pveBtn = CreateFrame("Button", nil, toggleContainer, "BackdropTemplate")
    pveBtn:SetSize(btnW, 22)
    pveBtn:SetPoint("LEFT", pvpBtn, "RIGHT", 4, 0)
    pveBtn:SetBackdrop(toggleBackdrop)
    pveBtn.label = pveBtn:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
    pveBtn.label:SetPoint("CENTER")
    pveBtn.label:SetText("PvE")
    browserFrame.pveBtn = pveBtn

    pvpBtn:SetScript("OnClick", function()
        browserShowPvP = not browserShowPvP
        EnsureDB()
        SKToolsNotesDB.browserShowPvP = browserShowPvP
        if not browserShowPvP and browserSelectedType and SECTION_MODES[browserSelectedType] == "pvp" then
            browserSelectedKey = nil
            browserSelectedType = nil
            browserEditMode = false
            browserShowBuilder = false
        end
        SKArenaNotes_RefreshBrowser()
    end)

    pveBtn:SetScript("OnClick", function()
        browserShowPvE = not browserShowPvE
        EnsureDB()
        SKToolsNotesDB.browserShowPvE = browserShowPvE
        if not browserShowPvE and browserSelectedType and SECTION_MODES[browserSelectedType] == "pve" then
            browserSelectedKey = nil
            browserSelectedType = nil
            browserEditMode = false
            browserShowBuilder = false
        end
        SKArenaNotes_RefreshBrowser()
    end)

    -- Left panel: scrollable list with all categories
    local listFrame = CreateFrame("Frame", nil, browserFrame)
    listFrame:SetPoint("TOPLEFT", toggleContainer, "BOTTOMLEFT", 0, -4)
    listFrame:SetSize(BROWSER_LIST_WIDTH, 678)
    listFrame:SetClipsChildren(true)
    browserFrame.listFrame = listFrame

    local listScroll = CreateFrame("ScrollFrame", "SKToolsNotesListScroll", listFrame, "FauxScrollFrameTemplate")
    listScroll:SetPoint("TOPLEFT", listFrame, "TOPLEFT", 0, 0)
    listScroll:SetPoint("BOTTOMRIGHT", listFrame, "BOTTOMRIGHT", -22, 0)
    browserFrame.listScroll = listScroll

    -- Empty state message (shown when no notes exist)
    local emptyMsg = listFrame:CreateFontString(nil, "OVERLAY", "GameFontNormal")
    emptyMsg:SetPoint("TOP", listFrame, "TOP", 0, -60)
    emptyMsg:SetPoint("LEFT", listFrame, "LEFT", 16, 0)
    emptyMsg:SetPoint("RIGHT", listFrame, "RIGHT", -16, 0)
    emptyMsg:SetJustifyH("CENTER")
    emptyMsg:SetWordWrap(true)
    emptyMsg:SetText(
        "No notes yet\n\n" ..
        "|cff666666Notes are created automatically during\n" ..
        "arena prep, or click|r |cff00E5EE+ New Note|r |cff666666above.|r"
    )
    emptyMsg:SetTextColor(0.45, 0.45, 0.45)
    emptyMsg:Hide()
    browserFrame.emptyMsg = emptyMsg

    -- Create row frames (reused for both headers and note entries)
    for i = 1, BROWSER_VISIBLE_ROWS do
        local row = CreateFrame("Button", nil, listFrame)
        row:SetSize(BROWSER_LIST_WIDTH - 22, BROWSER_ROW_HEIGHT)
        row:SetPoint("TOPLEFT", listFrame, "TOPLEFT", 0, 0)  -- repositioned in refresh

        -- Background
        local bg = row:CreateTexture(nil, "BACKGROUND")
        bg:SetAllPoints()
        bg:SetColorTexture(0.05, 0.05, 0.07, 0)
        row.bg = bg

        -- Selection highlight
        local sel = row:CreateTexture(nil, "BACKGROUND", nil, 1)
        sel:SetAllPoints()
        sel:SetColorTexture(CYAN.r * 0.10, CYAN.g * 0.10, CYAN.b * 0.10, 0.9)
        sel:Hide()
        row.selHighlight = sel

        -- Hover highlight
        local hover = row:CreateTexture(nil, "HIGHLIGHT")
        hover:SetAllPoints()
        hover:SetColorTexture(1, 1, 1, 0.04)
        row.hoverHighlight = hover

        -- Left accent bar (shown for notes with content)
        local noteAccent = row:CreateTexture(nil, "OVERLAY")
        noteAccent:SetWidth(2)
        noteAccent:SetPoint("TOPLEFT", row, "TOPLEFT", 0, -2)
        noteAccent:SetPoint("BOTTOMLEFT", row, "BOTTOMLEFT", 0, 2)
        noteAccent:SetColorTexture(CYAN.r, CYAN.g, CYAN.b, 0.6)
        noteAccent:Hide()
        row.noteAccent = noteAccent

        -- Keyboard focus highlight (left edge accent)
        local focusAccent = row:CreateTexture(nil, "OVERLAY")
        focusAccent:SetWidth(2)
        focusAccent:SetPoint("TOPLEFT", row, "TOPLEFT", 0, 0)
        focusAccent:SetPoint("BOTTOMLEFT", row, "BOTTOMLEFT", 0, 0)
        focusAccent:SetColorTexture(CYAN.r, CYAN.g, CYAN.b, 1)
        focusAccent:Hide()
        row.focusAccent = focusAccent

        -- Title text (used for note title or section header)
        local titleText = row:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
        titleText:SetPoint("TOPLEFT", row, "TOPLEFT", 8, -3)
        titleText:SetPoint("RIGHT", row, "RIGHT", -60, 0)
        titleText:SetJustifyH("LEFT")
        titleText:SetMaxLines(1)
        row.titleText = titleText

        -- Timestamp text (right-aligned, subtle)
        local timeText = row:CreateFontString(nil, "OVERLAY", "GameFontDisableSmall")
        timeText:SetPoint("TOPRIGHT", row, "TOPRIGHT", -4, -5)
        timeText:SetJustifyH("RIGHT")
        timeText:SetTextColor(0.35, 0.35, 0.40)
        row.timeText = timeText

        -- Preview text (hidden for headers, muted italic feel)
        local preview = row:CreateFontString(nil, "OVERLAY", "GameFontDisableSmall")
        preview:SetPoint("TOPLEFT", titleText, "BOTTOMLEFT", 0, -1)
        preview:SetPoint("RIGHT", row, "RIGHT", -6, 0)
        preview:SetJustifyH("LEFT")
        preview:SetWordWrap(true)
        preview:SetMaxLines(1)
        row.previewText = preview

        -- Bottom separator line (subtle, between entries)
        local sepLine = row:CreateTexture(nil, "ARTWORK")
        sepLine:SetHeight(1)
        sepLine:SetPoint("BOTTOMLEFT", row, "BOTTOMLEFT", 6, 0)
        sepLine:SetPoint("BOTTOMRIGHT", row, "BOTTOMRIGHT", -6, 0)
        sepLine:SetColorTexture(1, 1, 1, 0.04)
        row.sepLine = sepLine

        -- Section header underline (hidden for notes)
        local headerLine = row:CreateTexture(nil, "ARTWORK")
        headerLine:SetHeight(1)
        headerLine:SetPoint("BOTTOMLEFT", row, "BOTTOMLEFT", 0, 0)
        headerLine:SetPoint("BOTTOMRIGHT", row, "BOTTOMRIGHT", 0, 0)
        headerLine:SetColorTexture(CYAN.r, CYAN.g, CYAN.b, 0.12)
        headerLine:Hide()
        row.headerLine = headerLine

        row:SetScript("OnClick", function(self)
            -- Section header toggles collapse
            if self.sectionKey then
                EnsureDB()
                SKToolsNotesDB.collapsedSections[self.sectionKey] = not SKToolsNotesDB.collapsedSections[self.sectionKey]
                SKArenaNotes_RefreshBrowser()
                return
            end
            if self.noteKey and self.noteType then
                -- Toggle: click again to deselect (shows generic notes)
                if browserSelectedKey == self.noteKey and browserSelectedType == self.noteType then
                    browserSelectedKey = nil
                    browserSelectedType = nil
                else
                    browserSelectedKey = self.noteKey
                    browserSelectedType = self.noteType
                end
                -- Reset to view mode when switching notes
                browserEditMode = false
                browserShowBuilder = false
                SKArenaNotes_RefreshBrowser()
            end
        end)

        -- Tooltip preview on hover (show full note text)
        row:SetScript("OnEnter", function(self)
            if not self.noteKey or not self.noteTbl then return end
            local text = self.noteTbl[self.noteKey]
            if not text or text == "" then return end
            GameTooltip:SetOwner(self, "ANCHOR_RIGHT", 0, 0)
            GameTooltip:ClearLines()
            local totalLines = 0
            local maxChars = 55
            for line in text:gmatch("[^\n]+") do
                if totalLines >= 12 then
                    GameTooltip:AddLine("...", 0.5, 0.5, 0.5)
                    break
                end
                -- Manually word-wrap long lines
                while #line > maxChars and totalLines < 12 do
                    local breakAt = maxChars
                    local spaceAt = line:sub(1, maxChars):find("%s[^%s]*$")
                    if spaceAt and spaceAt > 10 then breakAt = spaceAt end
                    GameTooltip:AddLine(line:sub(1, breakAt), 0.85, 0.85, 0.85)
                    line = line:sub(breakAt + 1)
                    totalLines = totalLines + 1
                end
                if totalLines < 12 then
                    GameTooltip:AddLine(line, 0.85, 0.85, 0.85)
                    totalLines = totalLines + 1
                end
            end
            GameTooltip:Show()
        end)
        row:SetScript("OnLeave", function(self)
            GameTooltip:Hide()
        end)

        browserListButtons[i] = row
    end

    -- FauxScrollFrame OnVerticalScroll (makes the scrollbar draggable/clickable)
    listScroll:SetScript("OnVerticalScroll", function(self, offset)
        FauxScrollFrame_OnVerticalScroll(self, offset, BROWSER_ROW_HEIGHT_COMPACT, function()
            SKArenaNotes_RefreshBrowser()
        end)
    end)

    -- Override mouse wheel: scroll 2 rows per tick instead of default huge jumps
    -- Disable built-in scroll on FauxScrollFrame scrollbar
    local scrollBar = listScroll.ScrollBar or _G["SKToolsNotesListScrollScrollBar"]
    if scrollBar then
        scrollBar:EnableMouseWheel(false)
    end
    listScroll:EnableMouseWheel(false)

    listFrame:EnableMouseWheel(true)
    listFrame:SetScript("OnMouseWheel", function(self, delta)
        local current = listScroll:GetVerticalScroll()
        -- Max scroll = max offset * row height (so last entries fill the view exactly)
        local maxOff = browserFrame.maxScrollOffset or 0
        local maxScroll = maxOff * BROWSER_ROW_HEIGHT_COMPACT
        local step = BROWSER_ROW_HEIGHT_COMPACT * 2
        local newScroll = math.max(0, math.min(current - delta * step, maxScroll))
        listScroll:SetVerticalScroll(newScroll)
        FauxScrollFrame_OnVerticalScroll(listScroll, newScroll, BROWSER_ROW_HEIGHT_COMPACT, function()
            SKArenaNotes_RefreshBrowser()
        end)
    end)

    -- Vertical divider
    local vDivider = browserFrame:CreateTexture(nil, "ARTWORK")
    vDivider:SetWidth(1)
    vDivider:SetPoint("TOP", browserFrame, "TOP", 0, -28)
    vDivider:SetPoint("BOTTOM", browserFrame, "BOTTOM", 0, 8)
    vDivider:SetPoint("LEFT", listFrame, "RIGHT", 8, 0)
    vDivider:SetColorTexture(CYAN.r, CYAN.g, CYAN.b, 0.12)

    -- Right panel: editor
    local rightPanel = CreateFrame("Frame", nil, browserFrame)
    rightPanel:SetPoint("TOPLEFT", vDivider, "TOPRIGHT", 8, 6)
    rightPanel:SetPoint("BOTTOMRIGHT", browserFrame, "BOTTOMRIGHT", -8, 8)
    browserFrame.rightPanel = rightPanel

    -- Header for selected note
    local editorHeader = rightPanel:CreateFontString(nil, "OVERLAY", "GameFontNormal")
    editorHeader:SetPoint("TOPLEFT", rightPanel, "TOPLEFT", 4, 0)
    editorHeader:SetPoint("RIGHT", rightPanel, "RIGHT", -120, 0)
    editorHeader:SetJustifyH("LEFT")
    editorHeader:SetWordWrap(true)
    editorHeader:SetMaxLines(2)
    editorHeader:SetText("Select a note to edit")
    editorHeader:SetTextColor(0.7, 0.7, 0.7)
    browserFrame.editorHeader = editorHeader

    -- Title bar for comp/vscomp (clickable to set title)
    local titleBar = CreateFrame("Frame", nil, rightPanel)
    titleBar:SetPoint("TOPLEFT", editorHeader, "BOTTOMLEFT", 0, -1)
    titleBar:SetPoint("RIGHT", rightPanel, "RIGHT", -8, 0)
    titleBar:SetHeight(1)
    browserFrame.titleBar = titleBar

    local titleLabel = titleBar:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
    titleLabel:SetPoint("LEFT", titleBar, "LEFT", 0, 0)
    titleLabel:SetJustifyH("LEFT")
    titleLabel:SetTextColor(0.6, 0.6, 0.6)
    browserFrame.titleLabel = titleLabel

    local titleEditBtn = CreateFrame("Button", nil, titleBar)
    titleEditBtn:SetSize(60, 16)
    titleEditBtn:SetPoint("LEFT", titleLabel, "RIGHT", 6, 0)
    local titleEditText = titleEditBtn:CreateFontString(nil, "OVERLAY", "GameFontDisableSmall")
    titleEditText:SetAllPoints()
    titleEditText:SetJustifyH("LEFT")
    titleEditText:SetText("|cff00B4CC[rename]|r")
    titleEditBtn:SetScript("OnEnter", function() titleEditText:SetText("|cff00E5EE[rename]|r") end)
    titleEditBtn:SetScript("OnLeave", function() titleEditText:SetText("|cff00B4CC[rename]|r") end)
    browserFrame.titleEditBtn = titleEditBtn

    -- Also a set-title button for when no title exists
    local titleSetBtn = CreateFrame("Button", nil, titleBar)
    titleSetBtn:SetSize(90, 16)
    titleSetBtn:SetPoint("LEFT", titleBar, "LEFT", 0, 0)
    local titleSetText = titleSetBtn:CreateFontString(nil, "OVERLAY", "GameFontDisableSmall")
    titleSetText:SetAllPoints()
    titleSetText:SetJustifyH("LEFT")
    titleSetText:SetText("|cff00B4CC[+ set title]|r")
    titleSetBtn:SetScript("OnEnter", function() titleSetText:SetText("|cff00E5EE[+ set title]|r") end)
    titleSetBtn:SetScript("OnLeave", function() titleSetText:SetText("|cff00B4CC[+ set title]|r") end)
    browserFrame.titleSetBtn = titleSetBtn

    -- Title edit popup
    StaticPopupDialogs["SKNOTES_SET_COMP_TITLE"] = {
        text = "Set a title for this comp:",
        button1 = "Save",
        button2 = "Cancel",
        hasEditBox = true,
        editBoxWidth = 220,
        OnShow = function(self, data)
            local current = GetCompTitle(data.compKey) or ""
            self.editBox:SetText(current)
            self.editBox:HighlightText()
            self.editBox:SetFocus()
        end,
        OnAccept = function(self, data)
            local text = self.editBox:GetText()
            SetCompTitle(data.compKey, text)
            SKArenaNotes_RefreshBrowser()
        end,
        EditBoxOnEnterPressed = function(self, data)
            local text = self:GetText()
            SetCompTitle(data.compKey, text)
            self:GetParent():Hide()
            SKArenaNotes_RefreshBrowser()
        end,
        EditBoxOnEscapePressed = function(self)
            self:GetParent():Hide()
        end,
        timeout = 0,
        whileDead = true,
        hideOnEscape = true,
        preferredIndex = 3,
    }

    local function ShowTitlePopup(compKey)
        StaticPopup_Show("SKNOTES_SET_COMP_TITLE", nil, nil, { compKey = compKey })
    end

    titleEditBtn:SetScript("OnClick", function()
        if browserSelectedKey and (browserSelectedType == "comp" or browserSelectedType == "vscomp") then
            ShowTitlePopup(browserSelectedKey)
        end
    end)
    titleSetBtn:SetScript("OnClick", function()
        if browserSelectedKey and (browserSelectedType == "comp" or browserSelectedType == "vscomp") then
            ShowTitlePopup(browserSelectedKey)
        end
    end)

    -- Category label under header
    local editorCategory = rightPanel:CreateFontString(nil, "OVERLAY", "GameFontDisableSmall")
    editorCategory:SetPoint("TOPLEFT", titleBar, "BOTTOMLEFT", 0, -1)
    editorCategory:SetJustifyH("LEFT")
    editorCategory:SetText("")
    browserFrame.editorCategory = editorCategory

    -- === View mode: clean transparent display ===
    local viewContainer = CreateFrame("Frame", nil, rightPanel, "BackdropTemplate")
    viewContainer:SetPoint("TOPLEFT", editorCategory, "BOTTOMLEFT", -4, -6)
    viewContainer:SetPoint("BOTTOMRIGHT", rightPanel, "BOTTOMRIGHT", 0, 36)
    viewContainer:SetBackdrop({
        bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
        edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
        tile = true, tileSize = 8, edgeSize = 10,
        insets = { left = 2, right = 2, top = 2, bottom = 2 },
    })
    viewContainer:SetBackdropColor(0.04, 0.04, 0.06, 0.6)
    viewContainer:SetBackdropBorderColor(0.15, 0.15, 0.18, 0.4)
    viewContainer:Hide()
    browserFrame.viewContainer = viewContainer

    -- View scroll frame
    local viewScroll = CreateFrame("ScrollFrame", nil, viewContainer, "UIPanelScrollFrameTemplate")
    viewScroll:SetPoint("TOPLEFT", viewContainer, "TOPLEFT", 10, -8)
    viewScroll:SetPoint("BOTTOMRIGHT", viewContainer, "BOTTOMRIGHT", -10, 8)
    local vsBar = viewScroll.ScrollBar
    if vsBar then
        vsBar:SetWidth(10)
        if vsBar.Background then vsBar.Background:SetAlpha(0) end
        if vsBar.NineSlice then vsBar.NineSlice:SetAlpha(0) end
        local vThumb = vsBar.ThumbTexture or vsBar.thumbTexture
        if vThumb then
            vThumb:SetColorTexture(CYAN.r, CYAN.g, CYAN.b, 0.2)
            vThumb:SetWidth(8)
        end
    end

    local viewContent = CreateFrame("Frame", nil, viewScroll)
    local initW = viewScroll:GetWidth()
    viewContent:SetWidth((initW and initW > 1) and initW or 500)
    viewContent:SetHeight(1)
    viewScroll:SetScrollChild(viewContent)

    local viewText = viewContent:CreateFontString(nil, "OVERLAY", "ChatFontNormal")
    viewText:SetPoint("TOPLEFT", viewContent, "TOPLEFT", 4, 0)
    viewText:SetPoint("TOPRIGHT", viewContent, "TOPRIGHT", -8, 0)
    viewText:SetJustifyH("LEFT")
    viewText:SetJustifyV("TOP")
    viewText:SetWordWrap(true)
    viewText:SetSpacing(3)
    viewText:SetTextColor(0.82, 0.82, 0.82)
    browserFrame.viewScroll = viewScroll
    browserFrame.viewContent = viewContent
    browserFrame.viewText = viewText

    viewScroll:SetScript("OnSizeChanged", function(self, w, h)
        viewContent:SetWidth(w)
        local textH = viewText:GetStringHeight() or 0
        viewContent:SetHeight(math.max(textH + 20, h or 300))
    end)

    -- Empty view placeholder
    local viewEmpty = viewContainer:CreateFontString(nil, "OVERLAY", "GameFontDisable")
    viewEmpty:SetPoint("CENTER", viewContainer, "CENTER", 0, 10)
    viewEmpty:SetJustifyH("CENTER")
    viewEmpty:SetWordWrap(true)
    viewEmpty:SetWidth(280)
    viewEmpty:SetSpacing(6)
    viewEmpty:SetText("No notes yet.\n\nClick Edit below to start writing.")
    viewEmpty:Hide()
    browserFrame.viewEmpty = viewEmpty

    -- === Edit mode container (functional editor styling) ===
    local editorContainer = CreateFrame("Frame", nil, rightPanel, "BackdropTemplate")
    editorContainer:SetPoint("TOPLEFT", editorCategory, "BOTTOMLEFT", -4, -6)
    editorContainer:SetPoint("BOTTOMRIGHT", rightPanel, "BOTTOMRIGHT", 0, 36)
    editorContainer:SetBackdrop({
        bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
        edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
        tile = true, tileSize = 8, edgeSize = 10,
        insets = { left = 2, right = 2, top = 2, bottom = 2 },
    })
    editorContainer:SetBackdropColor(0.03, 0.03, 0.05, 0.95)
    editorContainer:SetBackdropBorderColor(CYAN.r * 0.15, CYAN.g * 0.15, CYAN.b * 0.15, 0.3)
    editorContainer:Hide()
    browserFrame.editorContainer = editorContainer

    -- Edit mode indicator accent
    local editAccentTop = editorContainer:CreateTexture(nil, "ARTWORK")
    editAccentTop:SetHeight(2)
    editAccentTop:SetPoint("TOPLEFT", editorContainer, "TOPLEFT", 4, -3)
    editAccentTop:SetPoint("TOPRIGHT", editorContainer, "TOPRIGHT", -4, -3)
    editAccentTop:SetColorTexture(0.9, 0.7, 0.2, 0.5)

    -- Editor scrollframe + editbox
    local editorScroll = CreateFrame("ScrollFrame", nil, editorContainer, "UIPanelScrollFrameTemplate")
    editorScroll:SetPoint("TOPLEFT", editorContainer, "TOPLEFT", 12, -10)
    editorScroll:SetPoint("BOTTOMRIGHT", editorContainer, "BOTTOMRIGHT", -16, 6)
    StyleScrollBar(editorScroll)

    local editorBox = CreateFrame("EditBox", nil, editorScroll)
    editorBox:SetMultiLine(true)
    editorBox:SetAutoFocus(false)
    editorBox:SetFontObject("GameFontNormal")
    local eW = editorScroll:GetWidth()
    editorBox:SetWidth((eW and eW > 1) and eW or 500)
    editorBox:SetHeight(400)
    editorBox:SetTextColor(0.92, 0.92, 0.92)
    editorBox:SetScript("OnEscapePressed", function(self) self:ClearFocus() end)
    editorScroll:SetScrollChild(editorBox)
    browserFrame.editorBox = editorBox
    browserFrame.editorScroll = editorScroll

    -- Click anywhere in scroll area to focus the editbox
    editorScroll:EnableMouse(true)
    editorScroll:SetScript("OnMouseUp", function() editorBox:SetFocus() end)

    editorScroll:SetScript("OnSizeChanged", function(self, w, h)
        editorBox:SetWidth(w)
        local textH = editorBox:GetHeight() or 0
        if textH < (h or 300) then
            editorBox:SetHeight(h or 300)
        end
    end)

    -- No auto-save; editing is manual save only
    editorBox:SetScript("OnTextChanged", function(self, userInput)
        if not userInput then return end
        -- Update unsaved indicator
        if browserFrame.saveBtn then
            local current = self:GetText()
            if current ~= browserSavedText then
                browserFrame.saveBtn.label:SetText("Save *")
            else
                browserFrame.saveBtn.label:SetText("Save")
            end
        end
    end)

    -- === Bottom button bar ===
    local buttonBar = CreateFrame("Frame", nil, rightPanel)
    buttonBar:SetPoint("BOTTOMLEFT", rightPanel, "BOTTOMLEFT", 0, 6)
    buttonBar:SetPoint("BOTTOMRIGHT", rightPanel, "BOTTOMRIGHT", 0, 6)
    buttonBar:SetHeight(26)
    browserFrame.buttonBar = buttonBar

    -- Edit button (view mode)
    local editBtn = CreateThemedButton(buttonBar, "Edit", 80, 24, "primary")
    editBtn:SetPoint("BOTTOMLEFT", buttonBar, "BOTTOMLEFT", 0, 0)
    editBtn:SetScript("OnClick", function()
        browserEditMode = true
        -- Snapshot text for revert
        EnsureDB()
        if browserSelectedKey and browserSelectedType then
            local tbl = ({ comp = SKToolsNotesDB.compNotes, vscomp = SKToolsNotesDB.vsCompNotes,
                matchup = SKToolsNotesDB.matchupNotes, spec = SKToolsNotesDB.specNotes,
                bg = SKToolsNotesDB.bgNotes, dungeon = SKToolsNotesDB.dungeonNotes, raid = SKToolsNotesDB.raidNotes, general = SKToolsNotesDB.generalNotes })[browserSelectedType]
            browserSavedText = (tbl and tbl[browserSelectedKey]) or ""
        end
        SKArenaNotes_RefreshBrowser()
        browserFrame.editorBox:SetFocus()
    end)
    browserFrame.editBtn = editBtn

    -- Save button (edit mode)
    local saveBtn = CreateThemedButton(buttonBar, "Save", 80, 24, "primary")
    saveBtn:SetPoint("BOTTOMLEFT", buttonBar, "BOTTOMLEFT", 0, 0)
    saveBtn:SetScript("OnClick", function()
        EnsureDB()
        local text = browserFrame.editorBox:GetText()
        if browserSelectedKey and browserSelectedType then
            local tbl = ({ comp = SKToolsNotesDB.compNotes, vscomp = SKToolsNotesDB.vsCompNotes,
                matchup = SKToolsNotesDB.matchupNotes, spec = SKToolsNotesDB.specNotes,
                bg = SKToolsNotesDB.bgNotes, dungeon = SKToolsNotesDB.dungeonNotes, raid = SKToolsNotesDB.raidNotes, general = SKToolsNotesDB.generalNotes })[browserSelectedType]
            if tbl then
                if text == "" then
                    tbl[browserSelectedKey] = nil
                    ClearNoteTimestamp(browserSelectedType, browserSelectedKey)
                else
                    tbl[browserSelectedKey] = text
                    TouchNote(browserSelectedType, browserSelectedKey)
                end
            end
        end
        browserSavedText = text
        browserEditMode = false
        browserFrame.editorBox:ClearFocus()
        SKArenaNotes_RefreshBrowser()
    end)
    browserFrame.saveBtn = saveBtn

    -- Revert button (edit mode)
    local revertBtn = CreateThemedButton(buttonBar, "Revert", 80, 24, "secondary")
    revertBtn:SetPoint("LEFT", saveBtn, "RIGHT", 6, 0)
    revertBtn:SetScript("OnClick", function()
        browserEditMode = false
        browserFrame.editorBox:SetText(browserSavedText)
        browserFrame.editorBox:ClearFocus()
        SKArenaNotes_RefreshBrowser()
    end)
    browserFrame.revertBtn = revertBtn

    -- Delete confirmation dialog
    StaticPopupDialogs["SKNOTES_DELETE_CONFIRM"] = {
        text = "Delete this note?",
        button1 = "Delete",
        button2 = "Cancel",
        OnAccept = function()
            if browserSelectedKey and browserSelectedType then
                local tbl = ({ comp = SKToolsNotesDB.compNotes, vscomp = SKToolsNotesDB.vsCompNotes,
                    matchup = SKToolsNotesDB.matchupNotes, spec = SKToolsNotesDB.specNotes,
                    bg = SKToolsNotesDB.bgNotes, dungeon = SKToolsNotesDB.dungeonNotes, raid = SKToolsNotesDB.raidNotes, general = SKToolsNotesDB.generalNotes })[browserSelectedType]
                if tbl then
                    tbl[browserSelectedKey] = nil
                end
                ClearNoteTimestamp(browserSelectedType, browserSelectedKey)
                browserSelectedKey = nil
                browserSelectedType = nil
                browserEditMode = false
                SKArenaNotes_RefreshBrowser()
            end
        end,
        timeout = 0,
        whileDead = true,
        hideOnEscape = true,
        preferredIndex = 3,
    }

    -- Delete button (always available when note selected)
    local deleteBtn = CreateThemedButton(buttonBar, "Delete", 70, 24, "danger")
    deleteBtn:SetPoint("BOTTOMRIGHT", buttonBar, "BOTTOMRIGHT", 0, 0)
    deleteBtn:SetScript("OnClick", function()
        if browserSelectedKey then
            StaticPopup_Show("SKNOTES_DELETE_CONFIRM")
        end
    end)
    browserFrame.deleteBtn = deleteBtn

    -- ===== Comp Builder Panel (shown when no note selected) =====
    local builderPanel = CreateFrame("Frame", nil, rightPanel)
    builderPanel:SetAllPoints()
    builderPanel:Hide()
    browserFrame.builderPanel = builderPanel

    local builderTitle = builderPanel:CreateFontString(nil, "OVERLAY", "GameFontNormalLarge")
    builderTitle:SetPoint("TOPLEFT", builderPanel, "TOPLEFT", 4, 0)
    builderTitle:SetText("Create New Note")
    builderTitle:SetTextColor(CYAN.r, CYAN.g, CYAN.b)

    -- Type selector buttons
    local builderTypeButtons = {}
    local builderTypes = {
        { key = "general", label = "General" },
        { key = "comp", label = "Your Comp" },
        { key = "vscomp", label = "Vs Comp" },
        { key = "matchup", label = "Matchup" },
    }
    local prevTypeBtn
    for i, bt in ipairs(builderTypes) do
        local btn = CreateFrame("Button", nil, builderPanel, "BackdropTemplate")
        btn:SetSize(95, 24)
        btn:SetBackdrop({
            bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
            edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
            tile = true, tileSize = 8, edgeSize = 10,
            insets = { left = 2, right = 2, top = 2, bottom = 2 },
        })
        if i == 1 then
            btn:SetPoint("TOPLEFT", builderTitle, "BOTTOMLEFT", 0, -10)
        else
            btn:SetPoint("LEFT", prevTypeBtn, "RIGHT", 6, 0)
        end
        btn.label = btn:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
        btn.label:SetPoint("CENTER")
        btn.label:SetText(bt.label)
        btn.typeKey = bt.key
        btn:SetScript("OnClick", function(self)
            browserFrame.builderType = self.typeKey
            wipe(browserFrame.builderSelected)
            browserFrame.builderMatchupPhase = 1
            wipe(browserFrame.builderMyTeam)
            SKArenaNotes_RefreshBuilder()
        end)
        builderTypeButtons[bt.key] = btn
        prevTypeBtn = btn
    end
    browserFrame.builderTypeButtons = builderTypeButtons

    -- Instructions
    local builderInstr = builderPanel:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
    builderInstr:SetPoint("TOPLEFT", builderTypeButtons["general"], "BOTTOMLEFT", 0, -8)
    builderInstr:SetPoint("RIGHT", builderPanel, "RIGHT", -4, 0)
    builderInstr:SetJustifyH("LEFT")
    builderInstr:SetWordWrap(true)
    browserFrame.builderInstr = builderInstr

    -- Spec grid scrollframe
    local builderScroll = CreateFrame("ScrollFrame", nil, builderPanel, "UIPanelScrollFrameTemplate")
    builderScroll:SetPoint("TOPLEFT", builderInstr, "BOTTOMLEFT", 0, -8)
    builderScroll:SetPoint("BOTTOMRIGHT", builderPanel, "BOTTOMRIGHT", -16, 36)
    StyleScrollBar(builderScroll)
    browserFrame.builderScroll = builderScroll

    local builderContent = CreateFrame("Frame", nil, builderScroll)
    builderContent:SetWidth(500)
    builderContent:SetHeight(1)
    builderScroll:SetScrollChild(builderContent)
    browserFrame.builderContent = builderContent

    builderScroll:SetScript("OnSizeChanged", function(self, w)
        builderContent:SetWidth(w)
    end)

    builderPanel:EnableMouseWheel(true)
    builderPanel:SetScript("OnMouseWheel", function(self, delta)
        local current = builderScroll:GetVerticalScroll()
        local maxScroll = builderScroll:GetVerticalScrollRange()
        builderScroll:SetVerticalScroll(math.max(0, math.min(current - delta * 30, maxScroll)))
    end)

    -- Selected display + action buttons at bottom
    local builderSelectedText = builderPanel:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
    builderSelectedText:SetPoint("BOTTOMLEFT", builderPanel, "BOTTOMLEFT", 4, 10)
    builderSelectedText:SetPoint("RIGHT", builderPanel, "RIGHT", -180, 0)
    builderSelectedText:SetJustifyH("LEFT")
    browserFrame.builderSelectedText = builderSelectedText

    local builderCreateBtn = CreateThemedButton(builderPanel, "Create", 100, 24, "primary")
    builderCreateBtn:SetPoint("BOTTOMRIGHT", builderPanel, "BOTTOMRIGHT", -4, 6)
    browserFrame.builderCreateBtn = builderCreateBtn

    local builderClearBtn = CreateThemedButton(builderPanel, "Clear", 70, 24, "secondary")
    builderClearBtn:SetPoint("RIGHT", builderCreateBtn, "LEFT", -6, 0)
    browserFrame.builderClearBtn = builderClearBtn
    builderClearBtn:SetScript("OnClick", function()
        wipe(browserFrame.builderSelected)
        browserFrame.builderMatchupPhase = 1
        wipe(browserFrame.builderMyTeam)
        SKArenaNotes_RefreshBuilder()
    end)

    -- Text input (shown only in "general" builder mode)
    local builderBGInput = CreateFrame("EditBox", nil, builderPanel, "InputBoxTemplate")
    builderBGInput:SetSize(300, 22)
    builderBGInput:SetPoint("TOPLEFT", builderInstr, "BOTTOMLEFT", 4, -8)
    builderBGInput:SetAutoFocus(false)
    builderBGInput:SetFontObject(GameFontHighlightSmall)
    builderBGInput:SetScript("OnEscapePressed", function(self) self:ClearFocus() end)
    builderBGInput:SetScript("OnEnterPressed", function(self) self:ClearFocus() end)
    builderBGInput:Hide()
    browserFrame.builderBGInput = builderBGInput

    builderCreateBtn:SetScript("OnClick", function()
        local selected = browserFrame.builderSelected
        local bType = browserFrame.builderType

        -- General mode: use text input for custom note title
        if bType == "general" then
            local title = browserFrame.builderBGInput:GetText()
            if not title or title:trim() == "" then
                print("|cff00E5EESKTools:|r Enter a title for your note.")
                return
            end
            title = title:trim()
            EnsureDB()
            if not SKToolsNotesDB.generalNotes[title] then
                SKToolsNotesDB.generalNotes[title] = ""
            end
            browserSelectedKey = title
            browserSelectedType = "general"
            browserShowBuilder = false
            browserEditMode = true
            browserSavedText = SKToolsNotesDB.generalNotes[title] or ""
            browserFrame.builderBGInput:SetText("")
            SKArenaNotes_RefreshBrowser()
            return
        end

        local team = {}
        for specKey in pairs(selected) do
            local spec, class = specKey:match("^(.+) (%u+)$")
            if spec and class then
                team[#team + 1] = { spec = spec, class = class }
            end
        end
        if #team < 2 then
            print("|cff00E5EESKTools:|r Select at least 2 specs.")
            return
        end

        EnsureDB()
        local key = MakeCompKey(team)

        if bType == "comp" then
            if not SKToolsNotesDB.compNotes[key] then
                SKToolsNotesDB.compNotes[key] = ""
            end
            browserSelectedKey = key
            browserSelectedType = "comp"
        elseif bType == "vscomp" then
            if not SKToolsNotesDB.vsCompNotes[key] then
                SKToolsNotesDB.vsCompNotes[key] = ""
            end
            browserSelectedKey = key
            browserSelectedType = "vscomp"
        elseif bType == "matchup" then
            if browserFrame.builderMatchupPhase == 1 then
                wipe(browserFrame.builderMyTeam)
                for k in pairs(selected) do browserFrame.builderMyTeam[k] = true end
                browserFrame.builderMatchupPhase = 2
                wipe(browserFrame.builderSelected)
                SKArenaNotes_RefreshBuilder()
                return
            else
                local myTeam = {}
                for specKey in pairs(browserFrame.builderMyTeam) do
                    local spec, class = specKey:match("^(.+) (%u+)$")
                    if spec and class then myTeam[#myTeam + 1] = { spec = spec, class = class } end
                end
                local matchupKey = MakeMatchupKey(myTeam, team)
                if not SKToolsNotesDB.matchupNotes[matchupKey] then
                    SKToolsNotesDB.matchupNotes[matchupKey] = ""
                end
                browserSelectedKey = matchupKey
                browserSelectedType = "matchup"
                browserFrame.builderMatchupPhase = 1
                wipe(browserFrame.builderMyTeam)
            end
        end

        wipe(browserFrame.builderSelected)
        -- Jump directly into edit mode for newly created notes
        browserEditMode = true
        browserSavedText = ""
        browserShowBuilder = false
        SKArenaNotes_RefreshBrowser()
    end)

    -- Builder state
    browserFrame.builderType = "comp"
    browserFrame.builderSelected = {}
    browserFrame.builderMatchupPhase = 1
    browserFrame.builderMyTeam = {}
    browserFrame.builderSpecButtons = {}

    -- Back button on builder panel (returns to generic notes)
    local builderBackBtn = CreateThemedButton(builderPanel, "Back", 70, 24, "secondary")
    builderBackBtn:SetPoint("BOTTOMLEFT", builderPanel, "BOTTOMLEFT", 0, 6)
    builderBackBtn:SetScript("OnClick", function()
        browserShowBuilder = false
        SKArenaNotes_RefreshBrowser()
    end)
    browserFrame.builderBackBtn = builderBackBtn

    -- Default right panel state: hide everything (refresh will show generic notes)
    editorHeader:Hide()
    editorCategory:Hide()
    viewContainer:Hide()
    editorContainer:Hide()
    buttonBar:Hide()
    editBtn:Hide()
    saveBtn:Hide()
    revertBtn:Hide()
    deleteBtn:Hide()
    builderPanel:Hide()

    browserFrame:Hide()
    return browserFrame
end

-- Create spec buttons in the builder grid (lazy, called once)
local function EnsureBuilderSpecButtons()
    if not browserFrame or not browserFrame.builderContent then return end
    if #browserFrame.builderSpecButtons > 0 then return end

    local content = browserFrame.builderContent
    local byClass = GetAllSpecsByClass()
    local yOff = 0

    for _, classEntry in ipairs(byClass) do
        local r, g, b = GetClassColor(classEntry.classFile)
        local header = content:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
        header:SetPoint("TOPLEFT", content, "TOPLEFT", 2, -yOff)
        header:SetText(ClassIconString(classEntry.classFile, 14) .. " " .. classEntry.className)
        header:SetTextColor(r, g, b)
        yOff = yOff + 18

        for _, spec in ipairs(classEntry.specs) do
            local btn = CreateFrame("Button", nil, content, "BackdropTemplate")
            btn:SetHeight(20)
            btn:SetPoint("TOPLEFT", content, "TOPLEFT", 16, -yOff)
            btn:SetPoint("RIGHT", content, "RIGHT", -4, 0)
            btn:SetBackdrop({
                bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
                edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
                tile = true, tileSize = 8, edgeSize = 8,
                insets = { left = 1, right = 1, top = 1, bottom = 1 },
            })
            btn:SetBackdropColor(0.07, 0.07, 0.09, 0.6)
            btn:SetBackdropBorderColor(0.2, 0.2, 0.25, 0.3)

            btn.label = btn:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
            btn.label:SetPoint("LEFT", btn, "LEFT", 6, 0)
            btn.label:SetText(spec.name)

            btn.specKey = spec.key

            local hover = btn:CreateTexture(nil, "HIGHLIGHT")
            hover:SetAllPoints()
            hover:SetColorTexture(1, 1, 1, 0.12)

            local sel = btn:CreateTexture(nil, "BACKGROUND", nil, 1)
            sel:SetAllPoints()
            sel:SetColorTexture(CYAN.r * 0.3, CYAN.g * 0.3, CYAN.b * 0.3, 0.7)
            sel:Hide()
            btn.selOverlay = sel

            -- Phase 1 matchup highlight (locked-in "your team" — dimmer CYAN)
            local myTeamOverlay = btn:CreateTexture(nil, "BACKGROUND", nil, 2)
            myTeamOverlay:SetAllPoints()
            myTeamOverlay:SetColorTexture(0, CYAN.g * 0.15, CYAN.b * 0.15, 0.5)
            myTeamOverlay:Hide()
            btn.myTeamOverlay = myTeamOverlay

            btn:SetScript("OnClick", function(self)
                -- In "spec" builder mode, clicking a spec immediately creates/selects the note
                if browserFrame.builderType == "spec" then
                    EnsureDB()
                    local key = self.specKey
                    if not SKToolsNotesDB.specNotes[key] then
                        SKToolsNotesDB.specNotes[key] = ""
                    end
                    browserSelectedKey = key
                    browserSelectedType = "spec"
                    SKArenaNotes_RefreshBrowser()
                    return
                end
                local bSelected = browserFrame.builderSelected
                if bSelected[self.specKey] then
                    bSelected[self.specKey] = nil
                else
                    bSelected[self.specKey] = true
                end
                SKArenaNotes_RefreshBuilder()
            end)

            browserFrame.builderSpecButtons[#browserFrame.builderSpecButtons + 1] = btn
            yOff = yOff + 22
        end
        yOff = yOff + 4
    end
    content:SetHeight(yOff + 10)
end

local CATEGORY_LABELS = { comp = "Your Comps", vscomp = "Vs Comps", matchup = "Matchups", spec = "Specs", bg = "Battlegrounds", dungeon = "Dungeons", raid = "Raids", general = "General" }

function SKArenaNotes_RefreshBuilder()
    if not browserFrame or not browserFrame.builderPanel then return end
    EnsureBuilderSpecButtons()

    local bType = browserFrame.builderType
    local bSelected = browserFrame.builderSelected

    -- Update type buttons (active = CYAN accent, inactive = subtle)
    for key, btn in pairs(browserFrame.builderTypeButtons) do
        if key == bType then
            btn:SetBackdropColor(0, CYAN.g * 0.2, CYAN.b * 0.2, 0.95)
            btn:SetBackdropBorderColor(CYAN.r, CYAN.g, CYAN.b, 0.8)
            btn.label:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
        else
            btn:SetBackdropColor(0.08, 0.08, 0.10, 0.8)
            btn:SetBackdropBorderColor(0.25, 0.25, 0.30, 0.5)
            btn.label:SetTextColor(0.6, 0.6, 0.6)
        end
    end

    -- Update instructions
    if bType == "comp" then
        browserFrame.builderInstr:SetText("Select 2-3 specs for your comp, then click Create:")
    elseif bType == "vscomp" then
        browserFrame.builderInstr:SetText("Select 2-3 specs for the enemy comp, then click Create:")
    elseif bType == "matchup" then
        if browserFrame.builderMatchupPhase == 1 then
            browserFrame.builderInstr:SetText("Step 1: Select your team's specs, then click Next:")
        else
            browserFrame.builderInstr:SetText("Step 2: Select enemy team's specs, then click Create:")
        end
    elseif bType == "spec" then
        browserFrame.builderInstr:SetText("Click a spec to create or edit its note:")
    elseif bType == "general" then
        browserFrame.builderInstr:SetText("Type a title for your note, then click Create:")
    end

    -- Show/hide text input (used in general mode)
    if browserFrame.builderBGInput then
        if bType == "general" then
            browserFrame.builderBGInput:Show()
        else
            browserFrame.builderBGInput:Hide()
        end
    end

    -- Hide Create/Clear buttons in spec mode (clicking directly opens the note)
    -- General mode: show Create, hide Clear (uses text input, not spec selection)
    if bType == "spec" then
        browserFrame.builderCreateBtn:Hide()
        browserFrame.builderClearBtn:Hide()
        browserFrame.builderSelectedText:SetText("")
    elseif bType == "general" then
        browserFrame.builderCreateBtn:Show()
        browserFrame.builderClearBtn:Hide()
        browserFrame.builderSelectedText:SetText("")
    else
        browserFrame.builderCreateBtn:Show()
        browserFrame.builderClearBtn:Show()
    end

    -- Hide spec grid in general mode, show in other modes
    if browserFrame.builderScroll then
        if bType == "general" then
            browserFrame.builderScroll:Hide()
        else
            browserFrame.builderScroll:Show()
        end
    end

    -- Update spec button highlights
    for _, btn in ipairs(browserFrame.builderSpecButtons) do
        if bSelected[btn.specKey] then
            btn.selOverlay:Show()
            btn.label:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
        else
            btn.selOverlay:Hide()
            btn.label:SetTextColor(0.8, 0.8, 0.8)
        end
        -- Show dimmer CYAN for locked-in "your team" in matchup phase 2
        if bType == "matchup" and browserFrame.builderMatchupPhase == 2
            and browserFrame.builderMyTeam[btn.specKey] then
            btn.myTeamOverlay:Show()
        else
            btn.myTeamOverlay:Hide()
        end
    end

    -- Update selected text (display-friendly names) — skip for bg mode
    if bType ~= "general" then
        local count = 0
        local names = {}
        for k in pairs(bSelected) do
            count = count + 1
            names[#names + 1] = k
        end
        table.sort(names)
        local displayNames = {}
        for _, k in ipairs(names) do
            displayNames[#displayNames + 1] = SpecKeyToDisplay(k)
        end
        browserFrame.builderSelectedText:SetText("Selected: " .. (count > 0 and table.concat(displayNames, ", ") or "none"))
    end

    -- Button label
    if bType == "matchup" and browserFrame.builderMatchupPhase == 1 then
        browserFrame.builderCreateBtn:SetText("Next  >")
    else
        browserFrame.builderCreateBtn:SetText("Create")
    end
end

local function StyleToggleBtn(btn, active)
    if active then
        btn:SetBackdropColor(0, CYAN.g * 0.2, CYAN.b * 0.2, 0.95)
        btn:SetBackdropBorderColor(CYAN.r, CYAN.g, CYAN.b, 0.8)
        btn.label:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
    else
        btn:SetBackdropColor(0.08, 0.08, 0.10, 0.8)
        btn:SetBackdropBorderColor(0.25, 0.25, 0.30, 0.5)
        btn.label:SetTextColor(0.6, 0.6, 0.6)
    end
end

local function UpdateViewToggle()
    if not browserFrame or not browserFrame.pvpBtn or not browserFrame.pveBtn then return end
    StyleToggleBtn(browserFrame.pvpBtn, browserShowPvP)
    StyleToggleBtn(browserFrame.pveBtn, browserShowPvE)
end

local function _RefreshBrowserImpl()
    -- Restore view filters from saved variables
    if SKToolsNotesDB then
        browserShowPvP = SKToolsNotesDB.browserShowPvP ~= false
        browserShowPvE = SKToolsNotesDB.browserShowPvE ~= false
    end
    UpdateViewToggle()

    local entries = BuildBrowserEntries()
    browserEntries = entries  -- cache for keyboard navigation
    local numEntries = #entries

    -- Clamp focused index to valid range
    if browserFocusedIdx and browserFocusedIdx > numEntries then
        browserFocusedIdx = nil
    end

    -- Show empty state if no entries at all (not even spec headers when showing all)
    local hasAnyNotes = numEntries > 0
    if browserFrame.emptyMsg then
        if hasAnyNotes then
            browserFrame.emptyMsg:Hide()
        else
            browserFrame.emptyMsg:Show()
        end
    end

    -- Compute max scroll offset: count entries from bottom that fill the view
    local viewH = browserFrame.listFrame:GetHeight() or 706
    local bottomH = 0
    local fitCount = 0
    for j = numEntries, 1, -1 do
        local e = entries[j]
        local h = BROWSER_ROW_HEIGHT_COMPACT
        if e.type == "header" then
            h = BROWSER_HEADER_HEIGHT
        else
            local nt = e.tbl[e.key] or ""
            if nt ~= "" then h = BROWSER_ROW_HEIGHT end
        end
        if bottomH + h > viewH then break end
        bottomH = bottomH + h
        fitCount = fitCount + 1
    end
    local maxOffset = math.max(0, numEntries - fitCount)
    browserFrame.maxScrollOffset = maxOffset

    FauxScrollFrame_Update(browserFrame.listScroll, numEntries, BROWSER_VISIBLE_ROWS, BROWSER_ROW_HEIGHT_COMPACT)
    local offset = FauxScrollFrame_GetOffset(browserFrame.listScroll) or 0

    local yOff = 0
    local listHeight = browserFrame.listFrame:GetHeight()
    for i = 1, BROWSER_VISIBLE_ROWS do
        local row = browserListButtons[i]
        local idx = offset + i
        if idx <= numEntries and yOff < listHeight then
            local entry = entries[idx]
            row:ClearAllPoints()

            if entry.type == "header" then
                row:SetHeight(BROWSER_HEADER_HEIGHT)
                row:SetPoint("TOPLEFT", browserFrame.listFrame, "TOPLEFT", 0, -yOff)
                row.titleText:SetText(entry.label)
                row.titleText:ClearAllPoints()
                row.titleText:SetPoint("BOTTOMLEFT", row, "BOTTOMLEFT", 4, 4)
                row.titleText:SetPoint("RIGHT", row, "RIGHT", -6, 0)
                row.titleText:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
                row.previewText:SetText("")
                row.previewText:Hide()
                row.timeText:SetText("")
                row.headerLine:Show()
                row.sepLine:Hide()
                row.selHighlight:Hide()
                row.noteAccent:Hide()
                row.bg:SetColorTexture(0, 0, 0, 0)
                row.noteKey = nil
                row.noteType = nil
                row.noteTbl = nil
                row.sectionKey = entry.sectionKey or nil
                row.focusAccent:Hide()
                row.hoverHighlight:SetColorTexture(1, 1, 1, 0.04)
                yOff = yOff + BROWSER_HEADER_HEIGHT
            else
                local displayText = FormatKeyForDisplay(entry.key, entry.tabType)
                local noteText = entry.tbl[entry.key] or ""
                local hasNote = noteText ~= ""
                local rowH = hasNote and BROWSER_ROW_HEIGHT or BROWSER_ROW_HEIGHT_COMPACT

                row:SetHeight(rowH)
                row:SetPoint("TOPLEFT", browserFrame.listFrame, "TOPLEFT", 0, -yOff)

                -- Timestamp
                local ts = GetNoteTimestamp(entry.tabType, entry.key)
                row.timeText:SetText(hasNote and (FormatRelativeTime(ts) or "") or "")

                row.titleText:ClearAllPoints()
                row.previewText:ClearAllPoints()
                if hasNote then
                    row.titleText:SetPoint("TOPLEFT", row, "TOPLEFT", 8, -6)
                    row.titleText:SetPoint("RIGHT", row.timeText, "LEFT", -6, 0)
                    row.previewText:SetPoint("BOTTOMLEFT", row, "BOTTOMLEFT", 8, 5)
                    row.previewText:SetPoint("RIGHT", row, "RIGHT", -8, 0)
                    local preview = noteText:sub(1, 80):gsub("\n", " ")
                    row.previewText:SetText("|cff666666" .. preview .. "|r")
                    row.previewText:Show()
                else
                    row.titleText:SetPoint("LEFT", row, "LEFT", 8, 0)
                    row.titleText:SetPoint("RIGHT", row, "RIGHT", -8, 0)
                    row.previewText:Hide()
                end

                row.titleText:SetText(displayText)
                if entry.tabType == "spec" then
                    local _, classToken = entry.key:match("^(.+) (%u+)$")
                    if classToken then
                        local r, g, b = GetClassColor(classToken)
                        row.titleText:SetTextColor(r, g, b)
                    else
                        row.titleText:SetTextColor(0.78, 0.78, 0.78)
                    end
                else
                    row.titleText:SetTextColor(0.82, 0.82, 0.82)
                end
                row.headerLine:Hide()
                row.sepLine:Show()
                row.hoverHighlight:SetColorTexture(1, 1, 1, 0.04)
                row.noteKey = entry.key
                row.noteType = entry.tabType
                row.noteTbl = entry.tbl
                row.sectionKey = nil

                if hasNote then
                    row.noteAccent:Show()
                else
                    row.noteAccent:Hide()
                end

                -- Backgrounds: items with notes get a subtle fill, others transparent
                if hasNote then
                    row.bg:SetColorTexture(0.06, 0.06, 0.08, 0.5)
                else
                    row.bg:SetColorTexture(0, 0, 0, 0)
                end

                if entry.key == browserSelectedKey and entry.tabType == browserSelectedType then
                    row.selHighlight:Show()
                else
                    row.selHighlight:Hide()
                end

                if browserFocusedIdx == idx then
                    row.focusAccent:Show()
                    row.bg:SetColorTexture(0.08, 0.10, 0.14, 0.7)
                else
                    row.focusAccent:Hide()
                end

                yOff = yOff + rowH
            end

            row:Show()
        else
            row:Hide()
            row.noteKey = nil
            row.noteType = nil
            row.noteTbl = nil
            row.sectionKey = nil
            row.focusAccent:Hide()
            row.noteAccent:Hide()
            row.sepLine:Hide()
        end
    end

    -- Right panel: note selected → view/edit, builder mode → builder, else → generic notes
    local function HideAllRightPanels()
        browserFrame.editorHeader:Hide()
        browserFrame.editorCategory:Hide()
        browserFrame.titleBar:SetHeight(1)
        browserFrame.titleLabel:SetText("")
        browserFrame.titleLabel:Hide()
        browserFrame.titleEditBtn:Hide()
        browserFrame.titleSetBtn:Hide()
        browserFrame.editorBox:ClearFocus()
        browserFrame.viewContainer:Hide()
        browserFrame.editorContainer:Hide()
        browserFrame.buttonBar:Hide()
        browserFrame.editBtn:Hide()
        browserFrame.saveBtn:Hide()
        browserFrame.revertBtn:Hide()
        browserFrame.deleteBtn:Hide()
        browserFrame.builderPanel:Hide()
    end

    -- Default to "General Notes" when nothing is selected
    if not browserSelectedKey and not browserSelectedType and not browserShowBuilder then
        EnsureDB()
        browserSelectedKey = "General Notes"
        browserSelectedType = "general"
    end

    if browserSelectedKey and browserSelectedType then
        HideAllRightPanels()
        local tbl = ({ comp = SKToolsNotesDB.compNotes, vscomp = SKToolsNotesDB.vsCompNotes,
            matchup = SKToolsNotesDB.matchupNotes, spec = SKToolsNotesDB.specNotes,
            bg = SKToolsNotesDB.bgNotes, general = SKToolsNotesDB.generalNotes })[browserSelectedType]

        -- Build header text and title bar based on type
        if browserSelectedType == "matchup" then
            -- Matchup: nicely formatted two-line header
            local myPart, enemyPart = browserSelectedKey:match("^(.+)|(.+)$")
            if myPart and enemyPart then
                local myTeam = CompKeyToTeam(myPart)
                local enemyTeam = CompKeyToTeam(enemyPart)
                local myIcons = CompIconString(myTeam, 18)
                local enemyIcons = CompIconString(enemyTeam, 18)
                local myTitle = GetCompTitle(myPart)
                local enemyTitle = GetCompTitle(enemyPart)
                local myLabel = myTitle and ("|cffFFD100" .. myTitle .. "|r  " .. myIcons) or (myIcons .. "  " .. CompKeyToDisplay(myPart))
                local enemyLabel = enemyTitle and ("|cffFF6666" .. enemyTitle .. "|r  " .. enemyIcons) or (enemyIcons .. "  " .. CompKeyToDisplay(enemyPart))
                browserFrame.editorHeader:SetText(myLabel .. "\n|cff888888vs|r\n" .. enemyLabel)
                browserFrame.editorHeader:SetMaxLines(4)
            else
                browserFrame.editorHeader:SetText(FormatKeyForDisplay(browserSelectedKey, browserSelectedType))
                browserFrame.editorHeader:SetMaxLines(2)
            end
            browserFrame.editorHeader:SetTextColor(0.9, 0.9, 0.9)
        elseif browserSelectedType == "comp" or browserSelectedType == "vscomp" then
            -- Comp/VsComp: show icons + specs, with title bar
            local team = CompKeyToTeam(browserSelectedKey)
            local icons = CompIconString(team, 18)
            local title = GetCompTitle(browserSelectedKey)
            if title then
                browserFrame.editorHeader:SetText(icons .. "  |cffFFD100" .. title .. "|r")
                browserFrame.titleLabel:SetText("|cff999999" .. CompKeyToDisplay(browserSelectedKey) .. "|r")
                browserFrame.titleLabel:Show()
                browserFrame.titleEditBtn:Show()
                browserFrame.titleSetBtn:Hide()
            else
                browserFrame.editorHeader:SetText(icons .. "  " .. CompKeyToDisplay(browserSelectedKey))
                browserFrame.titleLabel:SetText("")
                browserFrame.titleLabel:Hide()
                browserFrame.titleEditBtn:Hide()
                browserFrame.titleSetBtn:Show()
            end
            browserFrame.editorHeader:SetTextColor(0.9, 0.9, 0.9)
            browserFrame.editorHeader:SetMaxLines(2)
            browserFrame.titleBar:SetHeight(18)
        elseif browserSelectedType == "spec" then
            browserFrame.editorHeader:SetText(FormatKeyForDisplay(browserSelectedKey, browserSelectedType))
            browserFrame.editorHeader:SetMaxLines(2)
            local _, classToken = browserSelectedKey:match("^(.+) (%u+)$")
            if classToken then
                local r, g, b = GetClassColor(classToken)
                browserFrame.editorHeader:SetTextColor(r, g, b)
            else
                browserFrame.editorHeader:SetTextColor(0.9, 0.9, 0.9)
            end
        elseif browserSelectedType == "bg" or browserSelectedType == "dungeon" or browserSelectedType == "raid" then
            browserFrame.editorHeader:SetText(browserSelectedKey)
            browserFrame.editorHeader:SetMaxLines(2)
            browserFrame.editorHeader:SetTextColor(CYAN.r, CYAN.g, CYAN.b)
        elseif browserSelectedType == "general" then
            browserFrame.editorHeader:SetText(browserSelectedKey)
            browserFrame.editorHeader:SetMaxLines(2)
            browserFrame.editorHeader:SetTextColor(0.9, 0.9, 0.9)
        end

        local catLabel = CATEGORY_LABELS[browserSelectedType] or ""
        local ts = GetNoteTimestamp(browserSelectedType, browserSelectedKey)
        local relTime = FormatRelativeTime(ts)
        if relTime then
            catLabel = catLabel .. "  |cff666666edited " .. relTime .. "|r"
        end
        browserFrame.editorCategory:SetText(catLabel)
        browserFrame.editorHeader:Show()
        browserFrame.editorCategory:Show()
        browserFrame.buttonBar:Show()

        local noteText = (tbl and tbl[browserSelectedKey]) or ""

        -- "Clear" for specs (row stays in list), "Delete" for others
        if browserSelectedType == "spec" or browserSelectedType == "bg" or browserSelectedType == "dungeon" or browserSelectedType == "raid"
            or (browserSelectedType == "general" and browserSelectedKey == "General Notes") then
            browserFrame.deleteBtn.label:SetText("Clear")
        else
            browserFrame.deleteBtn.label:SetText("Delete")
        end
        browserFrame.deleteBtn:Show()

        if browserEditMode then
            -- Edit mode: show editor container with editbox
            browserFrame.editorContainer:Show()
            -- Only set text if editbox doesn't already have focus (avoid resetting mid-edit)
            if not browserFrame.editorBox:HasFocus() then
                browserFrame.editorBox:SetText(noteText)
            end
            browserFrame.editorBox:SetFocus()
            browserFrame.saveBtn:Show()
            -- Update unsaved indicator
            local current = browserFrame.editorBox:GetText()
            if current ~= browserSavedText then
                browserFrame.saveBtn.label:SetText("Save *")
            else
                browserFrame.saveBtn.label:SetText("Save")
            end
            browserFrame.revertBtn:Show()
        else
            -- View mode: show parchment scroll
            browserFrame.editBtn:Show()
            browserFrame.viewContainer:Show()

            if noteText ~= "" then
                browserFrame.viewText:SetText(noteText)
                browserFrame.viewEmpty:Hide()
                browserFrame.viewScroll:Show()
                -- Update content size to fit text (deferred so layout is ready)
                C_Timer.After(0, function()
                    if browserFrame.viewText and browserFrame.viewContent and browserFrame.viewScroll then
                        local scrollW = browserFrame.viewScroll:GetWidth() or 500
                        if scrollW > 1 then browserFrame.viewContent:SetWidth(scrollW) end
                        local textH = browserFrame.viewText:GetStringHeight() or 0
                        local scrollH = browserFrame.viewScroll:GetHeight() or 300
                        browserFrame.viewContent:SetHeight(math.max(textH + 16, scrollH))
                    end
                end)
            else
                browserFrame.viewScroll:Hide()
                browserFrame.viewEmpty:Show()
            end
        end
    elseif browserShowBuilder then
        -- Builder mode (creating a new structured note)
        HideAllRightPanels()
        browserEditMode = false
        browserFrame.builderPanel:Show()
        SKArenaNotes_RefreshBuilder()
    end
end

function SKArenaNotes_RefreshBrowser()
    if not browserFrame or not browserFrame:IsShown() then return end
    local ok, err = pcall(_RefreshBrowserImpl)
    if not ok then
        print("|cff00E5EESKTools|r |cffFF4444error|r [NotesBrowser]: " .. tostring(err))
    end
end

function SKArenaNotes_ToggleBrowser()
    -- In PvP: toggle contextual prep notes instead of full browser
    if inPvPInstance and prepFrame then
        if prepFrame:IsShown() then
            prepFrame:Hide()
        else
            FadeIn(prepFrame, 0.2)
        end
        return
    end

    -- Outside PvP: toggle the full browser
    CreateBrowserFrame()
    if browserFrame.matchNotesBtn then browserFrame.matchNotesBtn:Hide() end
    if browserFrame:IsShown() then
        -- Stop any in-progress fade before hiding
        if browserFrame._fadeTicker then browserFrame._fadeTicker:Cancel(); browserFrame._fadeTicker = nil end
        browserFrame:SetAlpha(1)
        browserFrame:Hide()
    else
        FadeIn(browserFrame, 0.2)
        SKArenaNotes_RefreshBrowser()
    end
end



-----------------------------
-- Event Handling
-----------------------------
local prepPollTimer
local prepShownForMatch = false    -- true once ALL data detected (final state)
local prepLastEnemyCount = 0       -- track changes to avoid unnecessary rebuilds
local prepLastPlayerComplete = false
local cachedPlayerTeam = nil       -- best resolved player team (survives flaky re-detection)

-- Figure out how many opponents to expect based on bracket
local function GetExpectedEnemyCount()
    if not IsActiveBattlefieldArena() then return 0 end
    local bracket = C_PvP.GetActiveMatchBracket and C_PvP.GetActiveMatchBracket()
    if bracket == 0 then return 2 end  -- 2v2
    return 3  -- 3v3, shuffle, etc.
end

-- Check that all player team members have resolved specs (not "Unknown")
local function IsPlayerTeamComplete(team, expected)
    if #team < expected then return false end
    for _, p in ipairs(team) do
        if p.spec == "Unknown" then return false end
    end
    return true
end

-- Progressive update: show/update the popup as data becomes available.
-- Only rebuilds when detected data actually changes to avoid interrupting typing.
-- Shows whatever is available: enemy specs don't need your team, YOUR COMP needs
-- your team, MATCHUP needs both.
local function TryUpdatePrepNotes()
    if not SKToolsDB or not SKToolsDB.arenaNotes then return false end
    if not IsActiveBattlefieldArena() then return false end

    local playerTeam = DetectPlayerTeam()
    local enemyTeam = DetectEnemyTeam()
    local expected = GetExpectedEnemyCount()
    local playerComplete = IsPlayerTeamComplete(playerTeam, expected)
    local eCount = #enemyTeam

    -- Cache best player team data — GetInspectSpecialization can flake out in arena
    -- and return 0 on re-query even after previously returning valid specs
    if playerComplete then
        cachedPlayerTeam = playerTeam
    elseif cachedPlayerTeam and IsPlayerTeamComplete(cachedPlayerTeam, expected) then
        playerTeam = cachedPlayerTeam
        playerComplete = true
    end

    -- Need at least SOMETHING to show (player team resolved OR any enemy detected)
    if not playerComplete and eCount == 0 then return false end

    -- Only rebuild the popup if data actually changed
    if playerComplete == prepLastPlayerComplete and eCount == prepLastEnemyCount then
        return eCount >= expected and playerComplete  -- already up to date
    end

    prepLastPlayerComplete = playerComplete
    prepLastEnemyCount = eCount

    -- Show/update popup with whatever data we have so far
    PopulateAndShowPrepNotes(playerTeam, enemyTeam, expected, playerComplete)

    -- Final state: all enemies detected AND player team resolved
    if eCount >= expected and playerComplete then
        prepShownForMatch = true
        if prepPollTimer then prepPollTimer:Cancel(); prepPollTimer = nil end
        return true
    end

    return false
end

-- Poll every 0.5s, progressively updating the popup as enemies are detected.
-- Stops when all enemies detected or after 60s timeout.
local function StartPrepPolling()
    if prepPollTimer then prepPollTimer:Cancel() end
    local attempts = 0
    local maxAttempts = 120  -- 60 seconds (entire prep phase)
    prepPollTimer = C_Timer.NewTicker(0.5, function(ticker)
        attempts = attempts + 1
        if prepShownForMatch or not IsActiveBattlefieldArena() or attempts >= maxAttempts then
            ticker:Cancel()
            prepPollTimer = nil
            return
        end
        TryUpdatePrepNotes()
    end)
end

local function OnMatchStart()
    if prepPollTimer then prepPollTimer:Cancel(); prepPollTimer = nil end
    if prepFrame and prepFrame:IsShown() then
        prepFrame:Hide()
    end
end

local function OnMatchComplete()
    if not SKToolsDB or not SKToolsDB.arenaNotes then return end

    C_Timer.After(1.5, function()
        if not SKToolsArenaDB or not SKToolsArenaDB.matches or #SKToolsArenaDB.matches == 0 then return end

        local lastMatch = SKToolsArenaDB.matches[1]
        if not lastMatch.playerTeam or not lastMatch.enemyTeam then return end
        if #lastMatch.playerTeam == 0 or #lastMatch.enemyTeam == 0 then return end

        ShowPostMatchPrompt(lastMatch.playerTeam, lastMatch.enemyTeam, lastMatch.won)
    end)
end

local function OnPlayerEnteringWorld()
    local _, instanceType = IsInInstance()
    if instanceType == "arena" then
        -- Skip if already in combat (mid-fight reload or reconnect)
        if UnitAffectingCombat("player") then return end
        if not SKToolsDB or not SKToolsDB.arenaNotesArena then return end
        -- Reset state for new arena
        inPvPInstance = true
        currentBGMapName = nil
        prepShownForMatch = false
        prepLastEnemyCount = 0
        prepLastPlayerComplete = false
        cachedPlayerTeam = nil
        -- Start polling immediately — the popup will show YOUR COMP as soon
        -- as player team specs are resolved, then add enemies as they appear.
        -- Short delay to let the instance finish loading.
        C_Timer.After(0.5, function()
            if prepShownForMatch then return end
            if not IsActiveBattlefieldArena() then return end
            if UnitAffectingCombat("player") then return end
            -- Try once immediately, then start polling
            if not TryUpdatePrepNotes() then
                StartPrepPolling()
            end
        end)
    elseif instanceType == "pvp" then
        -- Skip if already in combat (mid-fight reload or reconnect)
        if UnitAffectingCombat("player") then return end
        if not SKToolsDB or not SKToolsDB.arenaNotesBG then return end
        -- Battleground — show BG notes after a short delay for zone text to resolve
        inPvPInstance = true
        prepShownForMatch = false
        prepLastEnemyCount = 0
        prepLastPlayerComplete = false
        cachedPlayerTeam = nil
        if prepPollTimer then prepPollTimer:Cancel(); prepPollTimer = nil end
        C_Timer.After(1.0, function()
            local _, iType = IsInInstance()
            if iType ~= "pvp" then return end
            if UnitAffectingCombat("player") then return end
            local mapName = GetRealZoneText()
            if (not mapName or mapName == "") and C_Map and C_Map.GetBestMapForUnit then
                local mapID = C_Map.GetBestMapForUnit("player")
                if mapID then
                    local info = C_Map.GetMapInfo(mapID)
                    if info and info.name and info.name ~= "" then mapName = info.name end
                end
            end
            if not mapName or mapName == "" then return end
            currentBGMapName = mapName
            EnsureDB()
            -- Auto-create entry so the map appears in the browser
            if not SKToolsNotesDB.bgNotes[mapName] then
                SKToolsNotesDB.bgNotes[mapName] = ""
            end
            ShowInstancePrepNote(mapName, SKToolsNotesDB.bgNotes, "bg", "BG Notes")
        end)
    elseif instanceType == "party" then
        if UnitAffectingCombat("player") then return end
        if not SKToolsDB or not SKToolsDB.arenaNotesDungeon then return end
        inPvPInstance = false
        C_Timer.After(1.0, function()
            local _, iType = IsInInstance()
            if iType ~= "party" then return end
            if UnitAffectingCombat("player") then return end
            local mapName = GetRealZoneText()
            if not mapName or mapName == "" then return end
            EnsureDB()
            if not SKToolsNotesDB.dungeonNotes[mapName] then
                SKToolsNotesDB.dungeonNotes[mapName] = ""
            end
            ShowInstancePrepNote(mapName, SKToolsNotesDB.dungeonNotes, "dungeon", "Dungeon Notes")
        end)
    elseif instanceType == "raid" then
        if UnitAffectingCombat("player") then return end
        if not SKToolsDB or not SKToolsDB.arenaNotesRaid then return end
        inPvPInstance = false
        C_Timer.After(1.0, function()
            local _, iType = IsInInstance()
            if iType ~= "raid" then return end
            if UnitAffectingCombat("player") then return end
            local mapName = GetRealZoneText()
            if not mapName or mapName == "" then return end
            EnsureDB()
            if not SKToolsNotesDB.raidNotes[mapName] then
                SKToolsNotesDB.raidNotes[mapName] = ""
            end
            ShowInstancePrepNote(mapName, SKToolsNotesDB.raidNotes, "raid", "Raid Notes")
        end)
    else
        prepShownForMatch = false
        prepLastEnemyCount = 0
        prepLastPlayerComplete = false
        cachedPlayerTeam = nil
        if prepPollTimer then prepPollTimer:Cancel(); prepPollTimer = nil end
        inPvPInstance = false
        currentBGMapName = nil
        if browserFrame and browserFrame.matchNotesBtn then browserFrame.matchNotesBtn:Hide() end
        if prepFrame and prepFrame:IsShown() then
            prepFrame:Hide()
        end
        if postMatchFrame and postMatchFrame:IsShown() then
            postMatchFrame:Hide()
        end
    end
end

local function OnEvent(self, event, ...)
    local ok, err = pcall(function()
        if event == "ARENA_PREP_OPPONENT_SPECIALIZATIONS" then
            -- Opponent specs available — update immediately
            TryUpdatePrepNotes()
        elseif event == "ARENA_OPPONENT_UPDATE" then
            -- New opponent detected — update immediately
            TryUpdatePrepNotes()
        elseif event == "GROUP_ROSTER_UPDATE" then
            -- Party member data changed — re-check player team specs
            if IsActiveBattlefieldArena() and not prepShownForMatch then
                TryUpdatePrepNotes()
            end
        elseif event == "PLAYER_REGEN_DISABLED" then
            if IsActiveBattlefieldArena() then
                OnMatchStart()
            end
        elseif event == "PVP_MATCH_COMPLETE" then
            if IsActiveBattlefieldArena() then
                OnMatchComplete()
            end
        elseif event == "PLAYER_ENTERING_WORLD" then
            OnPlayerEnteringWorld()
        end
    end)
    if not ok then
        print("|cff00E5EESKTools|r |cffFF4444error|r [ArenaNotes:" .. event .. "]: " .. tostring(err))
    end
end

-----------------------------
-- Enable / Disable
-----------------------------
function SKArenaNotesEnable()
    EnsureDB()
    if not eventFrame then
        eventFrame = CreateFrame("Frame")
    end
    eventFrame:RegisterEvent("ARENA_PREP_OPPONENT_SPECIALIZATIONS")
    eventFrame:RegisterEvent("ARENA_OPPONENT_UPDATE")
    eventFrame:RegisterEvent("GROUP_ROSTER_UPDATE")
    eventFrame:RegisterEvent("PLAYER_REGEN_DISABLED")
    eventFrame:RegisterEvent("PVP_MATCH_COMPLETE")
    eventFrame:RegisterEvent("PLAYER_ENTERING_WORLD")
    eventFrame:SetScript("OnEvent", OnEvent)
end

function SKArenaNotesDisable()
    if eventFrame then
        eventFrame:UnregisterAllEvents()
        eventFrame:SetScript("OnEvent", nil)
    end
    if prepPollTimer then prepPollTimer:Cancel(); prepPollTimer = nil end
    if prepFrame then prepFrame:Hide() end
    if postMatchFrame then postMatchFrame:Hide() end
end
